﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/rds/RDSRequest.h>
#include <aws/rds/RDS_EXPORTS.h>
#include <aws/rds/model/Filter.h>

#include <utility>

namespace Aws {
namespace RDS {
namespace Model {

/**
 */
class DescribeDBClusterAutomatedBackupsRequest : public RDSRequest {
 public:
  AWS_RDS_API DescribeDBClusterAutomatedBackupsRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "DescribeDBClusterAutomatedBackups"; }

  AWS_RDS_API Aws::String SerializePayload() const override;

 protected:
  AWS_RDS_API void DumpBodyToUrl(Aws::Http::URI& uri) const override;

 public:
  ///@{
  /**
   * <p>The resource ID of the DB cluster that is the source of the automated backup.
   * This parameter isn't case-sensitive.</p>
   */
  inline const Aws::String& GetDbClusterResourceId() const { return m_dbClusterResourceId; }
  inline bool DbClusterResourceIdHasBeenSet() const { return m_dbClusterResourceIdHasBeenSet; }
  template <typename DbClusterResourceIdT = Aws::String>
  void SetDbClusterResourceId(DbClusterResourceIdT&& value) {
    m_dbClusterResourceIdHasBeenSet = true;
    m_dbClusterResourceId = std::forward<DbClusterResourceIdT>(value);
  }
  template <typename DbClusterResourceIdT = Aws::String>
  DescribeDBClusterAutomatedBackupsRequest& WithDbClusterResourceId(DbClusterResourceIdT&& value) {
    SetDbClusterResourceId(std::forward<DbClusterResourceIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>(Optional) The user-supplied DB cluster identifier. If this parameter is
   * specified, it must match the identifier of an existing DB cluster. It returns
   * information from the specific DB cluster's automated backup. This parameter
   * isn't case-sensitive.</p>
   */
  inline const Aws::String& GetDBClusterIdentifier() const { return m_dBClusterIdentifier; }
  inline bool DBClusterIdentifierHasBeenSet() const { return m_dBClusterIdentifierHasBeenSet; }
  template <typename DBClusterIdentifierT = Aws::String>
  void SetDBClusterIdentifier(DBClusterIdentifierT&& value) {
    m_dBClusterIdentifierHasBeenSet = true;
    m_dBClusterIdentifier = std::forward<DBClusterIdentifierT>(value);
  }
  template <typename DBClusterIdentifierT = Aws::String>
  DescribeDBClusterAutomatedBackupsRequest& WithDBClusterIdentifier(DBClusterIdentifierT&& value) {
    SetDBClusterIdentifier(std::forward<DBClusterIdentifierT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A filter that specifies which resources to return based on status.</p>
   * <p>Supported filters are the following:</p> <ul> <li> <p> <code>status</code>
   * </p> <ul> <li> <p> <code>retained</code> - Automated backups for deleted
   * clusters and after backup replication is stopped.</p> </li> </ul> </li> <li> <p>
   * <code>db-cluster-id</code> - Accepts DB cluster identifiers and Amazon Resource
   * Names (ARNs). The results list includes only information about the DB cluster
   * automated backups identified by these ARNs.</p> </li> <li> <p>
   * <code>db-cluster-resource-id</code> - Accepts DB resource identifiers and Amazon
   * Resource Names (ARNs). The results list includes only information about the DB
   * cluster resources identified by these ARNs.</p> </li> </ul> <p>Returns all
   * resources by default. The status for each resource is specified in the
   * response.</p>
   */
  inline const Aws::Vector<Filter>& GetFilters() const { return m_filters; }
  inline bool FiltersHasBeenSet() const { return m_filtersHasBeenSet; }
  template <typename FiltersT = Aws::Vector<Filter>>
  void SetFilters(FiltersT&& value) {
    m_filtersHasBeenSet = true;
    m_filters = std::forward<FiltersT>(value);
  }
  template <typename FiltersT = Aws::Vector<Filter>>
  DescribeDBClusterAutomatedBackupsRequest& WithFilters(FiltersT&& value) {
    SetFilters(std::forward<FiltersT>(value));
    return *this;
  }
  template <typename FiltersT = Filter>
  DescribeDBClusterAutomatedBackupsRequest& AddFilters(FiltersT&& value) {
    m_filtersHasBeenSet = true;
    m_filters.emplace_back(std::forward<FiltersT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The maximum number of records to include in the response. If more records
   * exist than the specified <code>MaxRecords</code> value, a pagination token
   * called a marker is included in the response so that you can retrieve the
   * remaining results.</p>
   */
  inline int GetMaxRecords() const { return m_maxRecords; }
  inline bool MaxRecordsHasBeenSet() const { return m_maxRecordsHasBeenSet; }
  inline void SetMaxRecords(int value) {
    m_maxRecordsHasBeenSet = true;
    m_maxRecords = value;
  }
  inline DescribeDBClusterAutomatedBackupsRequest& WithMaxRecords(int value) {
    SetMaxRecords(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The pagination token provided in the previous request. If this parameter is
   * specified the response includes only records beyond the marker, up to
   * <code>MaxRecords</code>.</p>
   */
  inline const Aws::String& GetMarker() const { return m_marker; }
  inline bool MarkerHasBeenSet() const { return m_markerHasBeenSet; }
  template <typename MarkerT = Aws::String>
  void SetMarker(MarkerT&& value) {
    m_markerHasBeenSet = true;
    m_marker = std::forward<MarkerT>(value);
  }
  template <typename MarkerT = Aws::String>
  DescribeDBClusterAutomatedBackupsRequest& WithMarker(MarkerT&& value) {
    SetMarker(std::forward<MarkerT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_dbClusterResourceId;
  bool m_dbClusterResourceIdHasBeenSet = false;

  Aws::String m_dBClusterIdentifier;
  bool m_dBClusterIdentifierHasBeenSet = false;

  Aws::Vector<Filter> m_filters;
  bool m_filtersHasBeenSet = false;

  int m_maxRecords{0};
  bool m_maxRecordsHasBeenSet = false;

  Aws::String m_marker;
  bool m_markerHasBeenSet = false;
};

}  // namespace Model
}  // namespace RDS
}  // namespace Aws
