﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSStreamFwd.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/rds/RDS_EXPORTS.h>

#include <utility>

namespace Aws {
namespace Utils {
namespace Xml {
class XmlNode;
}  // namespace Xml
}  // namespace Utils
namespace RDS {
namespace Model {

/**
 * <p>Information about an Amazon Web Services Identity and Access Management (IAM)
 * role that is associated with a DB instance.</p><p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/rds-2014-10-31/DBInstanceRole">AWS
 * API Reference</a></p>
 */
class DBInstanceRole {
 public:
  AWS_RDS_API DBInstanceRole() = default;
  AWS_RDS_API DBInstanceRole(const Aws::Utils::Xml::XmlNode& xmlNode);
  AWS_RDS_API DBInstanceRole& operator=(const Aws::Utils::Xml::XmlNode& xmlNode);

  AWS_RDS_API void OutputToStream(Aws::OStream& ostream, const char* location, unsigned index, const char* locationValue) const;
  AWS_RDS_API void OutputToStream(Aws::OStream& oStream, const char* location) const;

  ///@{
  /**
   * <p>The Amazon Resource Name (ARN) of the IAM role that is associated with the DB
   * instance.</p>
   */
  inline const Aws::String& GetRoleArn() const { return m_roleArn; }
  inline bool RoleArnHasBeenSet() const { return m_roleArnHasBeenSet; }
  template <typename RoleArnT = Aws::String>
  void SetRoleArn(RoleArnT&& value) {
    m_roleArnHasBeenSet = true;
    m_roleArn = std::forward<RoleArnT>(value);
  }
  template <typename RoleArnT = Aws::String>
  DBInstanceRole& WithRoleArn(RoleArnT&& value) {
    SetRoleArn(std::forward<RoleArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name of the feature associated with the Amazon Web Services Identity and
   * Access Management (IAM) role. For information about supported feature names, see
   * <code>DBEngineVersion</code>.</p>
   */
  inline const Aws::String& GetFeatureName() const { return m_featureName; }
  inline bool FeatureNameHasBeenSet() const { return m_featureNameHasBeenSet; }
  template <typename FeatureNameT = Aws::String>
  void SetFeatureName(FeatureNameT&& value) {
    m_featureNameHasBeenSet = true;
    m_featureName = std::forward<FeatureNameT>(value);
  }
  template <typename FeatureNameT = Aws::String>
  DBInstanceRole& WithFeatureName(FeatureNameT&& value) {
    SetFeatureName(std::forward<FeatureNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Information about the state of association between the IAM role and the DB
   * instance. The Status property returns one of the following values:</p> <ul> <li>
   * <p> <code>ACTIVE</code> - the IAM role ARN is associated with the DB instance
   * and can be used to access other Amazon Web Services services on your behalf.</p>
   * </li> <li> <p> <code>PENDING</code> - the IAM role ARN is being associated with
   * the DB instance.</p> </li> <li> <p> <code>INVALID</code> - the IAM role ARN is
   * associated with the DB instance, but the DB instance is unable to assume the IAM
   * role in order to access other Amazon Web Services services on your behalf.</p>
   * </li> </ul>
   */
  inline const Aws::String& GetStatus() const { return m_status; }
  inline bool StatusHasBeenSet() const { return m_statusHasBeenSet; }
  template <typename StatusT = Aws::String>
  void SetStatus(StatusT&& value) {
    m_statusHasBeenSet = true;
    m_status = std::forward<StatusT>(value);
  }
  template <typename StatusT = Aws::String>
  DBInstanceRole& WithStatus(StatusT&& value) {
    SetStatus(std::forward<StatusT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_roleArn;
  bool m_roleArnHasBeenSet = false;

  Aws::String m_featureName;
  bool m_featureNameHasBeenSet = false;

  Aws::String m_status;
  bool m_statusHasBeenSet = false;
};

}  // namespace Model
}  // namespace RDS
}  // namespace Aws
