﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/rds/RDSRequest.h>
#include <aws/rds/RDS_EXPORTS.h>
#include <aws/rds/model/Tag.h>

#include <utility>

namespace Aws {
namespace RDS {
namespace Model {

/**
 * <p/><p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/rds-2014-10-31/CreateDBClusterSnapshotMessage">AWS
 * API Reference</a></p>
 */
class CreateDBClusterSnapshotRequest : public RDSRequest {
 public:
  AWS_RDS_API CreateDBClusterSnapshotRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "CreateDBClusterSnapshot"; }

  AWS_RDS_API Aws::String SerializePayload() const override;

 protected:
  AWS_RDS_API void DumpBodyToUrl(Aws::Http::URI& uri) const override;

 public:
  ///@{
  /**
   * <p>The identifier of the DB cluster snapshot. This parameter is stored as a
   * lowercase string.</p> <p>Constraints:</p> <ul> <li> <p>Must contain from 1 to 63
   * letters, numbers, or hyphens.</p> </li> <li> <p>First character must be a
   * letter.</p> </li> <li> <p>Can't end with a hyphen or contain two consecutive
   * hyphens.</p> </li> </ul> <p>Example: <code>my-cluster1-snapshot1</code> </p>
   */
  inline const Aws::String& GetDBClusterSnapshotIdentifier() const { return m_dBClusterSnapshotIdentifier; }
  inline bool DBClusterSnapshotIdentifierHasBeenSet() const { return m_dBClusterSnapshotIdentifierHasBeenSet; }
  template <typename DBClusterSnapshotIdentifierT = Aws::String>
  void SetDBClusterSnapshotIdentifier(DBClusterSnapshotIdentifierT&& value) {
    m_dBClusterSnapshotIdentifierHasBeenSet = true;
    m_dBClusterSnapshotIdentifier = std::forward<DBClusterSnapshotIdentifierT>(value);
  }
  template <typename DBClusterSnapshotIdentifierT = Aws::String>
  CreateDBClusterSnapshotRequest& WithDBClusterSnapshotIdentifier(DBClusterSnapshotIdentifierT&& value) {
    SetDBClusterSnapshotIdentifier(std::forward<DBClusterSnapshotIdentifierT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The identifier of the DB cluster to create a snapshot for. This parameter
   * isn't case-sensitive.</p> <p>Constraints:</p> <ul> <li> <p>Must match the
   * identifier of an existing DBCluster.</p> </li> </ul> <p>Example:
   * <code>my-cluster1</code> </p>
   */
  inline const Aws::String& GetDBClusterIdentifier() const { return m_dBClusterIdentifier; }
  inline bool DBClusterIdentifierHasBeenSet() const { return m_dBClusterIdentifierHasBeenSet; }
  template <typename DBClusterIdentifierT = Aws::String>
  void SetDBClusterIdentifier(DBClusterIdentifierT&& value) {
    m_dBClusterIdentifierHasBeenSet = true;
    m_dBClusterIdentifier = std::forward<DBClusterIdentifierT>(value);
  }
  template <typename DBClusterIdentifierT = Aws::String>
  CreateDBClusterSnapshotRequest& WithDBClusterIdentifier(DBClusterIdentifierT&& value) {
    SetDBClusterIdentifier(std::forward<DBClusterIdentifierT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The tags to be assigned to the DB cluster snapshot.</p>
   */
  inline const Aws::Vector<Tag>& GetTags() const { return m_tags; }
  inline bool TagsHasBeenSet() const { return m_tagsHasBeenSet; }
  template <typename TagsT = Aws::Vector<Tag>>
  void SetTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags = std::forward<TagsT>(value);
  }
  template <typename TagsT = Aws::Vector<Tag>>
  CreateDBClusterSnapshotRequest& WithTags(TagsT&& value) {
    SetTags(std::forward<TagsT>(value));
    return *this;
  }
  template <typename TagsT = Tag>
  CreateDBClusterSnapshotRequest& AddTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags.emplace_back(std::forward<TagsT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_dBClusterSnapshotIdentifier;
  bool m_dBClusterSnapshotIdentifierHasBeenSet = false;

  Aws::String m_dBClusterIdentifier;
  bool m_dBClusterIdentifierHasBeenSet = false;

  Aws::Vector<Tag> m_tags;
  bool m_tagsHasBeenSet = false;
};

}  // namespace Model
}  // namespace RDS
}  // namespace Aws
