﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/logs/CloudWatchLogs_EXPORTS.h>
#include <aws/logs/model/PolicyType.h>
#include <aws/logs/model/Scope.h>

#include <utility>

namespace Aws {
namespace Utils {
namespace Json {
class JsonValue;
class JsonView;
}  // namespace Json
}  // namespace Utils
namespace CloudWatchLogs {
namespace Model {

/**
 * <p>A structure that contains information about one CloudWatch Logs account
 * policy.</p><p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/logs-2014-03-28/AccountPolicy">AWS
 * API Reference</a></p>
 */
class AccountPolicy {
 public:
  AWS_CLOUDWATCHLOGS_API AccountPolicy() = default;
  AWS_CLOUDWATCHLOGS_API AccountPolicy(Aws::Utils::Json::JsonView jsonValue);
  AWS_CLOUDWATCHLOGS_API AccountPolicy& operator=(Aws::Utils::Json::JsonView jsonValue);
  AWS_CLOUDWATCHLOGS_API Aws::Utils::Json::JsonValue Jsonize() const;

  ///@{
  /**
   * <p>The name of the account policy.</p>
   */
  inline const Aws::String& GetPolicyName() const { return m_policyName; }
  inline bool PolicyNameHasBeenSet() const { return m_policyNameHasBeenSet; }
  template <typename PolicyNameT = Aws::String>
  void SetPolicyName(PolicyNameT&& value) {
    m_policyNameHasBeenSet = true;
    m_policyName = std::forward<PolicyNameT>(value);
  }
  template <typename PolicyNameT = Aws::String>
  AccountPolicy& WithPolicyName(PolicyNameT&& value) {
    SetPolicyName(std::forward<PolicyNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The policy document for this account policy.</p> <p>The JSON specified in
   * <code>policyDocument</code> can be up to 30,720 characters.</p>
   */
  inline const Aws::String& GetPolicyDocument() const { return m_policyDocument; }
  inline bool PolicyDocumentHasBeenSet() const { return m_policyDocumentHasBeenSet; }
  template <typename PolicyDocumentT = Aws::String>
  void SetPolicyDocument(PolicyDocumentT&& value) {
    m_policyDocumentHasBeenSet = true;
    m_policyDocument = std::forward<PolicyDocumentT>(value);
  }
  template <typename PolicyDocumentT = Aws::String>
  AccountPolicy& WithPolicyDocument(PolicyDocumentT&& value) {
    SetPolicyDocument(std::forward<PolicyDocumentT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The date and time that this policy was most recently updated.</p>
   */
  inline long long GetLastUpdatedTime() const { return m_lastUpdatedTime; }
  inline bool LastUpdatedTimeHasBeenSet() const { return m_lastUpdatedTimeHasBeenSet; }
  inline void SetLastUpdatedTime(long long value) {
    m_lastUpdatedTimeHasBeenSet = true;
    m_lastUpdatedTime = value;
  }
  inline AccountPolicy& WithLastUpdatedTime(long long value) {
    SetLastUpdatedTime(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The type of policy for this account policy.</p>
   */
  inline PolicyType GetPolicyType() const { return m_policyType; }
  inline bool PolicyTypeHasBeenSet() const { return m_policyTypeHasBeenSet; }
  inline void SetPolicyType(PolicyType value) {
    m_policyTypeHasBeenSet = true;
    m_policyType = value;
  }
  inline AccountPolicy& WithPolicyType(PolicyType value) {
    SetPolicyType(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The scope of the account policy.</p>
   */
  inline Scope GetScope() const { return m_scope; }
  inline bool ScopeHasBeenSet() const { return m_scopeHasBeenSet; }
  inline void SetScope(Scope value) {
    m_scopeHasBeenSet = true;
    m_scope = value;
  }
  inline AccountPolicy& WithScope(Scope value) {
    SetScope(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The log group selection criteria that is used for this policy.</p>
   */
  inline const Aws::String& GetSelectionCriteria() const { return m_selectionCriteria; }
  inline bool SelectionCriteriaHasBeenSet() const { return m_selectionCriteriaHasBeenSet; }
  template <typename SelectionCriteriaT = Aws::String>
  void SetSelectionCriteria(SelectionCriteriaT&& value) {
    m_selectionCriteriaHasBeenSet = true;
    m_selectionCriteria = std::forward<SelectionCriteriaT>(value);
  }
  template <typename SelectionCriteriaT = Aws::String>
  AccountPolicy& WithSelectionCriteria(SelectionCriteriaT&& value) {
    SetSelectionCriteria(std::forward<SelectionCriteriaT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The Amazon Web Services account ID that the policy applies to.</p>
   */
  inline const Aws::String& GetAccountId() const { return m_accountId; }
  inline bool AccountIdHasBeenSet() const { return m_accountIdHasBeenSet; }
  template <typename AccountIdT = Aws::String>
  void SetAccountId(AccountIdT&& value) {
    m_accountIdHasBeenSet = true;
    m_accountId = std::forward<AccountIdT>(value);
  }
  template <typename AccountIdT = Aws::String>
  AccountPolicy& WithAccountId(AccountIdT&& value) {
    SetAccountId(std::forward<AccountIdT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_policyName;
  bool m_policyNameHasBeenSet = false;

  Aws::String m_policyDocument;
  bool m_policyDocumentHasBeenSet = false;

  long long m_lastUpdatedTime{0};
  bool m_lastUpdatedTimeHasBeenSet = false;

  PolicyType m_policyType{PolicyType::NOT_SET};
  bool m_policyTypeHasBeenSet = false;

  Scope m_scope{Scope::NOT_SET};
  bool m_scopeHasBeenSet = false;

  Aws::String m_selectionCriteria;
  bool m_selectionCriteriaHasBeenSet = false;

  Aws::String m_accountId;
  bool m_accountIdHasBeenSet = false;
};

}  // namespace Model
}  // namespace CloudWatchLogs
}  // namespace Aws
