﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/kendra/KendraRequest.h>
#include <aws/kendra/Kendra_EXPORTS.h>
#include <aws/kendra/model/CustomDocumentEnrichmentConfiguration.h>
#include <aws/kendra/model/Document.h>

#include <utility>

namespace Aws {
namespace kendra {
namespace Model {

/**
 */
class BatchPutDocumentRequest : public KendraRequest {
 public:
  AWS_KENDRA_API BatchPutDocumentRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "BatchPutDocument"; }

  AWS_KENDRA_API Aws::String SerializePayload() const override;

  AWS_KENDRA_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The identifier of the index to add the documents to. You need to create the
   * index first using the <code>CreateIndex</code> API.</p>
   */
  inline const Aws::String& GetIndexId() const { return m_indexId; }
  inline bool IndexIdHasBeenSet() const { return m_indexIdHasBeenSet; }
  template <typename IndexIdT = Aws::String>
  void SetIndexId(IndexIdT&& value) {
    m_indexIdHasBeenSet = true;
    m_indexId = std::forward<IndexIdT>(value);
  }
  template <typename IndexIdT = Aws::String>
  BatchPutDocumentRequest& WithIndexId(IndexIdT&& value) {
    SetIndexId(std::forward<IndexIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The Amazon Resource Name (ARN) of an IAM role with permission to access your
   * S3 bucket. For more information, see <a
   * href="https://docs.aws.amazon.com/kendra/latest/dg/iam-roles.html">IAM access
   * roles for Amazon Kendra</a>.</p>
   */
  inline const Aws::String& GetRoleArn() const { return m_roleArn; }
  inline bool RoleArnHasBeenSet() const { return m_roleArnHasBeenSet; }
  template <typename RoleArnT = Aws::String>
  void SetRoleArn(RoleArnT&& value) {
    m_roleArnHasBeenSet = true;
    m_roleArn = std::forward<RoleArnT>(value);
  }
  template <typename RoleArnT = Aws::String>
  BatchPutDocumentRequest& WithRoleArn(RoleArnT&& value) {
    SetRoleArn(std::forward<RoleArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>One or more documents to add to the index.</p> <p>Documents have the
   * following file size limits.</p> <ul> <li> <p>50 MB total size for any file</p>
   * </li> <li> <p>5 MB extracted text for any file</p> </li> </ul> <p>For more
   * information, see <a
   * href="https://docs.aws.amazon.com/kendra/latest/dg/quotas.html">Quotas</a>.</p>
   */
  inline const Aws::Vector<Document>& GetDocuments() const { return m_documents; }
  inline bool DocumentsHasBeenSet() const { return m_documentsHasBeenSet; }
  template <typename DocumentsT = Aws::Vector<Document>>
  void SetDocuments(DocumentsT&& value) {
    m_documentsHasBeenSet = true;
    m_documents = std::forward<DocumentsT>(value);
  }
  template <typename DocumentsT = Aws::Vector<Document>>
  BatchPutDocumentRequest& WithDocuments(DocumentsT&& value) {
    SetDocuments(std::forward<DocumentsT>(value));
    return *this;
  }
  template <typename DocumentsT = Document>
  BatchPutDocumentRequest& AddDocuments(DocumentsT&& value) {
    m_documentsHasBeenSet = true;
    m_documents.emplace_back(std::forward<DocumentsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Configuration information for altering your document metadata and content
   * during the document ingestion process when you use the
   * <code>BatchPutDocument</code> API.</p> <p>For more information on how to create,
   * modify and delete document metadata, or make other content alterations when you
   * ingest documents into Amazon Kendra, see <a
   * href="https://docs.aws.amazon.com/kendra/latest/dg/custom-document-enrichment.html">Customizing
   * document metadata during the ingestion process</a>.</p>
   */
  inline const CustomDocumentEnrichmentConfiguration& GetCustomDocumentEnrichmentConfiguration() const {
    return m_customDocumentEnrichmentConfiguration;
  }
  inline bool CustomDocumentEnrichmentConfigurationHasBeenSet() const { return m_customDocumentEnrichmentConfigurationHasBeenSet; }
  template <typename CustomDocumentEnrichmentConfigurationT = CustomDocumentEnrichmentConfiguration>
  void SetCustomDocumentEnrichmentConfiguration(CustomDocumentEnrichmentConfigurationT&& value) {
    m_customDocumentEnrichmentConfigurationHasBeenSet = true;
    m_customDocumentEnrichmentConfiguration = std::forward<CustomDocumentEnrichmentConfigurationT>(value);
  }
  template <typename CustomDocumentEnrichmentConfigurationT = CustomDocumentEnrichmentConfiguration>
  BatchPutDocumentRequest& WithCustomDocumentEnrichmentConfiguration(CustomDocumentEnrichmentConfigurationT&& value) {
    SetCustomDocumentEnrichmentConfiguration(std::forward<CustomDocumentEnrichmentConfigurationT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_indexId;
  bool m_indexIdHasBeenSet = false;

  Aws::String m_roleArn;
  bool m_roleArnHasBeenSet = false;

  Aws::Vector<Document> m_documents;
  bool m_documentsHasBeenSet = false;

  CustomDocumentEnrichmentConfiguration m_customDocumentEnrichmentConfiguration;
  bool m_customDocumentEnrichmentConfigurationHasBeenSet = false;
};

}  // namespace Model
}  // namespace kendra
}  // namespace Aws
