﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSMap.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/ivs/IVS_EXPORTS.h>

#include <utility>

namespace Aws {
namespace Utils {
namespace Json {
class JsonValue;
class JsonView;
}  // namespace Json
}  // namespace Utils
namespace IVS {
namespace Model {

/**
 * <p>An object representing a policy to constrain playback by country and/or
 * origin sites.</p><p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/ivs-2020-07-14/PlaybackRestrictionPolicy">AWS
 * API Reference</a></p>
 */
class PlaybackRestrictionPolicy {
 public:
  AWS_IVS_API PlaybackRestrictionPolicy() = default;
  AWS_IVS_API PlaybackRestrictionPolicy(Aws::Utils::Json::JsonView jsonValue);
  AWS_IVS_API PlaybackRestrictionPolicy& operator=(Aws::Utils::Json::JsonView jsonValue);
  AWS_IVS_API Aws::Utils::Json::JsonValue Jsonize() const;

  ///@{
  /**
   * <p>A list of country codes that control geoblocking restriction. Allowed values
   * are the officially assigned <a
   * href="https://en.wikipedia.org/wiki/ISO_3166-1_alpha-2">ISO 3166-1 alpha-2</a>
   * codes. Default: All countries (an empty array).</p>
   */
  inline const Aws::Vector<Aws::String>& GetAllowedCountries() const { return m_allowedCountries; }
  inline bool AllowedCountriesHasBeenSet() const { return m_allowedCountriesHasBeenSet; }
  template <typename AllowedCountriesT = Aws::Vector<Aws::String>>
  void SetAllowedCountries(AllowedCountriesT&& value) {
    m_allowedCountriesHasBeenSet = true;
    m_allowedCountries = std::forward<AllowedCountriesT>(value);
  }
  template <typename AllowedCountriesT = Aws::Vector<Aws::String>>
  PlaybackRestrictionPolicy& WithAllowedCountries(AllowedCountriesT&& value) {
    SetAllowedCountries(std::forward<AllowedCountriesT>(value));
    return *this;
  }
  template <typename AllowedCountriesT = Aws::String>
  PlaybackRestrictionPolicy& AddAllowedCountries(AllowedCountriesT&& value) {
    m_allowedCountriesHasBeenSet = true;
    m_allowedCountries.emplace_back(std::forward<AllowedCountriesT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A list of origin sites that control CORS restriction. Allowed values are the
   * same as valid values of the Origin header defined at <a
   * href="https://developer.mozilla.org/en-US/docs/Web/HTTP/Headers/Origin">https://developer.mozilla.org/en-US/docs/Web/HTTP/Headers/Origin</a>.
   * Default: All origins (an empty array).</p>
   */
  inline const Aws::Vector<Aws::String>& GetAllowedOrigins() const { return m_allowedOrigins; }
  inline bool AllowedOriginsHasBeenSet() const { return m_allowedOriginsHasBeenSet; }
  template <typename AllowedOriginsT = Aws::Vector<Aws::String>>
  void SetAllowedOrigins(AllowedOriginsT&& value) {
    m_allowedOriginsHasBeenSet = true;
    m_allowedOrigins = std::forward<AllowedOriginsT>(value);
  }
  template <typename AllowedOriginsT = Aws::Vector<Aws::String>>
  PlaybackRestrictionPolicy& WithAllowedOrigins(AllowedOriginsT&& value) {
    SetAllowedOrigins(std::forward<AllowedOriginsT>(value));
    return *this;
  }
  template <typename AllowedOriginsT = Aws::String>
  PlaybackRestrictionPolicy& AddAllowedOrigins(AllowedOriginsT&& value) {
    m_allowedOriginsHasBeenSet = true;
    m_allowedOrigins.emplace_back(std::forward<AllowedOriginsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Playback-restriction-policy ARN</p>
   */
  inline const Aws::String& GetArn() const { return m_arn; }
  inline bool ArnHasBeenSet() const { return m_arnHasBeenSet; }
  template <typename ArnT = Aws::String>
  void SetArn(ArnT&& value) {
    m_arnHasBeenSet = true;
    m_arn = std::forward<ArnT>(value);
  }
  template <typename ArnT = Aws::String>
  PlaybackRestrictionPolicy& WithArn(ArnT&& value) {
    SetArn(std::forward<ArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Whether channel playback is constrained by origin site. Default:
   * <code>false</code>.</p>
   */
  inline bool GetEnableStrictOriginEnforcement() const { return m_enableStrictOriginEnforcement; }
  inline bool EnableStrictOriginEnforcementHasBeenSet() const { return m_enableStrictOriginEnforcementHasBeenSet; }
  inline void SetEnableStrictOriginEnforcement(bool value) {
    m_enableStrictOriginEnforcementHasBeenSet = true;
    m_enableStrictOriginEnforcement = value;
  }
  inline PlaybackRestrictionPolicy& WithEnableStrictOriginEnforcement(bool value) {
    SetEnableStrictOriginEnforcement(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Playback-restriction-policy name. The value does not need to be unique.</p>
   */
  inline const Aws::String& GetName() const { return m_name; }
  inline bool NameHasBeenSet() const { return m_nameHasBeenSet; }
  template <typename NameT = Aws::String>
  void SetName(NameT&& value) {
    m_nameHasBeenSet = true;
    m_name = std::forward<NameT>(value);
  }
  template <typename NameT = Aws::String>
  PlaybackRestrictionPolicy& WithName(NameT&& value) {
    SetName(std::forward<NameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Tags attached to the resource. Array of 1-50 maps, each of the form
   * <code>string:string (key:value)</code>. See <a
   * href="https://docs.aws.amazon.com/tag-editor/latest/userguide/best-practices-and-strats.html">Best
   * practices and strategies</a> in <i>Tagging Amazon Web Services Resources and Tag
   * Editor</i> for details, including restrictions that apply to tags and "Tag
   * naming limits and requirements"; Amazon IVS has no service-specific constraints
   * beyond what is documented there.</p>
   */
  inline const Aws::Map<Aws::String, Aws::String>& GetTags() const { return m_tags; }
  inline bool TagsHasBeenSet() const { return m_tagsHasBeenSet; }
  template <typename TagsT = Aws::Map<Aws::String, Aws::String>>
  void SetTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags = std::forward<TagsT>(value);
  }
  template <typename TagsT = Aws::Map<Aws::String, Aws::String>>
  PlaybackRestrictionPolicy& WithTags(TagsT&& value) {
    SetTags(std::forward<TagsT>(value));
    return *this;
  }
  template <typename TagsKeyT = Aws::String, typename TagsValueT = Aws::String>
  PlaybackRestrictionPolicy& AddTags(TagsKeyT&& key, TagsValueT&& value) {
    m_tagsHasBeenSet = true;
    m_tags.emplace(std::forward<TagsKeyT>(key), std::forward<TagsValueT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::Vector<Aws::String> m_allowedCountries;
  bool m_allowedCountriesHasBeenSet = false;

  Aws::Vector<Aws::String> m_allowedOrigins;
  bool m_allowedOriginsHasBeenSet = false;

  Aws::String m_arn;
  bool m_arnHasBeenSet = false;

  bool m_enableStrictOriginEnforcement{false};
  bool m_enableStrictOriginEnforcementHasBeenSet = false;

  Aws::String m_name;
  bool m_nameHasBeenSet = false;

  Aws::Map<Aws::String, Aws::String> m_tags;
  bool m_tagsHasBeenSet = false;
};

}  // namespace Model
}  // namespace IVS
}  // namespace Aws
