﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/iot/IoTRequest.h>
#include <aws/iot/IoT_EXPORTS.h>

#include <utility>

namespace Aws {
namespace Http {
class URI;
}  // namespace Http
namespace IoT {
namespace Model {

/**
 */
class DeleteSecurityProfileRequest : public IoTRequest {
 public:
  AWS_IOT_API DeleteSecurityProfileRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "DeleteSecurityProfile"; }

  AWS_IOT_API Aws::String SerializePayload() const override;

  AWS_IOT_API void AddQueryStringParameters(Aws::Http::URI& uri) const override;

  ///@{
  /**
   * <p>The name of the security profile to be deleted.</p>
   */
  inline const Aws::String& GetSecurityProfileName() const { return m_securityProfileName; }
  inline bool SecurityProfileNameHasBeenSet() const { return m_securityProfileNameHasBeenSet; }
  template <typename SecurityProfileNameT = Aws::String>
  void SetSecurityProfileName(SecurityProfileNameT&& value) {
    m_securityProfileNameHasBeenSet = true;
    m_securityProfileName = std::forward<SecurityProfileNameT>(value);
  }
  template <typename SecurityProfileNameT = Aws::String>
  DeleteSecurityProfileRequest& WithSecurityProfileName(SecurityProfileNameT&& value) {
    SetSecurityProfileName(std::forward<SecurityProfileNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The expected version of the security profile. A new version is generated
   * whenever the security profile is updated. If you specify a value that is
   * different from the actual version, a <code>VersionConflictException</code> is
   * thrown.</p>
   */
  inline long long GetExpectedVersion() const { return m_expectedVersion; }
  inline bool ExpectedVersionHasBeenSet() const { return m_expectedVersionHasBeenSet; }
  inline void SetExpectedVersion(long long value) {
    m_expectedVersionHasBeenSet = true;
    m_expectedVersion = value;
  }
  inline DeleteSecurityProfileRequest& WithExpectedVersion(long long value) {
    SetExpectedVersion(value);
    return *this;
  }
  ///@}
 private:
  Aws::String m_securityProfileName;
  bool m_securityProfileNameHasBeenSet = false;

  long long m_expectedVersion{0};
  bool m_expectedVersionHasBeenSet = false;
};

}  // namespace Model
}  // namespace IoT
}  // namespace Aws
