﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/iot/IoTRequest.h>
#include <aws/iot/IoT_EXPORTS.h>
#include <aws/iot/model/MitigationActionParams.h>
#include <aws/iot/model/Tag.h>

#include <utility>

namespace Aws {
namespace IoT {
namespace Model {

/**
 */
class CreateMitigationActionRequest : public IoTRequest {
 public:
  AWS_IOT_API CreateMitigationActionRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "CreateMitigationAction"; }

  AWS_IOT_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p>A friendly name for the action. Choose a friendly name that accurately
   * describes the action (for example, <code>EnableLoggingAction</code>).</p>
   */
  inline const Aws::String& GetActionName() const { return m_actionName; }
  inline bool ActionNameHasBeenSet() const { return m_actionNameHasBeenSet; }
  template <typename ActionNameT = Aws::String>
  void SetActionName(ActionNameT&& value) {
    m_actionNameHasBeenSet = true;
    m_actionName = std::forward<ActionNameT>(value);
  }
  template <typename ActionNameT = Aws::String>
  CreateMitigationActionRequest& WithActionName(ActionNameT&& value) {
    SetActionName(std::forward<ActionNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The ARN of the IAM role that is used to apply the mitigation action.</p>
   */
  inline const Aws::String& GetRoleArn() const { return m_roleArn; }
  inline bool RoleArnHasBeenSet() const { return m_roleArnHasBeenSet; }
  template <typename RoleArnT = Aws::String>
  void SetRoleArn(RoleArnT&& value) {
    m_roleArnHasBeenSet = true;
    m_roleArn = std::forward<RoleArnT>(value);
  }
  template <typename RoleArnT = Aws::String>
  CreateMitigationActionRequest& WithRoleArn(RoleArnT&& value) {
    SetRoleArn(std::forward<RoleArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Defines the type of action and the parameters for that action.</p>
   */
  inline const MitigationActionParams& GetActionParams() const { return m_actionParams; }
  inline bool ActionParamsHasBeenSet() const { return m_actionParamsHasBeenSet; }
  template <typename ActionParamsT = MitigationActionParams>
  void SetActionParams(ActionParamsT&& value) {
    m_actionParamsHasBeenSet = true;
    m_actionParams = std::forward<ActionParamsT>(value);
  }
  template <typename ActionParamsT = MitigationActionParams>
  CreateMitigationActionRequest& WithActionParams(ActionParamsT&& value) {
    SetActionParams(std::forward<ActionParamsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Metadata that can be used to manage the mitigation action.</p>
   */
  inline const Aws::Vector<Tag>& GetTags() const { return m_tags; }
  inline bool TagsHasBeenSet() const { return m_tagsHasBeenSet; }
  template <typename TagsT = Aws::Vector<Tag>>
  void SetTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags = std::forward<TagsT>(value);
  }
  template <typename TagsT = Aws::Vector<Tag>>
  CreateMitigationActionRequest& WithTags(TagsT&& value) {
    SetTags(std::forward<TagsT>(value));
    return *this;
  }
  template <typename TagsT = Tag>
  CreateMitigationActionRequest& AddTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags.emplace_back(std::forward<TagsT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_actionName;
  bool m_actionNameHasBeenSet = false;

  Aws::String m_roleArn;
  bool m_roleArnHasBeenSet = false;

  MitigationActionParams m_actionParams;
  bool m_actionParamsHasBeenSet = false;

  Aws::Vector<Tag> m_tags;
  bool m_tagsHasBeenSet = false;
};

}  // namespace Model
}  // namespace IoT
}  // namespace Aws
