﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/gamelift/GameLiftRequest.h>
#include <aws/gamelift/GameLift_EXPORTS.h>
#include <aws/gamelift/model/ScalingStatusType.h>

#include <utility>

namespace Aws {
namespace GameLift {
namespace Model {

/**
 */
class DescribeScalingPoliciesRequest : public GameLiftRequest {
 public:
  AWS_GAMELIFT_API DescribeScalingPoliciesRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "DescribeScalingPolicies"; }

  AWS_GAMELIFT_API Aws::String SerializePayload() const override;

  AWS_GAMELIFT_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>A unique identifier for the fleet for which to retrieve scaling policies. You
   * can use either the fleet ID or ARN value.</p>
   */
  inline const Aws::String& GetFleetId() const { return m_fleetId; }
  inline bool FleetIdHasBeenSet() const { return m_fleetIdHasBeenSet; }
  template <typename FleetIdT = Aws::String>
  void SetFleetId(FleetIdT&& value) {
    m_fleetIdHasBeenSet = true;
    m_fleetId = std::forward<FleetIdT>(value);
  }
  template <typename FleetIdT = Aws::String>
  DescribeScalingPoliciesRequest& WithFleetId(FleetIdT&& value) {
    SetFleetId(std::forward<FleetIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Scaling policy status to filter results on. A scaling policy is only in force
   * when in an <code>ACTIVE</code> status.</p> <ul> <li> <p> <b>ACTIVE</b> -- The
   * scaling policy is currently in force.</p> </li> <li> <p> <b>UPDATEREQUESTED</b>
   * -- A request to update the scaling policy has been received.</p> </li> <li> <p>
   * <b>UPDATING</b> -- A change is being made to the scaling policy.</p> </li> <li>
   * <p> <b>DELETEREQUESTED</b> -- A request to delete the scaling policy has been
   * received.</p> </li> <li> <p> <b>DELETING</b> -- The scaling policy is being
   * deleted.</p> </li> <li> <p> <b>DELETED</b> -- The scaling policy has been
   * deleted.</p> </li> <li> <p> <b>ERROR</b> -- An error occurred in creating the
   * policy. It should be removed and recreated.</p> </li> </ul>
   */
  inline ScalingStatusType GetStatusFilter() const { return m_statusFilter; }
  inline bool StatusFilterHasBeenSet() const { return m_statusFilterHasBeenSet; }
  inline void SetStatusFilter(ScalingStatusType value) {
    m_statusFilterHasBeenSet = true;
    m_statusFilter = value;
  }
  inline DescribeScalingPoliciesRequest& WithStatusFilter(ScalingStatusType value) {
    SetStatusFilter(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The maximum number of results to return. Use this parameter with
   * <code>NextToken</code> to get results as a set of sequential pages.</p>
   */
  inline int GetLimit() const { return m_limit; }
  inline bool LimitHasBeenSet() const { return m_limitHasBeenSet; }
  inline void SetLimit(int value) {
    m_limitHasBeenSet = true;
    m_limit = value;
  }
  inline DescribeScalingPoliciesRequest& WithLimit(int value) {
    SetLimit(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A token that indicates the start of the next sequential page of results. Use
   * the token that is returned with a previous call to this operation. To start at
   * the beginning of the result set, do not specify a value.</p>
   */
  inline const Aws::String& GetNextToken() const { return m_nextToken; }
  inline bool NextTokenHasBeenSet() const { return m_nextTokenHasBeenSet; }
  template <typename NextTokenT = Aws::String>
  void SetNextToken(NextTokenT&& value) {
    m_nextTokenHasBeenSet = true;
    m_nextToken = std::forward<NextTokenT>(value);
  }
  template <typename NextTokenT = Aws::String>
  DescribeScalingPoliciesRequest& WithNextToken(NextTokenT&& value) {
    SetNextToken(std::forward<NextTokenT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p> The fleet location. If you don't specify this value, the response contains
   * the scaling policies of every location in the fleet. </p>
   */
  inline const Aws::String& GetLocation() const { return m_location; }
  inline bool LocationHasBeenSet() const { return m_locationHasBeenSet; }
  template <typename LocationT = Aws::String>
  void SetLocation(LocationT&& value) {
    m_locationHasBeenSet = true;
    m_location = std::forward<LocationT>(value);
  }
  template <typename LocationT = Aws::String>
  DescribeScalingPoliciesRequest& WithLocation(LocationT&& value) {
    SetLocation(std::forward<LocationT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_fleetId;
  bool m_fleetIdHasBeenSet = false;

  ScalingStatusType m_statusFilter{ScalingStatusType::NOT_SET};
  bool m_statusFilterHasBeenSet = false;

  int m_limit{0};
  bool m_limitHasBeenSet = false;

  Aws::String m_nextToken;
  bool m_nextTokenHasBeenSet = false;

  Aws::String m_location;
  bool m_locationHasBeenSet = false;
};

}  // namespace Model
}  // namespace GameLift
}  // namespace Aws
