﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/finspace/FinspaceRequest.h>
#include <aws/finspace/Finspace_EXPORTS.h>
#include <aws/finspace/model/KxClusterType.h>

#include <utility>

namespace Aws {
namespace Http {
class URI;
}  // namespace Http
namespace finspace {
namespace Model {

/**
 */
class ListKxClustersRequest : public FinspaceRequest {
 public:
  AWS_FINSPACE_API ListKxClustersRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "ListKxClusters"; }

  AWS_FINSPACE_API Aws::String SerializePayload() const override;

  AWS_FINSPACE_API void AddQueryStringParameters(Aws::Http::URI& uri) const override;

  ///@{
  /**
   * <p>A unique identifier for the kdb environment.</p>
   */
  inline const Aws::String& GetEnvironmentId() const { return m_environmentId; }
  inline bool EnvironmentIdHasBeenSet() const { return m_environmentIdHasBeenSet; }
  template <typename EnvironmentIdT = Aws::String>
  void SetEnvironmentId(EnvironmentIdT&& value) {
    m_environmentIdHasBeenSet = true;
    m_environmentId = std::forward<EnvironmentIdT>(value);
  }
  template <typename EnvironmentIdT = Aws::String>
  ListKxClustersRequest& WithEnvironmentId(EnvironmentIdT&& value) {
    SetEnvironmentId(std::forward<EnvironmentIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Specifies the type of KDB database that is being created. The following types
   * are available: </p> <ul> <li> <p>HDB – A Historical Database. The data is only
   * accessible with read-only permissions from one of the FinSpace managed kdb
   * databases mounted to the cluster.</p> </li> <li> <p>RDB – A Realtime Database.
   * This type of database captures all the data from a ticker plant and stores it in
   * memory until the end of day, after which it writes all of its data to a disk and
   * reloads the HDB. This cluster type requires local storage for temporary storage
   * of data during the savedown process. If you specify this field in your request,
   * you must provide the <code>savedownStorageConfiguration</code> parameter.</p>
   * </li> <li> <p>GATEWAY – A gateway cluster allows you to access data across
   * processes in kdb systems. It allows you to create your own routing logic using
   * the initialization scripts and custom code. This type of cluster does not
   * require a writable local storage.</p> </li> <li> <p>GP – A general purpose
   * cluster allows you to quickly iterate on code during development by granting
   * greater access to system commands and enabling a fast reload of custom code.
   * This cluster type can optionally mount databases including cache and savedown
   * storage. For this cluster type, the node count is fixed at 1. It does not
   * support autoscaling and supports only <code>SINGLE</code> AZ mode.</p> </li>
   * <li> <p>Tickerplant – A tickerplant cluster allows you to subscribe to feed
   * handlers based on IAM permissions. It can publish to RDBs, other Tickerplants,
   * and real-time subscribers (RTS). Tickerplants can persist messages to log, which
   * is readable by any RDB environment. It supports only single-node that is only
   * one kdb process.</p> </li> </ul>
   */
  inline KxClusterType GetClusterType() const { return m_clusterType; }
  inline bool ClusterTypeHasBeenSet() const { return m_clusterTypeHasBeenSet; }
  inline void SetClusterType(KxClusterType value) {
    m_clusterTypeHasBeenSet = true;
    m_clusterType = value;
  }
  inline ListKxClustersRequest& WithClusterType(KxClusterType value) {
    SetClusterType(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The maximum number of results to return in this request.</p>
   */
  inline int GetMaxResults() const { return m_maxResults; }
  inline bool MaxResultsHasBeenSet() const { return m_maxResultsHasBeenSet; }
  inline void SetMaxResults(int value) {
    m_maxResultsHasBeenSet = true;
    m_maxResults = value;
  }
  inline ListKxClustersRequest& WithMaxResults(int value) {
    SetMaxResults(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A token that indicates where a results page should begin.</p>
   */
  inline const Aws::String& GetNextToken() const { return m_nextToken; }
  inline bool NextTokenHasBeenSet() const { return m_nextTokenHasBeenSet; }
  template <typename NextTokenT = Aws::String>
  void SetNextToken(NextTokenT&& value) {
    m_nextTokenHasBeenSet = true;
    m_nextToken = std::forward<NextTokenT>(value);
  }
  template <typename NextTokenT = Aws::String>
  ListKxClustersRequest& WithNextToken(NextTokenT&& value) {
    SetNextToken(std::forward<NextTokenT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_environmentId;
  bool m_environmentIdHasBeenSet = false;

  KxClusterType m_clusterType{KxClusterType::NOT_SET};
  bool m_clusterTypeHasBeenSet = false;

  int m_maxResults{0};
  bool m_maxResultsHasBeenSet = false;

  Aws::String m_nextToken;
  bool m_nextTokenHasBeenSet = false;
};

}  // namespace Model
}  // namespace finspace
}  // namespace Aws
