﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/elasticmapreduce/EMRRequest.h>
#include <aws/elasticmapreduce/EMR_EXPORTS.h>

#include <utility>

namespace Aws {
namespace EMR {
namespace Model {

/**
 */
class UpdateStudioRequest : public EMRRequest {
 public:
  AWS_EMR_API UpdateStudioRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "UpdateStudio"; }

  AWS_EMR_API Aws::String SerializePayload() const override;

  AWS_EMR_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The ID of the Amazon EMR Studio to update.</p>
   */
  inline const Aws::String& GetStudioId() const { return m_studioId; }
  inline bool StudioIdHasBeenSet() const { return m_studioIdHasBeenSet; }
  template <typename StudioIdT = Aws::String>
  void SetStudioId(StudioIdT&& value) {
    m_studioIdHasBeenSet = true;
    m_studioId = std::forward<StudioIdT>(value);
  }
  template <typename StudioIdT = Aws::String>
  UpdateStudioRequest& WithStudioId(StudioIdT&& value) {
    SetStudioId(std::forward<StudioIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A descriptive name for the Amazon EMR Studio.</p>
   */
  inline const Aws::String& GetName() const { return m_name; }
  inline bool NameHasBeenSet() const { return m_nameHasBeenSet; }
  template <typename NameT = Aws::String>
  void SetName(NameT&& value) {
    m_nameHasBeenSet = true;
    m_name = std::forward<NameT>(value);
  }
  template <typename NameT = Aws::String>
  UpdateStudioRequest& WithName(NameT&& value) {
    SetName(std::forward<NameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A detailed description to assign to the Amazon EMR Studio.</p>
   */
  inline const Aws::String& GetDescription() const { return m_description; }
  inline bool DescriptionHasBeenSet() const { return m_descriptionHasBeenSet; }
  template <typename DescriptionT = Aws::String>
  void SetDescription(DescriptionT&& value) {
    m_descriptionHasBeenSet = true;
    m_description = std::forward<DescriptionT>(value);
  }
  template <typename DescriptionT = Aws::String>
  UpdateStudioRequest& WithDescription(DescriptionT&& value) {
    SetDescription(std::forward<DescriptionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A list of subnet IDs to associate with the Amazon EMR Studio. The list can
   * include new subnet IDs, but must also include all of the subnet IDs previously
   * associated with the Studio. The list order does not matter. A Studio can have a
   * maximum of 5 subnets. The subnets must belong to the same VPC as the Studio.
   * </p>
   */
  inline const Aws::Vector<Aws::String>& GetSubnetIds() const { return m_subnetIds; }
  inline bool SubnetIdsHasBeenSet() const { return m_subnetIdsHasBeenSet; }
  template <typename SubnetIdsT = Aws::Vector<Aws::String>>
  void SetSubnetIds(SubnetIdsT&& value) {
    m_subnetIdsHasBeenSet = true;
    m_subnetIds = std::forward<SubnetIdsT>(value);
  }
  template <typename SubnetIdsT = Aws::Vector<Aws::String>>
  UpdateStudioRequest& WithSubnetIds(SubnetIdsT&& value) {
    SetSubnetIds(std::forward<SubnetIdsT>(value));
    return *this;
  }
  template <typename SubnetIdsT = Aws::String>
  UpdateStudioRequest& AddSubnetIds(SubnetIdsT&& value) {
    m_subnetIdsHasBeenSet = true;
    m_subnetIds.emplace_back(std::forward<SubnetIdsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The Amazon S3 location to back up Workspaces and notebook files for the
   * Amazon EMR Studio.</p>
   */
  inline const Aws::String& GetDefaultS3Location() const { return m_defaultS3Location; }
  inline bool DefaultS3LocationHasBeenSet() const { return m_defaultS3LocationHasBeenSet; }
  template <typename DefaultS3LocationT = Aws::String>
  void SetDefaultS3Location(DefaultS3LocationT&& value) {
    m_defaultS3LocationHasBeenSet = true;
    m_defaultS3Location = std::forward<DefaultS3LocationT>(value);
  }
  template <typename DefaultS3LocationT = Aws::String>
  UpdateStudioRequest& WithDefaultS3Location(DefaultS3LocationT&& value) {
    SetDefaultS3Location(std::forward<DefaultS3LocationT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The KMS key identifier (ARN) used to encrypt Amazon EMR Studio workspace and
   * notebook files when backed up to Amazon S3.</p>
   */
  inline const Aws::String& GetEncryptionKeyArn() const { return m_encryptionKeyArn; }
  inline bool EncryptionKeyArnHasBeenSet() const { return m_encryptionKeyArnHasBeenSet; }
  template <typename EncryptionKeyArnT = Aws::String>
  void SetEncryptionKeyArn(EncryptionKeyArnT&& value) {
    m_encryptionKeyArnHasBeenSet = true;
    m_encryptionKeyArn = std::forward<EncryptionKeyArnT>(value);
  }
  template <typename EncryptionKeyArnT = Aws::String>
  UpdateStudioRequest& WithEncryptionKeyArn(EncryptionKeyArnT&& value) {
    SetEncryptionKeyArn(std::forward<EncryptionKeyArnT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_studioId;
  bool m_studioIdHasBeenSet = false;

  Aws::String m_name;
  bool m_nameHasBeenSet = false;

  Aws::String m_description;
  bool m_descriptionHasBeenSet = false;

  Aws::Vector<Aws::String> m_subnetIds;
  bool m_subnetIdsHasBeenSet = false;

  Aws::String m_defaultS3Location;
  bool m_defaultS3LocationHasBeenSet = false;

  Aws::String m_encryptionKeyArn;
  bool m_encryptionKeyArnHasBeenSet = false;
};

}  // namespace Model
}  // namespace EMR
}  // namespace Aws
