﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/elasticbeanstalk/ElasticBeanstalkRequest.h>
#include <aws/elasticbeanstalk/ElasticBeanstalk_EXPORTS.h>

#include <utility>

namespace Aws {
namespace ElasticBeanstalk {
namespace Model {

/**
 * <p>Request to add or change the operations role used by an
 * environment.</p><p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/elasticbeanstalk-2010-12-01/AssociateEnvironmentOperationsRoleMessage">AWS
 * API Reference</a></p>
 */
class AssociateEnvironmentOperationsRoleRequest : public ElasticBeanstalkRequest {
 public:
  AWS_ELASTICBEANSTALK_API AssociateEnvironmentOperationsRoleRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "AssociateEnvironmentOperationsRole"; }

  AWS_ELASTICBEANSTALK_API Aws::String SerializePayload() const override;

 protected:
  AWS_ELASTICBEANSTALK_API void DumpBodyToUrl(Aws::Http::URI& uri) const override;

 public:
  ///@{
  /**
   * <p>The name of the environment to which to set the operations role.</p>
   */
  inline const Aws::String& GetEnvironmentName() const { return m_environmentName; }
  inline bool EnvironmentNameHasBeenSet() const { return m_environmentNameHasBeenSet; }
  template <typename EnvironmentNameT = Aws::String>
  void SetEnvironmentName(EnvironmentNameT&& value) {
    m_environmentNameHasBeenSet = true;
    m_environmentName = std::forward<EnvironmentNameT>(value);
  }
  template <typename EnvironmentNameT = Aws::String>
  AssociateEnvironmentOperationsRoleRequest& WithEnvironmentName(EnvironmentNameT&& value) {
    SetEnvironmentName(std::forward<EnvironmentNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The Amazon Resource Name (ARN) of an existing IAM role to be used as the
   * environment's operations role.</p>
   */
  inline const Aws::String& GetOperationsRole() const { return m_operationsRole; }
  inline bool OperationsRoleHasBeenSet() const { return m_operationsRoleHasBeenSet; }
  template <typename OperationsRoleT = Aws::String>
  void SetOperationsRole(OperationsRoleT&& value) {
    m_operationsRoleHasBeenSet = true;
    m_operationsRole = std::forward<OperationsRoleT>(value);
  }
  template <typename OperationsRoleT = Aws::String>
  AssociateEnvironmentOperationsRoleRequest& WithOperationsRole(OperationsRoleT&& value) {
    SetOperationsRole(std::forward<OperationsRoleT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_environmentName;
  bool m_environmentNameHasBeenSet = false;

  Aws::String m_operationsRole;
  bool m_operationsRoleHasBeenSet = false;
};

}  // namespace Model
}  // namespace ElasticBeanstalk
}  // namespace Aws
