﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/elasticache/ElastiCacheRequest.h>
#include <aws/elasticache/ElastiCache_EXPORTS.h>
#include <aws/elasticache/model/Tag.h>

#include <utility>

namespace Aws {
namespace ElastiCache {
namespace Model {

/**
 * <p>Represents the input of a <code>CreateCacheSecurityGroup</code>
 * operation.</p><p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/elasticache-2015-02-02/CreateCacheSecurityGroupMessage">AWS
 * API Reference</a></p>
 */
class CreateCacheSecurityGroupRequest : public ElastiCacheRequest {
 public:
  AWS_ELASTICACHE_API CreateCacheSecurityGroupRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "CreateCacheSecurityGroup"; }

  AWS_ELASTICACHE_API Aws::String SerializePayload() const override;

 protected:
  AWS_ELASTICACHE_API void DumpBodyToUrl(Aws::Http::URI& uri) const override;

 public:
  ///@{
  /**
   * <p>A name for the cache security group. This value is stored as a lowercase
   * string.</p> <p>Constraints: Must contain no more than 255 alphanumeric
   * characters. Cannot be the word "Default".</p> <p>Example:
   * <code>mysecuritygroup</code> </p>
   */
  inline const Aws::String& GetCacheSecurityGroupName() const { return m_cacheSecurityGroupName; }
  inline bool CacheSecurityGroupNameHasBeenSet() const { return m_cacheSecurityGroupNameHasBeenSet; }
  template <typename CacheSecurityGroupNameT = Aws::String>
  void SetCacheSecurityGroupName(CacheSecurityGroupNameT&& value) {
    m_cacheSecurityGroupNameHasBeenSet = true;
    m_cacheSecurityGroupName = std::forward<CacheSecurityGroupNameT>(value);
  }
  template <typename CacheSecurityGroupNameT = Aws::String>
  CreateCacheSecurityGroupRequest& WithCacheSecurityGroupName(CacheSecurityGroupNameT&& value) {
    SetCacheSecurityGroupName(std::forward<CacheSecurityGroupNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A description for the cache security group.</p>
   */
  inline const Aws::String& GetDescription() const { return m_description; }
  inline bool DescriptionHasBeenSet() const { return m_descriptionHasBeenSet; }
  template <typename DescriptionT = Aws::String>
  void SetDescription(DescriptionT&& value) {
    m_descriptionHasBeenSet = true;
    m_description = std::forward<DescriptionT>(value);
  }
  template <typename DescriptionT = Aws::String>
  CreateCacheSecurityGroupRequest& WithDescription(DescriptionT&& value) {
    SetDescription(std::forward<DescriptionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A list of tags to be added to this resource. A tag is a key-value pair. A tag
   * key must be accompanied by a tag value, although null is accepted.</p>
   */
  inline const Aws::Vector<Tag>& GetTags() const { return m_tags; }
  inline bool TagsHasBeenSet() const { return m_tagsHasBeenSet; }
  template <typename TagsT = Aws::Vector<Tag>>
  void SetTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags = std::forward<TagsT>(value);
  }
  template <typename TagsT = Aws::Vector<Tag>>
  CreateCacheSecurityGroupRequest& WithTags(TagsT&& value) {
    SetTags(std::forward<TagsT>(value));
    return *this;
  }
  template <typename TagsT = Tag>
  CreateCacheSecurityGroupRequest& AddTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags.emplace_back(std::forward<TagsT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_cacheSecurityGroupName;
  bool m_cacheSecurityGroupNameHasBeenSet = false;

  Aws::String m_description;
  bool m_descriptionHasBeenSet = false;

  Aws::Vector<Tag> m_tags;
  bool m_tagsHasBeenSet = false;
};

}  // namespace Model
}  // namespace ElastiCache
}  // namespace Aws
