﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/ecs/ECSRequest.h>
#include <aws/ecs/ECS_EXPORTS.h>
#include <aws/ecs/model/NetworkConfiguration.h>
#include <aws/ecs/model/PropagateTags.h>
#include <aws/ecs/model/Tag.h>
#include <aws/ecs/model/TaskOverride.h>
#include <aws/ecs/model/TaskVolumeConfiguration.h>

#include <utility>

namespace Aws {
namespace ECS {
namespace Model {

/**
 */
class StartTaskRequest : public ECSRequest {
 public:
  AWS_ECS_API StartTaskRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "StartTask"; }

  AWS_ECS_API Aws::String SerializePayload() const override;

  AWS_ECS_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The short name or full Amazon Resource Name (ARN) of the cluster where to
   * start your task. If you do not specify a cluster, the default cluster is
   * assumed.</p>
   */
  inline const Aws::String& GetCluster() const { return m_cluster; }
  inline bool ClusterHasBeenSet() const { return m_clusterHasBeenSet; }
  template <typename ClusterT = Aws::String>
  void SetCluster(ClusterT&& value) {
    m_clusterHasBeenSet = true;
    m_cluster = std::forward<ClusterT>(value);
  }
  template <typename ClusterT = Aws::String>
  StartTaskRequest& WithCluster(ClusterT&& value) {
    SetCluster(std::forward<ClusterT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The container instance IDs or full ARN entries for the container instances
   * where you would like to place your task. You can specify up to 10 container
   * instances.</p>
   */
  inline const Aws::Vector<Aws::String>& GetContainerInstances() const { return m_containerInstances; }
  inline bool ContainerInstancesHasBeenSet() const { return m_containerInstancesHasBeenSet; }
  template <typename ContainerInstancesT = Aws::Vector<Aws::String>>
  void SetContainerInstances(ContainerInstancesT&& value) {
    m_containerInstancesHasBeenSet = true;
    m_containerInstances = std::forward<ContainerInstancesT>(value);
  }
  template <typename ContainerInstancesT = Aws::Vector<Aws::String>>
  StartTaskRequest& WithContainerInstances(ContainerInstancesT&& value) {
    SetContainerInstances(std::forward<ContainerInstancesT>(value));
    return *this;
  }
  template <typename ContainerInstancesT = Aws::String>
  StartTaskRequest& AddContainerInstances(ContainerInstancesT&& value) {
    m_containerInstancesHasBeenSet = true;
    m_containerInstances.emplace_back(std::forward<ContainerInstancesT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Specifies whether to use Amazon ECS managed tags for the task. For more
   * information, see <a
   * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-using-tags.html">Tagging
   * Your Amazon ECS Resources</a> in the <i>Amazon Elastic Container Service
   * Developer Guide</i>.</p>
   */
  inline bool GetEnableECSManagedTags() const { return m_enableECSManagedTags; }
  inline bool EnableECSManagedTagsHasBeenSet() const { return m_enableECSManagedTagsHasBeenSet; }
  inline void SetEnableECSManagedTags(bool value) {
    m_enableECSManagedTagsHasBeenSet = true;
    m_enableECSManagedTags = value;
  }
  inline StartTaskRequest& WithEnableECSManagedTags(bool value) {
    SetEnableECSManagedTags(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Whether or not the execute command functionality is turned on for the task.
   * If <code>true</code>, this turns on the execute command functionality on all
   * containers in the task.</p>
   */
  inline bool GetEnableExecuteCommand() const { return m_enableExecuteCommand; }
  inline bool EnableExecuteCommandHasBeenSet() const { return m_enableExecuteCommandHasBeenSet; }
  inline void SetEnableExecuteCommand(bool value) {
    m_enableExecuteCommandHasBeenSet = true;
    m_enableExecuteCommand = value;
  }
  inline StartTaskRequest& WithEnableExecuteCommand(bool value) {
    SetEnableExecuteCommand(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name of the task group to associate with the task. The default value is
   * the family name of the task definition (for example, family:my-family-name).</p>
   */
  inline const Aws::String& GetGroup() const { return m_group; }
  inline bool GroupHasBeenSet() const { return m_groupHasBeenSet; }
  template <typename GroupT = Aws::String>
  void SetGroup(GroupT&& value) {
    m_groupHasBeenSet = true;
    m_group = std::forward<GroupT>(value);
  }
  template <typename GroupT = Aws::String>
  StartTaskRequest& WithGroup(GroupT&& value) {
    SetGroup(std::forward<GroupT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The VPC subnet and security group configuration for tasks that receive their
   * own elastic network interface by using the <code>awsvpc</code> networking
   * mode.</p>
   */
  inline const NetworkConfiguration& GetNetworkConfiguration() const { return m_networkConfiguration; }
  inline bool NetworkConfigurationHasBeenSet() const { return m_networkConfigurationHasBeenSet; }
  template <typename NetworkConfigurationT = NetworkConfiguration>
  void SetNetworkConfiguration(NetworkConfigurationT&& value) {
    m_networkConfigurationHasBeenSet = true;
    m_networkConfiguration = std::forward<NetworkConfigurationT>(value);
  }
  template <typename NetworkConfigurationT = NetworkConfiguration>
  StartTaskRequest& WithNetworkConfiguration(NetworkConfigurationT&& value) {
    SetNetworkConfiguration(std::forward<NetworkConfigurationT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A list of container overrides in JSON format that specify the name of a
   * container in the specified task definition and the overrides it receives. You
   * can override the default command for a container (that's specified in the task
   * definition or Docker image) with a <code>command</code> override. You can also
   * override existing environment variables (that are specified in the task
   * definition or Docker image) on a container or add new environment variables to
   * it with an <code>environment</code> override.</p>  <p>A total of 8192
   * characters are allowed for overrides. This limit includes the JSON formatting
   * characters of the override structure.</p>
   */
  inline const TaskOverride& GetOverrides() const { return m_overrides; }
  inline bool OverridesHasBeenSet() const { return m_overridesHasBeenSet; }
  template <typename OverridesT = TaskOverride>
  void SetOverrides(OverridesT&& value) {
    m_overridesHasBeenSet = true;
    m_overrides = std::forward<OverridesT>(value);
  }
  template <typename OverridesT = TaskOverride>
  StartTaskRequest& WithOverrides(OverridesT&& value) {
    SetOverrides(std::forward<OverridesT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Specifies whether to propagate the tags from the task definition or the
   * service to the task. If no value is specified, the tags aren't propagated.</p>
   */
  inline PropagateTags GetPropagateTags() const { return m_propagateTags; }
  inline bool PropagateTagsHasBeenSet() const { return m_propagateTagsHasBeenSet; }
  inline void SetPropagateTags(PropagateTags value) {
    m_propagateTagsHasBeenSet = true;
    m_propagateTags = value;
  }
  inline StartTaskRequest& WithPropagateTags(PropagateTags value) {
    SetPropagateTags(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>This parameter is only used by Amazon ECS. It is not intended for use by
   * customers.</p>
   */
  inline const Aws::String& GetReferenceId() const { return m_referenceId; }
  inline bool ReferenceIdHasBeenSet() const { return m_referenceIdHasBeenSet; }
  template <typename ReferenceIdT = Aws::String>
  void SetReferenceId(ReferenceIdT&& value) {
    m_referenceIdHasBeenSet = true;
    m_referenceId = std::forward<ReferenceIdT>(value);
  }
  template <typename ReferenceIdT = Aws::String>
  StartTaskRequest& WithReferenceId(ReferenceIdT&& value) {
    SetReferenceId(std::forward<ReferenceIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>An optional tag specified when a task is started. For example, if you
   * automatically trigger a task to run a batch process job, you could apply a
   * unique identifier for that job to your task with the <code>startedBy</code>
   * parameter. You can then identify which tasks belong to that job by filtering the
   * results of a <a
   * href="https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_ListTasks.html">ListTasks</a>
   * call with the <code>startedBy</code> value. Up to 36 letters (uppercase and
   * lowercase), numbers, hyphens (-), forward slash (/), and underscores (_) are
   * allowed.</p> <p>If a task is started by an Amazon ECS service, the
   * <code>startedBy</code> parameter contains the deployment ID of the service that
   * starts it.</p>
   */
  inline const Aws::String& GetStartedBy() const { return m_startedBy; }
  inline bool StartedByHasBeenSet() const { return m_startedByHasBeenSet; }
  template <typename StartedByT = Aws::String>
  void SetStartedBy(StartedByT&& value) {
    m_startedByHasBeenSet = true;
    m_startedBy = std::forward<StartedByT>(value);
  }
  template <typename StartedByT = Aws::String>
  StartTaskRequest& WithStartedBy(StartedByT&& value) {
    SetStartedBy(std::forward<StartedByT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The metadata that you apply to the task to help you categorize and organize
   * them. Each tag consists of a key and an optional value, both of which you
   * define.</p> <p>The following basic restrictions apply to tags:</p> <ul> <li>
   * <p>Maximum number of tags per resource - 50</p> </li> <li> <p>For each resource,
   * each tag key must be unique, and each tag key can have only one value.</p> </li>
   * <li> <p>Maximum key length - 128 Unicode characters in UTF-8</p> </li> <li>
   * <p>Maximum value length - 256 Unicode characters in UTF-8</p> </li> <li> <p>If
   * your tagging schema is used across multiple services and resources, remember
   * that other services may have restrictions on allowed characters. Generally
   * allowed characters are: letters, numbers, and spaces representable in UTF-8, and
   * the following characters: + - = . _ : / @.</p> </li> <li> <p>Tag keys and values
   * are case-sensitive.</p> </li> <li> <p>Do not use <code>aws:</code>,
   * <code>AWS:</code>, or any upper or lowercase combination of such as a prefix for
   * either keys or values as it is reserved for Amazon Web Services use. You cannot
   * edit or delete tag keys or values with this prefix. Tags with this prefix do not
   * count against your tags per resource limit.</p> </li> </ul>
   */
  inline const Aws::Vector<Tag>& GetTags() const { return m_tags; }
  inline bool TagsHasBeenSet() const { return m_tagsHasBeenSet; }
  template <typename TagsT = Aws::Vector<Tag>>
  void SetTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags = std::forward<TagsT>(value);
  }
  template <typename TagsT = Aws::Vector<Tag>>
  StartTaskRequest& WithTags(TagsT&& value) {
    SetTags(std::forward<TagsT>(value));
    return *this;
  }
  template <typename TagsT = Tag>
  StartTaskRequest& AddTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags.emplace_back(std::forward<TagsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The <code>family</code> and <code>revision</code>
   * (<code>family:revision</code>) or full ARN of the task definition to start. If a
   * <code>revision</code> isn't specified, the latest <code>ACTIVE</code> revision
   * is used.</p>
   */
  inline const Aws::String& GetTaskDefinition() const { return m_taskDefinition; }
  inline bool TaskDefinitionHasBeenSet() const { return m_taskDefinitionHasBeenSet; }
  template <typename TaskDefinitionT = Aws::String>
  void SetTaskDefinition(TaskDefinitionT&& value) {
    m_taskDefinitionHasBeenSet = true;
    m_taskDefinition = std::forward<TaskDefinitionT>(value);
  }
  template <typename TaskDefinitionT = Aws::String>
  StartTaskRequest& WithTaskDefinition(TaskDefinitionT&& value) {
    SetTaskDefinition(std::forward<TaskDefinitionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The details of the volume that was <code>configuredAtLaunch</code>. You can
   * configure the size, volumeType, IOPS, throughput, snapshot and encryption in <a
   * href="https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_TaskManagedEBSVolumeConfiguration.html">TaskManagedEBSVolumeConfiguration</a>.
   * The <code>name</code> of the volume must match the <code>name</code> from the
   * task definition.</p>
   */
  inline const Aws::Vector<TaskVolumeConfiguration>& GetVolumeConfigurations() const { return m_volumeConfigurations; }
  inline bool VolumeConfigurationsHasBeenSet() const { return m_volumeConfigurationsHasBeenSet; }
  template <typename VolumeConfigurationsT = Aws::Vector<TaskVolumeConfiguration>>
  void SetVolumeConfigurations(VolumeConfigurationsT&& value) {
    m_volumeConfigurationsHasBeenSet = true;
    m_volumeConfigurations = std::forward<VolumeConfigurationsT>(value);
  }
  template <typename VolumeConfigurationsT = Aws::Vector<TaskVolumeConfiguration>>
  StartTaskRequest& WithVolumeConfigurations(VolumeConfigurationsT&& value) {
    SetVolumeConfigurations(std::forward<VolumeConfigurationsT>(value));
    return *this;
  }
  template <typename VolumeConfigurationsT = TaskVolumeConfiguration>
  StartTaskRequest& AddVolumeConfigurations(VolumeConfigurationsT&& value) {
    m_volumeConfigurationsHasBeenSet = true;
    m_volumeConfigurations.emplace_back(std::forward<VolumeConfigurationsT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_cluster;
  bool m_clusterHasBeenSet = false;

  Aws::Vector<Aws::String> m_containerInstances;
  bool m_containerInstancesHasBeenSet = false;

  bool m_enableECSManagedTags{false};
  bool m_enableECSManagedTagsHasBeenSet = false;

  bool m_enableExecuteCommand{false};
  bool m_enableExecuteCommandHasBeenSet = false;

  Aws::String m_group;
  bool m_groupHasBeenSet = false;

  NetworkConfiguration m_networkConfiguration;
  bool m_networkConfigurationHasBeenSet = false;

  TaskOverride m_overrides;
  bool m_overridesHasBeenSet = false;

  PropagateTags m_propagateTags{PropagateTags::NOT_SET};
  bool m_propagateTagsHasBeenSet = false;

  Aws::String m_referenceId;
  bool m_referenceIdHasBeenSet = false;

  Aws::String m_startedBy;
  bool m_startedByHasBeenSet = false;

  Aws::Vector<Tag> m_tags;
  bool m_tagsHasBeenSet = false;

  Aws::String m_taskDefinition;
  bool m_taskDefinitionHasBeenSet = false;

  Aws::Vector<TaskVolumeConfiguration> m_volumeConfigurations;
  bool m_volumeConfigurationsHasBeenSet = false;
};

}  // namespace Model
}  // namespace ECS
}  // namespace Aws
