﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/directconnect/DirectConnectRequest.h>
#include <aws/directconnect/DirectConnect_EXPORTS.h>
#include <aws/directconnect/model/LoaContentType.h>

#include <utility>

namespace Aws {
namespace DirectConnect {
namespace Model {

/**
 */
class DescribeLoaRequest : public DirectConnectRequest {
 public:
  AWS_DIRECTCONNECT_API DescribeLoaRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "DescribeLoa"; }

  AWS_DIRECTCONNECT_API Aws::String SerializePayload() const override;

  AWS_DIRECTCONNECT_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The ID of a connection, LAG, or interconnect.</p>
   */
  inline const Aws::String& GetConnectionId() const { return m_connectionId; }
  inline bool ConnectionIdHasBeenSet() const { return m_connectionIdHasBeenSet; }
  template <typename ConnectionIdT = Aws::String>
  void SetConnectionId(ConnectionIdT&& value) {
    m_connectionIdHasBeenSet = true;
    m_connectionId = std::forward<ConnectionIdT>(value);
  }
  template <typename ConnectionIdT = Aws::String>
  DescribeLoaRequest& WithConnectionId(ConnectionIdT&& value) {
    SetConnectionId(std::forward<ConnectionIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name of the service provider who establishes connectivity on your behalf.
   * If you specify this parameter, the LOA-CFA lists the provider name alongside
   * your company name as the requester of the cross connect.</p>
   */
  inline const Aws::String& GetProviderName() const { return m_providerName; }
  inline bool ProviderNameHasBeenSet() const { return m_providerNameHasBeenSet; }
  template <typename ProviderNameT = Aws::String>
  void SetProviderName(ProviderNameT&& value) {
    m_providerNameHasBeenSet = true;
    m_providerName = std::forward<ProviderNameT>(value);
  }
  template <typename ProviderNameT = Aws::String>
  DescribeLoaRequest& WithProviderName(ProviderNameT&& value) {
    SetProviderName(std::forward<ProviderNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The standard media type for the LOA-CFA document. The only supported value is
   * application/pdf.</p>
   */
  inline LoaContentType GetLoaContentType() const { return m_loaContentType; }
  inline bool LoaContentTypeHasBeenSet() const { return m_loaContentTypeHasBeenSet; }
  inline void SetLoaContentType(LoaContentType value) {
    m_loaContentTypeHasBeenSet = true;
    m_loaContentType = value;
  }
  inline DescribeLoaRequest& WithLoaContentType(LoaContentType value) {
    SetLoaContentType(value);
    return *this;
  }
  ///@}
 private:
  Aws::String m_connectionId;
  bool m_connectionIdHasBeenSet = false;

  Aws::String m_providerName;
  bool m_providerNameHasBeenSet = false;

  LoaContentType m_loaContentType{LoaContentType::NOT_SET};
  bool m_loaContentTypeHasBeenSet = false;
};

}  // namespace Model
}  // namespace DirectConnect
}  // namespace Aws
