﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/controltower/ControlTowerRequest.h>
#include <aws/controltower/ControlTower_EXPORTS.h>
#include <aws/core/utils/memory/stl/AWSString.h>

#include <utility>

namespace Aws {
namespace ControlTower {
namespace Model {

/**
 */
class DisableControlRequest : public ControlTowerRequest {
 public:
  AWS_CONTROLTOWER_API DisableControlRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "DisableControl"; }

  AWS_CONTROLTOWER_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p>The ARN of the control. Only <b>Strongly recommended</b> and <b>Elective</b>
   * controls are permitted, with the exception of the <b>Region deny</b> control.
   * For information on how to find the <code>controlIdentifier</code>, see <a
   * href="https://docs.aws.amazon.com/controltower/latest/APIReference/Welcome.html">the
   * overview page</a>.</p>
   */
  inline const Aws::String& GetControlIdentifier() const { return m_controlIdentifier; }
  inline bool ControlIdentifierHasBeenSet() const { return m_controlIdentifierHasBeenSet; }
  template <typename ControlIdentifierT = Aws::String>
  void SetControlIdentifier(ControlIdentifierT&& value) {
    m_controlIdentifierHasBeenSet = true;
    m_controlIdentifier = std::forward<ControlIdentifierT>(value);
  }
  template <typename ControlIdentifierT = Aws::String>
  DisableControlRequest& WithControlIdentifier(ControlIdentifierT&& value) {
    SetControlIdentifier(std::forward<ControlIdentifierT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The ARN of the organizational unit. For information on how to find the
   * <code>targetIdentifier</code>, see <a
   * href="https://docs.aws.amazon.com/controltower/latest/APIReference/Welcome.html">the
   * overview page</a>.</p>
   */
  inline const Aws::String& GetTargetIdentifier() const { return m_targetIdentifier; }
  inline bool TargetIdentifierHasBeenSet() const { return m_targetIdentifierHasBeenSet; }
  template <typename TargetIdentifierT = Aws::String>
  void SetTargetIdentifier(TargetIdentifierT&& value) {
    m_targetIdentifierHasBeenSet = true;
    m_targetIdentifier = std::forward<TargetIdentifierT>(value);
  }
  template <typename TargetIdentifierT = Aws::String>
  DisableControlRequest& WithTargetIdentifier(TargetIdentifierT&& value) {
    SetTargetIdentifier(std::forward<TargetIdentifierT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The ARN of the enabled control to be disabled, which uniquely identifies the
   * control instance on the target organizational unit.</p>
   */
  inline const Aws::String& GetEnabledControlIdentifier() const { return m_enabledControlIdentifier; }
  inline bool EnabledControlIdentifierHasBeenSet() const { return m_enabledControlIdentifierHasBeenSet; }
  template <typename EnabledControlIdentifierT = Aws::String>
  void SetEnabledControlIdentifier(EnabledControlIdentifierT&& value) {
    m_enabledControlIdentifierHasBeenSet = true;
    m_enabledControlIdentifier = std::forward<EnabledControlIdentifierT>(value);
  }
  template <typename EnabledControlIdentifierT = Aws::String>
  DisableControlRequest& WithEnabledControlIdentifier(EnabledControlIdentifierT&& value) {
    SetEnabledControlIdentifier(std::forward<EnabledControlIdentifierT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_controlIdentifier;
  bool m_controlIdentifierHasBeenSet = false;

  Aws::String m_targetIdentifier;
  bool m_targetIdentifierHasBeenSet = false;

  Aws::String m_enabledControlIdentifier;
  bool m_enabledControlIdentifierHasBeenSet = false;
};

}  // namespace Model
}  // namespace ControlTower
}  // namespace Aws
