﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/connect/ConnectRequest.h>
#include <aws/connect/Connect_EXPORTS.h>
#include <aws/connect/model/DisconnectReason.h>
#include <aws/core/utils/memory/stl/AWSString.h>

#include <utility>

namespace Aws {
namespace Connect {
namespace Model {

/**
 */
class StopContactRequest : public ConnectRequest {
 public:
  AWS_CONNECT_API StopContactRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "StopContact"; }

  AWS_CONNECT_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p>The ID of the contact.</p>
   */
  inline const Aws::String& GetContactId() const { return m_contactId; }
  inline bool ContactIdHasBeenSet() const { return m_contactIdHasBeenSet; }
  template <typename ContactIdT = Aws::String>
  void SetContactId(ContactIdT&& value) {
    m_contactIdHasBeenSet = true;
    m_contactId = std::forward<ContactIdT>(value);
  }
  template <typename ContactIdT = Aws::String>
  StopContactRequest& WithContactId(ContactIdT&& value) {
    SetContactId(std::forward<ContactIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The identifier of the Amazon Connect instance. You can <a
   * href="https://docs.aws.amazon.com/connect/latest/adminguide/find-instance-arn.html">find
   * the instance ID</a> in the Amazon Resource Name (ARN) of the instance.</p>
   */
  inline const Aws::String& GetInstanceId() const { return m_instanceId; }
  inline bool InstanceIdHasBeenSet() const { return m_instanceIdHasBeenSet; }
  template <typename InstanceIdT = Aws::String>
  void SetInstanceId(InstanceIdT&& value) {
    m_instanceIdHasBeenSet = true;
    m_instanceId = std::forward<InstanceIdT>(value);
  }
  template <typename InstanceIdT = Aws::String>
  StopContactRequest& WithInstanceId(InstanceIdT&& value) {
    SetInstanceId(std::forward<InstanceIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The reason a contact can be disconnected. Only Amazon Connect outbound
   * campaigns can provide this field. For a list and description of all the possible
   * disconnect reasons by channel (including outbound campaign voice contacts) see
   * DisconnectReason under <a
   * href="https://docs.aws.amazon.com/connect/latest/adminguide/ctr-data-model.html#ctr-ContactTraceRecord">ContactTraceRecord</a>
   * in the <i>Amazon Connect Administrator Guide</i>. </p>
   */
  inline const DisconnectReason& GetDisconnectReason() const { return m_disconnectReason; }
  inline bool DisconnectReasonHasBeenSet() const { return m_disconnectReasonHasBeenSet; }
  template <typename DisconnectReasonT = DisconnectReason>
  void SetDisconnectReason(DisconnectReasonT&& value) {
    m_disconnectReasonHasBeenSet = true;
    m_disconnectReason = std::forward<DisconnectReasonT>(value);
  }
  template <typename DisconnectReasonT = DisconnectReason>
  StopContactRequest& WithDisconnectReason(DisconnectReasonT&& value) {
    SetDisconnectReason(std::forward<DisconnectReasonT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_contactId;
  bool m_contactIdHasBeenSet = false;

  Aws::String m_instanceId;
  bool m_instanceIdHasBeenSet = false;

  DisconnectReason m_disconnectReason;
  bool m_disconnectReasonHasBeenSet = false;
};

}  // namespace Model
}  // namespace Connect
}  // namespace Aws
