/***************************************************************************
 *
 * knetworkmanager-settings.cpp - A NetworkManager frontend for KDE
 *
 * Copyright (C) 2005, 2006 Novell, Inc.
 *
 * Author: Helmut Schaa       <hschaa@suse.de>, <helmut.schaa@gmx.de>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 **************************************************************************/

#include <qpushbutton.h>
#include <qframe.h>
#include <qvbox.h>
#include <qvaluelist.h>
#include <qcombobox.h>
#include <qlayout.h>
#include <klocale.h>
#include <knotifydialog.h>
#include <kdebug.h>
#include <kpushbutton.h>
#include <knuminput.h>
#include <qgroupbox.h>
#include <qcheckbox.h>
#include <kiconloader.h>
#include <kprocess.h>
#include <kconfig.h>
#include <qradiobutton.h>
#include <kwallet.h>
#include <kactivelabel.h>
#include <kseparator.h>

#include "knetworkmanager-settings.h"
#include "knetworkmanager-ui-networklistview.h"
#include "knetworkmanager-storage.h"
#include "knetworkmanager-vpnconnectionsdialog.h"

/*
  GeneralSettingsWidget
*/

GeneralSettingsWidget::GeneralSettingsWidget(QWidget* parent, char* name)
	: GeneralSetting(parent, name)
{
	readConfig();
}

GeneralSettingsWidget::~GeneralSettingsWidget()
{

}

void GeneralSettingsWidget::readConfig()
{
	KConfig* config = KGlobal::config();
	bool warning = false;

	config->setGroup("General");
	chkAutostart->setChecked(config->readBoolEntry("Autostart", true));

	config->setGroup("Notification Messages");
	chkAskQuit->setChecked(config->readBoolEntry("systemtrayquitKNetworkManager", true));

	KNetworkManagerStorage * storage = KNetworkManagerStorage::getInstance();

	optKeysUnencrypted->setChecked(storage->getStoreKeysUnencrypted());
	optKeysInWallet->setChecked(!storage->getStoreKeysUnencrypted());

	// user may not select kwallet if the wallet is disabled
	optKeysInWallet->setEnabled(KWallet::Wallet::isEnabled());
	if (!KWallet::Wallet::isEnabled())
	{
		// show warning
		warning = true;
		imgWarning->setPixmap(KGlobal::instance()->iconLoader()->loadIcon("messagebox_warning", KIcon::NoGroup, KIcon::SizeMedium));
		lblWarning->setText("Warning: KWallet is disabled, passwords will be stored in clear text!");
	}
	else if (storage->getStoreKeysUnencrypted())
	{
		warning = true;
		imgWarning->setPixmap(KGlobal::instance()->iconLoader()->loadIcon("messagebox_warning", KIcon::NoGroup, KIcon::SizeMedium));
		lblWarning->setText("Warning: Passwords will be stored in clear text!");
	}

	lblImage->setPixmap(SmallIcon("password", QIconSet::Automatic));
	lblGeneralImage->setPixmap(SmallIcon("package_settings", QIconSet::Automatic));

	sepWarning->setShown(warning);
	imgWarning->setShown(warning);
	lblWarning->setShown(warning);
}

void GeneralSettingsWidget::apply()
{
	KConfig * config = KGlobal::config();

	config->setGroup ("General");
	config->writeEntry ("Autostart", chkAutostart->isChecked());

	config->setGroup ("Notification Messages");
	config->writeEntry("systemtrayquitKNetworkManager", chkAskQuit->isChecked());

	// Read from Storage whether to store keys in Wallet or in Config file
	KNetworkManagerStorage * storage = KNetworkManagerStorage::getInstance();
	storage->setStoreKeysUnencrypted(optKeysUnencrypted->isChecked());

	readConfig();
}

void GeneralSettingsWidget::discard()
{

}

/*
  NetworksConfigWidget
*/

NetworksConfigWidget::NetworksConfigWidget(KNetworkManager* ctx, QWidget* parent, char* name)
	: QWidget(parent, name)
{
	_ctx = ctx;
	QVBoxLayout* layout = new QVBoxLayout(this);
	layout->setSpacing( KDialog::spacingHint() );
	QWidget* mainWid = this;
	_lv = new KNetworkManagerNetworkListView( _ctx->getNetworkManagerInfo(), mainWid, "networklv" );
	QToolTip::add( _lv, i18n( "Drag and drop to move networks between groups" ) );
	layout->add(_lv);
	QWidget * wid = new QWidget( mainWid );
	layout->add(wid);
	QHBoxLayout * vbox = new QHBoxLayout( wid );
	vbox->setSpacing( KDialog::spacingHint() );
	QPushButton * addBtn = new QPushButton( i18n("Add Network..." ), wid );
	QPushButton * addApBtn = new QPushButton( i18n("Add Access Point..." ), wid );
	QPushButton * remBtn = new QPushButton( i18n("Remove Item" ), wid );
	QToolTip::add( addBtn, i18n( "Add a new network to the selected group" ) );
	QToolTip::add( addApBtn, i18n( "Add a new access point to the selected network" ) );
	QToolTip::add( remBtn, i18n( "Remove the selected network or access point" ) );
	vbox->addWidget( addBtn );
	vbox->addWidget( addApBtn );
	vbox->addStretch();
	vbox->addWidget( remBtn );
	
	connect( addBtn, SIGNAL( clicked() ), _lv, SLOT( slotAddNetworkClicked() ) );
	connect( addApBtn, SIGNAL( clicked() ), _lv, SLOT( slotAddAccessPointClicked() ) );
	connect( remBtn, SIGNAL( clicked() ), _lv, SLOT( slotRemoveItemClicked() ) );
	
	void slotAddNetworkClicked();
	void slotAddAccessPointClicked();
	void slotRemoveItemClicked();

	addBtn->setEnabled( false );
	addApBtn->setEnabled( false );
	addBtn->setHidden( true );
	addApBtn->setHidden( true );
}

NetworksConfigWidget::~NetworksConfigWidget()
{

}

void NetworksConfigWidget::apply()
{
	KNetworkManagerStorage * storage = KNetworkManagerStorage::getInstance();
	// changed networks
	QValueList< Network * > changes = _lv->changedNetworks();
	QValueList< Network * >::Iterator it = changes.begin();
	for ( ; it != changes.end(); ++it )
		storage->storeNetwork( *it, false );
	// deleted networks
	QValueList< Network * > deleted = _lv->deletedNetworks();
	QValueList< Network * >::Iterator delIt = deleted.begin();
	for ( ; delIt != deleted.end(); ++delIt )
		storage->removeNetwork( *delIt );
}

void NetworksConfigWidget::discard()
{

}

/*
  KNotifyWidget
*/

KNotifyWidget::KNotifyWidget(QWidget* parent, char * name)
	: KNotifyDialog(parent, name, false)
{
	// we want us displayed in our parent
	reparent(parent, 0, QPoint(0,0));

	// parent has buttons for us
	showButton(KDialogBase::Ok, false);
	showButton(KDialogBase::Apply, false);
	showButton(KDialogBase::Cancel, false);

	// we do not want borders
	layout()->setMargin(0);
}

KNotifyWidget::~KNotifyWidget()
{

}

void KNotifyWidget::apply()
{
	slotApply();
}

void KNotifyWidget::discard()
{
	slotCancel();
}

/*
  Dialup Connections
*/
DialupConfigWidget::DialupConfigWidget(QWidget* parent, char* name)
	: QWidget(parent, name)
{
	QGridLayout* layout = new QGridLayout(this, 2, 2);

	KPushButton* push = new KPushButton("Configure Dialup Connections", this);
	connect(push, SIGNAL(clicked()), this, SLOT(configureDialup()));
	layout->addWidget(push, 0, 0);

	QSpacerItem* spacer = new QSpacerItem(20,20, QSizePolicy::Expanding, QSizePolicy::Minimum );
	layout->addItem(spacer, 0, 1);

	spacer = new QSpacerItem(20,20, QSizePolicy::Minimum, QSizePolicy::Expanding);
	layout->addItem(spacer, 1, 0);

}

DialupConfigWidget::~DialupConfigWidget()
{

}

void DialupConfigWidget::configureDialup ()
{
#ifdef KNETWORKMANAGER_DIALUP_CONFIG
	KProcess* dialupConfig = new KProcess ();
	QStringList command = QStringList::split (" ", KNETWORKMANAGER_DIALUP_CONFIG, false);
	for (QStringList::Iterator it = command.begin (); it != command.end (); ++it)
		*dialupConfig << (*it);
	dialupConfig->start ();
#endif
}

/*
  SettingsDialog
*/

SettingsDialog::SettingsDialog(KNetworkManager* ctx, QWidget* parent, char* name, WFlags fl, bool modal, QString caption)
	: KDialogBase(KDialogBase::IconList, fl, parent, name, modal, caption)
{
	_ctx = ctx;
	QWidget* page = NULL;

	this->enableButtonSeparator(true);

	// general settings
	page = addVBoxPage(i18n("General Settings"), i18n("General Settings"), KGlobal::instance()->iconLoader()->loadIcon("package_settings", KIcon::NoGroup, KIcon::SizeMedium));
	GeneralSettingsWidget* general = new GeneralSettingsWidget(page, "gerenal_settings");
	connect(this, SIGNAL(okClicked()), general, SLOT(apply()));
	connect(this, SIGNAL(applyClicked()), general, SLOT(apply()));
	connect(this, SIGNAL(cancelClicked()), general, SLOT(discard()));

	// configure notifications
	page = addVBoxPage(i18n("Notifications"), i18n("Notifications"), KGlobal::instance()->iconLoader()->loadIcon("knotify", KIcon::NoGroup, KIcon::SizeMedium));
	KNotifyWidget* notify = new KNotifyWidget(page, "notify_dialog");
	connect(this, SIGNAL(okClicked()), notify, SLOT(apply()));
	connect(this, SIGNAL(applyClicked()), notify, SLOT(apply()));
	connect(this, SIGNAL(cancelClicked()), notify, SLOT(discard()));
	connect(notify, SIGNAL(cancelClicked()), this, SLOT());

	// configure networks
	page = addVBoxPage(i18n("Wireless Networks"), i18n("Wireless Networks"), KGlobal::instance()->iconLoader()->loadIcon("network", KIcon::NoGroup, KIcon::SizeMedium));
	NetworksConfigWidget* networks = new NetworksConfigWidget(_ctx, page, "networks");
	connect(this, SIGNAL(okClicked()), networks, SLOT(apply()));
	connect(this, SIGNAL(applyClicked()), networks, SLOT(apply()));
	connect(this, SIGNAL(cancelClicked()), notify, SLOT(cancel()));

	if (ctx->getVPN()->isAvailable())
	{
		// configure VPN connections
		page = addVBoxPage(i18n("VPN Connections"), i18n("VPN Connections"), KGlobal::instance()->iconLoader()->loadIcon("encrypted", KIcon::NoGroup, KIcon::SizeMedium));
		VPNConnectionsDialog* vpn = new VPNConnectionsDialog(_ctx, page, "vpn", false, 0);
		connect(vpn, SIGNAL(done()), _ctx->getVPN(), SLOT(updateVPNConnections()));
		connect(this, SIGNAL(okClicked()), vpn, SLOT(apply()));
		connect(this, SIGNAL(applyClicked()), vpn, SLOT(apply()));
		vpn->reparent(page, 0, QPoint(0,0));
		vpn->layout()->setMargin(0);
		vpn->pushClose->hide();
	}

#ifdef KNETWORKMANAGER_DIALUP_CONFIG
	page = addVBoxPage(i18n("Dial-Up Connections"), i18n("Dial-Up Connections"), KGlobal::instance()->iconLoader()->loadIcon("modem", KIcon::NoGroup, KIcon::SizeMedium));
	new DialupConfigWidget(page, "dialup");
#endif

	resize(minimumSize());

}

SettingsDialog::~SettingsDialog()
{

}
