
/***************************************************************************
 *                                                                         *
 *   KNetLoad is copyright (c) 1999-2000, Markus Gustavsson                *
 *                         (c) 2002, Ben Burton                            *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "knetproc.h"

#include <cstdio>
#include <cstdlib>
#include <cstring>

/**
 * Linux
 * -----
 *
 * We assume /proc/net/dev is filled with lines of the following form:
 *
 *   iface: bytesIn x x x x x x x bytesOut x x x x x x x
 *
 * We also assume that the first two lines of /proc/net/dev are headers
 * that can be ignored.
 *
 * Other Operating Systems
 * -----------------------
 *
 * Please, send in a patch!
 *
 * The KNetLoad maintainer is currently Ben Burton <bab@debian.org>, or
 * you could submit a patch through the KDE bug tracking system
 * (bugs.kde.org).
 */

#define MAX_NET_DEV_LINE 512
#define MAX_INTERFACE_NAME 128

const float KNetProc::megabyte = 1000000;

KNetProc::KNetProc() {
    currIn = currOut = 0;
    prevIn = prevOut = 0;
}

void KNetProc::setDevice(const char *interface) {
    // Actually set the device.
    device = interface;

    // Take an initial reading.
    currIn = currOut = 0;
    prevIn = prevOut = 0;
    readLoad();

    // Make the previous reading identical to the current reading to
    // avoid a massive spike before the next reading is taken.
    prevIn = currIn;
    prevOut = currOut;
}

void KNetProc::readLoad() {
    // The current readings must now become the previous readings.
    prevIn = currIn;
    prevOut = currOut;

    // Take the new set of readings.
    static FILE* fd;
    static char line[MAX_NET_DEV_LINE];
    static char* pos;
    static char* iface;
    static float x[7];

    if ((fd = fopen("/proc/net/dev", "r")) == 0)
        return;

    // Read the unwanted header lines.
    fgets(line, MAX_NET_DEV_LINE, fd);
    fgets(line, MAX_NET_DEV_LINE, fd);

    // Read through the remaining lines until we find our device.
    while (! feof(fd)) {
        fgets(line, MAX_NET_DEV_LINE, fd);

        // Find the interface name for this line.
        for (iface = line; *iface == ' '; iface++)
            ; // (skip initial spaces)
        for (pos = iface; *pos != ':' && *pos != 0; pos++)
            ; // (move to next ':' or end of string)
        if (*pos == 0)
            continue; // (was not ':')
        *pos = 0;

        // Now iface points to a null-terminated string containing the
        // interface name for this particular line.
        if (device != iface)
            continue;

        // This is the correct device!
        // Read off the numbers of bytes.
        sscanf(pos + 1, "%f %f %f %f %f %f %f %f %f",
            &currIn, x, x+1, x+2, x+3, x+4, x+5, x+6, &currOut);

        // Tidy up.
        if (currIn < prevIn)
            currIn = prevIn;
        if (currOut < prevOut)
            currOut = prevOut;

        fclose(fd);
        return;
    }

    // The device was not found.
    currIn = prevIn;
    currOut = prevOut;
    fclose(fd);
}

