/***************************************************************************
                          kpldoc.cpp  -  description
                             -------------------

    This file is a part of kpl - a program for graphical presentation of
    data sets and functions.

    begin                : Sat Apr 24 15:14:00 MEST 1999

    copyright            : (C) 2004 by Werner Stille
    email                : stille@uni-freiburg.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include <math.h>
#include <stdlib.h>
#include <qcombobox.h>
#include <qdir.h>
#include <qhbox.h>
#include <qlabel.h>
#include <qlineedit.h>
#include <kapp.h>
#include <kfiledialog.h>
#include <kglobal.h>
#include <kio/netaccess.h>
#include <klocale.h>
#include <kmessagebox.h>
#include <ksimpleconfig.h>
#include <ktempfile.h>
#include "arcitem.h"
#include "arrayitem.h"
#include "ellipseitem.h"
#include "fitdlg.h"
#include "frameitem.h"
#include "funitem.h"
#include "imageitem.h"
#include "kpldoc.h"
#include "legenditem.h"
#include "lineitem.h"
#include "rectitem.h"
#include "scalebaritem.h"
#include "splineitem.h"
#include "textitem.h"
#include "threeditem.h"
#include "utils.h"

KplDoc::KplDoc(QObject* _parent) :
 QObject(_parent), modified(false), fType(Unknown), chisq(0.0)
{
  memset(pLast, 0, sizeof(pLast));
  memset(pErr, 0, sizeof(pErr));
  separators << " " << "\t" << ",";
  itd.setAutoDelete(true);
  itb.setAutoDelete(true);
}

KplDoc::~KplDoc()
{
  freeAllItems();
  itb.clear();
}

void KplDoc::newDocument()
{
  docURL.setFileName(i18n("Untitled"));
  freeAllItems();
  aut.nyLeg = 0;
}

bool KplDoc::saveDocument(const KURL& url, QList<KplItem>* _items, bool _abs)
{
  QString filename = url.isLocalFile() ? url.path() : tmpFile();
  bool success = true;
  if (QFile::exists(filename))
    success = QFile::remove(filename);
  if (success) {
    {
      KSimpleConfig plo(filename);
      plo.setGroup("Items");
      plo.writeEntry("nitems", _items->count());
      for (unsigned int i = 0; i < _items->count(); i++) {
        plo.setGroup("Item " + QString::number(i));
        _items->at(i)->writePlo(&plo, url, _abs, this);
      }
    }
    if (!url.isLocalFile())
      copyTmp(filename, url);
  } else
    KMessageBox::sorry((QWidget*) parent(),
                       i18n("Writing to this file not possible."));
  return success;
}

void KplDoc::setModified(bool m)
{
  modified = m;
}

bool KplDoc::isModified() const
{
  return modified;
}

void KplDoc::setCurrentDir(const KURL& url)
{
  QFileInfo fi(url.path());
  KURL u = url;
  u.setPath(fi.dirPath(true));
  m_dir = u.url(+1);
}

const QString& KplDoc::currentDir() const
{
  return m_dir;
}

void KplDoc::setURL(const KURL& url)
{
  docURL = url;
  setCurrentDir(url);
  if (url.isLocalFile()) {
    QFileInfo fi(url.path());
    m_time = fi.lastModified();
  } else
    m_time = QDateTime();
}

const KURL& KplDoc::URL() const
{
  return docURL;
}

const QDateTime& KplDoc::URLTime() const
{
  return m_time;
}

KplDoc::FileType KplDoc::fileType() const
{
  return fType;
}

void KplDoc::setFileType(FileType iType)
{
  fType = iType;
}

void KplDoc::freeAllItems()
{
  itd.clear();
  setModified(false);
}

void KplDoc::newItem(KplItem::ItemTypes ityp, int index)
{
  KplItem* it = 0;
  switch (ityp) {
    case KplItem::Frame:
      it = new FrameItem(&aut);
      break;
    case KplItem::Array:
      it = new ArrayItem(&aut);
      break;
    case KplItem::Function:
      it = new FunItem(&aut);
      break;
    case KplItem::ParFunction:
      it = new ParFunItem(&aut);
      break;
    case KplItem::Spline:
      it = new SplineItem(&aut);
      break;
    case KplItem::Array3D:
      it = new Array3DItem(&aut);
      break;
    case KplItem::Function3D:
      it = new Fun3DItem(&aut);
      break;
    case KplItem::Legend:
      it = new LegendItem(&aut);
      break;
    case KplItem::Text:
      it = new TextItem(&aut);
      break;
    case KplItem::Line:
      it = new LineItem(&aut);
      break;
    case KplItem::Arrow:
      it = new ArrowItem(&aut);
      break;
    case KplItem::Arc:
      it = new ArcItem(&aut);
      break;
    case KplItem::Rectangle:
      it = new RectItem(&aut);
      break;
    case KplItem::Ellipse:
      it = new EllipseItem(&aut);
      break;
    case KplItem::Image:
      it = new ImageItem(&aut);
      break;
    case KplItem::ScaleBar:
      it = new ScaleBarItem(&aut);
  }
  if (index < 0)
    itd.append(it);
  else
    itd.insert(index, it);
}

void KplDoc::autoFile(double* xmi, double* xma, double* tic, int* mtic,
                      int* ndig, int* ie, double* fn, const double* x, int n,
                      double c, bool _log)
{
  double xmin, xmax;
  Utils::minMaxFile(&xmin, &xmax, x, n);
  if (aut.autoNorm)
    Utils::expo(QMAX(fabs(xmin), fabs(xmax)), ie, fn);
  else {
    *ie = 0;
    *fn = 1.0;
  }
  Utils::autoSc(xmi, xma, tic, mtic, ndig, xmin, xmax, *fn, c, _log);
}

bool KplDoc::autoScale(FrameItem* fd, ArrayItem* ad)
{
  if ((ad->x) && (aut.ixAuto < ad->ncols) && (aut.iyAuto < ad->ncols)) {
    autoFile(&fd->xmi, &fd->xma, &fd->xtic, &fd->mticx, &fd->ndigx,
             &fd->iex, &ad->fx, ad->x[aut.ixAuto], ad->nrows,
             0.2, aut.autoLogX);
    autoFile(&fd->ymi, &fd->yma, &fd->ytic, &fd->mticy, &fd->ndigy,
             &fd->iey, &ad->fy, ad->x[aut.iyAuto], ad->nrows,
             0.2 * aut.xlAuto / aut.ylAuto, aut.autoLogY);
    if (aut.autohPath)
      fd->sh = ad->url.isLocalFile() ? ad->url.path() : ad->url.url();
    int minCol = ad->ncols - 1;
    ad->ix = QMIN(aut.ixAuto, minCol);
    ad->iy = QMIN(aut.iyAuto, minCol);
    ad->ie = QMIN(aut.ieAuto, minCol);
    ad->symb = aut.autoSymb;
    ad->istart = 0;
    ad->n = ad->nrows;
    ad->errbars = aut.autoErr & (ad->ie < ad->ncols);
    return false;
  }
  return true;
}

void KplDoc::autoScaleFrame(bool autoNorm, int item, int* iex, int* iey,
                            double* fx, double* fy, double* xmi, double* xma,
                            double* ymi, double* yma, double* xtic, double* ytic,
                            int* mticx, int* mticy, int* ndigx, int* ndigy,
                            double c, bool logx, bool logy)
{
  if (autoNorm) {
    double fxt, fyt;
    int iext, ieyt;
    *iex = *iey = 999;
    for (KplItem* itm = itd.at(item + 1); itm; itm = itd.next()) {
      if (itm->iType() == KplItem::Frame)
        break;
      if (itm->active) {
        iext = *iex;
        ieyt = *iey;
        itm->expoItem(&iext, &ieyt, &fxt, &fyt);
        if (iext < *iex) {
          *iex = iext;
          *fx = fxt;
        }
        if (ieyt < *iey) {
          *iey = ieyt;
          *fy = fyt;
        }
      }
    }
  } else {
    *iex = *iey = 0;
    *fx = *fy = 1.0;
  }
  if ((*iex != 999) && (*iey != 999)) {
    double xmin = 1.0e300;
    double ymin = xmin;
    double xmax = -xmin;
    double ymax = xmax;
    for (KplItem *itm = itd.at(item + 1); itm; itm = itd.next()) {
      if (itm->iType() == KplItem::Frame)
        break;
      if (itm->active) {
        *xmi = xmin;
        *ymi = ymin;
        *xma = xmax;
        *yma = ymax;
        itm->minMax(xmi, xma, ymi, yma);
        xmin = QMIN(xmin, *xmi);
        xmax = QMAX(xmax, *xma);
        ymin = QMIN(ymin, *ymi);
        ymax = QMAX(ymax, *yma);
      }
    }
    Utils::autoSc(xmi, xma, xtic, mticx, ndigx, xmin, xmax, *fx, 0.2, logx);
    Utils::autoSc(ymi, yma, ytic, mticy, ndigy, ymin, ymax, *fy, 0.2 * c, logy);
  }
}

void KplDoc::setOptions(Kpl::AutoStruct* opt)
{
  aut = *opt;
}

const Kpl::AutoStruct* KplDoc::options() const
{
  return &aut;
}

const QString& KplDoc::separator() const
{
  return separators[aut.iSep];
}

QList<KplItem>* KplDoc::items()
{
  return &itd;
}

QString KplDoc::tmpFile() const
{
  KTempFile f;
  return f.name();
}

void KplDoc::copyTmp(const QString& src, const KURL& dest)
{
  connect(KIO::file_copy(KURL(src), dest, -1, true),
          SIGNAL(result(KIO::Job*)), this, SLOT(slotIOfinished(KIO::Job*)));
}

const QString KplDoc::number(double val) const
{
  return QString::number(val, aut.format, aut.prec);
}

bool KplDoc::getWriteURL(QWidget* parent, KURL& url, const QString& filter)
{
  url = KFileDialog::getSaveURL(currentDir(), filter);
  if (url.isEmpty())
    return false;
  if (url.isLocalFile()) {
    QFileInfo fi(url.path());
    if (!fi.exists())
      return true;
    if (!fi.isWritable()) {
      KMessageBox::sorry(parent, i18n("Writing to this file not possible."));
      return false;
    }
  } else {
#if KDE_VERSION >= 0x03015a
    if (!KIO::NetAccess::exists(url, true, parent))
      return true;
    if (!KIO::NetAccess::exists(url, false, parent)) {
      KMessageBox::sorry(parent, i18n("Writing to this file not possible."));
      return false;
    }
#else
    if (!KIO::NetAccess::exists(url))
      return true;
#endif
  }
  return (KMessageBox::warningYesNo(parent, url.path() + "\n" +
    i18n("A file with this name exists already.\n"
         "Do you want to overwrite it?")) == KMessageBox::Yes);
}

#if KDE_VERSION >= 0x030100
KURL::List KplDoc::getReadURLs(int& idec)
{
  QHBox* hbox = new QHBox();
  KFileDialog dlg(currentDir(), "*.dat *.DAT\n*", 0, 0, true, hbox);
  QComboBox* cb = decimalWidget(idec, &dlg, hbox, KFile::Files);
  dlg.exec();
  idec = cb->currentItem();
#else
KURL::List KplDoc::getReadURLs(int&)
{
  KFileDialog dlg(currentDir(), "*.dat *.DAT\n*", 0, 0, true);
  dlg.exec();
#endif
  return dlg.selectedURLs();
}

KURL KplDoc::getReadURL(int& idec, const QString& filename,
                        const QString& filter)
{
#if KDE_VERSION >= 0x030100
  QHBox* hbox = new QHBox();
  KFileDialog dlg(currentDir() + filename, filter, 0, 0, true, hbox);
  QComboBox* cb = decimalWidget(idec, &dlg, hbox, KFile::File);
  dlg.exec();
  idec = cb->currentItem();
#else
  KFileDialog dlg(currentDir() + filename, filter, 0, 0, true);
  dlg.exec();
#endif
  return dlg.selectedURL();
}

void KplDoc::getFile(QLineEdit* e)
{
  QString s = KFileDialog::getOpenFileName(currentDir(), "*.so\n*");
  if (!s.isEmpty()) {
    e->setText(s);
    setCurrentDir(s);
  }
}

bool KplDoc::loadPar(const KURL& url, QWidget* parent, double* p, int idec)
{
  bool success = false;
  QString fn;
#if KDE_VERSION >= 0x03015a
  if (KIO::NetAccess::download(url, fn, parent)) {
#else
  if (KIO::NetAccess::download(url, fn)) {
#endif
    QFile f(fn);
    if (f.open(IO_ReadOnly)) {
      memset(p, 0, KPL_NPMAX * sizeof(double));
      QTextStream t(&f);
      int i = -1;
      while (!t.eof() && (++i < KPL_NPMAX)){
        QString line = t.readLine();
        Utils::translate(line, idec);
        p[i] = line.toDouble();
      }
      f.close();
      setCurrentDir(url);
      success = true;
    } else
      KMessageBox::error(parent,
                         i18n("while trying to open parameter file"));
    KIO::NetAccess::removeTempFile(fn);
  }
  return success;
}

void KplDoc::getPar(QWidget* parent, double* p)
{
  int idec = aut.iDec;
  KURL url = getReadURL(idec, "", "*.par *.PAR\n*");
  if (!url.isEmpty())
    loadPar(url, parent, p, idec);
}

bool KplDoc::saveParameter(QWidget* parent, const KURL& url, double* p,
                           double* err)
{
  QFile f(url.isLocalFile() ? url.path() : tmpFile());
  bool success = f.open(IO_WriteOnly);
  if (success) {
    QTextStream t(&f);
    for (int i = 0; i < KPL_NPMAX; i++) {
      t << number(p[i]);
      if (err)
        t << separator() << number(err[i]);
      t << "\n";
    }
    f.close();
    setCurrentDir(url);
    if (!url.isLocalFile())
      copyTmp(f.name(), url);
  } else
    KMessageBox::error(parent, i18n("while trying to open parameter file"));
  return success;
}

void KplDoc::saveFunPar(QWidget* parent, double* p)
{
  KURL url;
  if (getWriteURL(parent, url, "*.par\n*"))
    saveParameter(parent, url, p, 0);
}

void KplDoc::autoFit(ArrayItem* ad, FunItem* fd)
{
  fd->pathy = aut.fitPath;
  fd->namey = aut.fitName;
  memcpy(fd->py, aut.pFit, KPL_NPMAX * sizeof(double));
  fd->symb = aut.fitSymb;
  fd->color = aut.fitColor;
  fd->tmin = ad->x[ad->ix][ad->istart];
  fd->tmax = ad->x[ad->ix][ad->istart + ad->n - 1];
  fd->fx = ad->fx;
  fd->fy = ad->fy;
  if (FunItem::getFuncAddr(fd->pathy.path(), fd->namey, &fd->hmody, &fd->fkty))
    return;
  int mode = 0;
  if (aut.fitShowDlg)
    mode |= FitDlg::ShowDlg;
  if (aut.fitSavePar)
    mode |= FitDlg::SavePar;
  if (aut.fitFollow)
    mode |= FitDlg::Follow;
  QList<ArrayItem> arr;
  arr.append(ad);
  QList<FunItem> fun;
  fun.append(fd);
  KConfig* config = KGlobal::config();
  config->setGroup("DataErrors0");
  config->writeEntry("FitErrorColumn", aut.err.fitErrCol);
  if (!aut.err.fitErrCol) {
    config->writeEntry("ErrorModelPath", aut.err.errModPath);
    config->writeEntry("ErrorModelName", aut.err.errModName);
    QStrList s1;
    for (int i = 0; i < KPL_NPMAX; i++)
      s1.insert(i, number(aut.err.pErrMod[i]));
    config->writeEntry("ErrorModelParam", s1, ' ');
    config->writeEntry("ErrorModelArgument", aut.err.errModArg);
  }
  FitDlg dlg((QWidget*) parent(), this, &arr, &fun, mode);
  dlg.fit();
  if (aut.fitShowDlg) {
    backupItems();
    dlg.exec();
  } else
    dlg.getValues(false);
  return;
}

QComboBox* KplDoc::decimalWidget(int idec, KFileDialog* dlg, QHBox* hbox,
                                 unsigned mode)
{
  new QLabel(i18n("Decimal symbol"), hbox);
  QComboBox* cb = new QComboBox(false, hbox);
  QStringList list;
  list << "." << ",";
  cb->insertStringList(list);
  cb->setCurrentItem(idec);
  cb->setFixedWidth(80);
#if QT_VERSION >= 0x030100
  (new QLabel("", hbox))->setSizePolicy(QSizePolicy::MinimumExpanding,
                                        QSizePolicy::Minimum);
#else
  (new QLabel("", hbox))->setSizePolicy(QSizePolicy(QSizePolicy::MinimumExpanding,
                                        QSizePolicy::Minimum));
#endif
  hbox->setSpacing(dlg->spacingHint());
  dlg->setOperationMode(KFileDialog::Opening);
  dlg->setCaption(i18n("Open"));
  dlg->setMode(mode);
  return cb;
}

bool KplDoc::openDocument(const KURL& url, int idec, FileType type)
{
  fType = type;
  if (fType == Unknown) {
    QString s = url.fileName();
    int pos = s.findRev(".plo", -1, false);
    fType = ((pos >= 0) && ((s.length() - pos) == 4)) ? Plot : Data;
  }
  bool success;
  bool newFrame = false;
  if (fType == Data) {
    int iItem = -1;
    if (aut.addData) {
      for (iItem = itd.count() - 1; iItem >= 0; iItem--)
        if (itd.at(iItem)->iType() == KplItem::Frame)
          break;
    }
    if (iItem == -1) {
      if (!aut.addData)
        freeAllItems();
      newItem(KplItem::Frame);
      newFrame = true;
      iItem = itd.count() - 1;
    }
    FrameItem* fd = (FrameItem*) itd.at(iItem);
    newItem(KplItem::Array);
    ArrayItem* ad = (ArrayItem*) itd.last();
    ad->url = url;
    ad->idec = idec;
    ad->n = ad->nrows = ArrayItem::readFile(url, &ad->ncols, &ad->x, ad->idec);
    success = ad->n;
    if (success) {
      if (itd.count() == 2) {
        if (autoScale(fd, ad))
          success = false;
      } else {
        double fx, fy;
        int iex = 0, iey = 0;
        autoScaleFrame(aut.autoNorm, iItem, &iex, &iey, &fx, &fy,
                       &fd->xmi, &fd->xma, &fd->ymi, &fd->yma,
                       &fd->xtic, &fd->ytic, &fd->mticx, &fd->mticy,
                       &fd->ndigx, &fd->ndigy, fd->w / fd->h,
                       fd->logx, fd->logy);
        if ((iex != 999) && (iey != 999)) {
          fd->iex = iex;
          fd->iey = iey;
          for (KplItem* itm = itd.at(iItem + 1); itm; itm = itd.next()) {
            if (itm->iType() == KplItem::Frame)
              break;
            itm->normalize(fx, fy);
          }
        } else
          success = false;
      }
      if (success) {
        setURL(url);
        if (aut.autoFit) {
          newItem(KplItem::Function);
          autoFit(ad, (FunItem*) itd.last());
        }
        setModified();
      }
    }
  } else {
    QString tmpfile;
#if KDE_VERSION >= 0x03015a
    success = KIO::NetAccess::download(url, tmpfile, (QWidget*) parent());
#else
    success = KIO::NetAccess::download(url, tmpfile);
#endif
    if (success) {
      QFileInfo f2(tmpfile);
      success = f2.isReadable();
      if (success) {
        if (!aut.addData)
          newDocument();
        KSimpleConfig plo(tmpfile, true);
        plo.setGroup("Items");
        if (int nItems = QMAX(plo.readNumEntry("nitems"), 0)) {
          const char *itemName[] = {"FRAMEITEM", "ARRAYITEM", "FUNITEM",
                                    "PARFUNITEM", "SPLINEITEM", "ARRAY3DITEM",
                                    "FUN3DITEM", "LEGENDITEM", "TEXTITEM",
                                    "LINEITEM", "ARROWITEM", "ARCITEM",
                                    "RECTITEM", "ELLIPSEITEM", "IMAGEITEM",
                                    "SCALEBARITEM"};
          const KplItem::ItemTypes iTypes[] = {KplItem::Frame, KplItem::Array,
                                               KplItem::Function,
                                               KplItem::ParFunction,
                                               KplItem::Spline,
                                               KplItem::Array3D,
                                               KplItem::Function3D,
                                               KplItem::Legend, KplItem::Text,
                                               KplItem::Line, KplItem::Arrow,
                                               KplItem::Arc, KplItem::Rectangle,
                                               KplItem::Ellipse, KplItem::Image,
                                               KplItem::ScaleBar};
          setURL(url);
          for (int i = 0; i < nItems; i++) {
            QString s;
            s.sprintf("Item %i", i);
            plo.setGroup(s);
            s = plo.readEntry("Type");
            unsigned j;
            for (j = 0; j < (sizeof(iTypes) / sizeof(int)); j++)
              if (s == itemName[j])
                break;
            switch (iTypes[j]) {
              case KplItem::Frame:
                itd.append(new FrameItem(&plo, &aut));
                break;
              case KplItem::Array:
                itd.append(new ArrayItem(&plo, &aut, url));
                break;
              case KplItem::Function:
                itd.append(new FunItem(&plo, &aut, url));
                break;
              case KplItem::ParFunction:
                itd.append(new ParFunItem(&plo, &aut, url));
                break;
              case KplItem::Spline:
                itd.append(new SplineItem(&plo, &aut, url));
                break;
              case KplItem::Array3D:
                itd.append(new Array3DItem(&plo, &aut, url));
                break;
              case KplItem::Function3D:
                itd.append(new Fun3DItem(&plo, &aut, url));
                break;
              case KplItem::Legend:
                itd.append(new LegendItem(&plo, &aut));
                break;
              case KplItem::Text:
                itd.append(new TextItem(&plo, &aut));
                break;
              case KplItem::Line:
                itd.append(new LineItem(&plo, &aut));
                break;
              case KplItem::Arrow:
                itd.append(new ArrowItem(&plo, &aut));
                break;
              case KplItem::Arc:
                itd.append(new ArcItem(&plo, &aut));
                break;
              case KplItem::Rectangle:
                itd.append(new RectItem(&plo, &aut));
                break;
              case KplItem::Ellipse:
                itd.append(new EllipseItem(&plo, &aut));
                break;
              case KplItem::Image:
                itd.append(new ImageItem(&plo, url));
                break;
              case KplItem::ScaleBar:
                itd.append(new ScaleBarItem(&plo, &aut));
            }
          }
        } else
          success = false;
      }
      KIO::NetAccess::removeTempFile(tmpfile);
    }
  }
  if (success)
    backupItems();
  else {
    if ((fType == Data) && aut.addData) {
      itd.removeLast();
      if (newFrame)
        itd.removeLast();
    } else
      if (!aut.addData)
        restoreItems();
    KMessageBox::error((QWidget*) parent(),
                       QString(i18n("while loading the file\n")) + url.path());
  }
  return success;
}

bool KplDoc::restoreItems(bool undo)
{
  QList<KplItem>* itc = 0;
  if (undo) {
    if (undoAllowed())
      itc = itb.prev();
  } else
    if (redoAllowed())
      itc = itb.next();
  if (itc) {
    itd.clear();
    for (unsigned int i = 0; i < itc->count(); i++)
    itd.append(itc->at(i)->copy());
    emit modelChanged(true, true);
    return true;
  } else
    return false;
}

bool KplDoc::undoAllowed() const
{
  if (!itb.current())
    return false;
  return (itb.current() != itb.getFirst());
}

bool KplDoc::redoAllowed() const
{
  if (!itb.current())
    return false;
  return (itb.current() != itb.getLast());
}

void KplDoc::slotIOfinished(KIO::Job* job)
{
  if (job) {
    if (job->error()) {
      job->showErrorDialog();
      QFileInfo fi(((KIO::FileCopyJob*) job)->destURL().path());
      if (fi.extension().lower() == "plo") {
        setModified();
        emit modelChanged(false, false);
      }
    }
    remove(((KIO::FileCopyJob*) job)->srcURL().path());
  }
}

void KplDoc::backupItems(bool list)
{
  QList<KplItem>* itc = itb.current();
  if (itc) {
    if (itc == itb.getLast()) {
      if (itb.count() >= 20)
        itb.removeFirst();
    } else
      do
        itb.removeLast();
      while (itc != itb.current());
  }
  itb.append(new QList<KplItem>);
  itb.current()->setAutoDelete(true);
  for (unsigned int i = 0; i < itd.count(); i++)
    itb.current()->append(itd.at(i)->copy());
  emit modelChanged(true, list);
}

void KplDoc::deleteItem(int i)
{
  itd.remove(i);
  setModified();
  backupItems();
}

void KplDoc::moveItem(int is, int id)
{
  KplItem *item = itd.at(is);
  itd.setAutoDelete(false);
  itd.remove(is);
  itd.insert(id, item);
  itd.setAutoDelete(true);
  setModified();
  backupItems();
}
