/* -*- Mode: C; indent-tabs-mode: t; c-basic-offset: 8; tab-width: 8 -*-
 * 
 * Copyright (C) 2003-2004 Bastien Nocera <hadess@hadess.net>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * Authors: Bastien Nocera <hadess@hadess.net>
 *
 */

#include <glib.h>
#include <string.h>
#include "nautilus-burn-drive.h"

static void
list_drives (void)
{
	GList             *drives, *l;
	NautilusBurnDrive *drive;

	drives = nautilus_burn_drive_get_list (FALSE, FALSE);

	for (l = drives; l != NULL; l = l->next) {
		char       *type_str;
		char       *display_name;
		const char *media;
		int         media_type;
		int         max_speed_read;
		int         max_speed_write;
		gboolean    is_rewritable;
		gboolean    is_blank;
		gboolean    has_data;
		gboolean    has_audio;
		gboolean    door_is_open;
		gint64      size;

		drive = l->data;
		type_str = nautilus_burn_drive_get_supported_media_string (drive, FALSE);
		media_type = nautilus_burn_drive_get_media_type_full (drive,
								      &is_rewritable,
								      &is_blank,
								      &has_data,
								      &has_audio);
		media = nautilus_burn_drive_media_type_get_string (media_type);

		door_is_open = nautilus_burn_drive_door_is_open (drive);
		display_name = nautilus_burn_drive_get_name_for_display (drive);
		max_speed_read = nautilus_burn_drive_get_max_speed_read (drive);
		max_speed_write = nautilus_burn_drive_get_max_speed_read (drive);

		g_print ("name: %s device: %s max_speed_read: %d\n",
			 display_name,
			 drive->device,
			 max_speed_read);

		g_print ("door: %s\n", door_is_open ? "open" : "closed");
		g_print ("type: %s\n", type_str);
		g_print ("media type: %s%s%s%s%s\n",
			 media,
			 is_rewritable ? " (rewritable)" : "",
			 is_blank ? " (blank)" : "",
			 has_data ? " (has-data)" : "",
			 has_audio ? " (has-audio)" : "");

		size = nautilus_burn_drive_get_media_size (drive);
		if (size >= 0) {
			g_print ("media size: %0.2f MiB", (float) size / 1024 / 1024);

			if (media_type == NAUTILUS_BURN_MEDIA_TYPE_CD
			    || media_type == NAUTILUS_BURN_MEDIA_TYPE_CDR
			    || media_type == NAUTILUS_BURN_MEDIA_TYPE_CDRW) {
				g_print (" approx. or %d mins %d secs\n",
					 NAUTILUS_BURN_DRIVE_SIZE_TO_TIME (size) / 60,
					 NAUTILUS_BURN_DRIVE_SIZE_TO_TIME (size) % 60);
			} else {
				g_print ("\n");
			}
		} else {
			g_print ("media size: Could not be determined\n");
		}

		g_print ("CD-Recorder/SCSI devices only: max_speed_write: %d\n", max_speed_write);

		g_print ("\n");

		g_free (display_name);
		g_free (type_str);

		nautilus_burn_drive_unref (drive);
	}

	g_list_free (drives);
}

int main (int argc, char **argv)
{
	g_type_init ();
	list_drives ();

	return 0;
}

