/* -*- Mode: C; indent-tabs-mode: t; c-basic-offset: 8; tab-width: 8 -*-

   nautilus-cd-burner.c: easy to use cd burner software
 
   Copyright (C) 2002 Red Hat, Inc.
  
   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License as
   published by the Free Software Foundation; either version 2 of the
   License, or (at your option) any later version.
  
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.
  
   You should have received a copy of the GNU General Public
   License along with this program; if not, write to the
   Free Software Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.
  
   Authors: Alexander Larsson <alexl@redhat.com>
*/
#include <gtk/gtk.h>
#include <gtk/gtkseparatormenuitem.h>
#include <glade/glade.h>
#include <libgnomevfs/gnome-vfs.h>
#include <eel/eel-vfs-extensions.h>
#include <time.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>

#include "nautilus-cd-burner.h"
#include "make-iso.h"

static GladeXML *xml;

static GList *recorders = NULL;
static CancelFunc cancel_func = NULL;
static gboolean cancel_dangerous = FALSE;
static gpointer cancel_user_data;
/* For the image spinning */
static GList *image_list;

GtkWindow *
cd_progress_get_window (void)
{
	return GTK_WINDOW (glade_xml_get_widget (xml, "progress_window"));
}

void 
cd_progress_set_fraction (double fraction)
{
	GtkWidget *progress_bar;
	
	progress_bar = glade_xml_get_widget (xml, "cd_progress");

	gtk_progress_bar_set_fraction (GTK_PROGRESS_BAR (progress_bar), fraction);
}

void 
cd_progress_set_text (const char *text)
{
	GtkWidget *progress_label;
	
	progress_label = glade_xml_get_widget (xml, "cd_progress_label");

	gtk_label_set_text (GTK_LABEL (progress_label), text);
}

void
cd_progress_set_cancel_func (CancelFunc _cancel_func,
			     gboolean _cancel_dangerous,
			     gpointer user_data)
{
	cancel_func = _cancel_func;
	cancel_dangerous = _cancel_dangerous;
	cancel_user_data = user_data;
}

static gboolean
cd_progress_set_image (gpointer user_data)
{
	GtkWidget *image;
	GdkPixbuf *pixbuf;

	if (image_list == NULL) {
		return FALSE;
	}

	image = glade_xml_get_widget (xml, "cd_image");
	pixbuf = image_list->data;

	if (pixbuf == NULL || image == NULL) {
		return FALSE;
	}

	gtk_image_set_from_pixbuf (GTK_IMAGE (image), pixbuf);

	if (image_list->next != NULL) {
		image_list = image_list->next;
	} else {
		image_list = g_list_first (image_list);
	}

	return TRUE;
}

void
cd_progress_set_image_spinning (gboolean spinning)
{
	static guint spin_id = 0;

	if (spinning) {
		if (spin_id == 0) {
			spin_id = g_timeout_add (100, (GSourceFunc) cd_progress_set_image, NULL);
		}
	} else {
		if (spin_id != 0) {
			g_source_remove (spin_id);
			spin_id = 0;
		}
	}
}

static void
cd_progress_image_setup (void)
{
	GdkPixbuf *pixbuf;
	char *filename;
	int i;

	image_list = NULL;
	i = 1;

	/* Setup the pixbuf list */
	filename = g_strdup_printf (DATADIR "/cdspin%d.png", i);
	while (g_file_test (filename, G_FILE_TEST_IS_REGULAR)) {
		pixbuf = gdk_pixbuf_new_from_file (filename, NULL);

		if (pixbuf != NULL) {
			image_list = g_list_prepend (image_list, (gpointer) pixbuf);
		}

		i++;
		g_free (filename);
		filename = g_strdup_printf (DATADIR "/cdspin%d.png", i);
	}

	g_free (filename);
	if (image_list != NULL) {
		image_list = g_list_reverse (image_list);
	}

	/* Set the first image */
	cd_progress_set_image (NULL);
}

static void
cd_progress_image_cleanup (void)
{
	GdkPixbuf *pixbuf;

	image_list = g_list_first (image_list);

	while (image_list != NULL) {
		pixbuf = (GdkPixbuf *) image_list->data;
		if (pixbuf != NULL) {
			gdk_pixbuf_unref (pixbuf);
		}
		image_list = g_list_remove (image_list, image_list->data);
	}
}

static void
cancel_func_changed_cb (CDRecorder *cdrecorder, CancelFunc _cancel_func,
			gboolean _cancel_dangerous, gpointer data)
{
	cd_progress_set_cancel_func (_cancel_func, _cancel_dangerous,
				     (gpointer) cdrecorder);
}

static void
animation_changed_cb (CDRecorder *cdrecorder, gboolean spinning, gpointer data)
{
	cd_progress_set_image_spinning (spinning);
}

static void
action_changed_cb (CDRecorder *cdrecorder, gint index, gpointer data)
{
	GList *list;

	list = cd_recorder_list_actions (cdrecorder);
	cd_progress_set_text ((const char *)g_list_nth (list, index)->data);
}

static void
progress_changed_cb (CDRecorder *cdrecorder, gdouble fraction, gpointer data)
{
	cd_progress_set_fraction (fraction);
}

static void
reload_dialog_response (GtkDialog *dialog, gint response_id, gpointer data)
{
	CDRecorder *cdrecorder = (CDRecorder *) data;

	gtk_widget_destroy (GTK_WIDGET (dialog));

	cd_recorder_insert_cd_retry (cdrecorder, TRUE);
}

static void
insert_cd_response (GtkDialog *dialog, gint response_id, gpointer data)
{
	CDRecorder *cdrecorder = (CDRecorder *) data;

	gtk_widget_destroy (GTK_WIDGET (dialog));

	cd_recorder_insert_cd_retry (cdrecorder, FALSE);
}

static void
insert_cdrw_response (GtkDialog *dialog, gint response_id, gpointer data)
{
	CDRecorder *cdrecorder = (CDRecorder *) data;

	gtk_widget_destroy (GTK_WIDGET (dialog));

	cd_recorder_insert_cd_retry (cdrecorder, FALSE);
}

static void
show_reload_dialog (CDRecorder *cdrecorder)
{
	GtkWidget *reload_dialog;

	reload_dialog = gtk_message_dialog_new (cd_progress_get_window (),
						GTK_DIALOG_DESTROY_WITH_PARENT,
						GTK_MESSAGE_INFO,
						GTK_BUTTONS_OK,
						_("Please reload the CD in the CD writer."));
	gtk_window_set_title (GTK_WINDOW (reload_dialog),
			      _("Reload CD"));
	g_signal_connect (reload_dialog, "response", (GCallback)reload_dialog_response, cdrecorder);
	gtk_widget_show (reload_dialog);
}

static void
show_insert_dialog (CDRecorder *cdrecorder)
{
	GtkWidget *insert_cd_dialog;

	insert_cd_dialog = gtk_message_dialog_new (cd_progress_get_window (),
						   GTK_DIALOG_DESTROY_WITH_PARENT,
						   GTK_MESSAGE_INFO,
						   GTK_BUTTONS_OK,
						   _("Please insert a blank CD in the CD writer."));
	gtk_window_set_title (GTK_WINDOW (insert_cd_dialog),
			      _("Insert blank CD"));
	g_signal_connect (insert_cd_dialog, "response", (GCallback)insert_cd_response, cdrecorder);
	gtk_widget_show (insert_cd_dialog);
}

static void
show_insert_rw_dialog (CDRecorder *cdrecorder)
{
	GtkWidget *insert_cdrw_dialog;

	insert_cdrw_dialog = gtk_message_dialog_new (cd_progress_get_window (),
						   GTK_DIALOG_DESTROY_WITH_PARENT,
						   GTK_MESSAGE_INFO,
						   GTK_BUTTONS_OK,
						   _("Please insert a blank CD-RW in the CD writer."));
	gtk_window_set_title (GTK_WINDOW (insert_cdrw_dialog),
			      _("Insert blank CD-RW"));
	g_signal_connect (insert_cdrw_dialog, "response", (GCallback)insert_cdrw_response, cdrecorder);
	gtk_widget_show (insert_cdrw_dialog);
}

static void
insert_cd_request_cb (CDRecorder *cdrecorder, gboolean is_reload,
		      gpointer data)
{
	if (is_reload) {
		show_reload_dialog (cdrecorder);
	} else {
		show_insert_dialog (cdrecorder);
	}
}

static void
insert_rwcd_request_cb (CDRecorder *cdrecorder, gboolean is_reload,
		      gpointer data)
{
	if (is_reload) {
		show_reload_dialog (cdrecorder);
	} else {
		show_insert_rw_dialog (cdrecorder);
	}
}

static void
details_clicked (GtkButton *button, gpointer data)
{
	CDRecorder *cdrecorder = (CDRecorder *) data;
	GtkWidget *dialog, *label, *text, *scrolled_window;
	GtkTextBuffer *buffer;

	dialog = gtk_dialog_new_with_buttons (_("CD writing error details"),
					      cd_progress_get_window (),
					      GTK_DIALOG_DESTROY_WITH_PARENT | GTK_DIALOG_MODAL,
					      GTK_STOCK_CLOSE, GTK_RESPONSE_CANCEL,
					      NULL);

	label = gtk_label_new (_("Detailed error output from cdrecord:"));
	gtk_widget_show (label);
	gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->vbox),
			    label,
			    FALSE, FALSE, 0);

	buffer = gtk_text_buffer_new (NULL);
	gtk_text_buffer_set_text (buffer,
				  cd_recorder_get_error_message_details (cdrecorder), -1);

	text = gtk_text_view_new_with_buffer (buffer);
	g_object_unref (buffer);
	gtk_text_view_set_editable (GTK_TEXT_VIEW (text), FALSE);
	gtk_widget_show (text);

	scrolled_window = gtk_scrolled_window_new (NULL, NULL);
	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrolled_window),
					GTK_POLICY_AUTOMATIC,
					GTK_POLICY_ALWAYS);

	gtk_container_add (GTK_CONTAINER (scrolled_window), text);
	gtk_widget_show (scrolled_window);

	gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->vbox),
			    scrolled_window,
			    TRUE, TRUE, 0);


	g_signal_connect (dialog, "response", (GCallback)gtk_widget_destroy, NULL);

	gtk_widget_show (dialog);
}


static void
show_error_message (CDRecorder *cdrecorder)
{
	GtkWidget *button, *dialog;

	dialog = gtk_message_dialog_new (cd_progress_get_window (),
					 GTK_DIALOG_DESTROY_WITH_PARENT,
					 GTK_MESSAGE_ERROR,
					 GTK_BUTTONS_OK,
					 cd_recorder_get_error_message (cdrecorder) ?
					 _("There was an error writing to the CD:\n%s") : _("There was an error writing to the CD"),
					 cd_recorder_get_error_message (cdrecorder));
	gtk_window_set_title (GTK_WINDOW (dialog),
			      _("Error writing to CD"));
	button = gtk_button_new_with_mnemonic (_("_Details"));
	gtk_widget_show (button);
	gtk_box_pack_end (GTK_BOX (GTK_DIALOG (dialog)->action_area),
			  button,
			  FALSE, TRUE, 0);
	g_signal_connect (button, "clicked", (GCallback)details_clicked, cdrecorder);
	gtk_dialog_run (GTK_DIALOG (dialog));
	gtk_widget_destroy (dialog);
}

static gboolean
estimate_size_callback (const gchar *rel_path,
			GnomeVFSFileInfo *info,
			gboolean recursing_will_loop,
			gpointer data,
			gboolean *recurse)
{
	GnomeVFSFileSize *size = data;

	if (info->type != GNOME_VFS_FILE_TYPE_DIRECTORY) {
		*size += info->size;
	}
	
	*recurse = TRUE;
	return TRUE;
}


static GnomeVFSFileSize
estimate_size (char *uri)
{
	GnomeVFSFileSize size;

	size = 0;
	gnome_vfs_directory_visit (uri,
				   GNOME_VFS_FILE_INFO_DEFAULT,
				   GNOME_VFS_DIRECTORY_VISIT_DEFAULT,
				   estimate_size_callback,
				   &size);

	return size;
}

static void
setup_close_button (void)
{
	GtkWidget *close_button;
	GtkWidget *cancel_button;

	cancel_button = glade_xml_get_widget (xml, "cancel_button");
	close_button = glade_xml_get_widget (xml, "close_button");
	gtk_widget_hide (cancel_button);
	gtk_widget_show (close_button);
	g_signal_connect (close_button, "clicked", (GCallback)gtk_main_quit, NULL);
}

static int
burn_cd (CDDrive *rec, const char *source_iso, const char *label, gboolean eject_cd, gboolean blank_cd, int speed, gboolean dummy)
{
	CDRecorder *cdrecorder;
	GtkWidget *filesel;
	char *filename, *str;
	const char *f;
	int fd;
	int res = RESULT_ERROR;

	if (rec->type == CDDRIVE_TYPE_FILE) {
		if (source_iso != NULL) {
			/* TODO: display some message here */
			g_print ("Writing ISO onto ISO, you freak!\n");
			return RESULT_FINISHED;
		}
		filesel = gtk_file_selection_new (_("Choose a filename for the cdrom image"));
		gtk_file_selection_set_filename (GTK_FILE_SELECTION (filesel),
						 "~/image.iso");

		res = gtk_dialog_run (GTK_DIALOG (filesel));
		f = gtk_file_selection_get_filename (GTK_FILE_SELECTION (filesel));
		filename = g_strdup (f);
		gtk_widget_destroy (filesel);
		
		if (res == GTK_RESPONSE_CANCEL) {
			return RESULT_CANCEL;
		}
	} else {
		if (source_iso == NULL) {
			filename = g_build_filename (g_get_tmp_dir (), "image.iso.XXXXXX", NULL);
			fd = mkstemp(filename);
			close (fd);
		} else {
			filename = g_strdup (source_iso);
		}
	}

	if (source_iso == NULL) {
		/* TODO: Check for availible diskspace */
		res = make_iso (filename, label);
		if (!res) {
			/* User cancelled or we had an error. */
			goto out;
		}
	}

	if (rec->type == CDDRIVE_TYPE_FILE) {
		str = g_strdup_printf (_("Completed writing %s"), filename);
		cd_progress_set_text (str);
		g_free (str);
		setup_close_button ();
		gtk_main ();
	} else {
		cdrecorder = cd_recorder_new ();
		
		g_signal_connect (G_OBJECT (cdrecorder),
				  "progress-changed",
				  G_CALLBACK (progress_changed_cb),
				  NULL);
		g_signal_connect (G_OBJECT (cdrecorder),
				  "action-changed",
				  G_CALLBACK (action_changed_cb),
				  NULL);
		g_signal_connect (G_OBJECT (cdrecorder),
				  "animation-changed",
				  G_CALLBACK (animation_changed_cb),
				  NULL);
		g_signal_connect (G_OBJECT (cdrecorder),
				  "cancel-func-changed",
				  G_CALLBACK (cancel_func_changed_cb),
				  NULL);
		g_signal_connect (G_OBJECT (cdrecorder),
				  "insert-cd-request",
				  G_CALLBACK (insert_cd_request_cb),
				  NULL);
		g_signal_connect (G_OBJECT (cdrecorder),
				  "insert-rwcd-request",
				  G_CALLBACK (insert_rwcd_request_cb),
				  NULL);
		res = cd_recorder_write_iso (cdrecorder, rec, filename, speed, eject_cd, blank_cd, dummy);
		if (source_iso == NULL) {
			unlink (filename);
		}

		g_print ("res: %d\n", res);
		
		if (res == RESULT_FINISHED) {
			cd_progress_set_text (_("Complete"));
			setup_close_button ();
			gtk_main ();
		} else {
			show_error_message (cdrecorder);
		}
		
		g_object_unref (cdrecorder);
	}
	
 out:
	g_free (filename);
	return res;
}


static CDDrive *
lookup_current_recorder (GladeXML *xml)
{
	GtkWidget *option_menu;
	GtkWidget *item;
	CDDrive *rec;
	
	option_menu = glade_xml_get_widget (xml, "target_optionmenu");
	item = gtk_menu_get_active (GTK_MENU (GTK_OPTION_MENU (option_menu)->menu));
	rec = g_object_get_data (G_OBJECT (item), "cd-drive");
	
	return rec;
}


static void
refresh_dialog (GladeXML *xml)
{
	GtkWidget *option_menu;
	GtkWidget *menu;
	GtkWidget *item;
	GtkWidget *eject_check;
	GtkWidget *blank_check;
	GList *l, *children;
	char *name;
	int i;
	CDDrive *rec;

	/* Find active recorder: */
	rec = lookup_current_recorder (xml);

	/* add speed items: */
	option_menu = glade_xml_get_widget (xml, "speed_optionmenu");
	menu = gtk_option_menu_get_menu (GTK_OPTION_MENU (option_menu));

	children = gtk_container_get_children (GTK_CONTAINER (menu));
	for (l = children; l != NULL; l = l ->next) {
		gtk_container_remove (GTK_CONTAINER (menu),
				      l->data);
	}
	g_list_free (children);

	item = gtk_menu_item_new_with_label (_("Maximum Possible"));
	g_object_set_data (G_OBJECT (item), "speed", GINT_TO_POINTER (0));
	gtk_widget_show (item);
	gtk_menu_shell_append (GTK_MENU_SHELL (menu), item);
	for (i = 1; i <= rec->max_speed_write; i++) {
		name = g_strdup_printf ("%dx", i);
		item = gtk_menu_item_new_with_label (name);
		g_object_set_data (G_OBJECT (item), "speed", GINT_TO_POINTER (i));
		gtk_widget_show (item);
		gtk_menu_shell_append (GTK_MENU_SHELL (menu), item);
		g_free (name);
	}
	/* Default to automatic */
	gtk_option_menu_set_history (GTK_OPTION_MENU (option_menu), 0);

	/* Disable speed if max speed < 1 */
	gtk_widget_set_sensitive (option_menu, rec->max_speed_write > 0);

	/* Disable eject for burn-to-iso */
	eject_check = glade_xml_get_widget (xml, "eject_check");
	gtk_widget_set_sensitive (eject_check, rec->type != CDDRIVE_TYPE_FILE);

	/* Disable blank for burn-to-iso */
	blank_check = glade_xml_get_widget (xml, "blank_check");
	gtk_widget_set_sensitive (blank_check, rec->type != CDDRIVE_TYPE_FILE);
	
}

static void
target_changed (GtkOptionMenu *option_menu, gpointer data)
{
  GladeXML *xml = data;

  refresh_dialog (xml);
}

static void
init_dialog (GladeXML *xml, GtkWidget *dialog, GnomeVFSFileSize size, char *disc_name, gboolean disable_reuse)
{
	GList *l;
	CDDrive *rec;
	GtkWidget *option_menu;
	GtkWidget *menu;
	GtkWidget *item;
	GtkWidget *label;
	GtkWidget *entry;
	GtkWidget *reuse;
	struct tm *tm;
	int size_mb;
	char *str;
	char buf[129];
	time_t now;

	label = glade_xml_get_widget (xml, "top_label");

	size_mb = (size + 1024 * 1024 - 1)/ (1024 * 1024);
	str = g_strdup_printf (_("You have selected %d megabyte of files to be written to the CD."),
				size_mb);
	gtk_label_set_markup (GTK_LABEL (label), str);
	g_free (str);
	
	/* Fill in targets */
	option_menu = glade_xml_get_widget (xml, "target_optionmenu");
	menu = gtk_option_menu_get_menu	(GTK_OPTION_MENU (option_menu));
	for (l = recorders; l != NULL; l = l->next) {
		rec = l->data;
		if (rec->type == CDDRIVE_TYPE_FILE) {
			item = gtk_separator_menu_item_new ();
			gtk_widget_show (item);
			gtk_menu_shell_append (GTK_MENU_SHELL (menu), item);
		}
		item = gtk_menu_item_new_with_label (rec->display_name);
		g_object_set_data (G_OBJECT (item), "cd-drive", rec);
		gtk_widget_show (item);
		gtk_menu_shell_append (GTK_MENU_SHELL (menu), item);
	}
	gtk_option_menu_set_history (GTK_OPTION_MENU (option_menu), 0);

	g_signal_connect (option_menu, "changed", (GCallback)target_changed, xml);
	
	/* Set default name */
	
	entry = glade_xml_get_widget (xml, "cdname_entry");
	if (disc_name != NULL) {
		gtk_entry_set_text (GTK_ENTRY (entry), disc_name);
		gtk_editable_set_editable (GTK_EDITABLE (entry), FALSE);
	} else {
		time (&now);
		tm = localtime (&now);
		strftime (buf, sizeof (buf), _("Personal Data, %b %d, %Y"), tm);	
		gtk_entry_set_text (GTK_ENTRY (entry), buf);
	}

	reuse = glade_xml_get_widget (xml, "reuse_check");
	g_assert (GTK_IS_WIDGET (reuse));
	gtk_widget_set_sensitive (reuse, !disable_reuse);
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (reuse), TRUE);

	refresh_dialog (xml);
}

static void
do_cancel (void)
{
	GtkWidget *warning_dialog;
	gint res;
	
	if (cancel_dangerous) {
		warning_dialog = gtk_message_dialog_new (cd_progress_get_window (),
							 GTK_DIALOG_DESTROY_WITH_PARENT | GTK_DIALOG_MODAL,
							 GTK_MESSAGE_WARNING,
							 GTK_BUTTONS_YES_NO,
							 _("Are you sure you want to cancel the CD write operation?\n"
							   "Some CD writers may require that you restart the machine to get it working again."));
		gtk_window_set_title (GTK_WINDOW (warning_dialog),
				      _("Are you sure you want to cancel?"));
		res = gtk_dialog_run (GTK_DIALOG (warning_dialog));
		gtk_widget_destroy (warning_dialog);
		if (res != GTK_RESPONSE_YES) {
			return;
		}
	}

	if (cancel_func != NULL) {
		(*cancel_func) (cancel_user_data);
	}
}

static gboolean
handle_delete_event (GtkWidget	     *widget,
		     GdkEventAny     *event)
{
	do_cancel ();
	return TRUE;
}

static void
cancel_clicked (GtkButton *button, gpointer data)
{
	do_cancel ();
}

static gboolean
verify_iso (const char *filename, char **iso_label)
{
	FILE *file;
#define BUFFER_SIZE 128
	char buf[BUFFER_SIZE+1];
	int res;
	char *str, *str2;

	file = fopen (filename, "rb");
	if (file == NULL) {
		return FALSE;
	}
	/* Verify we have an ISO image */
	/* This check is for the raw sector images */
	res = fseek (file, 37633L, SEEK_SET);
	if (res) {
		return FALSE;
	}
	res = fread (buf, sizeof (char), 5, file);
	if (res != 5 || strncmp (buf, "CD001", 5) != 0) {
		/* Standard ISO images */
		res = fseek (file, 32769L, SEEK_SET);
		if (res) {
			return FALSE;
		}
		res = fread (buf, sizeof (char), 5, file);
		if (res != 5 || strncmp (buf, "CD001", 5) != 0) {
			/* High Sierra images */
			res = fseek (file, 32776L, SEEK_SET);
			if (res) {
				return FALSE;
			}
			res = fread (buf, sizeof (char), 5, file);
			if (res != 5 || strncmp (buf, "CDROM", 5) != 0) {
				return FALSE;
			}
		}
	}
	/* Extract the volume label from the image */
	res = fseek (file, 32808L, SEEK_SET);
	if (res) {
		return FALSE;
	}
	res = fread (buf, sizeof(char), BUFFER_SIZE, file);
	if (res != BUFFER_SIZE) {
		return FALSE;
	}
	buf[BUFFER_SIZE+1] = '\0';
	str = g_strdup (g_strstrip (buf));
	if (!g_utf8_validate (str, -1, NULL)) {
		/* Hmm, not UTF-8. Try the current locale. */
		str2 = g_locale_to_utf8 (str, -1, NULL, NULL, NULL);
		if (str2 == NULL) {
			str2 = eel_make_valid_utf8 (str);
		}
		g_free (str);
		str = str2;
	}
	fclose (file);
	*iso_label = str;
	return TRUE;
}

int
main (int argc, char *argv[])
{
	CDDrive *rec;
	GtkWidget *dialog, *entry, *eject_check, *blank_check, *reuse_check, *progress_window;
	GtkWidget *cancel_button, *dummy_check, *option_menu, *item;
	const char *label;
	int res, speed;
	gboolean eject, blank, reuse, dummy;
	GnomeVFSURI *uri;
	GnomeVFSFileSize size;
	char *source_iso, *disc_name = NULL;
	struct stat stat_buf;

	bindtextdomain (GETTEXT_PACKAGE, GNOMELOCALEDIR);
	bind_textdomain_codeset (GETTEXT_PACKAGE, "UTF-8");
	textdomain (GETTEXT_PACKAGE);
    
	recorders = scan_for_cdroms (TRUE, TRUE);
	
	gtk_init (&argc, &argv);
	gnome_vfs_init ();

	if (argc > 2) {
		dialog = gtk_message_dialog_new (NULL,
						 GTK_DIALOG_MODAL,
						 GTK_MESSAGE_ERROR,
						 GTK_BUTTONS_OK,
						 _("Too many parameters to nautilus-cd-burner"));
		gtk_window_set_title (GTK_WINDOW (dialog),
				      _("Too many parameters"));
		res = gtk_dialog_run (GTK_DIALOG (dialog));
		gtk_widget_destroy (dialog);
		return 1;
	}

	source_iso = (argc == 2) ? argv[1] : NULL;

	if (source_iso != NULL) {
		if (!verify_iso (source_iso, &disc_name)) {
			dialog = gtk_message_dialog_new (NULL,
							 GTK_DIALOG_MODAL,
							 GTK_MESSAGE_ERROR,
							 GTK_BUTTONS_OK,
							 _("The file '%s' is not a valid CD image."),
							 source_iso);
			gtk_window_set_title (GTK_WINDOW (dialog),
					      _("Not a valid CD image"));
			res = gtk_dialog_run (GTK_DIALOG (dialog));
			gtk_widget_destroy (dialog);
			return 1;
		}
		/* TODO: is this close enough for the CD size? */
		res = stat (source_iso, &stat_buf);
		size = stat_buf.st_size;
	} else {
		size = estimate_size ("burn:///");

		if (size == 0) {
			dialog = gtk_message_dialog_new (NULL,
							 GTK_DIALOG_MODAL,
							 GTK_MESSAGE_ERROR,
							 GTK_BUTTONS_NONE,
							 _("You need to copy the files you want to write to CD to the CD Creator window. Would you like to open it now?"));
			gtk_dialog_add_buttons (GTK_DIALOG (dialog), GTK_STOCK_CLOSE, GTK_RESPONSE_CLOSE, _("Open CD Creator"), GTK_RESPONSE_YES, NULL);
			gtk_dialog_set_default_response (GTK_DIALOG (dialog), GTK_RESPONSE_YES);
			gtk_window_set_title (GTK_WINDOW (dialog),
					      _("No files selected"));
			res = gtk_dialog_run (GTK_DIALOG (dialog));
			gtk_widget_destroy (dialog);
			if (res == GTK_RESPONSE_YES) {
				char *argv[] = {"nautilus", "--no-desktop", "burn:///", NULL};
				g_spawn_async (NULL, argv, NULL,
					       G_SPAWN_SEARCH_PATH,
					       NULL, NULL, NULL, NULL);
			}
			return 1;
		}
	}
	
	xml = glade_xml_new (DATADIR "/cdburn.glade", NULL, NULL);
	dialog = glade_xml_get_widget (xml, "cdr_dialog");

	init_dialog (xml, dialog, size, disc_name, source_iso != NULL);

	if (disc_name != NULL) g_free (disc_name);

	gtk_widget_show (dialog);

	dummy_check = glade_xml_get_widget (xml, "dummy_check");

#ifndef ENABLE_DUMMY_WRITE
	gtk_widget_hide (dummy_check);
#endif
	
	res = gtk_dialog_run (GTK_DIALOG (dialog));

	gtk_widget_hide (dialog);
	progress_window = glade_xml_get_widget (xml, "progress_window");
	cd_progress_image_setup ();
	gtk_widget_show (progress_window);
	cancel_button = glade_xml_get_widget (xml, "cancel_button");
	g_signal_connect (cancel_button, "clicked", (GCallback)cancel_clicked, NULL);
	g_signal_connect (progress_window, "delete_event", (GCallback)handle_delete_event, NULL);

	if (res == 0) {
		rec = lookup_current_recorder (xml);
		
		entry = glade_xml_get_widget (xml, "cdname_entry");
		label = gtk_entry_get_text (GTK_ENTRY (entry));

		eject_check = glade_xml_get_widget (xml, "eject_check");
		eject = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (eject_check));

		blank_check = glade_xml_get_widget (xml, "blank_check");
		blank = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (blank_check));

		dummy = FALSE;

#ifdef ENABLE_DUMMY_WRITE
		dummy = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (dummy_check));
#endif

		option_menu = glade_xml_get_widget (xml, "speed_optionmenu");
		item = gtk_menu_get_active (GTK_MENU (gtk_option_menu_get_menu (GTK_OPTION_MENU (option_menu))));
		speed = GPOINTER_TO_INT (g_object_get_data (G_OBJECT (item), "speed"));

		res = burn_cd (rec, source_iso, label, eject, blank, speed, dummy);

		reuse_check = glade_xml_get_widget (xml, "reuse_check");
		reuse = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (reuse_check));

		if (!reuse && res == RESULT_FINISHED) {
			GList l;
			
			uri = gnome_vfs_uri_new ("burn:///");
			
			l.data = uri;
			l.next = NULL;
			l.prev = NULL;

			gnome_vfs_xfer_delete_list (&l,
						    GNOME_VFS_XFER_ERROR_MODE_ABORT,
						    GNOME_VFS_XFER_RECURSIVE,
						    NULL, NULL);
		}
	}

	cd_progress_image_cleanup ();

	return 0;
}
