/*
 * Java-Gnome Bindings Library
 * 
 * Copyright 1998-2004 the Java-Gnome Team, all rights reserved.
 * 
 * The Java-Gnome bindings library is free software distributed under the terms
 * of the GNU Library General Public License version 2.
 */

package org.gnu.gtk;

import org.gnu.gdk.Pixbuf;
import org.gnu.glib.Boxed;
import org.gnu.glib.Handle;

public class IconSource extends Boxed {
    /**
     * Construct a new IconSource object.
     */
    public IconSource() {
        super(gtk_icon_source_new());
    }

    /**
     * Sets the source filename
     */
    public void setFilename(String filename) {
        gtk_icon_source_set_filename(getHandle(), filename);
    }

    /**
     * Retrieves the source filename or null if one does not exist.
     */
    public String getFilename() {
        return gtk_icon_source_get_filename(getHandle());
    }

    /**
     * Sets the source pixbuf.
     */
    public void setPixbuf(Pixbuf pixbuf) {
        gtk_icon_source_set_pixbuf(getHandle(), pixbuf.getHandle());
    }

    /**
     * Retrieves the source pixbuf or null if one does not exist.
     */
    public Pixbuf getPixbuf() {
        Handle hndl = gtk_icon_source_get_pixbuf(getHandle());
        if (hndl == null) {
            return null;
        }
        return new Pixbuf(hndl);
    }

    /**
     * Set the icon size.
     */
    public void setSize(IconSize iconSize) {
        gtk_icon_source_set_size(getHandle(), iconSize.getValue());
    }

    /**
     * Return the icon size.
     */
    public IconSize getSize() {
        return IconSize.intern(gtk_icon_source_get_size(getHandle()));
    }

    /**
     * Set the icon state.
     */
    public void setState(StateType state) {
        gtk_icon_source_set_state(getHandle(), state.getValue());
    }

    /**
     * Return the icon state.
     */
    public StateType getState() {
        return StateType.intern(gtk_icon_source_get_state(getHandle()));
    }

    /**
     * Set the text direction the icon source is intended to be used with.
     */
    public void setDirection(TextDirection direction) {
        gtk_icon_source_set_direction(getHandle(), direction.getValue());
    }

    /**
     * Returns the text direction for the icon source.
     */
    public TextDirection getDirection() {
        return TextDirection.intern(gtk_icon_source_get_direction(getHandle()));
    }

    /**
     * Sets the name of an icon to look up in the current icon theme to use as a
     * base image when creating icon variants for {@link org.gnu.gtk.IconSet}.
     */
    public void setIconName(String iconName) {
        gtk_icon_source_set_icon_name(getHandle(), iconName);
    }

    /**
     * Retrieves the source icon name, or NULL if none is set. The icon_name is
     * not a copy, and should not be modified or expected to persist beyond the
     * lifetime of the icon source.
     */
    public String getIconName() {
        return gtk_icon_source_get_icon_name(getHandle());
    }

    native static final protected Handle gtk_icon_source_new();

    native static final protected void gtk_icon_source_free(Handle source);

    native static final protected void gtk_icon_source_set_filename(
            Handle source, String filename);

    native static final protected void gtk_icon_source_set_pixbuf(
            Handle source, Handle pixbuf);

    native static final protected String gtk_icon_source_get_filename(
            Handle source);

    native static final protected Handle gtk_icon_source_get_pixbuf(
            Handle source);

    native static final protected void gtk_icon_source_set_direction_wildcarded(
            Handle source, boolean setting);

    native static final protected void gtk_icon_source_set_state_wildcarded(
            Handle source, boolean setting);

    native static final protected void gtk_icon_source_set_size_wildcarded(
            Handle source, boolean setting);

    native static final protected boolean gtk_icon_source_get_size_wildcarded(
            Handle source);

    native static final protected boolean gtk_icon_source_get_state_wildcarded(
            Handle source);

    native static final protected boolean gtk_icon_source_get_direction_wildcarded(
            Handle source);

    native static final protected void gtk_icon_source_set_direction(
            Handle source, int direction);

    native static final protected void gtk_icon_source_set_state(Handle source,
            int state);

    native static final protected void gtk_icon_source_set_size(Handle source,
            int size);

    native static final protected int gtk_icon_source_get_direction(
            Handle source);

    native static final protected int gtk_icon_source_get_state(Handle source);

    native static final protected int gtk_icon_source_get_size(Handle source);

    native static final private int gtk_icon_source_get_type();

    native static final private void gtk_icon_source_set_icon_name(
            Handle source, String icon_name);

    native static final private String gtk_icon_source_get_icon_name(
            Handle source);

}
