/* graph-utility.c
 * Copyright (C) 2004 Vivien Malerba <malerba@gnome-db.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include "graph-utility.h"
#include <math.h>

static gboolean compute_intersect_rect_line (gdouble rectx1, gdouble recty1, gdouble rectx2, gdouble recty2,
					     gdouble P1x, gdouble P1y, gdouble P2x, gdouble P2y,
					     gdouble *R1x, gdouble *R1y, gdouble *R2x, gdouble *R2y);

static void     compute_text_marks_offsets (gdouble x1, gdouble y1, gdouble x2, gdouble y2,
					    gdouble *xoff, gdouble *yoff, GtkAnchorType *anchor_type);

/*
 * Computes the points' coordinates of the line going from
 * @ref_pk_ent to @fk_ent (which are themselves rectangles)
 *
 * Returns a list of CanvasShape structures, use free_canvas_shapes() to free it.
 */
GSList *
graph_util_compute_anchor_shapes (GnomeDbCanvasEntity *fk_ent, GnomeDbCanvasEntity *ref_pk_ent, 
				  guint nb_anchors, guint ext)
{
	GSList *retval = NULL;
	guint i;
	gdouble fx1, fy1, fx2, fy2; /* FK entity item (bounds) */
	gdouble rx1, ry1, rx2, ry2; /* REF PK entity item  (bounds) */

	gdouble rcx, rcy; /* center of ref_pk entity item */
	gdouble cx, cy;

	gdouble rux, ruy; /* current ref_pk point for the arrow line */
	gdouble dx, dy; /* increments to compute the new ref_pk point for the arrow line */

	g_return_val_if_fail (nb_anchors > 0, NULL);

	gnome_canvas_item_get_bounds (GNOME_CANVAS_ITEM (fk_ent), &fx1, &fy1, &fx2, &fy2);
	gnome_canvas_item_get_bounds (GNOME_CANVAS_ITEM (ref_pk_ent), &rx1, &ry1, &rx2, &ry2);

	/* compute the cx, cy, dx and dy values */
	rcx = (rx1 + rx2) / 2.;
	rcy = (ry1 + ry2) / 2.;
	cx = (fx1 + fx2) / 2.;
	cy = (fy1 + fy2) / 2.;
	rux = rcx;
	ruy = rcy;
	dx = 0;
	dy = 0;

	for (i = 0; i < nb_anchors; i++) {
		/* TODO:
		   - detect tables overlapping
		*/		
		if ((rcx == cx) && (rcy == cy)) {
			/* tables have the same center (includes case when they are equal) */
			gdouble Dy, Dx;
			GnomeCanvasPoints *ap, *points;
			CanvasShape *shape = g_new0 (CanvasShape, 1);
			
			points = gnome_canvas_points_new (4);
			ap = gnome_canvas_points_new (4);

			Dy = (ry2 - ry1) / 2. / (gdouble ) (nb_anchors + 1) * (gdouble) (i + 1);
			Dx = (rx2 - rx1) * (0.8 + 0.1 * i);
			g_assert (compute_intersect_rect_line (rx1, ry1, rx2, ry2,
							       cx, cy, cx + Dx, cy - Dy,
							       &(ap->coords[0]), &(ap->coords[1]),
							       &(ap->coords[2]), &(ap->coords[3])));
			
			if (ap->coords[0] > ap->coords[2]) {
				points->coords[0] = ap->coords[0];
				points->coords[1] = ap->coords[1];
			}
			else {
				points->coords[0] = ap->coords[2];
				points->coords[1] = ap->coords[3];
			}

			points->coords[2] = cx + Dx;
			points->coords[3] = cy - Dy;

			Dy = (fy2 - fy1) / 2. / (gdouble ) (nb_anchors + 1) * (gdouble) (i + 1);
			Dx = (fx2 - fx1) * (0.8 + 0.1 * i);
			points->coords[4] = cx + Dx;
			points->coords[5] = cy + Dy;

			g_assert (compute_intersect_rect_line (fx1, fy1, fx2, fy2,
							       cx, cy, cx + Dx, cy + Dy,
							       &(ap->coords[0]), &(ap->coords[1]),
							       &(ap->coords[2]), &(ap->coords[3])));
			
			if (ap->coords[0] > ap->coords[2]) {
				points->coords[6] = ap->coords[0];
				points->coords[7] = ap->coords[1];
			}
			else {
				points->coords[6] = ap->coords[2];
				points->coords[7] = ap->coords[3];
			}
			
			shape->d.points = points;
			shape->type = GNOME_TYPE_CANVAS_LINE;
			gnome_canvas_points_free (ap);

			retval = g_slist_append (retval, shape);

			/* extension marks as text */
			if (ext & CANVAS_SHAPE_EXT_JOIN_OUTER_1) {
				ap = gnome_canvas_points_new (2);
				ap->coords[0] = points->coords[2] + 5.;
				ap->coords[1] = points->coords[3] - 5.;

				shape = g_new0 (CanvasShape, 1);
				shape->type = GNOME_TYPE_CANVAS_TEXT;
				shape->d.points = ap;
				shape->text = g_strdup ("*");
				shape->text_anchor = GTK_ANCHOR_SOUTH;
				retval = g_slist_append (retval, shape);
			}

			if (ext & CANVAS_SHAPE_EXT_JOIN_OUTER_2) {
				ap = gnome_canvas_points_new (2);
				ap->coords[0] = points->coords[4] + 5.;
				ap->coords[1] = points->coords[5] + 5.;

				shape = g_new0 (CanvasShape, 1);
				shape->type = GNOME_TYPE_CANVAS_TEXT;
				shape->d.points = ap;
				shape->text = g_strdup ("*");
				shape->text_anchor = GTK_ANCHOR_NORTH;
				retval = g_slist_append (retval, shape);
			}

			/* handle in the middle */
			retval = g_slist_concat (retval, 
						 graph_util_compute_handle_shapes (points->coords[2], points->coords[3], 
										   points->coords[4], points->coords[5]));

			/* GnomeCanvasPathDef *path_def; */
			
			/* path_def = gnome_canvas_path_def_new(); */
			
			/* gnome_canvas_path_def_moveto(path_def, 500.0, 175.0); */
			/* gnome_canvas_path_def_curveto(path_def, 550.0, 175.0, 550.0, 275.0, 500.0, 275.0); */
			/* shape->d.path_def = path_def; */
			
		}
		else {
			GnomeCanvasPoints *ap, *points;
			CanvasShape *shape = g_new0 (CanvasShape, 1);

			points = gnome_canvas_points_new (2);
			ap = gnome_canvas_points_new (4);

			if (nb_anchors > 1) {
				if ((dx == 0) && (dy == 0)) {
					/* compute perpendicular to D={(rcx, rcy), (cx, cy)} */
					gdouble vx = (rcx - cx), vy = (rcy - cy);
					gdouble tmp;
					
					tmp = vx;
					vx = vy;
					vy = - tmp;
					
					/* compute intersect of ref_pkey rectangle and D=[vx, vy] passing at (rcx, rcy) */
					g_assert (compute_intersect_rect_line (rx1, ry1, rx2, ry2,
									       rcx, rcy, rcx + vx, rcy + vy,
									       &(ap->coords[0]), &(ap->coords[1]),
									       &(ap->coords[2]), &(ap->coords[3])));
					dx = (ap->coords[2] - ap->coords[0]) / (gdouble) (nb_anchors  + 1);
					dy = (ap->coords[3] - ap->coords[1]) / (gdouble) (nb_anchors  + 1);
					rux = ap->coords[0];
					ruy = ap->coords[1];
				}

				rux += dx;
				ruy += dy;
			}
			
			/* compute the 4 intersection points */
			g_assert (compute_intersect_rect_line (rx1, ry1, rx2, ry2,
							       rux, ruy, cx, cy,
							       &(ap->coords[0]), &(ap->coords[1]),
							       &(ap->coords[2]), &(ap->coords[3])));
			g_assert (compute_intersect_rect_line (fx1, fy1, fx2, fy2,
							       rux, ruy, cx, cy,
							       &(ap->coords[4]), &(ap->coords[5]),
							       &(ap->coords[6]), &(ap->coords[7])));
			
			/* choosing between point coords(0,1) and coords(2,3) */
			if (((ap->coords[0] - ap->coords[4]) * (ap->coords[0] - ap->coords[4]) + 
			     (ap->coords[1] - ap->coords[5]) * (ap->coords[1] - ap->coords[5])) <
			    ((ap->coords[2] - ap->coords[4]) * (ap->coords[2] - ap->coords[4]) + 
			     (ap->coords[3] - ap->coords[5]) * (ap->coords[3] - ap->coords[5]))) {
				points->coords[0] = ap->coords[0];
				points->coords[1] = ap->coords[1];
			}
			else {
				points->coords[0] = ap->coords[2];
				points->coords[1] = ap->coords[3];
			}
			
			/* choosing between point coords(4,5) and coords(6,7) */
			if (((points->coords[0] - ap->coords[4]) * (points->coords[0] - ap->coords[4]) +
			     (points->coords[1] - ap->coords[5]) * (points->coords[1] - ap->coords[5])) <
			    ((points->coords[0] - ap->coords[6]) * (points->coords[0] - ap->coords[6]) +
			     (points->coords[1] - ap->coords[7]) * (points->coords[1] - ap->coords[7]))) {
				points->coords[2] = ap->coords[4];
				points->coords[3] = ap->coords[5];
			}
			else {
				points->coords[2] = ap->coords[6];
				points->coords[3] = ap->coords[7];
			}
			
			shape->d.points = points;
			shape->type = GNOME_TYPE_CANVAS_LINE;
			gnome_canvas_points_free (ap);
			retval = g_slist_append (retval, shape);

			/* extension marks as text */
			if (ext & CANVAS_SHAPE_EXT_JOIN_OUTER_1) {
				gdouble mxoff = 0., myoff = 0.;
				GtkAnchorType atype;

				compute_text_marks_offsets (points->coords[0], points->coords[1], 
							    points->coords[2], points->coords[3],
							    &mxoff, &myoff, &atype);

				ap = gnome_canvas_points_new (2);
				ap->coords[0] = points->coords[2] + mxoff;
				ap->coords[1] = points->coords[3] + myoff;

				shape = g_new0 (CanvasShape, 1);
				shape->type = GNOME_TYPE_CANVAS_TEXT;
				shape->d.points = ap;
				shape->text = g_strdup ("*");
				shape->text_anchor = GTK_ANCHOR_NORTH;
				retval = g_slist_append (retval, shape);
			}

			if (ext & CANVAS_SHAPE_EXT_JOIN_OUTER_2) {
				gdouble mxoff, myoff;
				GtkAnchorType atype;
				
				compute_text_marks_offsets (points->coords[2], points->coords[3], 
							    points->coords[0], points->coords[1],
							    &mxoff, &myoff, &atype);

				ap = gnome_canvas_points_new (2);
				ap->coords[0] = points->coords[0] + mxoff;
				ap->coords[1] = points->coords[1] + myoff;

				shape = g_new0 (CanvasShape, 1);
				shape->type = GNOME_TYPE_CANVAS_TEXT;
				shape->d.points = ap;
				shape->text = g_strdup ("*");
				shape->text_anchor = atype;
				retval = g_slist_append (retval, shape);
			}

			/* handle in the middle */
			retval = g_slist_concat (retval, 
						 graph_util_compute_handle_shapes (points->coords[0], points->coords[1], 
										   points->coords[2], points->coords[3]));
		}
	}

	return retval;
}

/*
 * Computes the position offsets, relative to X2=(x2, y2) of a text to be written
 * close to the X2 point, also computes the anchor type
 */
static void
compute_text_marks_offsets (gdouble x1, gdouble y1, gdouble x2, gdouble y2,
			    gdouble *xoff, gdouble *yoff, GtkAnchorType *anchor_type)
{
	gdouble mxoff, myoff;
	GtkAnchorType atype = GTK_ANCHOR_CENTER; /* FIXME */
	gdouble sint, cost;
	gdouble sina = 0.5;
	gdouble cosa = 0.866025; /* sqrt(3)/2 */
	gdouble hyp;
	gdouble d = 15.;

	hyp = sqrt ((y2 - y1) * (y2 - y1) + (x2 - x1) * (x2 - x1));
	sint = - (y2 - y1) / hyp;
	cost = (x2 - x1) / hyp;
	
	mxoff = -d * (sina * sint + cosa * cost);
	myoff = -d * (sina * cost - cosa * sint);

	if (xoff)
		*xoff = mxoff;
	if (yoff)
		*yoff = myoff;
	if (anchor_type)
		*anchor_type = atype;
}

/*
 * Computes the points of intersection between a rectangle (defined by the first 2 points)
 * and a line (defined by the next 2 points).
 *
 * The result is returned in place of the line's point definition
 *
 *             --------- -----   D1
 *             |       |
 *             |       |
 *             |       |
 *             |       |
 *             |       |
 *             |       |
 *             |       |
 *             --------- ----   D2
 *             
 *             |       |
 *             |       |
 *             D3      D4
 *
 * Returns: TRUE if the line crosses the rectangle, and FALSE if it misses it.
 */
static gboolean
compute_intersect_rect_line (gdouble rectx1, gdouble recty1, gdouble rectx2, gdouble recty2,
			     gdouble P1x, gdouble P1y, gdouble P2x, gdouble P2y,
			     gdouble *R1x, gdouble *R1y, gdouble *R2x, gdouble *R2y)
{
	gboolean retval = FALSE;
	gboolean rotated = FALSE;
	gdouble a=0.; /* line slope   y = a x + b */
	gdouble b;    /* line offset  y = a x + b */
	gdouble offset = 2.;
		
	gdouble ptsx[4]; /* points' X coordinate: 0 for intersect with D1, 1 for D2,... */
	gdouble ptsy[4]; /* points' Y coordinate */

	g_return_val_if_fail ((rectx1 != rectx2) || (recty1 != recty2), FALSE);
	g_return_val_if_fail ((rectx1 < rectx2) && (recty1 < recty2), FALSE);
	g_return_val_if_fail ((P1x != P2x) || (P1y != P2y), FALSE);

	/* rotate the coordinates to invert X and Y to avoid rounding problems ? */
	if (P1x != P2x)
		a = (P1y - P2y) / (P1x - P2x);
	if ((P1x == P2x) || (fabs (a) > 1)) {
		gdouble tmp;
		rotated = TRUE;
		tmp = rectx1; rectx1 = recty1; recty1 = tmp;
		tmp = rectx2; rectx2 = recty2; recty2 = tmp;
		tmp = P1x; P1x = P1y; P1y = tmp;
		tmp = P2x; P2x = P2y; P2y = tmp;
		a = (P1y - P2y) / (P1x - P2x);
	}

	/* here we have (P1x != P2x), non vertical line */
	b = P1y - a * P1x;

	if (a == 0) {
		/* horizontal line */

		if ((b <= recty2) && (b >= recty1)) {
			retval = TRUE;
			*R1x = rectx1 - offset; *R1y = b;
			*R2x = rectx2 + offset; *R2y = b;
		}
	}
	else {
		gdouble retx[2] = {0., 0.};
		gdouble rety[2] = {0., 0.};
		gint i = 0;

		/* non horizontal and non vertical line */
		/* D1 */
		ptsy[0] = recty1 - offset;
		ptsx[0] = (recty1 - b) / a;

		/* D2 */
		ptsy[1] = recty2 + offset;
		ptsx[1] = (recty2 - b) / a;

		/* D3 */
		ptsx[2] = rectx1 - offset;
		ptsy[2] = a * rectx1 + b;

		/* D4 */
		ptsx[3] = rectx2 + offset;
		ptsy[3] = a * rectx2 + b;
		
		if ((ptsx[0] >= rectx1) && (ptsx[0] <= rectx2)) {
			retval = TRUE;
			retx[i] = ptsx[0]; rety[i] = ptsy[0];
			i ++;
		}
		if ((ptsx[1] >= rectx1) && (ptsx[1] <= rectx2)) {
			retval = TRUE;
			retx[i] = ptsx[1]; rety[i] = ptsy[1];
			i ++;
		}
		if ((i<2) && (ptsy[2] >= recty1) && (ptsy[2] <= recty2)) {
			retval = TRUE;
			retx[i] = ptsx[2]; rety[i] = ptsy[2];
			i ++;
		}
		if ((i<2) && (ptsy[3] >= recty1) && (ptsy[3] <= recty2)) {
			retval = TRUE;
			retx[i] = ptsx[3]; rety[i] = ptsy[3];
			i++;
		}

		if (retval) {
			g_assert (i == 2); /* wee need 2 points! */
			*R1x = retx[0]; *R1y = rety[0];
			*R2x = retx[1]; *R2y = rety[1];
		}
	}

	if (retval && rotated) {
		/* rotate it back */
		gdouble tmp;

		tmp = *R1x; *R1x = *R1y; *R1y = tmp;
		tmp = *R2x; *R2x = *R2y; *R2y = tmp;
	}

	return retval;
}

/*
 * Compute the anchor shapes to link field1 to field2 from ent1 to ent2
 */
GSList *
graph_util_compute_connect_shapes (GnomeDbCanvasEntity *ent1, GdaEntityField *field1, GnomeDbCanvasEntity *ent2, GdaEntityField *field2,
				   guint ext)
{
	GSList *retval;
	CanvasShape *shape;
	GnomeCanvasPoints *points;
	gdouble xl1, xr1, xl2, xr2, yt1, yt2; /* X boundings and Y top of ent1 and ent2 */
	gdouble x1, x2; /* X positions of the lines extremities close to ent1 and ent2 */
	gdouble x1offset, x2offset; /* offsets for the horizontal part of the lines */
	double sq = 5.;
	double eps = 0.5;

	/* line made of 4 points */
	shape = g_new0 (CanvasShape, 1);
	points = gnome_canvas_points_new (4);
	gnome_canvas_item_get_bounds (GNOME_CANVAS_ITEM (ent1), &xl1, &yt1, &xr1, NULL);
	gnome_canvas_item_get_bounds (GNOME_CANVAS_ITEM (ent2), &xl2, &yt2, &xr2, NULL);

	if (xl2 > xr1) {
		x1 = xr1 + eps;
		x2 = xl2 - eps;
		x1offset = 2 * sq;
		x2offset = -2 * sq;
	}
	else {
		if (xl1 >= xr2) {
			x1 = xl1 - eps;
			x2 = xr2 + eps;
			x1offset = - 2 * sq;
			x2offset = 2 * sq;
		}
		else {
			if ((xl1 + xr1) < (xl2 + xr2)) {
				x1 = xl1 - eps;
				x2 = xl2 - eps;
				x1offset = -2 * sq;
				x2offset = -2 * sq;
			}
			else {
				x1 = xr1 + eps;
				x2 = xr2 + eps;
				x1offset = 2 * sq;
				x2offset = 2 * sq;
			}
		}
	}

	points->coords[0] = x1;
	points->coords[1] = gnome_db_canvas_entity_get_field_ypos (ent1, field1) + yt1;

	points->coords[2] = x1 + x1offset;
	points->coords[3] = points->coords[1];

	points->coords[4] = x2 + x2offset;
	points->coords[5] = gnome_db_canvas_entity_get_field_ypos (ent2, field2) + yt2;

	points->coords[6] = x2;
	points->coords[7] = points->coords[5];

	shape->d.points = points;
	shape->type = GNOME_TYPE_CANVAS_LINE;
	retval = g_slist_append (NULL, shape);

	/* extension marks as text */
	if (ext & CANVAS_SHAPE_EXT_JOIN_OUTER_1) {
		gdouble mxoff = 0., myoff = 0.;
		GtkAnchorType atype;
		GnomeCanvasPoints *ap;
		
		compute_text_marks_offsets (points->coords[4], points->coords[5], 
					    points->coords[2], points->coords[3],
					    &mxoff, &myoff, &atype);
		
		ap = gnome_canvas_points_new (2);
		ap->coords[0] = points->coords[2] + mxoff;
		ap->coords[1] = points->coords[3] + myoff;
		
		shape = g_new0 (CanvasShape, 1);
		shape->type = GNOME_TYPE_CANVAS_TEXT;
		shape->d.points = ap;
		shape->text = g_strdup ("*");
		shape->text_anchor = GTK_ANCHOR_NORTH;
		retval = g_slist_append (retval, shape);
	}
	
	if (ext & CANVAS_SHAPE_EXT_JOIN_OUTER_2) {
		gdouble mxoff, myoff;
		GtkAnchorType atype;
		GnomeCanvasPoints *ap;
		
		compute_text_marks_offsets (points->coords[2], points->coords[3], 
					    points->coords[4], points->coords[5],
					    &mxoff, &myoff, &atype);
		
		ap = gnome_canvas_points_new (2);
		ap->coords[0] = points->coords[4] + mxoff;
		ap->coords[1] = points->coords[5] + myoff;
		
		shape = g_new0 (CanvasShape, 1);
		shape->type = GNOME_TYPE_CANVAS_TEXT;
		shape->d.points = ap;
		shape->text = g_strdup ("*");
		shape->text_anchor = atype;
		retval = g_slist_append (retval, shape);
	}
	
	
	/* handle in the middle */
	retval = g_slist_concat (retval, 
				 graph_util_compute_handle_shapes (points->coords[2], points->coords[3], 
								   points->coords[4], points->coords[5]));
 

	/* g_print ("(%.2f,%.2f) -> (%.2f,%.2f) \n", */
/* 		 points->coords[0], points->coords[1], */
/* 		 points->coords[2], points->coords[3]); */

	return retval;
}

/*
 * computes a "handle" in the middle of the 2 points passed as argument
 */
GSList *
graph_util_compute_handle_shapes (gdouble x1, gdouble y1, gdouble x2, gdouble y2)
{
	gdouble x, y, sq = 5.;
	CanvasShape *shape;
	GnomeCanvasPoints *points;

	/* rectangle in the middle */
	x = (x1 + x2) / 2.;
	y = (y1 + y2) / 2.;
	shape = g_new0 (CanvasShape, 1);
	points = gnome_canvas_points_new (2);
	points->coords[0] = x - sq;
	points->coords[1] = y - sq;
	points->coords[2] = x + sq;
	points->coords[3] = y + sq;
	shape->d.points = points;
	shape->type = GNOME_TYPE_CANVAS_RECT;

	return g_slist_append (NULL, shape);
}

/*
 * Free the list of points
 */
void
graph_util_free_canvas_shapes (GSList *canvas_shapes)
{
	GSList *list = canvas_shapes;

	if (!list)
		return;

	while (list) {
		CanvasShape *shape = CANVAS_SHAPE (list->data);
		if (shape->type == GNOME_TYPE_CANVAS_BPATH) {
			if (shape->d.path_def)
				gnome_canvas_path_def_unref (shape->d.path_def);
		}
		else {
			if (shape->d.points)
				gnome_canvas_points_free (shape->d.points);
		}
		if (shape->text)
			g_free (shape->text);
		g_free (shape);
		list = g_slist_next (list);
	}
	g_slist_free (canvas_shapes);
}

/*
 * Draws the shapes stored in @canvas_shapes.
 *
 * Returns: a new list of dranw items 
 */
GSList *
graph_util_draw_canvas_shapes (GnomeCanvasGroup *cg, GSList *canvas_shapes)
{
	GSList *list = canvas_shapes;
	GSList *retval = NULL;

	/* drawing each shape */
	while (list) {
		GnomeCanvasItem *item = NULL;
		CanvasShape *shape = CANVAS_SHAPE (list->data);
		
		if (shape->type == GNOME_TYPE_CANVAS_LINE) {
			item = gnome_canvas_item_new (cg,
						      GNOME_TYPE_CANVAS_LINE,
						      "points", shape->d.points,
						      "fill_color", "black",
						      "width_units", 2.,
						      "cap_style", GDK_CAP_ROUND,
						      "smooth", TRUE,
						      NULL);
		}

		if (!item && (shape->type == GNOME_TYPE_CANVAS_RECT)) {
			if (shape->d.points->num_points != 2)
				g_warning ("CanvasShape has %d points for a rectangle/ellipse", 
					   shape->d.points->num_points);
			else
				item = gnome_canvas_item_new (cg,
							      GNOME_TYPE_CANVAS_RECT,
							      "x1", shape->d.points->coords[0],
							      "y1", shape->d.points->coords[1],
							      "x2", shape->d.points->coords[2],
							      "y2", shape->d.points->coords[3],
							      "fill_color", "black",
							      "width_units", 2.,
							      NULL);
		}
		
		if (!item && (shape->type == GNOME_TYPE_CANVAS_BPATH)) {
			item = gnome_canvas_item_new (cg,
						      GNOME_TYPE_CANVAS_BPATH,
						      "bpath", shape->d.path_def,
						      "fill_color", "black",
						      "width_units", 2.,
						      "cap_style", GDK_CAP_ROUND,
						      NULL);
		}

		if (!item && (shape->type == GNOME_TYPE_CANVAS_TEXT)) {
			item = gnome_canvas_item_new (cg,
						      GNOME_TYPE_CANVAS_TEXT,
						      "x", shape->d.points->coords[0],
						      "y", shape->d.points->coords[1],
						      "text", shape->text,
						      "fill_color", "black",
						      NULL);
		}

		if (item) 
			retval = g_slist_prepend (retval, item);
		
		list = g_slist_next (list);
	}

	return g_slist_reverse (retval);
}
