/* gnome-db-entry-pict.c
 *
 * Copyright (C) 2006 Vivien Malerba
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <glib/gi18n-lib.h>
#include <glib/gstdio.h>
#include <gtk/gtk.h>
#include <gdk-pixbuf/gdk-pixdata.h>
#include "gnome-db-entry-pict.h"
#include "gnome-db-entry-filesel.h"
#include <libgda/gda-data-handler.h>
#include <string.h>
#include <unistd.h>
#include <gdk/gdk.h>

#include "gbase64.h"

/* 
 * Main static functions 
 */
static void gnome_db_entry_pict_class_init (GnomeDbEntryPictClass * class);
static void gnome_db_entry_pict_init (GnomeDbEntryPict *srv);
static void gnome_db_entry_pict_dispose (GObject *object);
static void gnome_db_entry_pict_finalize (GObject *object);

/* virtual functions */
static GtkWidget *create_entry (GnomeDbEntryWrapper *mgwrap);
static void       real_set_value (GnomeDbEntryWrapper *mgwrap, const GValue *value);
static GValue    *real_get_value (GnomeDbEntryWrapper *mgwrap);
static void       connect_signals(GnomeDbEntryWrapper *mgwrap, GCallback callback);
static gboolean   expand_in_layout (GnomeDbEntryWrapper *mgwrap);
static void       set_editable (GnomeDbEntryWrapper *mgwrap, gboolean editable);
static gboolean   value_is_equal_to (GnomeDbEntryWrapper *mgwrap, const GValue *value);
static gboolean   value_is_null (GnomeDbEntryWrapper *mgwrap);

/* get a pointer to the parents to be able to call their destructor */
static GObjectClass  *parent_class = NULL;

typedef enum {
	ENCODING_NONE,
	ENCODING_BASE64
} EncodeType;

/* private structure */
struct _GnomeDbEntryPictPrivate
{
	GtkWidget *sw;
	GtkWidget *pict;
	GtkWidget *notice;
	gboolean   editable;
	
	GtkWidget *menu; /* popup */
	GtkWidget *load_mitem;
	GtkWidget *save_mitem;

	guchar    *data;
	glong      data_length;
	EncodeType encoding;
	gboolean   serialize; /* TRUE if must use gdk_pixdata_serialize() */
};


GType
gnome_db_entry_pict_get_type (void)
{
	static GType type = 0;

	if (!type) {
		static const GTypeInfo info = {
			sizeof (GnomeDbEntryPictClass),
			(GBaseInitFunc) NULL,
			(GBaseFinalizeFunc) NULL,
			(GClassInitFunc) gnome_db_entry_pict_class_init,
			NULL,
			NULL,
			sizeof (GnomeDbEntryPict),
			0,
			(GInstanceInitFunc) gnome_db_entry_pict_init
		};
		
		type = g_type_register_static (GNOME_DB_TYPE_ENTRY_WRAPPER, "GnomeDbEntryPict", &info, 0);
	}
	return type;
}

static void
gnome_db_entry_pict_class_init (GnomeDbEntryPictClass * class)
{
	GObjectClass   *object_class = G_OBJECT_CLASS (class);

	parent_class = g_type_class_peek_parent (class);

	object_class->dispose = gnome_db_entry_pict_dispose;
	object_class->finalize = gnome_db_entry_pict_finalize;

	GNOME_DB_ENTRY_WRAPPER_CLASS (class)->create_entry = create_entry;
	GNOME_DB_ENTRY_WRAPPER_CLASS (class)->real_set_value = real_set_value;
	GNOME_DB_ENTRY_WRAPPER_CLASS (class)->real_get_value = real_get_value;
	GNOME_DB_ENTRY_WRAPPER_CLASS (class)->connect_signals = connect_signals;
	GNOME_DB_ENTRY_WRAPPER_CLASS (class)->expand_in_layout = expand_in_layout;
	GNOME_DB_ENTRY_WRAPPER_CLASS (class)->set_editable = set_editable;
	GNOME_DB_ENTRY_WRAPPER_CLASS (class)->value_is_equal_to = value_is_equal_to;
	GNOME_DB_ENTRY_WRAPPER_CLASS (class)->value_is_null = value_is_null;
}

static void
gnome_db_entry_pict_init (GnomeDbEntryPict * gnome_db_entry_pict)
{
	gnome_db_entry_pict->priv = g_new0 (GnomeDbEntryPictPrivate, 1);
	gnome_db_entry_pict->priv->pict = NULL;
	gnome_db_entry_pict->priv->data = NULL;
	gnome_db_entry_pict->priv->data_length = 0;
	gnome_db_entry_pict->priv->encoding = ENCODING_NONE;
	gnome_db_entry_pict->priv->serialize = FALSE;
	gnome_db_entry_pict->priv->editable = TRUE;
}

/**
 * gnome_db_entry_pict_new
 * @dh: the data handler to be used by the new widget
 * @type: the requested data type (compatible with @dh)
 * @options: optional parameters
 *
 * Creates a new widget which is mainly a GtkEntry
 *
 * Returns: the new widget
 */
GtkWidget *
gnome_db_entry_pict_new (GdaDataHandler *dh, GType type, const gchar *options)
{
	GObject *obj;
	GnomeDbEntryPict *mgpict;

	g_return_val_if_fail (GDA_IS_DATA_HANDLER (dh), NULL);
	g_return_val_if_fail (type != G_TYPE_INVALID, NULL);
	g_return_val_if_fail (gda_data_handler_accepts_g_type (dh, type), NULL);

	obj = g_object_new (GNOME_DB_TYPE_ENTRY_PICT, "handler", dh, NULL);
	mgpict = GNOME_DB_ENTRY_PICT (obj);
	gnome_db_data_entry_set_value_type (GNOME_DB_DATA_ENTRY (mgpict), type);

	if (options && *options) {
		GdaQuarkList *params;
		const gchar *str;

		params = gda_quark_list_new_from_string (options);
		str = gda_quark_list_find (params, "ENCODING");
		if (str) {
			if (!strcmp (str, "base64")) 
				mgpict->priv->encoding = ENCODING_BASE64;
		}
		str = gda_quark_list_find (params, "SERIALIZE");
		if (str) {
			if ((*str == 't') || (*str == 'T'))
				mgpict->priv->serialize = TRUE;
		}
		gda_quark_list_free (params);
	}

	return GTK_WIDGET (obj);
}


static void
gnome_db_entry_pict_dispose (GObject   * object)
{
	GnomeDbEntryPict *mgpict;

	g_return_if_fail (object != NULL);
	g_return_if_fail (GNOME_DB_IS_ENTRY_PICT (object));

	mgpict = GNOME_DB_ENTRY_PICT (object);
	if (mgpict->priv) {
		if (mgpict->priv->data) {
			g_free (mgpict->priv->data);
			mgpict->priv->data = NULL;
			mgpict->priv->data_length = 0;
		}

		if (mgpict->priv->menu) {
			gtk_widget_destroy (mgpict->priv->menu);
			mgpict->priv->menu = NULL;
		}
	}

	/* parent class */
	parent_class->dispose (object);
}

static void
gnome_db_entry_pict_finalize (GObject   * object)
{
	GnomeDbEntryPict *mgpict;

	g_return_if_fail (object != NULL);
	g_return_if_fail (GNOME_DB_IS_ENTRY_PICT (object));

	mgpict = GNOME_DB_ENTRY_PICT (object);
	if (mgpict->priv) {
		g_free (mgpict->priv);
		mgpict->priv = NULL;
	}

	/* parent class */
	parent_class->finalize (object);
}

static void display_image (GnomeDbEntryPict *mgpict, const gchar *error_stock, const gchar *notice);
static gboolean popup_menu_cb (GtkWidget *button, GnomeDbEntryPict *mgpict);
static gboolean event_cb (GtkWidget *button, GdkEvent *event, GnomeDbEntryPict *mgpict);

static GtkWidget *
create_entry (GnomeDbEntryWrapper *mgwrap)
{
	GtkWidget *vbox, *wid;
	GnomeDbEntryPict *mgpict;

	g_return_val_if_fail (mgwrap && GNOME_DB_IS_ENTRY_PICT (mgwrap), NULL);
	mgpict = GNOME_DB_ENTRY_PICT (mgwrap);
	g_return_val_if_fail (mgpict->priv, NULL);

	vbox = gtk_vbox_new (FALSE, 0);

	/* sw */
	wid = gtk_scrolled_window_new (NULL, NULL);
	gtk_box_pack_start (GTK_BOX (vbox), wid, TRUE, TRUE, 0);
	gtk_widget_show (wid);
	mgpict->priv->sw = wid;
	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (wid), 
					GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
	gtk_scrolled_window_set_shadow_type (GTK_SCROLLED_WINDOW (wid), GTK_SHADOW_NONE);

	/* image */
	wid = gtk_image_new ();
	gtk_scrolled_window_add_with_viewport (GTK_SCROLLED_WINDOW (mgpict->priv->sw), wid);
	gtk_widget_show (wid);
	mgpict->priv->pict = wid;

	wid = gtk_bin_get_child (GTK_BIN (mgpict->priv->sw));
	gtk_viewport_set_shadow_type (GTK_VIEWPORT (wid), GTK_SHADOW_NONE);

	/* notice, not shown */
	wid = gtk_label_new ("");
	mgpict->priv->notice = wid;
	gtk_box_pack_start (GTK_BOX (vbox), wid, TRUE, TRUE, 0);

	/* connect signals for popup menu */
	g_signal_connect (G_OBJECT (mgpict), "popup-menu",
			  G_CALLBACK (popup_menu_cb), mgpict);
	g_signal_connect (G_OBJECT (mgpict), "event",
			  G_CALLBACK (event_cb), mgpict);

	display_image (mgpict, GTK_STOCK_MISSING_IMAGE, _("No data to display"));

	return vbox;
}

static void file_load_cb (GtkWidget *button, GnomeDbEntryPict *mgpict);
static void file_save_cb (GtkWidget *button, GnomeDbEntryPict *mgpict);
static void
do_popup_menu (GtkWidget *widget, GdkEventButton *event, GnomeDbEntryPict *mgpict)
{
	int button, event_time;
	
	if (!mgpict->priv->menu) {
		GtkWidget *menu, *mitem;

		menu = gtk_menu_new ();
		mgpict->priv->menu = menu;
		g_signal_connect (menu, "deactivate", 
				  G_CALLBACK (gtk_widget_hide), NULL);
		
		mitem = gtk_menu_item_new_with_mnemonic (_("_Load from file"));
		mgpict->priv->load_mitem = mitem;
		gtk_widget_show (mitem);
		gtk_container_add (GTK_CONTAINER (menu), mitem);
		g_signal_connect (mitem, "activate",
				  G_CALLBACK (file_load_cb), mgpict);
		gtk_widget_set_sensitive (mgpict->priv->load_mitem, mgpict->priv->editable);
		
		
		mitem = gtk_menu_item_new_with_mnemonic (_("_Save to file"));
		mgpict->priv->save_mitem = mitem;
		gtk_widget_show (mitem);
		gtk_container_add (GTK_CONTAINER (menu), mitem);
		g_signal_connect (mitem, "activate",
				  G_CALLBACK (file_save_cb), mgpict);
		gtk_widget_set_sensitive (mgpict->priv->save_mitem, mgpict->priv->data ? TRUE : FALSE);

		gtk_menu_attach_to_widget (GTK_MENU (mgpict->priv->menu), widget, NULL);
	}		

	if (event) {
		button = event->button;
		event_time = event->time;
	}
	else {
		button = 0;
		event_time = gtk_get_current_event_time ();
	}
	
	gtk_menu_popup (GTK_MENU (mgpict->priv->menu), NULL, NULL, NULL, NULL, 
			button, event_time);
}

static gboolean
popup_menu_cb (GtkWidget *widget, GnomeDbEntryPict *mgpict)
{
	do_popup_menu (widget, NULL, mgpict);
	return TRUE;
}

static gboolean
event_cb (GtkWidget *widget, GdkEvent *event, GnomeDbEntryPict *mgpict)
{
	if ((event->type == GDK_BUTTON_PRESS) && (((GdkEventButton *) event)->button == 3)) {
		do_popup_menu (widget, (GdkEventButton *) event, mgpict);
		return TRUE;
	}
	if ((event->type == GDK_2BUTTON_PRESS) && (((GdkEventButton *) event)->button == 1))
		file_load_cb (widget, mgpict);
	
	return FALSE;
}

static void
file_load_cb (GtkWidget *widget, GnomeDbEntryPict *mgpict)
{
	GtkWidget *dlg;
	GtkFileFilter *filter;

	dlg = gtk_file_chooser_dialog_new (_("Select a file to load"), 
					   GTK_WINDOW (gtk_widget_get_toplevel (widget)),
					   GTK_FILE_CHOOSER_ACTION_OPEN, 
					   GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
					   GTK_STOCK_OPEN, GTK_RESPONSE_ACCEPT,
					   NULL);
	filter = gtk_file_filter_new ();
	gtk_file_filter_add_pixbuf_formats (filter);
	gtk_file_chooser_set_filter (GTK_FILE_CHOOSER (dlg), filter);
	
	if (gtk_dialog_run (GTK_DIALOG (dlg)) == GTK_RESPONSE_ACCEPT) {
		char *filename;
		gsize length;
		GError *error = NULL;
		gchar *data;

		filename = gtk_file_chooser_get_filename (GTK_FILE_CHOOSER (dlg));

		if (g_file_get_contents (filename, &data, &length, &error)) {
			if (mgpict->priv->data) {
				g_free (mgpict->priv->data);
				mgpict->priv->data = NULL;
				mgpict->priv->data_length = 0;
			}
			
			mgpict->priv->data = (guchar *) data;
			mgpict->priv->data_length = length;
			display_image (mgpict, NULL, NULL);
			gnome_db_entry_wrapper_contents_changed (GNOME_DB_ENTRY_WRAPPER (mgpict));
		}
		else {
			GtkWidget *msg;

			msg = gtk_message_dialog_new_with_markup (GTK_WINDOW (gtk_widget_get_toplevel (widget)), 
								  GTK_DIALOG_MODAL, GTK_MESSAGE_ERROR,
								  GTK_BUTTONS_CLOSE,
								  _("Could not load the contents of '%s':\n %s"), 
								  filename, 
								  error && error->message ? error->message : _("No detail"));
			if (error)
				g_error_free (error);
			gtk_widget_destroy (dlg);
			dlg = NULL;

			gtk_dialog_run (GTK_DIALOG (msg));
			gtk_widget_destroy (msg);
		}
		g_free (filename);
	}
	
	if (dlg)
		gtk_widget_destroy (dlg);

}

static void
file_save_cb (GtkWidget *widget, GnomeDbEntryPict *mgpict)
{
	GtkWidget *dlg;

	dlg = gtk_file_chooser_dialog_new (_("Select a file to save the image to"), 
					   GTK_WINDOW (gtk_widget_get_toplevel (widget)),
					   GTK_FILE_CHOOSER_ACTION_SAVE, 
					   GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
					   GTK_STOCK_OPEN, GTK_RESPONSE_ACCEPT,
					   NULL);
	if (gtk_dialog_run (GTK_DIALOG (dlg)) == GTK_RESPONSE_ACCEPT) {
		char *filename;
		GError *error = NULL;

		filename = gtk_file_chooser_get_filename (GTK_FILE_CHOOSER (dlg));

		if (!g_file_set_contents (filename, mgpict->priv->data, mgpict->priv->data_length, &error)) {
			GtkWidget *msg;

			msg = gtk_message_dialog_new_with_markup (GTK_WINDOW (gtk_widget_get_toplevel (widget)), 
								  GTK_DIALOG_MODAL, GTK_MESSAGE_ERROR,
								  GTK_BUTTONS_CLOSE,
								  _("Could not save the image to '%s':\n %s"), 
								  filename, 
								  error && error->message ? error->message : _("No detail"));
			if (error)
				g_error_free (error);
			gtk_widget_destroy (dlg);
			dlg = NULL;

			gtk_dialog_run (GTK_DIALOG (msg));
			gtk_widget_destroy (msg);
		}
		g_free (filename);
	}
	
	if (dlg)
		gtk_widget_destroy (dlg);

}

static void
real_set_value (GnomeDbEntryWrapper *mgwrap, const GValue *value)
{
	GnomeDbEntryPict *mgpict;
	gchar *stock = NULL;
	gchar *notice_msg = NULL;

	g_return_if_fail (mgwrap && GNOME_DB_IS_ENTRY_PICT (mgwrap));
	mgpict = GNOME_DB_ENTRY_PICT (mgwrap);
	g_return_if_fail (mgpict->priv);

	if (mgpict->priv->data) {
		g_free (mgpict->priv->data);
		mgpict->priv->data = NULL;
		mgpict->priv->data_length = 0;
	}

	/* fill in mgpict->priv->data */
	if (value) {
		if (gda_value_is_null ((GValue *) value)) {
			stock = GTK_STOCK_MISSING_IMAGE;
			notice_msg = _("No data to display");
		}
		else {
			if (G_VALUE_TYPE ((GValue *) value) == GDA_TYPE_BLOB) {
				TO_IMPLEMENT;
				stock = GTK_STOCK_DIALOG_WARNING;
				notice_msg = _("Unhandled type of data");
			}
			else if (G_VALUE_TYPE ((GValue *) value) == GDA_TYPE_BINARY) {
				GdaBinary *bin;

				bin = (GdaBinary *) gda_value_get_binary ((GValue *) value);
				g_assert (bin);

				if (bin->binary_length > 0) {
					mgpict->priv->data = g_new (guchar, bin->binary_length);
					mgpict->priv->data_length = bin->binary_length;
					memcpy (mgpict->priv->data, bin->data, bin->binary_length);
				}
			}
			else if (G_VALUE_TYPE ((GValue *) value) == G_TYPE_STRING) {
				const gchar *str;

				str = g_value_get_string (value);
				if (str) {
					switch (mgpict->priv->encoding) {
					case ENCODING_NONE:
						mgpict->priv->data = g_strdup (str);
						mgpict->priv->data_length = strlen (mgpict->priv->data);
						break;
					case ENCODING_BASE64: {
						gsize out_len;
						mgpict->priv->data = g_base64_decode (str, &out_len);
						if (out_len > 0)
							mgpict->priv->data_length = out_len;
						else {
							g_free (mgpict->priv->data);
							mgpict->priv->data = NULL;
							mgpict->priv->data_length = 0;
						}
						break;
					}
					}
				}
				else {
					stock = GTK_STOCK_MISSING_IMAGE;
					notice_msg = _("Empty data");
				}
			}
			else {
				stock = GTK_STOCK_DIALOG_ERROR;
				notice_msg = _("Unhandled type of data");
			}
		}
	}
	else 
		stock = GTK_STOCK_MISSING_IMAGE;

	/* create (if possible) a pixbuf from mgpict->priv->data */
	display_image (mgpict, stock, notice_msg);
}

static void 
loader_size_prepared_cb (GdkPixbufLoader *loader, gint width, gint height, GnomeDbEntryPict *mgpict)
{
	gint reqw, reqh;

	reqw = mgpict->priv->sw->allocation.width;
	reqh = mgpict->priv->sw->allocation.height;

	if ((reqw < width) || (reqh < height)) {
		gint w, h;
	
		if ((double) height * (double) reqw > (double) width * (double) reqh) {
			w = 0.5 + (double) width * (double) reqh / (double) height;
			h =  reqh;
		} else {
			h = 0.5 + (double) height * (double) reqw / (double) width;
			w =  reqw;
		}
		
		gdk_pixbuf_loader_set_size (loader, w, h);
	}
}

static void 
display_image (GnomeDbEntryPict *mgpict, const gchar *error_stock, const gchar *notice)
{
	const gchar *stock = error_stock;
	gchar *notice_msg = NULL;

	if (mgpict->priv->data) {
		if (mgpict->priv->serialize) {
			GdkPixdata pixdata;
			GError *error = NULL;
			
			if (!gdk_pixdata_deserialize (&pixdata, mgpict->priv->data_length, 
						      mgpict->priv->data, &error)) {
				g_free (mgpict->priv->data);
				mgpict->priv->data = NULL;
				mgpict->priv->data_length = 0;

				stock = GTK_STOCK_DIALOG_ERROR;
				notice_msg = g_strdup_printf (_("Error while deserializing data:\n%s"),
							      error && error->message ? error->message : _("No detail"));
				g_error_free (error);
			}
			else {
				GdkPixbuf *pixbuf;

				pixbuf = gdk_pixbuf_from_pixdata (&pixdata, FALSE, &error);
				if (!pixbuf) {
					notice_msg = g_strdup_printf (_("Error while interpreting data as an image:\n%s"),
								      error && error->message ? error->message : _("No detail"));
					g_error_free (error);
					stock = GTK_STOCK_DIALOG_ERROR;
				}
				else {
					gtk_image_set_from_pixbuf (GTK_IMAGE (mgpict->priv->pict), 
								   pixbuf);
					g_object_unref (pixbuf);
				}
			}
		}
		else {
			GdkPixbufLoader *loader;
			GError *error = NULL;
			
			loader = gdk_pixbuf_loader_new ();
			g_signal_connect (G_OBJECT (loader), "size-prepared",
					  G_CALLBACK (loader_size_prepared_cb), mgpict);
			gdk_pixbuf_loader_write (loader, mgpict->priv->data, mgpict->priv->data_length, NULL);
			if (gdk_pixbuf_loader_close (loader, &error)) {
				GdkPixbuf *pixbuf;
				
				pixbuf = gdk_pixbuf_loader_get_pixbuf (loader);
				if (pixbuf) 
					gtk_image_set_from_pixbuf (GTK_IMAGE (mgpict->priv->pict), 
								   pixbuf);
				else
					stock = GTK_STOCK_MISSING_IMAGE;
			}
			else {
				notice_msg = g_strdup_printf (_("Error while interpreting data as an image:\n%s"),
							      error && error->message ? error->message : _("No detail"));
				g_error_free (error);
				stock = GTK_STOCK_DIALOG_WARNING;
			}
			
			g_object_unref (loader);
		}
	}

	if (stock)
		gtk_image_set_from_stock (GTK_IMAGE (mgpict->priv->pict), 
					  stock, GTK_ICON_SIZE_DIALOG);
	if (notice || notice_msg) {
		gtk_label_set_text (GTK_LABEL (mgpict->priv->notice), notice ? notice : notice_msg);
		gtk_widget_show (mgpict->priv->notice);
	}
	else  
		gtk_widget_hide (mgpict->priv->notice);

	g_free (notice_msg);
	if (mgpict->priv->save_mitem)
		gtk_widget_set_sensitive (mgpict->priv->save_mitem, mgpict->priv->data ? TRUE : FALSE);
}

static GValue *
real_get_value (GnomeDbEntryWrapper *mgwrap)
{
	GValue *value = NULL;
	GnomeDbEntryPict *mgpict;
	GType type;

	g_return_val_if_fail (mgwrap && GNOME_DB_IS_ENTRY_PICT (mgwrap), NULL);
	mgpict = GNOME_DB_ENTRY_PICT (mgwrap);
	g_return_val_if_fail (mgpict->priv, NULL);

	type = gnome_db_data_entry_get_value_type (GNOME_DB_DATA_ENTRY (mgpict));

	if (mgpict->priv->data) {
		if (type == GDA_TYPE_BLOB) {
			TO_IMPLEMENT;
		}
		else if (type == GDA_TYPE_BINARY) {
			GdaBinary *bin;
			
			bin = g_new0 (GdaBinary, 1);
			bin->data = g_new0 (guchar, mgpict->priv->data_length);
			bin->binary_length = mgpict->priv->data_length;
			memcpy (bin->data, mgpict->priv->data, bin->binary_length);
			
			value = gda_value_new (GDA_TYPE_BINARY);
			gda_value_take_binary (value, bin);
		}
		else if (type == G_TYPE_STRING) {
			gchar *str = NULL;

			switch (mgpict->priv->encoding) {
			case ENCODING_NONE:
				str = g_strndup (mgpict->priv->data, mgpict->priv->data_length);
				break;
			case ENCODING_BASE64: 
				str = g_base64_encode (mgpict->priv->data, mgpict->priv->data_length);
				break;
			}
			
			value = gda_value_new (G_TYPE_STRING);
			g_value_take_string (value, str);
		}
	}

	if (!value) {
		/* in case the gda_data_handler_get_value_from_sql() returned an error because
		   the contents of the GtkEntry cannot be interpreted as a GValue */
		value = gda_value_new_null ();
	}

	return value;
}

static void
connect_signals(GnomeDbEntryWrapper *mgwrap, GCallback callback)
{
	/* doe nothing because we manullay call gnome_db_entry_wrapper_contents_changed() */
}

static gboolean
expand_in_layout (GnomeDbEntryWrapper *mgwrap)
{
	return TRUE;
}

static void
set_editable (GnomeDbEntryWrapper *mgwrap, gboolean editable)
{
	GnomeDbEntryPict *mgpict;

	g_return_if_fail (mgwrap && GNOME_DB_IS_ENTRY_PICT (mgwrap));
	mgpict = GNOME_DB_ENTRY_PICT (mgwrap);
	g_return_if_fail (mgpict->priv);
	
	mgpict->priv->editable = editable;
	if (mgpict->priv->load_mitem)
		gtk_widget_set_sensitive (mgpict->priv->load_mitem, editable);
}

static gboolean
value_is_equal_to (GnomeDbEntryWrapper *mgwrap, const GValue *value)
{
	GnomeDbEntryPict *mgpict;

	g_return_val_if_fail (mgwrap && GNOME_DB_IS_ENTRY_PICT (mgwrap), FALSE);
	mgpict = GNOME_DB_ENTRY_PICT (mgwrap);
	g_return_val_if_fail (mgpict->priv, FALSE);
	
	if (value) {
		if (gda_value_is_null (value) && !mgpict->priv->data)
			return TRUE;
		if (G_VALUE_TYPE (value) == GDA_TYPE_BLOB) {
			TO_IMPLEMENT;
			return FALSE;
		}
		if (G_VALUE_TYPE (value) == GDA_TYPE_BINARY) {
			GdaBinary *bin;

			bin = (GdaBinary *) gda_value_get_binary ((GValue *) value);
			g_assert (bin);
			if (mgpict->priv->data)
				return !memcmp (bin->data, mgpict->priv->data, MIN (mgpict->priv->data_length, bin->binary_length));
			else
				return FALSE;
		}
		return FALSE;
	}
	else {
		if (mgpict->priv->data)
			return TRUE;
		else
			return FALSE;
	}
	
	return FALSE;
}

static gboolean
value_is_null (GnomeDbEntryWrapper *mgwrap)
{
	GnomeDbEntryPict *mgpict;

	g_return_val_if_fail (mgwrap && GNOME_DB_IS_ENTRY_PICT (mgwrap), TRUE);
	mgpict = GNOME_DB_ENTRY_PICT (mgwrap);
	g_return_val_if_fail (mgpict->priv, TRUE);

	return mgpict->priv->data ? FALSE : TRUE;
}
