/* GNOME-DB Components
 *
 * Copyright (C) 2000 - 2006 The GNOME Foundation.
 *
 * AUTHORS:
 *      Rodrigo Moya <rodrigo@gnome-db.org>
 *      Vivien Malerba <malerba@gnome-db.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Library General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <glib/gi18n-lib.h>
#include <libgda/gda-client.h>
#include <libgda/gda-config.h>
#include <libgda/gda-data-model-array.h>
#include <gtk/gtkstock.h>
#include <gtk/gtktable.h>
#include <gtk/gtkvbbox.h>
#include <libgnomedb/libgnomedb.h>
#include <libgnomedb/gnome-db-util.h>
#include <libgnomeui/gnome-window-icon.h>
#include "dsn-config.h"
#include "dsn-properties-dialog.h"

#define DSN_CONFIG_DATA "DSN_ConfigData"

typedef struct {
	GtkWidget *title;
	GtkWidget *dsn_list;
	GtkWidget *dialog;
	guint      config_listener;
} DsnConfigPrivate;

static void
free_private_data (gpointer data)
{
	DsnConfigPrivate *priv = (DsnConfigPrivate *) data;

	if (priv->config_listener != 0) {
		gda_config_remove_listener (priv->config_listener);
		priv->config_listener = 0;
	}

	g_free (priv);
}

static void
show_data_sources (DsnConfigPrivate *priv)
{
	GList *dsn_list;
	GList *l;
	GdaDataModel *model;
	GtkTreeViewColumn *col;

	/* create the data model */
	model = gda_data_model_array_new (3);
	gda_data_model_set_column_title (model, 0, _("Name"));
	gda_data_model_set_column_title (model, 1, _("Description"));
	gda_data_model_set_column_title (model, 2, _("Global"));

	dsn_list = gda_config_get_data_source_list ();
	for (l = dsn_list; l != NULL; l = l->next) {
		GList *value_list;
		GdaDataSourceInfo *dsn_info = l->data;
		GValue *tmpval;

		g_value_set_string (tmpval = gda_value_new (G_TYPE_STRING), dsn_info->name);
		value_list = g_list_append (NULL, tmpval);

		g_value_set_string (tmpval = gda_value_new (G_TYPE_STRING), dsn_info->description);
		value_list = g_list_append (value_list, tmpval);

		g_value_set_boolean (tmpval = gda_value_new (G_TYPE_BOOLEAN), dsn_info->is_global);
		value_list = g_list_append (value_list, tmpval);

		gda_data_model_append_values (model, (const GList *) value_list, NULL);

		g_list_foreach (value_list, (GFunc) gda_value_free, NULL);
		g_list_free (value_list);
	}
	g_object_set (G_OBJECT (model), "read_only", TRUE, NULL);
	gda_config_free_data_source_list (dsn_list);

	/* display the new data model */
	gnome_db_data_widget_set_gda_model (GNOME_DB_DATA_WIDGET (priv->dsn_list), model);
	g_object_unref (G_OBJECT (model));

	col = gtk_tree_view_get_column (GTK_TREE_VIEW (priv->dsn_list), 1);
	g_object_set (G_OBJECT (col), "expand", TRUE, NULL);
}

/*
 * Callbacks
 */

static void
config_changed_cb (const gchar *path, gpointer user_data)
{
	show_data_sources (user_data);
}

static void
list_double_clicked_cb (GnomeDbGrid *grid, gint row, gpointer user_data)
{
	dsn_config_edit_properties (GTK_WIDGET (user_data));
}

static void
list_popup_properties_cb (GtkWidget *menu, gpointer user_data)
{
	dsn_config_edit_properties (GTK_WIDGET (user_data));
}

static void
list_popup_delete_cb (GtkWidget *menu, gpointer user_data)
{
	dsn_config_delete (GTK_WIDGET (user_data));
}

static void
list_popup_cb (GnomeDbRawGrid *grid, GtkMenu *menu, gpointer user_data)
{
	GtkWidget *item_delete, *item_properties;
	gboolean ok;
	GList *list;

	item_delete = gnome_db_new_menu_item (GTK_STOCK_DELETE,
					      TRUE,
					      G_CALLBACK (list_popup_delete_cb),
					      user_data);
	item_properties = gnome_db_new_menu_item (GTK_STOCK_PROPERTIES,
						  TRUE,
						  G_CALLBACK (list_popup_properties_cb),
						  user_data);

	list = gnome_db_raw_grid_get_selection (grid);
	ok = list != NULL;
	if (list)
		g_list_free (list);
	gtk_widget_set_sensitive (item_delete, ok);
	gtk_widget_set_sensitive (item_properties, ok);

	gtk_menu_prepend (menu, gtk_separator_menu_item_new ());
	gtk_menu_prepend (menu, item_delete);
	gtk_menu_prepend (menu, gtk_separator_menu_item_new ());
	gtk_menu_prepend (menu, item_properties);
}

/*
 * Public functions
 */

GtkWidget *
dsn_config_new (void)
{
	DsnConfigPrivate *priv;
	GtkWidget *dsn;
	GtkWidget *table;
	GtkWidget *box;
	GtkWidget *button;
	GtkWidget *label;
	gchar *title;

	priv = g_new0 (DsnConfigPrivate, 1);
	dsn = gnome_db_new_vbox_widget (FALSE, 6);
        gtk_container_set_border_width (GTK_CONTAINER (dsn), 6);
	g_object_set_data_full (G_OBJECT (dsn), DSN_CONFIG_DATA, priv,
				(GDestroyNotify) free_private_data);

	/* set up config listener */
	priv->config_listener = gda_config_add_listener ("/apps/libgda/Datasources",
							 (GdaConfigListenerFunc) config_changed_cb,
							 priv);

	/* create the main table */
	table = gnome_db_new_table_widget (3, 1, FALSE);
	gtk_box_pack_start (GTK_BOX (dsn), table, TRUE, TRUE, 0);

	/* title */
	title = g_strdup_printf ("<b>%s</b>\n%s", _("Data Sources"),
				 _("Configured data sources in the system"));
	priv->title = gnome_db_gray_bar_new (title);
	g_free (title);
	gnome_db_gray_bar_set_icon_from_stock (GNOME_DB_GRAY_BAR (priv->title),
					       GNOME_DB_STOCK_DATABASE, GTK_ICON_SIZE_BUTTON);
	gtk_table_attach (GTK_TABLE (table), priv->title, 0, 1, 0, 1,
			  GTK_FILL | GTK_SHRINK,
			  GTK_FILL | GTK_SHRINK,
			  0, 0);
	gtk_widget_show (priv->title);

	/* create the data source list */
	priv->dsn_list = gnome_db_raw_grid_new (NULL);
 	gnome_db_data_widget_column_set_editable (GNOME_DB_DATA_WIDGET (priv->dsn_list), 0, FALSE);
	g_object_set (priv->dsn_list, "info_cell_visible", FALSE, NULL);
	
	gtk_widget_show (priv->dsn_list);
	g_signal_connect (priv->dsn_list, "double_clicked",
			  G_CALLBACK (list_double_clicked_cb), dsn);
	g_signal_connect (priv->dsn_list, "populate_popup",
			  G_CALLBACK (list_popup_cb), dsn);
	gtk_table_attach (GTK_TABLE (table), priv->dsn_list, 0, 1, 1, 2,
			  GTK_FILL | GTK_SHRINK | GTK_EXPAND,
			  GTK_FILL | GTK_SHRINK | GTK_EXPAND,
			  0, 0);
	show_data_sources (priv);

	/* add tip */
	box = gnome_db_new_hbox_widget (FALSE, 6);
        gtk_container_set_border_width (GTK_CONTAINER (box), 6);
	gtk_table_attach (GTK_TABLE (table), box, 0, 1, 2, 3,
			  GTK_FILL,
			  GTK_FILL,
                          0, 0);

	button = gtk_image_new_from_stock (GTK_STOCK_DIALOG_INFO, GTK_ICON_SIZE_DIALOG);
        gtk_misc_set_alignment (GTK_MISC (button), 0.5, 0.0);
	gtk_widget_show (button);
	gtk_box_pack_start (GTK_BOX (box), button, FALSE, FALSE, 0);

	label = gnome_db_new_label_widget (
		_("Data sources are the means by which database "
		  "connections are identified in GNOME-DB. All "
		  "information needed to open a connection to "
		  "a specific provider/database combo is stored using "
		  "a unique name. It is by use of this unique name "
		  "you identify the connections in the applications "
		  "that make use of GNOME-DB for database access."));
	gtk_label_set_line_wrap (GTK_LABEL (label), TRUE);
        gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.0);
        gtk_label_set_justify (GTK_LABEL (label), GTK_JUSTIFY_LEFT);
	gtk_box_pack_start (GTK_BOX (box), label, TRUE, TRUE, 0);
	
	return dsn;
}

void
dsn_config_edit_properties (GtkWidget *dsn)
{
	DsnConfigPrivate *priv;
	GdaDataModel *model;
	GList *selection;
	gchar *str;

	priv = g_object_get_data (G_OBJECT (dsn), DSN_CONFIG_DATA);
	
	selection = gnome_db_raw_grid_get_selection (GNOME_DB_RAW_GRID (priv->dsn_list));
	if (!selection)
		return;

	model = gnome_db_data_widget_get_gda_model (GNOME_DB_DATA_WIDGET (priv->dsn_list));
	if (!GDA_IS_DATA_MODEL (model))
		return;

	str = gda_value_stringify ((GValue *) gda_data_model_get_value_at (model, 0, GPOINTER_TO_INT (selection->data)));

	dsn_properties_dialog (GTK_WINDOW (gtk_widget_get_toplevel (dsn)), str);

	g_list_free (selection);
	g_free (str);
}

static void
foreach_remove_cb (GtkTreeModel *tree_model, GtkTreePath *path, GtkTreeIter *iter, DsnConfigPrivate *priv)
{
	GdaDataModel *model;
	GtkWidget *dialog, *label;
	gchar *str1, *str;
	gchar *dsn;
	gint row;

	gtk_tree_model_get (tree_model, iter, DATA_STORE_COL_MODEL_ROW, &row, -1);
	model = gnome_db_data_widget_get_gda_model (GNOME_DB_DATA_WIDGET (priv->dsn_list));
	if (!GDA_IS_DATA_MODEL (model))
		return;

	dsn = gda_value_stringify ((GValue *) gda_data_model_get_value_at (model, 0, row));
	if (dsn == NULL)
		return;

	dialog = gtk_dialog_new_with_buttons (_("Data source delete confirmation"),
					      GTK_WINDOW (gtk_widget_get_toplevel (priv->dsn_list)),
					      GTK_DIALOG_MODAL,
					      GTK_STOCK_YES, GTK_RESPONSE_YES,
					      GTK_STOCK_NO, GTK_RESPONSE_NO, NULL);
	str1 = g_strdup_printf (_("Are you sure you want to remove the data source '%s'?"), dsn);
	str = g_strdup_printf ("<b>Data source removal confirmation:</b>\n\n%s", str1);
	g_free (str1);

	label = gtk_label_new ("");
	gtk_label_set_markup (GTK_LABEL (label), str);
	gtk_widget_show (label);
	gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->vbox), label, TRUE, TRUE, 10);
					      
	gtk_dialog_set_default_response (GTK_DIALOG (dialog),
					 GTK_RESPONSE_NO);
	gtk_widget_show (dialog);

	if (gtk_dialog_run (GTK_DIALOG (dialog)) == GTK_RESPONSE_YES) 
		gda_config_remove_data_source (dsn);

	/* free memory */
	gtk_widget_destroy (dialog);
	g_free (dsn);
	g_free (str);
}

void
dsn_config_delete (GtkWidget *dsn)
{
	DsnConfigPrivate *priv;
	GtkTreeSelection *sel;

	priv = g_object_get_data (G_OBJECT (dsn), DSN_CONFIG_DATA);

	sel = gtk_tree_view_get_selection (GTK_TREE_VIEW (priv->dsn_list));

	gtk_tree_selection_selected_foreach (sel, 
					     (GtkTreeSelectionForeachFunc) foreach_remove_cb, priv);
	show_data_sources (priv);
}

void
dsn_config_select_all (GtkWidget *dsn)
{
	DsnConfigPrivate *priv;
	GtkTreeSelection *sel;

	priv = g_object_get_data (G_OBJECT (dsn), DSN_CONFIG_DATA);

	sel = gtk_tree_view_get_selection (GTK_TREE_VIEW (priv->dsn_list));

	gtk_tree_selection_select_all (sel);
}

void
dsn_config_unselect_all (GtkWidget *dsn)
{
	DsnConfigPrivate *priv;
	GtkTreeSelection *sel;

	priv = g_object_get_data (G_OBJECT (dsn), DSN_CONFIG_DATA);

	sel = gtk_tree_view_get_selection (GTK_TREE_VIEW (priv->dsn_list));

	gtk_tree_selection_unselect_all (sel);
}

gboolean
dsn_config_has_selection (GtkWidget *dsn)
{	
	DsnConfigPrivate *priv;
	GtkTreeSelection *sel;

	priv = g_object_get_data (G_OBJECT (dsn), DSN_CONFIG_DATA);

	sel = gtk_tree_view_get_selection (GTK_TREE_VIEW (priv->dsn_list));

	if (gtk_tree_selection_count_selected_rows (sel) >= 1)
		return TRUE;
	else
		return FALSE;
}
