#include <stdlib.h>
#include <string.h>
#include <glib.h>
#include <libgda/libgda.h>

#define fail(x) g_warning (x)
#define fail_if(x,y) if (x) g_warning (y)
#define fail_unless(x,y) if (!(x)) g_warning (y)

#define CHECK_EXTRA_INFO
static gboolean do_test_load_file (const gchar *filename);

int
main (int argc, char **argv)
{
	int number_failed = 0;
	GDir *dir;
	gchar *dirname;
	GError *err = NULL;
	const gchar *name;

	gda_init ("check-model-import", PACKAGE_VERSION, argc, argv);

	dirname = g_build_filename (CHECK_FILES, "tests", "providers", NULL);
	if (!(dir = g_dir_open (dirname, 0, &err))) {
#ifdef CHECK_EXTRA_INFO
		g_warning ("Could not open directory '%s': %s", dirname,
			   err && err->message ? err->message : "No detail");
#endif
		return EXIT_FAILURE;
	}
	g_free (dirname);

	if (argc == 2) {
		if (g_str_has_suffix (argv[1], ".xml")) {
			g_print ("Tested: %s\n", argv[1]);
			if (!do_test_load_file (argv[1]))
				number_failed ++;
		}
	}
	else 
		while ((name = g_dir_read_name (dir))) {
			if (g_str_has_suffix (name, ".xml")) {
				g_print ("Tested: %s\n", name);
				if (!do_test_load_file (name))
					number_failed ++;
			}
		}
	g_dir_close (dir);

	if (number_failed == 0)
		g_print ("Ok.\n");
	else
		g_print ("%d failed\n", number_failed);
	return (number_failed == 0) ? EXIT_SUCCESS : EXIT_FAILURE;
}

static gboolean
do_test_load_file (const gchar *filename)
{
	gchar *file;
	GdaDataModel *import;
	GSList *errors;
	gboolean retval = TRUE;
	gchar *export, *contents;

	file = g_build_filename (CHECK_FILES, "tests", "providers", filename, NULL);
	import = gda_data_model_import_new_file (file, TRUE, NULL);

	if ((errors = gda_data_model_import_get_errors (GDA_DATA_MODEL_IMPORT (import)))) {
#ifdef CHECK_EXTRA_INFO
		g_warning ("Could not load the expected schema file '%s': ", file);
		for (; errors; errors = errors->next) 
			g_print ("\t%s\n", 
				 ((GError *)(errors->data))->message ? ((GError *)(errors->data))->message : "No detail");
#endif
		retval = FALSE;
		goto out;
	}

	export = gda_data_model_export_to_string (import, GDA_DATA_MODEL_IO_DATA_ARRAY_XML, NULL, 0, NULL, 0, NULL);
	g_assert (g_file_get_contents (file, &contents, NULL, NULL));
	if (strcmp (export, contents)) {
#ifdef CHECK_EXTRA_INFO
		g_print ("========== Imported data model ==========\n");
		gda_data_model_dump (import, stdout);

		g_warning ("Model exported as string differs from loaded data model:");
		g_print ("========== Export as string ==========\n%s\n", export);
		g_file_set_contents ("err.xml", export, -1, NULL);
#endif
		retval = FALSE;
		goto out;
	}
	g_free (export);
	g_free (contents);

 out:
	g_object_unref (import);
	return retval;
}
