# Copyright (C) 2004,2005 by SICEm S.L.
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU Lesser General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

import gobject
import gtk

from gazpacho import util
from gazpacho.constants import DND_TARGETS, \
     TARGET_TYPE_ADAPTOR, TARGET_TYPE_WIDGET, TARGET_TYPE_XML, \
     MIME_TYPE_OBJECT_ADAPTOR, MIME_TYPE_OBJECT, MIME_TYPE_OBJECT_XML
from gazpacho.cursor import Cursor
from gazpacho.popup import PlaceholderPopup

placeholder_xpm = [
    # columns rows colors chars-per-pixel
    "8 8 2 1",
    "  c #bbbbbb",
    ". c #d6d6d6",
    # pixels
    " .  .   ",
    ".    .  ",
    "      ..",
    "      ..",
    ".    .  ",
    " .  .   ",
    "  ..    ",
    "  ..    ",    
    ]

MIN_WIDTH = MIN_HEIGHT = 20

class Placeholder(gtk.Widget):
    __gsignals__ = {
        'realize': 'override',
        'size-request': 'override',
        'size-allocate': 'override',
        'expose-event': 'override',
        'motion-notify-event': 'override',
        'button-press-event': 'override',
        'popup-menu': 'override'
        }

    def __init__(self, app=None):
        gtk.Widget.__init__(self)

        self._app = app
        
        self._xpm_data = placeholder_xpm
        self._pixmap = None
        
        self.set_flags(self.flags() | gtk.CAN_FOCUS)

        # Initialize Drag and Drop
        self.drag_dest_set(gtk.DEST_DEFAULT_MOTION | gtk.DEST_DEFAULT_HIGHLIGHT,
                           DND_TARGETS,
                           gtk.gdk.ACTION_MOVE | gtk.gdk.ACTION_COPY)
        self.connect('drag_drop', self._dnd_drag_drop_cb)
        self.connect('drag_data_received', self._dnd_drag_data_received_cb)

        self.show()

    def do_realize(self):
        self.set_flags(self.flags() | gtk.REALIZED)

        events = (gtk.gdk.EXPOSURE_MASK |
                  gtk.gdk.BUTTON_PRESS_MASK |
                  gtk.gdk.POINTER_MOTION_MASK)
        
        self.window = gtk.gdk.Window(self.get_parent_window(),
                                     x=self.allocation.x,
                                     y=self.allocation.y,
                                     width=self.allocation.width,
                                     height=self.allocation.height,
                                     window_type=gtk.gdk.WINDOW_CHILD,
                                     wclass=gtk.gdk.INPUT_OUTPUT,
                                     visual=self.get_visual(),
                                     colormap=self.get_colormap(),
                                     event_mask=self.get_events() | events)
        
        self.window.set_user_data(self)
        self.style.attach(self.window)
        self.style.set_background(self.window, gtk.STATE_NORMAL)

        if not self._pixmap:
            t = gtk.gdk.pixmap_colormap_create_from_xpm_d(None,
                                                          self.get_colormap(),
                                                          None,
                                                          self._xpm_data)
            self._pixmap = t[0] # t[1] is the transparent color

        self.window.set_back_pixmap(self._pixmap, False)

    def get_project(self):
        """Get the project to which this placeholder belong."""
        project = None
        parent = util.get_parent(self)
        if parent:
            project = parent.project
        return project

    def _dnd_drag_drop_cb(self, target_widget, context, x, y, time):
        """Callback for handling the 'drag_drop' event."""
        
        if not context.targets:
            return False

        if context.get_source_widget():
            # For DnD within the application we request to use the
            # widget or adaptor directly
            if MIME_TYPE_OBJECT_ADAPTOR in context.targets:
                mime_type = MIME_TYPE_OBJECT_ADAPTOR
            else:
                mime_type = MIME_TYPE_OBJECT
        else:
            # otherwise we'll have request the data to be passed as XML
            mime_type = MIME_TYPE_OBJECT_XML
            
        target_widget.drag_get_data(context, mime_type, time)

        return True
        

    def _dnd_drag_data_received_cb(self, target_widget, context, x, y, data,
                                   info, time):
        """Callback for handling the 'drag_data_received' event. If
        the drag/drop is in the same project the target handles
        everything. If it's between different projects the target only
        handles the drop (paste) and the source takes care of the drag
        (cut)."""
        
        from gazpacho.widget import load_widget_from_xml
        from gazpacho.widget import copy_widget
        from gazpacho.widget import Widget

        # If there is no data we indicate that the drop was not
        # successful
        if not data.data:
            context.finish(False, False, time)
            return

        # The DnD is done within a single application so we can use
        # the drag widget directly.
        if info == TARGET_TYPE_WIDGET:
            gtk_source = context.get_source_widget()            
            gsource = Widget.from_widget(gtk_source)

            # Get the correct widget to drag. Sometimes this differ
            # from the source widget
            gwidget = gsource.dnd_widget

            # If we can't get the widget we indicate that the drop was
            # not successful
            if not gwidget:
                context.finish(False, False, time)
                return

            # When DnD in the same project we move the widget
            if self.get_project() == gwidget.project:
                self._app._command_manager.execute_drag_drop(gwidget, self)
                # Do not tell the source to delete the widget. We
                # handle this ourselves
                context.finish(True, False, time)
            # When DnD between projects we copy the widget
            else:
                gwidget_copy = copy_widget(gwidget, self.get_project())
                self._app._command_manager.execute_drop(gwidget_copy, self)
                context.finish(True, False, time)

        # The DnD is done between two applications and we only get the
        # widget as an XML representation. We only copy the widget
        elif info == TARGET_TYPE_XML:
            gwidget = load_widget_from_xml(data.data, self.get_project())
            if not gwidget:
                # If we can't get the widget we indicate that the drop
                # was not successful
                context.finish(False, False, time)
                return

            self._app._command_manager.execute_drop(gwidget, self)
            context.finish(True, False, time)

        elif info == TARGET_TYPE_ADAPTOR:
            mgr = self._app.get_command_manager()
            mgr.create(self._app.add_class, self, None)
            context.finish(True, False, time)
            
        # We don't know how to handle the data and ndicate that the
        # drop was not successful
        else:
            context.finish(False, False, time)

    def do_size_allocate(self, allocation):
        self.allocation = allocation
        if self.flags() & gtk.REALIZED:
            self.window.move_resize(*allocation)

    def do_size_request(self, requisition):
        requisition.width = MIN_WIDTH
        requisition.height = MIN_HEIGHT
        
    def do_expose_event(self, event):
        light_gc = self.style.light_gc[gtk.STATE_NORMAL]
        dark_gc = self.style.dark_gc[gtk.STATE_NORMAL]
        w, h = event.window.get_size()

        # These lines make the Placeholder looks like a button
        event.window.draw_line(light_gc, 0, 0, w - 1, 0)
        event.window.draw_line(light_gc, 0, 0, 0, h - 1)
        event.window.draw_line(dark_gc, 0, h -1, w - 1, h - 1)
        event.window.draw_line(dark_gc, w - 1, 0, w - 1, h - 1)

        util.draw_annotations(self, event.window)
        
        return False

    def do_motion_notify_event(self, event):
        if self._app is None:
            Cursor.set_for_widget_adaptor(event.window, None)
        else:
            Cursor.set_for_widget_adaptor(event.window, self._app.add_class)

        return False

    def do_button_press_event(self, event):
        if not self.flags() & gtk.HAS_FOCUS:
            self.grab_focus()

        if self._app is None:
            return True
        
        if event.button == 1 and event.type == gtk.gdk.BUTTON_PRESS:
            if self._app.add_class:
                # A widget type is selected in the palette.
                # Add a new widget of that type
                mgr = self._app.get_command_manager()
                mgr.create(self._app.add_class, self, None)

            # Shift clicking circles through the widget tree by
            # choosing the parent of the currently selected widget.
            elif event.state & gtk.gdk.SHIFT_MASK:
                util.circle_select(self)

            # otherwise we should be able to select placeholder
            # for paste operations  
            else:
                parent = util.get_parent(self)
                parent.project.selection_set(self, True)
                
        elif event.button == 3 and event.type == gtk.gdk.BUTTON_PRESS:
            mgr = self._app.get_command_manager()
            popup = PlaceholderPopup(mgr, self)
            popup.pop(event)

        return True

    def do_popup_menu(self):
        mgr = self._app.get_command_manager()
        popup = PlaceholderPopup(mgr, self)
        popup.pop(None)
        return True

gobject.type_register(Placeholder)

if __name__ == '__main__':
    win = gtk.Window()
    win.connect('destroy', gtk.main_quit)

    p = Placeholder()
    win.add(p)
    win.show_all()

    gtk.main()
