/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/* privileges-table.c: this file is part of users-admin, a ximian-setup-tool frontend 
 * for user administration.
 * 
 * Copyright (C) 2004 Carlos Garnacho
 * Copyright (C) 2005 Carlos Garnacho, Sivan Greenberg
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA.
 *
 * Authors: Carlos Garnacho Parro <carlosg@gnome.org>
 *          Sivan Greenberg       <sivan@workaround.org>
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <gtk/gtk.h>

#include "gst.h"
#include "privileges-table.h"
#include "user_group.h"

extern GstTool *tool;

static void
on_user_privilege_toggled (GtkCellRendererToggle *cell, gchar *path_str, gpointer data)
{
	GtkTreeModel *model = (GtkTreeModel*) data;
	GtkTreePath  *path  = gtk_tree_path_new_from_string (path_str);
	GtkTreeIter   iter;
	gboolean      value;

	gtk_tree_model_get_iter (GTK_TREE_MODEL (model), &iter, path);
	gtk_tree_model_get (model, &iter, 0, &value, -1);
	gtk_list_store_set (GTK_LIST_STORE (model), &iter, 0, !value, -1);
}

static void
create_privileges_table (GtkWidget *list)
{
	GtkTreeModel      *model;
	GtkCellRenderer   *renderer;
	GtkTreeViewColumn *column;
	GtkTreeIter        iter;

	model = GTK_TREE_MODEL (gtk_list_store_new (3, G_TYPE_BOOLEAN, G_TYPE_STRING, G_TYPE_STRING));

	gtk_tree_view_set_model (GTK_TREE_VIEW (list), model);
	g_object_unref (model);

	column = gtk_tree_view_column_new ();

	renderer = gtk_cell_renderer_toggle_new ();
	gtk_tree_view_column_pack_start (column, renderer, FALSE);
	gtk_tree_view_column_set_attributes (column,
					     renderer,
					     "active", 0,
					     NULL);
	g_signal_connect (G_OBJECT (renderer), "toggled", G_CALLBACK (on_user_privilege_toggled), model);

	renderer = gtk_cell_renderer_text_new ();
	gtk_tree_view_column_pack_end (column, renderer, TRUE);
	gtk_tree_view_column_set_attributes (column,
					     renderer,
					     "text", 1,
					     NULL);

	gtk_tree_view_column_set_sort_column_id (column, 1);
	gtk_tree_view_insert_column (GTK_TREE_VIEW (list), column, 0);
	gtk_tree_view_column_clicked (column);
}

void
create_user_privileges_table (void)
{
	GtkWidget *list = gst_dialog_get_widget (tool->main_dialog, "user_privileges");
	create_privileges_table (list);
}

void
create_profile_privileges_table (void)
{
	GtkWidget *list = gst_dialog_get_widget (tool->main_dialog, "profile_privileges");
	create_privileges_table (list);
}

void
populate_privileges_table (GtkWidget *list, gchar *username)
{
	GtkTreeModel *model = gtk_tree_view_get_model (GTK_TREE_VIEW (list));
	xmlNodePtr    root = get_root_node (NODE_GROUP);
	xmlNodePtr    group, user;
	GtkTreeIter   iter;
	gchar        *groupname, *desc, *buf;
	gboolean      val;

	gtk_list_store_clear (GTK_LIST_STORE (model));
	
	for (group = gst_xml_element_find_first (root, "group");
	     group;
	     group = gst_xml_element_find_next (group, "group")) {
		desc = gst_xml_get_child_content (group, "allows_to");

		if (desc) {
			val = FALSE;

			groupname = gst_xml_get_child_content (group, "name");
			
			/* check whether the user is already in the group */
			if (username) {
				user = gst_xml_element_find_first (group, "users");
				
				for (user = gst_xml_element_find_first (user, "user");
				     user;
				     user = gst_xml_element_find_next (user, "user")) {
					buf = gst_xml_element_get_content (user);

					if (strcmp (username, buf) == 0)
						val = TRUE;

					g_free (buf);
				}
			}
			
			gtk_list_store_append (GTK_LIST_STORE (model), &iter);
			gtk_list_store_set (GTK_LIST_STORE (model), &iter,
					    0, val,
					    1, desc,
					    2, groupname,
					    -1);
			g_free (desc);
			g_free (groupname);
		}
	}
}

static gboolean
is_group_in_profile (xmlNodePtr profile,gchar *groupname)
{
	xmlNodePtr  root, group;
	gchar      *buf;
	gboolean    val = FALSE;

	g_return_if_fail (groupname != NULL);
	root = gst_xml_element_find_first (profile, "groups");

	if (!root)
		return FALSE;

	for (group = gst_xml_element_find_first (root, "group");
	     group != NULL;
	     group = gst_xml_element_find_next (group, "group")) {
		buf = gst_xml_element_get_content (group);

		if (buf && strcmp (groupname, buf) == 0)
			val = TRUE;

		g_free (buf);
	}

	return val;
}

void
populate_privileges_table_from_profile (xmlNodePtr profile)
{
	GtkWidget    *list = gst_dialog_get_widget (tool->main_dialog, "user_privileges");
	GtkTreeModel *model = gtk_tree_view_get_model (GTK_TREE_VIEW (list));
	xmlNodePtr    root = get_root_node (NODE_GROUP);
	xmlNodePtr    group;
	GtkTreeIter   iter;
	gchar        *groupname, *desc, *buf;
	gboolean      val;

	gtk_list_store_clear (GTK_LIST_STORE (model));

	for (group = gst_xml_element_find_first (root, "group");
	     group;
	     group = gst_xml_element_find_next (group, "group"))
	{
		desc = gst_xml_get_child_content (group, "allows_to");

		if (desc) {
			groupname = gst_xml_get_child_content (group, "name");
			val = is_group_in_profile (profile, groupname);
		
			gtk_list_store_append (GTK_LIST_STORE (model), &iter);
			gtk_list_store_set (GTK_LIST_STORE (model), &iter,
					    0, val,
					    1, desc,
					    2, groupname,
					    -1);
			g_free (desc);
			g_free (groupname);
		}
	}
}

void
profile_populate_groups (xmlNodePtr profile)
{
	GtkWidget    *list = gst_dialog_get_widget (tool->main_dialog, "profile_privileges");
	GtkTreeModel *model = gtk_tree_view_get_model (GTK_TREE_VIEW (list));
	xmlNodePtr    root = get_root_node (NODE_GROUP);
	xmlNodePtr    group;
	GtkTreeIter   iter;
	gchar        *groupname, *desc, *buf;
	gboolean      val;

	gtk_list_store_clear (GTK_LIST_STORE (model));
	
	for (group = gst_xml_element_find_first (root, "group");
	     group;
	     group = gst_xml_element_find_next (group, "group"))
	{
		desc = gst_xml_get_child_content (group, "allows_to");

		if (desc) {
			groupname = gst_xml_get_child_content (group, "name");
			val = is_group_in_profile (profile, groupname);
		
			gtk_list_store_append (GTK_LIST_STORE (model), &iter);
			gtk_list_store_set (GTK_LIST_STORE (model), &iter,
					    0, val,
					    1, desc,
					    2, groupname,
					    -1);
			g_free (desc);
			g_free (groupname);
		}
	}
}

GList*
user_privileges_get_list (GList *groups)
{
	GtkWidget    *list  = gst_dialog_get_widget (tool->main_dialog, "user_privileges");
	GtkTreeModel *model = gtk_tree_view_get_model (GTK_TREE_VIEW (list));
	GList        *elem  = NULL;
	gboolean      valid, active;
	GtkTreeIter   iter;
	gchar        *group;

	valid = gtk_tree_model_get_iter_first (model, &iter);

	while (valid) {
		gtk_tree_model_get (model, &iter, 0, &active, 2, &group, -1);
		elem = g_list_find_custom (groups, group, my_strcmp);

		if (active && !elem)
			groups = g_list_prepend (groups, group);
		else if (!active && elem)
			groups = g_list_remove (groups, elem->data);

		valid = gtk_tree_model_iter_next (model, &iter);
	}

	return groups;
}

void
del_profile_groups (xmlNodePtr profile) 
{
	xmlNodePtr root, group;

	if (!profile)
		return;

	g_return_if_fail (profile != NULL);

	root = gst_xml_element_find_first (profile, "groups");

	if (root) {
		gst_xml_element_destroy_children (root);
		gst_xml_element_destroy (root);
	}
}

void
profile_groups_save_data (xmlNodePtr profile)
{
	GtkWidget    *list  = gst_dialog_get_widget (tool->main_dialog, "profile_privileges");
	GtkTreeModel *model = gtk_tree_view_get_model (GTK_TREE_VIEW (list));
	GList        *elem  = NULL;
	gboolean      valid, active;
	GtkTreeIter   iter;
	gchar        *groupname;
	xmlNodePtr    groups_root,orig_profile, buf;

	if (!profile)
		return;

	orig_profile = profile;

	del_profile_groups (profile);
	profile = orig_profile;
	groups_root = gst_xml_element_add (profile, "groups");
	buf = groups_root;

	valid = gtk_tree_model_get_iter_first (model, &iter);

	while (valid) {
		gtk_tree_model_get (model, &iter, 0, &active, 2, &groupname, -1);

		if (active) {
			groups_root = gst_xml_element_add (buf, "group");
			gst_xml_element_set_content (groups_root, (const gchar *) groupname);
		}

		valid = gtk_tree_model_iter_next (model, &iter);
	}
}
