/**
 * Copyright (c) 2010-2016, Abel Hegedus, IncQuery Labs Ltd.
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v. 2.0 which is available at
 * http://www.eclipse.org/legal/epl-v20.html.
 * 
 * SPDX-License-Identifier: EPL-2.0
 */
package org.eclipse.viatra.query.tooling.ui.queryresult;

import com.google.common.base.Objects;
import com.google.common.base.Preconditions;
import com.google.common.collect.HashBasedTable;
import com.google.common.collect.Iterables;
import com.google.common.collect.Maps;
import com.google.common.collect.Multimap;
import com.google.common.collect.Sets;
import com.google.common.collect.Table;
import com.google.common.collect.TreeMultimap;
import java.util.Collections;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;
import java.util.function.Consumer;
import org.eclipse.viatra.query.patternlanguage.emf.specification.SpecificationBuilder;
import org.eclipse.viatra.query.patternlanguage.emf.ui.EMFPatternLanguageUIPlugin;
import org.eclipse.viatra.query.runtime.api.AdvancedViatraQueryEngine;
import org.eclipse.viatra.query.runtime.api.IPatternMatch;
import org.eclipse.viatra.query.runtime.api.IQuerySpecification;
import org.eclipse.viatra.query.runtime.api.ViatraQueryEngineLifecycleListener;
import org.eclipse.viatra.query.runtime.api.ViatraQueryMatcher;
import org.eclipse.viatra.query.runtime.api.scope.QueryScope;
import org.eclipse.viatra.query.runtime.base.api.BaseIndexOptions;
import org.eclipse.viatra.query.runtime.emf.EMFScope;
import org.eclipse.viatra.query.runtime.extensibility.IQuerySpecificationProvider;
import org.eclipse.viatra.query.runtime.matchers.backend.QueryEvaluationHint;
import org.eclipse.viatra.query.runtime.matchers.psystem.queries.PQuery;
import org.eclipse.viatra.query.runtime.registry.IQuerySpecificationRegistry;
import org.eclipse.viatra.query.runtime.registry.IQuerySpecificationRegistryChangeListener;
import org.eclipse.viatra.query.runtime.registry.IQuerySpecificationRegistryEntry;
import org.eclipse.viatra.query.runtime.registry.IRegistryView;
import org.eclipse.viatra.query.runtime.registry.IRegistryViewFactory;
import org.eclipse.viatra.query.runtime.registry.view.AbstractRegistryView;
import org.eclipse.viatra.query.runtime.ui.modelconnector.IModelConnector;
import org.eclipse.viatra.query.tooling.ui.preferences.RuntimePreferencesInterpreter;
import org.eclipse.viatra.query.tooling.ui.queryregistry.index.IPatternBasedSpecificationProvider;
import org.eclipse.viatra.query.tooling.ui.queryresult.IQueryResultViewModelListener;
import org.eclipse.viatra.query.tooling.ui.queryresult.QueryResultTreeMatcher;
import org.eclipse.viatra.query.tooling.ui.util.IFilteredMatcherCollection;
import org.eclipse.viatra.query.tooling.ui.util.IFilteredMatcherContent;
import org.eclipse.viatra.transformation.evm.api.ExecutionSchema;
import org.eclipse.viatra.transformation.evm.api.Job;
import org.eclipse.viatra.transformation.evm.api.RuleSpecification;
import org.eclipse.viatra.transformation.evm.specific.ExecutionSchemas;
import org.eclipse.viatra.transformation.evm.specific.Jobs;
import org.eclipse.viatra.transformation.evm.specific.Lifecycles;
import org.eclipse.viatra.transformation.evm.specific.Rules;
import org.eclipse.viatra.transformation.evm.specific.Schedulers;
import org.eclipse.viatra.transformation.evm.specific.crud.CRUDActivationStateEnum;
import org.eclipse.viatra.transformation.evm.specific.resolver.InvertedDisappearancePriorityConflictResolver;
import org.eclipse.viatra.transformation.evm.specific.scheduler.UpdateCompleteBasedScheduler;
import org.eclipse.xtend.lib.annotations.AccessorType;
import org.eclipse.xtend.lib.annotations.Accessors;
import org.eclipse.xtend.lib.annotations.FinalFieldsConstructor;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Exceptions;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.Pure;

/**
 * @author Abel Hegedus
 */
@SuppressWarnings("all")
public class QueryResultTreeInput implements IFilteredMatcherCollection {
  @FinalFieldsConstructor
  public static class RegistryChangeListener implements IQuerySpecificationRegistryChangeListener {
    private final QueryResultTreeInput input;
    
    @Override
    public void entryAdded(final IQuerySpecificationRegistryEntry entry) {
      this.input.addMatcherIfLoaded(entry);
    }
    
    @Override
    public void entryRemoved(final IQuerySpecificationRegistryEntry entry) {
      this.input.removeMatcherIfLoaded(entry);
    }
    
    public RegistryChangeListener(final QueryResultTreeInput input) {
      super();
      this.input = input;
    }
  }
  
  @FinalFieldsConstructor
  public static class EngineLifecycleListener implements ViatraQueryEngineLifecycleListener {
    private final QueryResultTreeInput input;
    
    @Override
    public void engineBecameTainted(final String message, final Throwable t) {
      this.input.engineOperational = false;
      this.input.dispose();
    }
    
    @Override
    public void engineDisposed() {
      this.input.engineOperational = false;
      this.input.dispose();
    }
    
    @Override
    public void engineWiped() {
      this.input.resetInput();
    }
    
    @Override
    public void matcherInstantiated(final ViatraQueryMatcher<? extends IPatternMatch> matcher) {
      this.input.createMatcher(matcher);
    }
    
    public EngineLifecycleListener(final QueryResultTreeInput input) {
      super();
      this.input = input;
    }
  }
  
  @Accessors(AccessorType.PUBLIC_GETTER)
  private AdvancedViatraQueryEngine engine;
  
  @Accessors(AccessorType.PUBLIC_GETTER)
  private boolean readOnlyEngine;
  
  @Accessors(AccessorType.PUBLIC_GETTER)
  private boolean engineOperational;
  
  @Accessors(AccessorType.PUBLIC_GETTER)
  private Map<String, QueryResultTreeMatcher<?>> matchers;
  
  /**
   * @since 1.4
   */
  @Accessors({ AccessorType.PUBLIC_GETTER, AccessorType.PROTECTED_SETTER })
  private IModelConnector modelConnector;
  
  private Table<String, String, IQuerySpecificationRegistryEntry> loadedEntries;
  
  private Multimap<String, String> knownErrorEntries;
  
  private SpecificationBuilder builder;
  
  private ExecutionSchema schema;
  
  private QueryResultTreeInput.EngineLifecycleListener lifecycleListener;
  
  private QueryResultTreeInput.RegistryChangeListener registryListener;
  
  private IRegistryView view;
  
  @Accessors(AccessorType.PUBLIC_GETTER)
  private QueryEvaluationHint hintForBackendSelection;
  
  private Set<IQueryResultViewModelListener> listeners;
  
  public QueryResultTreeInput(final AdvancedViatraQueryEngine engine, final IQuerySpecificationRegistry registry, final boolean readOnlyEngine, final QueryEvaluationHint hint) {
    this.engine = engine;
    this.engineOperational = true;
    this.hintForBackendSelection = hint;
    this.readOnlyEngine = readOnlyEngine;
    this.matchers = Maps.<String, QueryResultTreeMatcher<?>>newTreeMap();
    this.loadedEntries = HashBasedTable.<String, String, IQuerySpecificationRegistryEntry>create();
    this.knownErrorEntries = TreeMultimap.<String, String>create();
    SpecificationBuilder _specificationBuilder = new SpecificationBuilder();
    this.builder = _specificationBuilder;
    this.listeners = Sets.<IQueryResultViewModelListener>newHashSet();
    UpdateCompleteBasedScheduler.UpdateCompleteBasedSchedulerFactory _queryEngineSchedulerFactory = Schedulers.getQueryEngineSchedulerFactory(engine);
    InvertedDisappearancePriorityConflictResolver _invertedDisappearancePriorityConflictResolver = new InvertedDisappearancePriorityConflictResolver();
    this.schema = ExecutionSchemas.createViatraQueryExecutionSchema(engine, _queryEngineSchedulerFactory, _invertedDisappearancePriorityConflictResolver);
    final Consumer<ViatraQueryMatcher<? extends IPatternMatch>> _function = (ViatraQueryMatcher<? extends IPatternMatch> it) -> {
      this.createMatcher(it);
    };
    engine.getCurrentMatchers().forEach(_function);
    this.schema.startUnscheduledExecution();
    QueryResultTreeInput.EngineLifecycleListener _engineLifecycleListener = new QueryResultTreeInput.EngineLifecycleListener(this);
    this.lifecycleListener = _engineLifecycleListener;
    engine.addLifecycleListener(this.lifecycleListener);
    if ((!readOnlyEngine)) {
      QueryResultTreeInput.RegistryChangeListener _registryChangeListener = new QueryResultTreeInput.RegistryChangeListener(this);
      this.registryListener = _registryChangeListener;
      final IRegistryViewFactory _function_1 = (IQuerySpecificationRegistry it) -> {
        return new AbstractRegistryView(registry, true) {
          @Override
          protected boolean isEntryRelevant(final IQuerySpecificationRegistryEntry entry) {
            return true;
          }
        };
      };
      this.view = registry.createView(_function_1);
      this.view.addViewListener(this.registryListener);
    }
  }
  
  public QueryEvaluationHint setHint(final QueryEvaluationHint hint) {
    QueryEvaluationHint _xblockexpression = null;
    {
      Preconditions.<QueryEvaluationHint>checkNotNull(hint);
      _xblockexpression = this.hintForBackendSelection = hint;
    }
    return _xblockexpression;
  }
  
  public <MATCH extends IPatternMatch> QueryResultTreeMatcher<MATCH> createMatcher(final ViatraQueryMatcher<MATCH> matcher) {
    final QueryResultTreeMatcher<MATCH> treeMatcher = new QueryResultTreeMatcher<MATCH>(this, matcher);
    final Consumer<MATCH> _function = (MATCH match) -> {
      final Consumer<IQueryResultViewModelListener> _function_1 = (IQueryResultViewModelListener it) -> {
        it.matchAdded(treeMatcher, match);
      };
      this.listeners.forEach(_function_1);
    };
    final Job<MATCH> matchCreatedJob = Jobs.<MATCH>newStatelessJob(CRUDActivationStateEnum.CREATED, _function);
    final Consumer<MATCH> _function_1 = (MATCH match) -> {
      final Consumer<IQueryResultViewModelListener> _function_2 = (IQueryResultViewModelListener it) -> {
        it.matchUpdated(treeMatcher, match);
      };
      this.listeners.forEach(_function_2);
    };
    final Job<MATCH> matchUpdatedJob = Jobs.<MATCH>newStatelessJob(CRUDActivationStateEnum.UPDATED, _function_1);
    final Consumer<MATCH> _function_2 = (MATCH match) -> {
      final Consumer<IQueryResultViewModelListener> _function_3 = (IQueryResultViewModelListener it) -> {
        it.matchRemoved(treeMatcher, match);
      };
      this.listeners.forEach(_function_3);
    };
    final Job<MATCH> matchDeletedJob = Jobs.<MATCH>newStatelessJob(CRUDActivationStateEnum.DELETED, _function_2);
    final RuleSpecification<MATCH> ruleSpec = Rules.<MATCH>newMatcherRuleSpecification(matcher, 
      Lifecycles.getDefault(true, true), 
      Collections.<Job<MATCH>>unmodifiableSet(CollectionLiterals.<Job<MATCH>>newHashSet(matchCreatedJob, matchUpdatedJob, matchDeletedJob)));
    treeMatcher.setRuleSpec(ruleSpec);
    final String fullyQualifiedName = matcher.getSpecification().getFullyQualifiedName();
    this.matchers.put(fullyQualifiedName, treeMatcher);
    final Consumer<IQueryResultViewModelListener> _function_3 = (IQueryResultViewModelListener it) -> {
      it.matcherAdded(treeMatcher);
    };
    this.listeners.forEach(_function_3);
    this.schema.<MATCH>addRule(ruleSpec);
    return treeMatcher;
  }
  
  public Object addMatcherIfLoaded(final IQuerySpecificationRegistryEntry entry) {
    Object _xifexpression = null;
    boolean _contains = this.loadedEntries.contains(entry.getSourceIdentifier(), entry.getFullyQualifiedName());
    if (_contains) {
      _xifexpression = null;
    }
    return _xifexpression;
  }
  
  public Object removeMatcherIfLoaded(final IQuerySpecificationRegistryEntry entry) {
    Object _xifexpression = null;
    boolean _contains = this.loadedEntries.contains(entry.getSourceIdentifier(), entry.getFullyQualifiedName());
    if (_contains) {
      _xifexpression = null;
    }
    return _xifexpression;
  }
  
  public QueryResultTreeMatcher<?> removeMatcher(final IQuerySpecificationRegistryEntry entry) {
    final QueryResultTreeMatcher<?> treeMatcher = this.matchers.get(entry.getFullyQualifiedName());
    if (((treeMatcher != null) && Objects.equal(treeMatcher.getEntry().getSourceIdentifier(), entry.getSourceIdentifier()))) {
      return this.removeMatcher(treeMatcher);
    }
    return null;
  }
  
  public QueryResultTreeMatcher<?> removeMatcher(final QueryResultTreeMatcher<?> matcher) {
    this.matchers.remove(matcher.getEntry().getFullyQualifiedName());
    final Consumer<IQueryResultViewModelListener> _function = (IQueryResultViewModelListener it) -> {
      it.matcherRemoved(matcher);
    };
    this.listeners.forEach(_function);
    RuleSpecification<? extends IPatternMatch> _ruleSpec = matcher.getRuleSpec();
    boolean _tripleNotEquals = (_ruleSpec != null);
    if (_tripleNotEquals) {
      this.schema.removeRule(matcher.getRuleSpec());
    }
    ViatraQueryMatcher<? extends IPatternMatch> _matcher = matcher.getMatcher();
    boolean _tripleNotEquals_1 = (_matcher != null);
    if (_tripleNotEquals_1) {
      this.builder.forgetSpecificationTransitively(matcher.getMatcher().getSpecification());
    } else {
      final Consumer<IQuerySpecification<?>> _function_1 = (IQuerySpecification<?> it) -> {
        this.builder.forgetSpecificationTransitively(it);
      };
      this.builder.getSpecification(matcher.getEntry().getFullyQualifiedName()).ifPresent(_function_1);
    }
    return matcher;
  }
  
  public void loadQueries(final Iterable<IQuerySpecificationRegistryEntry> entries) {
    if (this.readOnlyEngine) {
      throw new UnsupportedOperationException("Cannot load queries to read-only engine");
    }
    SpecificationBuilder _specificationBuilder = new SpecificationBuilder();
    this.builder = _specificationBuilder;
    final Iterator<IQuerySpecificationRegistryEntry> iterator = entries.iterator();
    while ((this.engineOperational && iterator.hasNext())) {
      {
        final IQuerySpecificationRegistryEntry entry = iterator.next();
        boolean _containsKey = this.matchers.containsKey(entry.getFullyQualifiedName());
        if (_containsKey) {
          this.removeMatcher(entry);
        }
        this.loadQuery(entry);
        this.loadedEntries.put(entry.getSourceIdentifier(), entry.getFullyQualifiedName(), entry);
      }
    }
    if (this.engineOperational) {
      this.schema.startUnscheduledExecution();
    }
  }
  
  private QueryResultTreeMatcher<?> loadQuery(final IQuerySpecificationRegistryEntry entry) {
    QueryResultTreeMatcher<IPatternMatch> _xblockexpression = null;
    {
      if ((!this.engineOperational)) {
        IllegalStateException _illegalStateException = new IllegalStateException("Query engine encountered a fatal error or has been disposed");
        this.addErroneousMatcher(entry, _illegalStateException);
      }
      final String entryFQN = entry.getFullyQualifiedName();
      boolean _containsKey = this.matchers.containsKey(entryFQN);
      if (_containsKey) {
        this.removeMatcher(entry);
      }
      QueryResultTreeMatcher<IPatternMatch> _xtrycatchfinallyexpression = null;
      try {
        QueryResultTreeMatcher<IPatternMatch> _xblockexpression_1 = null;
        {
          IQuerySpecification<?> _specificationOfProvider = this.getSpecificationOfProvider(entry.getProvider());
          final IQuerySpecification specification = ((IQuerySpecification) _specificationOfProvider);
          QueryResultTreeMatcher<IPatternMatch> _xifexpression = null;
          PQuery.PQueryStatus _status = specification.getInternalQueryRepresentation().getStatus();
          boolean _equals = Objects.equal(_status, PQuery.PQueryStatus.ERROR);
          if (_equals) {
            IllegalArgumentException _illegalArgumentException = new IllegalArgumentException("Query definition contains errors");
            _xifexpression = this.addErroneousMatcher(entry, _illegalArgumentException);
          } else {
            final QueryEvaluationHint currentHint = this.hintForBackendSelection.overrideBy(RuntimePreferencesInterpreter.getHintOverridesFromPreferences());
            final ViatraQueryMatcher<? extends IPatternMatch> matcher = this.engine.<ViatraQueryMatcher<? extends IPatternMatch>>getMatcher(specification, currentHint);
            final String specificationFQN = specification.getFullyQualifiedName();
            final QueryResultTreeMatcher<?> treeMatcher = this.matchers.get(specificationFQN);
            boolean _notEquals = (!Objects.equal(specificationFQN, entryFQN));
            if (_notEquals) {
              this.matchers.remove(specificationFQN);
              this.matchers.put(entryFQN, treeMatcher);
            }
            treeMatcher.setEntry(entry);
            treeMatcher.setHint(currentHint);
            this.knownErrorEntries.remove(entry.getSourceIdentifier(), entryFQN);
            return treeMatcher;
          }
          _xblockexpression_1 = _xifexpression;
        }
        _xtrycatchfinallyexpression = _xblockexpression_1;
      } catch (final Throwable _t) {
        if (_t instanceof Exception) {
          final Exception ex = (Exception)_t;
          return this.addErroneousMatcher(entry, ex);
        } else {
          throw Exceptions.sneakyThrow(_t);
        }
      }
      _xblockexpression = _xtrycatchfinallyexpression;
    }
    return _xblockexpression;
  }
  
  private QueryResultTreeMatcher<IPatternMatch> addErroneousMatcher(final IQuerySpecificationRegistryEntry entry, final Exception ex) {
    final String entryFQN = entry.getFullyQualifiedName();
    final QueryResultTreeMatcher<IPatternMatch> treeMatcher = new QueryResultTreeMatcher<IPatternMatch>(this, null);
    treeMatcher.setException(ex);
    treeMatcher.setEntry(entry);
    this.matchers.put(entryFQN, treeMatcher);
    final Consumer<IQueryResultViewModelListener> _function = (IQueryResultViewModelListener it) -> {
      it.matcherAdded(treeMatcher);
    };
    this.listeners.forEach(_function);
    boolean _put = this.knownErrorEntries.put(entry.getSourceIdentifier(), entryFQN);
    if (_put) {
      final String logMessage = String.format("Query Explorer has encountered an error during evaluation of query %s: %s", entryFQN, ex.getMessage());
      EMFPatternLanguageUIPlugin.getInstance().logException(logMessage, ex);
    }
    return treeMatcher;
  }
  
  private IQuerySpecification<?> getSpecificationOfProvider(final IQuerySpecificationProvider provider) {
    if ((provider instanceof IPatternBasedSpecificationProvider)) {
      final IQuerySpecification<?> specification = ((IPatternBasedSpecificationProvider)provider).getSpecification(this.builder);
      return specification;
    } else {
      final IQuerySpecification<?> specification_1 = provider.get();
      return specification_1;
    }
  }
  
  public boolean addListener(final IQueryResultViewModelListener listener) {
    return this.listeners.add(listener);
  }
  
  public boolean removeListener(final IQueryResultViewModelListener listener) {
    return this.listeners.remove(listener);
  }
  
  public IRegistryView dispose() {
    IRegistryView _xblockexpression = null;
    {
      if (this.schema!=null) {
        this.schema.dispose();
      }
      this.schema = null;
      if (this.engine!=null) {
        this.engine.removeLifecycleListener(this.lifecycleListener);
      }
      this.engine = null;
      this.resetMatchers();
      this.listeners.clear();
      if (this.view!=null) {
        this.view.removeViewListener(this.registryListener);
      }
      _xblockexpression = this.view = null;
    }
    return _xblockexpression;
  }
  
  protected void resetInput() {
    final Consumer<QueryResultTreeMatcher<?>> _function = (QueryResultTreeMatcher<?> it) -> {
      RuleSpecification<? extends IPatternMatch> _ruleSpec = it.getRuleSpec();
      boolean _tripleNotEquals = (_ruleSpec != null);
      if (_tripleNotEquals) {
        this.schema.removeRule(it.getRuleSpec());
      }
    };
    this.matchers.values().forEach(_function);
    this.resetMatchers();
    if ((!this.readOnlyEngine)) {
      this.engine.removeLifecycleListener(this.lifecycleListener);
      this.engine.wipe();
      this.engine.addLifecycleListener(this.lifecycleListener);
    }
  }
  
  protected void resetMatchers() {
    final Consumer<QueryResultTreeMatcher<?>> _function = (QueryResultTreeMatcher<?> matcher) -> {
      final Consumer<IQueryResultViewModelListener> _function_1 = (IQueryResultViewModelListener it) -> {
        it.matcherRemoved(matcher);
      };
      this.listeners.forEach(_function_1);
    };
    this.matchers.values().forEach(_function);
    SpecificationBuilder _specificationBuilder = new SpecificationBuilder();
    this.builder = _specificationBuilder;
    this.matchers.clear();
    this.loadedEntries.clear();
  }
  
  public BaseIndexOptions getBaseIndexOptions() {
    QueryScope _scope = this.engine.getScope();
    final EMFScope emfScope = ((EMFScope) _scope);
    return emfScope.getOptions();
  }
  
  @Override
  public Iterable<IFilteredMatcherContent<?>> getFilteredMatchers() {
    final Function1<Object, IFilteredMatcherContent<?>> _function = (Object matcher) -> {
      return ((IFilteredMatcherContent<?>) matcher);
    };
    return IterableExtensions.<IFilteredMatcherContent, IFilteredMatcherContent<?>>map(Iterables.<IFilteredMatcherContent>filter(this.matchers.values(), IFilteredMatcherContent.class), _function);
  }
  
  public void matcherFilterUpdated(final QueryResultTreeMatcher<?> matcher) {
    final Consumer<IQueryResultViewModelListener> _function = (IQueryResultViewModelListener it) -> {
      it.matcherFilterUpdated(matcher);
    };
    this.listeners.forEach(_function);
  }
  
  @Pure
  public AdvancedViatraQueryEngine getEngine() {
    return this.engine;
  }
  
  @Pure
  public boolean isReadOnlyEngine() {
    return this.readOnlyEngine;
  }
  
  @Pure
  public boolean isEngineOperational() {
    return this.engineOperational;
  }
  
  @Pure
  public Map<String, QueryResultTreeMatcher<?>> getMatchers() {
    return this.matchers;
  }
  
  @Pure
  public IModelConnector getModelConnector() {
    return this.modelConnector;
  }
  
  protected void setModelConnector(final IModelConnector modelConnector) {
    this.modelConnector = modelConnector;
  }
  
  @Pure
  public QueryEvaluationHint getHintForBackendSelection() {
    return this.hintForBackendSelection;
  }
}
