"use strict";
/*********************************************************************
 * Copyright (c) 2018, 2023 Ericsson and others
 *
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 *********************************************************************/
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.resolveLineTagLocations = exports.gdbVersionAtLeast = exports.logFileName = exports.hardwareBreakpoint = exports.defaultAdapter = exports.debugServerPort = exports.gdbServerPath = exports.gdbPath = exports.skipMake = exports.gdbNonStop = exports.gdbAsync = exports.isRemoteTest = exports.openGdbConsole = exports.fillDefaults = exports.standardBeforeEach = exports.testProgramsDir = exports.compareVariable = exports.verifyRegister = exports.verifyVariable = exports.expectRejection = exports.getScopes = void 0;
const chai_1 = require("chai");
const cp = require("child_process");
const fs = require("fs");
const path = require("path");
const os = require("os");
const debugClient_1 = require("./debugClient");
const util_1 = require("../util");
function getScopes(dc, threadIndex = 0, stackIndex = 0) {
    return __awaiter(this, void 0, void 0, function* () {
        // threads
        const threads = yield dc.threadsRequest();
        (0, chai_1.expect)(threads.body.threads.length, 'There are fewer threads than expected.').to.be.at.least(threadIndex + 1);
        const thread = threads.body.threads[threadIndex];
        const threadId = thread.id;
        // stack trace
        const stack = yield dc.stackTraceRequest({ threadId });
        (0, chai_1.expect)(stack.body.stackFrames.length, 'There are fewer stack frames than expected.').to.be.at.least(stackIndex + 1);
        const frame = stack.body.stackFrames[stackIndex];
        const frameId = frame.id;
        const scopes = yield dc.scopesRequest({ frameId });
        return Promise.resolve({ thread, frame, scopes });
    });
}
exports.getScopes = getScopes;
/**
 * Wrap `promise` in a new Promise that resolves if `promise` is rejected, and is rejected if `promise` is resolved.
 *
 * This is useful when we expect `promise` to be reject and want to test that it is indeed the case.
 */
function expectRejection(promise) {
    return new Promise((resolve, reject) => {
        promise.then(reject).catch(resolve);
    });
}
exports.expectRejection = expectRejection;
/**
 * Test a given variable returned from a variablesRequest against an expected name, type, and/or value.
 */
function verifyVariable(variable, expectedName, expectedType, expectedValue, flags) {
    (0, chai_1.expect)(variable.name, `The name of ${expectedName} is wrong`).to.equal(expectedName);
    if (expectedType) {
        (0, chai_1.expect)(variable.type, `The type of ${expectedName} is wrong`).to.equal(expectedType);
    }
    if (expectedValue) {
        (0, chai_1.expect)(variable.value, `The value of ${expectedName} is wrong`).to.equal(expectedValue);
    }
    if (flags === null || flags === void 0 ? void 0 : flags.hasChildren) {
        (0, chai_1.expect)(variable.variablesReference, `${expectedName} has no children`).to.not.equal(0);
    }
    else {
        (0, chai_1.expect)(variable.variablesReference, `${expectedName} has children`).to.equal(0);
    }
    if ((flags === null || flags === void 0 ? void 0 : flags.hasMemoryReference) || (flags === null || flags === void 0 ? void 0 : flags.hasMemoryReference) === undefined) {
        // Rather than actual read the memory, just verify that the memory
        // reference is to what is expected
        (0, chai_1.expect)(variable.memoryReference).eq(`&(${expectedName})`);
    }
    else {
        // For now we only support memory references for top-level
        // variables (e.g. no struct members). A possible
        // TODO is to support memoryReferences in these cases
        (0, chai_1.expect)(variable.memoryReference).is.undefined;
    }
}
exports.verifyVariable = verifyVariable;
/**
 * Test a given register variable returned from a variablesRequest against an expected name and/or value.
 */
function verifyRegister(variable, expectedName, expectedValue) {
    (0, chai_1.expect)(variable.name, `The name of ${expectedName} is wrong`).to.equal(expectedName);
    if (expectedValue) {
        (0, chai_1.expect)(variable.value, `The value of ${expectedName} is wrong`).to.equal(expectedValue);
    }
}
exports.verifyRegister = verifyRegister;
function compareVariable(varA, varB, namesMatch, typesMatch, valuesMatch) {
    if (namesMatch) {
        (0, chai_1.expect)(varA.name, `The name of ${varA.name} and ${varB.name} does not match`).to.equal(varB.name);
    }
    else {
        (0, chai_1.expect)(varA.name, `The name of ${varA.name} and ${varB.name} matches`).to.not.equal(varB.name);
    }
    if (typesMatch) {
        (0, chai_1.expect)(varA.type, `The type of ${varA.name} and ${varB.name} does not match`).to.equal(varB.type);
    }
    else {
        (0, chai_1.expect)(varA.type, `The type of ${varA.name} and ${varB.name} match`).to.equal(varB.type);
    }
    if (valuesMatch) {
        (0, chai_1.expect)(varA.value, `The value of ${varA.name} and ${varB.name} do not match`).to.equal(varB.value);
    }
    else {
        (0, chai_1.expect)(varA.value, `The value of ${varA.name} and ${varB.name} matches`).to.not.equal(varB.value);
    }
}
exports.compareVariable = compareVariable;
exports.testProgramsDir = path.join(__dirname, '..', '..', 'src', 'integration-tests', 'test-programs');
function standardBeforeEach(adapter, extraArgs) {
    return __awaiter(this, void 0, void 0, function* () {
        const dc = new debugClient_1.CdtDebugClient(adapter, extraArgs);
        yield dc.start(exports.debugServerPort);
        yield dc.initializeRequest();
        return dc;
    });
}
exports.standardBeforeEach = standardBeforeEach;
function fillDefaults(test, argsIn) {
    if (!test) {
        throw new Error('A Test object is required (this.test in test body or this.currentTest in beforeEach');
    }
    const args = argsIn !== undefined ? argsIn : {};
    args.verbose = true;
    args.logFile = logFileName(test);
    args.gdb = exports.gdbPath;
    args.openGdbConsole = exports.openGdbConsole;
    args.gdbAsync = exports.gdbAsync;
    args.gdbNonStop = exports.gdbNonStop;
    args.hardwareBreakpoint = exports.hardwareBreakpoint;
    return args;
}
exports.fillDefaults = fillDefaults;
exports.openGdbConsole = process.argv.indexOf('--run-in-terminal') !== -1;
exports.isRemoteTest = process.argv.indexOf('--test-remote') !== -1;
exports.gdbAsync = process.argv.indexOf('--test-gdb-async-off') === -1;
exports.gdbNonStop = process.argv.indexOf('--test-gdb-non-stop') !== -1;
exports.skipMake = process.argv.indexOf('--skip-make') !== -1;
exports.gdbPath = getGdbPathCli();
exports.gdbServerPath = getGdbServerPathCli();
exports.debugServerPort = getDebugServerPortCli();
exports.defaultAdapter = getDefaultAdapterCli();
exports.hardwareBreakpoint = process.argv.indexOf('--test-hw-breakpoint-on') !== -1;
before(function () {
    // Run make once per mocha execution, unless --skip-make
    // is specified. On the CI we run with --skip-make and the
    // make is its own explicit build step for two reasons:
    // 1. It makes it easier to see build errors in the make
    // 2. On CI we get errors running make on Windows like
    // ld.exe: cannot open output file empty.exe: Permission denied
    // The second reason may be because sometimes empty.exe is left
    // running after a remote test finishes.
    if (!exports.skipMake) {
        cp.execSync('make', { cwd: exports.testProgramsDir });
    }
    if ((exports.gdbNonStop || exports.hardwareBreakpoint) && os.platform() === 'win32') {
        // skip tests that are unsupported on Windows
        this.skip();
    }
});
beforeEach(function () {
    if (this.currentTest) {
        let prefix = '';
        if (exports.openGdbConsole) {
            prefix += 'run-in-terminal ';
        }
        if (exports.isRemoteTest) {
            prefix += 'remote ';
        }
        if (!exports.gdbAsync) {
            prefix += 'gdb-async-off ';
        }
        if (exports.gdbNonStop) {
            prefix += 'gdb-non-stop ';
        }
        if (exports.hardwareBreakpoint) {
            prefix += 'hw-breakpoint-on ';
        }
        if (prefix) {
            prefix = '/' + prefix.trim() + '/';
        }
        else {
            prefix = '/defaults/';
        }
        this.currentTest.title = prefix + this.currentTest.title;
    }
});
function logFileName(test) {
    // Clean up characters that GitHub actions doesn't like in filenames
    const cleaned = test
        .fullTitle()
        .replace('>', '&gt;')
        .replace('<', '&lt;')
        .split('/')
        .map((segment) => segment.trim())
        .join('/');
    return `${process.cwd()}/test-logs/${cleaned}.log`;
}
exports.logFileName = logFileName;
function getGdbPathCli() {
    const keyIndex = process.argv.indexOf('--gdb-path');
    if (keyIndex === -1) {
        return undefined;
    }
    return process.argv[keyIndex + 1];
}
function getGdbServerPathCli() {
    const keyIndex = process.argv.indexOf('--gdbserver-path');
    if (keyIndex === -1) {
        return 'gdbserver';
    }
    return process.argv[keyIndex + 1];
}
function getDebugServerPortCli() {
    const keyIndex = process.argv.indexOf('--debugserverport');
    if (keyIndex === -1) {
        return undefined;
    }
    return parseInt(process.argv[keyIndex + 1], 10);
}
function getDefaultAdapterCli() {
    if (!exports.isRemoteTest) {
        return 'debugAdapter.js';
    }
    return 'debugTargetAdapter.js';
}
function gdbVersionAtLeast(targetVersion) {
    return __awaiter(this, void 0, void 0, function* () {
        return ((0, util_1.compareVersions)(yield (0, util_1.getGdbVersion)(exports.gdbPath || 'gdb'), targetVersion) >=
            0);
    });
}
exports.gdbVersionAtLeast = gdbVersionAtLeast;
/**
 * Find locations of tags in `sourceFile`.
 *
 * Instead of referring to source line numbers of test programs directly,
 * tests should place tags (usually some comments) in the source files.  This
 * function finds the line number correspnding to each tag in `tags`.
 *
 * This function throws if a tag is found more than once or is not found.
 *
 * @param tags An object where keys are the tags to find, and values are 0.
 *             This function will modify the object in place to full the values
 *             with line number.
 */
function resolveLineTagLocations(sourceFile, tags) {
    const lines = fs
        .readFileSync(sourceFile, { encoding: 'utf-8' })
        .split('\n');
    for (let i = 0; i < lines.length; i++) {
        for (const tag of Object.keys(tags)) {
            if (lines[i].includes(tag)) {
                if (tags[tag] !== 0) {
                    throw new Error(`Tag ${tag} has been found twice.`);
                }
                tags[tag] = i + 1;
            }
        }
    }
    for (const tag of Object.keys(tags)) {
        const line = tags[tag];
        if (line === 0) {
            throw new Error(`Tag ${tag} was not found.`);
        }
    }
}
exports.resolveLineTagLocations = resolveLineTagLocations;
//# sourceMappingURL=utils.js.map