"use strict";
/*********************************************************************
 * Copyright (c) 2018 Ericsson and others
 *
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 *********************************************************************/
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
Object.defineProperty(exports, "__esModule", { value: true });
const chai_1 = require("chai");
const path = require("path");
const utils_1 = require("./utils");
// Allow non-arrow functions: https://mochajs.org/#arrow-functions
// tslint:disable:only-arrow-functions
describe('Disassembly Test Suite', function () {
    let dc;
    let frame;
    const disProgram = path.join(utils_1.testProgramsDir, 'disassemble');
    const disSrc = path.join(utils_1.testProgramsDir, 'disassemble.c');
    beforeEach(function () {
        return __awaiter(this, void 0, void 0, function* () {
            dc = yield utils_1.standardBeforeEach();
            yield dc.hitBreakpoint({
                gdb: utils_1.gdbPath,
                program: disProgram,
                openGdbConsole: utils_1.openGdbConsole,
            }, {
                path: disSrc,
                line: 2,
            });
            const threads = yield dc.threadsRequest();
            // On windows additional threads can exist to handle signals, therefore find
            // the real thread & frame running the user code. The other thread will
            // normally be running code from ntdll or similar.
            loop_threads: for (const thread of threads.body.threads) {
                const stack = yield dc.stackTraceRequest({ threadId: thread.id });
                if (stack.body.stackFrames.length >= 1) {
                    for (const f of stack.body.stackFrames) {
                        if (f.source && f.source.name === 'disassemble.c') {
                            frame = f;
                            break loop_threads;
                        }
                    }
                }
            }
            // Make sure we found the expected frame
            chai_1.expect(frame).not.eq(undefined);
        });
    });
    afterEach(function () {
        return __awaiter(this, void 0, void 0, function* () {
            yield dc.stop();
        });
    });
    it('can disassemble', function () {
        return __awaiter(this, void 0, void 0, function* () {
            const disassemble = (yield dc.send('disassemble', {
                memoryReference: 'main',
                instructionCount: 100,
            }));
            chai_1.expect(disassemble).not.eq(undefined);
            chai_1.expect(disassemble.body).not.eq(undefined);
            if (disassemble.body) {
                const instructions = disassemble.body.instructions;
                chai_1.expect(instructions).to.have.lengthOf(100);
                // the contents of the instructions are platform dependent, so instead
                // make sure we have read fully
                for (const i of instructions) {
                    chai_1.expect(i.address).to.have.lengthOf.greaterThan(0);
                    chai_1.expect(i.instructionBytes).to.have.lengthOf.greaterThan(0);
                    chai_1.expect(i.instruction).to.have.lengthOf.greaterThan(0);
                }
            }
        });
    });
    it('can disassemble with no source references', function () {
        return __awaiter(this, void 0, void 0, function* () {
            // In this case we attempt to read from where there is no source,
            // GDB returns data in a different format in that case
            const disassemble = (yield dc.send('disassemble', {
                memoryReference: 'main+1000',
                instructionCount: 100,
            }));
            chai_1.expect(disassemble).not.eq(undefined);
            chai_1.expect(disassemble.body).not.eq(undefined);
            if (disassemble.body) {
                const instructions = disassemble.body.instructions;
                chai_1.expect(instructions).to.have.lengthOf(100);
                // the contents of the instructions are platform dependent, so instead
                // make sure we have read fully
                for (const i of instructions) {
                    chai_1.expect(i.address).to.have.lengthOf.greaterThan(0);
                    chai_1.expect(i.instructionBytes).to.have.lengthOf.greaterThan(0);
                    chai_1.expect(i.instruction).to.have.lengthOf.greaterThan(0);
                }
            }
        });
    });
    it('can handle disassemble at bad address', function () {
        return __awaiter(this, void 0, void 0, function* () {
            const disassemble = (yield dc.send('disassemble', {
                memoryReference: '0x0',
                instructionCount: 10,
            }));
            chai_1.expect(disassemble).not.eq(undefined);
            chai_1.expect(disassemble.body).not.eq(undefined);
            if (disassemble.body) {
                const instructions = disassemble.body.instructions;
                chai_1.expect(instructions).to.have.lengthOf(10);
                // the contens of the instructions are platform dependent, so instead
                // make sure we have read fully
                for (const i of instructions) {
                    chai_1.expect(i.address).to.have.lengthOf.greaterThan(0);
                    chai_1.expect(i.instruction).to.have.lengthOf.greaterThan(0);
                    chai_1.expect(i.instructionBytes).eq(undefined);
                }
            }
        });
    });
});
//# sourceMappingURL=diassemble.spec.js.map