/* --------------------------------------------------------------------------------------------
 * Copyright (c) Remy Suen. All rights reserved.
 * Licensed under the MIT License. See License.txt in the project root for license information.
 * ------------------------------------------------------------------------------------------ */
'use strict';
Object.defineProperty(exports, "__esModule", { value: true });
const vscode_languageserver_types_1 = require("vscode-languageserver-types");
const imageTemplate_1 = require("./imageTemplate");
const from_1 = require("./instructions/from");
const util_1 = require("./util");
const main_1 = require("./main");
class Dockerfile extends imageTemplate_1.ImageTemplate {
    constructor(document) {
        super();
        this.initialInstructions = new imageTemplate_1.ImageTemplate();
        this.buildStages = [];
        this.directive = null;
        /**
         * Whether a FROM instruction has been added to this Dockerfile or not.
         */
        this.foundFrom = false;
        this.document = document;
    }
    getEscapeCharacter() {
        if (this.directive !== null && this.directive.getDirective() === main_1.Directive.escape) {
            let value = this.directive.getValue();
            if (value === '\\' || value === '`') {
                return value;
            }
        }
        return '\\';
    }
    getInitialARGs() {
        return this.initialInstructions.getARGs();
    }
    getContainingImage(position) {
        let range = vscode_languageserver_types_1.Range.create(vscode_languageserver_types_1.Position.create(0, 0), this.document.positionAt(this.document.getText().length));
        if (!util_1.Util.isInsideRange(position, range)) {
            // not inside the document, invalid position
            return null;
        }
        for (let buildStage of this.buildStages) {
            if (buildStage.contains(position)) {
                return buildStage;
            }
        }
        let instructions = this.initialInstructions.getInstructions();
        if (instructions.length > 0 &&
            (this.initialInstructions.contains(position) || instructions[instructions.length - 1].getRange().end.line >= position.line)) {
            return this.initialInstructions;
        }
        if (this.buildStages.length > 0) {
            if (this.buildStages[0].getInstructions()[0].getRange().start.line > position.line) {
                let instructions = this.initialInstructions.getInstructions();
                if (instructions.length > 0) {
                    return this.buildStages[0];
                }
            }
            let instructions = this.buildStages[this.buildStages.length - 1].getInstructions();
            if (instructions[instructions.length - 1].getRange().end.line < position.line) {
                return this;
            }
            for (let i = 0; i < this.buildStages.length - 1; i++) {
                let stageInstructions = this.buildStages[i].getInstructions();
                let stageInstructions2 = this.buildStages[i + 1].getInstructions();
                let between = vscode_languageserver_types_1.Range.create(stageInstructions[stageInstructions.length - 1].getRange().end, stageInstructions2[0].getRange().start);
                if (util_1.Util.isInsideRange(position, between)) {
                    return this.buildStages[i + 1];
                }
            }
        }
        return this;
    }
    addInstruction(instruction) {
        if (instruction.getKeyword() === main_1.Keyword.FROM) {
            this.currentBuildStage = new imageTemplate_1.ImageTemplate();
            this.buildStages.push(this.currentBuildStage);
            this.foundFrom = true;
        }
        else if (!this.foundFrom) {
            this.initialInstructions.addInstruction(instruction);
        }
        if (this.foundFrom) {
            this.currentBuildStage.addInstruction(instruction);
        }
        super.addInstruction(instruction);
    }
    setDirective(directive) {
        this.directive = directive;
    }
    getDirective() {
        return this.directive;
    }
    resolveVariable(variable, line) {
        for (let from of this.getFROMs()) {
            let range = from.getRange();
            if (range.start.line <= line && line <= range.end.line) {
                // resolve the FROM variable against the initial ARGs
                let initialARGs = new imageTemplate_1.ImageTemplate();
                for (let instruction of this.initialInstructions.getARGs()) {
                    initialARGs.addInstruction(instruction);
                }
                return initialARGs.resolveVariable(variable, line);
            }
        }
        let image = this.getContainingImage(vscode_languageserver_types_1.Position.create(line, 0));
        let resolvedVariable = image.resolveVariable(variable, line);
        if (resolvedVariable === null) {
            // refers to an uninitialized ARG variable,
            // try resolving it against the initial ARGs then
            let initialARGs = new imageTemplate_1.ImageTemplate();
            for (let instruction of this.initialInstructions.getARGs()) {
                initialARGs.addInstruction(instruction);
            }
            return initialARGs.resolveVariable(variable, line);
        }
        return resolvedVariable;
    }
    getAvailableVariables(currentLine) {
        if (this.getInstructionAt(currentLine) instanceof from_1.From) {
            let variables = [];
            for (let arg of this.getInitialARGs()) {
                let property = arg.getProperty();
                if (property) {
                    variables.push(property.getName());
                }
            }
            return variables;
        }
        let image = this.getContainingImage(vscode_languageserver_types_1.Position.create(currentLine, 0));
        return image ? image.getAvailableVariables(currentLine) : [];
    }
}
exports.Dockerfile = Dockerfile;
