/*
 * Copyright (c) 2013, 2015, 2016, 2018, 2019 Eike Stepper (Loehne, Germany) and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Eike Stepper - initial API and implementation
 */
package org.eclipse.net4j.internal.db;

import org.eclipse.net4j.db.DBException;
import org.eclipse.net4j.db.DBUtil;
import org.eclipse.net4j.db.IDBConnection;
import org.eclipse.net4j.db.IDBPreparedStatement;
import org.eclipse.net4j.db.IDBPreparedStatement.ReuseProbability;
import org.eclipse.net4j.db.IDBSchemaTransaction;
import org.eclipse.net4j.db.jdbc.DelegatingConnection;
import org.eclipse.net4j.util.CheckUtil;
import org.eclipse.net4j.util.collection.HashBag;
import org.eclipse.net4j.util.om.OMPlatform;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.TreeMap;

/**
 * @author Eike Stepper
 */
public final class DBConnection extends DelegatingConnection implements IDBConnection
{
  private static final boolean VALIDATE_CHECKOUTS = OMPlatform.INSTANCE.isProperty("org.eclipse.net4j.internal.db.DBConnection.VALIDATE_CHECKOUTS");

  private final TreeMap<String, DBPreparedStatement> cache = new TreeMap<>();

  private HashBag<DBPreparedStatement> checkOuts;

  private final DBDatabase database;

  private int cacheSize;

  private int lastTouch;

  private boolean closed;

  public DBConnection(DBDatabase database, Connection delegate)
  {
    super(delegate);
    this.database = database;

    if (VALIDATE_CHECKOUTS)
    {
      checkOuts = new HashBag<>();
    }

    try
    {
      delegate.setAutoCommit(false);
    }
    catch (SQLException ex)
    {
      throw new DBException(ex, "SET AUTO COMMIT = false");
    }
  }

  @Override
  public DBDatabase getDatabase()
  {
    return database;
  }

  @Override
  public String getUserID()
  {
    return database.getUserID();
  }

  @Override
  public void close()
  {
    DBUtil.close(getDelegate());
    // System.out.println("-- Open connections: " + --COUNT);
    closed = true;
    database.closeConnection(this);
  }

  @Override
  public boolean isClosed()
  {
    return closed;
  }

  @Override
  public IDBSchemaTransaction openSchemaTransaction()
  {
    DBSchemaTransaction schemaTransaction = database.openSchemaTransaction(this);
    return schemaTransaction;
  }

  @Override
  @Deprecated
  public PreparedStatement prepareStatement(String sql, int resultSetType, int resultSetConcurrency, int resultSetHoldability) throws SQLException
  {
    throw new UnsupportedOperationException();
  }

  @Override
  @Deprecated
  public PreparedStatement prepareStatement(String sql, int autoGeneratedKeys) throws SQLException
  {
    throw new UnsupportedOperationException();
  }

  @Override
  @Deprecated
  public PreparedStatement prepareStatement(String sql, int[] columnIndexes) throws SQLException
  {
    throw new UnsupportedOperationException();
  }

  @Override
  @Deprecated
  public PreparedStatement prepareStatement(String sql, String[] columnNames) throws SQLException
  {
    throw new UnsupportedOperationException();
  }

  @Override
  public IDBPreparedStatement prepareStatement(String sql, ReuseProbability reuseProbability)
  {
    return prepareStatement(sql, ResultSet.TYPE_FORWARD_ONLY, ResultSet.CONCUR_READ_ONLY, reuseProbability);
  }

  @Override
  public IDBPreparedStatement prepareStatement(String sql, int resultSetType, int resultSetConcurrency, ReuseProbability reuseProbability)
  {
    database.beginSchemaAccess(false);

    DBPreparedStatement preparedStatement;
    synchronized (this)
    {
      preparedStatement = cache.remove(sql);
      if (preparedStatement == null)
      {
        try
        {
          PreparedStatement delegate = getDelegate().prepareStatement(sql, resultSetType, resultSetConcurrency);
          preparedStatement = new DBPreparedStatement(this, sql, reuseProbability, delegate);
        }
        catch (SQLException ex)
        {
          throw new DBException(ex);
        }
      }
      else
      {
        --cacheSize;

        DBPreparedStatement nextCached = preparedStatement.getNextCached();
        if (nextCached != null)
        {
          cache.put(sql, nextCached);
          preparedStatement.setNextCached(null);
        }
      }

      if (VALIDATE_CHECKOUTS)
      {
        checkOuts.add(preparedStatement);
      }
    }

    return preparedStatement;
  }

  @Override
  public PreparedStatement prepareStatement(String sql) throws SQLException
  {
    return prepareStatement(sql, ReuseProbability.LOW);
  }

  @Override
  public PreparedStatement prepareStatement(String sql, int resultSetType, int resultSetConcurrency) throws SQLException
  {
    return prepareStatement(sql, resultSetType, resultSetConcurrency, ReuseProbability.LOW);
  }

  public void releasePreparedStatement(DBPreparedStatement preparedStatement)
  {
    try
    {
      if (preparedStatement == null)
      {
        // Bug 276926: Silently accept preparedStatement == null and do nothing.
        return;
      }

      synchronized (this)
      {
        if (VALIDATE_CHECKOUTS)
        {
          checkOuts.remove(preparedStatement);
        }

        preparedStatement.setTouch(++lastTouch);

        String sql = preparedStatement.getSQL();
        DBPreparedStatement cached = cache.put(sql, preparedStatement);
        if (cached != null)
        {
          preparedStatement.setNextCached(cached);
        }

        if (++cacheSize > database.getStatementCacheCapacity())
        {
          String firstKey = cache.firstKey();
          DBPreparedStatement old = cache.remove(firstKey);
          DBPreparedStatement nextCached = old.getNextCached();

          DBUtil.close(old.getDelegate());
          --cacheSize;

          if (nextCached != null)
          {
            cache.put(firstKey, nextCached);
          }
        }
      }
    }
    finally
    {
      database.endSchemaAccess();
    }
  }

  public void invalidateStatementCache()
  {
    synchronized (this)
    {
      if (VALIDATE_CHECKOUTS)
      {
        CheckUtil.checkState(checkOuts.isEmpty(), "Statements are checked out: " + checkOuts);
      }

      // Close all statements in the cache, then clear the cache.
      for (DBPreparedStatement preparedStatement : cache.values())
      {
        while (preparedStatement != null)
        {
          PreparedStatement delegate = preparedStatement.getDelegate();
          DBUtil.close(delegate);

          preparedStatement = preparedStatement.getNextCached();
        }
      }

      cache.clear();
      cacheSize = 0;
    }
  }

  public String convertString(DBPreparedStatement preparedStatement, int parameterIndex, String value)
  {
    return getDatabase().convertString(preparedStatement, parameterIndex, value);
  }

  public String convertString(DBResultSet resultSet, int columnIndex, String value)
  {
    return getDatabase().convertString(resultSet, columnIndex, value);
  }

  public String convertString(DBResultSet resultSet, String columnLabel, String value)
  {
    return getDatabase().convertString(resultSet, columnLabel, value);
  }
}
