/**
 * Copyright (c) 2011 Zoltan Ujhelyi, Andras Okros, Mark Czotter, Istvan Rath and Daniel Varro
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v. 2.0 which is available at
 * http://www.eclipse.org/legal/epl-v20.html.
 * 
 * SPDX-License-Identifier: EPL-2.0
 */
package org.eclipse.viatra.query.patternlanguage.emf.jvmmodel;

import com.google.inject.Inject;
import java.util.Arrays;
import org.apache.log4j.Logger;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.viatra.query.patternlanguage.emf.helper.PatternLanguageHelper;
import org.eclipse.viatra.query.patternlanguage.emf.jvmmodel.EMFPatternLanguageJvmModelInferrerUtil;
import org.eclipse.viatra.query.patternlanguage.emf.jvmmodel.PatternGroupClassInferrer;
import org.eclipse.viatra.query.patternlanguage.emf.jvmmodel.PatternMatchClassInferrer;
import org.eclipse.viatra.query.patternlanguage.emf.jvmmodel.PatternMatchProcessorClassInferrer;
import org.eclipse.viatra.query.patternlanguage.emf.jvmmodel.PatternMatcherClassInferrer;
import org.eclipse.viatra.query.patternlanguage.emf.jvmmodel.PatternQuerySpecificationClassInferrer;
import org.eclipse.viatra.query.patternlanguage.emf.util.EMFJvmTypesBuilder;
import org.eclipse.viatra.query.patternlanguage.emf.util.EMFPatternLanguageGeneratorConfig;
import org.eclipse.viatra.query.patternlanguage.emf.util.IErrorFeedback;
import org.eclipse.viatra.query.patternlanguage.emf.vql.Pattern;
import org.eclipse.viatra.query.patternlanguage.emf.vql.PatternModel;
import org.eclipse.viatra.query.runtime.api.GenericPatternMatch;
import org.eclipse.viatra.query.runtime.api.GenericPatternMatcher;
import org.eclipse.viatra.query.runtime.api.impl.BaseMatcher;
import org.eclipse.viatra.query.runtime.api.impl.BasePatternMatch;
import org.eclipse.xtext.common.types.JvmDeclaredType;
import org.eclipse.xtext.common.types.JvmGenericType;
import org.eclipse.xtext.common.types.JvmMember;
import org.eclipse.xtext.common.types.JvmType;
import org.eclipse.xtext.common.types.JvmTypeReference;
import org.eclipse.xtext.diagnostics.Severity;
import org.eclipse.xtext.xbase.compiler.GeneratorConfig;
import org.eclipse.xtext.xbase.compiler.IGeneratorConfigProvider;
import org.eclipse.xtext.xbase.jvmmodel.AbstractModelInferrer;
import org.eclipse.xtext.xbase.jvmmodel.IJvmDeclaredTypeAcceptor;
import org.eclipse.xtext.xbase.jvmmodel.IJvmModelAssociator;
import org.eclipse.xtext.xbase.lib.Exceptions;
import org.eclipse.xtext.xbase.lib.Extension;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.Procedures.Procedure1;
import org.eclipse.xtext.xbase.lib.StringExtensions;

/**
 * <p>Infers a JVM model from the source model.</p>
 * 
 * <p>The JVM model should contain all elements that would appear in the Java code
 * which is generated from the source model. Other models link against the JVM model rather than the source model.</p>
 * @noreference
 */
@SuppressWarnings("all")
public class EMFPatternLanguageJvmModelInferrer extends AbstractModelInferrer {
  private static final String JVM_MODEL_INFERRER_PREFIX = "org.eclipse.viatra.query.patternlanguage.emf.inferrer";
  
  public static final String INVALID_PATTERN_MODEL_CODE = (EMFPatternLanguageJvmModelInferrer.JVM_MODEL_INFERRER_PREFIX + ".invalid.patternmodel");
  
  public static final String INVALID_TYPEREF_CODE = (EMFPatternLanguageJvmModelInferrer.JVM_MODEL_INFERRER_PREFIX + ".invalid.typeref");
  
  public static final String SPECIFICATION_BUILDER_CODE = (EMFPatternLanguageJvmModelInferrer.JVM_MODEL_INFERRER_PREFIX + ".specification.builder");
  
  @Inject
  private Logger logger;
  
  @Inject
  private IErrorFeedback errorFeedback;
  
  @Inject
  private IGeneratorConfigProvider generatorConfigProvider;
  
  /**
   * convenience API to build and initialize JvmTypes and their members.
   */
  @Inject
  @Extension
  private EMFJvmTypesBuilder _eMFJvmTypesBuilder;
  
  @Inject
  @Extension
  private EMFPatternLanguageJvmModelInferrerUtil _eMFPatternLanguageJvmModelInferrerUtil;
  
  @Inject
  @Extension
  private PatternMatchClassInferrer _patternMatchClassInferrer;
  
  @Inject
  @Extension
  private PatternMatcherClassInferrer _patternMatcherClassInferrer;
  
  @Inject
  @Extension
  private PatternQuerySpecificationClassInferrer _patternQuerySpecificationClassInferrer;
  
  @Inject
  @Extension
  private PatternMatchProcessorClassInferrer _patternMatchProcessorClassInferrer;
  
  @Inject
  @Extension
  private PatternGroupClassInferrer _patternGroupClassInferrer;
  
  @Inject
  @Extension
  private IJvmModelAssociator associator;
  
  /**
   * Is called for each Pattern instance in a resource.
   * 
   * @param pattern - the model to create one or more JvmDeclaredTypes from.
   * @param acceptor - each created JvmDeclaredType without a container should be passed to the acceptor in order get attached to the
   *                   current resource.
   * @param isPreLinkingPhase - whether the method is called in a pre linking phase, i.e. when the global index isn't fully updated. You
   *        must not rely on linking using the index if this is <code>true</code>
   */
  public void inferPattern(final Pattern pattern, final IJvmDeclaredTypeAcceptor acceptor, final boolean isPrelinkingPhase) {
    final EMFPatternLanguageGeneratorConfig config = this.getConfiguration(pattern);
    boolean _isNullOrEmpty = StringExtensions.isNullOrEmpty(pattern.getName());
    boolean _not = (!_isNullOrEmpty);
    if (_not) {
      boolean _isPrivate = PatternLanguageHelper.isPrivate(pattern);
      final boolean isPublic = (!_isPrivate);
      try {
        if (isPublic) {
          EMFPatternLanguageGeneratorConfig.MatcherGenerationStrategy _matcherGenerationStrategy = config.getMatcherGenerationStrategy();
          if (_matcherGenerationStrategy != null) {
            switch (_matcherGenerationStrategy) {
              case SEPARATE_CLASS:
              case NESTED_CLASS:
                this.inferPublic(pattern, acceptor, config, isPrelinkingPhase);
                break;
              case USE_GENERIC:
                this.inferPublicWithNoMatchers(pattern, acceptor, config, isPrelinkingPhase);
                break;
              default:
                break;
            }
          }
        } else {
          this.inferPrivate(pattern, acceptor, config, isPrelinkingPhase);
        }
      } catch (final Throwable _t) {
        if (_t instanceof Exception) {
          final Exception e = (Exception)_t;
          this.logger.error(("Exception during Jvm Model Infer for: " + pattern), e);
        } else {
          throw Exceptions.sneakyThrow(_t);
        }
      }
    }
  }
  
  private EMFPatternLanguageGeneratorConfig getConfiguration(final EObject ctx) {
    final GeneratorConfig _config = this.generatorConfigProvider.get(ctx);
    EMFPatternLanguageGeneratorConfig _xifexpression = null;
    if ((_config instanceof EMFPatternLanguageGeneratorConfig)) {
      _xifexpression = ((EMFPatternLanguageGeneratorConfig)_config);
    } else {
      EMFPatternLanguageGeneratorConfig _xblockexpression = null;
      {
        final EMFPatternLanguageGeneratorConfig newConfig = new EMFPatternLanguageGeneratorConfig();
        newConfig.copy(_config);
        _xblockexpression = newConfig;
      }
      _xifexpression = _xblockexpression;
    }
    final EMFPatternLanguageGeneratorConfig config = _xifexpression;
    return config;
  }
  
  private void inferPublic(final Pattern pattern, final IJvmDeclaredTypeAcceptor acceptor, final EMFPatternLanguageGeneratorConfig config, final boolean isPrelinkingPhase) {
    String _name = pattern.getName();
    String _plus = ("Inferring Jvm Model for pattern" + _name);
    this.logger.debug(_plus);
    final boolean generateMatchProcessors = config.isGenerateMatchProcessors();
    EMFPatternLanguageGeneratorConfig.MatcherGenerationStrategy _matcherGenerationStrategy = config.getMatcherGenerationStrategy();
    final boolean nestedClasses = (_matcherGenerationStrategy == EMFPatternLanguageGeneratorConfig.MatcherGenerationStrategy.NESTED_CLASS);
    final String packageName = this._eMFPatternLanguageJvmModelInferrerUtil.getPackageName(pattern);
    final String utilPackageName = this._eMFPatternLanguageJvmModelInferrerUtil.getUtilPackageName(pattern);
    String _xifexpression = null;
    if (nestedClasses) {
      _xifexpression = packageName;
    } else {
      _xifexpression = utilPackageName;
    }
    final String specificationPackageName = _xifexpression;
    final Procedure1<JvmGenericType> _function = (JvmGenericType it) -> {
      if (nestedClasses) {
        it.setStatic(true);
      } else {
        it.setPackageName(packageName);
        this._eMFJvmTypesBuilder.setFileHeader(it, this._eMFPatternLanguageJvmModelInferrerUtil.getFileComment(pattern));
      }
      EList<JvmTypeReference> _superTypes = it.getSuperTypes();
      JvmTypeReference _typeRef = this._typeReferenceBuilder.typeRef(BasePatternMatch.class);
      this._eMFJvmTypesBuilder.<JvmTypeReference>operator_add(_superTypes, _typeRef);
    };
    final JvmGenericType matchClass = this._eMFJvmTypesBuilder.toClass(pattern, this._eMFPatternLanguageJvmModelInferrerUtil.matchClassName(pattern, config.getMatcherGenerationStrategy()), _function);
    final Procedure1<JvmGenericType> _function_1 = (JvmGenericType it) -> {
      if (nestedClasses) {
        it.setStatic(true);
      } else {
        it.setPackageName(packageName);
        this._eMFJvmTypesBuilder.setFileHeader(it, this._eMFPatternLanguageJvmModelInferrerUtil.getFileComment(pattern));
      }
      EList<JvmTypeReference> _superTypes = it.getSuperTypes();
      JvmTypeReference _typeRef = this._typeReferenceBuilder.typeRef(BaseMatcher.class, this._typeReferenceBuilder.typeRef(matchClass));
      this._eMFJvmTypesBuilder.<JvmTypeReference>operator_add(_superTypes, _typeRef);
    };
    final JvmGenericType matcherClass = this._eMFJvmTypesBuilder.toClass(pattern, this._eMFPatternLanguageJvmModelInferrerUtil.matcherClassName(pattern, config.getMatcherGenerationStrategy()), _function_1);
    final JvmDeclaredType querySpecificationClass = this._patternQuerySpecificationClassInferrer.inferQuerySpecificationClass(pattern, isPrelinkingPhase, specificationPackageName, matcherClass, this._typeReferenceBuilder, this._annotationTypesBuilder, config);
    if (nestedClasses) {
      EList<JvmMember> _members = querySpecificationClass.getMembers();
      this._eMFJvmTypesBuilder.<JvmGenericType>operator_add(_members, matchClass);
      EList<JvmMember> _members_1 = querySpecificationClass.getMembers();
      this._eMFJvmTypesBuilder.<JvmGenericType>operator_add(_members_1, matcherClass);
    }
    final Procedure1<JvmDeclaredType> _function_2 = (JvmDeclaredType it) -> {
      this._patternQuerySpecificationClassInferrer.initializeSpecification(it, pattern, matcherClass, matchClass, config);
    };
    acceptor.<JvmDeclaredType>accept(querySpecificationClass, _function_2);
    final Procedure1<JvmGenericType> _function_3 = (JvmGenericType it) -> {
      this._patternMatchClassInferrer.inferMatchClassElements(it, pattern, querySpecificationClass, this._typeReferenceBuilder, this._annotationTypesBuilder);
    };
    acceptor.<JvmGenericType>accept(matchClass, _function_3);
    final Procedure1<JvmGenericType> _function_4 = (JvmGenericType it) -> {
      this._patternMatcherClassInferrer.inferMatcherClassElements(it, pattern, querySpecificationClass, matchClass, this._typeReferenceBuilder, 
        this._annotationTypesBuilder, config);
    };
    acceptor.<JvmGenericType>accept(matcherClass, _function_4);
    this.associator.associatePrimary(pattern, querySpecificationClass);
    this.associator.associate(pattern, matcherClass);
    this.associator.associate(pattern, querySpecificationClass);
    if (generateMatchProcessors) {
      final JvmDeclaredType processorClass = this._patternMatchProcessorClassInferrer.inferProcessorClass(pattern, isPrelinkingPhase, utilPackageName, matchClass, 
        this._typeReferenceBuilder, this._annotationTypesBuilder, config);
      final Procedure1<JvmDeclaredType> _function_5 = (JvmDeclaredType it) -> {
        this._patternMatchProcessorClassInferrer.inferProcessorClassMethods(processorClass, pattern, matchClass);
      };
      acceptor.<JvmDeclaredType>accept(processorClass, _function_5);
      this.associator.associate(pattern, processorClass);
      if (nestedClasses) {
        EList<JvmMember> _members_2 = querySpecificationClass.getMembers();
        this._eMFJvmTypesBuilder.<JvmDeclaredType>operator_add(_members_2, processorClass);
      }
    }
  }
  
  private void inferPublicWithNoMatchers(final Pattern pattern, final IJvmDeclaredTypeAcceptor acceptor, final EMFPatternLanguageGeneratorConfig config, final boolean isPrelinkingPhase) {
    String _name = pattern.getName();
    String _plus = ("Inferring Jvm Model for private pattern " + _name);
    this.logger.debug(_plus);
    final String utilPackageName = this._eMFPatternLanguageJvmModelInferrerUtil.getPackageName(pattern);
    this.inferQuerySpecificationWithGeneric(pattern, acceptor, config, isPrelinkingPhase, utilPackageName);
  }
  
  private void inferPrivate(final Pattern pattern, final IJvmDeclaredTypeAcceptor acceptor, final EMFPatternLanguageGeneratorConfig config, final boolean isPrelinkingPhase) {
    String _name = pattern.getName();
    String _plus = ("Inferring Jvm Model for private pattern " + _name);
    this.logger.debug(_plus);
    final String utilPackageName = this._eMFPatternLanguageJvmModelInferrerUtil.getInternalSpecificationPackage(pattern);
    this.inferQuerySpecificationWithGeneric(pattern, acceptor, config, isPrelinkingPhase, utilPackageName);
  }
  
  private void inferQuerySpecificationWithGeneric(final Pattern pattern, final IJvmDeclaredTypeAcceptor acceptor, final EMFPatternLanguageGeneratorConfig config, final boolean isPrelinkingPhase, final String packageName) {
    final JvmType matcherClass = this._typeReferenceBuilder.typeRef(GenericPatternMatcher.class).getType();
    final JvmType matchClass = this._typeReferenceBuilder.typeRef(GenericPatternMatch.class).getType();
    final JvmDeclaredType querySpecificationClass = this._patternQuerySpecificationClassInferrer.inferQuerySpecificationClass(pattern, isPrelinkingPhase, packageName, matcherClass, 
      this._typeReferenceBuilder, this._annotationTypesBuilder, config);
    this.associator.associatePrimary(pattern, querySpecificationClass);
    final Procedure1<JvmDeclaredType> _function = (JvmDeclaredType it) -> {
      this._patternQuerySpecificationClassInferrer.initializeSpecification(querySpecificationClass, pattern, matcherClass, matchClass, config);
    };
    acceptor.<JvmDeclaredType>accept(querySpecificationClass, _function);
  }
  
  /**
   * Is called for each PatternModel instance in a resource.
   * 
   * @param model - the model to create one or more JvmDeclaredTypes from.
   * @param acceptor - each created JvmDeclaredType without a container should be passed to the acceptor in order get attached to the
   *                   current resource.
   * @param isPreLinkingPhase - whether the method is called in a pre linking phase, i.e. when the global index isn't fully updated. You
   *        must not rely on linking using the index if this is <code>true</code>
   */
  protected void _infer(final PatternModel model, final IJvmDeclaredTypeAcceptor acceptor, final boolean isPrelinkingPhase) {
    try {
      final EMFPatternLanguageGeneratorConfig config = this.getConfiguration(model);
      EList<Pattern> _patterns = model.getPatterns();
      for (final Pattern pattern : _patterns) {
        this.inferPattern(pattern, acceptor, isPrelinkingPhase);
      }
      String _modelFileName = this._eMFPatternLanguageJvmModelInferrerUtil.modelFileName(model);
      String _plus = ("Inferring Jvm Model for Pattern model " + _modelFileName);
      this.logger.debug(_plus);
      boolean _isEmpty = model.getPatterns().isEmpty();
      boolean _not = (!_isEmpty);
      if (_not) {
        final JvmGenericType groupClass = this._patternGroupClassInferrer.inferPatternGroupClass(model, this._typeReferenceBuilder, config, false);
        final Procedure1<JvmGenericType> _function = (JvmGenericType it) -> {
          this._patternGroupClassInferrer.initializePatternGroup(it, model, this._typeReferenceBuilder, config, false);
        };
        acceptor.<JvmGenericType>accept(groupClass, _function);
        final Function1<Pattern, Boolean> _function_1 = (Pattern it) -> {
          return Boolean.valueOf(PatternLanguageHelper.isPrivate(it));
        };
        boolean _exists = IterableExtensions.<Pattern>exists(model.getPatterns(), _function_1);
        if (_exists) {
          final JvmGenericType privateGroupClass = this._patternGroupClassInferrer.inferPatternGroupClass(model, this._typeReferenceBuilder, config, true);
          final Procedure1<JvmGenericType> _function_2 = (JvmGenericType it) -> {
            this._patternGroupClassInferrer.initializePatternGroup(it, model, this._typeReferenceBuilder, config, true);
          };
          acceptor.<JvmGenericType>accept(privateGroupClass, _function_2);
        }
        this.associator.associatePrimary(model, groupClass);
      }
    } catch (final Throwable _t) {
      if (_t instanceof IllegalArgumentException) {
        final IllegalArgumentException e = (IllegalArgumentException)_t;
        this.errorFeedback.reportErrorNoLocation(model, e.getMessage(), 
          EMFPatternLanguageJvmModelInferrer.INVALID_PATTERN_MODEL_CODE, Severity.ERROR, 
          IErrorFeedback.JVMINFERENCE_ERROR_TYPE);
      } else if (_t instanceof Exception) {
        final Exception e_1 = (Exception)_t;
        this.logger.error(("Exception during Jvm Model Infer for pattern model: " + model), e_1);
      } else {
        throw Exceptions.sneakyThrow(_t);
      }
    }
  }
  
  public void infer(final EObject model, final IJvmDeclaredTypeAcceptor acceptor, final boolean isPrelinkingPhase) {
    if (model instanceof PatternModel) {
      _infer((PatternModel)model, acceptor, isPrelinkingPhase);
      return;
    } else if (model != null) {
      _infer(model, acceptor, isPrelinkingPhase);
      return;
    } else {
      throw new IllegalArgumentException("Unhandled parameter types: " +
        Arrays.<Object>asList(model, acceptor, isPrelinkingPhase).toString());
    }
  }
}
