/**
 * Copyright (c) 2016 Codetrails GmbH.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 */
package org.eclipse.epp.internal.logging.aeri.ide.processors;

import java.util.Set;

import javax.inject.Inject;

import org.eclipse.core.runtime.IStatus;
import org.eclipse.e4.core.contexts.IEclipseContext;
import org.eclipse.epp.internal.logging.aeri.ide.IProcessorDescriptor;
import org.eclipse.epp.logging.aeri.core.IReport;
import org.eclipse.epp.logging.aeri.core.IReportProcessor;
import org.eclipse.epp.logging.aeri.core.IThrowable;
import org.eclipse.epp.logging.aeri.core.util.ModelSwitch;
import org.eclipse.epp.logging.aeri.core.util.Reports;

import com.google.common.base.Optional;
import com.google.common.collect.Sets;

public class WiringErrorAnalyzerProcessor implements IReportProcessor {

    private final class ContainsRequiredExceptionSwitch extends ModelSwitch<Void> {
        boolean isContained = false;

        private Set<String> possibleExceptions = Sets.newHashSet(NoClassDefFoundError.class.getName(), LinkageError.class.getName(),
                ClassNotFoundException.class.getName(), NoSuchMethodError.class.getName(), VerifyError.class.getName());

        @Override
        public Void caseThrowable(IThrowable throwable) {
            String className = throwable.getClassName();
            if (possibleExceptions.contains(className)) {
                isContained = true;
            }
            return null;
        }
    }

    private IProcessorDescriptor descriptor;

    @Inject
    public WiringErrorAnalyzerProcessor(IProcessorDescriptor descriptor) {
        this.descriptor = descriptor;
    }

    @Override
    public boolean canContribute(IStatus status, IEclipseContext context) {
        ContainsRequiredExceptionSwitch containsException = new ContainsRequiredExceptionSwitch();
        Reports.visit(Reports.newReport(status), containsException);
        return containsException.isContained;
    }

    @Override
    public boolean wantsToContribute(IStatus status, IEclipseContext context) {
        return canContribute(status, context);
    }

    @Override
    public void process(IReport report, IStatus status, IEclipseContext context) {
        String comment = (String) context.get(getContextKey());
        if (comment == null) {
            StringBuilder b = new StringBuilder();
            ModelSwitch<Void> visitor = new ModelSwitch<Void>() {

                @Override
                public Void caseThrowable(IThrowable throwable) {
                    Optional<String> bundleComment = new WiringErrorAnalyzer().computeComment(report.getPresentBundles(), throwable);
                    if (bundleComment.isPresent()) {
                        b.append(bundleComment.get());
                    }
                    return null;
                }
            };
            Reports.visit(report, visitor);
            comment = b.toString();
            context.set(getContextKey(), comment);
        }
        if (!comment.isEmpty()) {
            report.getAuxiliaryInformation().put(descriptor.getDirective(), comment);
        }
    }

    protected String getContextKey() {
        return "processor." + descriptor.getDirective() + ".value";
    }
}
