/**
 * Copyright (c) 2010-2012, Abel Hegedus, Istvan Rath and Daniel Varro
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   Abel Hegedus - initial API and implementation
 */
package org.eclipse.viatra.addon.querybasedfeatures.tooling;

import com.google.common.base.Objects;
import com.google.inject.Inject;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Map;
import java.util.Set;
import java.util.function.Consumer;
import org.apache.log4j.Logger;
import org.eclipse.core.runtime.IPath;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EClassifier;
import org.eclipse.emf.ecore.EPackage;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.viatra.addon.querybasedfeatures.runtime.QueryBasedFeatureKind;
import org.eclipse.viatra.addon.querybasedfeatures.runtime.handler.QueryBasedFeatures;
import org.eclipse.viatra.addon.querybasedfeatures.tooling.QueryBasedFeatureParameters;
import org.eclipse.viatra.addon.querybasedfeatures.tooling.SettingDelegateBasedGenerator;
import org.eclipse.viatra.query.patternlanguage.emf.helper.PatternLanguageHelper;
import org.eclipse.viatra.query.patternlanguage.emf.util.IErrorFeedback;
import org.eclipse.viatra.query.patternlanguage.emf.vql.Annotation;
import org.eclipse.viatra.query.patternlanguage.emf.vql.AnnotationParameter;
import org.eclipse.viatra.query.patternlanguage.emf.vql.ClassType;
import org.eclipse.viatra.query.patternlanguage.emf.vql.Pattern;
import org.eclipse.viatra.query.patternlanguage.emf.vql.StringValue;
import org.eclipse.viatra.query.patternlanguage.emf.vql.Type;
import org.eclipse.viatra.query.patternlanguage.emf.vql.ValueReference;
import org.eclipse.viatra.query.patternlanguage.emf.vql.Variable;
import org.eclipse.viatra.query.patternlanguage.emf.vql.VariableReference;
import org.eclipse.viatra.query.tooling.core.generator.ExtensionData;
import org.eclipse.viatra.query.tooling.core.generator.ExtensionGenerator;
import org.eclipse.viatra.query.tooling.core.generator.fragments.IGenerationFragment;
import org.eclipse.viatra.query.tooling.core.generator.genmodel.IVQGenmodelProvider;
import org.eclipse.xtext.diagnostics.Severity;
import org.eclipse.xtext.generator.IFileSystemAccess;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Conversions;
import org.eclipse.xtext.xbase.lib.Exceptions;
import org.eclipse.xtext.xbase.lib.Extension;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.Pair;
import org.eclipse.xtext.xbase.lib.Procedures.Procedure1;
import org.w3c.dom.Element;

/**
 * @author Abel Hegedus
 */
@SuppressWarnings("all")
public class QueryBasedFeatureGenerator implements IGenerationFragment {
  @Inject
  protected IVQGenmodelProvider provider;
  
  @Inject
  protected Logger logger;
  
  @Inject
  protected IErrorFeedback errorFeedback;
  
  @Inject
  @Extension
  protected ExtensionGenerator exGen;
  
  protected static String DERIVED_EXTENSION_POINT = "org.eclipse.viatra.query.runtime.base.wellbehaving.derived.features";
  
  protected static String DERIVED_ERROR_CODE = "org.eclipse.viatra.query.runtime.querybasedfeature.error";
  
  protected static String DERIVED_EXTENSION_PREFIX = "extension.derived.";
  
  protected static Map<String, QueryBasedFeatureKind> kinds = CollectionLiterals.<String, QueryBasedFeatureKind>newHashMap(
    Pair.<String, QueryBasedFeatureKind>of("single", QueryBasedFeatureKind.SINGLE_REFERENCE), 
    Pair.<String, QueryBasedFeatureKind>of("many", QueryBasedFeatureKind.MANY_REFERENCE), 
    Pair.<String, QueryBasedFeatureKind>of("sum", QueryBasedFeatureKind.SUM), 
    Pair.<String, QueryBasedFeatureKind>of("iteration", QueryBasedFeatureKind.ITERATION));
  
  private SettingDelegateBasedGenerator delegateBasedGenerator = new SettingDelegateBasedGenerator(this);
  
  @Override
  public IPath[] getAdditionalBinIncludes() {
    return ((IPath[])Conversions.unwrapArray(CollectionLiterals.<IPath>newArrayList(), IPath.class));
  }
  
  @Override
  public String[] getProjectDependencies() {
    return ((String[])Conversions.unwrapArray(CollectionLiterals.<String>newArrayList("org.eclipse.viatra.addon.querybasedfeatures.runtime"), String.class));
  }
  
  @Override
  public String getProjectPostfix() {
    return null;
  }
  
  @Override
  public void generateFiles(final Pattern pattern, final IFileSystemAccess fsa) {
    this.processAnnotations(pattern, true);
  }
  
  @Override
  public void cleanUp(final Pattern pattern, final IFileSystemAccess fsa) {
    this.processAnnotations(pattern, false);
  }
  
  private void processAnnotations(final Pattern pattern, final boolean generate) {
    Collection<Annotation> _annotationsByName = PatternLanguageHelper.getAnnotationsByName(pattern, QueryBasedFeatures.ANNOTATION_LITERAL);
    for (final Annotation annotation : _annotationsByName) {
      this.delegateBasedGenerator.updateAnnotations(pattern, annotation, generate);
    }
  }
  
  @Override
  public Iterable<ExtensionData> extensionContribution(final Pattern pattern) {
    final ArrayList<QueryBasedFeatureParameters> parameterList = CollectionLiterals.<QueryBasedFeatureParameters>newArrayList();
    Collection<Annotation> _annotationsByName = PatternLanguageHelper.getAnnotationsByName(pattern, QueryBasedFeatures.ANNOTATION_LITERAL);
    for (final Annotation annotation : _annotationsByName) {
      try {
        QueryBasedFeatureParameters _processDerivedFeatureAnnotation = this.processDerivedFeatureAnnotation(pattern, annotation, false);
        parameterList.add(_processDerivedFeatureAnnotation);
      } catch (final Throwable _t) {
        if (_t instanceof IllegalArgumentException) {
          final IllegalArgumentException e = (IllegalArgumentException)_t;
          this.logger.error(e.getMessage());
        } else {
          throw Exceptions.sneakyThrow(_t);
        }
      }
    }
    boolean _isEmpty = parameterList.isEmpty();
    if (_isEmpty) {
      return CollectionLiterals.<ExtensionData>newArrayList();
    }
    final Procedure1<Element> _function = (Element it) -> {
      final Consumer<QueryBasedFeatureParameters> _function_1 = (QueryBasedFeatureParameters parameters) -> {
        final Procedure1<Element> _function_2 = (Element it_1) -> {
          this.exGen.contribAttribute(it_1, "package-nsUri", parameters.ePackage.getNsURI());
          this.exGen.contribAttribute(it_1, "classifier-name", parameters.source.getName());
          this.exGen.contribAttribute(it_1, "feature-name", parameters.feature.getName());
        };
        this.exGen.contribElement(it, "wellbehaving-derived-feature", _function_2);
      };
      parameterList.forEach(_function_1);
    };
    final ArrayList<ExtensionData> wellbehaving = CollectionLiterals.<ExtensionData>newArrayList(
      this.exGen.contribExtension(this.derivedContributionId(pattern), QueryBasedFeatureGenerator.DERIVED_EXTENSION_POINT, _function));
    return wellbehaving;
  }
  
  @Override
  public Collection<Pair<String, String>> getRemovableExtensions() {
    return CollectionLiterals.<Pair<String, String>>newArrayList(
      Pair.<String, String>of(QueryBasedFeatureGenerator.DERIVED_EXTENSION_PREFIX, QueryBasedFeatureGenerator.DERIVED_EXTENSION_POINT));
  }
  
  @Override
  public Iterable<Pair<String, String>> removeExtension(final Pattern pattern) {
    return CollectionLiterals.<Pair<String, String>>newArrayList(
      Pair.<String, String>of(this.derivedContributionId(pattern), QueryBasedFeatureGenerator.DERIVED_EXTENSION_POINT));
  }
  
  protected String derivedContributionId(final Pattern pattern) {
    String _fullyQualifiedName = PatternLanguageHelper.getFullyQualifiedName(pattern);
    return (QueryBasedFeatureGenerator.DERIVED_EXTENSION_PREFIX + _fullyQualifiedName);
  }
  
  protected QueryBasedFeatureParameters processDerivedFeatureAnnotation(final Pattern pattern, final Annotation annotation, final boolean feedback) {
    final QueryBasedFeatureParameters parameters = new QueryBasedFeatureParameters();
    parameters.pattern = pattern;
    parameters.annotation = annotation;
    String sourceTmp = "";
    String targetTmp = "";
    String featureTmp = "";
    String kindTmp = "";
    boolean keepCacheTmp = true;
    boolean useAsSurrogate = false;
    int _size = pattern.getParameters().size();
    boolean _lessThan = (_size < 2);
    if (_lessThan) {
      if (feedback) {
        this.errorFeedback.reportError(pattern, "Pattern has less than 2 parameters!", QueryBasedFeatureGenerator.DERIVED_ERROR_CODE, Severity.ERROR, IErrorFeedback.FRAGMENT_ERROR_TYPE);
      }
      String _fullyQualifiedName = PatternLanguageHelper.getFullyQualifiedName(pattern);
      String _plus = ("Query-based feature pattern " + _fullyQualifiedName);
      String _plus_1 = (_plus + " has less than 2 parameters!");
      throw new IllegalArgumentException(_plus_1);
    }
    EList<AnnotationParameter> _parameters = annotation.getParameters();
    for (final AnnotationParameter ap : _parameters) {
      boolean _matches = ap.getName().matches("source");
      if (_matches) {
        ValueReference _value = ap.getValue();
        sourceTmp = ((VariableReference) _value).getVar();
      } else {
        boolean _matches_1 = ap.getName().matches("target");
        if (_matches_1) {
          ValueReference _value_1 = ap.getValue();
          targetTmp = ((VariableReference) _value_1).getVar();
        } else {
          boolean _matches_2 = ap.getName().matches("feature");
          if (_matches_2) {
            ValueReference _value_2 = ap.getValue();
            featureTmp = ((StringValue) _value_2).getValue();
          } else {
            boolean _matches_3 = ap.getName().matches("kind");
            if (_matches_3) {
              ValueReference _value_3 = ap.getValue();
              kindTmp = ((StringValue) _value_3).getValue();
            } else {
              boolean _matches_4 = ap.getName().matches("keepCache");
              if (_matches_4) {
                keepCacheTmp = (PatternLanguageHelper.<Boolean>getValue(ap.getValue(), Boolean.class)).booleanValue();
              } else {
                boolean _matches_5 = ap.getName().matches("useAsSurrogate");
                if (_matches_5) {
                  useAsSurrogate = (PatternLanguageHelper.<Boolean>getValue(ap.getValue(), Boolean.class)).booleanValue();
                }
              }
            }
          }
        }
      }
    }
    boolean _equals = Objects.equal(featureTmp, "");
    if (_equals) {
      featureTmp = pattern.getName();
    }
    boolean _equals_1 = Objects.equal(sourceTmp, "");
    if (_equals_1) {
      sourceTmp = pattern.getParameters().get(0).getName();
    }
    boolean _contains = PatternLanguageHelper.getParameterPositionsByName(pattern).keySet().contains(sourceTmp);
    boolean _not = (!_contains);
    if (_not) {
      if (feedback) {
        this.errorFeedback.reportError(annotation, (("No parameter for source " + sourceTmp) + " !"), QueryBasedFeatureGenerator.DERIVED_ERROR_CODE, Severity.ERROR, IErrorFeedback.FRAGMENT_ERROR_TYPE);
      }
      String _fullyQualifiedName_1 = PatternLanguageHelper.getFullyQualifiedName(pattern);
      String _plus_2 = ("Query-based feature pattern " + _fullyQualifiedName_1);
      String _plus_3 = (_plus_2 + ": No parameter for source ");
      String _plus_4 = (_plus_3 + sourceTmp);
      String _plus_5 = (_plus_4 + " !");
      throw new IllegalArgumentException(_plus_5);
    }
    final Variable sourcevar = pattern.getParameters().get((PatternLanguageHelper.getParameterPositionsByName(pattern).get(sourceTmp)).intValue());
    final Type sourceType = sourcevar.getType();
    if (((!(sourceType instanceof ClassType)) || (!(((ClassType) sourceType).getClassname() instanceof EClass)))) {
      if (feedback) {
        this.errorFeedback.reportError(sourcevar, (("Source " + sourceTmp) + " is not EClass!"), QueryBasedFeatureGenerator.DERIVED_ERROR_CODE, Severity.ERROR, IErrorFeedback.FRAGMENT_ERROR_TYPE);
      }
      String _fullyQualifiedName_2 = PatternLanguageHelper.getFullyQualifiedName(pattern);
      String _plus_6 = ("Query-based feature pattern " + _fullyQualifiedName_2);
      String _plus_7 = (_plus_6 + ": Source ");
      String _plus_8 = (_plus_7 + sourceTmp);
      String _plus_9 = (_plus_8 + " is not EClass!");
      throw new IllegalArgumentException(_plus_9);
    }
    EClassifier _classname = ((ClassType) sourceType).getClassname();
    EClass source = ((EClass) _classname);
    parameters.sourceVar = sourceTmp;
    parameters.source = source;
    if (((source == null) || (source.getEPackage() == null))) {
      if (feedback) {
        this.errorFeedback.reportError(sourcevar, "Source EClass or EPackage not found!", QueryBasedFeatureGenerator.DERIVED_ERROR_CODE, Severity.ERROR, IErrorFeedback.FRAGMENT_ERROR_TYPE);
      }
      String _fullyQualifiedName_3 = PatternLanguageHelper.getFullyQualifiedName(pattern);
      String _plus_10 = ("Query-based feature pattern " + _fullyQualifiedName_3);
      String _plus_11 = (_plus_10 + ": Source EClass or EPackage not found!");
      throw new IllegalArgumentException(_plus_11);
    }
    final EPackage pckg = source.getEPackage();
    if ((pckg == null)) {
      if (feedback) {
        this.errorFeedback.reportError(sourcevar, "EPackage not found!", QueryBasedFeatureGenerator.DERIVED_ERROR_CODE, Severity.ERROR, IErrorFeedback.FRAGMENT_ERROR_TYPE);
      }
      String _fullyQualifiedName_4 = PatternLanguageHelper.getFullyQualifiedName(pattern);
      String _plus_12 = ("Query-based feature pattern " + _fullyQualifiedName_4);
      String _plus_13 = (_plus_12 + ": EPackage not found!");
      throw new IllegalArgumentException(_plus_13);
    }
    parameters.ePackage = pckg;
    final String featureString = featureTmp;
    final Function1<EStructuralFeature, Boolean> _function = (EStructuralFeature it) -> {
      String _name = it.getName();
      return Boolean.valueOf(Objects.equal(_name, featureString));
    };
    final Iterable<EStructuralFeature> features = IterableExtensions.<EStructuralFeature>filter(source.getEAllStructuralFeatures(), _function);
    int _size_1 = IterableExtensions.size(features);
    boolean _notEquals = (_size_1 != 1);
    if (_notEquals) {
      if (feedback) {
        String _name = source.getName();
        String _plus_14 = ((("Feature " + featureTmp) + " not found in class ") + _name);
        String _plus_15 = (_plus_14 + "!");
        this.errorFeedback.reportError(annotation, _plus_15, QueryBasedFeatureGenerator.DERIVED_ERROR_CODE, Severity.ERROR, IErrorFeedback.FRAGMENT_ERROR_TYPE);
      }
      String _fullyQualifiedName_5 = PatternLanguageHelper.getFullyQualifiedName(pattern);
      String _plus_16 = ("Query-based feature pattern " + _fullyQualifiedName_5);
      String _plus_17 = (_plus_16 + ": Feature ");
      String _plus_18 = (_plus_17 + featureTmp);
      String _plus_19 = (_plus_18 + " not found in class ");
      String _name_1 = source.getName();
      String _plus_20 = (_plus_19 + _name_1);
      String _plus_21 = (_plus_20 + "!");
      throw new IllegalArgumentException(_plus_21);
    }
    final EStructuralFeature feature = features.iterator().next();
    parameters.feature = feature;
    parameters.useAsSurrogate = useAsSurrogate;
    if ((!useAsSurrogate)) {
      boolean _not_1 = (!((feature.isDerived() && feature.isTransient()) && feature.isVolatile()));
      if (_not_1) {
        if (feedback) {
          this.errorFeedback.reportError(annotation, (("Feature " + featureTmp) + " must be set derived, transient, volatile!"), QueryBasedFeatureGenerator.DERIVED_ERROR_CODE, Severity.ERROR, IErrorFeedback.FRAGMENT_ERROR_TYPE);
        }
        String _fullyQualifiedName_6 = PatternLanguageHelper.getFullyQualifiedName(pattern);
        String _plus_22 = ("Query-based feature pattern " + _fullyQualifiedName_6);
        String _plus_23 = (_plus_22 + ": Feature ");
        String _plus_24 = (_plus_23 + featureTmp);
        String _plus_25 = (_plus_24 + " must be set derived, transient, volatile!");
        throw new IllegalArgumentException(_plus_25);
      }
      final Resource resource = pckg.eResource();
      final URI uri = resource.getURI();
      boolean _not_2 = (!(uri.isFile() || uri.isPlatformResource()));
      if (_not_2) {
        parameters.resourceWritable = false;
        final ValueReference useModelCodeRef = PatternLanguageHelper.getFirstAnnotationParameter(annotation, "generateIntoModelCode");
        boolean useModelCode = false;
        if ((useModelCodeRef != null)) {
          useModelCode = (PatternLanguageHelper.<Boolean>getValue(useModelCodeRef, Boolean.class)).booleanValue();
        }
        final boolean annotationsOK = QueryBasedFeatures.checkEcoreAnnotation(pckg, feature, PatternLanguageHelper.getFullyQualifiedName(pattern), useModelCode);
        if ((!annotationsOK)) {
          final String message = String.format(
            "Ecore package of %s must be writable by Query-based Feature generator, but resource with URI %s is not!", 
            source.getName(), 
            uri.toString());
          this.errorFeedback.reportError(sourcevar, message, QueryBasedFeatureGenerator.DERIVED_ERROR_CODE, Severity.ERROR, 
            IErrorFeedback.FRAGMENT_ERROR_TYPE);
          String _fullyQualifiedName_7 = PatternLanguageHelper.getFullyQualifiedName(pattern);
          String _plus_26 = ("Query-based feature pattern " + _fullyQualifiedName_7);
          String _plus_27 = (_plus_26 + ": ");
          String _plus_28 = (_plus_27 + message);
          throw new IllegalArgumentException(_plus_28);
        }
      }
    }
    boolean _equals_2 = Objects.equal(kindTmp, "");
    if (_equals_2) {
      boolean _isMany = feature.isMany();
      if (_isMany) {
        kindTmp = "many";
      } else {
        kindTmp = "single";
      }
    }
    boolean _contains_1 = QueryBasedFeatureGenerator.kinds.keySet().contains(kindTmp);
    boolean _not_3 = (!_contains_1);
    if (_not_3) {
      if (feedback) {
        Set<String> _keySet = QueryBasedFeatureGenerator.kinds.keySet();
        String _plus_29 = ("Kind not set, or not in " + _keySet);
        String _plus_30 = (_plus_29 + "!");
        this.errorFeedback.reportError(annotation, _plus_30, QueryBasedFeatureGenerator.DERIVED_ERROR_CODE, Severity.ERROR, IErrorFeedback.FRAGMENT_ERROR_TYPE);
      }
      String _fullyQualifiedName_8 = PatternLanguageHelper.getFullyQualifiedName(pattern);
      String _plus_31 = ("Query-based feature pattern " + _fullyQualifiedName_8);
      String _plus_32 = (_plus_31 + ": Kind not set, or not in ");
      Set<String> _keySet_1 = QueryBasedFeatureGenerator.kinds.keySet();
      String _plus_33 = (_plus_32 + _keySet_1);
      String _plus_34 = (_plus_33 + "!");
      throw new IllegalArgumentException(_plus_34);
    }
    final QueryBasedFeatureKind kind = QueryBasedFeatureGenerator.kinds.get(kindTmp);
    parameters.kind = kind;
    boolean _equals_3 = Objects.equal(targetTmp, "");
    if (_equals_3) {
      targetTmp = pattern.getParameters().get(1).getName();
    } else {
      boolean _contains_2 = PatternLanguageHelper.getParameterPositionsByName(pattern).keySet().contains(targetTmp);
      boolean _not_4 = (!_contains_2);
      if (_not_4) {
        if (feedback) {
          this.errorFeedback.reportError(annotation, (("Target " + targetTmp) + " not set or no such parameter!"), QueryBasedFeatureGenerator.DERIVED_ERROR_CODE, Severity.ERROR, IErrorFeedback.FRAGMENT_ERROR_TYPE);
        }
        String _fullyQualifiedName_9 = PatternLanguageHelper.getFullyQualifiedName(pattern);
        String _plus_35 = ("Derived feature pattern " + _fullyQualifiedName_9);
        String _plus_36 = (_plus_35 + ": Target ");
        String _plus_37 = (_plus_36 + targetTmp);
        String _plus_38 = (_plus_37 + " not set or no such parameter!");
        throw new IllegalArgumentException(_plus_38);
      }
    }
    parameters.targetVar = targetTmp;
    parameters.keepCache = keepCacheTmp;
    return parameters;
  }
}
