/**
 * Copyright (c) 2021, 2026 Contributors to the Eclipse Foundation
 *
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 */
package org.eclipse.lsat.mpt.transformation;

import activity.Activity
import activity.ActivitySet
import common.CommonFactory
import common.TypeDefinition
import dispatching.ActivityDispatching
import java.util.List

import static activity.util.ActivityParametersUtil.*

import static extension activity.util.ActivityParametersUtil.findCandidate
import static extension activity.util.ActivityParametersUtil.getAllExpandedNames
import static extension activity.util.ActivityParametersUtil.getDeclarations
import common.HasName
import activity.Claim
import machine.ResourceItem
import machine.Resource
import machine.IResource

/**
 * Add a named Activity for all combinations of ResourceItems.
 * The original (unusable) activity is removed.
 */
final class ReduceActivityDispatching {

    private new(){}
    /**
     * Reduces expanded activities back to their original format stored on disk.
     * Activities refer to the supplied activitySet
     */
    static def void reduce(ActivityDispatching activityDispatching, ActivitySet activitySet) {
        //collect activities
        val allActivities = newHashSet
        allActivities += activitySet.activities
        allActivities += activitySet.loadAll.filter(ActivitySet).flatMap[activities];
        val expandMap = newLinkedHashMap
        allActivities.forEach[a| a.getAllExpandedNames.forEach[expandMap.put(it,a)]]
        val importActivitySet = CommonFactory.eINSTANCE.createImport();
        // make sure activitySet resource can be resolved.
        importActivitySet.importURI = activitySet.eResource.URI.toString
        activityDispatching.imports += importActivitySet
        activityDispatching.dispatchGroups.flatMap[dispatches].forEach [ dis |
            val activity = expandMap.get(dis.activity.name);
            val parNames = getParameterNames(activity.name, dis.activity.name);
            var resourceItemsFromActivity = dis.activity.nodes.filter(Claim).map[resource].filter[it instanceof ResourceItem || it instanceof Resource].toList
            
            val declarations = activity.declarations
            if (parNames.size != declarations.size) {
                throw new RuntimeException("Cannot convert expanded activity")
            }
            val iter = declarations.iterator
            val resources  = resourceItemsFromActivity
            for (parName : parNames) {
                val parameter = CommonFactory.eINSTANCE.createParameter();
                val declaration = iter.next()
                for (var i=0; i<parName.size; i++) {
                    val namedReference = activity.findItem(resources, declaration,  parName.get(i), parameter.references.map[reference])
                    if(namedReference instanceof ResourceItem ){
                        resources.add(namedReference);
                    }
                    if (namedReference === null) {
                        throw new RuntimeException("Cannot find item with name " + parName.get(i))
                    }
                    var ref = CommonFactory.eINSTANCE.createParameterReference();
                    ref.reference = namedReference
                    parameter.references += ref
                }
                dis.parameters += parameter
            }
            dis.activity = activity
        ]
        activityDispatching.imports -= importActivitySet
        // replace activity import with reduced version.
        // this construction is used to keep comment around it at the right place
         activityDispatching.imports.findFirst[importURI.contains(".activity")].importURI = activitySet.eResource.URI.toString
    }

    /**
     * returns the model item for itemName base on the type definition of parameter
     */
    private static def findItem(Activity activity, List<? extends IResource>  resources, TypeDefinition declaration,  String itemName, List<? extends HasName> current) {
        val candidates = activity.findCandidate(resources, declaration, current)
        return candidates.findFirst[name==itemName]
    }

    /**
     * returns a map with key all possible activity names and activity as values
     */
    private static def expandNames(List<Activity> activities) {
        ExpandActivities.expandNames(activities)
    }

}
