/*
 * Copyright (c) 2021, 2026 Contributors to the Eclipse Foundation
 *
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 */
package org.eclipse.lsat.motioncalculator.json.http;

import java.io.IOException;
import java.net.URI;
import java.net.http.HttpClient;
import java.net.http.HttpRequest;
import java.net.http.HttpResponse;
import java.nio.charset.StandardCharsets;
import java.time.Duration;
import java.util.LinkedHashMap;

import org.eclipse.core.runtime.preferences.InstanceScope;
import org.eclipse.lsat.motioncalculator.json.JsonRequest;
import org.eclipse.lsat.motioncalculator.json.JsonResponse;
import org.eclipse.lsat.motioncalculator.json.JsonSerializer;
import org.eclipse.lsat.motioncalculator.json.JsonServer;
import org.eclipse.swt.widgets.Display;
import org.eclipse.ui.console.ConsolePlugin;
import org.eclipse.ui.console.IConsole;
import org.eclipse.ui.console.IConsoleManager;
import org.eclipse.ui.console.MessageConsole;
import org.eclipse.ui.console.MessageConsoleStream;
import org.osgi.service.prefs.Preferences;

public class HttpJsonServer implements JsonServer {
    /**
     *
     */
    private static final int MAX_CACHE_SIZE = 1000;

    /**
     *
     */
    private static final String CONSOLE_NAME = "Console";

    private static final Preferences PREFERENCES = InstanceScope.INSTANCE
            .getNode("org.eclipse.lsat.motioncalculator.json.http");

    private static final String ARROW_LEFT = "\u2190 ";

    private static final String ARROW_RIGHT = "\u2192 ";

    private static Boolean write;

    private static HttpJsonServer instance;

    private final HttpClient httpClient;

    private final String serverUrl;

    private final LinkedHashMap<String, String> cache = new LinkedHashMap<String, String>();

    private MessageConsole console;

    public static HttpJsonServer getServer() {
        synchronized (PREFERENCES) {
            boolean writeMessages = PREFERENCES.getBoolean("WRITE_MESSAGES", false);
            String href = PREFERENCES.get("HREF", null);
            if (href != null && !href.toLowerCase().matches("https?://.*")) {
                href = "http://" + href;
            }
            if (write == null || instance == null || href != instance.serverUrl
                    || writeMessages != write.booleanValue())
            {
                write = writeMessages;
                instance = new HttpJsonServer(href, writeMessages);
            }
        }
        return instance;
    }

    /**
     * Creates a new HTTP client that will send requests to the specified URL.
     *
     * @param serverUrl The URL of the server to send requests to
     */
    public HttpJsonServer(String serverUrl, boolean write) {
        this.serverUrl = serverUrl;
        this.httpClient = HttpClient.newBuilder().version(HttpClient.Version.HTTP_2)
                .connectTimeout(Duration.ofSeconds(10)).build();
        if (write) {
            console = getConsole();
        }
    }

    /**
     * Gets or creates a console with the specified name
     */
    private static MessageConsole getConsole() {
        ConsolePlugin plugin = ConsolePlugin.getDefault();
        IConsoleManager consoleManager = plugin.getConsoleManager();
        IConsole[] existingConsoles = consoleManager.getConsoles();

        // Check if console already exists
        for (IConsole console: existingConsoles) {
            if (CONSOLE_NAME.equals(console.getName())) {
                return (MessageConsole)console;
            }
        }

        // Console does not exist, create a new one
        MessageConsole newConsole = new MessageConsole(CONSOLE_NAME, null);
        consoleManager.addConsoles(new IConsole[] {newConsole});

        // Show the console
        consoleManager.showConsoleView(newConsole);

        return newConsole;
    }

    /**
     * Sends a POST request with the provided content to the server.
     *
     * @param content The string content to send in the POST request
     * @return The server's response as a string
     * @throws Exception If an error occurs during the HTTP request
     */
    @Override
    public String request(String content) {
        var cachedResponse = cache.remove(content);
        if (cachedResponse != null) {
            cache.put(content, cachedResponse);
            return cachedResponse;
        }
        try {
            HttpRequest request = HttpRequest.newBuilder().uri(URI.create(serverUrl))
                    .header("Content-Type", "application/json")
                    .POST(HttpRequest.BodyPublishers.ofString(content, StandardCharsets.UTF_8)).build();

            logRequest(content);
            HttpResponse<String> response = httpClient.send(request, HttpResponse.BodyHandlers.ofString());

            var result = response.body();
            cache.put(content, result);
            if (cache.size() > MAX_CACHE_SIZE) {
                cache.entrySet().remove(cache.firstEntry());
            }
            logResponse(result);
            return result;
        } catch (IOException | InterruptedException e) {
            return error(content, e.getMessage(), e);
        }
    }

    private void logRequest(String request) {
        // no need to prettify
        log(ARROW_RIGHT, request);
    }

    private void logResponse(String response) {
        if (console != null) {
            // prettify
            try {
                JsonResponse r = JsonSerializer.createResponse(response);
                response = JsonSerializer.toJson(r);
                if (response == null || response.isEmpty()) {
                    response = "{}";
                }
            } catch (Exception e) {
                // use original if it can't be prettified;
            }
            log(ARROW_LEFT, response);
        }
    }

    /**
     * Write a message to the console with the specified color
     */
    private void log(String... strs) {
        if (console != null) {
            Display.getDefault().asyncExec(() -> {
                MessageConsoleStream stream = console.newMessageStream();

                for (String str: strs) {
                    stream.print(str);
                }
                stream.println();

                // Clean up resources
                try {
                    stream.flush();
                    stream.close();
                } catch (Exception e) {
                    e.printStackTrace();
                }
            });
        }
    }

    private String error(String request, String message, Throwable error) {
        JsonRequest jsonRequest = JsonSerializer.createRequest(request);
        JsonResponse jsonResponse = new JsonResponse();
        jsonResponse.setRequestType(jsonRequest.getRequestType());
        jsonResponse
                .setErrorMessage(message + " " + serverUrl + ((error != null) ? " error: " + error.getMessage() : ""));
        String response = JsonSerializer.toJson(jsonResponse);
        return response;
    }
}
