/**
 * Copyright (c) 2021, 2024 Contributors to the Eclipse Foundation
 *
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 */
/*
 * generated by Xtext
 */
package org.eclipse.lsat.machine.teditor.validation

import machine.Axis
import machine.BidirectionalPath
import machine.FullMeshPath
import machine.Import
import machine.MachinePackage
import machine.Peripheral
import machine.PeripheralType
import machine.Resource
import machine.SymbolicPosition
import machine.UnidirectionalPath
import org.eclipse.emf.common.util.URI
import org.eclipse.xtext.EcoreUtil2
import org.eclipse.xtext.validation.Check

import static extension org.eclipse.lsat.common.util.IterableUtil.*
import static extension org.eclipse.lsat.common.xtend.Queries.*
import machine.Machine

/**
 * Custom validation rules. 
 * 
 * see http://www.eclipse.org/Xtext/documentation.html#validation
 */
class MachineValidator extends AbstractMachineValidator {

    public static val SETPOINTS_NO_UNIT = 'noUnitForSetPoint'
    public static val UNKNOWN_CONVERSION_AXIS_TO_SETPOINT = 'unknownConversionAxisToSetPoint'
    public static val SAME_SOURCE_AND_TARGET_UNID_PATH_FOR_POSITION = 'sameSourceAndTargetForPositionUNI'
    public static val SAME_SOURCE_AND_TARGET_BID_PATH_FOR_POSITION = 'sameSourceAndTargetForPositionBI'
    public static val SAME_SOURCE_AND_TARGET_FUM_PATH_FOR_POSITION = 'sameSourceAndTargetForPositionFU'
    public static val PROFILE_NOT_USED = 'profileNotUsed'
    public static val CONFLICTING_POSITIONS = 'conflictingPositions'
    public static val INVALID_IMPORT = 'invalidImport'
    public static val INVALID_MUTUAL_EXCLUSIONS = 'invalidMutualExclusions'
    
    @Check
    def checkImportIsValid(Import imp) {
       try {
            val isImportUriValid = EcoreUtil2.isValidUri(imp, URI.createURI(imp.importURI))
            if (!isImportUriValid) {
                error('''The import «imp.importURI» cannot be resolved. Make sure that the name is spelled correctly.''',
                    imp, MachinePackage.Literals.IMPORT__IMPORT_URI, INVALID_IMPORT) 
            }
            val isUnderstood = imp.importURI.matches(".*\\.(machine)")
            if (!isUnderstood) {
                error('''Importing «imp.importURI» is not allowed. Only 'machine' files are allowed''',
                    imp, MachinePackage.Literals.IMPORT__IMPORT_URI, INVALID_IMPORT)
            }
       } catch (IllegalArgumentException e ) {
             error('''The import «imp.importURI» is not a valid URI.''', imp,
                MachinePackage.Literals.IMPORT__IMPORT_URI, INVALID_IMPORT)
       }  
    }

    @Check
    def checkUnknownConversionAxisToSetPoint(Axis axis) {
        if ((!axis.setPoints.isEmpty) && (axis.setPoints.xcollectOne[unit].asSet.size > 1)) {
            error('Setpoints of axes should have same unit', MachinePackage.Literals.AXIS__SET_POINTS,
                SETPOINTS_NO_UNIT)
        }
    }

    @Check
    def checkEqualSetPointUnits(PeripheralType peripheralType) {
        if (!peripheralType.conversion.isNullOrEmpty) {
            return
        }
        val conditionIsMet = peripheralType.axes.forall[axis|axis.setPoints.forall[setPoint|axis.unit == setPoint.unit]]

        if (!conditionIsMet) {
            error('Don\'t know how to convert axes to setpoints, please specifiy a conversion.',
                MachinePackage.Literals.PERIPHERAL_TYPE__SET_POINTS, UNKNOWN_CONVERSION_AXIS_TO_SETPOINT)
        }
    }

    @Check
    def checkEqualSourceTargetUnidirectional(UnidirectionalPath undPath) {
        if ((undPath.source !== null && undPath.target !== null) && (undPath.source.name == undPath.target.name)) {
            warning('Source and target should not refer to same position',
                MachinePackage.Literals.UNIDIRECTIONAL_PATH__SOURCE, SAME_SOURCE_AND_TARGET_UNID_PATH_FOR_POSITION)
        }
    }

    @Check
    def checkEqualEndPointsBidirectional(BidirectionalPath biPath) {
        if ((!biPath.endPoints.isEmpty) && (biPath.endPoints.xcollectOne[position].containsDuplicates)) {
            warning('Source and target should not refer to same position',
                MachinePackage.Literals.BIDIRECTIONAL_PATH__END_POINTS, SAME_SOURCE_AND_TARGET_BID_PATH_FOR_POSITION)
        }
    }

    @Check
    def checkEqualEndPointsFullMesh(FullMeshPath fullPath) {
        if ((!fullPath.endPoints.isEmpty) && (fullPath.endPoints.xcollectOne[position].containsDuplicates)) {
            error('Positions in full mesh should be unique', MachinePackage.Literals.FULL_MESH_PATH__END_POINTS,
                SAME_SOURCE_AND_TARGET_FUM_PATH_FOR_POSITION)
        }
    }


    @Check
    def checkResourcesUnique(Machine machine) {
        machine.resources.groupBy[name].values.filter[size > 1].flatten.forEach [
            error('''Resource names must be unique. Please remove all duplicate instances.''',
                machine, MachinePackage.Literals.MACHINE__RESOURCES, machine.resources.indexOf(it))
        ]
    }

    @Check
    def checkResourceItemsUnique(Resource resource) {
        resource.items.groupBy[name].values.filter[size > 1].flatten.forEach [
            error('''Items names must be unique. Please remove all duplicate instances.''',
                resource, MachinePackage.Literals.RESOURCE__ITEMS, resource.items.indexOf(it))
        ]
    }


    @Check
    def checkPositionHasAxes(Peripheral peripheral) {
        if (peripheral.type.axes.empty) {
            val items = newLinkedHashMap
            if (!peripheral.axisPositions.empty) {
                items.put('AxisPositions', MachinePackage.Literals.PERIPHERAL__AXIS_POSITIONS)
            }
            if (!peripheral.paths.empty) {
                items.put('Paths', MachinePackage.Literals.PERIPHERAL__PATHS)
            }
            if (!peripheral.positions.empty) {
                items.put('SymbolicPositions', MachinePackage.Literals.PERIPHERAL__POSITIONS)
            }
            if (!items.empty) {
                val msg = '''Positions can only be used for peripheral types with axes. Remove positions or add axes to PeripheralType «peripheral.type.name»'''
                for (e : items.entrySet) {
                    error(msg, peripheral, e.value, -1, INVALID_MUTUAL_EXCLUSIONS)
                }
            }
        }
    }

    @Check
    def checkMutualExclusiveness(Peripheral peripheral) {
        val distanceDefined = !peripheral.distances.empty
        if (distanceDefined) {
            val items = newLinkedHashMap
            if (!peripheral.axisPositions.empty) {
                items.put('AxisPositions', MachinePackage.Literals.PERIPHERAL__AXIS_POSITIONS)
            }
            if (!peripheral.paths.empty) {
                items.put('Paths', MachinePackage.Literals.PERIPHERAL__PATHS)
            }
            if (!peripheral.positions.empty) {
                items.put('SymbolicPositions', MachinePackage.Literals.PERIPHERAL__POSITIONS)
            }
            if (!items.empty) {
                val request = (items.size > 1) ? "Remove Distances or all others" : "Remove one" 
                val msg = '''Distances cannot be mixed with «items.keySet.join(' and ')». «request»''' 

                for (e : items.entrySet) {
                    error(msg, peripheral, e.value, 0, INVALID_MUTUAL_EXCLUSIONS)
                }
                error(msg, peripheral, MachinePackage.Literals.PERIPHERAL__DISTANCES, 0, INVALID_MUTUAL_EXCLUSIONS)
            }
        }
    }


    @Check
    def checkNotUsedProfile(Peripheral peripheral) {
        // distance peripherals do not have path.
        if( !peripheral.distances.empty) return
        val usedProfiles = peripheral.paths.xcollect[profiles]
        for (profile : peripheral.profiles) {
            if (!usedProfiles.contains(profile)) {
                warning('Profile ' + profile.name + ' is not used.', MachinePackage.Literals.PERIPHERAL__PROFILES,
                    peripheral.profiles.indexOf(profile), PROFILE_NOT_USED)
            }
        }
    }

    @Check
    def checkConflictingPositions(SymbolicPosition symbolicPosition) {
        if (symbolicPosition !== null) {
            val axesWithoutPosition = symbolicPosition.peripheral.type.axes.toSet
            axesWithoutPosition.removeAll(symbolicPosition.axisPosition.keySet)

            val conditionIsMet = axesWithoutPosition.forall [ axis |
                symbolicPosition.peripheral.axisPositions.get(axis).forall [ axisPos |
                    axisPos.name != symbolicPosition.name
                ]
            ]

            var peripheral = symbolicPosition.peripheral
            if ((peripheral !== null) && (!conditionIsMet)) {
                error('A position with name ' + symbolicPosition.name +
                    ' already exists for one of the axes, please specify these axis positions explicitly',
                    MachinePackage.Literals.SYMBOLIC_POSITION__AXIS_POSITION, CONFLICTING_POSITIONS)
            }
        }

    }

}
