//////////////////////////////////////////////////////////////////////////////
// Copyright (c) 2010, 2025 Contributors to the Eclipse Foundation
//
// See the NOTICE file(s) distributed with this work for additional
// information regarding copyright ownership.
//
// This program and the accompanying materials are made available
// under the terms of the MIT License which is available at
// https://opensource.org/licenses/MIT
//
// SPDX-License-Identifier: MIT
//////////////////////////////////////////////////////////////////////////////

package org.eclipse.escet.cif.bdd.spec;

import static org.eclipse.escet.common.java.Lists.last;
import static org.eclipse.escet.common.java.Lists.list;
import static org.eclipse.escet.common.java.Strings.fmt;

import java.util.BitSet;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.eclipse.escet.cif.bdd.settings.CifBddSettings;
import org.eclipse.escet.cif.bdd.utils.BddUtils;
import org.eclipse.escet.cif.metamodel.cif.declarations.Event;
import org.eclipse.escet.common.java.Assert;

import com.github.javabdd.BDD;
import com.github.javabdd.BDDFactory;
import com.github.javabdd.BDDVarSet;

/** CIF/BDD specification. Represents a linearized CIF specification in a BDD representation. */
public class CifBddSpec {
    /** The settings to use. */
    public final CifBddSettings settings;

    /** The BDD factory to use. */
    public BDDFactory factory;

    /** The alphabet of the specification. */
    public Set<Event> alphabet;

    /** The controllable subset of the {@link #alphabet} of the specification. */
    public Set<Event> controllables;

    /** The temporary events created for the input variables. */
    public Set<Event> inputVarEvents;

    /**
     * The CIF/BDD variables of the specification, including location pointer variables. This is the order used whenever
     * variables are ordered.
     */
    public CifBddVariable[] variables;

    /**
     * The names of the BDD {@link #variables}, or {@code null} if not yet available. Use {@link #getBddVarNames}
     * instead of using this field directly.
     */
    private String[] bddVarNames = null;

    /** The CIF/BDD edges of the specification. */
    public List<CifBddEdge> edges;

    /**
     * The CIF/BDD edges of the specification, ordered for backward reachability computations. Contains all edges from
     * {@link #edges} at least once.
     */
    public List<CifBddEdge> orderedEdgesBackward;

    /**
     * The CIF/BDD edges of the specification, ordered for forward reachability computations. Contains all edges from
     * {@link #edges} at least once.
     */
    public List<CifBddEdge> orderedEdgesForward;

    /** Mapping from events to their CIF/BDD edges. */
    public Map<Event, List<CifBddEdge>> eventEdges;

    /**
     * Per CIF/BDD edge in {@link #orderedEdgesBackward}, its backward edge dependencies set for the workset algorithm.
     * This field is {@code null} until it is computed.
     */
    public List<BitSet> worksetDependenciesBackward;

    /**
     * Per CIF/BDD edge in {@link #orderedEdgesForward}, its forward edge dependencies set for the workset algorithm.
     * This field is {@code null} until it is computed.
     */
    public List<BitSet> worksetDependenciesForward;

    /**
     * Initialization predicates for each of the CIF/BDD variables. Predicates are obtained from the initial values as
     * specified with the declarations of the discrete variables. For CIF/BDD variables that don't represent a discrete
     * variable, the predicate is {@code null}. Is {@code null} if not yet or no longer available.
     */
    public List<BDD> initialsVars;

    /** Initialization predicates from the components. Is {@code null} if not yet or no longer available. */
    public List<BDD> initialsComps;

    /**
     * Initialization predicates from the locations of the automata, per automaton. Is {@code null} if not yet or no
     * longer available.
     */
    public List<BDD> initialsLocs;

    /**
     * Initialization predicate for the discrete variables. Conjunction of {@link #initialsVars}. Is {@code null} if not
     * yet or no longer available.
     */
    public BDD initialVars;

    /**
     * Initialization predicate for the components. Conjunction of {@link #initialsComps}. Is {@code null} if not yet or
     * no longer available.
     */
    public BDD initialComps;

    /**
     * Initialization predicate for the locations of the automata. Conjunction of {@link #initialsLocs}. Is {@code null}
     * if not yet or no longer available.
     */
    public BDD initialLocs;

    /**
     * Initialization predicate of the uncontrolled system. Conjunction of {@link #initialVars}, {@link #initialComps}
     * and {@link #initialLocs}. Is {@code null} if not yet or no longer available.
     */
    public BDD initial;

    /**
     * Combined initialization and state plant invariant predicates of the uncontrolled system. Conjunction of
     * {@link #initial} and {@link #plantInv}. Is {@code null} if not yet or no longer available.
     */
    public BDD initialPlantInv;

    /**
     * Combined initialization and state invariant predicates of the uncontrolled system. Conjunction of
     * {@link #initialPlantInv} and {@link #reqInv}. Is {@code null} if not yet or no longer available.
     */
    public BDD initialInv;

    /** Marker predicates from the components. Is {@code null} if not yet or no longer available. */
    public List<BDD> markedsComps;

    /**
     * Marker predicates from the locations of the automata, combined per automaton. Is {@code null} if not yet or no
     * longer available.
     */
    public List<BDD> markedsLocs;

    /**
     * Marker predicate for the components. Conjunction of {@link #markedsComps}. Is {@code null} if not yet or no
     * longer available.
     */
    public BDD markedComps;

    /**
     * Marker predicate for the locations of the automata. Conjunction of {@link #markedsLocs}. Is {@code null} if not
     * yet or no longer available.
     */
    public BDD markedLocs;

    /**
     * Marker predicate for the uncontrolled system. Conjunction of {@link #markedComps} and {@link #markedLocs}. Is
     * {@code null} if not yet available.
     */
    public BDD marked;

    /**
     * Combined marking and state plant invariant predicates for the uncontrolled system. Conjunction of {@link #marked}
     * and {@link #plantInv}. Is {@code null} if not yet available.
     */
    public BDD markedPlantInv;

    /**
     * Combined marking and state invariant predicates of the uncontrolled system. Conjunction of
     * {@link #markedPlantInv} and {@link #reqInv}. Is {@code null} if not yet or no longer available.
     */
    public BDD markedInv;

    /** State plant invariants (predicates) from the components. Is {@code null} if not yet or no longer available. */
    public List<BDD> plantInvsComps;

    /**
     * State plant invariants (predicates) from the locations of the automata. Unlike initialization and marker
     * predicates, these are not combined per automaton, but instead individual state plant invariants (predicates) are
     * kept. Is {@code null} if not yet or no longer available.
     */
    public List<BDD> plantInvsLocs;

    /**
     * State plant invariant (predicate) for the components. Conjunction of {@link #plantInvsComps}. Is {@code null} if
     * not yet or no longer available.
     */
    public BDD plantInvComps;

    /**
     * State plant invariant (predicate) for the locations of the automata. Conjunction of {@link #plantInvsLocs}. Is
     * {@code null} if not yet or no longer available.
     */
    public BDD plantInvLocs;

    /**
     * State plant invariant (predicate) for the system. Conjunction of {@link #plantInvComps} and
     * {@link #plantInvLocs}. Is {@code null} if not yet or no longer available.
     */
    public BDD plantInv;

    /**
     * State requirement invariants (predicates) from the components. Is {@code null} if not yet or no longer available.
     */
    public List<BDD> reqInvsComps;

    /**
     * State requirement invariants (predicates) from the locations of the automata. Unlike initialization and marker
     * predicates, these are not combined per automaton, but instead individual state requirement invariants
     * (predicates) are kept. Is {@code null} if not yet or no longer available.
     */
    public List<BDD> reqInvsLocs;

    /**
     * State requirement invariant (predicate) for the components. Conjunction of {@link #reqInvsComps}. Is {@code null}
     * if not yet or no longer available.
     */
    public BDD reqInvComps;

    /**
     * State requirement invariant (predicate) for the locations of the automata. Conjunction of {@link #reqInvsLocs}.
     * Is {@code null} if not yet or no longer available.
     */
    public BDD reqInvLocs;

    /**
     * State requirement invariant (predicate) for the system. Conjunction of {@link #reqInvComps} and
     * {@link #reqInvLocs}. Is {@code null} if not yet or no longer available.
     */
    public BDD reqInv;

    /**
     * Mapping from controllable events to their corresponding state/event exclusion requirements, derived from
     * requirement automata. Per event, the state/event requirements are combined, using conjunctions. The state/event
     * requirement predicates indicate necessary global conditions for the event to be enabled/allowed. That is, the
     * predicates can be seen as additional global guards. The predicates originate only from the requirement automata,
     * not from the state/event exclusion requirement invariants of the input specification. Is {@code null} if not yet
     * or no longer available.
     */
    public Map<Event, BDD> stateEvtExclsReqAuts;

    /**
     * Mapping from controllable events to their corresponding state/event exclusion requirements, derived from
     * state/event exclusion requirement invariants from the input specification. Per event, the state/event
     * requirements are combined, using conjunctions. The state/event requirement predicates indicate necessary global
     * conditions for the event to be enabled/allowed. That is, the predicates can be seen as additional global guards.
     * The predicates originate only from the state/event exclusion requirement invariants of the input specification,
     * not from the requirement automata. Is {@code null} if not yet or no longer available.
     */
    public Map<Event, BDD> stateEvtExclsReqInvs;

    /**
     * Mapping from events to their corresponding state/event exclusion requirements. Does not map internal events that
     * are not in the original specification, e.g. for input variables. Per event, the separate state/event requirements
     * are collected. The state/event requirement predicates indicate necessary global conditions for the event to be
     * enabled/allowed. That is, the predicates can be seen as additional global guards. The predicates originate not
     * only from the state/event exclusion requirement invariants, but also from requirement automata. Is {@code null}
     * if not yet or no longer available.
     */
    public Map<Event, List<BDD>> stateEvtExclReqLists;

    /**
     * Mapping from events to their corresponding state/event exclusion requirements. Does not map internal events that
     * are not in the original specification, e.g. for input variables. Per event, the state/event requirements are
     * combined, using conjunctions, with respect to {@link #stateEvtExclReqLists}. The state/event requirement
     * predicates indicate necessary global conditions for the event to be enabled/allowed. That is, the predicates can
     * be seen as additional global guards. The predicates originate not only from the state/event exclusion requirement
     * invariants, but also from requirement automata. Is {@code null} if not yet or no longer available.
     */
    public Map<Event, BDD> stateEvtExclReqs;

    /**
     * Mapping from events to their corresponding state/event exclusion plants. Does not map internal events that are
     * not in the original specification, e.g. for input variables. Per event, the separate state/event plant invariants
     * are collected. The state/event plant predicates indicate necessary global conditions for the event to be enabled.
     * That is, the predicates can be seen as additional global guards. Is {@code null} if not yet or no longer
     * available.
     */
    public Map<Event, List<BDD>> stateEvtExclPlantLists;

    /**
     * Mapping from events to their corresponding state/event exclusion plants. Does not map internal events that are
     * not in the original specification, e.g. for input variables. Per event, the state/event plants are combined,
     * using conjunctions, with respect to {@link #stateEvtExclPlantLists}. The state/event plant predicates indicate
     * necessary global conditions for the event to be enabled. That is, the predicates can be seen as additional global
     * guards. Is {@code null} if no yet or no longer available.
     */
    public Map<Event, BDD> stateEvtExclPlants;

    /** Reachability requirement annotation predicates from the components. Is {@code null} if not yet available. */
    public List<BDD> reachReqPreds;

    /** The BDD domains created for this specification so far. */
    private List<CifBddDomain> domains = list();

    /** The BDD variable set containing all old variables, i.e. '{x, y, z, ...}'. Is {@code null} if not available. */
    public BDDVarSet varSetOld;

    /**
     * The BDD variable set containing all new variables, i.e. '{x+, y+, z+, ...}'. Is {@code null} if not available.
     */
    public BDDVarSet varSetNew;

    /**
     * The BDD variable sets for the extra domains, each containing all the domains' variables: '{x0, y0, z0, ...}, {z1,
     * y1, z0, ...}, ...'. Is {@code null} if not available.
     */
    public List<BDDVarSet> varSetsExtra;

    /**
     * Constructor for the {@link CifBddSpec} class.
     *
     * @param settings The settings to use.
     */
    public CifBddSpec(CifBddSettings settings) {
        this.settings = settings;
    }

    /**
     * Returns the number of BDD domains created for this specification.
     *
     * @return The number of BDD domains.
     */
    public int getDomainCount() {
        return domains.size();
    }

    /**
     * Creates a BDD domain for a CIF/BDD variable of this CIF/BDD specification.
     *
     * <p>
     * The newly created domain is put after any existing already created domains, and is built up from newly allocated
     * BDD variables.
     * </p>
     *
     * @param length The length of the domain to create, in number of bits.
     * @return The newly created BDD domain.
     */
    public CifBddDomain createDomain(int length) {
        return createDomains(length)[0];
    }

    /**
     * Creates a number of BDD domains for a CIF/BDD variable of this CIF/BDD specification.
     *
     * <p>
     * The newly created domains are put after any existing already created domains, and are built up from newly
     * allocated BDD variables. The newly created domains are created and returned in the order that the lengths are
     * given. The bits of the newly created domains are interleaved. For instance, if two domains are to created of
     * lengths 2 and 4, then the first domain would get bits x1 and x2, and the second domain would get bits y1, y2, y3
     * and y4. These bits of the two domains would be interleaved to the following order: x1, y1, x2, y2, y3, y4.
     * </p>
     *
     * @param lengths The lengths of the domains to create, in number of bits. There must be at least one length.
     * @return The newly created BDD domains.
     */
    public CifBddDomain[] createDomains(int... lengths) {
        Assert.check(lengths.length > 0);

        // Allocate variable index arrays for the domains, and determine the maximum length of the new domains.
        int[][] varArrays = new int[lengths.length][]; // Per new domain, the variable indices of its bits.
        int maxLength = 0;
        for (int domainIdx = 0; domainIdx < lengths.length; domainIdx++) {
            varArrays[domainIdx] = new int[lengths[domainIdx]];
            maxLength = Math.max(maxLength, lengths[domainIdx]);
        }

        // Allocate variable indices to the interleaved domains.
        int nextVarIdx = factory.varNum();
        for (int bitIdx = 0; bitIdx < maxLength; bitIdx++) {
            for (int domainIdx = 0; domainIdx < lengths.length; domainIdx++) {
                if (bitIdx < lengths[domainIdx]) {
                    varArrays[domainIdx][bitIdx] = nextVarIdx;
                    nextVarIdx++;
                }
            }
        }

        // Increase the number of variables of the BDD factory to make the new variables available.
        factory.setVarNum(nextVarIdx);

        // Create the new BDD domains and store them both in the CIF/BDD specification and in the method result.
        CifBddDomain[] newDomains = new CifBddDomain[lengths.length];
        for (int domainIdx = 0; domainIdx < lengths.length; domainIdx++) {
            CifBddDomain domain = new CifBddDomain(factory, domains.size(), varArrays[domainIdx]);
            domains.add(domain);
            newDomains[domainIdx] = domain;
        }
        return newDomains;
    }

    /**
     * Returns the number of BDD variables per domain. That is, it returns the number of old BDD variables, the number
     * of new BDD variables, or the number of extra BDD variables per extra variable domain.
     *
     * @return The number of BDD variables per domain.
     */
    public int getBddVarCountPerDomain() {
        int nrOfDomains = 2 + settings.getBddExtraVarDomainNames().size(); // '2' are the 'old' and 'new' domains.
        return factory.varNum() / nrOfDomains;
    }

    /**
     * Free the intermediate BDDs of this CIF/BDD specification, that were collected during conversion of the CIF
     * specification to this CIF/BDD specification. This information has been aggregated and will thus still be
     * available in aggregated form.
     *
     * @param freeReqsInvsCompsAndLocs Whether to free {@link #reqInvsComps} and {@link #reqInvsLocs}.
     */
    public void freeIntermediateBDDs(boolean freeReqsInvsCompsAndLocs) {
        plantInvsComps = BddUtils.free(plantInvsComps);
        plantInvsLocs = BddUtils.free(plantInvsLocs);
        plantInvComps = BddUtils.free(plantInvComps);
        plantInvLocs = BddUtils.free(plantInvLocs);

        if (freeReqsInvsCompsAndLocs) {
            reqInvsComps = BddUtils.free(reqInvsComps);
            reqInvsLocs = BddUtils.free(reqInvsLocs);
        }
        reqInvComps = BddUtils.free(reqInvComps);
        reqInvLocs = BddUtils.free(reqInvLocs);

        initialsVars = BddUtils.free(initialsVars);
        initialsComps = BddUtils.free(initialsComps);
        initialsLocs = BddUtils.free(initialsLocs);
        initialVars = BddUtils.free(initialVars);
        initialComps = BddUtils.free(initialComps);
        initialLocs = BddUtils.free(initialLocs);
        initialInv = BddUtils.free(initialInv);

        markedsComps = BddUtils.free(markedsComps);
        markedsLocs = BddUtils.free(markedsLocs);
        markedComps = BddUtils.free(markedComps);
        markedLocs = BddUtils.free(markedLocs);
        markedPlantInv = BddUtils.free(markedPlantInv);
        markedInv = BddUtils.free(markedInv);

        stateEvtExclPlantLists = BddUtils.free(stateEvtExclPlantLists, e -> e.getValue());
        stateEvtExclReqLists = BddUtils.free(stateEvtExclReqLists, e -> e.getValue());
    }

    /** Free all BDDs of this CIF/BDD specification, as well as the {@link #factory}. */
    public void freeAllBDDs() {
        // Free intermediate BDDs.
        freeIntermediateBDDs(true);

        // Free remaining BDDs.
        initial = BddUtils.free(initial);
        initialPlantInv = BddUtils.free(initialPlantInv);
        marked = BddUtils.free(marked);

        plantInv = BddUtils.free(plantInv);
        reqInv = BddUtils.free(reqInv);

        stateEvtExclsReqAuts = BddUtils.free(stateEvtExclsReqAuts, e -> Collections.singleton(e.getValue()));
        stateEvtExclsReqInvs = BddUtils.free(stateEvtExclsReqInvs, e -> Collections.singleton(e.getValue()));
        stateEvtExclReqs = BddUtils.free(stateEvtExclReqs, e -> Collections.singleton(e.getValue()));
        stateEvtExclPlants = BddUtils.free(stateEvtExclPlants, e -> Collections.singleton(e.getValue()));

        reachReqPreds = BddUtils.free(reachReqPreds);

        varSetOld = BddUtils.free(varSetOld);
        varSetNew = BddUtils.free(varSetNew);
        varSetsExtra = BddUtils.freeVarSets(varSetsExtra);

        for (CifBddEdge edge: edges) {
            edge.freeBDDs();
        }

        for (CifBddVariable cifBddVar: variables) {
            cifBddVar.domain.free();
            cifBddVar.domainNew.free();
            for (CifBddDomain domainExtra: cifBddVar.domainsExtra) {
                domainExtra.free();
            }
        }

        // Clean up the BDD factory.
        factory.done();
    }

    @Override
    public String toString() {
        return String.join("\n", getEdgesText());
    }

    /**
     * Returns a textual representation of the {@link #edges}.
     *
     * @return The lines of text.
     */
    public List<String> getEdgesText() {
        return getEdgesText(false);
    }

    /**
     * Returns a textual representation of the {@link #edges}.
     *
     * @param includeOnlyOrigGuard Whether to include only the {@link CifBddEdge#origGuard original edge guard}, or also
     *     the {@link CifBddEdge#guard current edge guard}.
     * @return The lines of text.
     */
    public List<String> getEdgesText(boolean includeOnlyOrigGuard) {
        return edges.stream().map(e -> e.toString("Edge: ", includeOnlyOrigGuard))
                .toList();
    }

    /**
     * Return the names of the BDD variables of the CIF/BDD specification.
     *
     * @return The names of the ordered BDD {@link #variables}.
     */
    public String[] getBddVarNames() {
        // If variable names have already been determined, return them.
        if (bddVarNames != null) {
            return bddVarNames;
        }

        // Get the CIF/BDD variables, per interleaving group.
        List<List<CifBddVariable>> varsPerGroup = list();
        int curGroupIdx = 0;
        for (CifBddVariable cifBddVar: variables) {
            if (varsPerGroup.isEmpty()) {
                Assert.areEqual(0, cifBddVar.group);
                varsPerGroup.add(list(cifBddVar));
            } else if (cifBddVar.group == curGroupIdx) {
                last(varsPerGroup).add(cifBddVar);
            } else {
                varsPerGroup.add(list(cifBddVar));
                curGroupIdx++;
                Assert.areEqual(cifBddVar.group, curGroupIdx);
            }
        }

        // Get the BDD variable names.
        int numberOfVarSetsPerCifVar = 2 + this.varSetsExtra.size(); // '2' for 'old' + 'new'.
        List<String> extraVarSetNames = settings.getBddExtraVarDomainNames();
        List<String> names = list();
        for (List<CifBddVariable> varsInGroup: varsPerGroup) {
            int maxLength = varsInGroup.stream().map(v -> v.getBddVarCount()).max(Integer::compareTo).get();
            for (int bitIdx = 0; bitIdx < maxLength; bitIdx++) {
                for (CifBddVariable cifBddVar: varsInGroup) {
                    if (bitIdx < cifBddVar.getBddVarCount()) {
                        for (int varSetIdx = 0; varSetIdx < numberOfVarSetsPerCifVar; varSetIdx++) {
                            // Build the postfix string based on the varset index.
                            String postfix = switch (varSetIdx) {
                                case 0 -> "";
                                case 1 -> "+";
                                default -> fmt("{%s}", extraVarSetNames.get(varSetIdx - 2));
                            };

                            // Add the name of the current BDD variable.
                            names.add(fmt("%s#%d%s", cifBddVar.name, bitIdx, postfix));
                        }
                    }
                }
            }
        }

        // Store and return the BDD variable names.
        bddVarNames = names.toArray(new String[names.size()]);
        return bddVarNames;
    }
}
