/**
 ********************************************************************************
 * Copyright (c) 2015-2020 Eclipse APP4MC contributors.
 * 
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 * 
 * SPDX-License-Identifier: EPL-2.0
 * 
 * Contributors:
 *     Timing-Architects Embedded Systems GmbH - initial API and implementation
 ********************************************************************************
 */

package org.eclipse.app4mc.atdb._import.btf;

import java.lang.reflect.InvocationTargetException;
import java.sql.SQLException;

import org.eclipse.app4mc.atdb.ATDBBuilder;
import org.eclipse.app4mc.atdb.ATDBConnection;
import org.eclipse.app4mc.atdb._import.btf.model.BTFEntityType;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.SubMonitor;
import org.eclipse.jface.operation.IRunnableWithProgress;


public class ImportTransformation implements IRunnableWithProgress {
	
	private final String btfFile;
	private final String atdbFile;
	private final boolean persistTraceEvents;
	
	public ImportTransformation(final String source, final String target, final boolean persistTraceEvents) {
		if (source.endsWith(".btf") && target.endsWith(".atdb")) {
			this.btfFile = source;
			this.atdbFile = target;
		} else {
			this.btfFile = "";
			this.atdbFile = "";
		}
		this.persistTraceEvents = persistTraceEvents;
	}

	@Override
	public void run(final IProgressMonitor progressMonitor) throws InvocationTargetException, InterruptedException {
		if (!this.btfFile.isEmpty() && !this.atdbFile.isEmpty()) {
			final SubMonitor subMon = SubMonitor.convert(progressMonitor, "Converting BTF trace to ATDB...", 100);
			try (final ATDBConnection con = new ATDBConnection(this.atdbFile, true)) {
				final SubMonitor createATDBMonitor = subMon.split(1);
				createATDBMonitor.beginTask("Creating empty ATDB...", 1);
				final ATDBBuilder atdbBuilder = new ATDBBuilder(con).createBasicDBStructure().createBasicViews()
						.createOptionalAndTemporaryTables(BTFEntityType.literals, this.persistTraceEvents);
				if (this.persistTraceEvents)
					atdbBuilder.createOptionalViews(BTFEntityType.literals);
				createATDBMonitor.done();
				
				final SubMonitor btfImportMonitor = subMon.split(69);
				final IRunnableWithProgress btfImporter = new BTFImporter(con, this.btfFile);
				btfImporter.run(btfImportMonitor);
				btfImportMonitor.done();
				
				final SubMonitor metricCalcMonitor = subMon.split(30);
				atdbBuilder.autoPopulateEntityFilteredTraceEventTables(BTFEntityType.literals);
				final IRunnableWithProgress metricCalc = new ATDBMetricCalculator(con);
				metricCalc.run(metricCalcMonitor);
				metricCalcMonitor.done();
			} catch (SQLException e) {
				throw new InvocationTargetException(e);
			} finally {
				progressMonitor.done();
			}
		}
	}
}
