#
# Copyright (C) 2007, 2008, 2014 Red Hat, Inc.
# Authors:
# Thomas Woerner <twoerner@redhat.com>
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.
#

import os.path

from fw_config import OLD_CONFIG, CONFIG, IP4TABLES_CFG, IP6TABLES_CFG
from fw_parser import parseSysconfigArgs
import fw_compat
import fw_iptables
import shutil

def read_sysconfig_args():
    filename = None
    if os.path.exists(CONFIG) and os.path.isfile(CONFIG):
        filename = CONFIG
    elif os.path.exists(OLD_CONFIG) and os.path.isfile(OLD_CONFIG):
        filename = OLD_CONFIG
    try:
        fd = open(filename, 'r')
    except:
        return None
    argv = [ ]
    for line in fd.xreadlines():
        if not line:
            break
        line = line.strip()
        if len(line) < 1 or line[0] == '#':
            continue
        argv.append(line)
    fd.close()
    return (argv, filename)

def parse_sysconfig_args(args, merge_config=None, compat=False, filename=None):
    config = parseSysconfigArgs(args, options=merge_config, compat=compat,
                                source=filename)
    if not config:
        return None
    config.filename = filename
    if config.filename == OLD_CONFIG:
        fw_compat.convertToServices(config)
        config.converted = True
    return config

def read_sysconfig_config(merge_config=None, compat=False):
    args = read_sysconfig_args() # returns: (args, filename) or None
    if not args:
        return merge_config
    return parse_sysconfig_args(args[0], merge_config, compat, args[1])

def write_sysconfig_config(filename, conf):
    if os.path.exists(filename):
        try:
            shutil.copy2(filename, "%s.old" % filename)
        except:
            return False

    try:
        fd = open(filename, 'w')
    except:
        return False

    os.chmod(filename, 0600)
    fd.write("# Configuration file for system-config-firewall\n")
    fd.write("\n")

    if conf.enabled == True:
        fd.write("--enabled\n")
    elif conf.enabled == False:
        fd.write("--disabled\n")

    if conf.trust:
        for dev in conf.trust:
            fd.write("--trust=%s\n" % dev)
    if conf.masq:
        for dev in conf.masq:
            fd.write("--masq=%s\n" % dev)

    if conf.ports and len(conf.ports) > 0:
        for (ports, proto) in conf.ports:
            fd.write("--port=%s:%s\n" % ('-'.join(map(str, ports)), proto))

    if conf.custom_rules and len(conf.custom_rules) > 0:
            for custom in conf.custom_rules:
                fd.write("--custom-rules=%s\n" % ":".join(custom))

    if conf.services:
        for service in conf.services:
            fd.write("--service=%s\n" % service)

    if conf.block_icmp:
        for icmp in conf.block_icmp:
            fd.write("--block-icmp=%s\n" % icmp)

    if conf.forward_port:
        for fwd in conf.forward_port:
            if len(fwd["port"]) == 1:
                port = "%s" % fwd["port"][0]
            else:
                port = "%s-%s" % (fwd["port"][0], fwd["port"][1])
            line = "if=%s:port=%s:proto=%s" % (fwd["if"], port, fwd["proto"])
            if fwd.has_key("toport"):
                if len(fwd["toport"]) == 1:
                    line += ":toport=%s" % fwd["toport"][0]
                else:
                    line += ":toport=%s-%s" % (fwd["toport"][0],
                                               fwd["toport"][1])
            if fwd.has_key("toaddr"):
                line += ":toaddr=%s" % fwd["toaddr"]
            fd.write("--forward-port=%s\n" % line)

    fd.close()
    return True

def read_service_settings():
    # load IPv4 configuration
    ipv4_failed = False
    ipv4_conf = fw_iptables.ip4tablesConfig(IP4TABLES_CFG)
    try:
        ipv4_conf.read()
    except:
        # no or empty config
        ipv4_failed = True

    # load IPv6 configuration
    ipv6_failed = False
    ipv6_conf = fw_iptables.ip6tablesConfig(IP6TABLES_CFG)
    try:
        ipv6_conf.read()
    except:
        # no or empty config
        ipv6_failed = True

    ipv4_settings = { }
    ipv6_settings = { }
    for setting in fw_iptables.setting_list:
        if not ipv4_failed:
            ipv4_settings[setting.key] = \
                (ipv4_conf.get(ipv4_conf.prefix+setting.key) == "yes")
        else:
            ipv4_settings[setting.key] = setting.iptables
        if not ipv6_failed:
            ipv6_settings[setting.key] = \
                (ipv6_conf.get(ipv6_conf.prefix+setting.key) == "yes")
        else:
            ipv6_settings[setting.key] = setting.ip6tables

    key = "MODULES"
    if not ipv4_failed:
        ipv4_settings[key] = ipv4_conf.get(ipv4_conf.prefix+key)
    else:
        ipv4_settings[key] = [ ]
    if not ipv6_failed:
        ipv6_settings[key] = ipv6_conf.get(ipv6_conf.prefix+key)
    else:
        ipv6_settings[key] = [ ]

    return { "iptables": ipv4_settings, "ip6tables": ipv6_settings }

def write_service_settings(settings):
    # load IPv4 configuration
    ipv4_failed = False
    ipv4_conf = fw_iptables.ip4tablesConfig(IP4TABLES_CFG)
    try:
        ipv4_conf.read()
    except:
        # ok: no or empty config
        pass

    # load IPv6 configuration
    ipv6_failed = False
    ipv6_conf = fw_iptables.ip6tablesConfig(IP6TABLES_CFG)
    try:
        ipv6_conf.read()
    except:
        # ok: no or empty config
        pass

    yes_no = { True: "yes", False: "no" }

    ipv4_settings = settings["iptables"]
    ipv6_settings = settings["ip6tables"]
    for key in ipv4_settings.keys():
        if key != "MODULES":
            ipv4_conf.set(ipv4_conf.prefix+key, yes_no[ipv4_settings[key]])
        else:
            ipv4_conf.set(ipv4_conf.prefix+key, ipv4_settings[key])
    for key in ipv6_settings.keys():
        if key != "MODULES":
            ipv6_conf.set(ipv6_conf.prefix+key, yes_no[ipv6_settings[key]])
        else:
            ipv6_conf.set(ipv6_conf.prefix+key, ipv6_settings[key])
    try:
        ipv4_conf.write()
        ipv6_conf.write()
    except:
        return False

    return True
