---
stage: none
group: unassigned
info: Any user with at least the Maintainer role can merge updates to this content. For details, see https://docs.gitlab.com/ee/development/development_processes.html#development-guidelines-review.
---

# SCSS style guide

## Utility Classes

In order to reduce the generation of more CSS as our site grows, prefer the use
of utility classes over adding new CSS. In complex cases, CSS can be addressed
by adding component classes.

### Where are CSS utility classes defined?

Utility classes are generated by [Tailwind CSS](https://tailwindcss.com/). Use [Tailwind CSS autocomplete](#tailwind-css-autocomplete) or the [official Tailwind CSS documentation](#official-tailwind-css-documentation) to see available CSS utility classes.

There are also legacy CSS utility classes defined in
`config/helpers/tailwind/css_in_js.js`. These CSS utility classes do not comply with
Tailwind CSS naming conventions and will be
[iteratively migrated](https://gitlab.com/groups/gitlab-org/-/epics/13521) to the
Tailwind CSS equivalent. Please do not add new instances of these CSS utility
classes, instead use the Tailwind CSS equivalent.

Classes in [`utilities.scss`](https://gitlab.com/gitlab-org/gitlab/-/blob/master/app/assets/stylesheets/utilities.scss)
and [`common.scss`](https://gitlab.com/gitlab-org/gitlab/-/blob/master/app/assets/stylesheets/framework/common.scss)
are being deprecated. Classes in [`common.scss`](https://gitlab.com/gitlab-org/gitlab/-/blob/master/app/assets/stylesheets/framework/common.scss)
that use non-design-system values should be avoided. Use classes with conforming values instead.

Avoid [Bootstrap's Utility Classes](https://getbootstrap.com/docs/4.3/utilities/).

NOTE:
While migrating [Bootstrap's Utility Classes](https://getbootstrap.com/docs/4.3/utilities/)
to the [GitLab UI](https://gitlab.com/gitlab-org/gitlab-ui/-/blob/main/doc/css.md#utilities)
utility classes, note both the classes for margin and padding differ. The size scale used at
GitLab differs from the scale used in the Bootstrap library. For a Bootstrap padding or margin
utility, you may need to double the size of the applied utility to achieve the same visual
result (such as `ml-1` becoming `gl-ml-2`).

### Tailwind CSS

We are in the process of migrating our CSS utility class setup to [Tailwind CSS](https://tailwindcss.com/).
See the [Tailwind CSS design document](https://handbook.gitlab.com/handbook/engineering/architecture/design-documents/tailwindcss/) for motivation, proposal,
and implementation details.

#### Tailwind CSS basics

Below are some Tailwind CSS basics and information about how it has been
configured to use the [Pajamas design system](https://design.gitlab.com/). For a
more in-depth guide see the [official Tailwind CSS documentation](https://tailwindcss.com/docs/utility-first).

##### Prefix

We have configured Tailwind CSS to use a
[prefix](https://tailwindcss.com/docs/configuration#prefix) so all utility classes are prefixed with `gl-`.
When using responsive utilities or state modifiers the prefix goes after the colon.

**Examples:** `gl-mt-5`, `lg:gl-mt-5`.

##### Responsive CSS utility classes

[Responsive CSS utility classes](https://tailwindcss.com/docs/responsive-design) are prefixed with the breakpoint name, followed by the `:` character.
The available breakpoints are configured in [tailwind.defaults.js#L44](https://gitlab.com/gitlab-org/gitlab-ui/-/blob/6612eaee37cdb4dd0258468c9f415be28c1053f0/tailwind.defaults.js#L44)

**Example:** `lg:gl-mt-5`

##### Hover, focus, and other state modifiers

[State modifiers](https://tailwindcss.com/docs/hover-focus-and-other-states)
can be used to conditionally apply any Tailwind CSS class. Prefix the CSS utility class
with the name of the modifier, followed by the `:` character.

**Example:** `hover:gl-underline`

##### `!important` modifier

You can use the [important modifier](https://tailwindcss.com/docs/configuration#important-modifier) by adding `!` to the beginning of the CSS utility class. When using in conjunction with responsive utility classes or state modifiers the `!` goes after the `:` character.

**Examples:** `!gl-mt-5`, `lg:!gl-mt-5`, `hover:!gl-underline`

##### Spacing and sizing CSS utility classes

Spacing and sizing CSS utility classes (e.g. `margin`, `padding`, `width`, `height`) use our spacing scale defined in
Color CSS utility classes (e.g. `color` and `background-color`) use colors defined in
[src/tokens/build/tailwind/tokens.cjs](https://gitlab.com/gitlab-org/gitlab-ui/-/blob/24a08b50da6bd3d34fb3f8d24f84436d90d165f6/src/tokens/build/tailwind/tokens.cjs).
They will use the naming conventions documented in the [official Tailwind CSS documentation](https://tailwindcss.com/docs/padding)
but the color names will not match. When using the [Tailwind CSS autocomplete](#tailwind-css-autocomplete)
our configured colors will be shown.

**Example:** `gl-mt-5` will be `margin-top: 1rem;`

##### Color CSS utility classes

Color CSS utility classes (e.g. `color` and `background-color`) use colors defined in
[src/tokens/build/tailwind/tokens.cjs](https://gitlab.com/gitlab-org/gitlab-ui/-/blob/24a08b50da6bd3d34fb3f8d24f84436d90d165f6/src/tokens/build/tailwind/tokens.cjs).
They will use the naming conventions documented in the [official Tailwind CSS documentation](https://tailwindcss.com/docs/text-color)
but the color names will not match. When using the [Tailwind CSS autocomplete](#tailwind-css-autocomplete)
our configured colors will be shown.

**Example:** `gl-text-red-500` will be `color: var(--red-500, #dd2b0e);`

#### Building the Tailwind CSS bundle

When using Vite or Webpack with the GitLab Development Kit, Tailwind CSS watches for file changes to
build detected utilities on the fly.

To build a fresh Tailwind CSS bundle, run `yarn tailwindcss:build`. This is the script that gets
called internally when building production assets with `bundle exec rake gitlab:assets:compile`.

However the bundle gets built, the output is saved to `app/assets/builds/tailwind.css`.

#### Tailwind CSS autocomplete

Tailwind CSS autocomplete will list all available classes in your code editor.
Keep in mind it will also list legacy CSS utilities. Unfortunately we don't have
a way to mark the legacy CSS utility classes in the autocomplete so try to cross
reference with the [official Tailwind CSS documentation](#official-tailwind-css-documentation) if you are unsure.

##### VS Code

Install the [Tailwind CSS IntelliSense](https://marketplace.visualstudio.com/items?itemName=bradlc.vscode-tailwindcss)
extension. For best results and HAML and custom `*-class` prop support these are the recommended settings:

```json
{
  "tailwindCSS.experimental.classRegex": [
    ["class: [\"|']+([^\"|']*)[\"|']+", "([a-zA-Z0-9\\-:!/]+)"],
    ["(\\.[\\w\\-.]+)[\\n\\=\\{\\s]", "([\\w\\-]+)"],
    ["[a-z]+-class(?:es)?=\"([^'\"]*)\""]
  ],
  "tailwindCSS.emmetCompletions": true,
  "editor.inlineSuggest.enabled": true,
  "editor.quickSuggestions": {
    "strings": true
  },
  "css.validate": false
}
```

##### RubyMine

Tailwind CSS autocomplete is [enabled by default](https://www.jetbrains.com/help/ruby/tailwind-css.html).
For full HAML and custom `*-class` prop support these are the recommended updates to the default settings:

```json
{
  "includeLanguages": {
    "haml": "html"
  },
  "emmetCompletions": true,
  "experimental": {
    "classRegex": [
      ["class: [\"|']+([^\"|']*)[\"|']+", "([a-zA-Z0-9\\-:!/]+)"],
      ["(\\.[\\w\\-.]+)[\\n\\=\\{\\s]", "([\\w\\-]+)"],
      ["[a-z]+-class(?:es)?=\"([^'\"]*)\""]
    ]
  }
}
```

#### Official Tailwind CSS documentation

GitLab defines its own Tailwind CSS config in [tailwind.defaults.js](https://gitlab.com/gitlab-org/gitlab-ui/-/blob/6612eaee37cdb4dd0258468c9f415be28c1053f0/tailwind.defaults.js)
to match the Pajamas design system and to prefix CSS utility classes with `gl-`.
This means that in the [official Tailwind CSS documentation](https://tailwindcss.com/docs/padding)
the spacing, sizing, and color CSS utility classes may not match. Also, the `gl-`
prefix will not be shown. Here is our
[spacing scale](https://gitlab.com/gitlab-org/gitlab-ui/-/blob/6612eaee37cdb4dd0258468c9f415be28c1053f0/tailwind.defaults.js#L4)
and [colors](https://gitlab.com/gitlab-org/gitlab-ui/-/blob/24a08b50da6bd3d34fb3f8d24f84436d90d165f6/src/tokens/build/tailwind/tokens.cjs).
In the future we plan to utilize [Tailwind config viewer](https://github.com/rogden/tailwind-config-viewer)
to have a Tailwind CSS documentation site specific to GitLab.

### Where should you put new utility classes?

Utility classes are generated by [Tailwind CSS](https://tailwindcss.com/) which
supports most CSS features. If there is something that is not available we should
update `tailwind.defaults.js` in GitLab UI.

### When should you create component classes?

We recommend a "utility-first" approach.

1. Start with utility classes.
1. If composing utility classes into a component class removes code duplication and encapsulates a clear responsibility, do it.

This encourages an organic growth of component classes and prevents the creation of
one-off non-reusable classes. Also, the kind of classes that emerge from "utility-first"
tend to be design-centered (for example, `.button`, `.alert`, `.card`) rather than
domain-centered (for example, `.security-report-widget`, `.commit-header-icon`).

Inspiration:

- <https://tailwindcss.com/docs/utility-first>
- <https://tailwindcss.com/docs/extracting-components>

### Utility mixins

We are currently in the process of [migrating to Tailwind](#tailwind-css). The migration
removes utility mixins so please do not add any new usages of utility mixins.
Instead, you can use the [`@apply` directive](https://tailwindcss.com/docs/reusing-styles#extracting-classes-with-apply)
to add Tailwind styles to a CSS selector. `@apply` should be used for any CSS properties
that are dependent on our design system (e.g. `margin`, `padding`). For CSS properties
that are unit-less (e.g `display: flex`) it is okay to use CSS properties directly.

```scss
// Bad
.my-class {
  @include gl-mt-3;
}

// Bad
.my-class {
  margin-top: 0.5rem;
}

// Okay
.my-class {
  display: flex;
}

// Good
.my-class {
  @apply gl-mt-5 gl-flex;
}
```

## Naming

Filenames should use `snake_case`.

CSS classes should use the `lowercase-hyphenated` format rather than
`snake_case` or `camelCase`.

```scss
// Bad
.class_name {
  color: #fff;
}

// Bad
.className {
  color: #fff;
}

// Good
.class-name {
  color: #fff;
}
```

Avoid making compound class names with SCSS `&` features. It makes
searching for usages harder, and provides limited benefit.

```scss
// Bad
.class {
  &-name {
    color: orange;
  }
}

// Good
.class-name {
  color: #fff;
}
```

Class names should be used instead of tag name selectors.
Using tag name selectors is discouraged because they can affect
unintended elements in the hierarchy.

```scss
// Bad
ul {
  color: #fff;
}

// Good
.class-name {
  color: #fff;
}

// Best
// prefer an existing utility class over adding existing styles
```

Class names are also preferable to IDs. Rules that use IDs
are not-reusable, as there can only be one affected element on
the page.

```scss
// Bad
#my-element {
  padding: 0;
}

// Good
.my-element {
  padding: 0;
}
```

## Nesting

Avoid unnecessary nesting. The extra specificity of a wrapper component
makes things harder to override.

```scss
// Bad
.component-container {
  .component-header {
    /* ... */
  }

  .component-body {
    /* ... */
  }
}

// Good
.component-container {
  /* ... */
}

.component-header {
  /* ... */
}

.component-body {
  /* ... */
}
```

## Selectors with a `js-` Prefix

Do not use any selector prefixed with `js-` for styling purposes. These
selectors are intended for use only with JavaScript to allow for removal or
renaming without breaking styling.

## Selectors with Util CSS Classes

Do not use utility CSS classes as selectors in your stylesheets. These classes
are likely to change, requiring updates to the selectors and making the
implementation harder to maintain. Instead, use another existing CSS class or
add a new custom CSS class for styling elements. This approach improves
maintainability and reduces the risk of bugs.

```scss
// ❌ Bad
.gl-mb-5 {
  /* ... */
}

// ✅ Good
.component-header {
  /* ... */
}
```

## Selectors with ARIA attributes

Do not use any attribute selector with ARIA for styling purposes. These
attributes and roles are intended for supporting assistive technology.
The structure of the components annotated with ARIA might change
and so its styling. We need to be able to move these roles and attributes
to different elements, without breaking styling.

```scss
// Bad
&[aria-expanded=false] &-header {
  border-bottom: 0;
}

// Good
&.is-collapsed &-header {
  border-bottom: 0;
}
```

## Using `extend` at-rule

Usage of the `extend` at-rule is prohibited due to
[memory leaks](https://gitlab.com/gitlab-org/gitlab/-/issues/323021) and
[the rule doesn't work as it should](https://sass-lang.com/documentation/breaking-changes/extend-compound/).

## Linting

We use [stylelint](https://stylelint.io) to check for style guide conformity. It uses the
ruleset in `.stylelintrc` and rules from
[our SCSS configuration](https://gitlab.com/gitlab-org/frontend/gitlab-stylelint-config).
`.stylelintrc` is located in the home directory of the project.

To check if any warnings are produced by your changes, run `yarn lint:stylelint`
in the GitLab directory. Stylelint also runs in GitLab CI/CD to
catch any warnings.

If the Rake task is throwing warnings you don't understand, SCSS Lint's
documentation includes [a full list of their rules](https://stylelint.io/user-guide/rules/).
