## ----include = FALSE----------------------------------------------------------
knitr::opts_chunk$set(
  message = FALSE, warning = FALSE, fig.width = 8
)

## ----setup, include = FALSE---------------------------------------------------
library(edfinr)
library(dplyr)
library(tidyr)
library(stringr)
library(ggplot2)

## ----eval = FALSE-------------------------------------------------------------
# library(tidyverse)
# library(edfinr)

## ----example-nominal----------------------------------------------------------
# Get nominal (unadjusted) data - this is the default
nominal_data <- get_finance_data(yr = "2015:2022", geo = "KY")

# View the nominal revenue for a specific district
nominal_data |>
  filter(dist_name == "Jefferson County") |>
  select(year, dist_name, rev_total, rev_total_pp)

## ----example-adjusted---------------------------------------------------------
# Get data adjusted to 2022 dollars
real_2022_data <- get_finance_data(yr = "2015:2022", geo = "KY", cpi_adj = 2022)

# View the same district with inflation-adjusted values
real_2022_data |>
  filter(dist_name == "Jefferson County") |>
  select(year, dist_name, rev_total, rev_total_pp)

## ----cpi-index----------------------------------------------------------------
# Examine the CPI index values
cpi_values <- get_finance_data(yr = "all", geo = "KY") |>
  select(year, cpi_sy12) |>
  distinct() |>
  arrange(year)

print(cpi_values)

# Calculate cumulative inflation since 2012
cpi_values |>
  mutate(
    inflation_since_2012 = (cpi_sy12 - 1) * 100,
    inflation_label = paste0(round(inflation_since_2012, 1), "%")
  )

## ----real-spending-analysis---------------------------------------------------
# get multiyear data in nominal dollars
ky_nominal <- get_finance_data(yr = "all", geo = "KY", cpi_adj = "none") |> 
  mutate(type = "Nominal dollars")

# get multi-year data adjusted to 2022 dollars
ky_real <- get_finance_data(yr = "all", geo = "KY", cpi_adj = "2022") |> 
  mutate(type = "Real 2022 dollars")

# join data
ky_data <- bind_rows(ky_nominal, ky_real)

# calculate statewide per-pupil revenue trends for real dollars
rev_trends <- ky_data |>
  group_by(type, year) |>
  summarize(
    rev_local = sum(rev_local, na.rm = TRUE),
    rev_state = sum(rev_state, na.rm = TRUE),
    rev_fed = sum(rev_fed, na.rm = TRUE),
    enroll = sum(enroll, na.rm = TRUE)
  ) |> 
  mutate(
    rev_local_pp = rev_local / enroll,
    rev_state_pp = rev_state / enroll,
    rev_fed_pp = rev_fed / enroll
  ) |> 
  select(type, year, rev_local_pp:rev_fed_pp) |>
  pivot_longer(
    cols = rev_local_pp:rev_fed_pp,
    names_to = "var", values_to = "val") |> 
  mutate(
    var = stringr::str_remove_all(var, "rev_"),
    var = stringr::str_remove_all(var, "_pp"),
    var = stringr::str_to_title(var),
    var = str_replace_all(var, "Fed", "Federal")
  )    

# plot trends
ggplot(rev_trends) +
  geom_line(
    aes(x = year, y = val, color = var, group = var)
    ) +
  facet_wrap(~type) +
  scale_x_discrete(breaks = c("2012", "2014", "2016", "2018", "2020", "2022")) +
  scale_y_continuous(labels = scales::dollar) +
  labs(
    title = "Comparing Nominal and Real Per-Pupil Revenue in Kentucky",
    subtitle = "Statewide average per-pupil revenue by source, 2012-2022",
    x = "Year",
    y = "Per-Pupil Revenue",
    color = "Revenue Source"
  ) +
  theme_minimal()

## ----base-year-comparison-----------------------------------------------------
# select ky district to assess
district_sample <- "Jefferson County"

# get data with nominal dollars and cpi-adjusted for different base years
nominal <- get_finance_data(yr = "2012:2022", geo = "KY") |>
  filter(dist_name == district_sample) |>
  select(year, rev_total_pp) |>
  mutate(type = "Nominal")

adjusted_2012 <- get_finance_data(yr = "2012:2022", geo = "KY", cpi_adj = 2012) |>
  filter(dist_name == district_sample) |>
  select(year, rev_total_pp) |>
  mutate(type = "2012 Dollars")

adjusted_2022 <- get_finance_data(yr = "2012:2022", geo = "KY", cpi_adj = 2022) |>
  filter(dist_name == district_sample) |>
  select(year, rev_total_pp) |>
  mutate(type = "2022 Dollars")

# join and plot data
bind_rows(nominal, adjusted_2012, adjusted_2022) |>
  ggplot(aes(x = year, y = rev_total_pp, color = type, group = type)) +
  geom_line(size = 1.2) +
  scale_y_continuous(labels = scales::dollar) +
  labs(
    title = paste("Per-Pupil Revenue:", district_sample),
    x = "Year",
    y = "Revenue per Pupil",
    color = "CPI Adjustment"
  ) +
  theme_minimal()

