## ----include = FALSE----------------------------------------------------------
knitr::opts_chunk$set(
  collapse = TRUE,
  comment = "#>",
  out.width = "100%"
)

## ----load-data, echo=FALSE----------------------------------------------------
# This loads pre-saved data to avoid network calls during CRAN checks
# See vignette creation script in data-raw/vignettes/save_equity_data.R
load("data_equity.RData")

## ----setup, message=FALSE, warning=FALSE--------------------------------------
library(rb3)
library(dplyr)
library(lubridate)
library(ggplot2)

## ----download-yearly, eval=FALSE----------------------------------------------
# # Download data for multiple years (example: 2018 to 2024)
# fetch_marketdata("b3-cotahist-yearly", year = 2018:2024)

## ----download-daily, eval=FALSE-----------------------------------------------
# # Download data for specific dates
# fetch_marketdata("b3-cotahist-daily", refdate = as.Date("2024-01-02"))

## ----access-data, eval=FALSE--------------------------------------------------
# # Access yearly dataset
# ch <- cotahist_get("yearly")
# 
# # Access daily dataset
# ch_daily <- cotahist_get("daily")

## ----equity-filter, eval=FALSE------------------------------------------------
# # Filter for stocks from 2024
# eq <- ch |>
#   filter(year(refdate) == 2024) |>
#   cotahist_filter_equity()

## ----etf-filter, eval=FALSE---------------------------------------------------
# # Filter for ETFs from 2024
# etfs <- ch |>
#   filter(year(refdate) == 2024) |>
#   cotahist_filter_etf()

## ----fii-filter, eval=FALSE---------------------------------------------------
# # Filter for REITs from 2024
# fii <- ch |>
#   filter(year(refdate) == 2024) |>
#   cotahist_filter_fii()

## ----bdr-filter, eval=FALSE---------------------------------------------------
# # Filter for BDRs
# bdrs <- cotahist_get() |>
#   cotahist_filter_bdr()

## ----equity-volume-fake, eval=FALSE-------------------------------------------
# # Get top 10 stocks by volume
# symbols_eq <- eq |>
#   group_by(symbol) |>
#   summarise(volume = sum(volume)) |>
#   arrange(desc(volume)) |>
#   head(10) |>
#   pull(symbol, as_vector = TRUE)
# 
# # Calculate monthly volumes for top stocks
# eq_monthly <- eq |>
#   filter(symbol %in% symbols_eq) |>
#   mutate(month = floor_date(refdate, "month")) |>
#   group_by(month, symbol) |>
#   summarise(volume = sum(volume)) |>
#   collect()

## ----equity-volume-plot, echo=FALSE, fig.width=7, fig.height=5.5, fig.cap="Monthly Trading Volume for Top 10 Brazilian Stocks"----
# Using pre-loaded data to create the plot without fetching data
ggplot(eq_monthly, aes(x = month, y = volume, group = symbol, colour = symbol)) +
  geom_line() +
  scale_y_continuous(labels = scales::label_number()) +
  labs(
    title = "Monthly Trading Volume for Top 10 Brazilian Stocks",
    x = "Month",
    y = "Volume (BRL)",
    caption = "Data source: B3 via rb3 package"
  )

## ----etf-analysis-fake, eval=FALSE--------------------------------------------
# # Calculate total ETF volume
# total_volume <- etfs |>
#   summarise(volume = sum(volume)) |>
#   pull(volume, as_vector = TRUE)
# 
# # Get volume share for top ETFs
# etf_shares <- etfs |>
#   group_by(symbol) |>
#   summarise(volume = sum(volume)) |>
#   collect() |>
#   mutate(volume_ratio = volume / total_volume) |>
#   slice_max(volume_ratio, n = 10) |>
#   mutate(volume_ratio_acc = cumsum(volume_ratio))

## ----etf-share-plot, echo=FALSE, fig.width=7, fig.height=5.5, fig.cap="Volume Share of Top 10 ETFs"----
# Using pre-loaded data to create the plot
fmt <- scales::label_percent(accuracy = 0.1)

ggplot(etf_shares, aes(
  x = reorder(symbol, -volume_ratio), y = volume_ratio,
  label = fmt(volume_ratio)
)) +
  geom_bar(stat = "identity", fill = "royalblue") +
  geom_text(nudge_y = 0.01) +
  scale_y_continuous(labels = scales::label_percent()) +
  labs(
    x = NULL, y = NULL,
    title = "Volume Share of Top 10 ETFs",
    subtitle = "Percentage of Total ETF Trading Volume",
    caption = "Data source: B3 via rb3 package"
  )

## ----distribution-analysis-fake, eval=FALSE-----------------------------------
# # Select a specific stock
# symbol_ <- "ITUB4"
# 
# # Extract data for the selected stock
# stock_data <- equities |>
#   filter(symbol == symbol_) |>
#   arrange(refdate) |>
#   collect()

## ----distribution-id-plot, echo=FALSE, fig.width=7, fig.height=5, fig.cap="Distribution ID Changes for ITUB4"----
# Using pre-loaded data to create the plot
ggplot(stock_data, aes(x = refdate, y = distribution_id)) +
  geom_line() +
  labs(
    title = "Distribution ID Changes for ITUB4",
    x = "Date",
    y = "Distribution ID",
    caption = "Data source: B3 via rb3 package"
  )

## ----price-by-distribution-plot, echo=FALSE, fig.width=7, fig.height=5, fig.cap="Price History for ITUB4 by Distribution ID"----
# Using pre-loaded data to create the plot
ggplot(stock_data, aes(x = refdate, y = close, colour = factor(distribution_id))) +
  geom_line() +
  labs(
    title = "Price History for ITUB4 by Distribution ID",
    x = "Date",
    y = "Closing Price (BRL)",
    color = "Distribution ID",
    caption = "Data source: B3 via rb3 package"
  )

## ----bdr-analysis-fake, eval=FALSE--------------------------------------------
# # Find the most common BDRs in the dataset
# top_bdrs <- bdrs |>
#   group_by(symbol) |>
#   count(sort = TRUE) |>
#   head(15) |>
#   collect()
# 
# # Analyze trading volume for a specific BDR
# bdr_data <- bdrs |>
#   filter(symbol == "AAPL34") |>
#   arrange(refdate) |>
#   collect()

## ----bdr-volume-plot, echo=FALSE, fig.width=7, fig.height=5, fig.cap="Trading Volume for AAPL34 (Apple BDR)"----
# Using pre-loaded data to create the plot
ggplot(bdr_data, aes(x = refdate, y = trade_quantity, colour = factor(distribution_id))) +
  geom_line() +
  labs(
    title = "Trading Volume for AAPL34 (Apple BDR)",
    x = "Date",
    y = "Trade Quantity",
    color = "Distribution ID",
    caption = "Data source: B3 via rb3 package"
  )

