from fastapi import Depends, HTTPException, APIRouter
from sqlalchemy.orm import Session

from ..database import session_provider
from ..models.models import Report, ApplicationStats, CpuInfo, ThreadTime, HealthCheck, ThreadConcurrency, ProcessUtilisation
from ..models.threading_analysis.threading_analysis import ThreadingAnalysisInDB, ThreadingAnalysisCreate

router = APIRouter()


@router.get("/threading_analysis/{report_id}", tags=['Threading Analysis'], response_model=ThreadingAnalysisInDB)
def get_threading_analysis(report_id: int, db: Session = Depends(session_provider.get_session)):
    report_db: Report = db.query(Report).filter(Report.id == report_id).first()
    if not report_db:
        raise HTTPException(status_code=404, detail="Report not found")
    return ThreadingAnalysisInDB.model_validate(report_db)


@router.post("/threading_analysis", tags=['Threading Analysis'], response_model=ThreadingAnalysisInDB)
def create_threading_analysis(threading_analysis: ThreadingAnalysisCreate, db: Session = Depends(session_provider.get_session)):
    db_report: Report = db.query(Report).filter(Report.id == threading_analysis.report_id).first()
    if not db_report:
        raise HTTPException(status_code=404, detail="Report not found")

    # clean up threading_analysis data for the report if exists
    for model in (ThreadTime, ThreadConcurrency, ProcessUtilisation, ApplicationStats, CpuInfo, HealthCheck):
        db.query(model).filter_by(report_id=threading_analysis.report_id).delete()

    db_application_stats = ApplicationStats(report_id=db_report.id, **threading_analysis.application_stats.model_dump())
    db.add(db_application_stats)

    db_cpu_info = CpuInfo(report_id=db_report.id, **threading_analysis.cpu_info.model_dump())
    db.add(db_cpu_info)

    for data in threading_analysis.thread_time:
        db_data = ThreadTime(report_id=db_report.id, **data.model_dump())
        db.add(db_data)

    for data in threading_analysis.health_check:
        db_data = HealthCheck(report_id=db_report.id, **data.model_dump())
        db.add(db_data)

    for data in threading_analysis.thread_concurrency:
        db_data = ThreadConcurrency(report_id=db_report.id, **data.model_dump())
        db.add(db_data)

    for data in threading_analysis.process_utilisation:
        db_data = ProcessUtilisation(report_id=db_report.id, **data.model_dump())
        db.add(db_data)

    db.commit()
    db.refresh(db_report)
    return ThreadingAnalysisInDB.model_validate(db_report)
