from typing import Optional, List

from pydantic import BaseModel
from pydantic_partial import PartialModelMixin
from sqlalchemy import Column, Integer, ForeignKey, String, Double, Boolean, BigInteger
from sqlalchemy.orm import relationship

from .region_app_dxg_krnl_profile_ranges import RegionAppDxgKernelProfileRangesInDB, \
    RegionAppDxgKernelProfileRangesCreate, RegionAppDxgKernelProfileRanges
from .region_etw_events import RegionEtwEventsInDB, RegionEtwEventsCreate, RegionEtwEvents
from .region_issues import RegionIssuesInDB, RegionIssuesCreate, RegionIssues, RegionIssuesShortInDB
from .region_metrics import RegionMetricsInDB, RegionMetricsCreate, RegionMetrics
from .region_thread import RegionThread, RegionThreadShortInDB
from ..base import Base


class RegionSubmitThreads(Base):
    __tablename__ = 'region_submit_threads'
    id = Column(Integer, primary_key=True, index=True)
    thread_id = Column(BigInteger)

    region_id = Column(Integer, ForeignKey('region.id'))
    region = relationship('Region', back_populates='submit_threads')


class RegionSubmitThreadsInDB(BaseModel):
    thread_id: int

    class Config:
        from_attributes = True


class RegionSubmitThreadsCreate(BaseModel):
    id: Optional[int] = None
    thread_id: int

    class Config:
        from_attributes = True


class RegionPresentThreads(Base):
    __tablename__ = 'region_present_threads'
    id = Column(Integer, primary_key=True, index=True)
    thread_id = Column(BigInteger)

    region_id = Column(Integer, ForeignKey('region.id'))
    region = relationship('Region', back_populates='present_threads')


class RegionPresentThreadsInDB(BaseModel):
    thread_id: int

    class Config:
        from_attributes = True


class RegionPresentThreadsCreate(BaseModel):
    id: Optional[int] = None
    thread_id: int

    class Config:
        from_attributes = True


class RegionCustomFields(Base):
    __tablename__ = 'region_custom_fields'
    id = Column(Integer, primary_key=True, index=True)
    name = Column(String)
    value = Column(String)

    region_id = Column(Integer, ForeignKey('region.id'))
    region = relationship('Region', back_populates='custom_fields')


class RegionCustomFieldsInDB(BaseModel):
    name: str
    value: str

    class Config:
        from_attributes = True


class RegionCustomFieldsCreate(BaseModel):
    id: Optional[int] = None
    name: str
    value: str

    class Config:
        from_attributes = True


class Region(Base):
    __tablename__ = 'region'
    id = Column(Integer, primary_key=True, index=True)
    region_index = Column(Integer, nullable=True)
    region_index_global = Column(Integer, nullable=True)
    region_type = Column(String, nullable=False)
    region_name = Column(String, nullable=False)
    duration_ms = Column(Double, nullable=True)
    start_time_s = Column(Double, nullable=True)

    summed_metric_value = Column(Integer, nullable=True)
    threads_count = Column(Integer, nullable=True)
    is_cpu_bound = Column(Boolean, nullable=True)
    is_median = Column(Boolean, nullable=False, default=False)

    hotspot_analysis_id = Column(Integer, ForeignKey('hotspot_analysis.id'))
    hotspot_analysis = relationship('HotspotAnalysis', back_populates='regions')

    submit_threads = relationship(RegionSubmitThreads, back_populates='region')
    present_threads = relationship(RegionPresentThreads, back_populates='region')
    region_app_dxg_kernel_profile_ranges = relationship(RegionAppDxgKernelProfileRanges, back_populates='region')
    region_etw_events = relationship(RegionEtwEvents, back_populates='region')
    region_issues = relationship(RegionIssues, back_populates='region')
    custom_fields = relationship(RegionCustomFields, back_populates='region')
    region_metrics = relationship(RegionMetrics, back_populates='region')

    region_threads = relationship(RegionThread, back_populates='region')


class RegionShortBaseInDB(BaseModel):
    id: int
    region_name: str
    region_type: str
    is_median: bool
    region_index: Optional[int]
    region_index_global: Optional[int]
    duration_ms: Optional[float]
    start_time_s: Optional[float]
    summed_metric_value: Optional[int]
    threads_count: Optional[int]
    is_cpu_bound: Optional[bool]
    hotspot_analysis_id: int

    class Config:
        from_attributes = True


class RegionShortInDB(RegionShortBaseInDB):
    region_issues: List[RegionIssuesShortInDB]

    class Config:
        from_attributes = True


class RegionInDB(RegionShortBaseInDB):
    submit_threads: List[RegionSubmitThreadsInDB]
    region_issues: List[RegionIssuesInDB]
    present_threads: List[RegionPresentThreadsInDB]
    region_app_dxg_kernel_profile_ranges: List[RegionAppDxgKernelProfileRangesInDB]
    region_etw_events: List[RegionEtwEventsInDB]
    region_threads: List[RegionThreadShortInDB]
    custom_fields: List[RegionCustomFieldsInDB]
    region_metrics: List[RegionMetricsInDB]

    class Config:
        from_attributes = True


class RegionBaseCreate(BaseModel):
    hotspot_analysis_id: int
    region_name: str
    region_type: str
    is_median: bool
    region_index: Optional[int] = None
    region_index_global: Optional[int] = None
    duration_ms: Optional[float] = None
    start_time_s: Optional[float] = None
    summed_metric_value: Optional[int] = None
    threads_count: Optional[int] = None
    is_cpu_bound: Optional[bool] = None

    class Config:
        from_attributes = True


class RegionCreate(PartialModelMixin, RegionBaseCreate):
    submit_threads: List[RegionSubmitThreadsCreate] = []
    present_threads: List[RegionPresentThreadsCreate] = []
    region_app_dxg_kernel_profile_ranges: List[RegionAppDxgKernelProfileRangesCreate] = []
    region_etw_events: List[RegionEtwEventsCreate] = []
    region_issues: List[RegionIssuesCreate] = []
    custom_fields: List[RegionCustomFieldsCreate] = []
    region_metrics: List[RegionMetricsCreate] = []

    class Config:
        from_attributes = True
