from fastapi import FastAPI
from starlette.middleware.cors import CORSMiddleware
from starlette.responses import JSONResponse

from .database import session_provider
from .routes.routes import register_routers


def create_app() -> FastAPI:
    if session_provider.SessionLocal is None:
        session_provider.set_up_database()

    fast_api_app = FastAPI(
        docs_url="/swagger",
        title="nsys-cpu-stats-server",
        description="Web server for nsys-cpu-stats based on FastAPI with Swagger and Sqlalchemy",
        version="1.0.0"
    )

    # Register the API router
    register_routers(fast_api_app)

    # Dependency
    @fast_api_app.middleware("http")
    async def db_session_middleware(request, call_next):
        try:
            request.state.db = session_provider.get_session()
            response = await call_next(request)
        finally:
            request.state.db.close()
        return response

    @fast_api_app.exception_handler(Exception)
    def validation_exception_handler(request, err):
        base_error_message = f"Failed to execute: {request.method}: {request.url}"
        return JSONResponse(status_code=400, content={"message": f"{base_error_message}. Detail: {err}"})

    origins = [
        "http://localhost:4200",
        "http://127.0.0.1:4200",
    ]

    fast_api_app.add_middleware(
        CORSMiddleware,
        allow_origins=origins,
        allow_credentials=True,
        allow_methods=["*"],
        allow_headers=["*"]
    )

    return fast_api_app


app = create_app()
