!
!     Copyright (c) 2017-2018, NVIDIA CORPORATION.  All rights reserved.
!
! NVIDIA CORPORATION and its licensors retain all intellectual property
! and proprietary rights in and to this software, related documentation
! and any modifications thereto.  Any use, reproduction, disclosure or
! distribution of this software and related documentation without an express
! license agreement from NVIDIA CORPORATION is strictly prohibited.
!
!          THIS CODE AND INFORMATION ARE PROVIDED "AS IS" WITHOUT
!   WARRANTY OF ANY KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING BUT
!   NOT LIMITED TO THE IMPLIED WARRANTIES OF MERCHANTABILITY AND/OR
!   FITNESS FOR A PARTICULAR PURPOSE.
!

module cusparse
  use iso_c_binding
  use nvf_acc_common, only : c_devptr

  integer, parameter :: CUSPARSE_VER_MAJOR = 12
  integer, parameter :: CUSPARSE_VER_MINOR = 1
  integer, parameter :: CUSPARSE_VER_PATCH = 2
  integer, parameter :: CUSPARSE_VER_BUILD = 129
  integer, parameter :: CUSPARSE_VERSION   = (CUSPARSE_VER_MAJOR * 1000 + &
                          CUSPARSE_VER_MINOR *  100 + CUSPARSE_VER_PATCH)

  enum, bind(C) ! cusparseStatus_t
     enumerator :: CUSPARSE_STATUS_SUCCESS=0
     enumerator :: CUSPARSE_STATUS_NOT_INITIALIZED=1
     enumerator :: CUSPARSE_STATUS_ALLOC_FAILED=2
     enumerator :: CUSPARSE_STATUS_INVALID_VALUE=3
     enumerator :: CUSPARSE_STATUS_ARCH_MISMATCH=4
     enumerator :: CUSPARSE_STATUS_MAPPING_ERROR=5
     enumerator :: CUSPARSE_STATUS_EXECUTION_FAILED=6
     enumerator :: CUSPARSE_STATUS_INTERNAL_ERROR=7
     enumerator :: CUSPARSE_STATUS_MATRIX_TYPE_NOT_SUPPORTED=8
     enumerator :: CUSPARSE_STATUS_ZERO_PIVOT=9
     enumerator :: CUSPARSE_STATUS_NOT_SUPPORTED=10
     enumerator :: CUSPARSE_STATUS_INSUFFICIENT_RESOURCES=11
  end enum

  enum, bind(c) ! cusparsePointerMode_t
     enumerator :: CUSPARSE_POINTER_MODE_HOST = 0  
     enumerator :: CUSPARSE_POINTER_MODE_DEVICE = 1        
  end enum

  enum, bind(c) ! cusparseAction_t
     enumerator :: CUSPARSE_ACTION_SYMBOLIC = 0
     enumerator :: CUSPARSE_ACTION_NUMERIC = 1        
  end enum

  enum, bind(C) ! cusparseMatrixType_t
     enumerator :: CUSPARSE_MATRIX_TYPE_GENERAL = 0 
     enumerator :: CUSPARSE_MATRIX_TYPE_SYMMETRIC = 1     
     enumerator :: CUSPARSE_MATRIX_TYPE_HERMITIAN = 2 
     enumerator :: CUSPARSE_MATRIX_TYPE_TRIANGULAR = 3 
  end enum

  enum, bind(C) ! cusparseFillMode_t
     enumerator :: CUSPARSE_FILL_MODE_LOWER = 0
     enumerator :: CUSPARSE_FILL_MODE_UPPER = 1
  end enum

  enum, bind(C) ! cusparseDiagType_t
     enumerator :: CUSPARSE_DIAG_TYPE_NON_UNIT = 0 
     enumerator :: CUSPARSE_DIAG_TYPE_UNIT = 1
  end enum

  enum, bind(C) ! cusparseIndexBase_t
     enumerator :: CUSPARSE_INDEX_BASE_ZERO = 0
     enumerator :: CUSPARSE_INDEX_BASE_ONE = 1
  end enum

  enum, bind(C) ! cusparseOperation_t
     enumerator :: CUSPARSE_OPERATION_NON_TRANSPOSE = 0
     enumerator :: CUSPARSE_OPERATION_TRANSPOSE = 1
     enumerator :: CUSPARSE_OPERATION_CONJUGATE_TRANSPOSE = 2  
  end enum

  enum, bind(C) ! cusparseDirection_t
     enumerator :: CUSPARSE_DIRECTION_ROW = 0
     enumerator :: CUSPARSE_DIRECTION_COLUMN = 1  
  end enum

  enum, bind(C) ! cusparseHybPartition_t
     enumerator :: CUSPARSE_HYB_PARTITION_AUTO = 0 
     enumerator :: CUSPARSE_HYB_PARTITION_USER = 1 
     enumerator :: CUSPARSE_HYB_PARTITION_MAX = 2  
  end enum

  enum, bind(C) ! cusparseSolvePolicy_t
     enumerator :: CUSPARSE_SOLVE_POLICY_NO_LEVEL = 0
     enumerator :: CUSPARSE_SOLVE_POLICY_USE_LEVEL = 1
  end enum

  enum, bind(C) ! cusparseSideMode_t
     enumerator :: CUSPARSE_SIDE_LEFT  = 0
     enumerator :: CUSPARSE_SIDE_RIGHT = 1
  end enum

  enum, bind(C) ! cusparseColorAlg_t
     enumerator :: CUSPARSE_COLOR_ALG0 = 0
     enumerator :: CUSPARSE_COLOR_ALG1 = 1
  end enum

  enum, bind(C) ! cusparseAlgMode_t;
     enumerator :: CUSPARSE_ALG0           = 0
     enumerator :: CUSPARSE_ALG1           = 1
     enumerator :: CUSPARSE_ALG_NAIVE      = 0
     enumerator :: CUSPARSE_ALG_MERGE_PATH = 0
  end enum

  enum, bind(C) ! cusparseCsr2CscAlg_t;
     enumerator :: CUSPARSE_CSR2CSC_ALG_DEFAULT = 1
     enumerator :: CUSPARSE_CSR2CSC_ALG1 = 1
     enumerator :: CUSPARSE_CSR2CSC_ALG2 = 2
  end enum

  enum, bind(C) ! cusparseFormat_t;
     enumerator :: CUSPARSE_FORMAT_CSR     = 1
     enumerator :: CUSPARSE_FORMAT_CSC     = 2
     enumerator :: CUSPARSE_FORMAT_COO     = 3
     enumerator :: CUSPARSE_FORMAT_COO_AOS = 4
     enumerator :: CUSPARSE_FORMAT_BLOCKED_ELL = 5
     enumerator :: CUSPARSE_FORMAT_BSR     = 6
     enumerator :: CUSPARSE_FORMAT_SLICED_ELLPACK = 7
  end enum

  enum, bind(C) ! cusparseOrder_t;
     enumerator :: CUSPARSE_ORDER_COL = 1
     enumerator :: CUSPARSE_ORDER_ROW = 2
  end enum

  enum, bind(C) ! cusparseSpMVAlg_t;
     enumerator :: CUSPARSE_MV_ALG_DEFAULT = 0
     enumerator :: CUSPARSE_COOMV_ALG      = 1
     enumerator :: CUSPARSE_CSRMV_ALG1     = 2
     enumerator :: CUSPARSE_CSRMV_ALG2     = 3
     enumerator :: CUSPARSE_SPMV_ALG_DEFAULT = 0
     enumerator :: CUSPARSE_SPMV_CSR_ALG1    = 2
     enumerator :: CUSPARSE_SPMV_CSR_ALG2    = 3
     enumerator :: CUSPARSE_SPMV_COO_ALG1    = 1
     enumerator :: CUSPARSE_SPMV_COO_ALG2    = 4
     enumerator :: CUSPARSE_SPMV_SELL_ALG1   = 5
  end enum

  enum, bind(C) ! cusparseSpMMAlg_t;
     enumerator :: CUSPARSE_MM_ALG_DEFAULT = 0
     enumerator :: CUSPARSE_COOMM_ALG1 = 1
     enumerator :: CUSPARSE_COOMM_ALG2 = 2
     enumerator :: CUSPARSE_COOMM_ALG3 = 3
     enumerator :: CUSPARSE_CSRMM_ALG1 = 4
     enumerator :: CUSPARSE_SPMM_ALG_DEFAULT = 0
     enumerator :: CUSPARSE_SPMM_COO_ALG1    = 1
     enumerator :: CUSPARSE_SPMM_COO_ALG2    = 2
     enumerator :: CUSPARSE_SPMM_COO_ALG3    = 3
     enumerator :: CUSPARSE_SPMM_COO_ALG4    = 5
     enumerator :: CUSPARSE_SPMM_CSR_ALG1    = 4
     enumerator :: CUSPARSE_SPMM_CSR_ALG2    = 6
     enumerator :: CUSPARSE_SPMM_CSR_ALG3    = 12
     enumerator :: CUSPARSE_SPMM_BLOCKED_ELL_ALG1 = 13
  end enum

  enum, bind(C) ! cusparseIndexType_t;
     enumerator :: CUSPARSE_INDEX_16U = 1
     enumerator :: CUSPARSE_INDEX_32I = 2
     enumerator :: CUSPARSE_INDEX_64I = 3
  end enum

  enum, bind(C) ! cusparseSpMatAttribute_t;
     enumerator :: CUSPARSE_SPMAT_FILL_MODE = 0
     enumerator :: CUSPARSE_SPMAT_DIAG_TYPE = 1
  end enum

  enum, bind(C) ! cusparseSparseToDenseAlg_t;
     enumerator :: CUSPARSE_SPARSETODENSE_ALG_DEFAULT = 0
     enumerator :: CUSPARSE_DENSETOSPARSE_ALG_DEFAULT = 0
  end enum

  enum, bind(C) ! cusparseSpSVAlg_t;
     enumerator :: CUSPARSE_SPSV_ALG_DEFAULT = 0
  end enum

  enum, bind(C) ! cusparseSpSVUpdate_t;
     enumerator :: CUSPARSE_SPSV_UPDATE_GENERAL  = 0
     enumerator :: CUSPARSE_SPSV_UPDATE_DIAGONAL = 1
  end enum
  enum, bind(C) ! cusparseSpSMAlg_t;
     enumerator :: CUSPARSE_SPSM_ALG_DEFAULT = 0
  end enum

  enum, bind(C) ! cusparseSpMMOpAlg_t;
     enumerator :: CUSPARSE_SPMM_OP_ALG_DEFAULT = 0
  end enum

  enum, bind(C) ! cusparseSpGEMMAlg_t;
     enumerator :: CUSPARSE_SPGEMM_DEFAULT = 0
     enumerator :: CUSPARSE_SPGEMM_CSR_ALG_DETERMINISTIC = 1
     enumerator :: CUSPARSE_SPGEMM_CSR_ALG_DETERMINITIC  = 1
     enumerator :: CUSPARSE_SPGEMM_CSR_ALG_NONDETERMINISTIC = 2
     enumerator :: CUSPARSE_SPGEMM_CSR_ALG_NONDETERMINITIC  = 2
     enumerator :: CUSPARSE_SPGEMM_ALG1 = 3
     enumerator :: CUSPARSE_SPGEMM_ALG2 = 4
     enumerator :: CUSPARSE_SPGEMM_ALG3 = 5
  end enum

  enum, bind(C) ! cusparseSDDMMAlg_t;
     enumerator :: CUSPARSE_SDDMM_ALG_DEFAULT = 0
  end enum

  ! ---------
  ! types
  ! ---------

  type cusparseHandle
     type(c_ptr) :: handle
  end type cusparseHandle

  type cusparseMatDescr
     type(c_ptr) :: descr
  end type cusparseMatDescr
     
  type cusparseSolveAnalysisInfo
     type(c_ptr) :: info
  end type cusparseSolveAnalysisInfo

  type cusparseHybMat
     type(c_ptr) :: mat
  end type cusparseHybMat

  type cusparseCsrsv2Info
     type(c_ptr) :: info
  end type cusparseCsrsv2Info

  type cusparseCsrsm2Info
     type(c_ptr) :: info
  end type cusparseCsrsm2Info

  type cusparseCsric02Info
     type(c_ptr) :: info
  end type cusparseCsric02Info

  type cusparseCsrilu02Info
     type(c_ptr) :: info
  end type cusparseCsrilu02Info

  type cusparseBsrsv2Info
     type(c_ptr) :: info
  end type cusparseBsrsv2Info

  type cusparseBsric02Info
     type(c_ptr) :: info
  end type cusparseBsric02Info

  type cusparseBsrilu02Info
     type(c_ptr) :: info
  end type cusparseBsrilu02Info

  type cusparseBsrsm2Info
     type(c_ptr) :: info
  end type cusparseBsrsm2Info

  type cusparseCsrgemm2Info
     type(c_ptr) :: info
  end type cusparseCsrgemm2Info

  type cusparseColorInfo
     type(c_ptr) :: info
  end type cusparseColorInfo

  type cusparseCsru2csrInfo
     type(c_ptr) :: info
  end type cusparseCsru2csrInfo

  type cusparseSpVecDescr
     type(c_ptr) :: descr
  end type cusparseSpVecDescr

  type cusparseDnVecDescr
     type(c_ptr) :: descr
  end type cusparseDnVecDescr

  type cusparseSpMatDescr
     type(c_ptr) :: descr
  end type cusparseSpMatDescr

  type cusparseDnMatDescr
     type(c_ptr) :: descr
  end type cusparseDnMatDescr

  type cusparseSpSVDescr
     type(c_ptr) :: descr
  end type cusparseSpSVDescr

  type cusparseSpSMDescr
     type(c_ptr) :: descr
  end type cusparseSpSMDescr

  type cusparseSpGEMMDescr
     type(c_ptr) :: descr
  end type cusparseSpGEMMDescr

  ! --------------------
  ! 5.5 (and later) helper functions
  ! --------------------

! cusparseCreate
  interface
     integer(c_int) function cusparseCreate(handle) bind(C,name='cusparseCreate')
       import cusparseHandle
       type(cusparseHandle) :: handle
     end function cusparseCreate
  end interface

! cusparseDestroy
  interface
     integer(c_int) function cusparseDestroy(handle) bind(C,name='cusparseDestroy')
       import cusparseHandle
       type(cusparseHandle), value :: handle
     end function cusparseDestroy
  end interface

! cusparseGetErrorName
  interface
     function cusparseGetErrorName(ierr) result(res)
     integer :: ierr
     character*128 :: res
     end function cusparseGetErrorName
  end interface

! cusparseGetErrorString
  interface
     function cusparseGetErrorString(ierr) result(res)
     integer :: ierr
     character*128 :: res
     end function cusparseGetErrorString
  end interface

! cusparseGetProperty
  interface
    integer(c_int) function cusparseGetProperty(ptype, pval) bind(C,name='cusparseGetProperty')
       use iso_c_binding
       integer, value :: ptype
       integer :: pval
     end function cusparseGetProperty
  end interface

! cusparseGetVersion
  interface
     integer(c_int) function cusparseGetVersion(handle, version) bind(C,name='cusparseGetVersion')
       use iso_c_binding
       import cusparseHandle
       type(cusparseHandle), value :: handle
       integer(c_int) :: version
     end function cusparseGetVersion
  end interface

! cusparseSetStream
  interface
     integer(c_int) function cusparseSetStream(handle, stream) bind(C,name='cusparseSetStream')
       use cudafor
       import cusparseHandle
       type(cusparseHandle), value :: handle
       !dir$ ignore_tkr (k) stream
       integer(cuda_stream_kind), value :: stream
     end function cusparseSetStream
  end interface

! cusparseGetStream
  interface
     integer(c_int) function cusparseGetStream(handle, stream) bind(C,name='cusparseGetStream')
       use cudafor
       import cusparseHandle
       type(cusparseHandle), value :: handle
       integer(cuda_stream_kind) :: stream
     end function cusparseGetStream
  end interface

! cusparseGetPointerMode
  interface
     integer(c_int) function cusparseGetPointerMode(handle, mode) bind(C,name='cusparseGetPointerMode')
       use iso_c_binding
       import cusparseHandle
       type(cusparseHandle), value :: handle
       integer(c_int) :: mode
     end function cusparseGetPointerMode
  end interface

! cusparseSetPointerMode
  interface
     integer(c_int) function cusparseSetPointerMode(handle, mode) bind(C,name='cusparseSetPointerMode')
       use iso_c_binding
       import cusparseHandle
       type(cusparseHandle), value :: handle
       integer(c_int), value :: mode
     end function cusparseSetPointerMode
  end interface

! cusparseCreateMatDescr
  interface
     integer(c_int) function cusparseCreateMatDescr(descrA) bind(C,name='cusparseCreateMatDescr')
       import cusparseMatDescr
       type(cusparseMatDescr) :: descrA
     end function cusparseCreateMatDescr
  end interface

! cusparseDestroyMatDescr
  interface
     integer(c_int) function cusparseDestroyMatDescr(descrA) bind(C,name='cusparseDestroyMatDescr')
       import cusparseMatDescr
       type(cusparseMatDescr), value :: descrA
     end function cusparseDestroyMatDescr
  end interface

! cusparseSetMatType
  interface
     integer(c_int) function cusparseSetMatType(descrA, type) bind(C,name='cusparseSetMatType')
       use iso_c_binding
       import cusparseMatDescr
       type(cusparseMatDescr), value :: descrA
       integer(c_int), value :: type
     end function cusparseSetMatType
  end interface

! cusparseGetMatType
  interface
     integer(c_int) function cusparseGetMatType(descrA) bind(C,name='cusparseGetMatType')
       import cusparseMatDescr
       type(cusparseMatDescr), value :: descrA
     end function cusparseGetMatType
  end interface

! cusparseSetMatFillMode
  interface
     integer(c_int) function cusparseSetMatFillMode(descrA, mode) bind(C,name='cusparseSetMatFillMode')
       use iso_c_binding
       import cusparseMatDescr
       type(cusparseMatDescr), value :: descrA
       integer(c_int), value :: mode
     end function cusparseSetMatFillMode
  end interface

! cusparseGetMatFillMode
  interface
     integer(c_int) function cusparseGetMatFillMode(descrA) bind(C,name='cusparseGetMatFillMode')
       import cusparseMatDescr
       type(cusparseMatDescr), value :: descrA
     end function cusparseGetMatFillMode
  end interface

! cusparseSetMatDiagType
  interface
     integer(c_int) function cusparseSetMatDiagType(descrA, type) bind(C,name='cusparseSetMatDiagType')
       use iso_c_binding
       import cusparseMatDescr
       type(cusparseMatDescr), value :: descrA
       integer(c_int), value :: type
     end function cusparseSetMatDiagType
  end interface

! cusparseGetMatDiagType
  interface
     integer(c_int) function cusparseGetMatDiagType(descrA) bind(C,name='cusparseGetMatDiagType')
       import cusparseMatDescr
       type(cusparseMatDescr), value :: descrA
     end function cusparseGetMatDiagType
  end interface

! cusparseSetMatIndexBase
  interface
     integer(c_int) function cusparseSetMatIndexBase(descrA, base) bind(C,name='cusparseSetMatIndexBase')
       use iso_c_binding
       import cusparseMatDescr
       type(cusparseMatDescr), value :: descrA
       integer(c_int), value :: base
     end function cusparseSetMatIndexBase
  end interface

! cusparseGetMatIndexBase
  interface
     integer(c_int) function cusparseGetMatIndexBase(descrA) bind(C,name='cusparseGetMatIndexBase')
       import cusparseMatDescr
       type(cusparseMatDescr), value :: descrA
     end function cusparseGetMatIndexBase
  end interface

! cusparseCreateSolveAnalysisInfo
  interface
     integer(c_int) function cusparseCreateSolveAnalysisInfo(info) bind(C,name='cusparseCreateSolveAnalysisInfo')
       import cusparseSolveAnalysisInfo
       type(cusparseSolveAnalysisinfo) :: info
     end function cusparseCreateSolveAnalysisInfo
  end interface

! cusparseDestroySolveAnalysisInfo
  interface
     integer(c_int) function cusparseDestroySolveAnalysisInfo(info) bind(C,name='cusparseDestroySolveAnalysisInfo')
       import cusparseSolveAnalysisInfo
       type(cusparseSolveAnalysisinfo), value :: info
     end function cusparseDestroySolveAnalysisInfo
  end interface

! cusparseGetLevelInfo
  interface
     integer(c_int) function cusparseGetLevelInfo(handle, info, nlevels, levelPtr, levelInd) &
          bind(C,name='cusparseGetLevelInfo')
       use iso_c_binding
       import cusparseHandle
       import cusparseSolveAnalysisInfo
       type(cusparseHandle), value :: handle
       type(cusparseSolveAnalysisinfo), value :: info
       integer(c_int) :: nlevels
       type(c_ptr) :: levelPtr
       type(c_ptr) :: levelInd
     end function cusparseGetLevelInfo
  end interface

! cusparseCreateHybMat
  interface
     integer(c_int) function cusparseCreateHybMat(hybA) bind(C,name='cusparseCreateHybMat')
       import cusparseHybMat
       type(cusparseHybMat) :: hybA
     end function cusparseCreateHybMat
  end interface

! cusparseDestroyHybMat
  interface
     integer(c_int) function cusparseDestroyHybMat(hybA) bind(C,name='cusparseCreateHybMat')
       import cusparseHybMat
       type(cusparseHybMat), value :: hybA
     end function cusparseDestroyHybMat
  end interface

  ! --------------------
  ! 6.0 Helper Functions
  ! --------------------

! cusparseCreateCsrsv2Info
  interface
     integer(c_int) function cusparseCreateCsrsv2Info(info) bind(C,name='cusparseCreateCsrsv2Info')
       import cusparseCsrsv2Info
       type(cusparseCsrsv2Info) :: info
     end function cusparseCreateCsrsv2Info
  end interface

! cusparseDestroyCsrsv2Info
  interface
     integer(c_int) function cusparseDestroyCsrsv2Info(info) bind(C,name='cusparseDestroyCsrsv2Info')
       use iso_c_binding
       import cusparseCsrsv2Info
       type(cusparseCsrsv2Info), value :: info
     end function cusparseDestroyCsrsv2Info
  end interface

! cusparseCreateCsric02Info
  interface
     integer(c_int) function cusparseCreateCsric02Info(info) bind(C,name='cusparseCreateCsric02Info')
       import cusparseCsric02Info
       type(cusparseCsric02Info) :: info
     end function cusparseCreateCsric02Info
  end interface

! cusparseDestroyCsric02Info
  interface
     integer(c_int) function cusparseDestroyCsric02Info(info) bind(C,name='cusparseDestroyCsric02Info')
       use iso_c_binding
       import cusparseCsric02Info
       type(cusparseCsric02Info), value :: info
     end function cusparseDestroyCsric02Info
  end interface
  
! cusparseCreateCsrilu02Info
  interface
     integer(c_int) function cusparseCreateCsrilu02Info(info) bind(C,name='cusparseCreateCsrilu02Info')
       import cusparseCsrilu02Info
       type(cusparseCsrilu02Info) :: info
     end function cusparseCreateCsrilu02Info
  end interface

! cusparseDestroyCsrilu02Info
  interface
     integer(c_int) function cusparseDestroyCsrilu02Info(info) bind(C,name='cusparseDestroyCsrilu02Info')
       use iso_c_binding
       import cusparseCsrilu02Info
       type(cusparseCsrilu02Info), value :: info
     end function cusparseDestroyCsrilu02Info
  end interface

! cusparseCreateBsrsv2Info
  interface
     integer(c_int) function cusparseCreateBsrsv2Info(info) bind(C,name='cusparseCreateBsrsv2Info')
       import cusparseBsrsv2Info
       type(cusparseBsrsv2Info) :: info
     end function cusparseCreateBsrsv2Info
  end interface

! cusparseDestroyBsrsv2Info
  interface
     integer(c_int) function cusparseDestroyBsrsv2Info(info) bind(C,name='cusparseDestroyBsrsv2Info')
       use iso_c_binding
       import cusparseBsrsv2Info
       type(cusparseBsrsv2Info), value :: info
     end function cusparseDestroyBsrsv2Info
  end interface

! cusparseCreateBsric02Info
  interface
     integer(c_int) function cusparseCreateBsric02Info(info) bind(C,name='cusparseCreateBsric02Info')
       import cusparseBsric02Info
       type(cusparseBsric02Info) :: info
     end function cusparseCreateBsric02Info
  end interface

! cusparseDestroyBsric02Info
  interface
     integer(c_int) function cusparseDestroyBsric02Info(info) bind(C,name='cusparseDestroyBsric02Info')
       use iso_c_binding
       import cusparseBsric02Info
       type(cusparseBsric02Info), value :: info
     end function cusparseDestroyBsric02Info
  end interface
  
! cusparseCreateBsrilu02Info
  interface
     integer(c_int) function cusparseCreateBsrilu02Info(info) bind(C,name='cusparseCreateBsrilu02Info')
       import cusparseBsrilu02Info
       type(cusparseBsrilu02Info) :: info
     end function cusparseCreateBsrilu02Info
  end interface

! cusparseDestroyBsrilu02Info
  interface
     integer(c_int) function cusparseDestroyBsrilu02Info(info) bind(C,name='cusparseDestroyBsrilu02Info')
       use iso_c_binding
       import cusparseBsrilu02Info
       type(cusparseBsrilu02Info), value :: info
     end function cusparseDestroyBsrilu02Info
  end interface

  ! --------------------
  ! 6.5 helper functions
  ! --------------------

! cusparseCreateBsrsm2Info
  interface
     integer(c_int) function cusparseCreateBsrsm2Info(info) bind(C,name='cusparseCreateBsrsm2Info')
       import cusparseBsrsm2Info
       type(cusparseBsrsm2Info) :: info
     end function cusparseCreateBsrsm2Info
  end interface

! cusparseDestroyBsrsm2Info
  interface
     integer(c_int) function cusparseDestroyBsrsm2Info(info) bind(C,name='cusparseDestroyBsrsm2Info')
       use iso_c_binding
       import cusparseBsrsm2Info
       type(cusparseBsrsm2Info), value :: info
     end function cusparseDestroyBsrsm2Info
  end interface

  ! --------------------
  ! 7.0 helper functions
  ! --------------------

! cusparseCreateCsrgemm2Info
  interface
     integer(c_int) function cusparseCreateCsrgemm2Info(info) bind(C,name='cusparseCreateCsrgemm2Info')
       import cusparseCsrgemm2Info
       type(cusparseCsrgemm2Info) :: info
     end function cusparseCreateCsrgemm2Info
  end interface

! cusparseDestroyCsrgemm2Info
  interface
     integer(c_int) function cusparseDestroyCsrgemm2Info(info) bind(C,name='cusparseDestroyCsrgemm2Info')
       use iso_c_binding
       import cusparseCsrgemm2Info
       type(cusparseCsrgemm2Info), value :: info
     end function cusparseDestroyCsrgemm2Info
  end interface

! cusparseCreateColorInfo
  interface
     integer(c_int) function cusparseCreateColorInfo(info) bind(C,name='cusparseCreateColorInfo')
       import cusparseColorInfo
       type(cusparseColorInfo) :: info
     end function cusparseCreateColorInfo
  end interface

! cusparseDestroyColorInfo
  interface
     integer(c_int) function cusparseDestroyColorInfo(info) bind(C,name='cusparseDestroyColorInfo')
       import cusparseColorInfo
       type(cusparseColorInfo), value :: info
     end function cusparseDestroyColorInfo
  end interface

! cusparseCreateCsru2csrInfo
  interface
     integer(c_int) function cusparseCreateCsru2csrInfo(info) bind(C,name='cusparseCreateCsru2csrInfo')
       import cusparseCsru2csrInfo
       type(cusparseCsru2csrInfo) :: info
     end function cusparseCreateCsru2csrInfo
  end interface

! cusparseDestroyCsru2csrInfo
  interface
     integer(c_int) function cusparseDestroyCsru2csrInfo(info) bind(C,name='cusparseDestroyCsru2csrInfo')
        import cusparseCsru2csrInfo
        type(cusparseCsru2csrInfo), value :: info
      end function cusparseDestroyCsru2csrInfo
   end interface

  ! ---------------------
  ! 5.5 Level 1 functions
  ! ---------------------

! cusparseSaxpyi
  interface cusparseSaxpyi
    integer function cusparseSaxpyi_setHPM(handle, nnz, alpha, xVal, xInd, y, idxBase)
    import cusparseHandle
    type(cusparseHandle) :: handle
    integer :: nnz
    real :: alpha
    real, device :: xVal(*)
    integer, device :: xInd(*)
    real, device :: y(*)
    integer :: idxBase
    end function cusparseSaxpyi_setHPM

    integer function cusparseSaxpyi_setDPM(handle, nnz, alpha, xVal, xInd, y, idxBase)
    import cusparseHandle
    type(cusparseHandle) :: handle
    integer :: nnz
    real, device :: alpha
    real, device :: xVal(*)
    integer, device :: xInd(*)
    real, device :: y(*)
    integer :: idxBase
    end function cusparseSaxpyi_setDPM
  end interface

#ifdef KEEP_V2_INTERFACES
  interface cusparseSaxpyi_v2
     integer(c_int) function cusparseSaxpyi_v2(handle, nnz, alpha, xVal, xInd, y, idxBase) &
          bind(C,name='cusparseSaxpyi')
       use iso_c_binding
       import cusparseHandle
       type(cusparseHandle), value :: handle
       integer(c_int), value :: nnz
       !dir$ ignore_tkr (d) alpha
       real(c_float) :: alpha
       real(c_float), device :: xVal(*)
       integer(c_int), device :: xInd(*)
       real(c_float), device :: y(*)
       integer(c_int), value :: idxBase
     end function cusparseSaxpyi_v2
  end interface cusparseSaxpyi_v2
#endif

! cusparseDaxpyi
  interface cusparseDaxpyi
    integer function cusparseDaxpyi_setHPM(handle, nnz, alpha, xVal, xInd, y, idxBase)
    import cusparseHandle
    type(cusparseHandle) :: handle
    integer :: nnz
    real(8) :: alpha
    real(8), device :: xVal(*)
    integer, device :: xInd(*)
    real(8), device :: y(*)
    integer :: idxBase
  end function cusparseDaxpyi_setHPM

    integer function cusparseDaxpyi_setDPM(handle, nnz, alpha, xVal, xInd, y, idxBase)
    import cusparseHandle
    type(cusparseHandle) :: handle
    integer :: nnz
    real(8), device :: alpha
    real(8), device :: xVal(*)
    integer, device :: xInd(*)
    real(8), device :: y(*)
    integer :: idxBase
    end function cusparseDaxpyi_setDPM
  end interface cusparseDaxpyi

#ifdef KEEP_V2_INTERFACES
  interface cusparseDaxpyi_v2
     integer(c_int) function cusparseDaxpyi_v2(handle, nnz, alpha, xVal, xInd, y, idxBase) &
          bind(C,name='cusparseDaxpyi')
       use iso_c_binding
       import cusparseHandle
       type(cusparseHandle), value :: handle
       integer(c_int), value :: nnz
       !dir$ ignore_tkr (d) alpha
       real(c_double) :: alpha
       real(c_double), device :: xVal(*)
       integer(c_int), device :: xInd(*)
       real(c_double), device :: y(*)
       integer(c_int), value :: idxBase
     end function cusparseDaxpyi_v2
  end interface cusparseDaxpyi_v2
#endif

! cusparseCaxpyi
  interface cusparseCaxpyi
    integer function cusparseCaxpyi_setHPM(handle, nnz, alpha, xVal, xInd, y, idxBase)
    import cusparseHandle
    type(cusparseHandle) :: handle
    integer :: nnz
    complex :: alpha
    complex, device :: xVal(*)
    integer, device :: xInd(*)
    complex, device :: y(*)
    integer :: idxBase
    end function cusparseCaxpyi_setHPM
    integer function cusparseCaxpyi_setDPM(handle, nnz, alpha, xVal, xInd, y, idxBase)
    import cusparseHandle
    type(cusparseHandle) :: handle
    integer :: nnz
    complex, device :: alpha
    complex, device :: xVal(*)
    integer, device :: xInd(*)
    complex, device :: y(*)
    integer :: idxBase
    end function cusparseCaxpyi_setDPM
  end interface cusparseCaxpyi

#ifdef KEEP_V2_INTERFACES
  interface cusparseCaxpyi_v2
     integer(c_int) function cusparseCaxpyi_v2(handle, nnz, alpha, xVal, xInd, y, idxBase) &
          bind(C,name='cusparseCaxpyi')
       use iso_c_binding
       import cusparseHandle
       type(cusparseHandle), value :: handle
       integer(c_int), value :: nnz
       !dir$ ignore_tkr (d) alpha
       complex(4) :: alpha
       complex(4), device :: xVal(*)
       integer(c_int), device :: xInd(*)
       complex(4), device :: y(*)
       integer(c_int), value :: idxBase
     end function cusparseCaxpyi_v2
  end interface cusparseCaxpyi_v2
#endif

! cusparseZaxpyi
  interface cusparseZaxpyi
    integer function cusparseZaxpyi_setHPM(handle, nnz, alpha, xVal, xInd, y, idxBase)
    import cusparseHandle
    type(cusparseHandle) :: handle
    integer :: nnz
    complex(8) :: alpha
    complex(8), device :: xVal(*)
    integer, device :: xInd(*)
    complex(8), device :: y(*)
    integer :: idxBase
    end function cusparseZaxpyi_setHPM
    integer function cusparseZaxpyi_setDPM(handle, nnz, alpha, xVal, xInd, y, idxBase)
    import cusparseHandle
    type(cusparseHandle) :: handle
    integer :: nnz
    complex(8), device :: alpha
    complex(8), device :: xVal(*)
    integer, device :: xInd(*)
    complex(8), device :: y(*)
    integer :: idxBase
    end function cusparseZaxpyi_setDPM
  end interface cusparseZaxpyi

#ifdef KEEP_V2_INTERFACES
  interface cusparseZaxpyi_v2
     integer(c_int) function cusparseZaxpyi_v2(handle, nnz, alpha, xVal, xInd, y, idxBase) &
          bind(C,name='cusparseZaxpyi')
       use iso_c_binding
       import cusparseHandle
       type(cusparseHandle), value :: handle
       integer(c_int), value :: nnz
       !dir$ ignore_tkr (d) alpha
       complex(8) :: alpha
       complex(8), device :: xVal(*)
       integer(c_int), device :: xInd(*)
       complex(8), device :: y(*)
       integer(c_int), value :: idxBase
     end function cusparseZaxpyi_v2
  end interface cusparseZaxpyi_v2
#endif

! cusparseSdoti
  interface cusparseSdoti
    integer function cusparseSdoti_setHPM(handle, nnz, xVal, xInd, y, res, idxBase)
    import cusparseHandle
    type(cusparseHandle) :: handle
    integer :: nnz
    real(4), device :: xVal(*)
    integer, device :: xInd(*)
    real(4), device :: y(*)
    real(4) :: res
    integer :: idxBase
    end function cusparseSdoti_setHPM
    integer function cusparseSdoti_setDPM(handle, nnz, xVal, xInd, y, res, idxBase)
    import cusparseHandle
    type(cusparseHandle) :: handle
    integer :: nnz
    real(4), device :: xVal(*)
    integer, device :: xInd(*)
    real(4), device :: y(*)
    real(4), device :: res
    integer :: idxBase
    end function cusparseSdoti_setDPM
  end interface cusparseSdoti

#ifdef KEEP_V2_INTERFACES
  interface cusparseSdoti_v2
     integer(c_int) function cusparseSdoti_v2(handle, nnz, xVal, xInd, y, res, idxBase) &
          bind(C,name='cusparseSdoti')
       use iso_c_binding
       import cusparseHandle
       type(cusparseHandle), value :: handle
       integer(c_int), value :: nnz
       real(c_float), device :: xVal(*)
       integer(c_int), device :: xInd(*)
       real(c_float), device :: y(*)
       !dir$ ignore_tkr (d) res
       real(c_float) :: res
       integer(c_int), value :: idxBase
     end function cusparseSdoti_v2
  end interface cusparseSdoti_v2
#endif

! cusparseDdoti
  interface cusparseDdoti
    integer function cusparseDdoti_setHPM(handle, nnz, xVal, xInd, y, res, idxBase)
    import cusparseHandle
    type(cusparseHandle) :: handle
    integer :: nnz
    real(8), device :: xVal(*)
    integer, device :: xInd(*)
    real(8), device :: y(*)
    real(8) :: res
    integer :: idxBase
    end function cusparseDdoti_setHPM
    integer function cusparseDdoti_setDPM(handle, nnz, xVal, xInd, y, res, idxBase)
    import cusparseHandle
    type(cusparseHandle) :: handle
    integer :: nnz
    real(8), device :: xVal(*)
    integer, device :: xInd(*)
    real(8), device :: y(*)
    real(8), device :: res
    integer :: idxBase
    end function cusparseDdoti_setDPM
  end interface cusparseDdoti

#ifdef KEEP_V2_INTERFACES
  interface cusparseDdoti_v2
     integer(c_int) function cusparseDdoti_v2(handle, nnz, xVal, xInd, y, res, idxBase) &
          bind(C,name='cusparseDdoti')
       use iso_c_binding
       import cusparseHandle
       type(cusparseHandle), value :: handle
       integer(c_int), value :: nnz
       real(c_double), device :: xVal(*)
       integer(c_int), device :: xInd(*)
       real(c_double), device :: y(*)
       !dir$ ignore_tkr (d) res
       real(c_double) :: res
       integer(c_int), value :: idxBase
     end function cusparseDdoti_v2
  end interface cusparseDdoti_v2
#endif

! cusparseCdoti
  interface cusparseCdoti
    integer function cusparseCdoti_setHPM(handle, nnz, xVal, xInd, y, res, idxBase)
    import cusparseHandle
    type(cusparseHandle) :: handle
    integer :: nnz
    complex(4), device :: xVal(*)
    integer, device :: xInd(*)
    complex(4), device :: y(*)
    complex(4) :: res
    integer :: idxBase
    end function cusparseCdoti_setHPM
    integer function cusparseCdoti_setDPM(handle, nnz, xVal, xInd, y, res, idxBase)
    import cusparseHandle
    type(cusparseHandle) :: handle
    integer :: nnz
    complex(4), device :: xVal(*)
    integer, device :: xInd(*)
    complex(4), device :: y(*)
    complex(4), device :: res
    integer :: idxBase
    end function cusparseCdoti_setDPM
  end interface cusparseCdoti

#ifdef KEEP_V2_INTERFACES
  interface cusparseCdoti_v2
     integer(c_int) function cusparseCdoti_v2(handle, nnz, xVal, xInd, y, res, idxBase) &
          bind(C,name='cusparseCdoti')
       use iso_c_binding
       import cusparseHandle
       type(cusparseHandle), value :: handle
       integer(c_int), value :: nnz
       complex(4), device :: xVal(*)
       integer(c_int), device :: xInd(*)
       complex(4), device :: y(*)
       !dir$ ignore_tkr (d) res
       complex(4) :: res
       integer(c_int), value :: idxBase
     end function cusparseCdoti_v2
  end interface cusparseCdoti_v2
#endif

! cusparseZdoti
  interface cusparseZdoti
    integer function cusparseZdoti_setHPM(handle, nnz, xVal, xInd, y, res, idxBase)
    import cusparseHandle
    type(cusparseHandle) :: handle
    integer :: nnz
    complex(8), device :: xVal(*)
    integer, device :: xInd(*)
    complex(8), device :: y(*)
    complex(8) :: res
    integer :: idxBase
    end function cusparseZdoti_setHPM
    integer function cusparseZdoti_setDPM(handle, nnz, xVal, xInd, y, res, idxBase)
    import cusparseHandle
    type(cusparseHandle) :: handle
    integer :: nnz
    complex(8), device :: xVal(*)
    integer, device :: xInd(*)
    complex(8), device :: y(*)
    complex(8), device :: res
    integer :: idxBase
    end function cusparseZdoti_setDPM
  end interface cusparseZdoti

#ifdef KEEP_V2_INTERFACES
  interface cusparseZdoti_v2
     integer(c_int) function cusparseZdoti_v2(handle, nnz, xVal, xInd, y, res, idxBase) &
          bind(C,name='cusparseZdoti')
       use iso_c_binding
       import cusparseHandle
       type(cusparseHandle), value :: handle
       integer(c_int), value :: nnz
       complex(8), device :: xVal(*)
       integer(c_int), device :: xInd(*)
       complex(8), device :: y(*)
       !dir$ ignore_tkr (d) res
       complex(8) :: res
       integer(c_int), value :: idxBase
     end function cusparseZdoti_v2
  end interface cusparseZdoti_v2
#endif

! cusparseCdotci
  interface cusparseCdotci
    integer function cusparseCdotci_setHPM(handle, nnz, xVal, xInd, y, res, idxBase)
    import cusparseHandle
    type(cusparseHandle) :: handle
    integer :: nnz
    complex(4), device :: xVal(*)
    integer, device :: xInd(*)
    complex(4), device :: y(*)
    complex(4) :: res
    integer :: idxBase
    end function cusparseCdotci_setHPM
    integer function cusparseCdotci_setDPM(handle, nnz, xVal, xInd, y, res, idxBase)
    import cusparseHandle
    type(cusparseHandle) :: handle
    integer :: nnz
    complex(4), device :: xVal(*)
    integer, device :: xInd(*)
    complex(4), device :: y(*)
    complex(4), device :: res
    integer :: idxBase
    end function cusparseCdotci_setDPM
  end interface cusparseCdotci

#ifdef KEEP_V2_INTERFACES
  interface cusparseCdotci_v2
     integer(c_int) function cusparseCdotci_v2(handle, nnz, xVal, xInd, y, res, idxBase) &
          bind(C,name='cusparseCdotci')
       use iso_c_binding
       import cusparseHandle
       type(cusparseHandle), value :: handle
       integer(c_int), value :: nnz
       complex(4), device :: xVal(*)
       integer(c_int), device :: xInd(*)
       complex(4), device :: y(*)
       !dir$ ignore_tkr (d) res
       complex(4) :: res
       integer(c_int), value :: idxBase
     end function cusparseCdotci_v2
  end interface cusparseCdotci_v2
#endif

! cusparseZdotci
  interface cusparseZdotci
    integer function cusparseZdotci_setHPM(handle, nnz, xVal, xInd, y, res, idxBase)
    import cusparseHandle
    type(cusparseHandle) :: handle
    integer :: nnz
    complex(8), device :: xVal(*)
    integer, device :: xInd(*)
    complex(8), device :: y(*)
    complex(8) :: res
    integer :: idxBase
    end function cusparseZdotci_setHPM
    integer function cusparseZdotci_setDPM(handle, nnz, xVal, xInd, y, res, idxBase)
    import cusparseHandle
    type(cusparseHandle) :: handle
    integer :: nnz
    complex(8), device :: xVal(*)
    integer, device :: xInd(*)
    complex(8), device :: y(*)
    complex(8), device :: res
    integer :: idxBase
    end function cusparseZdotci_setDPM
  end interface cusparseZdotci

#ifdef KEEP_V2_INTERFACES
  interface cusparseZdotci_v2
     integer(c_int) function cusparseZdotci_v2(handle, nnz, xVal, xInd, y, res, idxBase) &
          bind(C,name='cusparseZdotci')
       use iso_c_binding
       import cusparseHandle
       type(cusparseHandle), value :: handle
       integer(c_int), value :: nnz
       complex(8), device :: xVal(*)
       integer(c_int), device :: xInd(*)
       complex(8), device :: y(*)
       !dir$ ignore_tkr (d) res
       complex(8) :: res
       integer(c_int), value :: idxBase
     end function cusparseZdotci_v2
  end interface cusparseZdotci_v2
#endif

! cusparseSgthr
  interface 
     integer(c_int) function cusparseSgthr(handle, nnz, y, xVal, xInd, idxBase) &
          bind(C,name='cusparseSgthr')
       use iso_c_binding
       import cusparseHandle
       type(cusparseHandle), value :: handle
       integer(c_int), value :: nnz
       real(c_float), device :: y(*)
       real(c_float), device :: xVal(*)
       integer(c_int), device :: xInd(*)
       integer(c_int), value :: idxBase
     end function cusparseSgthr
  end interface

! cusparseDgthr
  interface 
     integer(c_int) function cusparseDgthr(handle, nnz, y, xVal, xInd, idxBase) &
          bind(C,name='cusparseDgthr')
       use iso_c_binding
       import cusparseHandle
       type(cusparseHandle), value :: handle
       integer(c_int), value :: nnz
       real(c_double), device :: y(*)
       real(c_double), device :: xVal(*)
       integer(c_int), device :: xInd(*)
       integer(c_int), value :: idxBase
     end function cusparseDgthr
  end interface

! cusparseCgthr
  interface 
     integer(c_int) function cusparseCgthr(handle, nnz, y, xVal, xInd, idxBase) &
          bind(C,name='cusparseCgthr')
       use iso_c_binding
       import cusparseHandle
       type(cusparseHandle), value :: handle
       integer(c_int), value :: nnz
       complex(4), device :: y(*)
       complex(4), device :: xVal(*)
       integer(c_int), device :: xInd(*)
       integer(c_int), value :: idxBase
     end function cusparseCgthr
  end interface

! cusparseZgthr
  interface 
     integer(c_int) function cusparseZgthr(handle, nnz, y, xVal, xInd, idxBase) &
          bind(C,name='cusparseZgthr')
       use iso_c_binding
       import cusparseHandle
       type(cusparseHandle), value :: handle
       integer(c_int), value :: nnz
       complex(8), device :: y(*)
       complex(8), device :: xVal(*)
       integer(c_int), device :: xInd(*)
       integer(c_int), value :: idxBase
     end function cusparseZgthr
  end interface

! cusparseSgthrz
  interface 
     integer(c_int) function cusparseSgthrz(handle, nnz, y, xVal, xInd, idxBase) &
          bind(C,name='cusparseSgthrz')
       use iso_c_binding
       import cusparseHandle
       type(cusparseHandle), value :: handle
       integer(c_int), value :: nnz
       real(c_float), device :: y(*)
       real(c_float), device :: xVal(*)
       integer(c_int), device :: xInd(*)
       integer(c_int), value :: idxBase
     end function cusparseSgthrz
  end interface

! cusparseDgthrz
  interface 
     integer(c_int) function cusparseDgthrz(handle, nnz, y, xVal, xInd, idxBase) &
          bind(C,name='cusparseDgthrz')
       use iso_c_binding
       import cusparseHandle
       type(cusparseHandle), value :: handle
       integer(c_int), value :: nnz
       real(c_double), device :: y(*)
       real(c_double), device :: xVal(*)
       integer(c_int), device :: xInd(*)
       integer(c_int), value :: idxBase
     end function cusparseDgthrz
  end interface

! cusparseCgthrz
  interface 
     integer(c_int) function cusparseCgthrz(handle, nnz, y, xVal, xInd, idxBase) &
          bind(C,name='cusparseCgthrz')
       use iso_c_binding
       import cusparseHandle
       type(cusparseHandle), value :: handle
       integer(c_int), value :: nnz
       complex(4), device :: y(*)
       complex(4), device :: xVal(*)
       integer(c_int), device :: xInd(*)
       integer(c_int), value :: idxBase
     end function cusparseCgthrz
  end interface

! cusparseZgthrz
  interface 
     integer(c_int) function cusparseZgthrz(handle, nnz, y, xVal, xInd, idxBase) &
          bind(C,name='cusparseZgthrz')
       use iso_c_binding
       import cusparseHandle
       type(cusparseHandle), value :: handle
       integer(c_int), value :: nnz
       complex(8), device :: y(*)
       complex(8), device :: xVal(*)
       integer(c_int), device :: xInd(*)
       integer(c_int), value :: idxBase
     end function cusparseZgthrz
  end interface

! cusparseSsctr
  interface
     integer(c_int) function cusparseSsctr(handle, nnz, xVal, xInd, y, idxBase) &
          bind(C,name='cusparseSsctr')
       use iso_c_binding
       import cusparseHandle
       type(cusparseHandle), value :: handle
       integer(c_int), value :: nnz
       real(c_float), device :: xVal(*)
       integer(c_int), device :: xInd(*)
       real(c_float), device :: y(*)
       integer(c_int), value :: idxBase
     end function cusparseSsctr
  end interface

! cusparseDsctr
  interface
     integer(c_int) function cusparseDsctr(handle, nnz, xVal, xInd, y, idxBase) &
          bind(C,name='cusparseDsctr')
       use iso_c_binding
       import cusparseHandle
       type(cusparseHandle), value :: handle
       integer(c_int), value :: nnz
       real(c_double), device :: xVal(*)
       integer(c_int), device :: xInd(*)
       real(c_double), device :: y(*)
       integer(c_int), value :: idxBase
     end function cusparseDsctr
  end interface

! cusparseCsctr
  interface
     integer(c_int) function cusparseCsctr(handle, nnz, xVal, xInd, y, idxBase) &
          bind(C,name='cusparseCsctr')
       use iso_c_binding
       import cusparseHandle
       type(cusparseHandle), value :: handle
       integer(c_int), value :: nnz
       complex(4), device :: xVal(*)
       integer(c_int), device :: xInd(*)
       complex(4), device :: y(*)
       integer(c_int), value :: idxBase
     end function cusparseCsctr
  end interface

! cusparseZsctr
  interface
     integer(c_int) function cusparseZsctr(handle, nnz, xVal, xInd, y, idxBase) &
          bind(C,name='cusparseZsctr')
       use iso_c_binding
       import cusparseHandle
       type(cusparseHandle), value :: handle
       integer(c_int), value :: nnz
       complex(8), device :: xVal(*)
       integer(c_int), device :: xInd(*)
       complex(8), device :: y(*)
       integer(c_int), value :: idxBase
     end function cusparseZsctr
  end interface

! cusparseSroti
  interface cusparseSroti
    integer function cusparseSroti_setHPM(handle, nnz, xVal, xInd, y, c, s, idxBase)
    import cusparseHandle
    type(cusparseHandle) :: handle
    integer :: nnz
    real, device :: xVal(*)
    integer, device :: xInd(*)
    real, device :: y(*)
    real :: c, s
    integer :: idxBase
    end function cusparseSroti_setHPM
    integer function cusparseSroti_setDPM(handle, nnz, xVal, xInd, y, c, s, idxBase)
    import cusparseHandle
    type(cusparseHandle) :: handle
    integer :: nnz
    real, device :: xVal(*)
    integer, device :: xInd(*)
    real, device :: y(*)
    real, device :: c, s
    integer :: idxBase
    end function cusparseSroti_setDPM
  end interface cusparseSroti

#ifdef KEEP_V2_INTERFACES
  interface
     integer(c_int) function cusparseSroti_v2(handle, nnz, xVal, xInd, y, c, s, idxBase) &
          bind(C,name='cusparseSroti')
       use iso_c_binding
       import cusparseHandle
       type(cusparseHandle), value :: handle
       integer(c_int), value :: nnz
       real(c_float), device :: xVal(*)
       integer(c_int), device :: xInd(*)
       real(c_float), device :: y(*)
       !dir$ ignore_tkr(d) c, (d) s
       real(c_float) :: c, s
       integer(c_int), value :: idxBase
     end function cusparseSroti_v2
  end interface
#endif

! cusparseDroti
  interface cusparseDroti
    integer function cusparseDroti_setHPM(handle, nnz, xVal, xInd, y, c, s, idxBase)
    import cusparseHandle
    type(cusparseHandle) :: handle
    integer :: nnz
    real(8), device :: xVal(*)
    integer, device :: xInd(*)
    real(8), device :: y(*)
    real(8) :: c, s
    integer :: idxBase
    end function cusparseDroti_setHPM
    integer function cusparseDroti_setDPM(handle, nnz, xVal, xInd, y, c, s, idxBase)
    import cusparseHandle
    type(cusparseHandle) :: handle
    integer :: nnz
    real(8), device :: xVal(*)
    integer, device :: xInd(*)
    real(8), device :: y(*)
    real(8), device :: c, s
    integer :: idxBase
    end function cusparseDroti_setDPM
  end interface cusparseDroti

#ifdef KEEP_V2_INTERFACES
  interface
     integer(c_int) function cusparseDroti_v2(handle, nnz, xVal, xInd, y, c, s, idxBase) &
          bind(C,name='cusparseDroti')
       use iso_c_binding
       import cusparseHandle
       type(cusparseHandle), value :: handle
       integer(c_int), value :: nnz
       real(c_double), device :: xVal(*)
       integer(c_int), device :: xInd(*)
       real(c_double), device :: y(*)
       !dir$ ignore_tkr(d) c, (d) s
       real(c_double) :: c, s
       integer(c_int), value :: idxBase
     end function cusparseDroti_v2
  end interface
#endif

  ! ---------------------
  ! 6.0 Level 1 functions 
  ! ---------------------
  ! none

  ! ---------------------
  ! 6.5 Level 1 functions
  ! ---------------------
  ! none

  ! ---------------------
  ! 7.0 Level 1 functions
  ! ---------------------
  ! none

  ! ---------------------
  ! 5.5 Level 2 functions
  ! ---------------------
  
! cusparseSbsrmv
  interface cusparseSbsrmv
    integer function cusparseSbsrmv_setHPM(handle, dir, trans, mb, nb, nnzb, &
       alpha, descr, bsrVal, bsrRowPtr, bsrColInd, blockDim, x, beta, y)
    import cusparseHandle, cusparseMatDescr
    type(cusparseHandle) :: handle
    integer :: dir
    integer :: trans
    integer :: mb, nb, nnzb
    real(4) :: alpha
    type(cusparseMatDescr) :: descr
    real(4), device :: bsrVal(*)
    integer, device :: bsrRowPtr(*)
    integer, device :: bsrColInd(*)
    integer :: blockDim
    real(4), device :: x(*)
    real(4) :: beta
    real(4), device :: y(*)     
    end function cusparseSbsrmv_setHPM
    integer function cusparseSbsrmv_setDPM(handle, dir, trans, mb, nb, nnzb, &
       alpha, descr, bsrVal, bsrRowPtr, bsrColInd, blockDim, x, beta, y)
    import cusparseHandle, cusparseMatDescr
    type(cusparseHandle) :: handle
    integer :: dir
    integer :: trans
    integer :: mb, nb, nnzb
    real(4), device :: alpha
    type(cusparseMatDescr) :: descr
    real(4), device :: bsrVal(*)
    integer, device :: bsrRowPtr(*)
    integer, device :: bsrColInd(*)
    integer :: blockDim
    real(4), device :: x(*)
    real(4), device :: beta
    real(4), device :: y(*)     
    end function cusparseSbsrmv_setDPM
  end interface cusparseSbsrmv

#ifdef KEEP_V2_INTERFACES
  interface cusparseSbsrmv_v2
     integer(c_int) function cusparseSbsrmv_v2(handle, dir, trans, mb, nb, nnzb, &
          alpha, descr, bsrVal, bsrRowPtr, bsrColInd, blockDim, x, beta, y) bind(C,name='cusparseSbsrmv')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr
       type(cusparseHandle), value :: handle
       integer(c_int), value :: dir
       integer(c_int), value :: trans
       integer(c_int), value :: mb, nb, nnzb
       !dir$ ignore_tkr (d) alpha, (d) beta
       real(c_float) :: alpha
       type(cusparseMatDescr), value :: descr
       real(c_float), device :: bsrVal(*)
       integer(c_int), device :: bsrRowPtr(*)
       integer(c_int), device :: bsrColInd(*)
       integer(c_int), value :: blockDim
       real(c_float), device :: x(*)
       real(c_float) :: beta
       real(c_float), device :: y(*) 
     end function cusparseSbsrmv_v2
  end interface cusparseSbsrmv_v2
#endif

! cusparseDbsrmv
  interface cusparseDbsrmv
    integer function cusparseDbsrmv_setHPM(handle, dir, trans, mb, nb, nnzb, &
       alpha, descr, bsrVal, bsrRowPtr, bsrColInd, blockDim, x, beta, y)
    import cusparseHandle, cusparseMatDescr
    type(cusparseHandle) :: handle
    integer :: dir
    integer :: trans
    integer :: mb, nb, nnzb
    real(8) :: alpha
    type(cusparseMatDescr) :: descr
    real(8), device :: bsrVal(*)
    integer, device :: bsrRowPtr(*)
    integer, device :: bsrColInd(*)
    integer :: blockDim
    real(8), device :: x(*)
    real(8) :: beta
    real(8), device :: y(*)     
    end function cusparseDbsrmv_setHPM
    integer function cusparseDbsrmv_setDPM(handle, dir, trans, mb, nb, nnzb, &
       alpha, descr, bsrVal, bsrRowPtr, bsrColInd, blockDim, x, beta, y)
    import cusparseHandle, cusparseMatDescr
    type(cusparseHandle) :: handle
    integer :: dir
    integer :: trans
    integer :: mb, nb, nnzb
    real(8), device :: alpha
    type(cusparseMatDescr) :: descr
    real(8), device :: bsrVal(*)
    integer, device :: bsrRowPtr(*)
    integer, device :: bsrColInd(*)
    integer :: blockDim
    real(8), device :: x(*)
    real(8), device :: beta
    real(8), device :: y(*)     
    end function cusparseDbsrmv_setDPM
  end interface cusparseDbsrmv

#ifdef KEEP_V2_INTERFACES
  interface cusparseDbsrmv_v2
     integer(c_int) function cusparseDbsrmv_v2(handle, dir, trans, mb, nb, nnzb, &
          alpha, descr, bsrVal, bsrRowPtr, bsrColInd, blockDim, x, beta, y) bind(C,name='cusparseDbsrmv')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr
       type(cusparseHandle), value :: handle
       integer(c_int), value :: dir
       integer(c_int), value :: trans
       integer(c_int), value :: mb, nb, nnzb
       !dir$ ignore_tkr (d) alpha, (d) beta
       real(c_double) :: alpha
       type(cusparseMatDescr), value :: descr
       real(c_double), device :: bsrVal(*)
       integer(c_int), device :: bsrRowPtr(*)
       integer(c_int), device :: bsrColInd(*)
       integer(c_int), value :: blockDim
       real(c_double), device :: x(*)
       real(c_double) :: beta
       real(c_double), device :: y(*) 
     end function cusparseDbsrmv_v2
  end interface cusparseDbsrmv_v2
#endif

! cusparseCbsrmv
  interface cusparseCbsrmv
    integer function cusparseCbsrmv_setHPM(handle, dir, trans, mb, nb, nnzb, &
       alpha, descr, bsrVal, bsrRowPtr, bsrColInd, blockDim, x, beta, y)
    import cusparseHandle, cusparseMatDescr
    type(cusparseHandle) :: handle
    integer :: dir
    integer :: trans
    integer :: mb, nb, nnzb
    complex(4) :: alpha
    type(cusparseMatDescr) :: descr
    complex(4), device :: bsrVal(*)
    integer, device :: bsrRowPtr(*)
    integer, device :: bsrColInd(*)
    integer :: blockDim
    complex(4), device :: x(*)
    complex(4) :: beta
    complex(4), device :: y(*)     
    end function cusparseCbsrmv_setHPM
    integer function cusparseCbsrmv_setDPM(handle, dir, trans, mb, nb, nnzb, &
       alpha, descr, bsrVal, bsrRowPtr, bsrColInd, blockDim, x, beta, y)
    import cusparseHandle, cusparseMatDescr
    type(cusparseHandle) :: handle
    integer :: dir
    integer :: trans
    integer :: mb, nb, nnzb
    complex(4), device :: alpha
    type(cusparseMatDescr) :: descr
    complex(4), device :: bsrVal(*)
    integer, device :: bsrRowPtr(*)
    integer, device :: bsrColInd(*)
    integer :: blockDim
    complex(4), device :: x(*)
    complex(4), device :: beta
    complex(4), device :: y(*)     
    end function cusparseCbsrmv_setDPM
  end interface cusparseCbsrmv

#ifdef KEEP_V2_INTERFACES
  interface cusparseCbsrmv_v2
     integer(c_int) function cusparseCbsrmv_v2(handle, dir, trans, mb, nb, nnzb, &
          alpha, descr, bsrVal, bsrRowPtr, bsrColInd, blockDim, x, beta, y) bind(C,name='cusparseCbsrmv')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr
       type(cusparseHandle), value :: handle
       integer(c_int), value :: dir
       integer(c_int), value :: trans
       integer(c_int), value :: mb, nb, nnzb
       !dir$ ignore_tkr (d) alpha, (d) beta
       complex(4) :: alpha
       type(cusparseMatDescr), value :: descr
       complex(4), device :: bsrVal(*)
       integer(c_int), device :: bsrRowPtr(*)
       integer(c_int), device :: bsrColInd(*)
       integer(c_int), value :: blockDim
       complex(4), device :: x(*)
       complex(4) :: beta
       complex(4), device :: y(*) 
     end function cusparseCbsrmv_v2
  end interface cusparseCbsrmv_v2
#endif

! cusparseZbsrmv
  interface cusparseZbsrmv
    integer function cusparseZbsrmv_setHPM(handle, dir, trans, mb, nb, nnzb, &
       alpha, descr, bsrVal, bsrRowPtr, bsrColInd, blockDim, x, beta, y)
    import cusparseHandle, cusparseMatDescr
    type(cusparseHandle) :: handle
    integer :: dir
    integer :: trans
    integer :: mb, nb, nnzb
    complex(8) :: alpha
    type(cusparseMatDescr) :: descr
    complex(8), device :: bsrVal(*)
    integer, device :: bsrRowPtr(*)
    integer, device :: bsrColInd(*)
    integer :: blockDim
    complex(8), device :: x(*)
    complex(8) :: beta
    complex(8), device :: y(*)     
    end function cusparseZbsrmv_setHPM
    integer function cusparseZbsrmv_setDPM(handle, dir, trans, mb, nb, nnzb, &
       alpha, descr, bsrVal, bsrRowPtr, bsrColInd, blockDim, x, beta, y)
    import cusparseHandle, cusparseMatDescr
    type(cusparseHandle) :: handle
    integer :: dir
    integer :: trans
    integer :: mb, nb, nnzb
    complex(8), device :: alpha
    type(cusparseMatDescr) :: descr
    complex(8), device :: bsrVal(*)
    integer, device :: bsrRowPtr(*)
    integer, device :: bsrColInd(*)
    integer :: blockDim
    complex(8), device :: x(*)
    complex(8), device :: beta
    complex(8), device :: y(*)     
    end function cusparseZbsrmv_setDPM
  end interface cusparseZbsrmv

#ifdef KEEP_V2_INTERFACES
  interface cusparseZbsrmv_v2
     integer(c_int) function cusparseZbsrmv_v2(handle, dir, trans, mb, nb, nnzb, &
          alpha, descr, bsrVal, bsrRowPtr, bsrColInd, blockDim, x, beta, y) bind(C,name='cusparseZbsrmv')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr
       type(cusparseHandle), value :: handle
       integer(c_int), value :: dir
       integer(c_int), value :: trans
       integer(c_int), value :: mb, nb, nnzb
       !dir$ ignore_tkr (d) alpha, (d) beta
       complex(8) :: alpha
       type(cusparseMatDescr), value :: descr
       complex(8), device :: bsrVal(*)
       integer(c_int), device :: bsrRowPtr(*)
       integer(c_int), device :: bsrColInd(*)
       integer(c_int), value :: blockDim
       complex(8), device :: x(*)
       complex(8) :: beta
       complex(8), device :: y(*) 
     end function cusparseZbsrmv_v2
  end interface cusparseZbsrmv_v2
#endif

! cusparseSbsrxmv
  interface cusparseSbsrxmv
    integer function cusparseSbsrxmv_setHPM(handle, dir, trans, sizeOfMask, mb, nb, nnzb, &
       alpha, descr, bsrVal, bsrMaskPtr, bsrRowPtr, bsrEndPtr, bsrColInd, blockDim, x, beta, y)
    import cusparseHandle, cusparseMatDescr
    type(cusparseHandle) :: handle
    integer :: dir
    integer :: trans
    integer :: sizeOfMask
    integer :: mb, nb, nnzb
    real(4) :: alpha
    type(cusparseMatDescr) :: descr
    real(4), device :: bsrVal(*)
    integer, device :: bsrMaskPtr(*), bsrRowPtr(*), bsrEndPtr(*)
    integer, device :: bsrColInd(*)
    integer :: blockDim
    real(4), device :: x(*)
    real(4) :: beta
    real(4), device :: y(*)     
    end function cusparseSbsrxmv_setHPM
    integer function cusparseSbsrxmv_setDPM(handle, dir, trans, sizeOfMask, mb, nb, nnzb, &
       alpha, descr, bsrVal, bsrMaskPtr, bsrRowPtr, bsrEndPtr, bsrColInd, blockDim, x, beta, y)
    import cusparseHandle, cusparseMatDescr
    type(cusparseHandle) :: handle
    integer :: dir
    integer :: trans
    integer :: sizeOfMask
    integer :: mb, nb, nnzb
    real(4), device :: alpha
    type(cusparseMatDescr) :: descr
    real(4), device :: bsrVal(*)
    integer, device :: bsrMaskPtr(*), bsrRowPtr(*), bsrEndPtr(*)
    integer, device :: bsrColInd(*)
    integer :: blockDim
    real(4), device :: x(*)
    real(4), device :: beta
    real(4), device :: y(*)     
    end function cusparseSbsrxmv_setDPM
  end interface cusparseSbsrxmv

#ifdef KEEP_V2_INTERFACES
  interface cusparseSbsrxmv_v2
     integer(c_int) function cusparseSbsrxmv_v2(handle, dir, trans, sizeOfMask, mb, nb, nnzb, &
          alpha, descr, bsrVal, bsrMaskPtr, bsrRowPtr, bsrEndPtr, bsrColInd, blockDim, x, beta, y) &
          bind(C,name='cusparseSbsrxmv')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr
       type(cusparseHandle), value :: handle
       integer(c_int), value :: dir
       integer(c_int), value :: trans
       integer(c_int), value :: sizeOfMask
       integer(c_int), value :: mb, nb, nnzb
       !dir$ ignore_tkr (d) alpha, (d) beta
       real(c_float) :: alpha
       type(cusparseMatDescr), value :: descr
       real(c_float), device :: bsrVal(*)
       integer(c_int), device :: bsrMaskPtr(*), bsrRowPtr(*), bsrEndPtr(*)
       integer(c_int), device :: bsrColInd(*)
       integer(c_int), value :: blockDim
       real(c_float), device :: x(*)
       real(c_float) :: beta
       real(c_float), device :: y(*) 
     end function cusparseSbsrxmv_v2
  end interface cusparseSbsrxmv_v2
#endif
  
! cusparseDbsrxmv
  interface cusparseDbsrxmv
    integer function cusparseDbsrxmv_setHPM(handle, dir, trans, sizeOfMask, mb, nb, nnzb, &
       alpha, descr, bsrVal, bsrMaskPtr, bsrRowPtr, bsrEndPtr, bsrColInd, blockDim, x, beta, y)
    import cusparseHandle, cusparseMatDescr
    type(cusparseHandle) :: handle
    integer :: dir
    integer :: trans
    integer :: sizeOfMask
    integer :: mb, nb, nnzb
    real(8) :: alpha
    type(cusparseMatDescr) :: descr
    real(8), device :: bsrVal(*)
    integer, device :: bsrMaskPtr(*), bsrRowPtr(*), bsrEndPtr(*)
    integer, device :: bsrColInd(*)
    integer :: blockDim
    real(8), device :: x(*)
    real(8) :: beta
    real(8), device :: y(*)     
    end function cusparseDbsrxmv_setHPM
    integer function cusparseDbsrxmv_setDPM(handle, dir, trans, sizeOfMask, mb, nb, nnzb, &
       alpha, descr, bsrVal, bsrMaskPtr, bsrRowPtr, bsrEndPtr, bsrColInd, blockDim, x, beta, y)
    import cusparseHandle, cusparseMatDescr
    type(cusparseHandle) :: handle
    integer :: dir
    integer :: trans
    integer :: sizeOfMask
    integer :: mb, nb, nnzb
    real(8), device :: alpha
    type(cusparseMatDescr) :: descr
    real(8), device :: bsrVal(*)
    integer, device :: bsrMaskPtr(*), bsrRowPtr(*), bsrEndPtr(*)
    integer, device :: bsrColInd(*)
    integer :: blockDim
    real(8), device :: x(*)
    real(8), device :: beta
    real(8), device :: y(*)     
    end function cusparseDbsrxmv_setDPM
  end interface cusparseDbsrxmv

#ifdef KEEP_V2_INTERFACES
  interface cusparseDbsrxmv_v2
     integer(c_int) function cusparseDbsrxmv_v2(handle, dir, trans, sizeOfMask, mb, nb, nnzb, &
          alpha, descr, bsrVal, bsrMaskPtr, bsrRowPtr, bsrEndPtr, bsrColInd, blockDim, x, beta, y) &
          bind(C,name='cusparseDbsrxmv')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr
       type(cusparseHandle), value :: handle
       integer(c_int), value :: dir
       integer(c_int), value :: trans
       integer(c_int), value :: sizeOfMask
       integer(c_int), value :: mb, nb, nnzb
       !dir$ ignore_tkr(d) alpha, (d) beta
       real(c_double) :: alpha
       type(cusparseMatDescr), value :: descr
       real(c_double), device :: bsrVal(*)
       integer(c_int), device :: bsrMaskPtr(*), bsrRowPtr(*), bsrEndPtr(*)
       integer(c_int), device :: bsrColInd(*)
       integer(c_int), value :: blockDim
       real(c_double), device :: x(*)
       real(c_double) :: beta
       real(c_double), device :: y(*) 
     end function cusparseDbsrxmv_v2
  end interface cusparseDbsrxmv_v2
#endif

! cusparseCbsrxmv
  interface cusparseCbsrxmv
    integer function cusparseCbsrxmv_setHPM(handle, dir, trans, sizeOfMask, mb, nb, nnzb, &
       alpha, descr, bsrVal, bsrMaskPtr, bsrRowPtr, bsrEndPtr, bsrColInd, blockDim, x, beta, y)
    import cusparseHandle, cusparseMatDescr
    type(cusparseHandle) :: handle
    integer :: dir
    integer :: trans
    integer :: sizeOfMask
    integer :: mb, nb, nnzb
    complex(4) :: alpha
    type(cusparseMatDescr) :: descr
    complex(4), device :: bsrVal(*)
    integer, device :: bsrMaskPtr(*), bsrRowPtr(*), bsrEndPtr(*)
    integer, device :: bsrColInd(*)
    integer :: blockDim
    complex(4), device :: x(*)
    complex(4) :: beta
    complex(4), device :: y(*)     
    end function cusparseCbsrxmv_setHPM
    integer function cusparseCbsrxmv_setDPM(handle, dir, trans, sizeOfMask, mb, nb, nnzb, &
       alpha, descr, bsrVal, bsrMaskPtr, bsrRowPtr, bsrEndPtr, bsrColInd, blockDim, x, beta, y)
    import cusparseHandle, cusparseMatDescr
    type(cusparseHandle) :: handle
    integer :: dir
    integer :: trans
    integer :: sizeOfMask
    integer :: mb, nb, nnzb
    complex(4), device :: alpha
    type(cusparseMatDescr) :: descr
    complex(4), device :: bsrVal(*)
    integer, device :: bsrMaskPtr(*), bsrRowPtr(*), bsrEndPtr(*)
    integer, device :: bsrColInd(*)
    integer :: blockDim
    complex(4), device :: x(*)
    complex(4), device :: beta
    complex(4), device :: y(*)     
    end function cusparseCbsrxmv_setDPM
  end interface cusparseCbsrxmv

#ifdef KEEP_V2_INTERFACES
  interface cusparseCbsrxmv_v2
     integer(c_int) function cusparseCbsrxmv_v2(handle, dir, trans, sizeOfMask, mb, nb, nnzb, &
          alpha, descr, bsrVal, bsrMaskPtr, bsrRowPtr, bsrEndPtr, bsrColInd, blockDim, x, beta, y) &
          bind(C,name='cusparseCbsrxmv')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr
       type(cusparseHandle), value :: handle
       integer(c_int), value :: dir
       integer(c_int), value :: trans
       integer(c_int), value :: sizeOfMask
       integer(c_int), value :: mb, nb, nnzb
       !dir$ ignore_tkr (d) alpha, (d) beta
       complex(4) :: alpha
       type(cusparseMatDescr), value :: descr
       complex(4), device :: bsrVal(*)
       integer(c_int), device :: bsrMaskPtr(*), bsrRowPtr(*), bsrEndPtr(*)
       integer(c_int), device :: bsrColInd(*)
       integer(c_int), value :: blockDim
       complex(4), device :: x(*)
       complex(4) :: beta
       complex(4), device :: y(*) 
     end function cusparseCbsrxmv_v2
  end interface cusparseCbsrxmv_v2
#endif
  
! cusparseZbsrxmv
  interface cusparseZbsrxmv
    integer function cusparseZbsrxmv_setHPM(handle, dir, trans, sizeOfMask, mb, nb, nnzb, &
       alpha, descr, bsrVal, bsrMaskPtr, bsrRowPtr, bsrEndPtr, bsrColInd, blockDim, x, beta, y)
    import cusparseHandle, cusparseMatDescr
    type(cusparseHandle) :: handle
    integer :: dir
    integer :: trans
    integer :: sizeOfMask
    integer :: mb, nb, nnzb
    complex(8) :: alpha
    type(cusparseMatDescr) :: descr
    complex(8), device :: bsrVal(*)
    integer, device :: bsrMaskPtr(*), bsrRowPtr(*), bsrEndPtr(*)
    integer, device :: bsrColInd(*)
    integer :: blockDim
    complex(8), device :: x(*)
    complex(8) :: beta
    complex(8), device :: y(*)     
    end function cusparseZbsrxmv_setHPM
    integer function cusparseZbsrxmv_setDPM(handle, dir, trans, sizeOfMask, mb, nb, nnzb, &
       alpha, descr, bsrVal, bsrMaskPtr, bsrRowPtr, bsrEndPtr, bsrColInd, blockDim, x, beta, y)
    import cusparseHandle, cusparseMatDescr
    type(cusparseHandle) :: handle
    integer :: dir
    integer :: trans
    integer :: sizeOfMask
    integer :: mb, nb, nnzb
    complex(8), device :: alpha
    type(cusparseMatDescr) :: descr
    complex(8), device :: bsrVal(*)
    integer, device :: bsrMaskPtr(*), bsrRowPtr(*), bsrEndPtr(*)
    integer, device :: bsrColInd(*)
    integer :: blockDim
    complex(8), device :: x(*)
    complex(8), device :: beta
    complex(8), device :: y(*)     
    end function cusparseZbsrxmv_setDPM
  end interface cusparseZbsrxmv

#ifdef KEEP_V2_INTERFACES
  interface cusparseZbsrxmv_v2
     integer(c_int) function cusparseZbsrxmv_v2(handle, dir, trans, sizeOfMask, mb, nb, nnzb, &
          alpha, descr, bsrVal, bsrMaskPtr, bsrRowPtr, bsrEndPtr, bsrColInd, blockDim, x, beta, y) &
          bind(C,name='cusparseZbsrxmv')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr
       type(cusparseHandle), value :: handle
       integer(c_int), value :: dir
       integer(c_int), value :: trans
       integer(c_int), value :: sizeOfMask
       integer(c_int), value :: mb, nb, nnzb
       !dir$ ignore_tkr (d) alpha, (d) beta
       complex(8) :: alpha
       type(cusparseMatDescr), value :: descr
       complex(8), device :: bsrVal(*)
       integer(c_int), device :: bsrMaskPtr(*), bsrRowPtr(*), bsrEndPtr(*)
       integer(c_int), device :: bsrColInd(*)
       integer(c_int), value :: blockDim
       complex(8), device :: x(*)
       complex(8) :: beta
       complex(8), device :: y(*) 
     end function cusparseZbsrxmv_v2
  end interface cusparseZbsrxmv_v2
#endif
  
! cusparseScsrmv
  interface cusparseScsrmv
    integer function cusparseScsrmv_setHPM(handle, trans, m, n, nnz, &
       alpha, descr, csrVal, csrRowPtr, csrColInd, x, beta, y)
    import cusparseHandle, cusparseMatDescr
    type(cusparseHandle) :: handle
    integer :: trans
    integer :: m, n, nnz
    real(4) :: alpha
    type(cusparseMatDescr) :: descr
    real(4), device :: csrVal(*)
    integer, device :: csrRowPtr(*)
    integer, device :: csrColInd(*)
    real(4), device :: x(*)
    real(4) :: beta
    real(4), device :: y(*)     
    end function cusparseScsrmv_setHPM
    integer function cusparseScsrmv_setDPM(handle, trans, m, n, nnz, &
       alpha, descr, csrVal, csrRowPtr, csrColInd, x, beta, y)
    import cusparseHandle, cusparseMatDescr
    type(cusparseHandle) :: handle
    integer :: trans
    integer :: m, n, nnz
    real(4), device :: alpha
    type(cusparseMatDescr) :: descr
    real(4), device :: csrVal(*)
    integer, device :: csrRowPtr(*)
    integer, device :: csrColInd(*)
    real(4), device :: x(*)
    real(4), device :: beta
    real(4), device :: y(*)     
    end function cusparseScsrmv_setDPM
  end interface cusparseScsrmv

#ifdef KEEP_V2_INTERFACES
  interface cusparseScsrmv_v2
     integer(c_int) function cusparseScsrmv_v2(handle, trans, m, n, nnz, &
          alpha, descr, csrVal, csrRowPtr, csrColInd, x, beta, y) bind(C,name='cusparseScsrmv')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr
       type(cusparseHandle), value :: handle
       integer(c_int), value :: trans
       integer(c_int), value :: m, n, nnz
       !dir$ ignore_tkr (d) alpha, (d) beta
       real(c_float) :: alpha
       type(cusparseMatDescr), value :: descr
       real(c_float), device :: csrVal(*)
       integer(c_int), device :: csrRowPtr(*)
       integer(c_int), device :: csrColInd(*)
       real(c_float), device :: x(*)
       real(c_float) :: beta
       real(c_float), device :: y(*) 
     end function cusparseScsrmv_v2
  end interface cusparseScsrmv_v2
#endif

! cusparseDcsrmv
  interface cusparseDcsrmv
    integer function cusparseDcsrmv_setHPM(handle, trans, m, n, nnz, &
       alpha, descr, csrVal, csrRowPtr, csrColInd, x, beta, y)
    import cusparseHandle, cusparseMatDescr
    type(cusparseHandle) :: handle
    integer :: trans
    integer :: m, n, nnz
    real(8) :: alpha
    type(cusparseMatDescr) :: descr
    real(8), device :: csrVal(*)
    integer, device :: csrRowPtr(*)
    integer, device :: csrColInd(*)
    real(8), device :: x(*)
    real(8) :: beta
    real(8), device :: y(*)     
    end function cusparseDcsrmv_setHPM
    integer function cusparseDcsrmv_setDPM(handle, trans, m, n, nnz, &
       alpha, descr, csrVal, csrRowPtr, csrColInd, x, beta, y)
    import cusparseHandle, cusparseMatDescr
    type(cusparseHandle) :: handle
    integer :: trans
    integer :: m, n, nnz
    real(8), device :: alpha
    type(cusparseMatDescr) :: descr
    real(8), device :: csrVal(*)
    integer, device :: csrRowPtr(*)
    integer, device :: csrColInd(*)
    real(8), device :: x(*)
    real(8), device :: beta
    real(8), device :: y(*)     
    end function cusparseDcsrmv_setDPM
  end interface cusparseDcsrmv

#ifdef KEEP_V2_INTERFACES
  interface cusparseDcsrmv_v2
     integer(c_int) function cusparseDcsrmv_v2(handle, trans, m, n, nnz, &
          alpha, descr, csrVal, csrRowPtr, csrColInd, x, beta, y) bind(C,name='cusparseDcsrmv')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr
       type(cusparseHandle), value :: handle
       integer(c_int), value :: trans
       integer(c_int), value :: m, n, nnz
       !dir$ ignore_tkr (d) alpha, (d) beta
       real(c_double) :: alpha
       type(cusparseMatDescr), value :: descr
       real(c_double), device :: csrVal(*)
       integer(c_int), device :: csrRowPtr(*)
       integer(c_int), device :: csrColInd(*)
       real(c_double), device :: x(*)
       real(c_double) :: beta
       real(c_double), device :: y(*) 
     end function cusparseDcsrmv_v2
  end interface cusparseDcsrmv_v2
#endif

! cusparseCcsrmv
  interface cusparseCcsrmv
    integer function cusparseCcsrmv_setHPM(handle, trans, m, n, nnz, &
       alpha, descr, csrVal, csrRowPtr, csrColInd, x, beta, y)
    import cusparseHandle, cusparseMatDescr
    type(cusparseHandle) :: handle
    integer :: trans
    integer :: m, n, nnz
    complex(4) :: alpha
    type(cusparseMatDescr) :: descr
    complex(4), device :: csrVal(*)
    integer, device :: csrRowPtr(*)
    integer, device :: csrColInd(*)
    complex(4), device :: x(*)
    complex(4) :: beta
    complex(4), device :: y(*)     
    end function cusparseCcsrmv_setHPM
    integer function cusparseCcsrmv_setDPM(handle, trans, m, n, nnz, &
       alpha, descr, csrVal, csrRowPtr, csrColInd, x, beta, y)
    import cusparseHandle, cusparseMatDescr
    type(cusparseHandle) :: handle
    integer :: trans
    integer :: m, n, nnz
    complex(4), device :: alpha
    type(cusparseMatDescr) :: descr
    complex(4), device :: csrVal(*)
    integer, device :: csrRowPtr(*)
    integer, device :: csrColInd(*)
    complex(4), device :: x(*)
    complex(4), device :: beta
    complex(4), device :: y(*)     
    end function cusparseCcsrmv_setDPM
  end interface cusparseCcsrmv

#ifdef KEEP_V2_INTERFACES
  interface cusparseCcsrmv_v2
     integer(c_int) function cusparseCcsrmv_v2(handle, trans, m, n, nnz, &
          alpha, descr, csrVal, csrRowPtr, csrColInd, x, beta, y) bind(C,name='cusparseCcsrmv')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr
       type(cusparseHandle), value :: handle
       integer(c_int), value :: trans
       integer(c_int), value :: m, n, nnz
       !dir$ ignore_tkr (d) alpha, (d) beta
       complex(4) :: alpha
       type(cusparseMatDescr), value :: descr
       complex(4), device :: csrVal(*)
       integer(c_int), device :: csrRowPtr(*)
       integer(c_int), device :: csrColInd(*)
       complex(4), device :: x(*)
       complex(4) :: beta
       complex(4), device :: y(*) 
     end function cusparseCcsrmv_v2
  end interface cusparseCcsrmv_v2
#endif

! cusparseZcsrmv
  interface cusparseZcsrmv
    integer function cusparseZcsrmv_setHPM(handle, trans, m, n, nnz, &
       alpha, descr, csrVal, csrRowPtr, csrColInd, x, beta, y)
    import cusparseHandle, cusparseMatDescr
    type(cusparseHandle) :: handle
    integer :: trans
    integer :: m, n, nnz
    complex(8) :: alpha
    type(cusparseMatDescr) :: descr
    complex(8), device :: csrVal(*)
    integer, device :: csrRowPtr(*)
    integer, device :: csrColInd(*)
    complex(8), device :: x(*)
    complex(8) :: beta
    complex(8), device :: y(*)     
    end function cusparseZcsrmv_setHPM
    integer function cusparseZcsrmv_setDPM(handle, trans, m, n, nnz, &
       alpha, descr, csrVal, csrRowPtr, csrColInd, x, beta, y)
    import cusparseHandle, cusparseMatDescr
    type(cusparseHandle) :: handle
    integer :: trans
    integer :: m, n, nnz
    complex(8), device :: alpha
    type(cusparseMatDescr) :: descr
    complex(8), device :: csrVal(*)
    integer, device :: csrRowPtr(*)
    integer, device :: csrColInd(*)
    complex(8), device :: x(*)
    complex(8), device :: beta
    complex(8), device :: y(*)     
    end function cusparseZcsrmv_setDPM
  end interface cusparseZcsrmv

#ifdef KEEP_V2_INTERFACES
  interface cusparseZcsrmv_v2
     integer(c_int) function cusparseZcsrmv_v2(handle, trans, m, n, nnz, &
          alpha, descr, csrVal, csrRowPtr, csrColInd, x, beta, y) bind(C,name='cusparseZcsrmv')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr
       type(cusparseHandle), value :: handle
       integer(c_int), value :: trans
       integer(c_int), value :: m, n, nnz
       !dir$ ignore_tkr (d) alpha, (d) beta
       complex(8) :: alpha
       type(cusparseMatDescr), value :: descr
       complex(8), device :: csrVal(*)
       integer(c_int), device :: csrRowPtr(*)
       integer(c_int), device :: csrColInd(*)
       complex(8), device :: x(*)
       complex(8) :: beta
       complex(8), device :: y(*) 
     end function cusparseZcsrmv_v2
  end interface cusparseZcsrmv_v2
#endif

! cusparseScsrsv_analysis
  interface cusparseScsrsv_analysis
     integer(c_int) function cusparseScsrsv_analysis(handle, trans, m, nnz, descr, &
          csrVal, csrRowPtr, csrColInd, info) bind(C, name='cusparseScsrsv_analysis')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr, cusparseSolveAnalysisInfo
       type(cusparseHandle), value :: handle
       integer(c_int), value :: trans
       integer(c_int), value :: m, nnz
       type(cusparseMatDescr), value :: descr
       real(c_float), device :: csrVal(*)
       integer(c_int), device :: csrRowPtr(*)
       integer(c_int), device :: csrColInd(*)
       type(cusparseSolveAnalysisInfo), value :: info 
     end function cusparseScsrsv_analysis
  end interface cusparseScsrsv_analysis

! cusparseDcsrsv_analysis
  interface cusparseDcsrsv_analysis
     integer(c_int) function cusparseDcsrsv_analysis(handle, trans, m, nnz, descr, &
          csrVal, csrRowPtr, csrColInd, info) bind(C, name='cusparseDcsrsv_analysis')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr, cusparseSolveAnalysisInfo
       type(cusparseHandle), value :: handle
       integer(c_int), value :: trans
       integer(c_int), value :: m, nnz
       type(cusparseMatDescr), value :: descr
       real(c_double), device :: csrVal(*)
       integer(c_int), device :: csrRowPtr(*)
       integer(c_int), device :: csrColInd(*)
       type(cusparseSolveAnalysisInfo), value :: info 
     end function cusparseDcsrsv_analysis
  end interface cusparseDcsrsv_analysis

! cusparseCcsrsv_analysis
  interface cusparseCcsrsv_analysis
     integer(c_int) function cusparseCcsrsv_analysis(handle, trans, m, nnz, descr, &
          csrVal, csrRowPtr, csrColInd, info) bind(C, name='cusparseCcsrsv_analysis')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr, cusparseSolveAnalysisInfo
       type(cusparseHandle), value :: handle
       integer(c_int), value :: trans
       integer(c_int), value :: m, nnz
       type(cusparseMatDescr), value :: descr
       complex(4), device :: csrVal(*)
       integer(c_int), device :: csrRowPtr(*)
       integer(c_int), device :: csrColInd(*)
       type(cusparseSolveAnalysisInfo), value :: info 
     end function cusparseCcsrsv_analysis
  end interface cusparseCcsrsv_analysis

! cusparseZcsrsv_analysis
  interface cusparseZcsrsv_analysis
     integer(c_int) function cusparseZcsrsv_analysis(handle, trans, m, nnz, descr, &
          csrVal, csrRowPtr, csrColInd, info) bind(C, name='cusparseZcsrsv_analysis')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr, cusparseSolveAnalysisInfo
       type(cusparseHandle), value :: handle
       integer(c_int), value :: trans
       integer(c_int), value :: m, nnz
       type(cusparseMatDescr), value :: descr
       complex(8), device :: csrVal(*)
       integer(c_int), device :: csrRowPtr(*)
       integer(c_int), device :: csrColInd(*)
       type(cusparseSolveAnalysisInfo), value :: info 
     end function cusparseZcsrsv_analysis
  end interface cusparseZcsrsv_analysis

! cusparseScsrsv_solve
  interface cusparseScsrsv_solve
    integer function cusparseScsrsv_solve_setHPM(handle, trans, m, alpha, descr, &
       csrVal, csrRowPtr, csrColInd, info, x, y)
    import cusparseHandle, cusparseMatDescr, cusparseSolveAnalysisInfo
    type(cusparseHandle) :: handle
    integer :: trans
    integer :: m
    real(4) :: alpha
    type(cusparseMatDescr) :: descr
    real(4), device :: csrVal(*)
    integer, device :: csrRowPtr(*)
    integer, device :: csrColInd(*)
    type(cusparseSolveAnalysisInfo) :: info 
    real(4), device :: x(*)
    real(4), device :: y(*)     
    end function cusparseScsrsv_solve_setHPM
    integer function cusparseScsrsv_solve_setDPM(handle, trans, m, alpha, descr, &
       csrVal, csrRowPtr, csrColInd, info, x, y)
    import cusparseHandle, cusparseMatDescr, cusparseSolveAnalysisInfo
    type(cusparseHandle) :: handle
    integer :: trans
    integer :: m
    real(4), device :: alpha
    type(cusparseMatDescr) :: descr
    real(4), device :: csrVal(*)
    integer, device :: csrRowPtr(*)
    integer, device :: csrColInd(*)
    type(cusparseSolveAnalysisInfo) :: info 
    real(4), device :: x(*)
    real(4), device :: y(*)     
    end function cusparseScsrsv_solve_setDPM
  end interface cusparseScsrsv_solve

#ifdef KEEP_V2_INTERFACES
  interface cusparseScsrsv_solve_v2
     integer(c_int) function cusparseScsrsv_solve_v2(handle, trans, m, alpha, descr, &
          csrVal, csrRowPtr, csrColInd, info, x, y) bind(C, name='cusparseScsrsv_solve')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr, cusparseSolveAnalysisInfo
       type(cusparseHandle), value :: handle
       integer(c_int), value :: trans
       integer(c_int), value :: m
       !dir$ ignore_tkr (d) alpha
       real(c_float) :: alpha
       type(cusparseMatDescr), value :: descr
       real(c_float), device :: csrVal(*)
       integer(c_int), device :: csrRowPtr(*)
       integer(c_int), device :: csrColInd(*)
       type(cusparseSolveAnalysisInfo), value :: info 
       real(c_float), device :: x(*), y(*)
     end function cusparseScsrsv_solve_v2
  end interface cusparseScsrsv_solve_v2
#endif
  
! cusparseDcsrsv_solve
  interface cusparseDcsrsv_solve
    integer function cusparseDcsrsv_solve_setHPM(handle, trans, m, alpha, descr, &
       csrVal, csrRowPtr, csrColInd, info, x, y)
    import cusparseHandle, cusparseMatDescr, cusparseSolveAnalysisInfo
    type(cusparseHandle) :: handle
    integer :: trans
    integer :: m
    real(8) :: alpha
    type(cusparseMatDescr) :: descr
    real(8), device :: csrVal(*)
    integer, device :: csrRowPtr(*)
    integer, device :: csrColInd(*)
    type(cusparseSolveAnalysisInfo) :: info 
    real(8), device :: x(*)
    real(8), device :: y(*)     
    end function cusparseDcsrsv_solve_setHPM
    integer function cusparseDcsrsv_solve_setDPM(handle, trans, m, alpha, descr, &
       csrVal, csrRowPtr, csrColInd, info, x, y)
    import cusparseHandle, cusparseMatDescr, cusparseSolveAnalysisInfo
    type(cusparseHandle) :: handle
    integer :: trans
    integer :: m
    real(8), device :: alpha
    type(cusparseMatDescr) :: descr
    real(8), device :: csrVal(*)
    integer, device :: csrRowPtr(*)
    integer, device :: csrColInd(*)
    type(cusparseSolveAnalysisInfo) :: info 
    real(8), device :: x(*)
    real(8), device :: y(*)     
    end function cusparseDcsrsv_solve_setDPM
  end interface cusparseDcsrsv_solve

#ifdef KEEP_V2_INTERFACES
  interface cusparseDcsrsv_solve_v2
     integer(c_int) function cusparseDcsrsv_solve_v2(handle, trans, m, alpha, descr, &
          csrVal, csrRowPtr, csrColInd, info, x, y) bind(C, name='cusparseDcsrsv_solve')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr, cusparseSolveAnalysisInfo
       type(cusparseHandle), value :: handle
       integer(c_int), value :: trans
       integer(c_int), value :: m
       !dir$ ignore_tkr(d) alpha
       real(c_double) :: alpha
       type(cusparseMatDescr), value :: descr
       real(c_double), device :: csrVal(*)
       integer(c_int), device :: csrRowPtr(*)
       integer(c_int), device :: csrColInd(*)
       type(cusparseSolveAnalysisInfo), value :: info 
       real(c_double), device :: x(*), y(*)
     end function cusparseDcsrsv_solve_v2
  end interface cusparseDcsrsv_solve_v2
#endif

! cusparseCcsrsv_solve
  interface cusparseCcsrsv_solve
    integer function cusparseCcsrsv_solve_setHPM(handle, trans, m, alpha, descr, &
       csrVal, csrRowPtr, csrColInd, info, x, y)
    import cusparseHandle, cusparseMatDescr, cusparseSolveAnalysisInfo
    type(cusparseHandle) :: handle
    integer :: trans
    integer :: m
    complex(4) :: alpha
    type(cusparseMatDescr) :: descr
    complex(4), device :: csrVal(*)
    integer, device :: csrRowPtr(*)
    integer, device :: csrColInd(*)
    type(cusparseSolveAnalysisInfo) :: info 
    complex(4), device :: x(*)
    complex(4), device :: y(*)     
    end function cusparseCcsrsv_solve_setHPM
    integer function cusparseCcsrsv_solve_setDPM(handle, trans, m, alpha, descr, &
       csrVal, csrRowPtr, csrColInd, info, x, y)
    import cusparseHandle, cusparseMatDescr, cusparseSolveAnalysisInfo
    type(cusparseHandle) :: handle
    integer :: trans
    integer :: m
    complex(4), device :: alpha
    type(cusparseMatDescr) :: descr
    complex(4), device :: csrVal(*)
    integer, device :: csrRowPtr(*)
    integer, device :: csrColInd(*)
    type(cusparseSolveAnalysisInfo) :: info 
    complex(4), device :: x(*)
    complex(4), device :: y(*)     
    end function cusparseCcsrsv_solve_setDPM
  end interface cusparseCcsrsv_solve

#ifdef KEEP_V2_INTERFACES
  interface cusparseCcsrsv_solve_v2
     integer(c_int) function cusparseCcsrsv_solve_v2(handle, trans, m, alpha, descr, &
          csrVal, csrRowPtr, csrColInd, info, x, y) bind(C, name='cusparseCcsrsv_solve')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr, cusparseSolveAnalysisInfo
       type(cusparseHandle), value :: handle
       integer(c_int), value :: trans
       integer(c_int), value :: m
       !dir$ ignore_tkr(d) alpha
       complex(4) :: alpha
       type(cusparseMatDescr), value :: descr
       complex(4), device :: csrVal(*)
       integer(c_int), device :: csrRowPtr(*)
       integer(c_int), device :: csrColInd(*)
       type(cusparseSolveAnalysisInfo), value :: info 
       complex(4), device :: x(*), y(*)
     end function cusparseCcsrsv_solve_v2
  end interface cusparseCcsrsv_solve_v2
#endif

! cusparseZcsrsv_solve
  interface cusparseZcsrsv_solve
    integer function cusparseZcsrsv_solve_setHPM(handle, trans, m, alpha, descr, &
       csrVal, csrRowPtr, csrColInd, info, x, y)
    import cusparseHandle, cusparseMatDescr, cusparseSolveAnalysisInfo
    type(cusparseHandle) :: handle
    integer :: trans
    integer :: m
    complex(8) :: alpha
    type(cusparseMatDescr) :: descr
    complex(8), device :: csrVal(*)
    integer, device :: csrRowPtr(*)
    integer, device :: csrColInd(*)
    type(cusparseSolveAnalysisInfo) :: info 
    complex(8), device :: x(*)
    complex(8), device :: y(*)     
    end function cusparseZcsrsv_solve_setHPM
    integer function cusparseZcsrsv_solve_setDPM(handle, trans, m, alpha, descr, &
       csrVal, csrRowPtr, csrColInd, info, x, y)
    import cusparseHandle, cusparseMatDescr, cusparseSolveAnalysisInfo
    type(cusparseHandle) :: handle
    integer :: trans
    integer :: m
    complex(8), device :: alpha
    type(cusparseMatDescr) :: descr
    complex(8), device :: csrVal(*)
    integer, device :: csrRowPtr(*)
    integer, device :: csrColInd(*)
    type(cusparseSolveAnalysisInfo) :: info 
    complex(8), device :: x(*)
    complex(8), device :: y(*)     
    end function cusparseZcsrsv_solve_setDPM
  end interface cusparseZcsrsv_solve

#ifdef KEEP_V2_INTERFACES
  interface cusparseZcsrsv_solve_v2
     integer(c_int) function cusparseZcsrsv_solve_v2(handle, trans, m, alpha, descr, &
          csrVal, csrRowPtr, csrColInd, info, x, y) bind(C, name='cusparseZcsrsv_solve')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr, cusparseSolveAnalysisInfo
       type(cusparseHandle), value :: handle
       integer(c_int), value :: trans
       integer(c_int), value :: m
       !dir$ ignore_tkr(d) alpha
       complex(8) :: alpha
       type(cusparseMatDescr), value :: descr
       complex(8), device :: csrVal(*)
       integer(c_int), device :: csrRowPtr(*)
       integer(c_int), device :: csrColInd(*)
       type(cusparseSolveAnalysisInfo), value :: info 
       complex(8), device :: x(*), y(*)
     end function cusparseZcsrsv_solve_v2
  end interface cusparseZcsrsv_solve_v2
#endif

! cusparseShybmv
  interface cusparseShybmv
    integer function cusparseShybmv_setHPM(handle, trans, alpha, descr, hyb, x, beta, y)
    import cusparseHandle, cusparseMatDescr, cusparseHybMat
    type(cusparseHandle) :: handle
    integer :: trans
    real(4) :: alpha
    type(cusparseMatDescr) :: descr
    type(cusparseHybMat) :: hyb 
    real(4), device :: x(*)
    real(4) :: beta
    real(4), device :: y(*)
    end function cusparseShybmv_setHPM
    integer function cusparseShybmv_setDPM(handle, trans, alpha, descr, hyb, x, beta, y)
    import cusparseHandle, cusparseMatDescr, cusparseHybMat
    type(cusparseHandle) :: handle
    integer :: trans
    real(4), device :: alpha
    type(cusparseMatDescr) :: descr
    type(cusparseHybMat) :: hyb 
    real(4), device :: x(*)
    real(4), device :: beta
    real(4), device :: y(*)
    end function cusparseShybmv_setDPM
  end interface cusparseShybmv

#ifdef KEEP_V2_INTERFACES
  interface cusparseShybmv_v2
     integer(c_int) function cusparseShybmv_v2(handle, trans, alpha, descr, hyb, x, beta, y) &
          bind(C, name='cusparseShybmv')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr, cusparseHybMat
       type(cusparseHandle), value :: handle
       integer(c_int), value :: trans
       !dir$ ignore_tkr(d) alpha, (d) beta
       real(c_float) :: alpha
       type(cusparseMatDescr), value :: descr
       type(cusparseHybMat), value :: hyb 
       real(c_float), device :: x(*)
       real(c_float) :: beta
       real(c_float), device :: y(*)
     end function cusparseShybmv_v2
  end interface cusparseShybmv_v2
#endif

! cusparseDhybmv
  interface cusparseDhybmv
    integer function cusparseDhybmv_setHPM(handle, trans, alpha, descr, hyb, x, beta, y)
    import cusparseHandle, cusparseMatDescr, cusparseHybMat
    type(cusparseHandle) :: handle
    integer :: trans
    real(8) :: alpha
    type(cusparseMatDescr) :: descr
    type(cusparseHybMat) :: hyb 
    real(8), device :: x(*)
    real(8) :: beta
    real(8), device :: y(*)
    end function cusparseDhybmv_setHPM
    integer function cusparseDhybmv_setDPM(handle, trans, alpha, descr, hyb, x, beta, y)
    import cusparseHandle, cusparseMatDescr, cusparseHybMat
    type(cusparseHandle) :: handle
    integer :: trans
    real(8), device :: alpha
    type(cusparseMatDescr) :: descr
    type(cusparseHybMat) :: hyb 
    real(8), device :: x(*)
    real(8), device :: beta
    real(8), device :: y(*)
    end function cusparseDhybmv_setDPM
  end interface cusparseDhybmv

#ifdef KEEP_V2_INTERFACES
  interface cusparseDhybmv_v2
     integer(c_int) function cusparseDhybmv_v2(handle, trans, alpha, descr, hyb, x, beta, y) &          
          bind(C, name='cusparseDhybmv')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr, cusparseHybMat
       type(cusparseHandle), value :: handle
       integer(c_int), value :: trans
       !dir$ ignore_tkr(d) alpha, (d) beta
       real(c_double) :: alpha
       type(cusparseMatDescr), value :: descr
       type(cusparseHybMat), value :: hyb 
       real(c_double), device :: x(*)
       real(c_double) :: beta
       real(c_double), device :: y(*)
     end function cusparseDhybmv_v2
  end interface cusparseDhybmv_v2
#endif

! cusparseChybmv
  interface cusparseChybmv
    integer function cusparseChybmv_setHPM(handle, trans, alpha, descr, hyb, x, beta, y)
    import cusparseHandle, cusparseMatDescr, cusparseHybMat
    type(cusparseHandle) :: handle
    integer :: trans
    complex(4) :: alpha
    type(cusparseMatDescr) :: descr
    type(cusparseHybMat) :: hyb 
    complex(4), device :: x(*)
    complex(4) :: beta
    complex(4), device :: y(*)
    end function cusparseChybmv_setHPM
    integer function cusparseChybmv_setDPM(handle, trans, alpha, descr, hyb, x, beta, y)
    import cusparseHandle, cusparseMatDescr, cusparseHybMat
    type(cusparseHandle) :: handle
    integer :: trans
    complex(4), device :: alpha
    type(cusparseMatDescr) :: descr
    type(cusparseHybMat) :: hyb 
    complex(4), device :: x(*)
    complex(4), device :: beta
    complex(4), device :: y(*)
    end function cusparseChybmv_setDPM
  end interface cusparseChybmv

#ifdef KEEP_V2_INTERFACES
  interface cusparseChybmv_v2
     integer(c_int) function cusparseChybmv_v2(handle, trans, alpha, descr, hyb, x, beta, y) &
          bind(C, name='cusparseChybmv')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr, cusparseHybMat
       type(cusparseHandle), value :: handle
       integer(c_int), value :: trans
       !dir$ ignore_tkr(d) alpha, (d)beta
       complex(4) :: alpha
       type(cusparseMatDescr), value :: descr
       type(cusparseHybMat), value :: hyb 
       complex(4), device :: x(*)
       complex(4) :: beta
       complex(4), device :: y(*)
     end function cusparseChybmv_v2
  end interface cusparseChybmv_v2
#endif

! cusparseZhybmv
  interface cusparseZhybmv
    integer function cusparseZhybmv_setHPM(handle, trans, alpha, descr, hyb, x, beta, y)
    import cusparseHandle, cusparseMatDescr, cusparseHybMat
    type(cusparseHandle) :: handle
    integer :: trans
    complex(8) :: alpha
    type(cusparseMatDescr) :: descr
    type(cusparseHybMat) :: hyb 
    complex(8), device :: x(*)
    complex(8) :: beta
    complex(8), device :: y(*)
    end function cusparseZhybmv_setHPM
    integer function cusparseZhybmv_setDPM(handle, trans, alpha, descr, hyb, x, beta, y)
    import cusparseHandle, cusparseMatDescr, cusparseHybMat
    type(cusparseHandle) :: handle
    integer :: trans
    complex(8), device :: alpha
    type(cusparseMatDescr) :: descr
    type(cusparseHybMat) :: hyb 
    complex(8), device :: x(*)
    complex(8), device :: beta
    complex(8), device :: y(*)
    end function cusparseZhybmv_setDPM
  end interface cusparseZhybmv

#ifdef KEEP_V2_INTERFACES
  interface cusparseZhybmv_v2
     integer(c_int) function cusparseZhybmv_v2(handle, trans, alpha, descr, hyb, x, beta, y) &
          bind(C, name='cusparseZhybmv')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr, cusparseHybMat
       type(cusparseHandle), value :: handle
       integer(c_int), value :: trans
       !dir$ ignore_tkr(d) alpha, (d) beta
       complex(8) :: alpha
       type(cusparseMatDescr), value :: descr
       type(cusparseHybMat), value :: hyb 
       complex(8), device :: x(*)
       complex(8) :: beta
       complex(8), device :: y(*)
     end function cusparseZhybmv_v2
  end interface cusparseZhybmv_v2
#endif

! cusparseShybsv_analysis
  interface cusparseShybsv_analysis
     integer(c_int) function cusparseShybsv_analysis(handle, trans, descr, hyb, info) &
          bind(C, name='cusparseShybsv_analysis')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr, cusparseHybMat, cusparseSolveAnalysisInfo
       type(cusparseHandle), value :: handle
       integer(c_int), value :: trans
       type(cusparseMatDescr), value :: descr
       type(cusparseHybMat), value :: hyb 
       type(cusparseSolveAnalysisInfo), value :: info
     end function cusparseShybsv_analysis
  end interface cusparseShybsv_analysis

! cusparseDhybsv_analysis
  interface cusparseDhybsv_analysis
     integer(c_int) function cusparseDhybsv_analysis(handle, trans, descr, hyb, info) &
          bind(C, name='cusparseDhybsv_analysis')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr, cusparseHybMat, cusparseSolveAnalysisInfo
       type(cusparseHandle), value :: handle
       integer(c_int), value :: trans
       type(cusparseMatDescr), value :: descr
       type(cusparseHybMat), value :: hyb 
       type(cusparseSolveAnalysisInfo), value :: info
     end function cusparseDhybsv_analysis
  end interface cusparseDhybsv_analysis

! cusparseChybsv_analysis
  interface cusparseChybsv_analysis
     integer(c_int) function cusparseChybsv_analysis(handle, trans, descr, hyb, info) &
          bind(C, name='cusparseChybsv_analysis')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr, cusparseHybMat, cusparseSolveAnalysisInfo
       type(cusparseHandle), value :: handle
       integer(c_int), value :: trans
       type(cusparseMatDescr), value :: descr
       type(cusparseHybMat), value :: hyb 
       type(cusparseSolveAnalysisInfo), value :: info
     end function cusparseChybsv_analysis
  end interface cusparseChybsv_analysis

! cusparseZhybsv_analysis
  interface cusparseZhybsv_analysis
     integer(c_int) function cusparseZhybsv_analysis(handle, trans, descr, hyb, info) &
          bind(C, name='cusparseZhybsv_analysis')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr, cusparseHybMat, cusparseSolveAnalysisInfo
       type(cusparseHandle), value :: handle
       integer(c_int), value :: trans
       type(cusparseMatDescr), value :: descr
       type(cusparseHybMat), value :: hyb 
       type(cusparseSolveAnalysisInfo), value :: info
     end function cusparseZhybsv_analysis
  end interface cusparseZhybsv_analysis

! cusparseShybsv_solve
  interface cusparseShybsv_solve
    integer function cusparseShybsv_solve_setHPM(handle, trans, alpha, descr, hyb, info, x, y)
    import cusparseHandle, cusparseMatDescr, cusparseHybMat, cusparseSolveAnalysisInfo
    type(cusparseHandle) :: handle
    integer :: trans
    real(4) :: alpha
    type(cusparseMatDescr) :: descr
    type(cusparseHybMat) :: hyb 
    type(cusparseSolveAnalysisInfo) :: info
    real(4), device :: x(*)
    real(4), device :: y(*)
    end function cusparseShybsv_solve_setHPM
    integer function cusparseShybsv_solve_setDPM(handle, trans, alpha, descr, hyb, info, x, y)
    import cusparseHandle, cusparseMatDescr, cusparseHybMat, cusparseSolveAnalysisInfo
    type(cusparseHandle) :: handle
    integer :: trans
    real(4), device :: alpha
    type(cusparseMatDescr) :: descr
    type(cusparseHybMat) :: hyb 
    type(cusparseSolveAnalysisInfo) :: info
    real(4), device :: x(*)
    real(4), device :: y(*)
    end function cusparseShybsv_solve_setDPM
  end interface cusparseShybsv_solve

#ifdef KEEP_V2_INTERFACES
  interface cusparseShybsv_solve_v2
     integer(c_int) function cusparseShybsv_solve_v2(handle, trans, alpha, descr, hyb, info, x, y) &
          bind(C, name='cusparseShybsv_solve')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr, cusparseHybMat, cusparseSolveAnalysisInfo
       type(cusparseHandle), value :: handle
       integer(c_int), value :: trans
       !dir$ ignore_tkr(d) alpha
       real(c_float) :: alpha
       type(cusparseMatDescr), value :: descr
       type(cusparseHybMat), value :: hyb 
       type(cusparseSolveAnalysisInfo), value :: info
       real(c_float), device :: x(*)
       real(c_float), device :: y(*)
     end function cusparseShybsv_solve_v2
  end interface cusparseShybsv_solve_v2
#endif

! cusparseDhybsv_solve
  interface cusparseDhybsv_solve
    integer function cusparseDhybsv_solve_setHPM(handle, trans, alpha, descr, hyb, info, x, y)
    import cusparseHandle, cusparseMatDescr, cusparseHybMat, cusparseSolveAnalysisInfo
    type(cusparseHandle) :: handle
    integer :: trans
    real(8) :: alpha
    type(cusparseMatDescr) :: descr
    type(cusparseHybMat) :: hyb 
    type(cusparseSolveAnalysisInfo) :: info
    real(8), device :: x(*)
    real(8), device :: y(*)
    end function cusparseDhybsv_solve_setHPM
    integer function cusparseDhybsv_solve_setDPM(handle, trans, alpha, descr, hyb, info, x, y)
    import cusparseHandle, cusparseMatDescr, cusparseHybMat, cusparseSolveAnalysisInfo
    type(cusparseHandle) :: handle
    integer :: trans
    real(8), device :: alpha
    type(cusparseMatDescr) :: descr
    type(cusparseHybMat) :: hyb 
    type(cusparseSolveAnalysisInfo) :: info
    real(8), device :: x(*)
    real(8), device :: y(*)
    end function cusparseDhybsv_solve_setDPM
  end interface cusparseDhybsv_solve

#ifdef KEEP_V2_INTERFACES
  interface cusparseDhybsv_solve_v2
     integer(c_int) function cusparseDhybsv_solve_v2(handle, trans, alpha, descr, hyb, info, x, y) &
          bind(C, name='cusparseDhybsv_solve')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr, cusparseHybMat, cusparseSolveAnalysisInfo
       type(cusparseHandle), value :: handle
       integer(c_int), value :: trans
       !dir$ ignore_tkr(d) alpha
       real(c_double) :: alpha
       type(cusparseMatDescr), value :: descr
       type(cusparseHybMat), value :: hyb 
       type(cusparseSolveAnalysisInfo), value :: info
       real(c_double), device :: x(*)
       real(c_double), device :: y(*)
     end function cusparseDhybsv_solve_v2
  end interface cusparseDhybsv_solve_v2
#endif

! cusparseChybsv_solve
  interface cusparseChybsv_solve
    integer function cusparseChybsv_solve_setHPM(handle, trans, alpha, descr, hyb, info, x, y)
    import cusparseHandle, cusparseMatDescr, cusparseHybMat, cusparseSolveAnalysisInfo
    type(cusparseHandle) :: handle
    integer :: trans
    complex(4) :: alpha
    type(cusparseMatDescr) :: descr
    type(cusparseHybMat) :: hyb 
    type(cusparseSolveAnalysisInfo) :: info
    complex(4), device :: x(*)
    complex(4), device :: y(*)
    end function cusparseChybsv_solve_setHPM
    integer function cusparseChybsv_solve_setDPM(handle, trans, alpha, descr, hyb, info, x, y)
    import cusparseHandle, cusparseMatDescr, cusparseHybMat, cusparseSolveAnalysisInfo
    type(cusparseHandle) :: handle
    integer :: trans
    complex(4), device :: alpha
    type(cusparseMatDescr) :: descr
    type(cusparseHybMat) :: hyb 
    type(cusparseSolveAnalysisInfo) :: info
    complex(4), device :: x(*)
    complex(4), device :: y(*)
    end function cusparseChybsv_solve_setDPM
  end interface cusparseChybsv_solve

#ifdef KEEP_V2_INTERFACES
  interface cusparseChybsv_solve_v2
     integer(c_int) function cusparseChybsv_solve_v2(handle, trans, alpha, descr, hyb, info, x, y) &
          bind(C, name='cusparseChybsv_solve')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr, cusparseHybMat, cusparseSolveAnalysisInfo
       type(cusparseHandle), value :: handle
       integer(c_int), value :: trans
       !dir$ ignore_tkr(d) alpha
       complex(4) :: alpha
       type(cusparseMatDescr), value :: descr
       type(cusparseHybMat), value :: hyb 
       type(cusparseSolveAnalysisInfo), value :: info
       complex(4), device :: x(*)
       complex(4), device :: y(*)
     end function cusparseChybsv_solve_v2
  end interface cusparseChybsv_solve_v2
#endif

! cusparseZhybsv_solve
  interface cusparseZhybsv_solve
    integer function cusparseZhybsv_solve_setHPM(handle, trans, alpha, descr, hyb, info, x, y)
    import cusparseHandle, cusparseMatDescr, cusparseHybMat, cusparseSolveAnalysisInfo
    type(cusparseHandle) :: handle
    integer :: trans
    complex(8) :: alpha
    type(cusparseMatDescr) :: descr
    type(cusparseHybMat) :: hyb 
    type(cusparseSolveAnalysisInfo) :: info
    complex(8), device :: x(*)
    complex(8), device :: y(*)
    end function cusparseZhybsv_solve_setHPM
    integer function cusparseZhybsv_solve_setDPM(handle, trans, alpha, descr, hyb, info, x, y)
    import cusparseHandle, cusparseMatDescr, cusparseHybMat, cusparseSolveAnalysisInfo
    type(cusparseHandle) :: handle
    integer :: trans
    complex(8), device :: alpha
    type(cusparseMatDescr) :: descr
    type(cusparseHybMat) :: hyb 
    type(cusparseSolveAnalysisInfo) :: info
    complex(8), device :: x(*)
    complex(8), device :: y(*)
    end function cusparseZhybsv_solve_setDPM
  end interface cusparseZhybsv_solve

#ifdef KEEP_V2_INTERFACES
  interface cusparseZhybsv_solve_v2
     integer(c_int) function cusparseZhybsv_solve_v2(handle, trans, alpha, descr, hyb, info, x, y) &
          bind(C, name='cusparseZhybsv_solve')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr, cusparseHybMat, cusparseSolveAnalysisInfo
       type(cusparseHandle), value :: handle
       integer(c_int), value :: trans
       !dir$ ignore_tkr(d) alpha
       complex(8) :: alpha
       type(cusparseMatDescr), value :: descr
       type(cusparseHybMat), value :: hyb 
       type(cusparseSolveAnalysisInfo), value :: info
       complex(8), device :: x(*)
       complex(8), device :: y(*)
     end function cusparseZhybsv_solve_v2
  end interface cusparseZhybsv_solve_v2
#endif

  ! ---------------------
  ! 6.0 Level 2 functions
  ! ---------------------

! cusparseSbsrsv2_bufferSize
  interface cusparseSbsrsv2_bufferSize
     integer(c_int) function cusparseSbsrsv2_bufferSize(handle, dirA, transA, mb, nnzb, &
          descrA, bsrValA, bsrRowPtrA, bsrColIndA, blockDim, info, pBufferSize)  bind(C,name='cusparseSbsrsv2_bufferSize')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr, cusparseBsrsv2Info
       type(cusparseHandle), value :: handle
       integer(c_int), value :: dirA, transA
       integer(c_int), value :: mb, nnzb
       type(cusparseMatDescr), value :: descrA
       real(c_float), device :: bsrValA(*)
       integer(c_int), device :: bsrRowPtrA(*), bsrColIndA(*)
       integer(c_int), value :: blockDim
       type(cusparseBsrsv2Info), value :: info
       integer(4) :: pBufferSize
     end function cusparseSbsrsv2_bufferSize
     integer(c_int) function cusparseSbsrsv2_bufferSizeExt(handle, dirA, transA, mb, nnzb, &
          descrA, bsrValA, bsrRowPtrA, bsrColIndA, blockDim, info, pBufferSize)  bind(C,name='cusparseSbsrsv2_bufferSizeExt')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr, cusparseBsrsv2Info
       type(cusparseHandle), value :: handle
       integer(c_int), value :: dirA, transA
       integer(c_int), value :: mb, nnzb
       type(cusparseMatDescr), value :: descrA
       real(c_float), device :: bsrValA(*)
       integer(c_int), device :: bsrRowPtrA(*), bsrColIndA(*)
       integer(c_int), value :: blockDim
       type(cusparseBsrsv2Info), value :: info
       integer(8) :: pBufferSize
     end function cusparseSbsrsv2_bufferSizeExt
  end interface

! cusparseDbsrsv2_bufferSize
  interface cusparseDbsrsv2_bufferSize
     integer(c_int) function cusparseDbsrsv2_bufferSize(handle, dirA, transA, mb, nnzb, &
          descrA, bsrValA, bsrRowPtrA, bsrColIndA, blockDim, info, pBufferSize) bind(C,name='cusparseDbsrsv2_bufferSize')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr, cusparseBsrsv2Info
       type(cusparseHandle), value :: handle
       integer(c_int), value :: dirA, transA
       integer(c_int), value :: mb, nnzb
       type(cusparseMatDescr), value :: descrA
       real(c_double), device :: bsrValA(*)
       integer(c_int), device :: bsrRowPtrA(*), bsrColIndA(*)
       integer(c_int), value :: blockDim
       type(cusparseBsrsv2Info), value :: info
       integer(4) :: pBufferSize
     end function cusparseDbsrsv2_bufferSize
     integer(c_int) function cusparseDbsrsv2_bufferSizeExt(handle, dirA, transA, mb, nnzb, &
          descrA, bsrValA, bsrRowPtrA, bsrColIndA, blockDim, info, pBufferSize) bind(C,name='cusparseDbsrsv2_bufferSizeExt')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr, cusparseBsrsv2Info
       type(cusparseHandle), value :: handle
       integer(c_int), value :: dirA, transA
       integer(c_int), value :: mb, nnzb
       type(cusparseMatDescr), value :: descrA
       real(c_double), device :: bsrValA(*)
       integer(c_int), device :: bsrRowPtrA(*), bsrColIndA(*)
       integer(c_int), value :: blockDim
       type(cusparseBsrsv2Info), value :: info
       integer(8) :: pBufferSize
     end function cusparseDbsrsv2_bufferSizeExt
  end interface

! cusparseCbsrsv2_bufferSize
  interface cusparseCbsrsv2_bufferSize
     integer(c_int) function cusparseCbsrsv2_bufferSize(handle, dirA, transA, mb, nnzb, &
          descrA, bsrValA, bsrRowPtrA, bsrColIndA, blockDim, info, pBufferSize) bind(C,name='cusparseCbsrsv2_bufferSize')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr, cusparseBsrsv2Info
       type(cusparseHandle), value :: handle
       integer(c_int), value :: dirA, transA
       integer(c_int), value :: mb, nnzb
       type(cusparseMatDescr), value :: descrA
       complex(4), device :: bsrValA(*)
       integer(c_int), device :: bsrRowPtrA(*), bsrColIndA(*)
       integer(c_int), value :: blockDim
       type(cusparseBsrsv2Info), value :: info
       integer(4) :: pBufferSize
     end function cusparseCbsrsv2_bufferSize
     integer(c_int) function cusparseCbsrsv2_bufferSizeExt(handle, dirA, transA, mb, nnzb, &
          descrA, bsrValA, bsrRowPtrA, bsrColIndA, blockDim, info, pBufferSize) bind(C,name='cusparseCbsrsv2_bufferSizeExt')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr, cusparseBsrsv2Info
       type(cusparseHandle), value :: handle
       integer(c_int), value :: dirA, transA
       integer(c_int), value :: mb, nnzb
       type(cusparseMatDescr), value :: descrA
       complex(4), device :: bsrValA(*)
       integer(c_int), device :: bsrRowPtrA(*), bsrColIndA(*)
       integer(c_int), value :: blockDim
       type(cusparseBsrsv2Info), value :: info
       integer(8) :: pBufferSize
     end function cusparseCbsrsv2_bufferSizeExt
  end interface

! cusparseZbsrsv2_bufferSize
  interface cusparseZbsrsv2_bufferSize
     integer(c_int) function cusparseZbsrsv2_bufferSize(handle, dirA, transA, mb, nnzb, &
          descrA, bsrValA, bsrRowPtrA, bsrColIndA, blockDim, info, pBufferSize) bind(C,name='cusparseZbsrsv2_bufferSize')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr, cusparseBsrsv2Info
       type(cusparseHandle), value :: handle
       integer(c_int), value :: dirA, transA
       integer(c_int), value :: mb, nnzb
       type(cusparseMatDescr), value :: descrA
       complex(8), device :: bsrValA(*)
       integer(c_int), device :: bsrRowPtrA(*), bsrColIndA(*)
       integer(c_int), value :: blockDim
       type(cusparseBsrsv2Info), value :: info
       integer(4) :: pBufferSize
     end function cusparseZbsrsv2_bufferSize
     integer(c_int) function cusparseZbsrsv2_bufferSizeExt(handle, dirA, transA, mb, nnzb, &
          descrA, bsrValA, bsrRowPtrA, bsrColIndA, blockDim, info, pBufferSize) bind(C,name='cusparseZbsrsv2_bufferSizeExt')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr, cusparseBsrsv2Info
       type(cusparseHandle), value :: handle
       integer(c_int), value :: dirA, transA
       integer(c_int), value :: mb, nnzb
       type(cusparseMatDescr), value :: descrA
       complex(8), device :: bsrValA(*)
       integer(c_int), device :: bsrRowPtrA(*), bsrColIndA(*)
       integer(c_int), value :: blockDim
       type(cusparseBsrsv2Info), value :: info
       integer(8) :: pBufferSize
     end function cusparseZbsrsv2_bufferSizeExt
  end interface

! cusparseSbsrsv2_analysis
  interface
     integer(c_int) function cusparseSbsrsv2_analysis(handle, dirA, transA, mb, nnzb, &
          descrA, bsrValA, bsrRowPtrA, bsrColIndA, blockDim, info, policy, pBuffer) bind(C,name='cusparseSbsrsv2_analysis')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr, cusparseBsrsv2Info
       type(cusparseHandle), value :: handle
       integer(c_int), value :: dirA, transA
       integer(c_int), value :: mb, nnzb
       type(cusparseMatDescr), value :: descrA
       real(c_float), device :: bsrValA(*)
       integer(c_int), device :: bsrRowPtrA(*), bsrColIndA(*)
       integer(c_int), value :: blockDim
       type(cusparseBsrsv2Info), value :: info
       integer(c_int), value :: policy
       !dir$ ignore_tkr pBuffer
       character(c_char), device :: pBuffer(*)
     end function cusparseSbsrsv2_analysis
  end interface

! cusparseDbsrsv2_analysis
  interface
     integer(c_int) function cusparseDbsrsv2_analysis(handle, dirA, transA, mb, nnzb, &
          descrA, bsrValA, bsrRowPtrA, bsrColIndA, blockDim, info, policy, pBuffer) bind(C,name='cusparseDbsrsv2_analysis')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr, cusparseBsrsv2Info
       type(cusparseHandle), value :: handle
       integer(c_int), value :: dirA, transA
       integer(c_int), value :: mb, nnzb
       type(cusparseMatDescr), value :: descrA
       real(c_double), device :: bsrValA(*)
       integer(c_int), device :: bsrRowPtrA(*), bsrColIndA(*)
       integer(c_int), value :: blockDim
       type(cusparseBsrsv2Info), value :: info
       integer(c_int), value :: policy
       !dir$ ignore_tkr pBuffer
       character(c_char), device :: pBuffer(*)
     end function cusparseDbsrsv2_analysis
  end interface

! cusparseCbsrsv2_analysis
  interface
     integer(c_int) function cusparseCbsrsv2_analysis(handle, dirA, transA, mb, nnzb, &
          descrA, bsrValA, bsrRowPtrA, bsrColIndA, blockDim, info, policy, pBuffer) bind(C,name='cusparseCbsrsv2_analysis')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr, cusparseBsrsv2Info
       type(cusparseHandle), value :: handle
       integer(c_int), value :: dirA, transA
       integer(c_int), value :: mb, nnzb
       type(cusparseMatDescr), value :: descrA
       complex(4), device :: bsrValA(*)
       integer(c_int), device :: bsrRowPtrA(*), bsrColIndA(*)
       integer(c_int), value :: blockDim
       type(cusparseBsrsv2Info), value :: info
       integer(c_int), value :: policy
       !dir$ ignore_tkr pBuffer
       character(c_char), device :: pBuffer(*)
     end function cusparseCbsrsv2_analysis
  end interface

! cusparseZbsrsv2_analysis
  interface
     integer(c_int) function cusparseZbsrsv2_analysis(handle, dirA, transA, mb, nnzb, &
          descrA, bsrValA, bsrRowPtrA, bsrColIndA, blockDim, info, policy, pBuffer) bind(C,name='cusparseZbsrsv2_analysis')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr, cusparseBsrsv2Info
       type(cusparseHandle), value :: handle
       integer(c_int), value :: dirA, transA
       integer(c_int), value :: mb, nnzb
       type(cusparseMatDescr), value :: descrA
       complex(8), device :: bsrValA(*)
       integer(c_int), device :: bsrRowPtrA(*), bsrColIndA(*)
       integer(c_int), value :: blockDim
       type(cusparseBsrsv2Info), value :: info
       integer(c_int), value :: policy
       !dir$ ignore_tkr pBuffer
       character(c_char), device :: pBuffer(*)
     end function cusparseZbsrsv2_analysis
  end interface

! cusparseSbsrsv2_solve
  interface cusparseSbsrsv2_solve
     integer function cusparseSbsrsv2_solve_setHPM(handle, dirA, transA, mb, nnzb, alpha, &
       descrA, bsrValA, bsrRowPtrA, bsrColIndA, blockDim, info, x, y, policy, pBuffer)
       import cusparseHandle, cusparseMatDescr, cusparseBsrsv2Info
       type(cusparseHandle) :: handle
       integer :: dirA, transA, mb, nnzb
       real(4) :: alpha 
       type(cusparseMatDescr) :: descrA
       real(4), device :: bsrValA(*)
       integer, device :: bsrRowPtrA(*), bsrColIndA(*)
       integer :: blockDim
       type(cusparseBsrsv2Info) :: info
       real(4), device :: x(*), y(*)
       integer :: policy
       !dir$ ignore_tkr pBuffer
       character, device :: pBuffer(*)
     end function cusparseSbsrsv2_solve_setHPM
  
     integer function cusparseSbsrsv2_solve_setDPM(handle, dirA, transA, mb, nnzb, alpha, &
       descrA, bsrValA, bsrRowPtrA, bsrColIndA, blockDim, info, x, y, policy, pBuffer)
       import cusparseHandle, cusparseMatDescr, cusparseBsrsv2Info
       type(cusparseHandle) :: handle
       integer :: dirA, transA, mb, nnzb
       real(4), device :: alpha 
       type(cusparseMatDescr) :: descrA
       real(4), device :: bsrValA(*)
       integer, device :: bsrRowPtrA(*), bsrColIndA(*)
       integer :: blockDim
       type(cusparseBsrsv2Info) :: info
       real(4), device :: x(*), y(*)
       integer :: policy
       !dir$ ignore_tkr pBuffer
       character, device :: pBuffer(*)
     end function cusparseSbsrsv2_solve_setDPM
  end interface cusparseSbsrsv2_solve

#ifdef KEEP_V2_INTERFACES
  interface
     integer(c_int) function cusparseSbsrsv2_solve_v2(handle, dirA, transA, mb, nnzb, alpha, &
          descrA, bsrValA, bsrRowPtrA, bsrColIndA, blockDim, info, x, y, policy, pBuffer) bind(C,name='cusparseSbsrsv2_solve')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr, cusparseBsrsv2Info
       type(cusparseHandle), value :: handle
       integer(c_int), value :: dirA, transA
       integer(c_int), value :: mb, nnzb
       !dir$ ignore_tkr (d) alpha
       real(c_float) :: alpha 
       type(cusparseMatDescr), value :: descrA
       real(c_float), device :: bsrValA(*)
       integer(c_int), device :: bsrRowPtrA(*), bsrColIndA(*)
       integer(c_int), value :: blockDim
       type(cusparseBsrsv2Info), value :: info
       real(c_float), device :: x(*), y(*)
       integer(c_int), value :: policy
       !dir$ ignore_tkr pBuffer
       character(c_char), device :: pBuffer(*)
     end function cusparseSbsrsv2_solve_v2
  end interface
#endif

! cusparseDbsrsv2_solve
  interface cusparseDbsrsv2_solve
     integer(c_int) function cusparseDbsrsv2_solve_setHPM(handle, dirA, transA, mb, nnzb, alpha, &
       descrA, bsrValA, bsrRowPtrA, bsrColIndA, blockDim, info, x, y, policy, pBuffer)
       import cusparseHandle, cusparseMatDescr, cusparseBsrsv2Info
       type(cusparseHandle) :: handle
       integer :: dirA, transA, mb, nnzb
       real(8) :: alpha 
       type(cusparseMatDescr) :: descrA
       real(8), device :: bsrValA(*)
       integer, device :: bsrRowPtrA(*), bsrColIndA(*)
       integer :: blockDim
       type(cusparseBsrsv2Info) :: info
       real(8), device :: x(*), y(*)
       integer :: policy
       !dir$ ignore_tkr pBuffer
       character, device :: pBuffer(*)
     end function cusparseDbsrsv2_solve_setHPM
  
     integer(c_int) function cusparseDbsrsv2_solve_setDPM(handle, dirA, transA, mb, nnzb, alpha, &
       descrA, bsrValA, bsrRowPtrA, bsrColIndA, blockDim, info, x, y, policy, pBuffer)
       import cusparseHandle, cusparseMatDescr, cusparseBsrsv2Info
       type(cusparseHandle) :: handle
       integer :: dirA, transA, mb, nnzb
       real(8), device :: alpha 
       type(cusparseMatDescr) :: descrA
       real(8), device :: bsrValA(*)
       integer, device :: bsrRowPtrA(*), bsrColIndA(*)
       integer :: blockDim
       type(cusparseBsrsv2Info) :: info
       real(8), device :: x(*), y(*)
       integer :: policy
       !dir$ ignore_tkr pBuffer
       character, device :: pBuffer(*)
     end function cusparseDbsrsv2_solve_setDPM
  end interface cusparseDbsrsv2_solve

#ifdef KEEP_V2_INTERFACES
  interface
     integer(c_int) function cusparseDbsrsv2_solve_v2(handle, dirA, transA, mb, nnzb, alpha, &
          descrA, bsrValA, bsrRowPtrA, bsrColIndA, blockDim, info, x, y, policy, pBuffer) bind(C,name='cusparseDbsrsv2_solve')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr, cusparseBsrsv2Info
       type(cusparseHandle), value :: handle
       integer(c_int), value :: dirA, transA
       integer(c_int), value :: mb, nnzb
       !dir$ ignore_tkr (d) alpha
       real(c_double) :: alpha 
       type(cusparseMatDescr), value :: descrA
       real(c_double), device :: bsrValA(*)
       integer(c_int), device :: bsrRowPtrA(*), bsrColIndA(*)
       integer(c_int), value :: blockDim
       type(cusparseBsrsv2Info), value :: info
       real(c_double), device :: x(*), y(*)
       integer(c_int), value :: policy
       !dir$ ignore_tkr pBuffer
       character(c_char), device :: pBuffer(*)
     end function cusparseDbsrsv2_solve_v2
  end interface
#endif

! cusparseCbsrsv2_solve
  interface cusparseCbsrsv2_solve
     integer(c_int) function cusparseCbsrsv2_solve_setHPM(handle, dirA, transA, mb, nnzb, alpha, &
       descrA, bsrValA, bsrRowPtrA, bsrColIndA, blockDim, info, x, y, policy, pBuffer)
       import cusparseHandle, cusparseMatDescr, cusparseBsrsv2Info
       type(cusparseHandle) :: handle
       integer :: dirA, transA, mb, nnzb
       complex(4) :: alpha 
       type(cusparseMatDescr) :: descrA
       complex(4), device :: bsrValA(*)
       integer, device :: bsrRowPtrA(*), bsrColIndA(*)
       integer :: blockDim
       type(cusparseBsrsv2Info) :: info
       complex(4), device :: x(*), y(*)
       integer :: policy
       !dir$ ignore_tkr pBuffer
       character, device :: pBuffer(*)
     end function cusparseCbsrsv2_solve_setHPM
  
     integer(c_int) function cusparseCbsrsv2_solve_setDPM(handle, dirA, transA, mb, nnzb, alpha, &
       descrA, bsrValA, bsrRowPtrA, bsrColIndA, blockDim, info, x, y, policy, pBuffer)
       import cusparseHandle, cusparseMatDescr, cusparseBsrsv2Info
       type(cusparseHandle) :: handle
       integer :: dirA, transA, mb, nnzb
       complex(4), device :: alpha 
       type(cusparseMatDescr) :: descrA
       complex(4), device :: bsrValA(*)
       integer, device :: bsrRowPtrA(*), bsrColIndA(*)
       integer :: blockDim
       type(cusparseBsrsv2Info) :: info
       complex(4), device :: x(*), y(*)
       integer :: policy
       !dir$ ignore_tkr pBuffer
       character, device :: pBuffer(*)
     end function cusparseCbsrsv2_solve_setDPM
  end interface cusparseCbsrsv2_solve

#ifdef KEEP_V2_INTERFACES
  interface
     integer(c_int) function cusparseCbsrsv2_solve_v2(handle, dirA, transA, mb, nnzb, alpha, &
          descrA, bsrValA, bsrRowPtrA, bsrColIndA, blockDim, info, x, y, policy, pBuffer) bind(C,name='cusparseCbsrsv2_solve')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr, cusparseBsrsv2Info
       type(cusparseHandle), value :: handle
       integer(c_int), value :: dirA, transA
       integer(c_int), value :: mb, nnzb
       !dir$ ignore_tkr (d) alpha
       complex(4) :: alpha 
       type(cusparseMatDescr), value :: descrA
       complex(4), device :: bsrValA(*)
       integer(c_int), device :: bsrRowPtrA(*), bsrColIndA(*)
       integer(c_int), value :: blockDim
       type(cusparseBsrsv2Info), value :: info
       complex(4), device :: x(*), y(*)
       integer(c_int), value :: policy
       !dir$ ignore_tkr pBuffer
       character(c_char), device :: pBuffer(*)
     end function cusparseCbsrsv2_solve_v2
  end interface
#endif

! cusparseZbsrsv2_solve
  interface cusparseZbsrsv2_solve
     integer(c_int) function cusparseZbsrsv2_solve_setHPM(handle, dirA, transA, mb, nnzb, alpha, &
       descrA, bsrValA, bsrRowPtrA, bsrColIndA, blockDim, info, x, y, policy, pBuffer)
       import cusparseHandle, cusparseMatDescr, cusparseBsrsv2Info
       type(cusparseHandle) :: handle
       integer :: dirA, transA, mb, nnzb
       complex(8) :: alpha 
       type(cusparseMatDescr) :: descrA
       complex(8), device :: bsrValA(*)
       integer, device :: bsrRowPtrA(*), bsrColIndA(*)
       integer :: blockDim
       type(cusparseBsrsv2Info) :: info
       complex(8), device :: x(*), y(*)
       integer :: policy
       !dir$ ignore_tkr pBuffer
       character, device :: pBuffer(*)
     end function cusparseZbsrsv2_solve_setHPM
  
  integer(c_int) function cusparseZbsrsv2_solve_setDPM(handle, dirA, transA, mb, nnzb, alpha, &
       descrA, bsrValA, bsrRowPtrA, bsrColIndA, blockDim, info, x, y, policy, pBuffer)
       import cusparseHandle, cusparseMatDescr, cusparseBsrsv2Info
       type(cusparseHandle) :: handle
       integer :: dirA, transA, mb, nnzb
       complex(8), device :: alpha 
       type(cusparseMatDescr) :: descrA
       complex(8), device :: bsrValA(*)
       integer, device :: bsrRowPtrA(*), bsrColIndA(*)
       integer :: blockDim
       type(cusparseBsrsv2Info) :: info
       complex(8), device :: x(*), y(*)
       integer :: policy
       !dir$ ignore_tkr pBuffer
       character, device :: pBuffer(*)
     end function cusparseZbsrsv2_solve_setDPM
  end interface cusparseZbsrsv2_solve

#ifdef KEEP_V2_INTERFACES
  interface
     integer(c_int) function cusparseZbsrsv2_solve_v2(handle, dirA, transA, mb, nnzb, alpha, &
          descrA, bsrValA, bsrRowPtrA, bsrColIndA, blockDim, info, x, y, policy, pBuffer) bind(C,name='cusparseZbsrsv2_solve')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr, cusparseBsrsv2Info
       type(cusparseHandle), value :: handle
       integer(c_int), value :: dirA, transA
       integer(c_int), value :: mb, nnzb
       !dir$ ignore_tkr (d) alpha
       complex(8) :: alpha 
       type(cusparseMatDescr), value :: descrA
       complex(8), device :: bsrValA(*)
       integer(c_int), device :: bsrRowPtrA(*), bsrColIndA(*)
       integer(c_int), value :: blockDim
       type(cusparseBsrsv2Info), value :: info
       complex(8), device :: x(*), y(*)
       integer(c_int), value :: policy
       !dir$ ignore_tkr pBuffer
       character(c_char), device :: pBuffer(*)
     end function cusparseZbsrsv2_solve_v2
  end interface
#endif

! cusparseXbsrsv2_zeroPivot
  interface cusparseXbsrsv2_zeroPivot
     integer function cusparseXbsrsv2_zeroPivot_setHPM(handle, info, position)
       import cusparseHandle, cusparseBsrsv2Info
       type(cusparseHandle) :: handle
       type(cusparseBsrsv2Info) :: info
       integer :: position
     end function cusparseXbsrsv2_zeroPivot_setHPM

     integer function cusparseXbsrsv2_zeroPivot_setDPM(handle, info, position)
       import cusparseHandle, cusparseBsrsv2Info
       type(cusparseHandle) :: handle
       type(cusparseBsrsv2Info) :: info
       integer, device :: position
     end function cusparseXbsrsv2_zeroPivot_setDPM
  end interface cusparseXbsrsv2_zeroPivot

#ifdef KEEP_V2_INTERFACES
  interface 
     integer(c_int) function cusparseXbsrsv2_zeroPivot_v2(handle, info, position) bind(C,name='cusparseXbsrsv2_zeroPivot')
       use iso_c_binding
       import cusparseHandle, cusparseBsrsv2Info
       type(cusparseHandle), value :: handle
       type(cusparseBsrsv2Info), value :: info
       !dir$ ignore_tkr (d) position
       integer(c_int) :: position
     end function cusparseXbsrsv2_zeroPivot_v2
  end interface
#endif


! cusparseScsrsv2_bufferSize
  interface cusparseScsrsv2_bufferSize
     integer(c_int) function cusparseScsrsv2_bufferSize(handle, transA, m, nnz, descrA, &
          csrValA, csrRowPtrA, csrColIndA, info, pBufferSize) bind(C,name='cusparseScsrsv2_bufferSize')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr, cusparseCsrsv2Info
       type(cusparseHandle), value :: handle
       integer(c_int), value :: transA, m, nnz
       type(cusparseMatDescr), value :: descrA
       real(c_float), device :: csrValA(*)
       integer(c_int), device :: csrRowPtrA(*), csrColIndA(*)
       type(cusparseCsrsv2Info), value :: info
       integer(4) :: pBufferSize
     end function cusparseScsrsv2_bufferSize
     integer(c_int) function cusparseScsrsv2_bufferSizeExt(handle, transA, m, nnz, descrA, &
          csrValA, csrRowPtrA, csrColIndA, info, pBufferSize) bind(C,name='cusparseScsrsv2_bufferSizeExt')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr, cusparseCsrsv2Info
       type(cusparseHandle), value :: handle
       integer(c_int), value :: transA, m, nnz
       type(cusparseMatDescr), value :: descrA
       real(c_float), device :: csrValA(*)
       integer(c_int), device :: csrRowPtrA(*), csrColIndA(*)
       type(cusparseCsrsv2Info), value :: info
       integer(8) :: pBufferSize
     end function cusparseScsrsv2_bufferSizeExt
  end interface

! cusparseDcsrsv2_bufferSize
  interface cusparseDcsrsv2_bufferSize
     integer(c_int) function cusparseDcsrsv2_bufferSize(handle, transA, m, nnz, descrA, &
          csrValA, csrRowPtrA, csrColIndA, info, pBufferSize) bind(C,name='cusparseDcsrsv2_bufferSize')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr, cusparseCsrsv2Info
       type(cusparseHandle), value :: handle
       integer(c_int), value :: transA, m, nnz
       type(cusparseMatDescr), value :: descrA
       real(c_double), device :: csrValA(*)
       integer(c_int), device :: csrRowPtrA(*), csrColIndA(*)
       type(cusparseCsrsv2Info), value :: info
       integer(4) :: pBufferSize
     end function cusparseDcsrsv2_bufferSize
     integer(c_int) function cusparseDcsrsv2_bufferSizeExt(handle, transA, m, nnz, descrA, &
          csrValA, csrRowPtrA, csrColIndA, info, pBufferSize) bind(C,name='cusparseDcsrsv2_bufferSizeExt')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr, cusparseCsrsv2Info
       type(cusparseHandle), value :: handle
       integer(c_int), value :: transA, m, nnz
       type(cusparseMatDescr), value :: descrA
       real(c_double), device :: csrValA(*)
       integer(c_int), device :: csrRowPtrA(*), csrColIndA(*)
       type(cusparseCsrsv2Info), value :: info
       integer(8) :: pBufferSize
     end function cusparseDcsrsv2_bufferSizeExt
  end interface

! cusparseCcsrsv2_bufferSize
  interface cusparseCcsrsv2_bufferSize
     integer(c_int) function cusparseCcsrsv2_bufferSize(handle, transA, m, nnz, descrA, &
          csrValA, csrRowPtrA, csrColIndA, info, pBufferSize) bind(C,name='cusparseCcsrsv2_bufferSize')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr, cusparseCsrsv2Info
       type(cusparseHandle), value :: handle
       integer(c_int), value :: transA, m, nnz
       type(cusparseMatDescr), value :: descrA
       complex(4), device :: csrValA(*)
       integer(c_int), device :: csrRowPtrA(*), csrColIndA(*)
       type(cusparseCsrsv2Info), value :: info
       integer(4) :: pBufferSize
     end function cusparseCcsrsv2_bufferSize
     integer(c_int) function cusparseCcsrsv2_bufferSizeExt(handle, transA, m, nnz, descrA, &
          csrValA, csrRowPtrA, csrColIndA, info, pBufferSize) bind(C,name='cusparseCcsrsv2_bufferSizeExt')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr, cusparseCsrsv2Info
       type(cusparseHandle), value :: handle
       integer(c_int), value :: transA, m, nnz
       type(cusparseMatDescr), value :: descrA
       complex(4), device :: csrValA(*)
       integer(c_int), device :: csrRowPtrA(*), csrColIndA(*)
       type(cusparseCsrsv2Info), value :: info
       integer(8) :: pBufferSize
     end function cusparseCcsrsv2_bufferSizeExt
  end interface

! cusparseZcsrsv2_bufferSize
  interface cusparseZcsrsv2_bufferSize
     integer(c_int) function cusparseZcsrsv2_bufferSize(handle, transA, m, nnz, descrA, &
          csrValA, csrRowPtrA, csrColIndA, info, pBufferSize) bind(C,name='cusparseZcsrsv2_bufferSize')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr, cusparseCsrsv2Info
       type(cusparseHandle), value :: handle
       integer(c_int), value :: transA, m, nnz
       type(cusparseMatDescr), value :: descrA
       complex(8), device :: csrValA(*)
       integer(c_int), device :: csrRowPtrA(*), csrColIndA(*)
       type(cusparseCsrsv2Info), value :: info
       integer(4) :: pBufferSize
     end function cusparseZcsrsv2_bufferSize
     integer(c_int) function cusparseZcsrsv2_bufferSizeExt(handle, transA, m, nnz, descrA, &
          csrValA, csrRowPtrA, csrColIndA, info, pBufferSize) bind(C,name='cusparseZcsrsv2_bufferSizeExt')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr, cusparseCsrsv2Info
       type(cusparseHandle), value :: handle
       integer(c_int), value :: transA, m, nnz
       type(cusparseMatDescr), value :: descrA
       complex(8), device :: csrValA(*)
       integer(c_int), device :: csrRowPtrA(*), csrColIndA(*)
       type(cusparseCsrsv2Info), value :: info
       integer(8) :: pBufferSize
     end function cusparseZcsrsv2_bufferSizeExt
  end interface

! cusparseScsrsv2_analysis
  interface 
     integer(c_int) function cusparseScsrsv2_analysis(handle, transA, m, nnz, descrA, &
          csrValA, csrRowPtrA, csrColIndA, info, policy, pBuffer) bind(C,name='cusparseScsrsv2_analysis')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr, cusparseCsrsv2Info
       type(cusparseHandle), value :: handle
       integer(c_int), value :: transA, m, nnz
       type(cusparseMatDescr), value :: descrA
       real(c_float), device :: csrValA(*)
       integer(c_int), device :: csrRowPtrA(*), csrColIndA(*)
       type(cusparseCsrsv2Info), value :: info
       integer(c_int), value :: policy
       !dir$ ignore_tkr pBuffer
       character(c_char), device :: pBuffer(*)
     end function cusparseScsrsv2_analysis
  end interface

! cusparseDcsrsv2_analysis
  interface 
     integer(c_int) function cusparseDcsrsv2_analysis(handle, transA, m, nnz, descrA, &
          csrValA, csrRowPtrA, csrColIndA, info, policy, pBuffer) bind(C,name='cusparseDcsrsv2_analysis')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr, cusparseCsrsv2Info
       type(cusparseHandle), value :: handle
       integer(c_int), value :: transA, m, nnz
       type(cusparseMatDescr), value :: descrA
       real(c_double), device :: csrValA(*)
       integer(c_int), device :: csrRowPtrA(*), csrColIndA(*)
       type(cusparseCsrsv2Info), value :: info
       integer(c_int), value :: policy
       !dir$ ignore_tkr pBuffer
       character(c_char), device :: pBuffer(*)
     end function cusparseDcsrsv2_analysis
  end interface

! cusparseCcsrsv2_analysis
  interface 
     integer(c_int) function cusparseCcsrsv2_analysis(handle, transA, m, nnz, descrA, &
          csrValA, csrRowPtrA, csrColIndA, info, policy, pBuffer) bind(C,name='cusparseCcsrsv2_analysis')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr, cusparseCsrsv2Info
       type(cusparseHandle), value :: handle
       integer(c_int), value :: transA, m, nnz
       type(cusparseMatDescr), value :: descrA
       complex(4), device :: csrValA(*)
       integer(c_int), device :: csrRowPtrA(*), csrColIndA(*)
       type(cusparseCsrsv2Info), value :: info
       integer(c_int), value :: policy
       !dir$ ignore_tkr pBuffer
       character(c_char), device :: pBuffer(*)
     end function cusparseCcsrsv2_analysis
  end interface

! cusparseZcsrsv2_analysis
  interface 
     integer(c_int) function cusparseZcsrsv2_analysis(handle, transA, m, nnz, descrA, &
          csrValA, csrRowPtrA, csrColIndA, info, policy, pBuffer) bind(C,name='cusparseZcsrsv2_analysis')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr, cusparseCsrsv2Info
       type(cusparseHandle), value :: handle
       integer(c_int), value :: transA, m, nnz
       type(cusparseMatDescr), value :: descrA
       complex(8), device :: csrValA(*)
       integer(c_int), device :: csrRowPtrA(*), csrColIndA(*)
       type(cusparseCsrsv2Info), value :: info
       integer(c_int), value :: policy
       !dir$ ignore_tkr pBuffer
       character(c_char), device :: pBuffer(*)
     end function cusparseZcsrsv2_analysis
  end interface

! cusparseScsrsv2_solve
  interface cusparseScsrsv2_solve
     integer function cusparseScsrsv2_solve_setHPM(handle, transA, m, nnz, alpha, descrA, &
          csrValA, csrRowPtrA, csrColIndA, info, x, y, policy, pBuffer)
       import cusparseHandle, cusparseMatDescr, cusparseCsrsv2Info
       type(cusparseHandle) :: handle
       integer:: transA, m, nnz
       real(4) :: alpha
       type(cusparseMatDescr) :: descrA
       real(4), device :: csrValA(*), x(*), y(*)
       integer, device :: csrRowPtrA(*), csrColIndA(*)
       type(cusparseCsrsv2Info) :: info
       integer :: policy
       !dir$ ignore_tkr pBuffer
       character, device :: pBuffer(*)
     end function cusparseScsrsv2_solve_setHPM

     integer function cusparseScsrsv2_solve_setDPM(handle, transA, m, nnz, alpha, descrA, &
          csrValA, csrRowPtrA, csrColIndA, info, x, y, policy, pBuffer)
       import cusparseHandle, cusparseMatDescr, cusparseCsrsv2Info
       type(cusparseHandle) :: handle
       integer:: transA, m, nnz
       real(4), device :: alpha
       type(cusparseMatDescr) :: descrA
       real(4), device :: csrValA(*), x(*), y(*)
       integer, device :: csrRowPtrA(*), csrColIndA(*)
       type(cusparseCsrsv2Info) :: info
       integer :: policy
       !dir$ ignore_tkr pBuffer
       character, device :: pBuffer(*)
     end function cusparseScsrsv2_solve_setDPM
  end interface cusparseScsrsv2_solve

#ifdef KEEP_V2_INTERFACES
  interface 
     integer(c_int) function cusparseScsrsv2_solve_v2(handle, transA, m, nnz, alpha, descrA, &
          csrValA, csrRowPtrA, csrColIndA, info, x, y, policy, pBuffer) bind(C,name='cusparseScsrsv2_solve')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr, cusparseCsrsv2Info
       type(cusparseHandle), value :: handle
       integer(c_int), value :: transA, m, nnz
       !dir$ ignore_tkr (d) alpha
       real(c_float) :: alpha
       type(cusparseMatDescr), value :: descrA
       real(c_float), device :: csrValA(*)
       integer(c_int), device :: csrRowPtrA(*), csrColIndA(*)
       type(cusparseCsrsv2Info), value :: info
       real(c_float), device :: x(*), y(*)
       integer(c_int), value :: policy
       !dir$ ignore_tkr pBuffer
       character(c_char), device :: pBuffer(*)
     end function cusparseScsrsv2_solve_v2
  end interface
#endif

! cusparseDcsrsv2_solve
  interface cusparseDcsrsv2_solve
     integer function cusparseDcsrsv2_solve_setHPM(handle, transA, m, nnz, alpha, descrA, &
          csrValA, csrRowPtrA, csrColIndA, info, x, y, policy, pBuffer)
       import cusparseHandle, cusparseMatDescr, cusparseCsrsv2Info
       type(cusparseHandle) :: handle
       integer:: transA, m, nnz
       real(8) :: alpha
       type(cusparseMatDescr) :: descrA
       real(8), device :: csrValA(*), x(*), y(*)
       integer, device :: csrRowPtrA(*), csrColIndA(*)
       type(cusparseCsrsv2Info) :: info
       integer :: policy
       !dir$ ignore_tkr pBuffer
       character, device :: pBuffer(*)
     end function cusparseDcsrsv2_solve_setHPM

     integer function cusparseDcsrsv2_solve_setDPM(handle, transA, m, nnz, alpha, descrA, &
          csrValA, csrRowPtrA, csrColIndA, info, x, y, policy, pBuffer)
       import cusparseHandle, cusparseMatDescr, cusparseCsrsv2Info
       type(cusparseHandle) :: handle
       integer:: transA, m, nnz
       real(8), device :: alpha
       type(cusparseMatDescr) :: descrA
       real(8), device :: csrValA(*), x(*), y(*)
       integer, device :: csrRowPtrA(*), csrColIndA(*)
       type(cusparseCsrsv2Info) :: info
       integer :: policy
       !dir$ ignore_tkr pBuffer
       character, device :: pBuffer(*)
     end function cusparseDcsrsv2_solve_setDPM
  end interface cusparseDcsrsv2_solve

#ifdef KEEP_V2_INTERFACES
  interface 
     integer(c_int) function cusparseDcsrsv2_solve_v2(handle, transA, m, nnz, alpha, descrA, &
          csrValA, csrRowPtrA, csrColIndA, info, x, y, policy, pBuffer) bind(C,name='cusparseDcsrsv2_solve')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr, cusparseCsrsv2Info
       type(cusparseHandle), value :: handle
       integer(c_int), value :: transA, m, nnz
       !dir$ ignore_tkr (d) alpha
       real(c_double) :: alpha
       type(cusparseMatDescr), value :: descrA
       real(c_double), device :: csrValA(*)
       integer(c_int), device :: csrRowPtrA(*), csrColIndA(*)
       type(cusparseCsrsv2Info), value :: info
       real(c_double), device :: x(*), y(*)
       integer(c_int), value :: policy
       !dir$ ignore_tkr pBuffer
       character(c_char), device :: pBuffer(*)
     end function cusparseDcsrsv2_solve_v2
  end interface
#endif

! cusparseCcsrsv2_solve
  interface cusparseCcsrsv2_solve
     integer function cusparseCcsrsv2_solve_setHPM(handle, transA, m, nnz, alpha, descrA, &
          csrValA, csrRowPtrA, csrColIndA, info, x, y, policy, pBuffer)
       import cusparseHandle, cusparseMatDescr, cusparseCsrsv2Info
       type(cusparseHandle) :: handle
       integer:: transA, m, nnz
       complex(4) :: alpha
       type(cusparseMatDescr) :: descrA
       complex(4), device :: csrValA(*), x(*), y(*)
       integer, device :: csrRowPtrA(*), csrColIndA(*)
       type(cusparseCsrsv2Info) :: info
       integer :: policy
       !dir$ ignore_tkr pBuffer
       character, device :: pBuffer(*)
     end function cusparseCcsrsv2_solve_setHPM

     integer function cusparseCcsrsv2_solve_setDPM(handle, transA, m, nnz, alpha, descrA, &
          csrValA, csrRowPtrA, csrColIndA, info, x, y, policy, pBuffer)
       import cusparseHandle, cusparseMatDescr, cusparseCsrsv2Info
       type(cusparseHandle) :: handle
       integer:: transA, m, nnz
       complex(4), device :: alpha
       type(cusparseMatDescr) :: descrA
       complex(4), device :: csrValA(*), x(*), y(*)
       integer, device :: csrRowPtrA(*), csrColIndA(*)
       type(cusparseCsrsv2Info) :: info
       integer :: policy
       !dir$ ignore_tkr pBuffer
       character, device :: pBuffer(*)
     end function cusparseCcsrsv2_solve_setDPM
  end interface cusparseCcsrsv2_solve

#ifdef KEEP_V2_INTERFACES
  interface 
     integer(c_int) function cusparseCcsrsv2_solve_v2(handle, transA, m, nnz, alpha, descrA, &
          csrValA, csrRowPtrA, csrColIndA, info, x, y, policy, pBuffer) bind(C,name='cusparseCcsrsv2_solve')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr, cusparseCsrsv2Info
       type(cusparseHandle), value :: handle
       integer(c_int), value :: transA, m, nnz
       !dir$ ignore_tkr (d) alpha
       complex(4) :: alpha
       type(cusparseMatDescr), value :: descrA
       complex(4), device :: csrValA(*)
       integer(c_int), device :: csrRowPtrA(*), csrColIndA(*)
       type(cusparseCsrsv2Info), value :: info
       complex(4), device :: x(*), y(*)
       integer(c_int), value :: policy
       !dir$ ignore_tkr pBuffer
       character(c_char), device :: pBuffer(*)
     end function cusparseCcsrsv2_solve_v2
  end interface
#endif

! cusparseZcsrsv2_solve
  interface cusparseZcsrsv2_solve
     integer function cusparseZcsrsv2_solve_setHPM(handle, transA, m, nnz, alpha, descrA, &
          csrValA, csrRowPtrA, csrColIndA, info, x, y, policy, pBuffer)
       import cusparseHandle, cusparseMatDescr, cusparseCsrsv2Info
       type(cusparseHandle) :: handle
       integer:: transA, m, nnz
       complex(8) :: alpha
       type(cusparseMatDescr) :: descrA
       complex(8), device :: csrValA(*), x(*), y(*)
       integer, device :: csrRowPtrA(*), csrColIndA(*)
       type(cusparseCsrsv2Info) :: info
       integer :: policy
       !dir$ ignore_tkr pBuffer
       character, device :: pBuffer(*)
     end function cusparseZcsrsv2_solve_setHPM

     integer function cusparseZcsrsv2_solve_setDPM(handle, transA, m, nnz, alpha, descrA, &
          csrValA, csrRowPtrA, csrColIndA, info, x, y, policy, pBuffer)
       import cusparseHandle, cusparseMatDescr, cusparseCsrsv2Info
       type(cusparseHandle) :: handle
       integer:: transA, m, nnz
       complex(8), device :: alpha
       type(cusparseMatDescr) :: descrA
       complex(8), device :: csrValA(*), x(*), y(*)
       integer, device :: csrRowPtrA(*), csrColIndA(*)
       type(cusparseCsrsv2Info) :: info
       integer :: policy
       !dir$ ignore_tkr pBuffer
       character, device :: pBuffer(*)
     end function cusparseZcsrsv2_solve_setDPM
  end interface cusparseZcsrsv2_solve

#ifdef KEEP_V2_INTERFACES
  interface 
     integer(c_int) function cusparseZcsrsv2_solve_v2(handle, transA, m, nnz, alpha, descrA, &
          csrValA, csrRowPtrA, csrColIndA, info, x, y, policy, pBuffer) bind(C,name='cusparseZcsrsv2_solve')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr, cusparseCsrsv2Info
       type(cusparseHandle), value :: handle
       integer(c_int), value :: transA, m, nnz
       !dir$ ignore_tkr (d) alpha
       complex(8) :: alpha
       type(cusparseMatDescr), value :: descrA
       complex(8), device :: csrValA(*)
       integer(c_int), device :: csrRowPtrA(*), csrColIndA(*)
       type(cusparseCsrsv2Info), value :: info
       complex(8), device :: x(*), y(*)
       integer(c_int), value :: policy
       !dir$ ignore_tkr pBuffer
       character(c_char), device :: pBuffer(*)
     end function cusparseZcsrsv2_solve_v2
  end interface
#endif

! cusparseXcsrsv2_zeroPivot
  interface cusparseXcsrsv2_zeroPivot
     integer function cusparseXcsrsv2_zeroPivot_setHPM(handle, info, position)
       import cusparseHandle, cusparseCsrsv2Info
       type(cusparseHandle), value :: handle
       type(cusparseCsrsv2Info), value :: info
       integer :: position
     end function cusparseXcsrsv2_zeroPivot_setHPM

     integer function cusparseXcsrsv2_zeroPivot_setDPM(handle, info, position)
       import cusparseHandle, cusparseCsrsv2Info
       type(cusparseHandle), value :: handle
       type(cusparseCsrsv2Info), value :: info
       integer, device :: position
     end function cusparseXcsrsv2_zeroPivot_setDPM
  end interface cusparseXcsrsv2_zeroPivot

#ifdef KEEP_V2_INTERFACES
  interface
     integer(c_int) function cusparseXcsrsv2_zeroPivot_v2(handle, info, position) bind(C, name='cusparseXcsrsv2_zeroPivot')
       use iso_c_binding
       import cusparseHandle, cusparseCsrsv2Info
       type(cusparseHandle), value :: handle
       type(cusparseCsrsv2Info), value :: info
       !dir$ ignore_tkr (d) position
       integer(c_int) :: position
     end function cusparseXcsrsv2_zeroPivot_v2
  end interface
#endif

  ! ---------------------
  ! 6.5 Level 2 functions
  ! ---------------------
  ! none
  
  ! ---------------------
  ! 7.0 Level 2 functions
  ! ---------------------
  ! none

  ! -----------
  ! 5.5 Level 3
  ! -----------

! cusparseScsrmm
  interface cusparseScsrmm
    integer function cusparseScsrmm_setHPM(handle, transA, m, n, k, nnz, alpha, &
          descrA, csrValA, csrRowPtrA, csrColIndA, B, ldb, beta, C, ldc)
    import cusparseHandle, cusparseMatDescr
    type(cusparseHandle) :: handle
    integer :: transA, m, n, k, nnz
    real(4) :: alpha, beta
    type(cusparseMatDescr) :: descrA
    !dir$ ignore_tkr (R) B, (r) C
    real(4), device :: csrValA(*), B(*), C(*)
    integer, device :: csrRowPtrA(*), csrColIndA(*)
    integer :: ldb, ldc
    end function cusparseScsrmm_setHPM
    integer function cusparseScsrmm_setDPM(handle, transA, m, n, k, nnz, alpha, &
          descrA, csrValA, csrRowPtrA, csrColIndA, B, ldb, beta, C, ldc)
    import cusparseHandle, cusparseMatDescr
    type(cusparseHandle) :: handle
    integer :: transA, m, n, k, nnz
    real(4), device :: alpha, beta
    type(cusparseMatDescr) :: descrA
    !dir$ ignore_tkr (R) B, (R) C
    real(4), device :: csrValA(*), B(*), C(*)
    integer, device :: csrRowPtrA(*), csrColIndA(*)
    integer :: ldb, ldc
    end function cusparseScsrmm_setDPM
  end interface cusparseScsrmm

#ifdef KEEP_V2_INTERFACES
  interface cusparseScsrmm_v2
     integer(c_int) function cusparseScsrmm_v2(handle, transA, m, n, k, nnz, alpha, &
          descrA, csrValA, csrRowPtrA, csrColIndA, B, ldb, beta, C, ldc) &
          bind(C, name='cusparseScsrmm')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr
       type(cusparseHandle), value :: handle
       integer(c_int), value :: transA, m, n, k, nnz
       !dir$ ignore_tkr (d) alpha, (d) beta
       real(c_float) :: alpha, beta
       type(cusparseMatDescr), value :: descrA
       !dir$ ignore_tkr (r) B, (r) C
       real(c_float), device :: csrValA(*), B(*), C(*)
       integer(c_int), device :: csrRowPtrA(*), csrColIndA(*)
       integer(c_int), value :: ldb, ldc
     end function cusparseScsrmm_v2
  end interface cusparseScsrmm_v2
#endif

! cusparseDcsrmm
  interface cusparseDcsrmm
    integer function cusparseDcsrmm_setHPM(handle, transA, m, n, k, nnz, alpha, &
          descrA, csrValA, csrRowPtrA, csrColIndA, B, ldb, beta, C, ldc)
    import cusparseHandle, cusparseMatDescr
    type(cusparseHandle) :: handle
    integer :: transA, m, n, k, nnz
    real(8) :: alpha, beta
    type(cusparseMatDescr) :: descrA
    !dir$ ignore_tkr (R) B, (R) C
    real(8), device :: csrValA(*), B(*), C(*)
    integer, device :: csrRowPtrA(*), csrColIndA(*)
    integer :: ldb, ldc
    end function cusparseDcsrmm_setHPM
    integer function cusparseDcsrmm_setDPM(handle, transA, m, n, k, nnz, alpha, &
          descrA, csrValA, csrRowPtrA, csrColIndA, B, ldb, beta, C, ldc)
    import cusparseHandle, cusparseMatDescr
    type(cusparseHandle) :: handle
    integer :: transA, m, n, k, nnz
    real(8), device :: alpha, beta
    type(cusparseMatDescr) :: descrA
    !dir$ ignore_tkr (R) B, (R) C
    real(8), device :: csrValA(*), B(*), C(*)
    integer, device :: csrRowPtrA(*), csrColIndA(*)
    integer :: ldb, ldc
    end function cusparseDcsrmm_setDPM
  end interface cusparseDcsrmm

#ifdef KEEP_V2_INTERFACES
  interface cusparseDcsrmm_v2
     integer(c_int) function cusparseDcsrmm_v2(handle, transA, m, n, k, nnz, alpha, &
          descrA, csrValA, csrRowPtrA, csrColIndA, B, ldb, beta, C, ldc) &
          bind(C, name='cusparseDcsrmm')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr
       type(cusparseHandle), value :: handle
       integer(c_int), value :: transA, m, n, k, nnz
       !dir$ ignore_tkr (d) alpha, (d) beta
       real(c_double) :: alpha, beta
       type(cusparseMatDescr), value :: descrA
       !dir$ ignore_tkr (r) B, (r) C
       real(c_double), device :: csrValA(*), B(*), C(*)
       integer(c_int), device :: csrRowPtrA(*), csrColIndA(*)
       integer(c_int), value :: ldb, ldc
     end function cusparseDcsrmm_v2
  end interface cusparseDcsrmm_v2
#endif

! cusparseCcsrmm
  interface cusparseCcsrmm
    integer function cusparseCcsrmm_setHPM(handle, transA, m, n, k, nnz, alpha, &
          descrA, csrValA, csrRowPtrA, csrColIndA, B, ldb, beta, C, ldc)
    import cusparseHandle, cusparseMatDescr
    type(cusparseHandle) :: handle
    integer :: transA, m, n, k, nnz
    complex(4) :: alpha, beta
    type(cusparseMatDescr) :: descrA
    !dir$ ignore_tkr (R) B, (R) C
    complex(4), device :: csrValA(*), B(*), C(*)
    integer, device :: csrRowPtrA(*), csrColIndA(*)
    integer :: ldb, ldc
    end function cusparseCcsrmm_setHPM
    integer function cusparseCcsrmm_setDPM(handle, transA, m, n, k, nnz, alpha, &
          descrA, csrValA, csrRowPtrA, csrColIndA, B, ldb, beta, C, ldc)
    import cusparseHandle, cusparseMatDescr
    type(cusparseHandle) :: handle
    integer :: transA, m, n, k, nnz
    complex(4), device :: alpha, beta
    type(cusparseMatDescr) :: descrA
    !dir$ ignore_tkr (R) B, (R) C
    complex(4), device :: csrValA(*), B(*), C(*)
    integer, device :: csrRowPtrA(*), csrColIndA(*)
    integer :: ldb, ldc
    end function cusparseCcsrmm_setDPM
  end interface cusparseCcsrmm

#ifdef KEEP_V2_INTERFACES
  interface cusparseCcsrmm_v2
     integer(c_int) function cusparseCcsrmm_v2(handle, transA, m, n, k, nnz, alpha, &
          descrA, csrValA, csrRowPtrA, csrColIndA, B, ldb, beta, C, ldc) &
          bind(C, name='cusparseCcsrmm')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr
       type(cusparseHandle), value :: handle
       integer(c_int), value :: transA, m, n, k, nnz
       !dir$ ignore_tkr (d) alpha, (d) beta
       complex(4) :: alpha, beta
       type(cusparseMatDescr), value :: descrA
       !dir$ ignore_tkr (r) B, (r) C
       complex(4), device :: csrValA(*), B(*), C(*)
       integer(c_int), device :: csrRowPtrA(*), csrColIndA(*)
       integer(c_int), value :: ldb, ldc
     end function cusparseCcsrmm_v2
  end interface cusparseCcsrmm_v2
#endif

! cusparseZcsrmm
  interface cusparseZcsrmm
    integer function cusparseZcsrmm_setHPM(handle, transA, m, n, k, nnz, alpha, &
          descrA, csrValA, csrRowPtrA, csrColIndA, B, ldb, beta, C, ldc)
    import cusparseHandle, cusparseMatDescr
    type(cusparseHandle) :: handle
    integer :: transA, m, n, k, nnz
    complex(8) :: alpha, beta
    type(cusparseMatDescr) :: descrA
    !dir$ ignore_tkr (R) B, (R) C
    complex(8), device :: csrValA(*), B(*), C(*)
    integer, device :: csrRowPtrA(*), csrColIndA(*)
    integer :: ldb, ldc
    end function cusparseZcsrmm_setHPM
    integer function cusparseZcsrmm_setDPM(handle, transA, m, n, k, nnz, alpha, &
          descrA, csrValA, csrRowPtrA, csrColIndA, B, ldb, beta, C, ldc)
    import cusparseHandle, cusparseMatDescr
    type(cusparseHandle) :: handle
    integer :: transA, m, n, k, nnz
    complex(8), device :: alpha, beta
    type(cusparseMatDescr) :: descrA
    !dir$ ignore_tkr (R) B, (R) C
    complex(8), device :: csrValA(*), B(*), C(*)
    integer, device :: csrRowPtrA(*), csrColIndA(*)
    integer :: ldb, ldc
    end function cusparseZcsrmm_setDPM
  end interface cusparseZcsrmm

#ifdef KEEP_V2_INTERFACES
  interface cusparseZcsrmm_v2
     integer(c_int) function cusparseZcsrmm_v2(handle, transA, m, n, k, nnz, alpha, &
          descrA, csrValA, csrRowPtrA, csrColIndA, B, ldb, beta, C, ldc) &
          bind(C, name='cusparseZcsrmm')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr
       type(cusparseHandle), value :: handle
       integer(c_int), value :: transA, m, n, k, nnz
       !dir$ ignore_tkr (d) alpha, (d) beta
       complex(8) :: alpha, beta
       type(cusparseMatDescr), value :: descrA
       !dir$ ignore_tkr (r) B, (r) C
       complex(8), device :: csrValA(*), B(*), C(*)
       integer(c_int), device :: csrRowPtrA(*), csrColIndA(*)
       integer(c_int), value :: ldb, ldc
     end function cusparseZcsrmm_v2
  end interface cusparseZcsrmm_v2
#endif

! cusparseScsrmm2
  interface cusparseScsrmm2
    integer function cusparseScsrmm2_setHPM(handle, transA, transB, m, n, k, nnz, alpha, &
          descrA, csrValA, csrRowPtrA, csrColIndA, B, ldb, beta, C, ldc)
    import cusparseHandle, cusparseMatDescr
    type(cusparseHandle) :: handle
    integer :: transA, transB, m, n, k, nnz
    real(4) :: alpha, beta
    type(cusparseMatDescr) :: descrA
    !dir$ ignore_tkr (R) B, (R) C
    real(4), device :: csrValA(*), B(*), C(*)
    integer, device :: csrRowPtrA(*), csrColIndA(*)
    integer :: ldb, ldc
    end function cusparseScsrmm2_setHPM
    integer function cusparseScsrmm2_setDPM(handle, transA, transB, m, n, k, nnz, alpha, &
          descrA, csrValA, csrRowPtrA, csrColIndA, B, ldb, beta, C, ldc)
    import cusparseHandle, cusparseMatDescr
    type(cusparseHandle) :: handle
    integer :: transA, transB, m, n, k, nnz
    real(4), device :: alpha, beta
    type(cusparseMatDescr) :: descrA
    !dir$ ignore_tkr (R) B, (R) C
    real(4), device :: csrValA(*), B(*), C(*)
    integer, device :: csrRowPtrA(*), csrColIndA(*)
    integer :: ldb, ldc
    end function cusparseScsrmm2_setDPM
  end interface cusparseScsrmm2

#ifdef KEEP_V2_INTERFACES
  interface cusparseScsrmm2_v2
     integer(c_int) function cusparseScsrmm2_v2(handle, transA, transB, m, n, k, nnz, alpha, &
          descrA, csrValA, csrRowPtrA, csrColIndA, B, ldb, beta, C, ldc) &
          bind(C, name='cusparseScsrmm2')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr
       type(cusparseHandle), value :: handle
       integer(c_int), value :: transA, transB, m, n, k, nnz
       !dir$ ignore_tkr (d) alpha, (d) beta
       real(c_float) :: alpha, beta
       type(cusparseMatDescr), value :: descrA
       !dir$ ignore_tkr (r) B, (r) C
       real(c_float), device :: csrValA(*), B(*), C(*)
       integer(c_int), device :: csrRowPtrA(*), csrColIndA(*)
       integer(c_int), value :: ldb, ldc
     end function cusparseScsrmm2_v2
  end interface cusparseScsrmm2_v2
#endif

! cusparseDcsrmm2
  interface cusparseDcsrmm2
    integer function cusparseDcsrmm2_setHPM(handle, transA, transB, m, n, k, nnz, alpha, &
          descrA, csrValA, csrRowPtrA, csrColIndA, B, ldb, beta, C, ldc)
    import cusparseHandle, cusparseMatDescr
    type(cusparseHandle) :: handle
    integer :: transA, transB, m, n, k, nnz
    real(8) :: alpha, beta
    type(cusparseMatDescr) :: descrA
    !dir$ ignore_tkr (R) B, (R) C
    real(8), device :: csrValA(*), B(*), C(*)
    integer, device :: csrRowPtrA(*), csrColIndA(*)
    integer :: ldb, ldc
    end function cusparseDcsrmm2_setHPM
    integer function cusparseDcsrmm2_setDPM(handle, transA, transB, m, n, k, nnz, alpha, &
          descrA, csrValA, csrRowPtrA, csrColIndA, B, ldb, beta, C, ldc)
    import cusparseHandle, cusparseMatDescr
    type(cusparseHandle) :: handle
    integer :: transA, transB, m, n, k, nnz
    real(8), device :: alpha, beta
    type(cusparseMatDescr) :: descrA
    !dir$ ignore_tkr (R) B, (R) C
    real(8), device :: csrValA(*), B(*), C(*)
    integer, device :: csrRowPtrA(*), csrColIndA(*)
    integer :: ldb, ldc
    end function cusparseDcsrmm2_setDPM
  end interface cusparseDcsrmm2

#ifdef KEEP_V2_INTERFACES
  interface cusparseDcsrmm2_v2
     integer(c_int) function cusparseDcsrmm2_v2(handle, transA, transB, m, n, k, nnz, alpha, &
          descrA, csrValA, csrRowPtrA, csrColIndA, B, ldb, beta, C, ldc) &
          bind(C, name='cusparseDcsrmm2')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr
       type(cusparseHandle), value :: handle
       integer(c_int), value :: transA, transB, m, n, k, nnz
       !dir$ ignore_tkr (d) alpha, (d) beta
       real(c_double) :: alpha, beta
       type(cusparseMatDescr), value :: descrA
       !dir$ ignore_tkr (r) B, (r) C
       real(c_double), device :: csrValA(*), B(*), C(*)
       integer(c_int), device :: csrRowPtrA(*), csrColIndA(*)
       integer(c_int), value :: ldb, ldc
     end function cusparseDcsrmm2_v2
  end interface cusparseDcsrmm2_v2
#endif

! cusparseCcsrmm2
  interface cusparseCcsrmm2
    integer function cusparseCcsrmm2_setHPM(handle, transA, transB, m, n, k, nnz, alpha, &
          descrA, csrValA, csrRowPtrA, csrColIndA, B, ldb, beta, C, ldc)
    import cusparseHandle, cusparseMatDescr
    type(cusparseHandle) :: handle
    integer :: transA, transB, m, n, k, nnz
    complex(4) :: alpha, beta
    type(cusparseMatDescr) :: descrA
    !dir$ ignore_tkr (R) B, (R) C
    complex(4), device :: csrValA(*), B(*), C(*)
    integer, device :: csrRowPtrA(*), csrColIndA(*)
    integer :: ldb, ldc
    end function cusparseCcsrmm2_setHPM
    integer function cusparseCcsrmm2_setDPM(handle, transA, transB, m, n, k, nnz, alpha, &
          descrA, csrValA, csrRowPtrA, csrColIndA, B, ldb, beta, C, ldc)
    import cusparseHandle, cusparseMatDescr
    type(cusparseHandle) :: handle
    integer :: transA, transB, m, n, k, nnz
    complex(4), device :: alpha, beta
    type(cusparseMatDescr) :: descrA
    !dir$ ignore_tkr (R) B, (R) C
    complex(4), device :: csrValA(*), B(*), C(*)
    integer, device :: csrRowPtrA(*), csrColIndA(*)
    integer :: ldb, ldc
    end function cusparseCcsrmm2_setDPM
  end interface cusparseCcsrmm2

#ifdef KEEP_V2_INTERFACES
  interface cusparseCcsrmm2_v2
     integer(c_int) function cusparseCcsrmm2_v2(handle, transA, transB, m, n, k, nnz, alpha, &
          descrA, csrValA, csrRowPtrA, csrColIndA, B, ldb, beta, C, ldc) &
          bind(C, name='cusparseCcsrmm2')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr
       type(cusparseHandle), value :: handle
       integer(c_int), value :: transA, transB, m, n, k, nnz
       !dir$ ignore_tkr (d) alpha, (d) beta
       complex(4) :: alpha, beta
       type(cusparseMatDescr), value :: descrA
       !dir$ ignore_tkr (r) B, (r) C
       complex(4), device :: csrValA(*), B(*), C(*)
       integer(c_int), device :: csrRowPtrA(*), csrColIndA(*)
       integer(c_int), value :: ldb, ldc
     end function cusparseCcsrmm2_v2
  end interface cusparseCcsrmm2_v2
#endif

! cusparseZcsrmm2
  interface cusparseZcsrmm2
    integer function cusparseZcsrmm2_setHPM(handle, transA, transB, m, n, k, nnz, alpha, &
          descrA, csrValA, csrRowPtrA, csrColIndA, B, ldb, beta, C, ldc)
    import cusparseHandle, cusparseMatDescr
    type(cusparseHandle) :: handle
    integer :: transA, transB, m, n, k, nnz
    complex(8) :: alpha, beta
    type(cusparseMatDescr) :: descrA
    !dir$ ignore_tkr (R) B, (R) C
    complex(8), device :: csrValA(*), B(*), C(*)
    integer, device :: csrRowPtrA(*), csrColIndA(*)
    integer :: ldb, ldc
    end function cusparseZcsrmm2_setHPM
    integer function cusparseZcsrmm2_setDPM(handle, transA, transB, m, n, k, nnz, alpha, &
          descrA, csrValA, csrRowPtrA, csrColIndA, B, ldb, beta, C, ldc)
    import cusparseHandle, cusparseMatDescr
    type(cusparseHandle) :: handle
    integer :: transA, transB, m, n, k, nnz
    complex(8), device :: alpha, beta
    type(cusparseMatDescr) :: descrA
    !dir$ ignore_tkr (R) B, (R) C
    complex(8), device :: csrValA(*), B(*), C(*)
    integer, device :: csrRowPtrA(*), csrColIndA(*)
    integer :: ldb, ldc
    end function cusparseZcsrmm2_setDPM
  end interface cusparseZcsrmm2

#ifdef KEEP_V2_INTERFACES
  interface cusparseZcsrmm2_v2
     integer(c_int) function cusparseZcsrmm2_v2(handle, transA, transB, m, n, k, nnz, alpha, &
          descrA, csrValA, csrRowPtrA, csrColIndA, B, ldb, beta, C, ldc) &
          bind(C, name='cusparseZcsrmm2')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr
       type(cusparseHandle), value :: handle
       integer(c_int), value :: transA, transB, m, n, k, nnz
       !dir$ ignore_tkr (d) alpha, (d) beta
       complex(8) :: alpha, beta
       type(cusparseMatDescr), value :: descrA
       !dir$ ignore_tkr (r) B, (r) C
       complex(8), device :: csrValA(*), B(*), C(*)
       integer(c_int), device :: csrRowPtrA(*), csrColIndA(*)
       integer(c_int), value :: ldb, ldc
     end function cusparseZcsrmm2_v2
  end interface cusparseZcsrmm2_v2
#endif

! cusparseScsrsm_analysis
  interface cusparseScsrsm_analysis
     integer(c_int) function cusparseScsrsm_analysis(handle, transA, m, nnz, descrA, &
          csrValA, csrRowPtrA, csrColIndA, info) bind(C, name='cusparseScsrsm_analysis')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr, cusparseSolveAnalysisInfo
       type(cusparseHandle), value :: handle       
       integer(c_int), value :: transA, m, nnz
       type(cusparseMatDescr), value :: descrA
       real(c_float), device :: csrValA(*)
       integer(c_int), device :: csrRowPtrA(*), csrColIndA(*)
       type(cusparseSolveAnalysisInfo), value :: info
     end function cusparseScsrsm_analysis
  end interface cusparseScsrsm_analysis

! cusparseDcsrsm_analysis
  interface cusparseDcsrsm_analysis
     integer(c_int) function cusparseDcsrsm_analysis(handle, transA, m, nnz, descrA, &
          csrValA, csrRowPtrA, csrColIndA, info) bind(C, name='cusparseDcsrsm_analysis')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr, cusparseSolveAnalysisInfo
       type(cusparseHandle), value :: handle       
       integer(c_int), value :: transA, m, nnz
       type(cusparseMatDescr), value :: descrA
       real(c_double), device :: csrValA(*)
       integer(c_int), device :: csrRowPtrA(*), csrColIndA(*)
       type(cusparseSolveAnalysisInfo), value :: info
     end function cusparseDcsrsm_analysis
  end interface cusparseDcsrsm_analysis

! cusparseCcsrsm_analysis
  interface cusparseCcsrsm_analysis
     integer(c_int) function cusparseCcsrsm_analysis(handle, transA, m, nnz, descrA, &
          csrValA, csrRowPtrA, csrColIndA, info) bind(C, name='cusparseCcsrsm_analysis')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr, cusparseSolveAnalysisInfo
       type(cusparseHandle), value :: handle       
       integer(c_int), value :: transA, m, nnz
       type(cusparseMatDescr), value :: descrA
       complex(4), device :: csrValA(*)
       integer(c_int), device :: csrRowPtrA(*), csrColIndA(*)
       type(cusparseSolveAnalysisInfo), value :: info
     end function cusparseCcsrsm_analysis
  end interface cusparseCcsrsm_analysis

! cusparseZcsrsm_analysis
  interface cusparseZcsrsm_analysis
     integer(c_int) function cusparseZcsrsm_analysis(handle, transA, m, nnz, descrA, &
          csrValA, csrRowPtrA, csrColIndA, info) bind(C, name='cusparseZcsrsm_analysis')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr, cusparseSolveAnalysisInfo
       type(cusparseHandle), value :: handle       
       integer(c_int), value :: transA, m, nnz
       type(cusparseMatDescr), value :: descrA
       complex(8), device :: csrValA(*)
       integer(c_int), device :: csrRowPtrA(*), csrColIndA(*)
       type(cusparseSolveAnalysisInfo), value :: info
     end function cusparseZcsrsm_analysis
  end interface cusparseZcsrsm_analysis

! cusparseScsrsm_solve
  interface cusparseScsrsm_solve
    integer function cusparseScsrsm_solve_setHPM(handle, transA, m, n, alpha, descrA, &
       csrValA, csrRowPtrA, csrColIndA, info, X, ldx, Y, ldy)
    import cusparseHandle, cusparseMatDescr, cusparseSolveAnalysisInfo
    type(cusparseHandle) :: handle       
    integer :: transA, m, n
    real(4) :: alpha
    type(cusparseMatDescr) :: descrA
    real(4), device :: csrValA(*)
    integer, device :: csrRowPtrA(*), csrColIndA(*)
    type(cusparseSolveAnalysisInfo) :: info
    !dir$ ignore_tkr (r) X, (r) Y
    real(4), device :: X(*), Y(*)
    integer :: ldx, ldy
    end function cusparseScsrsm_solve_setHPM
    integer function cusparseScsrsm_solve_setDPM(handle, transA, m, n, alpha, descrA, &
       csrValA, csrRowPtrA, csrColIndA, info, X, ldx, Y, ldy)
    import cusparseHandle, cusparseMatDescr, cusparseSolveAnalysisInfo
    type(cusparseHandle) :: handle       
    integer :: transA, m, n
    real(4), device :: alpha
    type(cusparseMatDescr) :: descrA
    real(4), device :: csrValA(*)
    integer, device :: csrRowPtrA(*), csrColIndA(*)
    type(cusparseSolveAnalysisInfo) :: info
    !dir$ ignore_tkr (r) X, (r) Y
    real(4), device :: X(*), Y(*)
    integer :: ldx, ldy
    end function cusparseScsrsm_solve_setDPM
  end interface cusparseScsrsm_solve

#ifdef KEEP_V2_INTERFACES
  interface cusparseScsrsm_solve_v2
     integer(c_int) function cusparseScsrsm_solve_v2(handle, transA, m, n, alpha, descrA, &
          csrValA, csrRowPtrA, csrColIndA, info, X, ldx, Y, ldy) bind(C, name='cusparseScsrsm_solve') 
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr, cusparseSolveAnalysisInfo
       type(cusparseHandle), value :: handle       
       integer(c_int), value :: transA, m, n
       !dir$ ignore_tkr (d) alpha
       real(c_float) :: alpha
       type(cusparseMatDescr), value :: descrA
       real(c_float), device :: csrValA(*)
       integer(c_int), device :: csrRowPtrA(*), csrColIndA(*)
       type(cusparseSolveAnalysisInfo), value :: info
       !dir$ ignore_tkr (r) X, (r) Y
       real(c_float), device :: X(*), Y(*)
       integer(c_int), value :: ldx, ldy
     end function cusparseScsrsm_solve_v2
  end interface cusparseScsrsm_solve_v2
#endif

! cusparseDcsrsm_solve
  interface cusparseDcsrsm_solve
    integer function cusparseDcsrsm_solve_setHPM(handle, transA, m, n, alpha, descrA, &
       csrValA, csrRowPtrA, csrColIndA, info, X, ldx, Y, ldy)
    import cusparseHandle, cusparseMatDescr, cusparseSolveAnalysisInfo
    type(cusparseHandle) :: handle       
    integer :: transA, m, n
    real(8) :: alpha
    type(cusparseMatDescr) :: descrA
    real(8), device :: csrValA(*)
    integer, device :: csrRowPtrA(*), csrColIndA(*)
    type(cusparseSolveAnalysisInfo) :: info
    !dir$ ignore_tkr (r) X, (r) Y
    real(8), device :: X(*), Y(*)
    integer :: ldx, ldy
    end function cusparseDcsrsm_solve_setHPM
    integer function cusparseDcsrsm_solve_setDPM(handle, transA, m, n, alpha, descrA, &
       csrValA, csrRowPtrA, csrColIndA, info, X, ldx, Y, ldy)
    import cusparseHandle, cusparseMatDescr, cusparseSolveAnalysisInfo
    type(cusparseHandle) :: handle       
    integer :: transA, m, n
    real(8), device :: alpha
    type(cusparseMatDescr) :: descrA
    real(8), device :: csrValA(*)
    integer, device :: csrRowPtrA(*), csrColIndA(*)
    type(cusparseSolveAnalysisInfo) :: info
    !dir$ ignore_tkr (r) X, (r) Y
    real(8), device :: X(*), Y(*)
    integer :: ldx, ldy
    end function cusparseDcsrsm_solve_setDPM
  end interface cusparseDcsrsm_solve

#ifdef KEEP_V2_INTERFACES
  interface cusparseDcsrsm_solve_v2
     integer(c_int) function cusparseDcsrsm_solve_v2(handle, transA, m, n, alpha, descrA, &
          csrValA, csrRowPtrA, csrColIndA, info, X, ldx, Y, ldy) bind(C, name='cusparseDcsrsm_solve') 
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr, cusparseSolveAnalysisInfo
       type(cusparseHandle), value :: handle       
       integer(c_int), value :: transA, m, n
       !dir$ ignore_tkr (d) alpha
       real(c_double) :: alpha
       type(cusparseMatDescr), value :: descrA
       real(c_double), device :: csrValA(*)
       integer(c_int), device :: csrRowPtrA(*), csrColIndA(*)
       type(cusparseSolveAnalysisInfo), value :: info
       !dir$ IGNORE_TKR (r) X, (r) Y
       real(c_double), device :: X(*), Y(*)
       integer(c_int), value :: ldx, ldy
     end function cusparseDcsrsm_solve_v2
  end interface cusparseDcsrsm_solve_v2
#endif

! cusparseCcsrsm_solve
  interface cusparseCcsrsm_solve
    integer function cusparseCcsrsm_solve_setHPM(handle, transA, m, n, alpha, descrA, &
       csrValA, csrRowPtrA, csrColIndA, info, X, ldx, Y, ldy)
    import cusparseHandle, cusparseMatDescr, cusparseSolveAnalysisInfo
    type(cusparseHandle) :: handle       
    integer :: transA, m, n
    complex(4) :: alpha
    type(cusparseMatDescr) :: descrA
    complex(4), device :: csrValA(*)
    integer, device :: csrRowPtrA(*), csrColIndA(*)
    type(cusparseSolveAnalysisInfo) :: info
    !dir$ ignore_tkr (r) X, (r) Y
    complex(4), device :: X(*), Y(*)
    integer :: ldx, ldy
    end function cusparseCcsrsm_solve_setHPM
    integer function cusparseCcsrsm_solve_setDPM(handle, transA, m, n, alpha, descrA, &
       csrValA, csrRowPtrA, csrColIndA, info, X, ldx, Y, ldy)
    import cusparseHandle, cusparseMatDescr, cusparseSolveAnalysisInfo
    type(cusparseHandle) :: handle       
    integer :: transA, m, n
    complex(4), device :: alpha
    type(cusparseMatDescr) :: descrA
    complex(4), device :: csrValA(*)
    integer, device :: csrRowPtrA(*), csrColIndA(*)
    type(cusparseSolveAnalysisInfo) :: info
    !dir$ ignore_tkr (r) X, (r) Y
    complex(4), device :: X(*), Y(*)
    integer :: ldx, ldy
    end function cusparseCcsrsm_solve_setDPM
  end interface cusparseCcsrsm_solve

#ifdef KEEP_V2_INTERFACES
  interface cusparseCcsrsm_solve_v2
     integer(c_int) function cusparseCcsrsm_solve_v2(handle, transA, m, n, alpha, descrA, &
          csrValA, csrRowPtrA, csrColIndA, info, X, ldx, Y, ldy) bind(C, name='cusparseCcsrsm_solve') 
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr, cusparseSolveAnalysisInfo
       type(cusparseHandle), value :: handle       
       integer(c_int), value :: transA, m, n
       !dir$ ignore_tkr (d) alpha
       complex(4) :: alpha
       type(cusparseMatDescr), value :: descrA
       complex(4), device :: csrValA(*)
       integer(c_int), device :: csrRowPtrA(*), csrColIndA(*)
       type(cusparseSolveAnalysisInfo), value :: info
       !dir$ IGNORE_TKR (r) X, (r) Y
       complex(4), device :: X(*), Y(*)
       integer(c_int), value :: ldx, ldy
     end function cusparseCcsrsm_solve_v2
  end interface cusparseCcsrsm_solve_v2
#endif

! cusparseZcsrsm_solve
  interface cusparseZcsrsm_solve
    integer function cusparseZcsrsm_solve_setHPM(handle, transA, m, n, alpha, descrA, &
       csrValA, csrRowPtrA, csrColIndA, info, X, ldx, Y, ldy)
    import cusparseHandle, cusparseMatDescr, cusparseSolveAnalysisInfo
    type(cusparseHandle) :: handle       
    integer :: transA, m, n
    complex(8) :: alpha
    type(cusparseMatDescr) :: descrA
    complex(8), device :: csrValA(*)
    integer, device :: csrRowPtrA(*), csrColIndA(*)
    type(cusparseSolveAnalysisInfo) :: info
    !dir$ ignore_tkr (r) X, (r) Y
    complex(8), device :: X(*), Y(*)
    integer :: ldx, ldy
    end function cusparseZcsrsm_solve_setHPM
    integer function cusparseZcsrsm_solve_setDPM(handle, transA, m, n, alpha, descrA, &
       csrValA, csrRowPtrA, csrColIndA, info, X, ldx, Y, ldy)
    import cusparseHandle, cusparseMatDescr, cusparseSolveAnalysisInfo
    type(cusparseHandle) :: handle       
    integer :: transA, m, n
    complex(8), device :: alpha
    type(cusparseMatDescr) :: descrA
    complex(8), device :: csrValA(*)
    integer, device :: csrRowPtrA(*), csrColIndA(*)
    type(cusparseSolveAnalysisInfo) :: info
    !dir$ ignore_tkr (r) X, (r) Y
    complex(8), device :: X(*), Y(*)
    integer :: ldx, ldy
    end function cusparseZcsrsm_solve_setDPM
  end interface cusparseZcsrsm_solve

#ifdef KEEP_V2_INTERFACES
  interface cusparseZcsrsm_solve_v2
     integer(c_int) function cusparseZcsrsm_solve_v2(handle, transA, m, n, alpha, descrA, &
          csrValA, csrRowPtrA, csrColIndA, info, X, ldx, Y, ldy) bind(C, name='cusparseZcsrsm_solve') 
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr, cusparseSolveAnalysisInfo
       type(cusparseHandle), value :: handle       
       integer(c_int), value :: transA, m, n
       !dir$ ignore_tkr (d) alpha
       complex(8) :: alpha
       type(cusparseMatDescr), value :: descrA
       complex(8), device :: csrValA(*)
       integer(c_int), device :: csrRowPtrA(*), csrColIndA(*)
       type(cusparseSolveAnalysisInfo), value :: info
       !dir$ IGNORE_TKR (r) X, (r) Y
       complex(8), device :: X(*), Y(*)
       integer(c_int), value :: ldx, ldy
     end function cusparseZcsrsm_solve_v2
  end interface cusparseZcsrsm_solve_v2
#endif

! cusparseCreateCsrsm2Info
  interface cusparseCreateCsrsm2Info
     integer(c_int) function cusparseCreateCsrsm2Info(info) bind(C, name='cusparseCreateCsrsm2Info')
       import cusparseCsrsm2Info
       type(cusparseCsrsm2Info) :: info
     end function cusparseCreateCsrsm2Info
  end interface cusparseCreateCsrsm2Info

! cusparseDestroyCsrsm2Info
  interface cusparseDestroyCsrsm2Info
     integer(c_int) function cusparseDestroyCsrsm2Info(info) bind(C, name='cusparseDestroyCsrsm2Info')
       import cusparseCsrsm2Info
       type(cusparseCsrsm2Info), value :: info
     end function cusparseDestroyCsrsm2Info
  end interface cusparseDestroyCsrsm2Info

! Xcsrxm2_bufferSizeExt
! cusparseScsrsm2_bufferSizeExt
  interface cusparseScsrsm2_bufferSizeExt
     integer(c_int) function cusparseScsrsm2_bufferSizeExt(handle, algo, transA, transB, &
          m, nrhs, nnz, alpha, descrA, csrSortedValA, csrSortedRowPtrA, csrSortedColIndA, &
          B, ldb, info, policy, pBufferSize) bind(C, name='cusparseScsrsm2_bufferSizeExt')
       import cusparseHandle, cusparseMatDescr, cusparseCsrsm2Info
       type(cusparseHandle), value :: handle       
       integer(4), value :: algo, transA, transB, m, nrhs, nnz, ldb, policy
       !dir$ ignore_tkr (d) alpha
       real(4) :: alpha
       type(cusparseMatDescr), value :: descrA
       real(4), device :: csrSortedValA(*)
       integer(4), device :: csrSortedRowPtrA(*), csrSortedColIndA(*)
       !dir$ ignore_tkr (r) B
       real(4), device :: B(ldb,*)
       type(cusparseCsrsm2Info), value :: info
       integer(8) :: pBufferSize
     end function cusparseScsrsm2_bufferSizeExt
  end interface cusparseScsrsm2_bufferSizeExt

! cusparseDcsrsm2_bufferSizeExt
  interface cusparseDcsrsm2_bufferSizeExt
     integer(c_int) function cusparseDcsrsm2_bufferSizeExt(handle, algo, transA, transB, &
          m, nrhs, nnz, alpha, descrA, csrSortedValA, csrSortedRowPtrA, csrSortedColIndA, &
          B, ldb, info, policy, pBufferSize) bind(C, name='cusparseDcsrsm2_bufferSizeExt')
       import cusparseHandle, cusparseMatDescr, cusparseCsrsm2Info
       type(cusparseHandle), value :: handle       
       integer(4), value :: algo, transA, transB, m, nrhs, nnz, ldb, policy
       !dir$ ignore_tkr (d) alpha
       real(8) :: alpha
       type(cusparseMatDescr), value :: descrA
       real(8), device :: csrSortedValA(*)
       integer(4), device :: csrSortedRowPtrA(*), csrSortedColIndA(*)
       !dir$ ignore_tkr (r) B
       real(8), device :: B(ldb,*)
       type(cusparseCsrsm2Info), value :: info
       integer(8) :: pBufferSize
     end function cusparseDcsrsm2_bufferSizeExt
  end interface cusparseDcsrsm2_bufferSizeExt

! cusparseCcsrsm2_bufferSizeExt
  interface cusparseCcsrsm2_bufferSizeExt
     integer(c_int) function cusparseCcsrsm2_bufferSizeExt(handle, algo, transA, transB, &
          m, nrhs, nnz, alpha, descrA, csrSortedValA, csrSortedRowPtrA, csrSortedColIndA, &
          B, ldb, info, policy, pBufferSize) bind(C, name='cusparseCcsrsm2_bufferSizeExt')
       import cusparseHandle, cusparseMatDescr, cusparseCsrsm2Info
       type(cusparseHandle), value :: handle       
       integer(4), value :: algo, transA, transB, m, nrhs, nnz, ldb, policy
       !dir$ ignore_tkr (d) alpha
       complex(4) :: alpha
       type(cusparseMatDescr), value :: descrA
       complex(4), device :: csrSortedValA(*)
       integer(4), device :: csrSortedRowPtrA(*), csrSortedColIndA(*)
       !dir$ ignore_tkr (r) B
       complex(4), device :: B(ldb,*)
       type(cusparseCsrsm2Info), value :: info
       integer(8) :: pBufferSize
     end function cusparseCcsrsm2_bufferSizeExt
  end interface cusparseCcsrsm2_bufferSizeExt

! cusparseZcsrsm2_bufferSizeExt
  interface cusparseZcsrsm2_bufferSizeExt
     integer(c_int) function cusparseZcsrsm2_bufferSizeExt(handle, algo, transA, transB, &
          m, nrhs, nnz, alpha, descrA, csrSortedValA, csrSortedRowPtrA, csrSortedColIndA, &
          B, ldb, info, policy, pBufferSize) bind(C, name='cusparseZcsrsm2_bufferSizeExt')
       import cusparseHandle, cusparseMatDescr, cusparseCsrsm2Info
       type(cusparseHandle), value :: handle       
       integer(4), value :: algo, transA, transB, m, nrhs, nnz, ldb, policy
       !dir$ ignore_tkr (d) alpha
       complex(4) :: alpha
       type(cusparseMatDescr), value :: descrA
       complex(4), device :: csrSortedValA(*)
       integer(4), device :: csrSortedRowPtrA(*), csrSortedColIndA(*)
       !dir$ ignore_tkr (r) B
       complex(4), device :: B(ldb,*)
       type(cusparseCsrsm2Info), value :: info
       integer(8) :: pBufferSize
     end function cusparseZcsrsm2_bufferSizeExt
  end interface cusparseZcsrsm2_bufferSizeExt

! Xcsrxm2_analysis
! cusparseScsrsm2_analysis
  interface cusparseScsrsm2_analysis
     integer(c_int) function cusparseScsrsm2_analysis(handle, algo, transA, transB, &
          m, nrhs, nnz, alpha, descrA, csrSortedValA, csrSortedRowPtrA, csrSortedColIndA, &
          B, ldb, info, policy, pBuffer) bind(C, name='cusparseScsrsm2_analysis')
       import cusparseHandle, cusparseMatDescr, cusparseCsrsm2Info
       type(cusparseHandle), value :: handle       
       integer(4), value :: algo, transA, transB, m, nrhs, nnz, ldb, policy
       !dir$ ignore_tkr (d) alpha
       real(4) :: alpha
       type(cusparseMatDescr), value :: descrA
       real(4), device :: csrSortedValA(*)
       integer(4), device :: csrSortedRowPtrA(*), csrSortedColIndA(*)
       !dir$ ignore_tkr (r) B
       real(4), device :: B(ldb,*)
       type(cusparseCsrsm2Info), value :: info
       !dir$ ignore_tkr pBuffer
       character, device :: pBuffer(*)       
     end function cusparseScsrsm2_analysis
  end interface cusparseScsrsm2_analysis

! cusparseDcsrsm2_analysis
  interface cusparseDcsrsm2_analysis
     integer(c_int) function cusparseDcsrsm2_analysis(handle, algo, transA, transB, &
          m, nrhs, nnz, alpha, descrA, csrSortedValA, csrSortedRowPtrA, csrSortedColIndA, &
          B, ldb, info, policy, pBuffer) bind(C, name='cusparseDcsrsm2_analysis')
       import cusparseHandle, cusparseMatDescr, cusparseCsrsm2Info
       type(cusparseHandle), value :: handle       
       integer(4), value :: algo, transA, transB, m, nrhs, nnz, ldb, policy
       !dir$ ignore_tkr (d) alpha
       real(8) :: alpha
       type(cusparseMatDescr), value :: descrA
       real(8), device :: csrSortedValA(*)
       integer(4), device :: csrSortedRowPtrA(*), csrSortedColIndA(*)
       !dir$ ignore_tkr (r) B
       real(8), device :: B(ldb,*)
       type(cusparseCsrsm2Info), value :: info
       !dir$ ignore_tkr pBuffer
       character, device :: pBuffer(*)       
     end function cusparseDcsrsm2_analysis
  end interface cusparseDcsrsm2_analysis

! cusparseCcsrsm2_analysis
  interface cusparseCcsrsm2_analysis
     integer(c_int) function cusparseCcsrsm2_analysis(handle, algo, transA, transB, &
          m, nrhs, nnz, alpha, descrA, csrSortedValA, csrSortedRowPtrA, csrSortedColIndA, &
          B, ldb, info, policy, pBuffer) bind(C, name='cusparseCcsrsm2_analysis')
       import cusparseHandle, cusparseMatDescr, cusparseCsrsm2Info
       type(cusparseHandle), value :: handle       
       integer(4), value :: algo, transA, transB, m, nrhs, nnz, ldb, policy
       !dir$ ignore_tkr (d) alpha
       complex(4) :: alpha
       type(cusparseMatDescr), value :: descrA
       complex(4), device :: csrSortedValA(*)
       integer(4), device :: csrSortedRowPtrA(*), csrSortedColIndA(*)
       !dir$ ignore_tkr (r) B
       complex(4), device :: B(ldb,*)
       type(cusparseCsrsm2Info), value :: info
       !dir$ ignore_tkr pBuffer
       character, device :: pBuffer(*)       
     end function cusparseCcsrsm2_analysis
  end interface cusparseCcsrsm2_analysis

! cusparseZcsrsm2_analysis
  interface cusparseZcsrsm2_analysis
     integer(c_int) function cusparseZcsrsm2_analysis(handle, algo, transA, transB, &
          m, nrhs, nnz, alpha, descrA, csrSortedValA, csrSortedRowPtrA, csrSortedColIndA, &
          B, ldb, info, policy, pBuffer) bind(C, name='cusparseZcsrsm2_analysis')
       import cusparseHandle, cusparseMatDescr, cusparseCsrsm2Info
       type(cusparseHandle), value :: handle       
       integer(4), value :: algo, transA, transB, m, nrhs, nnz, ldb, policy
       !dir$ ignore_tkr (d) alpha
       complex(8) :: alpha
       type(cusparseMatDescr), value :: descrA
       complex(8), device :: csrSortedValA(*)
       integer(4), device :: csrSortedRowPtrA(*), csrSortedColIndA(*)
       !dir$ ignore_tkr (r) B
       complex(8), device :: B(ldb,*)
       type(cusparseCsrsm2Info), value :: info
       !dir$ ignore_tkr pBuffer
       character, device :: pBuffer(*)       
     end function cusparseZcsrsm2_analysis
  end interface cusparseZcsrsm2_analysis

! Xcsrxm2_solve
! cusparseScsrsm2_solve
  interface cusparseScsrsm2_solve
     integer(c_int) function cusparseScsrsm2_solve(handle, algo, transA, transB, &
          m, nrhs, nnz, alpha, descrA, csrSortedValA, csrSortedRowPtrA, csrSortedColIndA, &
          B, ldb, info, policy, pBuffer) bind(C, name='cusparseScsrsm2_solve')
       import cusparseHandle, cusparseMatDescr, cusparseCsrsm2Info
       type(cusparseHandle), value :: handle       
       integer(4), value :: algo, transA, transB, m, nrhs, nnz, ldb, policy
       !dir$ ignore_tkr (d) alpha
       real(4) :: alpha
       type(cusparseMatDescr), value :: descrA
       real(4), device :: csrSortedValA(*)
       integer(4), device :: csrSortedRowPtrA(*), csrSortedColIndA(*)
       !dir$ ignore_tkr (r) B
       real(4), device :: B(ldb,*)
       type(cusparseCsrsm2Info), value :: info
       !dir$ ignore_tkr pBuffer
       character, device :: pBuffer(*)       
     end function cusparseScsrsm2_solve
  end interface cusparseScsrsm2_solve

! cusparseDcsrsm2_solve
  interface cusparseDcsrsm2_solve
     integer(c_int) function cusparseDcsrsm2_solve(handle, algo, transA, transB, &
          m, nrhs, nnz, alpha, descrA, csrSortedValA, csrSortedRowPtrA, csrSortedColIndA, &
          B, ldb, info, policy, pBuffer) bind(C, name='cusparseDcsrsm2_solve')
       import cusparseHandle, cusparseMatDescr, cusparseCsrsm2Info
       type(cusparseHandle), value :: handle       
       integer(4), value :: algo, transA, transB, m, nrhs, nnz, ldb, policy
       !dir$ ignore_tkr (d) alpha
       real(8) :: alpha
       type(cusparseMatDescr), value :: descrA
       real(8), device :: csrSortedValA(*)
       integer(4), device :: csrSortedRowPtrA(*), csrSortedColIndA(*)
       !dir$ ignore_tkr (r) B
       real(8), device :: B(ldb,*)
       type(cusparseCsrsm2Info), value :: info
       !dir$ ignore_tkr pBuffer
       character, device :: pBuffer(*)       
     end function cusparseDcsrsm2_solve
  end interface cusparseDcsrsm2_solve

! cusparseCcsrsm2_solve
  interface cusparseCcsrsm2_solve
     integer(c_int) function cusparseCcsrsm2_solve(handle, algo, transA, transB, &
          m, nrhs, nnz, alpha, descrA, csrSortedValA, csrSortedRowPtrA, csrSortedColIndA, &
          B, ldb, info, policy, pBuffer) bind(C, name='cusparseCcsrsm2_solve')
       import cusparseHandle, cusparseMatDescr, cusparseCsrsm2Info
       type(cusparseHandle), value :: handle       
       integer(4), value :: algo, transA, transB, m, nrhs, nnz, ldb, policy
       !dir$ ignore_tkr (d) alpha
       complex(4) :: alpha
       type(cusparseMatDescr), value :: descrA
       complex(4), device :: csrSortedValA(*)
       integer(4), device :: csrSortedRowPtrA(*), csrSortedColIndA(*)
       !dir$ ignore_tkr (r) B
       complex(4), device :: B(ldb,*)
       type(cusparseCsrsm2Info), value :: info
       !dir$ ignore_tkr pBuffer
       character, device :: pBuffer(*)       
     end function cusparseCcsrsm2_solve
  end interface cusparseCcsrsm2_solve

! cusparseZcsrsm2_solve
  interface cusparseZcsrsm2_solve
     integer(c_int) function cusparseZcsrsm2_solve(handle, algo, transA, transB, &
          m, nrhs, nnz, alpha, descrA, csrSortedValA, csrSortedRowPtrA, csrSortedColIndA, &
          B, ldb, info, policy, pBuffer) bind(C, name='cusparseZcsrsm2_solve')
       import cusparseHandle, cusparseMatDescr, cusparseCsrsm2Info
       type(cusparseHandle), value :: handle       
       integer(4), value :: algo, transA, transB, m, nrhs, nnz, ldb, policy
       !dir$ ignore_tkr (d) alpha
       complex(8) :: alpha
       type(cusparseMatDescr), value :: descrA
       complex(8), device :: csrSortedValA(*)
       integer(4), device :: csrSortedRowPtrA(*), csrSortedColIndA(*)
       !dir$ ignore_tkr (r) B
       complex(8), device :: B(ldb,*)
       type(cusparseCsrsm2Info), value :: info
       !dir$ ignore_tkr pBuffer
       character, device :: pBuffer(*)       
     end function cusparseZcsrsm2_solve
  end interface cusparseZcsrsm2_solve

! cusparseXcsrsm2_zeroPivot
  interface cusparseXcsrsm2_zeroPivot
     integer(c_int) function cusparseXcsrsm2_zeroPivot(handle, info, position) &
          bind(C, name='cusparseXcsrsm2_zeroPivot')
       import cusparseHandle, cusparseCsrsm2Info
       type(cusparseHandle), value :: handle       
       type(cusparseCsrsm2Info), value :: info
       !dir$ ignore_tkr (d) position
       integer(4) :: position
     end function cusparseXcsrsm2_zeroPivot
  end interface cusparseXcsrsm2_zeroPivot

  ! ---------------------
  ! 6.0 Level 3 functions
  ! ---------------------
  ! none


  ! ---------------------
  ! 6.5 Level 3 functions
  ! ---------------------

! cusparseSbsrmm
  interface cusparseSbsrmm
     integer function cusparseSbsrmm_setHPM(handle, dirA, transA, transB, mb, n, &
          kb, nnzb, alpha, descrA, bsrValA, bsrRowPtrA, bsrColIndA, blockDim, B, ldb, beta, C, ldc)
       import cusparseHandle, cusparseMatDescr
       type(cusparseHandle) :: handle
       integer  :: dirA, transA, transB, mb, n, kb, nnzb, blockDim
       real(4) :: alpha, beta
       type(cusparseMatDescr) :: descrA
       !dir$ ignore_tkr (r) B, (r) C
       real(4), device :: bsrValA(*), B(*), C(*)
       integer, device :: bsrRowPtrA(*), bsrColIndA(*)
       integer :: ldb, ldc
     end function cusparseSbsrmm_setHPM

     integer function cusparseSbsrmm_setDPM(handle, dirA, transA, transB, mb, n, kb, nnzb, alpha, &
          descrA, bsrValA, bsrRowPtrA, bsrColIndA, blockDim, B, ldb, beta, C, ldc)
       import cusparseHandle, cusparseMatDescr
       type(cusparseHandle) :: handle
       integer  :: dirA, transA, transB, mb, n, kb, nnzb, blockDim
       real(4), device :: alpha, beta
       type(cusparseMatDescr) :: descrA
       !dir$ ignore_tkr (r) B, (r) C
       real(4), device :: bsrValA(*), B(*), C(*)
       integer, device :: bsrRowPtrA(*), bsrColIndA(*)
       integer :: ldb, ldc
     end function cusparseSbsrmm_setDPM
  end interface cusparseSbsrmm

#ifdef KEEP_V2_INTERFACES
  interface cusparseSbsrmm_v2
     integer(c_int) function cusparseSbsrmm_v2(handle, dirA, transA, transB, &
          mb, n, kb, nnzb, alpha, &
          descrA, bsrValA, bsrRowPtrA, bsrColIndA, blockDim, B, ldb, beta, C, ldc) &
          bind(C, name='cusparseSbsrmm')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr
       type(cusparseHandle), value :: handle
       integer(c_int), value :: dirA, transA, transB, mb, n, kb, nnzb, blockDim
       !dir$ ignore_tkr (d) alpha, (d) beta
       real(c_float) :: alpha, beta
       type(cusparseMatDescr), value :: descrA
       !dir$ ignore_tkr (r) B, (r) C
       real(c_float), device :: bsrValA(*), B(*), C(*)
       integer(c_int), device :: bsrRowPtrA(*), bsrColIndA(*)
       integer(c_int), value :: ldb, ldc
     end function cusparseSbsrmm_v2
  end interface cusparseSbsrmm_v2
#endif

! cusparseDbsrmm
  interface cusparseDbsrmm
     integer function cusparseDbsrmm_setHPM(handle, dirA, transA, transB, mb, n, kb, nnzb, alpha, &
          descrA, bsrValA, bsrRowPtrA, bsrColIndA, blockDim, B, ldb, beta, C, ldc)
       import cusparseHandle, cusparseMatDescr
       type(cusparseHandle) :: handle
       integer  :: dirA, transA, transB, mb, n, kb, nnzb, blockDim
       real(8) :: alpha, beta
       type(cusparseMatDescr) :: descrA
       !dir$ ignore_tkr (r) B, (r) C
       real(8), device :: bsrValA(*), B(*), C(*)
       integer, device :: bsrRowPtrA(*), bsrColIndA(*)
       integer :: ldb, ldc
     end function cusparseDbsrmm_setHPM

     integer function cusparseDbsrmm_setDPM(handle, dirA, transA, transB, mb, n, kb, nnzb, alpha, &
       descrA, bsrValA, bsrRowPtrA, bsrColIndA, blockDim, B, ldb, beta, C, ldc)
       import cusparseHandle, cusparseMatDescr
       type(cusparseHandle) :: handle
       integer  :: dirA, transA, transB, mb, n, kb, nnzb, blockDim
       real(8), device :: alpha, beta
       type(cusparseMatDescr) :: descrA
       !dir$ ignore_tkr (r) B, (r) C
       real(8), device :: bsrValA(*), B(*), C(*)
       integer, device :: bsrRowPtrA(*), bsrColIndA(*)
       integer :: ldb, ldc
     end function cusparseDbsrmm_setDPM
  end interface cusparseDbsrmm

#ifdef KEEP_V2_INTERFACES
  interface cusparseDbsrmm_v2
     integer(c_int) function cusparseDbsrmm_v2(handle, dirA, transA, transB, &
          mb, n, kb, nnzb, alpha, &
          descrA, bsrValA, bsrRowPtrA, bsrColIndA, blockDim, B, ldb, beta, C, ldc) &
          bind(C, name='cusparseDbsrmm')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr
       type(cusparseHandle), value :: handle
       integer(c_int), value :: dirA, transA, transB, mb, n, kb, nnzb, blockDim
       !dir$ ignore_tkr (d) alpha, (d) beta
       real(c_double) :: alpha, beta
       type(cusparseMatDescr), value :: descrA
       !dir$ ignore_tkr (r) B, (r) C
       real(c_double), device :: bsrValA(*), B(*), C(*)
       integer(c_int), device :: bsrRowPtrA(*), bsrColIndA(*)
       integer(c_int), value :: ldb, ldc
     end function cusparseDbsrmm_v2
  end interface cusparseDbsrmm_v2
#endif

! cusparseCbsrmm
  interface cusparseCbsrmm
     integer function cusparseCbsrmm_setHPM(handle, dirA, transA, transB, mb, n, kb, nnzb, alpha, &
       descrA, bsrValA, bsrRowPtrA, bsrColIndA, blockDim, B, ldb, beta, C, ldc)
       import cusparseHandle, cusparseMatDescr
       type(cusparseHandle) :: handle
       integer  :: dirA, transA, transB, mb, n, kb, nnzb, blockDim
       complex(4) :: alpha, beta
       type(cusparseMatDescr) :: descrA
       !dir$ ignore_tkr (r) B, (r) C
       complex(4), device :: bsrValA(*), B(*), C(*)
       integer, device :: bsrRowPtrA(*), bsrColIndA(*)
       integer :: ldb, ldc
     end function cusparseCbsrmm_setHPM

     integer function cusparseCbsrmm_setDPM(handle, dirA, transA, transB, mb, n, kb, nnzb, alpha, &
          descrA, bsrValA, bsrRowPtrA, bsrColIndA, blockDim, B, ldb, beta, C, ldc)
       import cusparseHandle, cusparseMatDescr
       type(cusparseHandle) :: handle
       integer  :: dirA, transA, transB, mb, n, kb, nnzb, blockDim
       complex(4), device :: alpha, beta
       type(cusparseMatDescr) :: descrA
       !dir$ ignore_tkr (r) B, (r) C
       complex(4), device :: bsrValA(*), B(*), C(*)
       integer, device :: bsrRowPtrA(*), bsrColIndA(*)
       integer :: ldb, ldc
     end function cusparseCbsrmm_setDPM
  end interface cusparseCbsrmm

#ifdef KEEP_V2_INTERFACES
  interface cusparseCbsrmm_v2
     integer(c_int) function cusparseCbsrmm_v2(handle, dirA, transA, transB, &
          mb, n, kb, nnzb, alpha, &
          descrA, bsrValA, bsrRowPtrA, bsrColIndA, blockDim, B, ldb, beta, C, ldc) &
          bind(C, name='cusparseCbsrmm')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr
       type(cusparseHandle), value :: handle
       integer(c_int), value :: dirA, transA, transB, mb, n, kb, nnzb, blockDim
       !dir$ ignore_tkr (d) alpha, (d) beta
       complex(4) :: alpha, beta
       type(cusparseMatDescr), value :: descrA
       !dir$ ignore_tkr (r) B, (r) C
       complex(4), device :: bsrValA(*), B(*), C(*)
       integer(c_int), device :: bsrRowPtrA(*), bsrColIndA(*)
       integer(c_int), value :: ldb, ldc
     end function cusparseCbsrmm_v2
  end interface cusparseCbsrmm_v2
#endif

! cusparseZbsrmm
  interface cusparseZbsrmm
     integer function cusparseZbsrmm_setHPM(handle, dirA, transA, transB, mb, n, kb, nnzb, alpha, &
          descrA, bsrValA, bsrRowPtrA, bsrColIndA, blockDim, B, ldb, beta, C, ldc)
       import cusparseHandle, cusparseMatDescr
       type(cusparseHandle) :: handle
       integer  :: dirA, transA, transB, mb, n, kb, nnzb, blockDim
       complex(8) :: alpha, beta
       type(cusparseMatDescr) :: descrA
       !dir$ ignore_tkr (r) B, (r) C
       complex(8), device :: bsrValA(*), B(*), C(*)
       integer, device :: bsrRowPtrA(*), bsrColIndA(*)
       integer :: ldb, ldc
     end function cusparseZbsrmm_setHPM

     integer function cusparseZbsrmm_setDPM(handle, dirA, transA, transB, mb, n, kb, nnzb, alpha, &
       descrA, bsrValA, bsrRowPtrA, bsrColIndA, blockDim, B, ldb, beta, C, ldc)
       import cusparseHandle, cusparseMatDescr
       type(cusparseHandle) :: handle
       integer  :: dirA, transA, transB, mb, n, kb, nnzb, blockDim
       complex(8), device :: alpha, beta
       type(cusparseMatDescr) :: descrA
       !dir$ ignore_tkr (r) B, (r) C
       complex(8), device :: bsrValA(*), B(*), C(*)
       integer, device :: bsrRowPtrA(*), bsrColIndA(*)
       integer :: ldb, ldc
     end function cusparseZbsrmm_setDPM
  end interface cusparseZbsrmm

#ifdef KEEP_V2_INTERFACES
  interface cusparseZbsrmm_v2
     integer(c_int) function cusparseZbsrmm_v2(handle, dirA, transA, transB, &
          mb, n, kb, nnzb, alpha, &
          descrA, bsrValA, bsrRowPtrA, bsrColIndA, blockDim, B, ldb, beta, C, ldc) &
          bind(C, name='cusparseZbsrmm')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr
       type(cusparseHandle), value :: handle
       integer(c_int), value :: dirA, transA, transB, mb, n, kb, nnzb, blockDim
       !dir$ ignore_tkr (d) alpha, (d) beta
       complex(8) :: alpha, beta
       type(cusparseMatDescr), value :: descrA
       !dir$ ignore_tkr (r) B, (r) C
       complex(8), device :: bsrValA(*), B(*), C(*)
       integer(c_int), device :: bsrRowPtrA(*), bsrColIndA(*)
       integer(c_int), value :: ldb, ldc
     end function cusparseZbsrmm_v2
  end interface cusparseZbsrmm_v2
#endif

! cusparseSbsrsm2_bufferSize
  interface cusparseSbsrsm2_bufferSize
     integer(c_int) function cusparseSbsrsm2_bufferSize(handle, dirA, transA, transX, mb, n, nnzb, descrA, &
          bsrValA, bsrRowPtrA, bsrColIndA, blockDim, info, pBufferSize) bind(C,name='cusparseSbsrsm2_bufferSize')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr, cusparseBsrsm2Info
       type(cusparseHandle), value :: handle
       integer(c_int), value :: dirA, transA, transX, mb, n, nnzb, blockDim
       type(cusparseMatDescr), value :: descrA
       real(c_float), device :: bsrValA(*)
       integer(c_int), device :: bsrRowPtrA(*), bsrColIndA(*)
       type(cusparseBsrsm2Info), value :: info
       integer(4) :: pBufferSize
     end function cusparseSbsrsm2_bufferSize
     integer(c_int) function cusparseSbsrsm2_bufferSizeExt(handle, dirA, transA, transX, mb, n, nnzb, descrA, &
          bsrValA, bsrRowPtrA, bsrColIndA, blockDim, info, pBufferSize) bind(C,name='cusparseSbsrsm2_bufferSizeExt')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr, cusparseBsrsm2Info
       type(cusparseHandle), value :: handle
       integer(c_int), value :: dirA, transA, transX, mb, n, nnzb, blockDim
       type(cusparseMatDescr), value :: descrA
       real(c_float), device :: bsrValA(*)
       integer(c_int), device :: bsrRowPtrA(*), bsrColIndA(*)
       type(cusparseBsrsm2Info), value :: info
       integer(8) :: pBufferSize
     end function cusparseSbsrsm2_bufferSizeExt
  end interface

! cusparseDbsrsm2_bufferSize
  interface cusparseDbsrsm2_bufferSize
     integer(c_int) function cusparseDbsrsm2_bufferSize(handle, dirA, transA, transX, mb, n, nnzb, descrA, &
          bsrValA, bsrRowPtrA, bsrColIndA, blockDim, info, pBufferSize) bind(C,name='cusparseDbsrsm2_bufferSize')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr, cusparseBsrsm2Info
       type(cusparseHandle), value :: handle
       integer(c_int), value :: dirA, transA, transX, mb, n, nnzb, blockDim
       type(cusparseMatDescr), value :: descrA
       real(c_double), device :: bsrValA(*)
       integer(c_int), device :: bsrRowPtrA(*), bsrColIndA(*)
       type(cusparseBsrsm2Info), value :: info
       integer(4) :: pBufferSize
     end function cusparseDbsrsm2_bufferSize
     integer(c_int) function cusparseDbsrsm2_bufferSizeExt(handle, dirA, transA, transX, mb, n, nnzb, descrA, &
          bsrValA, bsrRowPtrA, bsrColIndA, blockDim, info, pBufferSize) bind(C,name='cusparseDbsrsm2_bufferSizeExt')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr, cusparseBsrsm2Info
       type(cusparseHandle), value :: handle
       integer(c_int), value :: dirA, transA, transX, mb, n, nnzb, blockDim
       type(cusparseMatDescr), value :: descrA
       real(c_double), device :: bsrValA(*)
       integer(c_int), device :: bsrRowPtrA(*), bsrColIndA(*)
       type(cusparseBsrsm2Info), value :: info
       integer(8) :: pBufferSize
     end function cusparseDbsrsm2_bufferSizeExt
  end interface

! cusparseCbsrsm2_bufferSize
  interface cusparseCbsrsm2_bufferSize
     integer(c_int) function cusparseCbsrsm2_bufferSize(handle, dirA, transA, transX, mb, n, nnzb, descrA, &
          bsrValA, bsrRowPtrA, bsrColIndA, blockDim, info, pBufferSize) bind(C,name='cusparseCbsrsm2_bufferSize')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr, cusparseBsrsm2Info
       type(cusparseHandle), value :: handle
       integer(c_int), value :: dirA, transA, transX, mb, n, nnzb, blockDim
       type(cusparseMatDescr), value :: descrA
       complex(4), device :: bsrValA(*)
       integer(c_int), device :: bsrRowPtrA(*), bsrColIndA(*)
       type(cusparseBsrsm2Info), value :: info
       integer(4) :: pBufferSize
     end function cusparseCbsrsm2_bufferSize
     integer(c_int) function cusparseCbsrsm2_bufferSizeExt(handle, dirA, transA, transX, mb, n, nnzb, descrA, &
          bsrValA, bsrRowPtrA, bsrColIndA, blockDim, info, pBufferSize) bind(C,name='cusparseCbsrsm2_bufferSizeExt')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr, cusparseBsrsm2Info
       type(cusparseHandle), value :: handle
       integer(c_int), value :: dirA, transA, transX, mb, n, nnzb, blockDim
       type(cusparseMatDescr), value :: descrA
       complex(4), device :: bsrValA(*)
       integer(c_int), device :: bsrRowPtrA(*), bsrColIndA(*)
       type(cusparseBsrsm2Info), value :: info
       integer(8) :: pBufferSize
     end function cusparseCbsrsm2_bufferSizeExt
  end interface

! cusparseZbsrsm2_bufferSize
  interface cusparseZbsrsm2_bufferSize
     integer(c_int) function cusparseZbsrsm2_bufferSize(handle, dirA, transA, transX, mb, n, nnzb, descrA, &
          bsrValA, bsrRowPtrA, bsrColIndA, blockDim, info, pBufferSize) bind(C,name='cusparseZbsrsm2_bufferSize')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr, cusparseBsrsm2Info
       type(cusparseHandle), value :: handle
       integer(c_int), value :: dirA, transA, transX, mb, n, nnzb, blockDim
       type(cusparseMatDescr), value :: descrA
       complex(8), device :: bsrValA(*)
       integer(c_int), device :: bsrRowPtrA(*), bsrColIndA(*)
       type(cusparseBsrsm2Info), value :: info
       integer(4) :: pBufferSize
     end function cusparseZbsrsm2_bufferSize
     integer(c_int) function cusparseZbsrsm2_bufferSizeExt(handle, dirA, transA, transX, mb, n, nnzb, descrA, &
          bsrValA, bsrRowPtrA, bsrColIndA, blockDim, info, pBufferSize) bind(C,name='cusparseZbsrsm2_bufferSizeExt')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr, cusparseBsrsm2Info
       type(cusparseHandle), value :: handle
       integer(c_int), value :: dirA, transA, transX, mb, n, nnzb, blockDim
       type(cusparseMatDescr), value :: descrA
       complex(8), device :: bsrValA(*)
       integer(c_int), device :: bsrRowPtrA(*), bsrColIndA(*)
       type(cusparseBsrsm2Info), value :: info
       integer(8) :: pBufferSize
     end function cusparseZbsrsm2_bufferSizeExt
  end interface

! cusparseSbsrsm2_analysis
  interface cusparseSbsrsm2_analysis
     integer(c_int) function cusparseSbsrsm2_analysis(handle, dirA, transA, transX, mb, n, nnzb, descrA, &
          bsrValA, bsrRowPtrA, bsrColIndA, blockDim, info, policy, pBuffer) bind(C, name='cusparseSbsrsm2_analysis')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr, cusparseBsrsm2Info
       type(cusparseHandle), value :: handle       
       integer(c_int), value :: dirA, transA, transX, mb, n, nnzb, blockDim
       type(cusparseMatDescr), value :: descrA
       real(c_float), device :: bsrValA(*)
       integer(c_int), device :: bsrRowPtrA(*), bsrColIndA(*)
       type(cusparseBsrsm2Info), value :: info
       integer(c_int), value :: policy
       !dir$ ignore_tkr pBuffer
       character(c_char), device :: pBuffer(*)       
     end function cusparseSbsrsm2_analysis
  end interface cusparseSbsrsm2_analysis

! cusparseDbsrsm2_analysis
  interface cusparseDbsrsm2_analysis
     integer(c_int) function cusparseDbsrsm2_analysis(handle, dirA, transA, transX, mb, n, nnzb, descrA, &
          bsrValA, bsrRowPtrA, bsrColIndA, blockDim, info, policy, pBuffer) bind(C, name='cusparseDbsrsm2_analysis')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr, cusparseBsrsm2Info
       type(cusparseHandle), value :: handle       
       integer(c_int), value :: dirA, transA, transX, mb, n, nnzb, blockDim
       type(cusparseMatDescr), value :: descrA
       real(c_double), device :: bsrValA(*)
       integer(c_int), device :: bsrRowPtrA(*), bsrColIndA(*)
       type(cusparseBsrsm2Info), value :: info
       integer(c_int), value :: policy
       !dir$ ignore_tkr pBuffer
       character(c_char), device :: pBuffer(*)       
     end function cusparseDbsrsm2_analysis
  end interface cusparseDbsrsm2_analysis

! cusparseCbsrsm2_analysis
  interface cusparseCbsrsm2_analysis
     integer(c_int) function cusparseCbsrsm2_analysis(handle, dirA, transA, transX, mb, n, nnzb, descrA, &
          bsrValA, bsrRowPtrA, bsrColIndA, blockDim, info, policy, pBuffer) bind(C, name='cusparseCbsrsm2_analysis')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr, cusparseBsrsm2Info
       type(cusparseHandle), value :: handle       
       integer(c_int), value :: dirA, transA, transX, mb, n, nnzb, blockDim
       type(cusparseMatDescr), value :: descrA
       complex(4), device :: bsrValA(*)
       integer(c_int), device :: bsrRowPtrA(*), bsrColIndA(*)
       type(cusparseBsrsm2Info), value :: info
       integer(c_int), value :: policy
       !dir$ ignore_tkr pBuffer
       character(c_char), device :: pBuffer(*)       
     end function cusparseCbsrsm2_analysis
  end interface cusparseCbsrsm2_analysis

! cusparseZbsrsm2_analysis
  interface cusparseZbsrsm2_analysis
     integer(c_int) function cusparseZbsrsm2_analysis(handle, dirA, transA, transX, mb, n, nnzb, descrA, &
          bsrValA, bsrRowPtrA, bsrColIndA, blockDim, info, policy, pBuffer) bind(C, name='cusparseZbsrsm2_analysis')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr, cusparseBsrsm2Info
       type(cusparseHandle), value :: handle       
       integer(c_int), value :: dirA, transA, transX, mb, n, nnzb, blockDim
       type(cusparseMatDescr), value :: descrA
       complex(8), device :: bsrValA(*)
       integer(c_int), device :: bsrRowPtrA(*), bsrColIndA(*)
       type(cusparseBsrsm2Info), value :: info
       integer(c_int), value :: policy
       !dir$ ignore_tkr pBuffer
       character(c_char), device :: pBuffer(*)       
     end function cusparseZbsrsm2_analysis
  end interface cusparseZbsrsm2_analysis

! cusparseSbsrsm2_solve
  interface cusparseSbsrsm2_solve
     integer function cusparseSbsrsm2_solve_setHPM(handle, dirA, transA, transX, mb, n, nnzb, alpha, &
       descrA, bsrValA, bsrRowPtrA, bsrColIndA, blockDim, info, x, ldx, y, ldy, policy, pBuffer)
       import cusparseHandle, cusparseMatDescr, cusparseBsrsm2Info
       type(cusparseHandle) :: handle
       integer :: dirA, transA, transX, mb, n, nnzb
       real(4) :: alpha 
       type(cusparseMatDescr) :: descrA
       !dir$ ignore_tkr (r) x, (r) y
       real(4), device :: bsrValA(*), x(*), y(*)
       integer, device :: bsrRowPtrA(*), bsrColIndA(*)
       integer :: blockDim, policy, ldx, ldy
       type(cusparseBsrsm2Info) :: info
       !dir$ ignore_tkr pBuffer
       character, device :: pBuffer(*)    
     end function cusparseSbsrsm2_solve_setHPM
  
     integer function cusparseSbsrsm2_solve_setDPM(handle, dirA, transA, transX, mb, n, nnzb, alpha, &
       descrA, bsrValA, bsrRowPtrA, bsrColIndA, blockDim, info, x, ldx, y, ldy, policy, pBuffer)
       import cusparseHandle, cusparseMatDescr, cusparseBsrsm2Info
       type(cusparseHandle) :: handle
       integer :: dirA, transA, transX, mb, n, nnzb
       real(4), device :: alpha 
       type(cusparseMatDescr) :: descrA
       !dir$ ignore_tkr (r) x, (r) y
       real(4), device :: bsrValA(*), x(*), y(*)
       integer, device :: bsrRowPtrA(*), bsrColIndA(*)
       integer :: blockDim, policy, ldx, ldy
       type(cusparseBsrsm2Info) :: info
       !dir$ ignore_tkr pBuffer
       character, device :: pBuffer(*)    
     end function cusparseSbsrsm2_solve_setDPM
  end interface cusparseSbsrsm2_solve

#ifdef KEEP_V2_INTERFACES
  interface
     integer(c_int) function cusparseSbsrsm2_solve_v2(handle, dirA, transA, transX, mb, n, nnzb, alpha, &
          descrA, bsrValA, bsrRowPtrA, bsrColIndA, blockDim, info, x, ldx, y, ldy, policy, pBuffer) bind(C,name='cusparseSbsrsm2_solve')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr, cusparseBsrsm2Info
       type(cusparseHandle), value :: handle
       integer(c_int), value :: dirA, transA, transX, mb, n, nnzb
       !dir$ ignore_tkr (d) alpha
       real(c_float) :: alpha 
       type(cusparseMatDescr), value :: descrA
       !dir$ ignore_tkr (r) x, (r) y
       real(c_float), device :: bsrValA(*), x(*), y(*)
       integer(c_int), device :: bsrRowPtrA(*), bsrColIndA(*)
       integer(c_int), value :: blockDim, policy, ldx, ldy
       type(cusparseBsrsm2Info), value :: info
       !dir$ ignore_tkr pBuffer
       character(c_char), device :: pBuffer(*)
     end function cusparseSbsrsm2_solve_v2
  end interface
#endif

! cusparseDbsrsm2_solve
  interface cusparseDbsrsm2_solve
     integer function cusparseDbsrsm2_solve_setHPM(handle, dirA, transA, transX, mb, n, nnzb, alpha, &
          descrA, bsrValA, bsrRowPtrA, bsrColIndA, blockDim, info, x, ldx, y, ldy, policy, pBuffer)
       import cusparseHandle, cusparseMatDescr, cusparseBsrsm2Info
       type(cusparseHandle) :: handle
       integer :: dirA, transA, transX, mb, n, nnzb
       real(8) :: alpha 
       type(cusparseMatDescr) :: descrA
       !dir$ ignore_tkr (r) x, (r) y
       real(8), device :: bsrValA(*), x(*), y(*)
       integer, device :: bsrRowPtrA(*), bsrColIndA(*)
       integer :: blockDim, policy, ldx, ldy
       type(cusparseBsrsm2Info) :: info
       !dir$ ignore_tkr pBuffer
       character, device :: pBuffer(*)    
     end function cusparseDbsrsm2_solve_setHPM
  
     integer function cusparseDbsrsm2_solve_setDPM(handle, dirA, transA, transX, mb, n, nnzb, alpha, &
       descrA, bsrValA, bsrRowPtrA, bsrColIndA, blockDim, info, x, ldx, y, ldy, policy, pBuffer)
       import cusparseHandle, cusparseMatDescr, cusparseBsrsm2Info
       type(cusparseHandle) :: handle
       integer :: dirA, transA, transX, mb, n, nnzb
       real(8), device :: alpha 
       type(cusparseMatDescr) :: descrA
       !dir$ ignore_tkr (r) x, (r) y
       real(8), device :: bsrValA(*), x(*), y(*)
       integer, device :: bsrRowPtrA(*), bsrColIndA(*)
       integer :: blockDim, policy, ldx, ldy
       type(cusparseBsrsm2Info) :: info
       !dir$ ignore_tkr pBuffer
       character, device :: pBuffer(*)    
     end function cusparseDbsrsm2_solve_setDPM
  end interface cusparseDbsrsm2_solve

#ifdef KEEP_V2_INTERFACES
  interface
     integer(c_int) function cusparseDbsrsm2_solve_v2(handle, dirA, transA, transX, mb, n, nnzb, alpha, &
          descrA, bsrValA, bsrRowPtrA, bsrColIndA, blockDim, info, x, ldx, y, ldy, policy, pBuffer) bind(C,name='cusparseDbsrsm2_solve')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr, cusparseBsrsm2Info
       type(cusparseHandle), value :: handle
       integer(c_int), value :: dirA, transA, transX, mb, n, nnzb
       !dir$ ignore_tkr (d) alpha
       real(c_double) :: alpha 
       type(cusparseMatDescr), value :: descrA
       !dir$ ignore_tkr (r) x, (r) y
       real(c_double), device :: bsrValA(*), x(*), y(*)
       integer(c_int), device :: bsrRowPtrA(*), bsrColIndA(*)
       integer(c_int), value :: blockDim, policy, ldx, ldy
       type(cusparseBsrsm2Info), value :: info
       !dir$ ignore_tkr pBuffer
       character(c_char), device :: pBuffer(*)
     end function cusparseDbsrsm2_solve_v2
  end interface
#endif

! cusparseCbsrsm2_solve
  interface cusparseCbsrsm2_solve
     integer function cusparseCbsrsm2_solve_setHPM(handle, dirA, transA, transX, mb, n, nnzb, alpha, &
          descrA, bsrValA, bsrRowPtrA, bsrColIndA, blockDim, info, x, ldx, y, ldy, policy, pBuffer)
       import cusparseHandle, cusparseMatDescr, cusparseBsrsm2Info
       type(cusparseHandle) :: handle
       integer :: dirA, transA, transX, mb, n, nnzb
       complex(4) :: alpha 
       type(cusparseMatDescr) :: descrA
       !dir$ ignore_tkr (r) x, (r) y
       complex(4), device :: bsrValA(*), x(*), y(*)
       integer, device :: bsrRowPtrA(*), bsrColIndA(*)
       integer :: blockDim, policy, ldx, ldy
       type(cusparseBsrsm2Info) :: info
       !dir$ ignore_tkr pBuffer
       character, device :: pBuffer(*)    
     end function cusparseCbsrsm2_solve_setHPM
  
     integer function cusparseCbsrsm2_solve_setDPM(handle, dirA, transA, transX, mb, n, nnzb, alpha, &
       descrA, bsrValA, bsrRowPtrA, bsrColIndA, blockDim, info, x, ldx, y, ldy, policy, pBuffer)
       import cusparseHandle, cusparseMatDescr, cusparseBsrsm2Info
       type(cusparseHandle) :: handle
       integer :: dirA, transA, transX, mb, n, nnzb
       complex(4), device :: alpha 
       type(cusparseMatDescr) :: descrA
       !dir$ ignore_tkr (r) x, (r) y
       complex(4), device :: bsrValA(*), x(*), y(*)
       integer, device :: bsrRowPtrA(*), bsrColIndA(*)
       integer :: blockDim, policy, ldx, ldy
       type(cusparseBsrsm2Info) :: info
       !dir$ ignore_tkr pBuffer
       character, device :: pBuffer(*)    
     end function cusparseCbsrsm2_solve_setDPM
  end interface cusparseCbsrsm2_solve

#ifdef KEEP_V2_INTERFACES
  interface
     integer(c_int) function cusparseCbsrsm2_solve_v2(handle, dirA, transA, transX, mb, n, nnzb, alpha, &
          descrA, bsrValA, bsrRowPtrA, bsrColIndA, blockDim, info, x, ldx, y, ldy, policy, pBuffer) bind(C,name='cusparseCbsrsm2_solve')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr, cusparseBsrsm2Info
       type(cusparseHandle), value :: handle
       integer(c_int), value :: dirA, transA, transX, mb, n, nnzb
       !dir$ ignore_tkr (d) alpha
       complex(4) :: alpha 
       type(cusparseMatDescr), value :: descrA
       !dir$ ignore_tkr (r) x, (r) y
       complex(4), device :: bsrValA(*), x(*), y(*)
       integer(c_int), device :: bsrRowPtrA(*), bsrColIndA(*)
       integer(c_int), value :: blockDim, policy, ldx, ldy
       type(cusparseBsrsm2Info), value :: info
       !dir$ ignore_tkr pBuffer
       character(c_char), device :: pBuffer(*)
     end function cusparseCbsrsm2_solve_v2
  end interface
#endif

! cusparseZbsrsm2_solve
  interface cusparseZbsrsm2_solve
     integer function cusparseZbsrsm2_solve_setHPM(handle, dirA, transA, transX, mb, n, nnzb, alpha, &
          descrA, bsrValA, bsrRowPtrA, bsrColIndA, blockDim, info, x, ldx, y, ldy, policy, pBuffer)
       import cusparseHandle, cusparseMatDescr, cusparseBsrsm2Info
       type(cusparseHandle) :: handle
       integer :: dirA, transA, transX, mb, n, nnzb
       complex(8) :: alpha 
       type(cusparseMatDescr) :: descrA
       !dir$ ignore_tkr (r) x, (r) y
       complex(8), device :: bsrValA(*), x(*), y(*)
       integer, device :: bsrRowPtrA(*), bsrColIndA(*)
       integer :: blockDim, policy, ldx, ldy
       type(cusparseBsrsm2Info) :: info
       !dir$ ignore_tkr pBuffer
       character, device :: pBuffer(*)    
     end function cusparseZbsrsm2_solve_setHPM
  
     integer function cusparseZbsrsm2_solve_setDPM(handle, dirA, transA, transX, mb, n, nnzb, alpha, &
       descrA, bsrValA, bsrRowPtrA, bsrColIndA, blockDim, info, x, ldx, y, ldy, policy, pBuffer)
       import cusparseHandle, cusparseMatDescr, cusparseBsrsm2Info
       type(cusparseHandle) :: handle
       integer :: dirA, transA, transX, mb, n, nnzb
       complex(8), device :: alpha 
       type(cusparseMatDescr) :: descrA
       !dir$ ignore_tkr (r) x, (r) y
       complex(8), device :: bsrValA(*), x(*), y(*)
       integer, device :: bsrRowPtrA(*), bsrColIndA(*)
       integer :: blockDim, policy, ldx, ldy
       type(cusparseBsrsm2Info) :: info
       !dir$ ignore_tkr pBuffer
       character, device :: pBuffer(*)    
     end function cusparseZbsrsm2_solve_setDPM
  end interface cusparseZbsrsm2_solve

#ifdef KEEP_V2_INTERFACES
  interface
     integer(c_int) function cusparseZbsrsm2_solve_v2(handle, dirA, transA, transX, mb, n, nnzb, alpha, &
          descrA, bsrValA, bsrRowPtrA, bsrColIndA, blockDim, info, x, ldx, y, ldy, policy, pBuffer) bind(C,name='cusparseZbsrsm2_solve')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr, cusparseBsrsm2Info
       type(cusparseHandle), value :: handle
       integer(c_int), value :: dirA, transA, transX, mb, n, nnzb
       !dir$ ignore_tkr (d) alpha
       complex(8) :: alpha 
       type(cusparseMatDescr), value :: descrA
       !dir$ ignore_tkr (r) x, (r) y
       complex(8), device :: bsrValA(*), x(*), y(*)
       integer(c_int), device :: bsrRowPtrA(*), bsrColIndA(*)
       integer(c_int), value :: blockDim, policy, ldx, ldy
       type(cusparseBsrsm2Info), value :: info
       !dir$ ignore_tkr pBuffer
       character(c_char), device :: pBuffer(*)
     end function cusparseZbsrsm2_solve_v2
  end interface
#endif

! cusparseXbsrsm2_zeroPivot
  interface cusparseXbsrsm2_zeroPivot
     integer function cusparseXbsrsm2_zeroPivot_setHPM(handle, info, position)
       import cusparseHandle, cusparseBsrsm2Info
       type(cusparseHandle) :: handle
       type(cusparseBsrsm2Info) :: info
       integer :: position
     end function cusparseXbsrsm2_zeroPivot_setHPM

     integer function cusparseXbsrsm2_zeroPivot_setDPM(handle, info, position)
       import cusparseHandle, cusparseBsrsm2Info
       type(cusparseHandle) :: handle
       type(cusparseBsrsm2Info) :: info    
       integer, device :: position
     end function cusparseXbsrsm2_zeroPivot_setDPM
  end interface cusparseXbsrsm2_zeroPivot

#ifdef KEEP_V2_INTERFACES
  interface 
     integer(c_int) function cusparseXbsrsm2_zeroPivot_v2(handle, info, position) bind(C,name='cusparseXbsrsm2_zeroPivot')
       use iso_c_binding
       import cusparseHandle, cusparseBsrsm2Info
       type(cusparseHandle), value :: handle
       type(cusparseBsrsm2Info), value :: info
       !dir$ ignore_tkr (d) position
       integer(c_int) :: position
     end function cusparseXbsrsm2_zeroPivot_v2
  end interface
#endif

  ! ---------------------
  ! 7.0 Level 3 functions
  ! ---------------------
  ! none

  ! -------------------
  ! 5.5 Extra Functions
  ! -------------------

! cusparseXcsrgeamNnz
  interface cusparseXcsrgeamNnz
    integer function cusparseXcsrgeamNnz_setHPM(handle, m, n, descrA, nnzA, csrRowPtrA, csrColIndA, &
       descrB, nnzB, csrRowPtrB, csrColIndB, descrC, csrRowPtrC, nnzTotalDevHostPtr)
    import cusparseHandle, cusparseMatDescr
    type(cusparseHandle) :: handle       
    integer :: m, n, nnzA, nnzB
    type(cusparseMatDescr) :: descrA, descrB, descrC
    integer, device :: csrRowPtrA(*), csrColIndA(*), csrRowPtrB(*), csrColIndB(*), csrRowPtrC(*)
    integer :: nnzTotalDevHostPtr
    end function cusparseXcsrgeamNnz_setHPM
    integer function cusparseXcsrgeamNnz_setDPM(handle, m, n, descrA, nnzA, csrRowPtrA, csrColIndA, &
       descrB, nnzB, csrRowPtrB, csrColIndB, descrC, csrRowPtrC, nnzTotalDevHostPtr)
    import cusparseHandle, cusparseMatDescr
    type(cusparseHandle) :: handle       
    integer :: m, n, nnzA, nnzB
    type(cusparseMatDescr) :: descrA, descrB, descrC
    integer, device :: csrRowPtrA(*), csrColIndA(*), csrRowPtrB(*), csrColIndB(*), csrRowPtrC(*)
    integer, device :: nnzTotalDevHostPtr
    end function cusparseXcsrgeamNnz_setDPM
  end interface cusparseXcsrgeamnnz

#ifdef KEEP_V2_INTERFACES
  interface cusparseXcsrgeamnnz_v2
     integer(c_int) function cusparseXcsrgeamNnz_v2(handle, m, n, descrA, nnzA, csrRowPtrA, csrColIndA, &
          descrB, nnzB, csrRowPtrB, csrColIndB, descrC, csrRowPtrC, nnzTotalDevHostPtr) &
          bind(C, name='cusparseXcsrgeamNnz')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr              
       type(cusparseHandle), value :: handle       
       integer(c_int), value :: m, n, nnzA, nnzB
       type(cusparseMatDescr), value :: descrA, descrB, descrC
       integer(c_int), device :: csrRowPtrA(*), csrColIndA(*), csrRowPtrB(*), csrColIndB(*), &
            csrRowPtrC(*)
       !dir$ ignore_tkr(d) nnzTotalDevHostPtr
       integer(c_int) :: nnzTotalDevHostPtr
     end function cusparseXcsrgeamNnz_v2
  end interface cusparseXcsrgeamnnz_v2

  interface cusparseXcsrgeam2nnz_v2
     integer(c_int) function cusparseXcsrgeam2Nnz_v2(handle, m, n, descrA, nnzA, csrRowPtrA, csrColIndA, &
          descrB, nnzB, csrRowPtrB, csrColIndB, descrC, csrRowPtrC, nnzTotalDevHostPtr, buffer) &
          bind(C, name='cusparseXcsrgeam2Nnz')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr              
       type(cusparseHandle), value :: handle       
       integer(c_int), value :: m, n, nnzA, nnzB
       type(cusparseMatDescr), value :: descrA, descrB, descrC
       !dir$ ignore_tkr buffer
       integer(c_int), device :: csrRowPtrA(*), csrColIndA(*), csrRowPtrB(*), csrColIndB(*), &
            csrRowPtrC(*), buffer(*)
       !dir$ ignore_tkr(d) nnzTotalDevHostPtr
       integer(c_int) :: nnzTotalDevHostPtr
     end function cusparseXcsrgeam2Nnz_v2
  end interface cusparseXcsrgeam2nnz_v2
#endif

  interface cusparseXcsrgeam2Nnz
    integer function cusparseXcsrgeam2Nnz_setHPM(handle, m, n, descrA, nnzA, csrRowPtrA, csrColIndA, &
       descrB, nnzB, csrRowPtrB, csrColIndB, descrC, csrRowPtrC, nnzTotalDevHostPtr, buffer)
    import cusparseHandle, cusparseMatDescr
    type(cusparseHandle) :: handle       
    integer :: m, n, nnzA, nnzB
    type(cusparseMatDescr) :: descrA, descrB, descrC
    integer, device :: csrRowPtrA(*), csrColIndA(*), csrRowPtrB(*), csrColIndB(*), csrRowPtrC(*)
    integer :: nnzTotalDevHostPtr
    !dir$ ignore_tkr buffer
    integer(4), device :: buffer
    end function cusparseXcsrgeam2Nnz_setHPM
    integer function cusparseXcsrgeam2Nnz_setDPM(handle, m, n, descrA, nnzA, csrRowPtrA, csrColIndA, &
       descrB, nnzB, csrRowPtrB, csrColIndB, descrC, csrRowPtrC, nnzTotalDevHostPtr, buffer)
    import cusparseHandle, cusparseMatDescr
    type(cusparseHandle) :: handle       
    integer :: m, n, nnzA, nnzB
    type(cusparseMatDescr) :: descrA, descrB, descrC
    integer, device :: csrRowPtrA(*), csrColIndA(*), csrRowPtrB(*), csrColIndB(*), csrRowPtrC(*)
    integer, device :: nnzTotalDevHostPtr
    !dir$ ignore_tkr buffer
    integer(4), device :: buffer
    end function cusparseXcsrgeam2Nnz_setDPM
  end interface cusparseXcsrgeam2Nnz


! cusparseScsrgeam
  interface cusparseScsrgeam
    integer function cusparseScsrgeam_setHPM(handle, m, n, &
       alpha, descrA, nnzA, csrValA, csrRowPtrA, csrColIndA, &
       beta,  descrB, nnzB, csrValB, csrRowPtrB, csrColIndB, &
       descrC, csrValC, csrRowPtrC, csrColIndC)
    import cusparseHandle, cusparseMatDescr
    type(cusparseHandle) :: handle       
    integer :: m, n, nnzA, nnzB
    real(4) :: alpha, beta
    type(cusparseMatDescr):: descrA, descrB, descrC
    real(4), device :: csrValA(*), csrValB(*), csrValC(*)
    integer, device :: csrRowPtrA(*), csrColIndA(*), csrRowPtrB(*), csrColIndB(*), csrRowPtrC(*), csrColIndC(*)
    end function cusparseScsrgeam_setHPM
    integer function cusparseScsrgeam_setDPM(handle, m, n, &
       alpha, descrA, nnzA, csrValA, csrRowPtrA, csrColIndA, &
       beta,  descrB, nnzB, csrValB, csrRowPtrB, csrColIndB, &
       descrC, csrValC, csrRowPtrC, csrColIndC)
    import cusparseHandle, cusparseMatDescr
    type(cusparseHandle) :: handle       
    integer :: m, n, nnzA, nnzB
    real(4), device :: alpha, beta
    type(cusparseMatDescr):: descrA, descrB, descrC
    real(4), device :: csrValA(*), csrValB(*), csrValC(*)
    integer, device :: csrRowPtrA(*), csrColIndA(*), csrRowPtrB(*), csrColIndB(*), csrRowPtrC(*), csrColIndC(*)
    end function cusparseScsrgeam_setDPM
  end interface cusparseScsrgeam

#ifdef KEEP_V2_INTERFACES
  interface cusparseScsrgeam_v2
     integer(c_int) function cusparseScsrgeam_v2(handle, m, n, &
          alpha, descrA, nnzA, csrValA, csrRowPtrA, csrColIndA, &
          beta,  descrB, nnzB, csrValB, csrRowPtrB, csrColIndB, &
          descrC, csrValC, csrRowPtrC, csrColIndC) &
          bind(C, name='cusparseScsrgeam')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr              
       type(cusparseHandle), value :: handle       
       integer(c_int), value :: m, n, nnzA, nnzB
       !dir$ ignore_tkr (d) alpha, (d) beta
       real(c_float) :: alpha, beta
       type(cusparseMatDescr), value :: descrA, descrB, descrC
       real(c_float), device :: csrValA(*), csrValB(*), csrValC(*)
       integer(c_int), device :: csrRowPtrA(*), csrColIndA(*), csrRowPtrB(*), csrColIndB(*), &
            csrRowPtrC(*), csrColIndC(*)
     end function cusparseScsrgeam_v2
  end interface cusparseScsrgeam_v2

  interface cusparseScsrgeam2_bufferSizeExt_v2
    integer function cusparseScsrgeam2_bufferSizeExt_v2(handle, m, n, &
       alpha, descrA, nnzA, csrValA, csrRowPtrA, csrColIndA, &
       beta,  descrB, nnzB, csrValB, csrRowPtrB, csrColIndB, &
       descrC, csrValC, csrRowPtrC, csrColIndC, bufferSizeInBytes) &
       bind(C, name='cusparseScsrgeam2_bufferSizeExt')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr
       type(cusparseHandle), value :: handle       
       integer(c_int), value :: m, n, nnzA, nnzB
       !dir$ ignore_tkr (d) alpha, (d) beta
       real(4) :: alpha, beta
       type(cusparseMatDescr), value :: descrA, descrB, descrC
       real(4), device :: csrValA(*), csrValB(*), csrValC(*)
       integer, device :: csrRowPtrA(*), csrColIndA(*), csrRowPtrB(*), csrColIndB(*), &
            csrRowPtrC(*), csrColIndC(*)
       integer(8) :: bufferSizeInBytes
    end function cusparseScsrgeam2_bufferSizeExt_v2
  end interface cusparseScsrgeam2_bufferSizeExt_v2

  interface cusparseScsrgeam2_v2
    integer function cusparseScsrgeam2_v2(handle, m, n, &
       alpha, descrA, nnzA, csrValA, csrRowPtrA, csrColIndA, &
       beta,  descrB, nnzB, csrValB, csrRowPtrB, csrColIndB, &
       descrC, csrValC, csrRowPtrC, csrColIndC, buffer) &
       bind(C, name='cusparseScsrgeam2')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr
       type(cusparseHandle), value :: handle       
       integer(c_int), value :: m, n, nnzA, nnzB
       !dir$ ignore_tkr (d) alpha, (d) beta
       real(4) :: alpha, beta
       type(cusparseMatDescr), value :: descrA, descrB, descrC
       real(4), device :: csrValA(*), csrValB(*), csrValC(*)
       integer, device :: csrRowPtrA(*), csrColIndA(*), csrRowPtrB(*), csrColIndB(*), &
            csrRowPtrC(*), csrColIndC(*)
       !dir$ ignore_tkr buffer
       integer(4), device :: buffer
    end function cusparseScsrgeam2_v2
  end interface cusparseScsrgeam2_v2
#endif

  interface cusparseScsrgeam2_bufferSizeExt
    integer function cusparseScsrgeam2_bufferSizeExt_setHPM(handle, m, n, &
       alpha, descrA, nnzA, csrValA, csrRowPtrA, csrColIndA, &
       beta,  descrB, nnzB, csrValB, csrRowPtrB, csrColIndB, &
       descrC, csrValC, csrRowPtrC, csrColIndC, bufferSizeInBytes)
    import cusparseHandle, cusparseMatDescr
    type(cusparseHandle) :: handle       
    integer :: m, n, nnzA, nnzB
    real(4) :: alpha, beta
    type(cusparseMatDescr):: descrA, descrB, descrC
    real(4), device :: csrValA(*), csrValB(*), csrValC(*)
    integer, device :: csrRowPtrA(*), csrColIndA(*), csrRowPtrB(*), csrColIndB(*), csrRowPtrC(*), csrColIndC(*)
    integer(8) :: bufferSizeInBytes
    end function cusparseScsrgeam2_bufferSizeExt_setHPM
    integer function cusparseScsrgeam2_bufferSizeExt_setDPM(handle, m, n, &
       alpha, descrA, nnzA, csrValA, csrRowPtrA, csrColIndA, &
       beta,  descrB, nnzB, csrValB, csrRowPtrB, csrColIndB, &
       descrC, csrValC, csrRowPtrC, csrColIndC, bufferSizeInBytes)
    import cusparseHandle, cusparseMatDescr
    type(cusparseHandle) :: handle       
    integer :: m, n, nnzA, nnzB
    real(4), device :: alpha, beta
    type(cusparseMatDescr):: descrA, descrB, descrC
    real(4), device :: csrValA(*), csrValB(*), csrValC(*)
    integer, device :: csrRowPtrA(*), csrColIndA(*), csrRowPtrB(*), csrColIndB(*), csrRowPtrC(*), csrColIndC(*)
    integer(8) :: bufferSizeInBytes
    end function cusparseScsrgeam2_bufferSizeExt_setDPM
  end interface cusparseScsrgeam2_bufferSizeExt

  interface cusparseScsrgeam2
    integer function cusparseScsrgeam2_setHPM(handle, m, n, &
       alpha, descrA, nnzA, csrValA, csrRowPtrA, csrColIndA, &
       beta,  descrB, nnzB, csrValB, csrRowPtrB, csrColIndB, &
       descrC, csrValC, csrRowPtrC, csrColIndC, buffer)
    import cusparseHandle, cusparseMatDescr
    type(cusparseHandle) :: handle       
    integer :: m, n, nnzA, nnzB
    real(4) :: alpha, beta
    type(cusparseMatDescr):: descrA, descrB, descrC
    real(4), device :: csrValA(*), csrValB(*), csrValC(*)
    integer, device :: csrRowPtrA(*), csrColIndA(*), csrRowPtrB(*), csrColIndB(*), csrRowPtrC(*), csrColIndC(*)
    !dir$ ignore_tkr buffer
    integer(4), device :: buffer
    end function cusparseScsrgeam2_setHPM
    integer function cusparseScsrgeam2_setDPM(handle, m, n, &
       alpha, descrA, nnzA, csrValA, csrRowPtrA, csrColIndA, &
       beta,  descrB, nnzB, csrValB, csrRowPtrB, csrColIndB, &
       descrC, csrValC, csrRowPtrC, csrColIndC, buffer)
    import cusparseHandle, cusparseMatDescr
    type(cusparseHandle) :: handle       
    integer :: m, n, nnzA, nnzB
    real(4), device :: alpha, beta
    type(cusparseMatDescr):: descrA, descrB, descrC
    real(4), device :: csrValA(*), csrValB(*), csrValC(*)
    integer, device :: csrRowPtrA(*), csrColIndA(*), csrRowPtrB(*), csrColIndB(*), csrRowPtrC(*), csrColIndC(*)
    !dir$ ignore_tkr buffer
    integer(4), device :: buffer
    end function cusparseScsrgeam2_setDPM
  end interface cusparseScsrgeam2

! cusparseDcsrgeam
  interface cusparseDcsrgeam
    integer function cusparseDcsrgeam_setHPM(handle, m, n, &
       alpha, descrA, nnzA, csrValA, csrRowPtrA, csrColIndA, &
       beta,  descrB, nnzB, csrValB, csrRowPtrB, csrColIndB, &
       descrC, csrValC, csrRowPtrC, csrColIndC)
    import cusparseHandle, cusparseMatDescr
    type(cusparseHandle) :: handle       
    integer :: m, n, nnzA, nnzB
    real(8) :: alpha, beta
    type(cusparseMatDescr):: descrA, descrB, descrC
    real(8), device :: csrValA(*), csrValB(*), csrValC(*)
    integer, device :: csrRowPtrA(*), csrColIndA(*), csrRowPtrB(*), csrColIndB(*), csrRowPtrC(*), csrColIndC(*)
    end function cusparseDcsrgeam_setHPM
    integer function cusparseDcsrgeam_setDPM(handle, m, n, &
       alpha, descrA, nnzA, csrValA, csrRowPtrA, csrColIndA, &
       beta,  descrB, nnzB, csrValB, csrRowPtrB, csrColIndB, &
       descrC, csrValC, csrRowPtrC, csrColIndC)
    import cusparseHandle, cusparseMatDescr
    type(cusparseHandle) :: handle       
    integer :: m, n, nnzA, nnzB
    real(8), device :: alpha, beta
    type(cusparseMatDescr):: descrA, descrB, descrC
    real(8), device :: csrValA(*), csrValB(*), csrValC(*)
    integer, device :: csrRowPtrA(*), csrColIndA(*), csrRowPtrB(*), csrColIndB(*), csrRowPtrC(*), csrColIndC(*)
    end function cusparseDcsrgeam_setDPM
  end interface cusparseDcsrgeam

#ifdef KEEP_V2_INTERFACES
  interface cusparseDcsrgeam_v2
     integer(c_int) function cusparseDcsrgeam_v2(handle, m, n, &
          alpha, descrA, nnzA, csrValA, csrRowPtrA, csrColIndA, &
          beta,  descrB, nnzB, csrValB, csrRowPtrB, csrColIndB, &
          descrC, csrValC, csrRowPtrC, csrColIndC) &
          bind(C, name='cusparseDcsrgeam')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr              
       type(cusparseHandle), value :: handle       
       integer(c_int), value :: m, n, nnzA, nnzB
       !dir$ ignore_tkr (d) alpha, (d) beta
       real(c_double) :: alpha, beta
       type(cusparseMatDescr), value :: descrA, descrB, descrC
       real(c_double), device :: csrValA(*), csrValB(*), csrValC(*)
       integer(c_int), device :: csrRowPtrA(*), csrColIndA(*), csrRowPtrB(*), csrColIndB(*), &
            csrRowPtrC(*), csrColIndC(*)
     end function cusparseDcsrgeam_v2
  end interface cusparseDcsrgeam_v2

  interface cusparseDcsrgeam2_bufferSizeExt_v2
    integer function cusparseDcsrgeam2_bufferSizeExt_v2(handle, m, n, &
       alpha, descrA, nnzA, csrValA, csrRowPtrA, csrColIndA, &
       beta,  descrB, nnzB, csrValB, csrRowPtrB, csrColIndB, &
       descrC, csrValC, csrRowPtrC, csrColIndC, bufferSizeInBytes) &
       bind(C, name='cusparseDcsrgeam2_bufferSizeExt')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr
       type(cusparseHandle), value :: handle       
       integer(c_int), value :: m, n, nnzA, nnzB
       !dir$ ignore_tkr (d) alpha, (d) beta
       real(8) :: alpha, beta
       type(cusparseMatDescr), value :: descrA, descrB, descrC
       real(8), device :: csrValA(*), csrValB(*), csrValC(*)
       integer, device :: csrRowPtrA(*), csrColIndA(*), csrRowPtrB(*), csrColIndB(*), &
            csrRowPtrC(*), csrColIndC(*)
       integer(8) :: bufferSizeInBytes
    end function cusparseDcsrgeam2_bufferSizeExt_v2
  end interface cusparseDcsrgeam2_bufferSizeExt_v2

  interface cusparseDcsrgeam2_v2
    integer function cusparseDcsrgeam2_v2(handle, m, n, &
       alpha, descrA, nnzA, csrValA, csrRowPtrA, csrColIndA, &
       beta,  descrB, nnzB, csrValB, csrRowPtrB, csrColIndB, &
       descrC, csrValC, csrRowPtrC, csrColIndC, buffer) &
       bind(C, name='cusparseDcsrgeam2')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr
       type(cusparseHandle), value :: handle       
       integer(c_int), value :: m, n, nnzA, nnzB
       !dir$ ignore_tkr (d) alpha, (d) beta
       real(8) :: alpha, beta
       type(cusparseMatDescr), value :: descrA, descrB, descrC
       real(8), device :: csrValA(*), csrValB(*), csrValC(*)
       integer, device :: csrRowPtrA(*), csrColIndA(*), csrRowPtrB(*), csrColIndB(*), &
            csrRowPtrC(*), csrColIndC(*)
       !dir$ ignore_tkr buffer
       integer(4), device :: buffer
    end function cusparseDcsrgeam2_v2
  end interface cusparseDcsrgeam2_v2
#endif

  interface cusparseDcsrgeam2_bufferSizeExt
    integer function cusparseDcsrgeam2_bufferSizeExt_setHPM(handle, m, n, &
       alpha, descrA, nnzA, csrValA, csrRowPtrA, csrColIndA, &
       beta,  descrB, nnzB, csrValB, csrRowPtrB, csrColIndB, &
       descrC, csrValC, csrRowPtrC, csrColIndC, bufferSizeInBytes)
    import cusparseHandle, cusparseMatDescr
    type(cusparseHandle) :: handle       
    integer :: m, n, nnzA, nnzB
    real(8) :: alpha, beta
    type(cusparseMatDescr):: descrA, descrB, descrC
    real(8), device :: csrValA(*), csrValB(*), csrValC(*)
    integer, device :: csrRowPtrA(*), csrColIndA(*), csrRowPtrB(*), csrColIndB(*), csrRowPtrC(*), csrColIndC(*)
    integer(8) :: bufferSizeInBytes
    end function cusparseDcsrgeam2_bufferSizeExt_setHPM
    integer function cusparseDcsrgeam2_bufferSizeExt_setDPM(handle, m, n, &
       alpha, descrA, nnzA, csrValA, csrRowPtrA, csrColIndA, &
       beta,  descrB, nnzB, csrValB, csrRowPtrB, csrColIndB, &
       descrC, csrValC, csrRowPtrC, csrColIndC, bufferSizeInBytes)
    import cusparseHandle, cusparseMatDescr
    type(cusparseHandle) :: handle       
    integer :: m, n, nnzA, nnzB
    real(8), device :: alpha, beta
    type(cusparseMatDescr):: descrA, descrB, descrC
    real(8), device :: csrValA(*), csrValB(*), csrValC(*)
    integer, device :: csrRowPtrA(*), csrColIndA(*), csrRowPtrB(*), csrColIndB(*), csrRowPtrC(*), csrColIndC(*)
    integer(8) :: bufferSizeInBytes
    end function cusparseDcsrgeam2_bufferSizeExt_setDPM
  end interface cusparseDcsrgeam2_bufferSizeExt

  interface cusparseDcsrgeam2
    integer function cusparseDcsrgeam2_setHPM(handle, m, n, &
       alpha, descrA, nnzA, csrValA, csrRowPtrA, csrColIndA, &
       beta,  descrB, nnzB, csrValB, csrRowPtrB, csrColIndB, &
       descrC, csrValC, csrRowPtrC, csrColIndC, buffer)
    import cusparseHandle, cusparseMatDescr
    type(cusparseHandle) :: handle       
    integer :: m, n, nnzA, nnzB
    real(8) :: alpha, beta
    type(cusparseMatDescr):: descrA, descrB, descrC
    real(8), device :: csrValA(*), csrValB(*), csrValC(*)
    integer, device :: csrRowPtrA(*), csrColIndA(*), csrRowPtrB(*), csrColIndB(*), csrRowPtrC(*), csrColIndC(*)
    !dir$ ignore_tkr buffer
    integer(4), device :: buffer
    end function cusparseDcsrgeam2_setHPM
    integer function cusparseDcsrgeam2_setDPM(handle, m, n, &
       alpha, descrA, nnzA, csrValA, csrRowPtrA, csrColIndA, &
       beta,  descrB, nnzB, csrValB, csrRowPtrB, csrColIndB, &
       descrC, csrValC, csrRowPtrC, csrColIndC, buffer)
    import cusparseHandle, cusparseMatDescr
    type(cusparseHandle) :: handle       
    integer :: m, n, nnzA, nnzB
    real(8), device :: alpha, beta
    type(cusparseMatDescr):: descrA, descrB, descrC
    real(8), device :: csrValA(*), csrValB(*), csrValC(*)
    integer, device :: csrRowPtrA(*), csrColIndA(*), csrRowPtrB(*), csrColIndB(*), csrRowPtrC(*), csrColIndC(*)
    !dir$ ignore_tkr buffer
    integer(4), device :: buffer
    end function cusparseDcsrgeam2_setDPM
  end interface cusparseDcsrgeam2

! cusparseCcsrgeam
  interface cusparseCcsrgeam
    integer function cusparseCcsrgeam_setHPM(handle, m, n, &
       alpha, descrA, nnzA, csrValA, csrRowPtrA, csrColIndA, &
       beta,  descrB, nnzB, csrValB, csrRowPtrB, csrColIndB, &
       descrC, csrValC, csrRowPtrC, csrColIndC)
    import cusparseHandle, cusparseMatDescr
    type(cusparseHandle) :: handle       
    integer :: m, n, nnzA, nnzB
    complex(4) :: alpha, beta
    type(cusparseMatDescr):: descrA, descrB, descrC
    complex(4), device :: csrValA(*), csrValB(*), csrValC(*)
    integer, device :: csrRowPtrA(*), csrColIndA(*), csrRowPtrB(*), csrColIndB(*), csrRowPtrC(*), csrColIndC(*)
    end function cusparseCcsrgeam_setHPM
    integer function cusparseCcsrgeam_setDPM(handle, m, n, &
       alpha, descrA, nnzA, csrValA, csrRowPtrA, csrColIndA, &
       beta,  descrB, nnzB, csrValB, csrRowPtrB, csrColIndB, &
       descrC, csrValC, csrRowPtrC, csrColIndC)
    import cusparseHandle, cusparseMatDescr
    type(cusparseHandle) :: handle       
    integer :: m, n, nnzA, nnzB
    complex(4), device :: alpha, beta
    type(cusparseMatDescr):: descrA, descrB, descrC
    complex(4), device :: csrValA(*), csrValB(*), csrValC(*)
    integer, device :: csrRowPtrA(*), csrColIndA(*), csrRowPtrB(*), csrColIndB(*), csrRowPtrC(*), csrColIndC(*)
    end function cusparseCcsrgeam_setDPM
  end interface cusparseCcsrgeam

#ifdef KEEP_V2_INTERFACES
  interface cusparseCcsrgeam_v2
     integer(c_int) function cusparseCcsrgeam_v2(handle, m, n, &
          alpha, descrA, nnzA, csrValA, csrRowPtrA, csrColIndA, &
          beta,  descrB, nnzB, csrValB, csrRowPtrB, csrColIndB, &
          descrC, csrValC, csrRowPtrC, csrColIndC) &
          bind(C, name='cusparseCcsrgeam')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr              
       type(cusparseHandle), value :: handle       
       integer(c_int), value :: m, n, nnzA, nnzB
       !dir$ ignore_tkr (d) alpha, (d) beta
       complex(4) :: alpha, beta
       type(cusparseMatDescr), value :: descrA, descrB, descrC
       complex(4), device :: csrValA(*), csrValB(*), csrValC(*)
       integer(c_int), device :: csrRowPtrA(*), csrColIndA(*), csrRowPtrB(*), csrColIndB(*), &
            csrRowPtrC(*), csrColIndC(*)
     end function cusparseCcsrgeam_v2
  end interface cusparseCcsrgeam_v2

  interface cusparseCcsrgeam2_bufferSizeExt_v2
    integer function cusparseCcsrgeam2_bufferSizeExt_v2(handle, m, n, &
       alpha, descrA, nnzA, csrValA, csrRowPtrA, csrColIndA, &
       beta,  descrB, nnzB, csrValB, csrRowPtrB, csrColIndB, &
       descrC, csrValC, csrRowPtrC, csrColIndC, bufferSizeInBytes) &
       bind(C, name='cusparseCcsrgeam2_bufferSizeExt')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr
       type(cusparseHandle), value :: handle       
       integer(c_int), value :: m, n, nnzA, nnzB
       !dir$ ignore_tkr (d) alpha, (d) beta
       complex(4) :: alpha, beta
       type(cusparseMatDescr), value :: descrA, descrB, descrC
       complex(4), device :: csrValA(*), csrValB(*), csrValC(*)
       integer, device :: csrRowPtrA(*), csrColIndA(*), csrRowPtrB(*), csrColIndB(*), &
            csrRowPtrC(*), csrColIndC(*)
       integer(8) :: bufferSizeInBytes
    end function cusparseCcsrgeam2_bufferSizeExt_v2
  end interface cusparseCcsrgeam2_bufferSizeExt_v2

  interface cusparseCcsrgeam2_v2
    integer function cusparseCcsrgeam2_v2(handle, m, n, &
       alpha, descrA, nnzA, csrValA, csrRowPtrA, csrColIndA, &
       beta,  descrB, nnzB, csrValB, csrRowPtrB, csrColIndB, &
       descrC, csrValC, csrRowPtrC, csrColIndC, buffer) &
       bind(C, name='cusparseCcsrgeam2')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr
       type(cusparseHandle), value :: handle       
       integer(c_int), value :: m, n, nnzA, nnzB
       !dir$ ignore_tkr (d) alpha, (d) beta
       complex(4) :: alpha, beta
       type(cusparseMatDescr), value :: descrA, descrB, descrC
       complex(4), device :: csrValA(*), csrValB(*), csrValC(*)
       integer, device :: csrRowPtrA(*), csrColIndA(*), csrRowPtrB(*), csrColIndB(*), &
            csrRowPtrC(*), csrColIndC(*)
       !dir$ ignore_tkr buffer
       integer(4), device :: buffer
    end function cusparseCcsrgeam2_v2
  end interface cusparseCcsrgeam2_v2
#endif

  interface cusparseCcsrgeam2_bufferSizeExt
    integer function cusparseCcsrgeam2_bufferSizeExt_setHPM(handle, m, n, &
       alpha, descrA, nnzA, csrValA, csrRowPtrA, csrColIndA, &
       beta,  descrB, nnzB, csrValB, csrRowPtrB, csrColIndB, &
       descrC, csrValC, csrRowPtrC, csrColIndC, bufferSizeInBytes)
    import cusparseHandle, cusparseMatDescr
    type(cusparseHandle) :: handle       
    integer :: m, n, nnzA, nnzB
    complex(4) :: alpha, beta
    type(cusparseMatDescr):: descrA, descrB, descrC
    complex(4), device :: csrValA(*), csrValB(*), csrValC(*)
    integer, device :: csrRowPtrA(*), csrColIndA(*), csrRowPtrB(*), csrColIndB(*), csrRowPtrC(*), csrColIndC(*)
    integer(8) :: bufferSizeInBytes
    end function cusparseCcsrgeam2_bufferSizeExt_setHPM
    integer function cusparseCcsrgeam2_bufferSizeExt_setDPM(handle, m, n, &
       alpha, descrA, nnzA, csrValA, csrRowPtrA, csrColIndA, &
       beta,  descrB, nnzB, csrValB, csrRowPtrB, csrColIndB, &
       descrC, csrValC, csrRowPtrC, csrColIndC, bufferSizeInBytes)
    import cusparseHandle, cusparseMatDescr
    type(cusparseHandle) :: handle       
    integer :: m, n, nnzA, nnzB
    complex(4), device :: alpha, beta
    type(cusparseMatDescr):: descrA, descrB, descrC
    complex(4), device :: csrValA(*), csrValB(*), csrValC(*)
    integer, device :: csrRowPtrA(*), csrColIndA(*), csrRowPtrB(*), csrColIndB(*), csrRowPtrC(*), csrColIndC(*)
    integer(8) :: bufferSizeInBytes
    end function cusparseCcsrgeam2_bufferSizeExt_setDPM
  end interface cusparseCcsrgeam2_bufferSizeExt

  interface cusparseCcsrgeam2
    integer function cusparseCcsrgeam2_setHPM(handle, m, n, &
       alpha, descrA, nnzA, csrValA, csrRowPtrA, csrColIndA, &
       beta,  descrB, nnzB, csrValB, csrRowPtrB, csrColIndB, &
       descrC, csrValC, csrRowPtrC, csrColIndC, buffer)
    import cusparseHandle, cusparseMatDescr
    type(cusparseHandle) :: handle       
    integer :: m, n, nnzA, nnzB
    complex(4) :: alpha, beta
    type(cusparseMatDescr):: descrA, descrB, descrC
    complex(4), device :: csrValA(*), csrValB(*), csrValC(*)
    integer, device :: csrRowPtrA(*), csrColIndA(*), csrRowPtrB(*), csrColIndB(*), csrRowPtrC(*), csrColIndC(*)
    !dir$ ignore_tkr buffer
    integer(4), device :: buffer
    end function cusparseCcsrgeam2_setHPM
    integer function cusparseCcsrgeam2_setDPM(handle, m, n, &
       alpha, descrA, nnzA, csrValA, csrRowPtrA, csrColIndA, &
       beta,  descrB, nnzB, csrValB, csrRowPtrB, csrColIndB, &
       descrC, csrValC, csrRowPtrC, csrColIndC, buffer)
    import cusparseHandle, cusparseMatDescr
    type(cusparseHandle) :: handle       
    integer :: m, n, nnzA, nnzB
    complex(4), device :: alpha, beta
    type(cusparseMatDescr):: descrA, descrB, descrC
    complex(4), device :: csrValA(*), csrValB(*), csrValC(*)
    integer, device :: csrRowPtrA(*), csrColIndA(*), csrRowPtrB(*), csrColIndB(*), csrRowPtrC(*), csrColIndC(*)
    !dir$ ignore_tkr buffer
    integer(4), device :: buffer
    end function cusparseCcsrgeam2_setDPM
  end interface cusparseCcsrgeam2

! cusparseZcsrgeam
  interface cusparseZcsrgeam
    integer function cusparseZcsrgeam_setHPM(handle, m, n, &
       alpha, descrA, nnzA, csrValA, csrRowPtrA, csrColIndA, &
       beta,  descrB, nnzB, csrValB, csrRowPtrB, csrColIndB, &
       descrC, csrValC, csrRowPtrC, csrColIndC)
    import cusparseHandle, cusparseMatDescr
    type(cusparseHandle) :: handle       
    integer :: m, n, nnzA, nnzB
    complex(8) :: alpha, beta
    type(cusparseMatDescr):: descrA, descrB, descrC
    complex(8), device :: csrValA(*), csrValB(*), csrValC(*)
    integer, device :: csrRowPtrA(*), csrColIndA(*), csrRowPtrB(*), csrColIndB(*), csrRowPtrC(*), csrColIndC(*)
    end function cusparseZcsrgeam_setHPM
    integer function cusparseZcsrgeam_setDPM(handle, m, n, &
       alpha, descrA, nnzA, csrValA, csrRowPtrA, csrColIndA, &
       beta,  descrB, nnzB, csrValB, csrRowPtrB, csrColIndB, &
       descrC, csrValC, csrRowPtrC, csrColIndC)
    import cusparseHandle, cusparseMatDescr
    type(cusparseHandle) :: handle       
    integer :: m, n, nnzA, nnzB
    complex(8), device :: alpha, beta
    type(cusparseMatDescr):: descrA, descrB, descrC
    complex(8), device :: csrValA(*), csrValB(*), csrValC(*)
    integer, device :: csrRowPtrA(*), csrColIndA(*), csrRowPtrB(*), csrColIndB(*), csrRowPtrC(*), csrColIndC(*)
    end function cusparseZcsrgeam_setDPM
  end interface cusparseZcsrgeam

#ifdef KEEP_V2_INTERFACES
  interface cusparseZcsrgeam_v2
     integer(c_int) function cusparseZcsrgeam_v2(handle, m, n, &
          alpha, descrA, nnzA, csrValA, csrRowPtrA, csrColIndA, &
          beta,  descrB, nnzB, csrValB, csrRowPtrB, csrColIndB, &
          descrC, csrValC, csrRowPtrC, csrColIndC) &
          bind(C, name='cusparseZcsrgeam')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr              
       type(cusparseHandle), value :: handle       
       integer(c_int), value :: m, n, nnzA, nnzB
       !dir$ ignore_tkr (d) alpha, (d) beta
       complex(8) :: alpha, beta
       type(cusparseMatDescr), value :: descrA, descrB, descrC
       complex(8), device :: csrValA(*), csrValB(*), csrValC(*)
       integer(c_int), device :: csrRowPtrA(*), csrColIndA(*), csrRowPtrB(*), csrColIndB(*), &
            csrRowPtrC(*), csrColIndC(*)
     end function cusparseZcsrgeam_v2
  end interface cusparseZcsrgeam_v2

  interface cusparseZcsrgeam2_bufferSizeExt_v2
    integer function cusparseZcsrgeam2_bufferSizeExt_v2(handle, m, n, &
       alpha, descrA, nnzA, csrValA, csrRowPtrA, csrColIndA, &
       beta,  descrB, nnzB, csrValB, csrRowPtrB, csrColIndB, &
       descrC, csrValC, csrRowPtrC, csrColIndC, bufferSizeInBytes) &
       bind(C, name='cusparseZcsrgeam2_bufferSizeExt')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr
       type(cusparseHandle), value :: handle       
       integer(c_int), value :: m, n, nnzA, nnzB
       !dir$ ignore_tkr (d) alpha, (d) beta
       complex(8) :: alpha, beta
       type(cusparseMatDescr), value :: descrA, descrB, descrC
       complex(8), device :: csrValA(*), csrValB(*), csrValC(*)
       integer, device :: csrRowPtrA(*), csrColIndA(*), csrRowPtrB(*), csrColIndB(*), &
            csrRowPtrC(*), csrColIndC(*)
       integer(8) :: bufferSizeInBytes
    end function cusparseZcsrgeam2_bufferSizeExt_v2
  end interface cusparseZcsrgeam2_bufferSizeExt_v2

  interface cusparseZcsrgeam2_v2
    integer function cusparseZcsrgeam2_v2(handle, m, n, &
       alpha, descrA, nnzA, csrValA, csrRowPtrA, csrColIndA, &
       beta,  descrB, nnzB, csrValB, csrRowPtrB, csrColIndB, &
       descrC, csrValC, csrRowPtrC, csrColIndC, buffer) &
       bind(C, name='cusparseZcsrgeam2')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr
       type(cusparseHandle), value :: handle       
       integer(c_int), value :: m, n, nnzA, nnzB
       !dir$ ignore_tkr (d) alpha, (d) beta
       complex(8) :: alpha, beta
       type(cusparseMatDescr), value :: descrA, descrB, descrC
       complex(8), device :: csrValA(*), csrValB(*), csrValC(*)
       integer, device :: csrRowPtrA(*), csrColIndA(*), csrRowPtrB(*), csrColIndB(*), &
            csrRowPtrC(*), csrColIndC(*)
       !dir$ ignore_tkr buffer
       integer(4), device :: buffer
    end function cusparseZcsrgeam2_v2
  end interface cusparseZcsrgeam2_v2
#endif

  interface cusparseZcsrgeam2_bufferSizeExt
    integer function cusparseZcsrgeam2_bufferSizeExt_setHPM(handle, m, n, &
       alpha, descrA, nnzA, csrValA, csrRowPtrA, csrColIndA, &
       beta,  descrB, nnzB, csrValB, csrRowPtrB, csrColIndB, &
       descrC, csrValC, csrRowPtrC, csrColIndC, bufferSizeInBytes)
    import cusparseHandle, cusparseMatDescr
    type(cusparseHandle) :: handle       
    integer :: m, n, nnzA, nnzB
    complex(8) :: alpha, beta
    type(cusparseMatDescr):: descrA, descrB, descrC
    complex(8), device :: csrValA(*), csrValB(*), csrValC(*)
    integer, device :: csrRowPtrA(*), csrColIndA(*), csrRowPtrB(*), csrColIndB(*), csrRowPtrC(*), csrColIndC(*)
    integer(8) :: bufferSizeInBytes
    end function cusparseZcsrgeam2_bufferSizeExt_setHPM
    integer function cusparseZcsrgeam2_bufferSizeExt_setDPM(handle, m, n, &
       alpha, descrA, nnzA, csrValA, csrRowPtrA, csrColIndA, &
       beta,  descrB, nnzB, csrValB, csrRowPtrB, csrColIndB, &
       descrC, csrValC, csrRowPtrC, csrColIndC, bufferSizeInBytes)
    import cusparseHandle, cusparseMatDescr
    type(cusparseHandle) :: handle       
    integer :: m, n, nnzA, nnzB
    complex(8), device :: alpha, beta
    type(cusparseMatDescr):: descrA, descrB, descrC
    complex(8), device :: csrValA(*), csrValB(*), csrValC(*)
    integer, device :: csrRowPtrA(*), csrColIndA(*), csrRowPtrB(*), csrColIndB(*), csrRowPtrC(*), csrColIndC(*)
    integer(8) :: bufferSizeInBytes
    end function cusparseZcsrgeam2_bufferSizeExt_setDPM
  end interface cusparseZcsrgeam2_bufferSizeExt

  interface cusparseZcsrgeam2
    integer function cusparseZcsrgeam2_setHPM(handle, m, n, &
       alpha, descrA, nnzA, csrValA, csrRowPtrA, csrColIndA, &
       beta,  descrB, nnzB, csrValB, csrRowPtrB, csrColIndB, &
       descrC, csrValC, csrRowPtrC, csrColIndC, buffer)
    import cusparseHandle, cusparseMatDescr
    type(cusparseHandle) :: handle       
    integer :: m, n, nnzA, nnzB
    complex(8) :: alpha, beta
    type(cusparseMatDescr):: descrA, descrB, descrC
    complex(8), device :: csrValA(*), csrValB(*), csrValC(*)
    integer, device :: csrRowPtrA(*), csrColIndA(*), csrRowPtrB(*), csrColIndB(*), csrRowPtrC(*), csrColIndC(*)
    !dir$ ignore_tkr buffer
    integer(4), device :: buffer
    end function cusparseZcsrgeam2_setHPM
    integer function cusparseZcsrgeam2_setDPM(handle, m, n, &
       alpha, descrA, nnzA, csrValA, csrRowPtrA, csrColIndA, &
       beta,  descrB, nnzB, csrValB, csrRowPtrB, csrColIndB, &
       descrC, csrValC, csrRowPtrC, csrColIndC, buffer)
    import cusparseHandle, cusparseMatDescr
    type(cusparseHandle) :: handle       
    integer :: m, n, nnzA, nnzB
    complex(8), device :: alpha, beta
    type(cusparseMatDescr):: descrA, descrB, descrC
    complex(8), device :: csrValA(*), csrValB(*), csrValC(*)
    integer, device :: csrRowPtrA(*), csrColIndA(*), csrRowPtrB(*), csrColIndB(*), csrRowPtrC(*), csrColIndC(*)
    !dir$ ignore_tkr buffer
    integer(4), device :: buffer
    end function cusparseZcsrgeam2_setDPM
  end interface cusparseZcsrgeam2

! cusparseXcsrgemmNnz
  interface cusparseXcsrgemmNnz
    integer function cusparseXcsrgemmNnz_setHPM(handle, transA, transB, m, n, k, &
       descrA, nnzA, csrRowPtrA, csrColIndA, &
       descrB, nnzB, csrRowPtrB, csrColIndB, &
       descrC, csrRowPtrC, nnzTotalDevHostPtr)
    import cusparseHandle, cusparseMatDescr
    type(cusparseHandle) :: handle       
    integer :: transA, transB, m, n, k, nnzA, nnzB
    type(cusparseMatDescr) :: descrA, descrB, descrC
    integer, device :: csrRowPtrA(*), csrColIndA(*), csrRowPtrB(*), csrColIndB(*), csrRowPtrC(*)
    integer :: nnzTotalDevHostPtr
    end function cusparseXcsrgemmNnz_setHPM
    integer function cusparseXcsrgemmNnz_setDPM(handle, transA, transB, m, n, k, & 
       descrA, nnzA, csrRowPtrA, csrColIndA, &
       descrB, nnzB, csrRowPtrB, csrColIndB, &
       descrC, csrRowPtrC, nnzTotalDevHostPtr)
    import cusparseHandle, cusparseMatDescr
    type(cusparseHandle) :: handle       
    integer :: transA, transB, m, n, k, nnzA, nnzB
    type(cusparseMatDescr) :: descrA, descrB, descrC
    integer, device :: csrRowPtrA(*), csrColIndA(*), csrRowPtrB(*), csrColIndB(*), csrRowPtrC(*)
    integer, device :: nnzTotalDevHostPtr
    end function cusparseXcsrgemmNnz_setDPM
  end interface cusparseXcsrgemmNnz

#ifdef KEEP_V2_INTERFACES
  interface cusparseXcsrgemmNnz_v2
     integer(c_int) function cusparseXcsrgemmNnz_v2(handle, transA, transB, m, n, k, &
          descrA, nnzA, csrRowPtrA, csrColIndA, &
          descrB, nnzB, csrRowPtrB, csrColIndB, &
          descrC, csrRowPtrC, nnzTotalDevHostPtr) bind(C, name='cusparseXcsrgemmNnz')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr              
       type(cusparseHandle), value :: handle       
       integer(c_int), value :: transA, transB, m, n, k, nnzA, nnzB
       type(cusparseMatDescr), value :: descrA, descrB, descrC
       integer(c_int), device :: csrRowPtrA(*), csrColIndA(*), csrRowPtrB(*), csrColIndB(*), &
            csrRowPtrC(*)
       !dir$ ignore_tkr(d) nnzTotalDevHostPtr
       integer(c_int) :: nnzTotalDevHostPtr
     end function cusparseXcsrgemmNnz_v2
  end interface cusparseXcsrgemmNnz_v2

  interface cusparseXcsrgemm2Nnz_v2
     integer(c_int) function cusparseXcsrgemm2Nnz_v2(handle, transA, transB, m, n, k, &
          descrA, nnzA, csrRowPtrA, csrColIndA, &
          descrB, nnzB, csrRowPtrB, csrColIndB, &
          descrC, csrRowPtrC, nnzTotalDevHostPtr, buffer) bind(C, name='cusparseXcsrgemm2Nnz')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr              
       type(cusparseHandle), value :: handle       
       integer(c_int), value :: transA, transB, m, n, k, nnzA, nnzB
       type(cusparseMatDescr), value :: descrA, descrB, descrC
       !dir$ ignore_tkr buffer
       integer(c_int), device :: csrRowPtrA(*), csrColIndA(*), csrRowPtrB(*), csrColIndB(*), &
            csrRowPtrC(*), buffer(*)
       !dir$ ignore_tkr(d) nnzTotalDevHostPtr
       integer(c_int) :: nnzTotalDevHostPtr
     end function cusparseXcsrgemm2Nnz_v2
  end interface cusparseXcsrgemm2Nnz_v2
#endif

  interface cusparseXcsrgemm2Nnz
    integer function cusparseXcsrgemm2Nnz_setHPM(handle, transA, transB, m, n, k, &
       descrA, nnzA, csrRowPtrA, csrColIndA, &
       descrB, nnzB, csrRowPtrB, csrColIndB, &
       descrC, csrRowPtrC, nnzTotalDevHostPtr, buffer)
    import cusparseHandle, cusparseMatDescr
    type(cusparseHandle) :: handle       
    integer :: transA, transB, m, n, k, nnzA, nnzB
    type(cusparseMatDescr) :: descrA, descrB, descrC
    integer, device :: csrRowPtrA(*), csrColIndA(*), csrRowPtrB(*), csrColIndB(*), csrRowPtrC(*)
    integer :: nnzTotalDevHostPtr
    !dir$ ignore_tkr buffer
    integer(4), device :: buffer
    end function cusparseXcsrgemm2Nnz_setHPM
    integer function cusparseXcsrgemm2Nnz_setDPM(handle, transA, transB, m, n, k, & 
       descrA, nnzA, csrRowPtrA, csrColIndA, &
       descrB, nnzB, csrRowPtrB, csrColIndB, &
       descrC, csrRowPtrC, nnzTotalDevHostPtr, buffer)
    import cusparseHandle, cusparseMatDescr
    type(cusparseHandle) :: handle       
    integer :: transA, transB, m, n, k, nnzA, nnzB
    type(cusparseMatDescr) :: descrA, descrB, descrC
    integer, device :: csrRowPtrA(*), csrColIndA(*), csrRowPtrB(*), csrColIndB(*), csrRowPtrC(*)
    integer, device :: nnzTotalDevHostPtr
    !dir$ ignore_tkr buffer
    integer(4), device :: buffer
    end function cusparseXcsrgemm2Nnz_setDPM
  end interface cusparseXcsrgemm2Nnz

! cusparseScsrgemm
  interface cusparseScsrgemm
     integer(c_int) function cusparseScsrgemm(handle, transA, transB, m, n, k, &
          descrA, nnzA, csrValA, csrRowPtrA, csrColIndA, &
          descrB, nnzB, csrValB, csrRowPtrB, csrColIndB, &
          descrC, csrValC, csrRowPtrC, csrColIndC) &
          bind(C, name='cusparseScsrgemm')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr              
       type(cusparseHandle), value :: handle       
       integer(c_int), value :: transA, transB, m, n, k, nnzA, nnzB
       type(cusparseMatDescr), value :: descrA, descrB, descrC
       real(c_float), device :: csrValA(*), csrValB(*), csrValC(*)
       integer(c_int), device :: csrRowPtrA(*), csrColIndA(*), csrRowPtrB(*), csrColIndB(*), &
            csrRowPtrC(*), csrColIndC(*)
     end function cusparseScsrgemm
  end interface cusparseScsrgemm

! cusparseDcsrgemm
  interface cusparseDcsrgemm
     integer(c_int) function cusparseDcsrgemm(handle, transA, transB, m, n, k, &
          descrA, nnzA, csrValA, csrRowPtrA, csrColIndA, &
          descrB, nnzB, csrValB, csrRowPtrB, csrColIndB, &
          descrC, csrValC, csrRowPtrC, csrColIndC) &
          bind(C, name='cusparseDcsrgemm')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr              
       type(cusparseHandle), value :: handle       
       integer(c_int), value :: transA, transB, m, n, k, nnzA, nnzB
       type(cusparseMatDescr), value :: descrA, descrB, descrC
       real(c_double), device :: csrValA(*), csrValB(*), csrValC(*)
       integer(c_int), device :: csrRowPtrA(*), csrColIndA(*), csrRowPtrB(*), csrColIndB(*), &
            csrRowPtrC(*), csrColIndC(*)
     end function cusparseDcsrgemm
  end interface cusparseDcsrgemm

! cusparseCcsrgemm
  interface cusparseCcsrgemm
     integer(c_int) function cusparseCcsrgemm(handle, transA, transB, m, n, k, &
          descrA, nnzA, csrValA, csrRowPtrA, csrColIndA, &
          descrB, nnzB, csrValB, csrRowPtrB, csrColIndB, &
          descrC, csrValC, csrRowPtrC, csrColIndC) &
          bind(C, name='cusparseCcsrgemm')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr              
       type(cusparseHandle), value :: handle       
       integer(c_int), value :: transA, transB, m, n, k, nnzA, nnzB
       type(cusparseMatDescr), value :: descrA, descrB, descrC
       complex(4), device :: csrValA(*), csrValB(*), csrValC(*)
       integer(c_int), device :: csrRowPtrA(*), csrColIndA(*), csrRowPtrB(*), csrColIndB(*), &
            csrRowPtrC(*), csrColIndC(*)
     end function cusparseCcsrgemm
  end interface cusparseCcsrgemm

! cusparseZcsrgemm
  interface cusparseZcsrgemm
     integer(c_int) function cusparseZcsrgemm(handle, transA, transB, m, n, k, &
          descrA, nnzA, csrValA, csrRowPtrA, csrColIndA, &
          descrB, nnzB, csrValB, csrRowPtrB, csrColIndB, &
          descrC, csrValC, csrRowPtrC, csrColIndC) &
          bind(C, name='cusparseZcsrgemm')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr              
       type(cusparseHandle), value :: handle       
       integer(c_int), value :: transA, transB, m, n, k, nnzA, nnzB
       type(cusparseMatDescr), value :: descrA, descrB, descrC
       complex(8), device :: csrValA(*), csrValB(*), csrValC(*)
       integer(c_int), device :: csrRowPtrA(*), csrColIndA(*), csrRowPtrB(*), csrColIndB(*), &
            csrRowPtrC(*), csrColIndC(*)
     end function cusparseZcsrgemm
  end interface cusparseZcsrgemm

  ! -------------------
  ! 6.0 Extra Functions
  ! -------------------
  ! none

  ! -------------------
  ! 6.5 Extra Functions
  ! -------------------
  ! none

  ! -------------------
  ! 7.0 Extra Functions
  ! -------------------

! cusparseScsrgemm2_bufferSizeExt
  interface cusparseScsrgemm2_bufferSizeExt
    integer function cusparseScsrgemm2_bufferSizeExt_setHPM(handle, m, n, k, &
       alpha, &
       descrA, nnzA, csrRowPtrA, csrColIndA, &
       descrB, nnzB, csrRowPtrB, csrColIndB, &
       beta, &
       descrD, nnzD, csrRowPtrD, csrColIndD, &
       info, pBufferSizeInBytes)
    import cusparseHandle, cusparseMatDescr, cusparseCsrgemm2Info              
    type(cusparseHandle) :: handle       
    real(4) :: alpha, beta
    integer :: m, n, k, nnzA, nnzB, nnzD
    type(cusparseMatDescr) :: descrA, descrB, descrD
    integer, device :: csrRowPtrA(*), csrColIndA(*), csrRowPtrB(*), csrColIndB(*), &
         csrRowPtrD(*), csrColIndD(*)
    type(cusparseCsrgemm2Info) :: info
    integer(8) :: pBufferSizeInBytes
    end function cusparseScsrgemm2_bufferSizeExt_setHPM
    integer function cusparseScsrgemm2_bufferSizeExt_setDPM(handle, m, n, k, &
       alpha, &
       descrA, nnzA, csrRowPtrA, csrColIndA, &
       descrB, nnzB, csrRowPtrB, csrColIndB, &
       beta, &
       descrD, nnzD, csrRowPtrD, csrColIndD, &
       info, pBufferSizeInBytes)
    import cusparseHandle, cusparseMatDescr, cusparseCsrgemm2Info              
    type(cusparseHandle) :: handle       
    real(4), device :: alpha, beta
    integer :: m, n, k, nnzA, nnzB, nnzD
    type(cusparseMatDescr) :: descrA, descrB, descrD
    integer, device :: csrRowPtrA(*), csrColIndA(*), csrRowPtrB(*), csrColIndB(*), &
         csrRowPtrD(*), csrColIndD(*)
    type(cusparseCsrgemm2Info) :: info
    integer(8) :: pBufferSizeInBytes
    end function cusparseScsrgemm2_bufferSizeExt_setDPM
  end interface cusparseScsrgemm2_bufferSizeExt

#ifdef KEEP_V2_INTERFACES
  interface cusparseScsrgemm2_bufferSizeExt_v2
     integer(c_int) function cusparseScsrgemm2_bufferSizeExt_v2(handle, m, n, k, &
          alpha, &
          descrA, nnzA, csrRowPtrA, csrColIndA, &
          descrB, nnzB, csrRowPtrB, csrColIndB, &
          beta, &
          descrD, nnzD, csrRowPtrD, csrColIndD, &
          info, pBufferSizeInBytes) bind(C, name='cusparseScsrgemm2_bufferSizeExt')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr, cusparseCsrgemm2Info              
       type(cusparseHandle), value :: handle       
       !dir$ ignore_tkr (d) alpha, (d) beta
       real(c_float) :: alpha, beta
       integer(c_int), value :: m, n, k, nnzA, nnzB, nnzD
       type(cusparseMatDescr), value :: descrA, descrB, descrD
       integer(c_int), device :: csrRowPtrA(*), csrColIndA(*), csrRowPtrB(*), csrColIndB(*), &
            csrRowPtrD(*), csrColIndD(*)
       type(cusparseCsrgemm2Info), value :: info
       integer(8) :: pBufferSizeInBytes
     end function cusparseScsrgemm2_bufferSizeExt_v2
  end interface cusparseScsrgemm2_bufferSizeExt_v2
#endif
  
! cusparseDcsrgemm2_bufferSizeExt
  interface cusparseDcsrgemm2_bufferSizeExt
    integer function cusparseDcsrgemm2_bufferSizeExt_setHPM(handle, m, n, k, &
       alpha, &
       descrA, nnzA, csrRowPtrA, csrColIndA, &
       descrB, nnzB, csrRowPtrB, csrColIndB, &
       beta, &
       descrD, nnzD, csrRowPtrD, csrColIndD, &
       info, pBufferSizeInBytes)
    import cusparseHandle, cusparseMatDescr, cusparseCsrgemm2Info              
    type(cusparseHandle) :: handle       
    real(8) :: alpha, beta
    integer :: m, n, k, nnzA, nnzB, nnzD
    type(cusparseMatDescr) :: descrA, descrB, descrD
    integer, device :: csrRowPtrA(*), csrColIndA(*), csrRowPtrB(*), csrColIndB(*), &
         csrRowPtrD(*), csrColIndD(*)
    type(cusparseCsrgemm2Info) :: info
    integer(8) :: pBufferSizeInBytes
    end function cusparseDcsrgemm2_bufferSizeExt_setHPM
    integer function cusparseDcsrgemm2_bufferSizeExt_setDPM(handle, m, n, k, &
       alpha, &
       descrA, nnzA, csrRowPtrA, csrColIndA, &
       descrB, nnzB, csrRowPtrB, csrColIndB, &
       beta, &
       descrD, nnzD, csrRowPtrD, csrColIndD, &
       info, pBufferSizeInBytes)
    import cusparseHandle, cusparseMatDescr, cusparseCsrgemm2Info              
    type(cusparseHandle) :: handle       
    real(8), device :: alpha, beta
    integer :: m, n, k, nnzA, nnzB, nnzD
    type(cusparseMatDescr) :: descrA, descrB, descrD
    integer, device :: csrRowPtrA(*), csrColIndA(*), csrRowPtrB(*), csrColIndB(*), &
         csrRowPtrD(*), csrColIndD(*)
    type(cusparseCsrgemm2Info) :: info
    integer(8) :: pBufferSizeInBytes
    end function cusparseDcsrgemm2_bufferSizeExt_setDPM
  end interface cusparseDcsrgemm2_bufferSizeExt

#ifdef KEEP_V2_INTERFACES
  interface cusparseDcsrgemm2_bufferSizeExt_v2
     integer(c_int) function cusparseDcsrgemm2_bufferSizeExt_v2(handle, m, n, k, &
          alpha, &
          descrA, nnzA, csrRowPtrA, csrColIndA, &
          descrB, nnzB, csrRowPtrB, csrColIndB, &
          beta, &
          descrD, nnzD, csrRowPtrD, csrColIndD, &
          info, pBufferSizeInBytes) bind(C, name='cusparseDcsrgemm2_bufferSizeExt')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr, cusparseCsrgemm2Info              
       type(cusparseHandle), value :: handle       
       !dir$ ignore_tkr (d) alpha, (d) beta
       real(c_double) :: alpha, beta
       integer(c_int), value :: m, n, k, nnzA, nnzB, nnzD
       type(cusparseMatDescr), value :: descrA, descrB, descrD
       integer(c_int), device :: csrRowPtrA(*), csrColIndA(*), csrRowPtrB(*), csrColIndB(*), &
            csrRowPtrD(*), csrColIndD(*)
       type(cusparseCsrgemm2Info), value :: info
       integer(8) :: pBufferSizeInBytes
     end function cusparseDcsrgemm2_bufferSizeExt_v2
  end interface cusparseDcsrgemm2_bufferSizeExt_v2
#endif

! cusparseCcsrgemm2_bufferSizeExt
  interface cusparseCcsrgemm2_bufferSizeExt
    integer function cusparseCcsrgemm2_bufferSizeExt_setHPM(handle, m, n, k, &
       alpha, &
       descrA, nnzA, csrRowPtrA, csrColIndA, &
       descrB, nnzB, csrRowPtrB, csrColIndB, &
       beta, &
       descrD, nnzD, csrRowPtrD, csrColIndD, &
       info, pBufferSizeInBytes)
    import cusparseHandle, cusparseMatDescr, cusparseCsrgemm2Info              
    type(cusparseHandle) :: handle       
    complex(4) :: alpha, beta
    integer :: m, n, k, nnzA, nnzB, nnzD
    type(cusparseMatDescr) :: descrA, descrB, descrD
    integer, device :: csrRowPtrA(*), csrColIndA(*), csrRowPtrB(*), csrColIndB(*), &
         csrRowPtrD(*), csrColIndD(*)
    type(cusparseCsrgemm2Info) :: info
    integer(8) :: pBufferSizeInBytes
    end function cusparseCcsrgemm2_bufferSizeExt_setHPM
    integer function cusparseCcsrgemm2_bufferSizeExt_setDPM(handle, m, n, k, &
       alpha, &
       descrA, nnzA, csrRowPtrA, csrColIndA, &
       descrB, nnzB, csrRowPtrB, csrColIndB, &
       beta, &
       descrD, nnzD, csrRowPtrD, csrColIndD, &
       info, pBufferSizeInBytes)
    import cusparseHandle, cusparseMatDescr, cusparseCsrgemm2Info              
    type(cusparseHandle) :: handle       
    complex(4), device :: alpha, beta
    integer :: m, n, k, nnzA, nnzB, nnzD
    type(cusparseMatDescr) :: descrA, descrB, descrD
    integer, device :: csrRowPtrA(*), csrColIndA(*), csrRowPtrB(*), csrColIndB(*), &
         csrRowPtrD(*), csrColIndD(*)
    type(cusparseCsrgemm2Info) :: info
    integer(8) :: pBufferSizeInBytes
    end function cusparseCcsrgemm2_bufferSizeExt_setDPM
  end interface cusparseCcsrgemm2_bufferSizeExt

#ifdef KEEP_V2_INTERFACES
  interface cusparseCcsrgemm2_bufferSizeExt_v2
     integer(c_int) function cusparseCcsrgemm2_bufferSizeExt_v2(handle, m, n, k, &
          alpha, &
          descrA, nnzA, csrRowPtrA, csrColIndA, &
          descrB, nnzB, csrRowPtrB, csrColIndB, &
          beta, &
          descrD, nnzD, csrRowPtrD, csrColIndD, &
          info, pBufferSizeInBytes) bind(C, name='cusparseCcsrgemm2_bufferSizeExt')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr, cusparseCsrgemm2Info              
       type(cusparseHandle), value :: handle       
       !dir$ ignore_tkr (d) alpha, (d) beta
       complex(4) :: alpha, beta
       integer(c_int), value :: m, n, k, nnzA, nnzB, nnzD
       type(cusparseMatDescr), value :: descrA, descrB, descrD
       integer(c_int), device :: csrRowPtrA(*), csrColIndA(*), csrRowPtrB(*), csrColIndB(*), &
            csrRowPtrD(*), csrColIndD(*)
       type(cusparseCsrgemm2Info), value :: info
       integer(8) :: pBufferSizeInBytes
     end function cusparseCcsrgemm2_bufferSizeExt_v2
  end interface cusparseCcsrgemm2_bufferSizeExt_v2
#endif

! cusparseZcsrgemm2_bufferSizeExt
  interface cusparseZcsrgemm2_bufferSizeExt
    integer function cusparseZcsrgemm2_bufferSizeExt_setHPM(handle, m, n, k, &
       alpha, &
       descrA, nnzA, csrRowPtrA, csrColIndA, &
       descrB, nnzB, csrRowPtrB, csrColIndB, &
       beta, &
       descrD, nnzD, csrRowPtrD, csrColIndD, &
       info, pBufferSizeInBytes)
    import cusparseHandle, cusparseMatDescr, cusparseCsrgemm2Info              
    type(cusparseHandle) :: handle       
    complex(8) :: alpha, beta
    integer :: m, n, k, nnzA, nnzB, nnzD
    type(cusparseMatDescr) :: descrA, descrB, descrD
    integer, device :: csrRowPtrA(*), csrColIndA(*), csrRowPtrB(*), csrColIndB(*), &
         csrRowPtrD(*), csrColIndD(*)
    type(cusparseCsrgemm2Info) :: info
    integer(8) :: pBufferSizeInBytes
    end function cusparseZcsrgemm2_bufferSizeExt_setHPM
    integer function cusparseZcsrgemm2_bufferSizeExt_setDPM(handle, m, n, k, &
       alpha, &
       descrA, nnzA, csrRowPtrA, csrColIndA, &
       descrB, nnzB, csrRowPtrB, csrColIndB, &
       beta, &
       descrD, nnzD, csrRowPtrD, csrColIndD, &
       info, pBufferSizeInBytes)
    import cusparseHandle, cusparseMatDescr, cusparseCsrgemm2Info              
    type(cusparseHandle) :: handle       
    complex(8), device :: alpha, beta
    integer :: m, n, k, nnzA, nnzB, nnzD
    type(cusparseMatDescr) :: descrA, descrB, descrD
    integer, device :: csrRowPtrA(*), csrColIndA(*), csrRowPtrB(*), csrColIndB(*), &
         csrRowPtrD(*), csrColIndD(*)
    type(cusparseCsrgemm2Info) :: info
    integer(8) :: pBufferSizeInBytes
    end function cusparseZcsrgemm2_bufferSizeExt_setDPM
  end interface cusparseZcsrgemm2_bufferSizeExt

#ifdef KEEP_V2_INTERFACES
  interface cusparseZcsrgemm2_bufferSizeExt_v2
     integer(c_int) function cusparseZcsrgemm2_bufferSizeExt_v2(handle, m, n, k, &
          alpha, &
          descrA, nnzA, csrRowPtrA, csrColIndA, &
          descrB, nnzB, csrRowPtrB, csrColIndB, &
          beta, &
          descrD, nnzD, csrRowPtrD, csrColIndD, &
          info, pBufferSizeInBytes) bind(C, name='cusparseZcsrgemm2_bufferSizeExt')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr, cusparseCsrgemm2Info              
       type(cusparseHandle), value :: handle       
       !dir$ ignore_tkr (d) alpha, (d) beta
       complex(8) :: alpha, beta
       integer(c_int), value :: m, n, k, nnzA, nnzB, nnzD
       type(cusparseMatDescr), value :: descrA, descrB, descrD
       integer(c_int), device :: csrRowPtrA(*), csrColIndA(*), csrRowPtrB(*), csrColIndB(*), &
            csrRowPtrD(*), csrColIndD(*)
       type(cusparseCsrgemm2Info), value :: info
       integer(8) :: pBufferSizeInBytes
     end function cusparseZcsrgemm2_bufferSizeExt_v2
  end interface cusparseZcsrgemm2_bufferSizeExt_v2
#endif

! cusparseXcsrgemm2Nnz
  interface cusparseXcsrgemm2Nnz
     integer(c_int) function cusparseXcsrgemm2Nnz(handle, &
          m, n, k, &
          descrA, nnzA, csrRowPtrA, csrColIndA, &
          descrB, nnzB, csrRowPtrB, csrColIndB, &
          descrD, nnzD, csrRowPtrD, csrColIndD, &
          descrC, csrRowPtrC, nnzTotalDevHostPtr, info, pBuffer) &
          bind(C, name='cusparseXcsrgemm2Nnz')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr, cusparseCsrgemm2Info
       type(cusparseHandle), value :: handle
       type(cusparseMatDescr), value :: descrA, descrB, descrD, descrC
       type(cusparseCsrgemm2Info), value :: info
       integer(c_int), value :: m, n, k, nnzA, nnzB, nnzD
       integer(c_int), device :: csrRowPtrA(*), csrColIndA(*), &
            csrRowPtrB(*), csrColIndB(*), &
            csrRowPtrD(*), csrColIndD(*), csrRowPtrC(*)
       !dir$ ignore_tkr (d) nnzTotalDevHostPtr
       integer(c_int) :: nnzTotalDevHostPtr
       !dir$ ignore_tkr pBuffer
       character(c_char), device :: pBuffer(*)
     end function cusparseXcsrgemm2Nnz
  end interface cusparseXcsrgemm2Nnz


! cusparseScsrgemm2
  interface cusparseScsrgemm2
    integer function cusparseScsrgemm2_setHPM(handle, &
       m, n, k, alpha, &
       descrA, nnzA, csrValA, csrRowPtrA, csrColIndA, &
       descrB, nnzB, csrValB, csrRowPtrB, csrColIndB, &
       beta, &
       descrD, nnzD, csrValD, csrRowPtrD, csrColIndD, &
       descrC,       csrValC, csrRowPtrC, csrColIndC, &
       info, pBuffer)
    import cusparseHandle, cusparseMatDescr, cusparseCsrgemm2Info              
    type(cusparseHandle) :: handle
    type(cusparseMatDescr) :: descrA, descrB, descrD, descrC
    type(cusparseCsrgemm2Info) :: info
    integer :: m, n, k, nnzA, nnzB, nnzD
    integer, device :: csrRowPtrA(*), csrColIndA(*), &
         csrRowPtrB(*), csrColIndB(*), &
         csrRowPtrD(*), csrColIndD(*), &
         csrRowPtrC(*), csrColIndC(*)
    real(4), device :: csrValA(*), csrValB(*), csrValD(*), csrValC(*)
    real(4) :: alpha, beta
    integer :: nnzTotalDevHostPtr
    !dir$ ignore_tkr pBuffer
    character, device :: pBuffer(*)
    end function cusparseScsrgemm2_setHPM
    integer function cusparseScsrgemm2_setDPM(handle, &
       m, n, k, alpha, &
       descrA, nnzA, csrValA, csrRowPtrA, csrColIndA, &
       descrB, nnzB, csrValB, csrRowPtrB, csrColIndB, &
       beta, &
       descrD, nnzD, csrValD, csrRowPtrD, csrColIndD, &
       descrC,       csrValC, csrRowPtrC, csrColIndC, &
       info, pBuffer)
    import cusparseHandle, cusparseMatDescr, cusparseCsrgemm2Info              
    type(cusparseHandle) :: handle
    type(cusparseMatDescr) :: descrA, descrB, descrD, descrC
    type(cusparseCsrgemm2Info) :: info
    integer :: m, n, k, nnzA, nnzB, nnzD
    integer, device :: csrRowPtrA(*), csrColIndA(*), &
         csrRowPtrB(*), csrColIndB(*), &
         csrRowPtrD(*), csrColIndD(*), &
         csrRowPtrC(*), csrColIndC(*)
    real(4), device :: csrValA(*), csrValB(*), csrValD(*), csrValC(*)
    real(4), device :: alpha, beta
    integer, device :: nnzTotalDevHostPtr
    !dir$ ignore_tkr pBuffer
    character, device :: pBuffer(*)
    end function cusparseScsrgemm2_setDPM
  end interface cusparseScsrgemm2

#ifdef KEEP_V2_INTERFACES
  interface cusparseScsrgemm2_v2
     integer(c_int) function cusparseScsrgemm2_v2(handle, &
          m, n, k, alpha, &
          descrA, nnzA, csrValA, csrRowPtrA, csrColIndA, &
          descrB, nnzB, csrValB, csrRowPtrB, csrColIndB, &
          beta, &
          descrD, nnzD, csrValD, csrRowPtrD, csrColIndD, &
          descrC,       csrValC, csrRowPtrC, csrColIndC, &
          info, pBuffer) bind(C, name='cusparseScsrgemm2')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr, cusparseCsrgemm2Info
       type(cusparseHandle), value :: handle
       type(cusparseMatDescr), value :: descrA, descrB, descrD, descrC
       type(cusparseCsrgemm2Info), value :: info
       integer(c_int), value :: m, n, k, nnzA, nnzB, nnzD
       integer(c_int), device :: csrRowPtrA(*), csrColIndA(*), &
            csrRowPtrB(*), csrColIndB(*), &
            csrRowPtrD(*), csrColIndD(*), &
            csrRowPtrC(*), csrColIndC(*)
       real(c_float), device :: csrValA(*), csrValB(*), csrValD(*), csrValC(*)
       !dir$ ignore_tkr (d) alpha, (d) beta
       real(c_float) :: alpha, beta
       !dir$ ignore_tkr pBuffer
       character(c_char), device :: pBuffer(*)
     end function cusparseScsrgemm2_v2
  end interface cusparseScsrgemm2_v2
#endif
     
! cusparseDcsrgemm2
  interface cusparseDcsrgemm2
    integer function cusparseDcsrgemm2_setHPM(handle, &
       m, n, k, alpha, &
       descrA, nnzA, csrValA, csrRowPtrA, csrColIndA, &
       descrB, nnzB, csrValB, csrRowPtrB, csrColIndB, &
       beta, &
       descrD, nnzD, csrValD, csrRowPtrD, csrColIndD, &
       descrC,       csrValC, csrRowPtrC, csrColIndC, &
       info, pBuffer)
    import cusparseHandle, cusparseMatDescr, cusparseCsrgemm2Info              
    type(cusparseHandle) :: handle
    type(cusparseMatDescr) :: descrA, descrB, descrD, descrC
    type(cusparseCsrgemm2Info) :: info
    integer :: m, n, k, nnzA, nnzB, nnzD
    integer, device :: csrRowPtrA(*), csrColIndA(*), &
         csrRowPtrB(*), csrColIndB(*), &
         csrRowPtrD(*), csrColIndD(*), &
         csrRowPtrC(*), csrColIndC(*)
    real(8), device :: csrValA(*), csrValB(*), csrValD(*), csrValC(*)
    real(8) :: alpha, beta
    integer :: nnzTotalDevHostPtr
    !dir$ ignore_tkr pBuffer
    character, device :: pBuffer(*)
    end function cusparseDcsrgemm2_setHPM
    integer function cusparseDcsrgemm2_setDPM(handle, &
       m, n, k, alpha, &
       descrA, nnzA, csrValA, csrRowPtrA, csrColIndA, &
       descrB, nnzB, csrValB, csrRowPtrB, csrColIndB, &
       beta, &
       descrD, nnzD, csrValD, csrRowPtrD, csrColIndD, &
       descrC,       csrValC, csrRowPtrC, csrColIndC, &
       info, pBuffer)
    import cusparseHandle, cusparseMatDescr, cusparseCsrgemm2Info              
    type(cusparseHandle) :: handle
    type(cusparseMatDescr) :: descrA, descrB, descrD, descrC
    type(cusparseCsrgemm2Info) :: info
    integer :: m, n, k, nnzA, nnzB, nnzD
    integer, device :: csrRowPtrA(*), csrColIndA(*), &
         csrRowPtrB(*), csrColIndB(*), &
         csrRowPtrD(*), csrColIndD(*), &
         csrRowPtrC(*), csrColIndC(*)
    real(8), device :: csrValA(*), csrValB(*), csrValD(*), csrValC(*)
    real(8), device :: alpha, beta
    integer, device :: nnzTotalDevHostPtr
    !dir$ ignore_tkr pBuffer
    character, device :: pBuffer(*)
    end function cusparseDcsrgemm2_setDPM
  end interface cusparseDcsrgemm2

#ifdef KEEP_V2_INTERFACES
  interface cusparseDcsrgemm2_v2
     integer(c_int) function cusparseDcsrgemm2_v2(handle, &
          m, n, k, alpha, &
          descrA, nnzA, csrValA, csrRowPtrA, csrColIndA, &
          descrB, nnzB, csrValB, csrRowPtrB, csrColIndB, &
          beta, &
          descrD, nnzD, csrValD, csrRowPtrD, csrColIndD, &
          descrC,       csrValC, csrRowPtrC, csrColIndC, &
          info, pBuffer) bind(C, name='cusparseDcsrgemm2')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr, cusparseCsrgemm2Info
       type(cusparseHandle), value :: handle
       type(cusparseMatDescr), value :: descrA, descrB, descrD, descrC
       type(cusparseCsrgemm2Info), value :: info
       integer(c_int), value :: m, n, k, nnzA, nnzB, nnzD
       integer(c_int), device :: csrRowPtrA(*), csrColIndA(*), &
            csrRowPtrB(*), csrColIndB(*), &
            csrRowPtrD(*), csrColIndD(*), &
            csrRowPtrC(*), csrColIndC(*)
       real(c_double), device :: csrValA(*), csrValB(*), csrValD(*), csrValC(*)
       !dir$ ignore_tkr (d) alpha, (d) beta
       real(c_double) :: alpha, beta
       !dir$ ignore_tkr pBuffer
       character(c_char), device :: pBuffer(*)
     end function cusparseDcsrgemm2_v2
  end interface cusparseDcsrgemm2_v2
#endif
     
! cusparseCcsrgemm2
  interface cusparseCcsrgemm2
    integer function cusparseCcsrgemm2_setHPM(handle, &
       m, n, k, alpha, &
       descrA, nnzA, csrValA, csrRowPtrA, csrColIndA, &
       descrB, nnzB, csrValB, csrRowPtrB, csrColIndB, &
       beta, &
       descrD, nnzD, csrValD, csrRowPtrD, csrColIndD, &
       descrC,       csrValC, csrRowPtrC, csrColIndC, &
       info, pBuffer)
    import cusparseHandle, cusparseMatDescr, cusparseCsrgemm2Info              
    type(cusparseHandle) :: handle
    type(cusparseMatDescr) :: descrA, descrB, descrD, descrC
    type(cusparseCsrgemm2Info) :: info
    integer :: m, n, k, nnzA, nnzB, nnzD
    integer, device :: csrRowPtrA(*), csrColIndA(*), &
         csrRowPtrB(*), csrColIndB(*), &
         csrRowPtrD(*), csrColIndD(*), &
         csrRowPtrC(*), csrColIndC(*)
    complex(4), device :: csrValA(*), csrValB(*), csrValD(*), csrValC(*)
    complex(4) :: alpha, beta
    integer :: nnzTotalDevHostPtr
    !dir$ ignore_tkr pBuffer
    character, device :: pBuffer(*)
    end function cusparseCcsrgemm2_setHPM
    integer function cusparseCcsrgemm2_setDPM(handle, &
       m, n, k, alpha, &
       descrA, nnzA, csrValA, csrRowPtrA, csrColIndA, &
       descrB, nnzB, csrValB, csrRowPtrB, csrColIndB, &
       beta, &
       descrD, nnzD, csrValD, csrRowPtrD, csrColIndD, &
       descrC,       csrValC, csrRowPtrC, csrColIndC, &
       info, pBuffer)
    import cusparseHandle, cusparseMatDescr, cusparseCsrgemm2Info              
    type(cusparseHandle) :: handle
    type(cusparseMatDescr) :: descrA, descrB, descrD, descrC
    type(cusparseCsrgemm2Info) :: info
    integer :: m, n, k, nnzA, nnzB, nnzD
    integer, device :: csrRowPtrA(*), csrColIndA(*), &
         csrRowPtrB(*), csrColIndB(*), &
         csrRowPtrD(*), csrColIndD(*), &
         csrRowPtrC(*), csrColIndC(*)
    complex(4), device :: csrValA(*), csrValB(*), csrValD(*), csrValC(*)
    complex(4), device :: alpha, beta
    integer, device :: nnzTotalDevHostPtr
    !dir$ ignore_tkr pBuffer
    character, device :: pBuffer(*)
    end function cusparseCcsrgemm2_setDPM
  end interface cusparseCcsrgemm2

#ifdef KEEP_V2_INTERFACES
  interface cusparseCcsrgemm2_v2
     integer(c_int) function cusparseCcsrgemm2_v2(handle, &
          m, n, k, alpha, &
          descrA, nnzA, csrValA, csrRowPtrA, csrColIndA, &
          descrB, nnzB, csrValB, csrRowPtrB, csrColIndB, &
          beta, &
          descrD, nnzD, csrValD, csrRowPtrD, csrColIndD, &
          descrC,       csrValC, csrRowPtrC, csrColIndC, &
          info, pBuffer) bind(C, name='cusparseCcsrgemm2')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr, cusparseCsrgemm2Info
       type(cusparseHandle), value :: handle
       type(cusparseMatDescr), value :: descrA, descrB, descrD, descrC
       type(cusparseCsrgemm2Info), value :: info
       integer(c_int), value :: m, n, k, nnzA, nnzB, nnzD
       integer(c_int), device :: csrRowPtrA(*), csrColIndA(*), &
            csrRowPtrB(*), csrColIndB(*), &
            csrRowPtrD(*), csrColIndD(*), &
            csrRowPtrC(*), csrColIndC(*)
       complex(4), device :: csrValA(*), csrValB(*), csrValD(*), csrValC(*)
       !dir$ ignore_tkr (d) alpha, (d) beta
       complex(4) :: alpha, beta
       !dir$ ignore_tkr pBuffer
       character(c_char), device :: pBuffer(*)
     end function cusparseCcsrgemm2_v2
  end interface cusparseCcsrgemm2_v2
#endif

! cusparseZcsrgemm2
  interface cusparseZcsrgemm2
    integer function cusparseZcsrgemm2_setHPM(handle, &
       m, n, k, alpha, &
       descrA, nnzA, csrValA, csrRowPtrA, csrColIndA, &
       descrB, nnzB, csrValB, csrRowPtrB, csrColIndB, &
       beta, &
       descrD, nnzD, csrValD, csrRowPtrD, csrColIndD, &
       descrC,       csrValC, csrRowPtrC, csrColIndC, &
       info, pBuffer)
    import cusparseHandle, cusparseMatDescr, cusparseCsrgemm2Info              
    type(cusparseHandle) :: handle
    type(cusparseMatDescr) :: descrA, descrB, descrD, descrC
    type(cusparseCsrgemm2Info) :: info
    integer :: m, n, k, nnzA, nnzB, nnzD
    integer, device :: csrRowPtrA(*), csrColIndA(*), &
         csrRowPtrB(*), csrColIndB(*), &
         csrRowPtrD(*), csrColIndD(*), &
         csrRowPtrC(*), csrColIndC(*)
    complex(8), device :: csrValA(*), csrValB(*), csrValD(*), csrValC(*)
    complex(8) :: alpha, beta
    integer :: nnzTotalDevHostPtr
    !dir$ ignore_tkr pBuffer
    character, device :: pBuffer(*)
    end function cusparseZcsrgemm2_setHPM
    integer function cusparseZcsrgemm2_setDPM(handle, &
       m, n, k, alpha, &
       descrA, nnzA, csrValA, csrRowPtrA, csrColIndA, &
       descrB, nnzB, csrValB, csrRowPtrB, csrColIndB, &
       beta, &
       descrD, nnzD, csrValD, csrRowPtrD, csrColIndD, &
       descrC,       csrValC, csrRowPtrC, csrColIndC, &
       info, pBuffer)
    import cusparseHandle, cusparseMatDescr, cusparseCsrgemm2Info              
    type(cusparseHandle) :: handle
    type(cusparseMatDescr) :: descrA, descrB, descrD, descrC
    type(cusparseCsrgemm2Info) :: info
    integer :: m, n, k, nnzA, nnzB, nnzD
    integer, device :: csrRowPtrA(*), csrColIndA(*), &
         csrRowPtrB(*), csrColIndB(*), &
         csrRowPtrD(*), csrColIndD(*), &
         csrRowPtrC(*), csrColIndC(*)
    complex(8), device :: csrValA(*), csrValB(*), csrValD(*), csrValC(*)
    complex(8), device :: alpha, beta
    integer, device :: nnzTotalDevHostPtr
    !dir$ ignore_tkr pBuffer
    character, device :: pBuffer(*)
    end function cusparseZcsrgemm2_setDPM
  end interface cusparseZcsrgemm2
  
#ifdef KEEP_V2_INTERFACES
  interface cusparseZcsrgemm2_v2
     integer(c_int) function cusparseZcsrgemm2_v2(handle, &
          m, n, k, alpha, &
          descrA, nnzA, csrValA, csrRowPtrA, csrColIndA, &
          descrB, nnzB, csrValB, csrRowPtrB, csrColIndB, &
          beta, &
          descrD, nnzD, csrValD, csrRowPtrD, csrColIndD, &
          descrC,       csrValC, csrRowPtrC, csrColIndC, &
          info, pBuffer) bind(C, name='cusparseZcsrgemm2')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr, cusparseCsrgemm2Info
       type(cusparseHandle), value :: handle
       type(cusparseMatDescr), value :: descrA, descrB, descrD, descrC
       type(cusparseCsrgemm2Info), value :: info
       integer(c_int), value :: m, n, k, nnzA, nnzB, nnzD
       integer(c_int), device :: csrRowPtrA(*), csrColIndA(*), &
            csrRowPtrB(*), csrColIndB(*), &
            csrRowPtrD(*), csrColIndD(*), &
            csrRowPtrC(*), csrColIndC(*)
       complex(8), device :: csrValA(*), csrValB(*), csrValD(*), csrValC(*)
       !dir$ ignore_tkr (d) alpha, (d) beta
       complex(8) :: alpha, beta
       !dir$ ignore_tkr pBuffer
       character(c_char), device :: pBuffer(*)
     end function cusparseZcsrgemm2_v2
  end interface cusparseZcsrgemm2_v2
#endif
     
  ! -------------------
  ! 5.5 Preconditioners
  ! -------------------

! cusparseScsric0
  interface 
     integer(c_int) function cusparseScsric0(handle, trans, m, &
          descrA, csrValM, csrRowPtrA, csrColIndA, info) &
          bind(C, name='cusparseScsric0')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr, cusparseSolveAnalysisInfo             
       type(cusparseHandle), value :: handle       
       integer(c_int), value :: trans, m
       type(cusparseMatDescr), value :: descrA
       real(c_float), device :: csrValM(*)
       integer(c_int), device :: csrRowPtrA(*), csrColIndA(*)
       type(cusparseSolveAnalysisInfo), value :: info
     end function cusparseScsric0
  end interface

! cusparseDcsric0
  interface 
     integer(c_int) function cusparseDcsric0(handle, trans, m, &
          descrA, csrValM, csrRowPtrA, csrColIndA, info) &
          bind(C, name='cusparseDcsric0')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr, cusparseSolveAnalysisInfo             
       type(cusparseHandle), value :: handle       
       integer(c_int), value :: trans, m
       type(cusparseMatDescr), value :: descrA
       real(c_double), device :: csrValM(*)
       integer(c_int), device :: csrRowPtrA(*), csrColIndA(*)
       type(cusparseSolveAnalysisInfo), value :: info
     end function cusparseDcsric0
  end interface

! cusparseCcsric0
  interface 
     integer(c_int) function cusparseCcsric0(handle, trans, m, &
          descrA, csrValM, csrRowPtrA, csrColIndA, info) &
          bind(C, name='cusparseCcsric0')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr, cusparseSolveAnalysisInfo             
       type(cusparseHandle), value :: handle       
       integer(c_int), value :: trans, m
       type(cusparseMatDescr), value :: descrA
       complex(4), device :: csrValM(*)
       integer(c_int), device :: csrRowPtrA(*), csrColIndA(*)
       type(cusparseSolveAnalysisInfo), value :: info
     end function cusparseCcsric0
  end interface

! cusparseZcsric0
  interface 
     integer(c_int) function cusparseZcsric0(handle, trans, m, &
          descrA, csrValM, csrRowPtrA, csrColIndA, info) &
          bind(C, name='cusparseZcsric0')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr, cusparseSolveAnalysisInfo             
       type(cusparseHandle), value :: handle       
       integer(c_int), value :: trans, m
       type(cusparseMatDescr), value :: descrA
       complex(8), device :: csrValM(*)
       integer(c_int), device :: csrRowPtrA(*), csrColIndA(*)
       type(cusparseSolveAnalysisInfo), value :: info
     end function cusparseZcsric0
  end interface

! cusparseScsrilu0
  interface 
     integer(c_int) function cusparseScsrilu0(handle, trans, m, &
          descrA, csrValM, csrRowPtrA, csrColIndA, info) &
          bind(C, name='cusparseScsrilu0')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr, cusparseSolveAnalysisInfo             
       type(cusparseHandle), value :: handle       
       integer(c_int), value :: trans, m
       type(cusparseMatDescr), value :: descrA
       real(c_float), device :: csrValM(*)
       integer(c_int), device :: csrRowPtrA(*), csrColIndA(*)
       type(cusparseSolveAnalysisInfo), value :: info
     end function cusparseScsrilu0
  end interface

! cusparseDcsrilu0
  interface 
     integer(c_int) function cusparseDcsrilu0(handle, trans, m, &
          descrA, csrValM, csrRowPtrA, csrColIndA, info) &
          bind(C, name='cusparseDcsrilu0')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr, cusparseSolveAnalysisInfo             
       type(cusparseHandle), value :: handle       
       integer(c_int), value :: trans, m
       type(cusparseMatDescr), value :: descrA
       real(c_double), device :: csrValM(*)
       integer(c_int), device :: csrRowPtrA(*), csrColIndA(*)
       type(cusparseSolveAnalysisInfo), value :: info
     end function cusparseDcsrilu0
  end interface

! cusparseCcsrilu0
  interface 
     integer(c_int) function cusparseCcsrilu0(handle, trans, m, &
          descrA, csrValM, csrRowPtrA, csrColIndA, info) &
          bind(C, name='cusparseCcsrilu0')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr, cusparseSolveAnalysisInfo             
       type(cusparseHandle), value :: handle       
       integer(c_int), value :: trans, m
       type(cusparseMatDescr), value :: descrA
       complex(4), device :: csrValM(*)
       integer(c_int), device :: csrRowPtrA(*), csrColIndA(*)
       type(cusparseSolveAnalysisInfo), value :: info
     end function cusparseCcsrilu0
  end interface

! cusparseZcsrilu0
  interface 
     integer(c_int) function cusparseZcsrilu0(handle, trans, m, &
          descrA, csrValM, csrRowPtrA, csrColIndA, info) &
          bind(C, name='cusparseZcsrilu0')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr, cusparseSolveAnalysisInfo             
       type(cusparseHandle), value :: handle       
       integer(c_int), value :: trans, m
       type(cusparseMatDescr), value :: descrA
       complex(8), device :: csrValM(*)
       integer(c_int), device :: csrRowPtrA(*), csrColIndA(*)
       type(cusparseSolveAnalysisInfo), value :: info
     end function cusparseZcsrilu0
  end interface

! cusparseSgtsv
  interface
     integer(c_int) function cusparseSgtsv(handle, m, n, dl, d, du, B, ldb) bind(C,name='cusparseSgtsv')
       use iso_c_binding
       import cusparseHandle
       type(cusparseHandle), value :: handle       
       integer(c_int), value :: m, n, ldb
       real(c_float), device :: dl(*), d(*), du(*), B(*)
     end function cusparseSgtsv
  end interface

! cusparseDgtsv
  interface
     integer(c_int) function cusparseDgtsv(handle, m, n, dl, d, du, B, ldb) bind(C,name='cusparseDgtsv')
       use iso_c_binding
       import cusparseHandle
       type(cusparseHandle), value :: handle       
       integer(c_int), value :: m, n, ldb
       real(c_double), device :: dl(*), d(*), du(*), B(*)
     end function cusparseDgtsv
  end interface

! cusparseCgtsv
  interface
     integer(c_int) function cusparseCgtsv(handle, m, n, dl, d, du, B, ldb) bind(C,name='cusparseCgtsv')
       use iso_c_binding
       import cusparseHandle
       type(cusparseHandle), value :: handle       
       integer(c_int), value :: m, n, ldb
       complex(4), device :: dl(*), d(*), du(*), B(*)
     end function cusparseCgtsv
  end interface

! cusparseZgtsv
  interface
     integer(c_int) function cusparseZgtsv(handle, m, n, dl, d, du, B, ldb) bind(C,name='cusparseZgtsv')
       use iso_c_binding
       import cusparseHandle
       type(cusparseHandle), value :: handle       
       integer(c_int), value :: m, n, ldb
       complex(8), device :: dl(*), d(*), du(*), B(*)
     end function cusparseZgtsv
  end interface

! cusparseSgtsv_nopivot
  interface
     integer(c_int) function cusparseSgtsv_nopivot(handle, m, n, dl, d, du, B, ldb) &
          bind(C,name='cusparseSgtsv_nopivot')
       use iso_c_binding
       import cusparseHandle
       type(cusparseHandle), value :: handle       
       integer(c_int), value :: m, n, ldb
       real(c_float), device :: dl(*), d(*), du(*), B(*)
     end function cusparseSgtsv_nopivot
  end interface

! cusparseDgtsv_nopivot
  interface
     integer(c_int) function cusparseDgtsv_nopivot(handle, m, n, dl, d, du, B, ldb) &
          bind(C,name='cusparseDgtsv_nopivot')
       use iso_c_binding
       import cusparseHandle
       type(cusparseHandle), value :: handle       
       integer(c_int), value :: m, n, ldb
       real(c_double), device :: dl(*), d(*), du(*), B(*)
     end function cusparseDgtsv_nopivot
  end interface

! cusparseCgtsv_nopivot
  interface
     integer(c_int) function cusparseCgtsv_nopivot(handle, m, n, dl, d, du, B, ldb) &
          bind(C,name='cusparseCgtsv_nopivot')
       use iso_c_binding
       import cusparseHandle
       type(cusparseHandle), value :: handle       
       integer(c_int), value :: m, n, ldb
       complex(4), device :: dl(*), d(*), du(*), B(*)
     end function cusparseCgtsv_nopivot
  end interface

! cusparseZgtsv_nopivot
  interface
     integer(c_int) function cusparseZgtsv_nopivot(handle, m, n, dl, d, du, B, ldb) &
          bind(C,name='cusparseZgtsv_nopivot')
       use iso_c_binding
       import cusparseHandle
       type(cusparseHandle), value :: handle       
       integer(c_int), value :: m, n, ldb
       complex(8), device :: dl(*), d(*), du(*), B(*)
     end function cusparseZgtsv_nopivot
  end interface

! cusparseSgtsvStridedBatch
  interface
     integer(c_int) function cusparseSgtsvStridedBatch(handle, m, dl, d, du, x, batchCount, batchStride) &
          bind(C,name='cusparseSgtsvStridedBatch')
       use iso_c_binding
       import cusparseHandle
       type(cusparseHandle), value :: handle       
       integer(c_int), value :: m, batchCount, batchStride
       real(c_float), device :: dl(*), d(*), du(*), x(*)
     end function cusparseSgtsvStridedBatch
  end interface

! cusparseDgtsvStridedBatch
  interface
     integer(c_int) function cusparseDgtsvStridedBatch(handle, m, dl, d, du, x, batchCount, batchStride) &
          bind(C,name='cusparseDgtsvStridedBatch')
       use iso_c_binding
       import cusparseHandle
       type(cusparseHandle), value :: handle       
       integer(c_int), value :: m, batchCount, batchStride
       real(c_double), device :: dl(*), d(*), du(*), x(*)
     end function cusparseDgtsvStridedBatch
  end interface

! cusparseCgtsvStridedBatch
  interface
     integer(c_int) function cusparseCgtsvStridedBatch(handle, m, dl, d, du, x, batchCount, batchStride) &
          bind(C,name='cusparseCgtsvStridedBatch')
       use iso_c_binding
       import cusparseHandle
       type(cusparseHandle), value :: handle       
       integer(c_int), value :: m, batchCount, batchStride
       complex(4), device :: dl(*), d(*), du(*), x(*)
     end function cusparseCgtsvStridedBatch
  end interface

! cusparseZgtsvStridedBatch
  interface
     integer(c_int) function cusparseZgtsvStridedBatch(handle, m, dl, d, du, x, batchCount, batchStride) &
          bind(C,name='cusparseZgtsvStridedBatch')
       use iso_c_binding
       import cusparseHandle
       type(cusparseHandle), value :: handle       
       integer(c_int), value :: m, batchCount, batchStride
       complex(8), device :: dl(*), d(*), du(*), x(*)
     end function cusparseZgtsvStridedBatch
  end interface

!
! --- New in CUDA 10.1  BBBBB
!
! Tridiagonal solve with buffer space
!
! cusparseSgtsv2_bufferSize
  interface cusparseSgtsv2_bufferSize
     integer(c_int) function cusparseSgtsv2_bufferSizeExt(handle, &
          m, n, dl, d, du, B, ldb, pBufferSizeInBytes) &
          bind(C,name='cusparseSgtsv2_bufferSizeExt')
       import cusparseHandle
       type(cusparseHandle), value :: handle       
       integer(4), value :: m, n, ldb
       real(4), device :: dl(*), d(*), du(*), B(*)
       integer(8) :: pBufferSizeInBytes
     end function cusparseSgtsv2_bufferSizeExt
  end interface

! cusparseDgtsv2_bufferSize
  interface cusparseDgtsv2_bufferSize
     integer(c_int) function cusparseDgtsv2_bufferSizeExt(handle, &
          m, n, dl, d, du, B, ldb, pBufferSizeInBytes) &
          bind(C,name='cusparseDgtsv2_bufferSizeExt')
       import cusparseHandle
       type(cusparseHandle), value :: handle       
       integer(4), value :: m, n, ldb
       real(8), device :: dl(*), d(*), du(*), B(*)
       integer(8) :: pBufferSizeInBytes
     end function cusparseDgtsv2_bufferSizeExt
  end interface

! cusparseCgtsv2_bufferSize
  interface cusparseCgtsv2_bufferSize
     integer(c_int) function cusparseCgtsv2_bufferSizeExt(handle, &
          m, n, dl, d, du, B, ldb, pBufferSizeInBytes) &
          bind(C,name='cusparseCgtsv2_bufferSizeExt')
       import cusparseHandle
       type(cusparseHandle), value :: handle       
       integer(4), value :: m, n, ldb
       complex(4), device :: dl(*), d(*), du(*), B(*)
       integer(8) :: pBufferSizeInBytes
     end function cusparseCgtsv2_bufferSizeExt
  end interface

! cusparseZgtsv2_bufferSize
  interface cusparseZgtsv2_bufferSize
     integer(c_int) function cusparseZgtsv2_bufferSizeExt(handle, &
          m, n, dl, d, du, B, ldb, pBufferSizeInBytes) &
          bind(C,name='cusparseZgtsv2_bufferSizeExt')
       import cusparseHandle
       type(cusparseHandle), value :: handle       
       integer(4), value :: m, n, ldb
       complex(8), device :: dl(*), d(*), du(*), B(*)
       integer(8) :: pBufferSizeInBytes
     end function cusparseZgtsv2_bufferSizeExt
  end interface

! cusparseSgtsv2
  interface
     integer(c_int) function cusparseSgtsv2(handle, m, n, dl, d, du, B, ldb, pBuffer) bind(C,name='cusparseSgtsv2')
       import cusparseHandle
       type(cusparseHandle), value :: handle       
       integer(4), value :: m, n, ldb
       real(4), device :: dl(*), d(*), du(*), B(*)
       !dir$ ignore_tkr pBuffer
       character*1, device :: pBuffer(*)
     end function cusparseSgtsv2
  end interface

! cusparseDgtsv2
  interface
     integer(c_int) function cusparseDgtsv2(handle, m, n, dl, d, du, B, ldb, pBuffer) bind(C,name='cusparseDgtsv2')
       import cusparseHandle
       type(cusparseHandle), value :: handle       
       integer(4), value :: m, n, ldb
       real(8), device :: dl(*), d(*), du(*), B(*)
       !dir$ ignore_tkr pBuffer
       character*1, device :: pBuffer(*)
     end function cusparseDgtsv2
  end interface

! cusparseCgtsv2
  interface
     integer(c_int) function cusparseCgtsv2(handle, m, n, dl, d, du, B, ldb, pBuffer) bind(C,name='cusparseCgtsv2')
       import cusparseHandle
       type(cusparseHandle), value :: handle       
       integer(4), value :: m, n, ldb
       complex(4), device :: dl(*), d(*), du(*), B(*)
       !dir$ ignore_tkr pBuffer
       character*1, device :: pBuffer(*)
     end function cusparseCgtsv2
  end interface

! cusparseZgtsv2
  interface
     integer(c_int) function cusparseZgtsv2(handle, m, n, dl, d, du, B, ldb, pBuffer) bind(C,name='cusparseZgtsv2')
       import cusparseHandle
       type(cusparseHandle), value :: handle       
       integer(4), value :: m, n, ldb
       complex(8), device :: dl(*), d(*), du(*), B(*)
       !dir$ ignore_tkr pBuffer
       character*1, device :: pBuffer(*)
     end function cusparseZgtsv2
  end interface

! Tridiagonal solve, no pivot, with buffer space
!
! cusparseSgtsv2_nopivot_bufferSize
  interface cusparseSgtsv2_nopivot_bufferSize
     integer(c_int) function cusparseSgtsv2_nopivot_bufferSizeExt(handle, &
          m, n, dl, d, du, B, ldb, pBufferSizeInBytes) &
          bind(C,name='cusparseSgtsv2_nopivot_bufferSizeExt')
       import cusparseHandle
       type(cusparseHandle), value :: handle       
       integer(4), value :: m, n, ldb
       real(4), device :: dl(*), d(*), du(*), B(*)
       integer(8) :: pBufferSizeInBytes
     end function cusparseSgtsv2_nopivot_bufferSizeExt
  end interface

! cusparseDgtsv2_nopivot_bufferSize
  interface cusparseDgtsv2_nopivot_bufferSize
     integer(c_int) function cusparseDgtsv2_nopivot_bufferSizeExt(handle, &
          m, n, dl, d, du, B, ldb, pBufferSizeInBytes) &
          bind(C,name='cusparseDgtsv2_nopivot_bufferSizeExt')
       import cusparseHandle
       type(cusparseHandle), value :: handle       
       integer(4), value :: m, n, ldb
       real(8), device :: dl(*), d(*), du(*), B(*)
       integer(8) :: pBufferSizeInBytes
     end function cusparseDgtsv2_nopivot_bufferSizeExt
  end interface

! cusparseCgtsv2_nopivot_bufferSize
  interface cusparseCgtsv2_nopivot_bufferSize
     integer(c_int) function cusparseCgtsv2_nopivot_bufferSizeExt(handle, &
          m, n, dl, d, du, B, ldb, pBufferSizeInBytes) &
          bind(C,name='cusparseCgtsv2_nopivot_bufferSizeExt')
       import cusparseHandle
       type(cusparseHandle), value :: handle       
       integer(4), value :: m, n, ldb
       complex(4), device :: dl(*), d(*), du(*), B(*)
       integer(8) :: pBufferSizeInBytes
     end function cusparseCgtsv2_nopivot_bufferSizeExt
  end interface

! cusparseZgtsv2_nopivot_bufferSize
  interface cusparseZgtsv2_nopivot_bufferSize
     integer(c_int) function cusparseZgtsv2_nopivot_bufferSizeExt(handle, &
          m, n, dl, d, du, B, ldb, pBufferSizeInBytes) &
          bind(C,name='cusparseZgtsv2_nopivot_bufferSizeExt')
       import cusparseHandle
       type(cusparseHandle), value :: handle       
       integer(4), value :: m, n, ldb
       complex(8), device :: dl(*), d(*), du(*), B(*)
       integer(8) :: pBufferSizeInBytes
     end function cusparseZgtsv2_nopivot_bufferSizeExt
  end interface

! cusparseSgtsv2_nopivot
  interface
     integer(c_int) function cusparseSgtsv2_nopivot(handle, m, n, dl, d, du, B, ldb, pBuffer) bind(C,name='cusparseSgtsv2_nopivot')
       import cusparseHandle
       type(cusparseHandle), value :: handle       
       integer(4), value :: m, n, ldb
       real(4), device :: dl(*), d(*), du(*), B(*)
       !dir$ ignore_tkr pBuffer
       character*1, device :: pBuffer(*)
     end function cusparseSgtsv2_nopivot
  end interface

! cusparseDgtsv2_nopivot
  interface
     integer(c_int) function cusparseDgtsv2_nopivot(handle, m, n, dl, d, du, B, ldb, pBuffer) bind(C,name='cusparseDgtsv2_nopivot')
       import cusparseHandle
       type(cusparseHandle), value :: handle       
       integer(4), value :: m, n, ldb
       real(8), device :: dl(*), d(*), du(*), B(*)
       !dir$ ignore_tkr pBuffer
       character*1, device :: pBuffer(*)
     end function cusparseDgtsv2_nopivot
  end interface

! cusparseCgtsv2_nopivot
  interface
     integer(c_int) function cusparseCgtsv2_nopivot(handle, m, n, dl, d, du, B, ldb, pBuffer) bind(C,name='cusparseCgtsv2_nopivot')
       import cusparseHandle
       type(cusparseHandle), value :: handle       
       integer(4), value :: m, n, ldb
       complex(4), device :: dl(*), d(*), du(*), B(*)
       !dir$ ignore_tkr pBuffer
       character*1, device :: pBuffer(*)
     end function cusparseCgtsv2_nopivot
  end interface

! cusparseZgtsv2_nopivot
  interface
     integer(c_int) function cusparseZgtsv2_nopivot(handle, m, n, dl, d, du, B, ldb, pBuffer) bind(C,name='cusparseZgtsv2_nopivot')
       import cusparseHandle
       type(cusparseHandle), value :: handle       
       integer(4), value :: m, n, ldb
       complex(8), device :: dl(*), d(*), du(*), B(*)
       !dir$ ignore_tkr pBuffer
       character*1, device :: pBuffer(*)
     end function cusparseZgtsv2_nopivot
  end interface
!
! Strided Batched
!
! cusparseSgtsv2StridedBatch_bufferSize
  interface cusparseSgtsv2StridedBatch_bufferSize
     integer(4) function cusparseSgtsv2StridedBatch_bufferSizeExt(handle, &
          m, dl, d, du, x, batchCount, batchStride, pBufferSizeInBytes) &
          bind(C, name='cusparseSgtsv2StridedBatch_bufferSizeExt')
       import cusparseHandle
       type(cusparseHandle), value :: handle       
       integer(4), value :: m, batchCount, batchStride
       real(4), device :: dl(*), d(*), du(*), x(*)
       integer(8) :: pBufferSizeInBytes
     end function cusparseSgtsv2StridedBatch_bufferSizeExt
  end interface cusparseSgtsv2StridedBatch_bufferSize

! cusparseDgtsv2StridedBatch_bufferSize
  interface cusparseDgtsv2StridedBatch_bufferSize
     integer(4) function cusparseDgtsv2StridedBatch_bufferSizeExt(handle, &
          m, dl, d, du, x, batchCount, batchStride, pBufferSizeInBytes) &
          bind(C, name='cusparseDgtsv2StridedBatch_bufferSizeExt')
       import cusparseHandle
       type(cusparseHandle), value :: handle       
       integer(4), value :: m, batchCount, batchStride
       real(8), device :: dl(*), d(*), du(*), x(*)
       integer(8) :: pBufferSizeInBytes
     end function cusparseDgtsv2StridedBatch_bufferSizeExt
  end interface cusparseDgtsv2StridedBatch_bufferSize

! cusparseCgtsv2StridedBatch_bufferSize
  interface cusparseCgtsv2StridedBatch_bufferSize
     integer(4) function cusparseCgtsv2StridedBatch_bufferSizeExt(handle, &
          m, dl, d, du, x, batchCount, batchStride, pBufferSizeInBytes) &
          bind(C, name='cusparseCgtsv2StridedBatch_bufferSizeExt')
       import cusparseHandle
       type(cusparseHandle), value :: handle       
       integer(4), value :: m, batchCount, batchStride
       complex(4), device :: dl(*), d(*), du(*), x(*)
       integer(8) :: pBufferSizeInBytes
     end function cusparseCgtsv2StridedBatch_bufferSizeExt
  end interface cusparseCgtsv2StridedBatch_bufferSize

! cusparseZgtsv2StridedBatch_bufferSize
  interface cusparseZgtsv2StridedBatch_bufferSize
     integer(4) function cusparseZgtsv2StridedBatch_bufferSizeExt(handle, &
          m, dl, d, du, x, batchCount, batchStride, pBufferSizeInBytes) &
          bind(C, name='cusparseZgtsv2StridedBatch_bufferSizeExt')
       import cusparseHandle
       type(cusparseHandle), value :: handle       
       integer(4), value :: m, batchCount, batchStride
       complex(8), device :: dl(*), d(*), du(*), x(*)
       integer(8) :: pBufferSizeInBytes
     end function cusparseZgtsv2StridedBatch_bufferSizeExt
  end interface cusparseZgtsv2StridedBatch_bufferSize

! cusparseSgtsv2StridedBatch
  interface
     integer(4) function cusparseSgtsv2StridedBatch(handle, m, dl, d, du, x, batchCount, batchStride, pBuffer) &
          bind(C,name='cusparseSgtsv2StridedBatch')
       import cusparseHandle
       type(cusparseHandle), value :: handle       
       integer(4), value :: m, batchCount, batchStride
       real(4), device :: dl(*), d(*), du(*), x(*)
       !dir$ ignore_tkr pBuffer
       character*1, device :: pBuffer(*)
     end function cusparseSgtsv2StridedBatch
  end interface

! cusparseDgtsv2StridedBatch
  interface
     integer(4) function cusparseDgtsv2StridedBatch(handle, m, dl, d, du, x, batchCount, batchStride, pBuffer) &
          bind(C,name='cusparseDgtsv2StridedBatch')
       import cusparseHandle
       type(cusparseHandle), value :: handle       
       integer(4), value :: m, batchCount, batchStride
       real(8), device :: dl(*), d(*), du(*), x(*)
       !dir$ ignore_tkr pBuffer
       character*1, device :: pBuffer(*)
     end function cusparseDgtsv2StridedBatch
  end interface

! cusparseCgtsv2StridedBatch
  interface
     integer(4) function cusparseCgtsv2StridedBatch(handle, m, dl, d, du, x, batchCount, batchStride, pBuffer) &
          bind(C,name='cusparseCgtsv2StridedBatch')
       import cusparseHandle
       type(cusparseHandle), value :: handle       
       integer(4), value :: m, batchCount, batchStride
       complex(4), device :: dl(*), d(*), du(*), x(*)
       !dir$ ignore_tkr pBuffer
       character*1, device :: pBuffer(*)
     end function cusparseCgtsv2StridedBatch
  end interface

! cusparseZgtsv2StridedBatch
  interface
     integer(4) function cusparseZgtsv2StridedBatch(handle, m, dl, d, du, x, batchCount, batchStride, pBuffer) &
          bind(C,name='cusparseZgtsv2StridedBatch')
       import cusparseHandle
       type(cusparseHandle), value :: handle       
       integer(4), value :: m, batchCount, batchStride
       complex(8), device :: dl(*), d(*), du(*), x(*)
       !dir$ ignore_tkr pBuffer
       character*1, device :: pBuffer(*)
     end function cusparseZgtsv2StridedBatch
  end interface
!
! Interleaved Batched Tridiagonal
!
! cusparseSgtsvInterleavedBatch_bufferSize
  interface cusparseSgtsvInterleavedBatch_bufferSize
     integer(4) function cusparseSgtsvInterleavedBatch_bufferSizeExt(handle, &
          algo, m, dl, d, du, x, batchCount, pBufferSizeInBytes) &
          bind(C, name='cusparseSgtsvInterleavedBatch_bufferSizeExt')
       import cusparseHandle
       type(cusparseHandle), value :: handle       
       integer(4), value :: algo, m, batchCount
       real(4), device :: dl(*), d(*), du(*), x(*)
       integer(8) :: pBufferSizeInBytes
     end function cusparseSgtsvInterleavedBatch_bufferSizeExt
  end interface cusparseSgtsvInterleavedBatch_bufferSize

! cusparseDgtsvInterleavedBatch_bufferSize
  interface cusparseDgtsvInterleavedBatch_bufferSize
     integer(4) function cusparseDgtsvInterleavedBatch_bufferSizeExt(handle, &
          algo, m, dl, d, du, x, batchCount, pBufferSizeInBytes) &
          bind(C, name='cusparseDgtsvInterleavedBatch_bufferSizeExt')
       import cusparseHandle
       type(cusparseHandle), value :: handle       
       integer(4), value :: algo, m, batchCount
       real(8), device :: dl(*), d(*), du(*), x(*)
       integer(8) :: pBufferSizeInBytes
     end function cusparseDgtsvInterleavedBatch_bufferSizeExt
  end interface cusparseDgtsvInterleavedBatch_bufferSize

! cusparseCgtsvInterleavedBatch_bufferSize
  interface cusparseCgtsvInterleavedBatch_bufferSize
     integer(4) function cusparseCgtsvInterleavedBatch_bufferSizeExt(handle, &
          algo, m, dl, d, du, x, batchCount, pBufferSizeInBytes) &
          bind(C, name='cusparseCgtsvInterleavedBatch_bufferSizeExt')
       import cusparseHandle
       type(cusparseHandle), value :: handle       
       integer(4), value :: algo, m, batchCount
       complex(4), device :: dl(*), d(*), du(*), x(*)
       integer(8) :: pBufferSizeInBytes
     end function cusparseCgtsvInterleavedBatch_bufferSizeExt
  end interface cusparseCgtsvInterleavedBatch_bufferSize

! cusparseZgtsvInterleavedBatch_bufferSize
  interface cusparseZgtsvInterleavedBatch_bufferSize
     integer(4) function cusparseZgtsvInterleavedBatch_bufferSizeExt(handle, &
          algo, m, dl, d, du, x, batchCount, pBufferSizeInBytes) &
          bind(C, name='cusparseZgtsvInterleavedBatch_bufferSizeExt')
       import cusparseHandle
       type(cusparseHandle), value :: handle       
       integer(4), value :: algo, m, batchCount
       complex(8), device :: dl(*), d(*), du(*), x(*)
       integer(8) :: pBufferSizeInBytes
     end function cusparseZgtsvInterleavedBatch_bufferSizeExt
  end interface cusparseZgtsvInterleavedBatch_bufferSize

! cusparseSgtsvInterleavedBatch
  interface
     integer(4) function cusparseSgtsvInterleavedBatch(handle, algo, m, dl, d, du, x, batchCount, pBuffer) &
          bind(C,name='cusparseSgtsvInterleavedBatch')
       import cusparseHandle
       type(cusparseHandle), value :: handle       
       integer(4), value :: algo, m, batchCount
       real(4), device :: dl(*), d(*), du(*), x(*)
       !dir$ ignore_tkr pBuffer
       character*1, device :: pBuffer(*)
     end function cusparseSgtsvInterleavedBatch
  end interface

! cusparseDgtsvInterleavedBatch
  interface
     integer(4) function cusparseDgtsvInterleavedBatch(handle, algo, m, dl, d, du, x, batchCount, pBuffer) &
          bind(C,name='cusparseDgtsvInterleavedBatch')
       import cusparseHandle
       type(cusparseHandle), value :: handle       
       integer(4), value :: algo, m, batchCount
       real(8), device :: dl(*), d(*), du(*), x(*)
       !dir$ ignore_tkr pBuffer
       character*1, device :: pBuffer(*)
     end function cusparseDgtsvInterleavedBatch
  end interface

! cusparseCgtsvInterleavedBatch
  interface
     integer(4) function cusparseCgtsvInterleavedBatch(handle, algo, m, dl, d, du, x, batchCount, pBuffer) &
          bind(C,name='cusparseCgtsvInterleavedBatch')
       import cusparseHandle
       type(cusparseHandle), value :: handle       
       integer(4), value :: algo, m, batchCount
       complex(4), device :: dl(*), d(*), du(*), x(*)
       !dir$ ignore_tkr pBuffer
       character*1, device :: pBuffer(*)
     end function cusparseCgtsvInterleavedBatch
  end interface

! cusparseZgtsvInterleavedBatch
  interface
     integer(4) function cusparseZgtsvInterleavedBatch(handle, algo, m, dl, d, du, x, batchCount, pBuffer) &
          bind(C,name='cusparseZgtsvInterleavedBatch')
       import cusparseHandle
       type(cusparseHandle), value :: handle       
       integer(4), value :: algo, m, batchCount
       complex(8), device :: dl(*), d(*), du(*), x(*)
       !dir$ ignore_tkr pBuffer
       character*1, device :: pBuffer(*)
     end function cusparseZgtsvInterleavedBatch
  end interface
!
! Interleaved Batched Pentadiagonal
!
! cusparseSgpsvInterleavedBatch_bufferSize
  interface cusparseSgpsvInterleavedBatch_bufferSize
     integer(4) function cusparseSgpsvInterleavedBatch_bufferSizeExt(handle, &
          algo, m, ds, dl, d, du, dw, x, batchCount, pBufferSizeInBytes) &
          bind(C, name='cusparseSgpsvInterleavedBatch_bufferSizeExt')
       import cusparseHandle
       type(cusparseHandle), value :: handle       
       integer(4), value :: algo, m, batchCount
       real(4), device :: ds(*), dl(*), d(*), du(*), dw(*), x(*)
       integer(8) :: pBufferSizeInBytes
     end function cusparseSgpsvInterleavedBatch_bufferSizeExt
  end interface cusparseSgpsvInterleavedBatch_bufferSize

! cusparseDgpsvInterleavedBatch_bufferSize
  interface cusparseDgpsvInterleavedBatch_bufferSize
     integer(4) function cusparseDgpsvInterleavedBatch_bufferSizeExt(handle, &
          algo, m, ds, dl, d, du, dw, x, batchCount, pBufferSizeInBytes) &
          bind(C, name='cusparseDgpsvInterleavedBatch_bufferSizeExt')
       import cusparseHandle
       type(cusparseHandle), value :: handle       
       integer(4), value :: algo, m, batchCount
       real(8), device :: ds(*), dl(*), d(*), du(*), dw(*), x(*)
       integer(8) :: pBufferSizeInBytes
     end function cusparseDgpsvInterleavedBatch_bufferSizeExt
  end interface cusparseDgpsvInterleavedBatch_bufferSize

! cusparseCgpsvInterleavedBatch_bufferSize
  interface cusparseCgpsvInterleavedBatch_bufferSize
     integer(4) function cusparseCgpsvInterleavedBatch_bufferSizeExt(handle, &
          algo, m, ds, dl, d, du, dw, x, batchCount, pBufferSizeInBytes) &
          bind(C, name='cusparseCgpsvInterleavedBatch_bufferSizeExt')
       import cusparseHandle
       type(cusparseHandle), value :: handle       
       integer(4), value :: algo, m, batchCount
       complex(4), device :: ds(*), dl(*), d(*), du(*), dw(*), x(*)
       integer(8) :: pBufferSizeInBytes
     end function cusparseCgpsvInterleavedBatch_bufferSizeExt
  end interface cusparseCgpsvInterleavedBatch_bufferSize

! cusparseZgpsvInterleavedBatch_bufferSize
  interface cusparseZgpsvInterleavedBatch_bufferSize
     integer(4) function cusparseZgpsvInterleavedBatch_bufferSizeExt(handle, &
          algo, m, ds, dl, d, du, dw, x, batchCount, pBufferSizeInBytes) &
          bind(C, name='cusparseZgpsvInterleavedBatch_bufferSizeExt')
       import cusparseHandle
       type(cusparseHandle), value :: handle       
       integer(4), value :: algo, m, batchCount
       complex(8), device :: ds(*), dl(*), d(*), du(*), dw(*), x(*)
       integer(8) :: pBufferSizeInBytes
     end function cusparseZgpsvInterleavedBatch_bufferSizeExt
  end interface cusparseZgpsvInterleavedBatch_bufferSize

! cusparseSgpsvInterleavedBatch
  interface
     integer(4) function cusparseSgpsvInterleavedBatch(handle, algo, m, ds, dl, d, du, dw, x, batchCount, pBuffer) &
          bind(C,name='cusparseSgpsvInterleavedBatch')
       import cusparseHandle
       type(cusparseHandle), value :: handle       
       integer(4), value :: algo, m, batchCount
       real(4), device :: ds(*), dl(*), d(*), du(*), dw(*), x(*)
       !dir$ ignore_tkr pBuffer
       character*1, device :: pBuffer(*)
     end function cusparseSgpsvInterleavedBatch
  end interface

! cusparseDgpsvInterleavedBatch
  interface
     integer(4) function cusparseDgpsvInterleavedBatch(handle, algo, m, ds, dl, d, du, dw, x, batchCount, pBuffer) &
          bind(C,name='cusparseDgpsvInterleavedBatch')
       import cusparseHandle
       type(cusparseHandle), value :: handle       
       integer(4), value :: algo, m, batchCount
       real(8), device :: ds(*), dl(*), d(*), du(*), dw(*), x(*)
       !dir$ ignore_tkr pBuffer
       character*1, device :: pBuffer(*)
     end function cusparseDgpsvInterleavedBatch
  end interface

! cusparseCgpsvInterleavedBatch
  interface
     integer(4) function cusparseCgpsvInterleavedBatch(handle, algo, m, ds, dl, d, du, dw, x, batchCount, pBuffer) &
          bind(C,name='cusparseCgpsvInterleavedBatch')
       import cusparseHandle
       type(cusparseHandle), value :: handle       
       integer(4), value :: algo, m, batchCount
       complex(4), device :: ds(*), dl(*), d(*), du(*), dw(*), x(*)
       !dir$ ignore_tkr pBuffer
       character*1, device :: pBuffer(*)
     end function cusparseCgpsvInterleavedBatch
  end interface

! cusparseZgpsvInterleavedBatch
  interface
     integer(4) function cusparseZgpsvInterleavedBatch(handle, algo, m, ds, dl, d, du, dw, x, batchCount, pBuffer) &
          bind(C,name='cusparseZgpsvInterleavedBatch')
       import cusparseHandle
       type(cusparseHandle), value :: handle       
       integer(4), value :: algo, m, batchCount
       complex(8), device :: ds(*), dl(*), d(*), du(*), dw(*), x(*)
       !dir$ ignore_tkr pBuffer
       character*1, device :: pBuffer(*)
     end function cusparseZgpsvInterleavedBatch
  end interface
!

  ! -------------------
  ! 6.0 Preconditioners
  ! -------------------

! cusparseScsric02_bufferSize
  interface cusparseScsric02_bufferSize
     integer(c_int) function cusparseScsric02_bufferSize(handle, m, nnz, descrA, &
          csrValA, csrRowPtrA, csrColIndA, info, pBufferSize) bind(C,name='cusparseScsric02_bufferSize')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr, cusparseCsric02Info
       type(cusparseHandle), value :: handle
       integer(c_int), value :: m, nnz
       type(cusparseMatDescr), value :: descrA
       real(c_float), device :: csrValA(*)
       integer(c_int), device :: csrRowPtrA(*), csrColIndA(*)
       type(cusparseCsric02Info), value :: info
       integer(4) :: pBufferSize
     end function cusparseScsric02_bufferSize
     integer(c_int) function cusparseScsric02_bufferSizeExt(handle, m, nnz, descrA, &
          csrValA, csrRowPtrA, csrColIndA, info, pBufferSize) bind(C,name='cusparseScsric02_bufferSizeExt')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr, cusparseCsric02Info
       type(cusparseHandle), value :: handle
       integer(c_int), value :: m, nnz
       type(cusparseMatDescr), value :: descrA
       real(c_float), device :: csrValA(*)
       integer(c_int), device :: csrRowPtrA(*), csrColIndA(*)
       type(cusparseCsric02Info), value :: info
       integer(8) :: pBufferSize
     end function cusparseScsric02_bufferSizeExt
  end interface

! cusparseDcsric02_bufferSize
  interface cusparseDcsric02_bufferSize
     integer(c_int) function cusparseDcsric02_bufferSize(handle, m, nnz, descrA, &
          csrValA, csrRowPtrA, csrColIndA, info, pBufferSize) bind(C,name='cusparseDcsric02_bufferSize')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr, cusparseCsric02Info
       type(cusparseHandle), value :: handle
       integer(c_int), value :: m, nnz
       type(cusparseMatDescr), value :: descrA
       real(c_double), device :: csrValA(*)
       integer(c_int), device :: csrRowPtrA(*), csrColIndA(*)
       type(cusparseCsric02Info), value :: info
       integer(4) :: pBufferSize
     end function cusparseDcsric02_bufferSize
     integer(c_int) function cusparseDcsric02_bufferSizeExt(handle, m, nnz, descrA, &
          csrValA, csrRowPtrA, csrColIndA, info, pBufferSize) bind(C,name='cusparseDcsric02_bufferSizeExt')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr, cusparseCsric02Info
       type(cusparseHandle), value :: handle
       integer(c_int), value :: m, nnz
       type(cusparseMatDescr), value :: descrA
       real(c_double), device :: csrValA(*)
       integer(c_int), device :: csrRowPtrA(*), csrColIndA(*)
       type(cusparseCsric02Info), value :: info
       integer(8) :: pBufferSize
     end function cusparseDcsric02_bufferSizeExt
  end interface

! cusparseCcsric02_bufferSize
  interface cusparseCcsric02_bufferSize
     integer(c_int) function cusparseCcsric02_bufferSize(handle, m, nnz, descrA, &
          csrValA, csrRowPtrA, csrColIndA, info, pBufferSize) bind(C,name='cusparseCcsric02_bufferSize')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr, cusparseCsric02Info
       type(cusparseHandle), value :: handle
       integer(c_int), value :: m, nnz
       type(cusparseMatDescr), value :: descrA
       complex(4), device :: csrValA(*)
       integer(c_int), device :: csrRowPtrA(*), csrColIndA(*)
       type(cusparseCsric02Info), value :: info
       integer(4) :: pBufferSize
     end function cusparseCcsric02_bufferSize
     integer(c_int) function cusparseCcsric02_bufferSizeExt(handle, m, nnz, descrA, &
          csrValA, csrRowPtrA, csrColIndA, info, pBufferSize) bind(C,name='cusparseCcsric02_bufferSizeExt')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr, cusparseCsric02Info
       type(cusparseHandle), value :: handle
       integer(c_int), value :: m, nnz
       type(cusparseMatDescr), value :: descrA
       complex(4), device :: csrValA(*)
       integer(c_int), device :: csrRowPtrA(*), csrColIndA(*)
       type(cusparseCsric02Info), value :: info
       integer(8) :: pBufferSize
     end function cusparseCcsric02_bufferSizeExt
  end interface

! cusparseZcsric02_bufferSize
  interface cusparseZcsric02_bufferSize
     integer(c_int) function cusparseZcsric02_bufferSize(handle, m, nnz, descrA, &
          csrValA, csrRowPtrA, csrColIndA, info, pBufferSize) bind(C,name='cusparseZcsric02_bufferSize')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr, cusparseCsric02Info
       type(cusparseHandle), value :: handle
       integer(c_int), value :: m, nnz
       type(cusparseMatDescr), value :: descrA
       complex(8), device :: csrValA(*)
       integer(c_int), device :: csrRowPtrA(*), csrColIndA(*)
       type(cusparseCsric02Info), value :: info
       integer(4) :: pBufferSize
     end function cusparseZcsric02_bufferSize
     integer(c_int) function cusparseZcsric02_bufferSizeExt(handle, m, nnz, descrA, &
          csrValA, csrRowPtrA, csrColIndA, info, pBufferSize) bind(C,name='cusparseZcsric02_bufferSizeExt')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr, cusparseCsric02Info
       type(cusparseHandle), value :: handle
       integer(c_int), value :: m, nnz
       type(cusparseMatDescr), value :: descrA
       complex(8), device :: csrValA(*)
       integer(c_int), device :: csrRowPtrA(*), csrColIndA(*)
       type(cusparseCsric02Info), value :: info
       integer(8) :: pBufferSize
     end function cusparseZcsric02_bufferSizeExt
  end interface

! cusparseScsric02_analysis
  interface
     integer(c_int) function cusparseScsric02_analysis(handle, m, nnz, descrA, &
          csrValA, csrRowPtrA, csrColIndA, info, policy, pBuffer) bind(C,name='cusparseScsric02_analysis')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr, cusparseCsric02Info
       type(cusparseHandle), value :: handle
       integer(c_int), value :: m, nnz
       type(cusparseMatDescr), value :: descrA
       real(c_float), device :: csrValA(*)
       integer(c_int), device :: csrRowPtrA(*), csrColIndA(*)
       type(cusparseCsric02Info), value :: info
       integer(c_int), value :: policy
       !dir$ ignore_tkr pBuffer
       character(c_char), device :: pBuffer(*)
     end function cusparseScsric02_analysis
  end interface
  
! cusparseDcsric02_analysis
  interface
     integer(c_int) function cusparseDcsric02_analysis(handle, m, nnz, descrA, &
          csrValA, csrRowPtrA, csrColIndA, info, policy, pBuffer) bind(C,name='cusparseDcsric02_analysis')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr, cusparseCsric02Info
       type(cusparseHandle), value :: handle
       integer(c_int), value :: m, nnz
       type(cusparseMatDescr), value :: descrA
       real(c_double), device :: csrValA(*)
       integer(c_int), device :: csrRowPtrA(*), csrColIndA(*)
       type(cusparseCsric02Info), value :: info
       integer(c_int), value :: policy
       !dir$ ignore_tkr pBuffer
       character(c_char), device :: pBuffer(*)
     end function cusparseDcsric02_analysis
  end interface
  
! cusparseCcsric02_analysis
  interface
     integer(c_int) function cusparseCcsric02_analysis(handle, m, nnz, descrA, &
          csrValA, csrRowPtrA, csrColIndA, info, policy, pBuffer) bind(C,name='cusparseCcsric02_analysis')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr, cusparseCsric02Info
       type(cusparseHandle), value :: handle
       integer(c_int), value :: m, nnz
       type(cusparseMatDescr), value :: descrA
       complex(4), device :: csrValA(*)
       integer(c_int), device :: csrRowPtrA(*), csrColIndA(*)
       type(cusparseCsric02Info), value :: info
       integer(c_int), value :: policy
       !dir$ ignore_tkr pBuffer
       character(c_char), device :: pBuffer(*)
     end function cusparseCcsric02_analysis
  end interface
  
! cusparseZcsric02_analysis
  interface
     integer(c_int) function cusparseZcsric02_analysis(handle, m, nnz, descrA, &
          csrValA, csrRowPtrA, csrColIndA, info, policy, pBuffer) bind(C,name='cusparseZcsric02_analysis')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr, cusparseCsric02Info
       type(cusparseHandle), value :: handle
       integer(c_int), value :: m, nnz
       type(cusparseMatDescr), value :: descrA
       complex(8), device :: csrValA(*)
       integer(c_int), device :: csrRowPtrA(*), csrColIndA(*)
       type(cusparseCsric02Info), value :: info
       integer(c_int), value :: policy
       !dir$ ignore_tkr pBuffer
       character(c_char), device :: pBuffer(*)
     end function cusparseZcsric02_analysis
  end interface
  
! cusparseScsric02
  interface
     integer(c_int) function cusparseScsric02(handle, m, nnz, descrA, &
          csrValA, csrRowPtrA, csrColIndA, info, policy, pBuffer) bind(C,name='cusparseScsric02')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr, cusparseCsric02Info
       type(cusparseHandle), value :: handle
       integer(c_int), value :: m, nnz
       type(cusparseMatDescr), value :: descrA
       real(c_float), device :: csrValA(*)
       integer(c_int), device :: csrRowPtrA(*), csrColIndA(*)
       type(cusparseCsric02Info), value :: info
       integer(c_int), value :: policy
       !dir$ ignore_tkr pBuffer
       character(c_char), device :: pBuffer(*)
     end function cusparseScsric02
  end interface

! cusparseDcsric02
  interface
     integer(c_int) function cusparseDcsric02(handle, m, nnz, descrA, &
          csrValA, csrRowPtrA, csrColIndA, info, policy, pBuffer) bind(C,name='cusparseDcsric02')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr, cusparseCsric02Info
       type(cusparseHandle), value :: handle
       integer(c_int), value :: m, nnz
       type(cusparseMatDescr), value :: descrA
       real(c_double), device :: csrValA(*)
       integer(c_int), device :: csrRowPtrA(*), csrColIndA(*)
       type(cusparseCsric02Info), value :: info
       integer(c_int), value :: policy
       !dir$ ignore_tkr pBuffer
       character(c_char), device :: pBuffer(*)
     end function cusparseDcsric02
  end interface
  
! cusparseCcsric02
  interface
     integer(c_int) function cusparseCcsric02(handle, m, nnz, descrA, &
          csrValA, csrRowPtrA, csrColIndA, info, policy, pBuffer) bind(C,name='cusparseCcsric02')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr, cusparseCsric02Info
       type(cusparseHandle), value :: handle
       integer(c_int), value :: m, nnz
       type(cusparseMatDescr), value :: descrA
       complex(4), device :: csrValA(*)
       integer(c_int), device :: csrRowPtrA(*), csrColIndA(*)
       type(cusparseCsric02Info), value :: info
       integer(c_int), value :: policy
       !dir$ ignore_tkr pBuffer
       character(c_char), device :: pBuffer(*)
     end function cusparseCcsric02
  end interface
  
! cusparseZcsric02
  interface
     integer(c_int) function cusparseZcsric02(handle, m, nnz, descrA, &
          csrValA, csrRowPtrA, csrColIndA, info, policy, pBuffer) bind(C,name='cusparseZcsric02')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr, cusparseCsric02Info
       type(cusparseHandle), value :: handle
       integer(c_int), value :: m, nnz
       type(cusparseMatDescr), value :: descrA
       complex(8), device :: csrValA(*)
       integer(c_int), device :: csrRowPtrA(*), csrColIndA(*)
       type(cusparseCsric02Info), value :: info
       integer(c_int), value :: policy
       !dir$ ignore_tkr pBuffer
       character(c_char), device :: pBuffer(*)
     end function cusparseZcsric02
  end interface
  
! cusparseXcsric02_zeroPivot
  interface cusparseXcsric02_zeroPivot
     integer function cusparseXcsric02_zeroPivot_setHPM(handle, info, position)
       import cusparseHandle, cusparseCsric02Info
       type(cusparseHandle) :: handle
       type(cusparseCsric02Info) :: info
       integer :: position
     end function cusparseXcsric02_zeroPivot_setHPM

     integer function cusparseXcsric02_zeroPivot_setDPM(handle, info, position)
       import cusparseHandle, cusparseCsric02Info
       type(cusparseHandle) :: handle
       type(cusparseCsric02Info) :: info
       integer, device :: position
     end function cusparseXcsric02_zeroPivot_setDPM
  end interface cusparseXcsric02_zeroPivot

#ifdef KEEP_V2_INTERFACES
  interface 
     integer(c_int) function cusparseXcsric02_zeroPivot_v2(handle, info, position) bind(C,name='cusparseXcsric02_zeroPivot')
       use iso_c_binding
       import cusparseHandle, cusparseCsric02Info
       type(cusparseHandle), value :: handle
       type(cusparseCsric02Info), value :: info
       !dir$ ignore_tkr (d) position
       integer(c_int) :: position
     end function cusparseXcsric02_zeroPivot_v2
  end interface
#endif

! cusparseScsrilu02_numericBoost
  interface cusparseScsrilu02_numericBoost
     integer function cusparseScsrilu02_numericBoost_setHPM(handle, info, enable_boost, tol, boost_val)
       import cusparseHandle, cusparseCsrilu02Info
       type(cusparseHandle) :: handle
       type(cusparseCsrilu02Info) :: info
       integer :: enable_boost
       real(8) :: tol
       real(4) :: boost_val
     end function cusparseScsrilu02_numericBoost_setHPM

     integer function cusparseScsrilu02_numericBoost_setDPM(handle, info, enable_boost, tol, boost_val)
       import cusparseHandle, cusparseCsrilu02Info
       type(cusparseHandle) :: handle
       type(cusparseCsrilu02Info) :: info
       integer :: enable_boost
       real(8), device :: tol
       real(4), device :: boost_val
     end function cusparseScsrilu02_numericBoost_setDPM
  end interface cusparseScsrilu02_numericBoost

#ifdef KEEP_V2_INTERFACES
  interface
     integer(c_int) function cusparseScsrilu02_numericBoost_v2(handle, info, enable_boost, tol, boost_val) &
          bind(C,name='cusparseScsrilu02_numericBoost')
       use iso_c_binding
       import cusparseHandle, cusparseCsrilu02Info
       type(cusparseHandle), value :: handle
       type(cusparseCsrilu02Info), value :: info
       integer(c_int), value :: enable_boost
       !dir$ ignore_tkr (d) tol, boost_val
       real(c_double) :: tol
       real(c_float) :: boost_val
     end function cusparseScsrilu02_numericBoost_v2
  end interface
#endif

! cusparseDcsrilu02_numericBoost
  interface cusparseDcsrilu02_numericBoost
     integer function cusparseDcsrilu02_numericBoost_setHPM(handle, info, enable_boost, tol, boost_val)
       import cusparseHandle, cusparseCsrilu02Info
       type(cusparseHandle) :: handle
       type(cusparseCsrilu02Info) :: info
       integer :: enable_boost
       real(8) :: tol
       real(8) :: boost_val
     end function cusparseDcsrilu02_numericBoost_setHPM

     integer function cusparseDcsrilu02_numericBoost_setDPM(handle, info, enable_boost, tol, boost_val)
       import cusparseHandle, cusparseCsrilu02Info
       type(cusparseHandle) :: handle
       type(cusparseCsrilu02Info) :: info
       integer :: enable_boost
       real(8), device :: tol
       real(8), device :: boost_val
     end function cusparseDcsrilu02_numericBoost_setDPM
  end interface cusparseDcsrilu02_numericBoost

#ifdef KEEP_V2_INTERFACES
  interface
     integer(c_int) function cusparseDcsrilu02_numericBoost_v2(handle, info, enable_boost, tol, boost_val) &
          bind(C,name='cusparseDcsrilu02_numericBoost')
       use iso_c_binding
       import cusparseHandle, cusparseCsrilu02Info
       type(cusparseHandle), value :: handle
       type(cusparseCsrilu02Info), value :: info
       integer(c_int), value :: enable_boost
       !dir$ ignore_tkr (d) tol, boost_val
       real(c_double) :: tol
       real(c_double) :: boost_val
     end function cusparseDcsrilu02_numericBoost_v2
  end interface
#endif

! cusparseCcsrilu02_numericBoost
  interface cusparseCcsrilu02_numericBoost
     integer function cusparseCcsrilu02_numericBoost_setHPM(handle, info, enable_boost, tol, boost_val)
       import cusparseHandle, cusparseCsrilu02Info
       type(cusparseHandle) :: handle
       type(cusparseCsrilu02Info) :: info
       integer :: enable_boost
       real(8) :: tol
       complex(4) :: boost_val
     end function cusparseCcsrilu02_numericBoost_setHPM

     integer function cusparseCcsrilu02_numericBoost_setDPM(handle, info, enable_boost, tol, boost_val)
       import cusparseHandle, cusparseCsrilu02Info
       type(cusparseHandle) :: handle
       type(cusparseCsrilu02Info) :: info
       integer :: enable_boost
       real(8), device :: tol
       complex(4), device :: boost_val
     end function cusparseCcsrilu02_numericBoost_setDPM
  end interface cusparseCcsrilu02_numericBoost

#ifdef KEEP_V2_INTERFACES
  interface
     integer(c_int) function cusparseCcsrilu02_numericBoost_v2(handle, info, enable_boost, tol, boost_val) &
          bind(C,name='cusparseCcsrilu02_numericBoost')
       use iso_c_binding
       import cusparseHandle, cusparseCsrilu02Info
       type(cusparseHandle), value :: handle
       type(cusparseCsrilu02Info), value :: info
       integer(c_int), value :: enable_boost
       !dir$ ignore_tkr (d) tol, boost_val
       real(c_double) :: tol
       complex(4) :: boost_val
     end function cusparseCcsrilu02_numericBoost_v2
  end interface
#endif

! cusparseZcsrilu02_numericBoost
  interface cusparseZcsrilu02_numericBoost
     integer function cusparseZcsrilu02_numericBoost_setHPM(handle, info, enable_boost, tol, boost_val)
       import cusparseHandle, cusparseCsrilu02Info
       type(cusparseHandle) :: handle
       type(cusparseCsrilu02Info) :: info
       integer :: enable_boost
       real(8) :: tol
       complex(8) :: boost_val
     end function cusparseZcsrilu02_numericBoost_setHPM

     integer function cusparseZcsrilu02_numericBoost_setDPM(handle, info, enable_boost, tol, boost_val)
       import cusparseHandle, cusparseCsrilu02Info
       type(cusparseHandle) :: handle
       type(cusparseCsrilu02Info) :: info
       integer :: enable_boost
       real(8), device :: tol
       complex(8), device :: boost_val
     end function cusparseZcsrilu02_numericBoost_setDPM
  end interface cusparseZcsrilu02_numericBoost

#ifdef KEEP_V2_INTERFACES
  interface
     integer(c_int) function cusparseZcsrilu02_numericBoost_v2(handle, info, enable_boost, tol, boost_val) &
          bind(C,name='cusparseZcsrilu02_numericBoost')
       use iso_c_binding
       import cusparseHandle, cusparseCsrilu02Info
       type(cusparseHandle), value :: handle
       type(cusparseCsrilu02Info), value :: info
       integer(c_int), value :: enable_boost
       !dir$ ignore_tkr (d) tol, boost_val
       real(c_double) :: tol
       complex(8) :: boost_val
     end function cusparseZcsrilu02_numericBoost_v2
  end interface
#endif

! cusparseScsrilu02_bufferSize
  interface cusparseScsrilu02_bufferSize
     integer(c_int) function cusparseScsrilu02_bufferSize(handle, m, nnz, descrA, &
          csrValA, csrRowPtrA, csrColIndA, info, pBufferSize) bind(C,name='cusparseScsrilu02_bufferSize')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr, cusparseCsrilu02Info
       type(cusparseHandle), value :: handle
       integer(c_int), value :: m, nnz
       type(cusparseMatDescr), value :: descrA
       real(c_float), device :: csrValA(*)
       integer(c_int), device :: csrRowPtrA(*), csrColIndA(*)
       type(cusparseCsrilu02Info), value :: info
       integer(4) :: pBufferSize
     end function cusparseScsrilu02_bufferSize
     integer(c_int) function cusparseScsrilu02_bufferSizeExt(handle, m, nnz, descrA, &
          csrValA, csrRowPtrA, csrColIndA, info, pBufferSize) bind(C,name='cusparseScsrilu02_bufferSizeExt')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr, cusparseCsrilu02Info
       type(cusparseHandle), value :: handle
       integer(c_int), value :: m, nnz
       type(cusparseMatDescr), value :: descrA
       real(c_float), device :: csrValA(*)
       integer(c_int), device :: csrRowPtrA(*), csrColIndA(*)
       type(cusparseCsrilu02Info), value :: info
       integer(8) :: pBufferSize
     end function cusparseScsrilu02_bufferSizeExt
  end interface

! cusparseDcsrilu02_bufferSize
  interface cusparseDcsrilu02_bufferSize
     integer(c_int) function cusparseDcsrilu02_bufferSize(handle, m, nnz, descrA, &
          csrValA, csrRowPtrA, csrColIndA, info, pBufferSize) bind(C,name='cusparseDcsrilu02_bufferSize')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr, cusparseCsrilu02Info
       type(cusparseHandle), value :: handle
       integer(c_int), value :: m, nnz
       type(cusparseMatDescr), value :: descrA
       real(c_double), device :: csrValA(*)
       integer(c_int), device :: csrRowPtrA(*), csrColIndA(*)
       type(cusparseCsrilu02Info), value :: info
       integer(4) :: pBufferSize
     end function cusparseDcsrilu02_bufferSize
     integer(c_int) function cusparseDcsrilu02_bufferSizeExt(handle, m, nnz, descrA, &
          csrValA, csrRowPtrA, csrColIndA, info, pBufferSize) bind(C,name='cusparseDcsrilu02_bufferSizeExt')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr, cusparseCsrilu02Info
       type(cusparseHandle), value :: handle
       integer(c_int), value :: m, nnz
       type(cusparseMatDescr), value :: descrA
       real(c_double), device :: csrValA(*)
       integer(c_int), device :: csrRowPtrA(*), csrColIndA(*)
       type(cusparseCsrilu02Info), value :: info
       integer(8) :: pBufferSize
     end function cusparseDcsrilu02_bufferSizeExt
  end interface

! cusparseCcsrilu02_bufferSize
  interface cusparseCcsrilu02_bufferSize
     integer(c_int) function cusparseCcsrilu02_bufferSize(handle, m, nnz, descrA, &
          csrValA, csrRowPtrA, csrColIndA, info, pBufferSize) bind(C,name='cusparseCcsrilu02_bufferSize')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr, cusparseCsrilu02Info
       type(cusparseHandle), value :: handle
       integer(c_int), value :: m, nnz
       type(cusparseMatDescr), value :: descrA
       complex(4), device :: csrValA(*)
       integer(c_int), device :: csrRowPtrA(*), csrColIndA(*)
       type(cusparseCsrilu02Info), value :: info
       integer(4) :: pBufferSize
     end function cusparseCcsrilu02_bufferSize
     integer(c_int) function cusparseCcsrilu02_bufferSizeExt(handle, m, nnz, descrA, &
          csrValA, csrRowPtrA, csrColIndA, info, pBufferSize) bind(C,name='cusparseCcsrilu02_bufferSizeExt')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr, cusparseCsrilu02Info
       type(cusparseHandle), value :: handle
       integer(c_int), value :: m, nnz
       type(cusparseMatDescr), value :: descrA
       complex(4), device :: csrValA(*)
       integer(c_int), device :: csrRowPtrA(*), csrColIndA(*)
       type(cusparseCsrilu02Info), value :: info
       integer(8) :: pBufferSize
     end function cusparseCcsrilu02_bufferSizeExt
  end interface

! cusparseZcsrilu02_bufferSize
  interface cusparseZcsrilu02_bufferSize
     integer(c_int) function cusparseZcsrilu02_bufferSize(handle, m, nnz, descrA, &
          csrValA, csrRowPtrA, csrColIndA, info, pBufferSize) bind(C,name='cusparseZcsrilu02_bufferSize')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr, cusparseCsrilu02Info
       type(cusparseHandle), value :: handle
       integer(c_int), value :: m, nnz
       type(cusparseMatDescr), value :: descrA
       complex(8), device :: csrValA(*)
       integer(c_int), device :: csrRowPtrA(*), csrColIndA(*)
       type(cusparseCsrilu02Info), value :: info
       integer(4) :: pBufferSize
     end function cusparseZcsrilu02_bufferSize
     integer(c_int) function cusparseZcsrilu02_bufferSizeExt(handle, m, nnz, descrA, &
          csrValA, csrRowPtrA, csrColIndA, info, pBufferSize) bind(C,name='cusparseZcsrilu02_bufferSizeExt')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr, cusparseCsrilu02Info
       type(cusparseHandle), value :: handle
       integer(c_int), value :: m, nnz
       type(cusparseMatDescr), value :: descrA
       complex(8), device :: csrValA(*)
       integer(c_int), device :: csrRowPtrA(*), csrColIndA(*)
       type(cusparseCsrilu02Info), value :: info
       integer(8) :: pBufferSize
     end function cusparseZcsrilu02_bufferSizeExt
  end interface

! cusparseScsrilu02_analysis
  interface
     integer(c_int) function cusparseScsrilu02_analysis(handle, m, nnz, descrA, &
          csrValA, csrRowPtrA, csrColIndA, info, policy, pBuffer) bind(C,name='cusparseScsrilu02_analysis')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr, cusparseCsrilu02Info
       type(cusparseHandle), value :: handle
       integer(c_int), value :: m, nnz
       type(cusparseMatDescr), value :: descrA
       real(c_float), device :: csrValA(*)
       integer(c_int), device :: csrRowPtrA(*), csrColIndA(*)
       type(cusparseCsrilu02Info), value :: info
       integer(c_int), value :: policy
       !dir$ ignore_tkr pBuffer
       character(c_char), device :: pBuffer(*)
     end function cusparseScsrilu02_analysis
  end interface
  
! cusparseDcsrilu02_analysis
  interface
     integer(c_int) function cusparseDcsrilu02_analysis(handle, m, nnz, descrA, &
          csrValA, csrRowPtrA, csrColIndA, info, policy, pBuffer) bind(C,name='cusparseDcsrilu02_analysis')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr, cusparseCsrilu02Info
       type(cusparseHandle), value :: handle
       integer(c_int), value :: m, nnz
       type(cusparseMatDescr), value :: descrA
       real(c_double), device :: csrValA(*)
       integer(c_int), device :: csrRowPtrA(*), csrColIndA(*)
       type(cusparseCsrilu02Info), value :: info
       integer(c_int), value :: policy
       !dir$ ignore_tkr pBuffer
       character(c_char), device :: pBuffer(*)
     end function cusparseDcsrilu02_analysis
  end interface
  
! cusparseCcsrilu02_analysis
  interface
     integer(c_int) function cusparseCcsrilu02_analysis(handle, m, nnz, descrA, &
          csrValA, csrRowPtrA, csrColIndA, info, policy, pBuffer) bind(C,name='cusparseCcsrilu02_analysis')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr, cusparseCsrilu02Info
       type(cusparseHandle), value :: handle
       integer(c_int), value :: m, nnz
       type(cusparseMatDescr), value :: descrA
       complex(4), device :: csrValA(*)
       integer(c_int), device :: csrRowPtrA(*), csrColIndA(*)
       type(cusparseCsrilu02Info), value :: info
       integer(c_int), value :: policy
       !dir$ ignore_tkr pBuffer
       character(c_char), device :: pBuffer(*)
     end function cusparseCcsrilu02_analysis
  end interface
  
! cusparseZcsrilu02_analysis
  interface
     integer(c_int) function cusparseZcsrilu02_analysis(handle, m, nnz, descrA, &
          csrValA, csrRowPtrA, csrColIndA, info, policy, pBuffer) bind(C,name='cusparseZcsrilu02_analysis')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr, cusparseCsrilu02Info
       type(cusparseHandle), value :: handle
       integer(c_int), value :: m, nnz
       type(cusparseMatDescr), value :: descrA
       complex(8), device :: csrValA(*)
       integer(c_int), device :: csrRowPtrA(*), csrColIndA(*)
       type(cusparseCsrilu02Info), value :: info
       integer(c_int), value :: policy
       !dir$ ignore_tkr pBuffer
       character(c_char), device :: pBuffer(*)
     end function cusparseZcsrilu02_analysis
  end interface
  
! cusparseScsrilu02
  interface
     integer(c_int) function cusparseScsrilu02(handle, m, nnz, descrA, &

          csrValA, csrRowPtrA, csrColIndA, info, policy, pBuffer) bind(C,name='cusparseScsrilu02')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr, cusparseCsrilu02Info
       type(cusparseHandle), value :: handle
       integer(c_int), value :: m, nnz
       type(cusparseMatDescr), value :: descrA
       real(c_float), device :: csrValA(*)
       integer(c_int), device :: csrRowPtrA(*), csrColIndA(*)
       type(cusparseCsrilu02Info), value :: info
       integer(c_int), value :: policy
       !dir$ ignore_tkr pBuffer
       character(c_char), device :: pBuffer(*)
     end function cusparseScsrilu02
  end interface

! cusparseDcsrilu02
  interface
     integer(c_int) function cusparseDcsrilu02(handle, m, nnz, descrA, &
          csrValA, csrRowPtrA, csrColIndA, info, policy, pBuffer) bind(C,name='cusparseDcsrilu02')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr, cusparseCsrilu02Info
       type(cusparseHandle), value :: handle
       integer(c_int), value :: m, nnz
       type(cusparseMatDescr), value :: descrA
       real(c_double), device :: csrValA(*)
       integer(c_int), device :: csrRowPtrA(*), csrColIndA(*)
       type(cusparseCsrilu02Info), value :: info
       integer(c_int), value :: policy
       !dir$ ignore_tkr pBuffer
       character(c_char), device :: pBuffer(*)
     end function cusparseDcsrilu02
  end interface

! cusparseCcsrilu02
  interface
     integer(c_int) function cusparseCcsrilu02(handle, m, nnz, descrA, &
          csrValA, csrRowPtrA, csrColIndA, info, policy, pBuffer) bind(C,name='cusparseCcsrilu02')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr, cusparseCsrilu02Info
       type(cusparseHandle), value :: handle
       integer(c_int), value :: m, nnz
       type(cusparseMatDescr), value :: descrA
       complex(4), device :: csrValA(*)
       integer(c_int), device :: csrRowPtrA(*), csrColIndA(*)
       type(cusparseCsrilu02Info), value :: info
       integer(c_int), value :: policy
       !dir$ ignore_tkr pBuffer
       character(c_char), device :: pBuffer(*)
     end function cusparseCcsrilu02
  end interface

! cusparseZcsrilu02
  interface
     integer(c_int) function cusparseZcsrilu02(handle, m, nnz, descrA, &
          csrValA, csrRowPtrA, csrColIndA, info, policy, pBuffer) bind(C,name='cusparseZcsrilu02')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr, cusparseCsrilu02Info
       type(cusparseHandle), value :: handle
       integer(c_int), value :: m, nnz
       type(cusparseMatDescr), value :: descrA
       complex(8), device :: csrValA(*)
       integer(c_int), device :: csrRowPtrA(*), csrColIndA(*)
       type(cusparseCsrilu02Info), value :: info
       integer(c_int), value :: policy
       !dir$ ignore_tkr pBuffer
       character(c_char), device :: pBuffer(*)
     end function cusparseZcsrilu02
  end interface

! cusparseXcsrilu02_zeroPivot
  interface cusparseXcsrilu02_zeroPivot
     integer function cusparseXcsrilu02_zeroPivot_setHPM(handle, info, position)
       import cusparseHandle, cusparseCsrilu02Info
       type(cusparseHandle) :: handle
       type(cusparseCsrilu02Info) :: info
       integer :: position
     end function cusparseXcsrilu02_zeroPivot_setHPM

     integer function cusparseXcsrilu02_zeroPivot_setDPM(handle, info, position)
       import cusparseHandle, cusparseCsrilu02Info
       type(cusparseHandle) :: handle
       type(cusparseCsrilu02Info) :: info
       integer, device :: position
     end function cusparseXcsrilu02_zeroPivot_setDPM
  end interface cusparseXcsrilu02_zeroPivot

#ifdef KEEP_V2_INTERFACES
  interface 
     integer(c_int) function cusparseXcsrilu02_zeroPivot_v2(handle, info, position) bind(C,name='cusparseXcsrilu02_zeroPivot')
       use iso_c_binding
       import cusparseHandle, cusparseCsrilu02Info
       type(cusparseHandle), value :: handle
       type(cusparseCsrilu02Info), value :: info
       !dir$ ignore_tkr (d) position
       integer(c_int) :: position
     end function cusparseXcsrilu02_zeroPivot_v2
  end interface
#endif

! cusparseSbsric02_bufferSize
  interface cusparseSbsric02_bufferSize
     integer(c_int) function cusparseSbsric02_bufferSize(handle, dirA, mb, nnzb, descrA, &
          bsrValA, bsrRowPtrA, bsrColIndA, blockDim, info, pBufferSize) bind(C,name='cusparseSbsric02_bufferSize')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr, cusparseBsric02Info
       type(cusparseHandle), value :: handle
       integer(c_int), value :: dirA
       integer(c_int), value :: mb, nnzb
       type(cusparseMatDescr), value :: descrA
       real(c_float), device :: bsrValA(*)
       integer(c_int), device :: bsrRowPtrA(*), bsrColIndA(*)
       integer(c_int), value :: blockDim
       type(cusparseBsric02Info), value :: info
       integer(4) :: pBufferSize
     end function cusparseSbsric02_bufferSize
     integer(c_int) function cusparseSbsric02_bufferSizeExt(handle, dirA, mb, nnzb, descrA, &
          bsrValA, bsrRowPtrA, bsrColIndA, blockDim, info, pBufferSize) bind(C,name='cusparseSbsric02_bufferSizeExt')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr, cusparseBsric02Info
       type(cusparseHandle), value :: handle
       integer(c_int), value :: dirA
       integer(c_int), value :: mb, nnzb
       type(cusparseMatDescr), value :: descrA
       real(c_float), device :: bsrValA(*)
       integer(c_int), device :: bsrRowPtrA(*), bsrColIndA(*)
       integer(c_int), value :: blockDim
       type(cusparseBsric02Info), value :: info
       integer(8) :: pBufferSize
     end function cusparseSbsric02_bufferSizeExt
  end interface

! cusparseDbsric02_bufferSize
  interface cusparseDbsric02_bufferSize
     integer(c_int) function cusparseDbsric02_bufferSize(handle, dirA, mb, nnzb, descrA, &
          bsrValA, bsrRowPtrA, bsrColIndA, blockDim, info, pBufferSize) bind(C,name='cusparseDbsric02_bufferSize')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr, cusparseBsric02Info
       type(cusparseHandle), value :: handle
       integer(c_int), value :: dirA
       integer(c_int), value :: mb, nnzb
       type(cusparseMatDescr), value :: descrA
       real(c_double), device :: bsrValA(*)
       integer(c_int), device :: bsrRowPtrA(*), bsrColIndA(*)
       integer(c_int), value :: blockDim
       type(cusparseBsric02Info), value :: info
       integer(4) :: pBufferSize
     end function cusparseDbsric02_bufferSize
     integer(c_int) function cusparseDbsric02_bufferSizeExt(handle, dirA, mb, nnzb, descrA, &
          bsrValA, bsrRowPtrA, bsrColIndA, blockDim, info, pBufferSize) bind(C,name='cusparseDbsric02_bufferSizeExt')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr, cusparseBsric02Info
       type(cusparseHandle), value :: handle
       integer(c_int), value :: dirA
       integer(c_int), value :: mb, nnzb
       type(cusparseMatDescr), value :: descrA
       real(c_double), device :: bsrValA(*)
       integer(c_int), device :: bsrRowPtrA(*), bsrColIndA(*)
       integer(c_int), value :: blockDim
       type(cusparseBsric02Info), value :: info
       integer(8) :: pBufferSize
     end function cusparseDbsric02_bufferSizeExt
  end interface

! cusparseCbsric02_bufferSize
  interface cusparseCbsric02_bufferSize
     integer(c_int) function cusparseCbsric02_bufferSize(handle, dirA, mb, nnzb, descrA, &
          bsrValA, bsrRowPtrA, bsrColIndA, blockDim, info, pBufferSize) bind(C,name='cusparseCbsric02_bufferSize')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr, cusparseBsric02Info
       type(cusparseHandle), value :: handle
       integer(c_int), value :: dirA
       integer(c_int), value :: mb, nnzb
       type(cusparseMatDescr), value :: descrA
       complex(4), device :: bsrValA(*)
       integer(c_int), device :: bsrRowPtrA(*), bsrColIndA(*)
       integer(c_int), value :: blockDim
       type(cusparseBsric02Info), value :: info
       integer(4) :: pBufferSize
     end function cusparseCbsric02_bufferSize
     integer(c_int) function cusparseCbsric02_bufferSizeExt(handle, dirA, mb, nnzb, descrA, &
          bsrValA, bsrRowPtrA, bsrColIndA, blockDim, info, pBufferSize) bind(C,name='cusparseCbsric02_bufferSizeExt')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr, cusparseBsric02Info
       type(cusparseHandle), value :: handle
       integer(c_int), value :: dirA
       integer(c_int), value :: mb, nnzb
       type(cusparseMatDescr), value :: descrA
       complex(4), device :: bsrValA(*)
       integer(c_int), device :: bsrRowPtrA(*), bsrColIndA(*)
       integer(c_int), value :: blockDim
       type(cusparseBsric02Info), value :: info
       integer(8) :: pBufferSize
     end function cusparseCbsric02_bufferSizeExt
  end interface

! cusparseZbsric02_bufferSize
  interface cusparseZbsric02_bufferSize
     integer(c_int) function cusparseZbsric02_bufferSize(handle, dirA, mb, nnzb, descrA, &
          bsrValA, bsrRowPtrA, bsrColIndA, blockDim, info, pBufferSize) bind(C,name='cusparseZbsric02_bufferSize')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr, cusparseBsric02Info
       type(cusparseHandle), value :: handle
       integer(c_int), value :: dirA
       integer(c_int), value :: mb, nnzb
       type(cusparseMatDescr), value :: descrA
       complex(8), device :: bsrValA(*)
       integer(c_int), device :: bsrRowPtrA(*), bsrColIndA(*)
       integer(c_int), value :: blockDim
       type(cusparseBsric02Info), value :: info
       integer(4) :: pBufferSize
     end function cusparseZbsric02_bufferSize
     integer(c_int) function cusparseZbsric02_bufferSizeExt(handle, dirA, mb, nnzb, descrA, &
          bsrValA, bsrRowPtrA, bsrColIndA, blockDim, info, pBufferSize) bind(C,name='cusparseZbsric02_bufferSizeExt')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr, cusparseBsric02Info
       type(cusparseHandle), value :: handle
       integer(c_int), value :: dirA
       integer(c_int), value :: mb, nnzb
       type(cusparseMatDescr), value :: descrA
       complex(8), device :: bsrValA(*)
       integer(c_int), device :: bsrRowPtrA(*), bsrColIndA(*)
       integer(c_int), value :: blockDim
       type(cusparseBsric02Info), value :: info
       integer(8) :: pBufferSize
     end function cusparseZbsric02_bufferSizeExt
  end interface

! cusparseSbsric02_analysis
  interface 
     integer(c_int) function cusparseSbsric02_analysis(handle, dirA, mb, nnzb, descrA, &
          bsrValA, bsrRowPtrA, bsrColIndA, blockDim, info, policy, pBuffer) bind(C,name='cusparseSbsric02_analysis')
       use iso_c_binding
       import cusparseHandle,  cusparseMatDescr, cusparseBsric02Info
       type(cusparseHandle), value :: handle
       integer(c_int), value :: dirA
       integer(c_int), value :: mb, nnzb
       type(cusparseMatDescr), value :: descrA
       real(c_float), device :: bsrValA(*)
       integer(c_int), device :: bsrRowPtrA(*), bsrColIndA(*)
       integer(c_int), value :: blockDim
       type(cusparseBsric02Info), value :: info
       integer(c_int), value :: policy
       !dir$ ignore_tkr pBuffer
       character(c_char), device :: pBuffer(*)
     end function cusparseSbsric02_analysis
  end interface
  
! cusparseDbsric02_analysis
  interface 
     integer(c_int) function cusparseDbsric02_analysis(handle, dirA, mb, nnzb, descrA, &
          bsrValA, bsrRowPtrA, bsrColIndA, blockDim, info, policy, pBuffer) bind(C,name='cusparseDbsric02_analysis')
       use iso_c_binding
       import cusparseHandle,  cusparseMatDescr, cusparseBsric02Info
       type(cusparseHandle), value :: handle
       integer(c_int), value :: dirA
       integer(c_int), value :: mb, nnzb
       type(cusparseMatDescr), value :: descrA
       real(c_double), device :: bsrValA(*)
       integer(c_int), device :: bsrRowPtrA(*), bsrColIndA(*)
       integer(c_int), value :: blockDim
       type(cusparseBsric02Info), value :: info
       integer(c_int), value :: policy
       !dir$ ignore_tkr pBuffer
       character(c_char), device :: pBuffer(*)
     end function cusparseDbsric02_analysis
  end interface

! cusparseCbsric02_analysis
  interface 
     integer(c_int) function cusparseCbsric02_analysis(handle, dirA, mb, nnzb, descrA, &
          bsrValA, bsrRowPtrA, bsrColIndA, blockDim, info, policy, pBuffer) bind(C,name='cusparseCbsric02_analysis')
       use iso_c_binding
       import cusparseHandle,  cusparseMatDescr, cusparseBsric02Info
       type(cusparseHandle), value :: handle
       integer(c_int), value :: dirA
       integer(c_int), value :: mb, nnzb
       type(cusparseMatDescr), value :: descrA
       complex(4), device :: bsrValA(*)
       integer(c_int), device :: bsrRowPtrA(*), bsrColIndA(*)
       integer(c_int), value :: blockDim
       type(cusparseBsric02Info), value :: info
       integer(c_int), value :: policy
       !dir$ ignore_tkr pBuffer
       character(c_char), device :: pBuffer(*)
     end function cusparseCbsric02_analysis
  end interface

! cusparseZbsric02_analysis
  interface 
     integer(c_int) function cusparseZbsric02_analysis(handle, dirA, mb, nnzb, descrA, &
          bsrValA, bsrRowPtrA, bsrColIndA, blockDim, info, policy, pBuffer) bind(C,name='cusparseZbsric02_analysis')
       use iso_c_binding
       import cusparseHandle,  cusparseMatDescr, cusparseBsric02Info
       type(cusparseHandle), value :: handle
       integer(c_int), value :: dirA
       integer(c_int), value :: mb, nnzb
       type(cusparseMatDescr), value :: descrA
       complex(8), device :: bsrValA(*)
       integer(c_int), device :: bsrRowPtrA(*), bsrColIndA(*)
       integer(c_int), value :: blockDim
       type(cusparseBsric02Info), value :: info
       integer(c_int), value :: policy
       !dir$ ignore_tkr pBuffer
       character(c_char), device :: pBuffer(*)
     end function cusparseZbsric02_analysis
  end interface

! cusparseSbsric02
  interface 
     integer(c_int) function cusparseSbsric02(handle, dirA, mb, nnzb, descrA, &
          bsrValA, bsrRowPtrA, bsrColIndA, blockDim, info, policy, pBuffer) bind(C,name='cusparseSbsric02')
       use iso_c_binding
       import cusparseHandle,  cusparseMatDescr, cusparseBsric02Info
       type(cusparseHandle), value :: handle
       integer(c_int), value :: dirA
       integer(c_int), value :: mb, nnzb
       type(cusparseMatDescr), value :: descrA
       real(c_float), device :: bsrValA(*)
       integer(c_int), device :: bsrRowPtrA(*), bsrColIndA(*)
       integer(c_int), value :: blockDim
       type(cusparseBsric02Info), value :: info
       integer(c_int), value :: policy
       !dir$ ignore_tkr pBuffer
       character(c_char), device :: pBuffer(*)
     end function cusparseSbsric02
  end interface
  
! cusparseDbsric02
  interface 
     integer(c_int) function cusparseDbsric02(handle, dirA, mb, nnzb, descrA, &
          bsrValA, bsrRowPtrA, bsrColIndA, blockDim, info, policy, pBuffer) bind(C,name='cusparseDbsric02')
       use iso_c_binding
       import cusparseHandle,  cusparseMatDescr, cusparseBsric02Info
       type(cusparseHandle), value :: handle
       integer(c_int), value :: dirA
       integer(c_int), value :: mb, nnzb
       type(cusparseMatDescr), value :: descrA
       real(c_double), device :: bsrValA(*)
       integer(c_int), device :: bsrRowPtrA(*), bsrColIndA(*)
       integer(c_int), value :: blockDim
       type(cusparseBsric02Info), value :: info
       integer(c_int), value :: policy
       !dir$ ignore_tkr pBuffer
       character(c_char), device :: pBuffer(*)
     end function cusparseDbsric02
  end interface
  
! cusparseCbsric02
  interface 
     integer(c_int) function cusparseCbsric02(handle, dirA, mb, nnzb, descrA, &
          bsrValA, bsrRowPtrA, bsrColIndA, blockDim, info, policy, pBuffer) bind(C,name='cusparseCbsric02')
       use iso_c_binding
       import cusparseHandle,  cusparseMatDescr, cusparseBsric02Info
       type(cusparseHandle), value :: handle
       integer(c_int), value :: dirA
       integer(c_int), value :: mb, nnzb
       type(cusparseMatDescr), value :: descrA
       complex(4), device :: bsrValA(*)
       integer(c_int), device :: bsrRowPtrA(*), bsrColIndA(*)
       integer(c_int), value :: blockDim
       type(cusparseBsric02Info), value :: info
       integer(c_int), value :: policy
       !dir$ ignore_tkr pBuffer
       character(c_char), device :: pBuffer(*)
     end function cusparseCbsric02
  end interface
  
! cusparseZbsric02
  interface 
     integer(c_int) function cusparseZbsric02(handle, dirA, mb, nnzb, descrA, &
          bsrValA, bsrRowPtrA, bsrColIndA, blockDim, info, policy, pBuffer) bind(C,name='cusparseZbsric02')
       use iso_c_binding
       import cusparseHandle,  cusparseMatDescr, cusparseBsric02Info
       type(cusparseHandle), value :: handle
       integer(c_int), value :: dirA
       integer(c_int), value :: mb, nnzb
       type(cusparseMatDescr), value :: descrA
       complex(8), device :: bsrValA(*)
       integer(c_int), device :: bsrRowPtrA(*), bsrColIndA(*)
       integer(c_int), value :: blockDim
       type(cusparseBsric02Info), value :: info
       integer(c_int), value :: policy
       !dir$ ignore_tkr pBuffer
       character(c_char), device :: pBuffer(*)
     end function cusparseZbsric02
  end interface
  
! cusparseXbsric02_zeroPivot
  interface cusparseXbsric02_zeroPivot
     integer function cusparseXbsric02_zeroPivot_setHPM(handle, info, position)
       import cusparseHandle, cusparseBsric02Info
       type(cusparseHandle) :: handle
       type(cusparseBsric02Info) :: info
       integer :: position
     end function cusparseXbsric02_zeroPivot_setHPM

     integer function cusparseXbsric02_zeroPivot_setDPM(handle, info, position)
       import cusparseHandle, cusparseBsric02Info
       type(cusparseHandle) :: handle
       type(cusparseBsric02Info) :: info
       integer, device :: position
     end function cusparseXbsric02_zeroPivot_setDPM
  end interface cusparseXbsric02_zeroPivot

#ifdef KEEP_V2_INTERFACES
  interface 
     integer(c_int) function cusparseXbsric02_zeroPivot_v2(handle, info, position) bind(C,name='cusparseXbsric02_zeroPivot')
       use iso_c_binding
       import cusparseHandle, cusparseBsric02Info
       type(cusparseHandle), value :: handle
       type(cusparseBsric02Info), value :: info
       !dir$ ignore_tkr (d) position
       integer(c_int) :: position
     end function cusparseXbsric02_zeroPivot_v2
  end interface
#endif

! cusparseSbsrilu02_numericBoost
  interface cusparseSbsrilu02_numericBoost
     integer function cusparseSbsrilu02_numericBoost_setHPM(handle, info, enable_boost, tol, boost_val)
       import cusparseHandle, cusparseBsrilu02Info
       type(cusparseHandle) :: handle
       type(cusparseBsrilu02Info) :: info
       integer :: enable_boost
       real(8) :: tol
       real(4) :: boost_val
     end function cusparseSbsrilu02_numericBoost_setHPM

     integer function cusparseSbsrilu02_numericBoost_setDPM(handle, info, enable_boost, tol, boost_val)
       import cusparseHandle, cusparseBsrilu02Info
       type(cusparseHandle) :: handle
       type(cusparseBsrilu02Info) :: info
       integer :: enable_boost
       real(8), device :: tol
       real(4), device :: boost_val
     end function cusparseSbsrilu02_numericBoost_setDPM
  end interface cusparseSbsrilu02_numericBoost

#ifdef KEEP_V2_INTERFACES
  interface
     integer(c_int) function cusparseSbsrilu02_numericBoost_v2(handle, info, enable_boost, tol, boost_val) &
          bind(C,name='cusparseSbsrilu02_numericBoost')
       use iso_c_binding
       import cusparseHandle, cusparseBsrilu02Info
       type(cusparseHandle), value :: handle
       type(cusparseBsrilu02Info), value :: info
       integer(c_int), value :: enable_boost
       !dir$ ignore_tkr (d) tol, boost_val
       real(c_double) :: tol
       real(c_float) :: boost_val
     end function cusparseSbsrilu02_numericBoost_v2
  end interface
#endif

! cusparseDbsrilu02_numericBoost
  interface cusparseDbsrilu02_numericBoost
     integer function cusparseDbsrilu02_numericBoost_setHPM(handle, info, enable_boost, tol, boost_val)
       import cusparseHandle, cusparseBsrilu02Info
       type(cusparseHandle) :: handle
       type(cusparseBsrilu02Info) :: info
       integer :: enable_boost
       real(8) :: tol
       real(8) :: boost_val
     end function cusparseDbsrilu02_numericBoost_setHPM

     integer function cusparseDbsrilu02_numericBoost_setDPM(handle, info, enable_boost, tol, boost_val)
       import cusparseHandle, cusparseBsrilu02Info
       type(cusparseHandle) :: handle
       type(cusparseBsrilu02Info) :: info
       integer :: enable_boost
       real(8), device :: tol
       real(8), device :: boost_val
     end function cusparseDbsrilu02_numericBoost_setDPM
  end interface cusparseDbsrilu02_numericBoost

#ifdef KEEP_V2_INTERFACES
  interface
     integer(c_int) function cusparseDbsrilu02_numericBoost_v2(handle, info, enable_boost, tol, boost_val) &
          bind(C,name='cusparseDbsrilu02_numericBoost')
       use iso_c_binding
       import cusparseHandle, cusparseBsrilu02Info
       type(cusparseHandle), value :: handle
       type(cusparseBsrilu02Info), value :: info
       integer(c_int), value :: enable_boost
       !dir$ ignore_tkr (d) tol, boost_val
       real(c_double) :: tol
       real(c_double) :: boost_val
     end function cusparseDbsrilu02_numericBoost_v2
  end interface
#endif

! cusparseCbsrilu02_numericBoost
  interface cusparseCbsrilu02_numericBoost
     integer function cusparseCbsrilu02_numericBoost_setHPM(handle, info, enable_boost, tol, boost_val)
       import cusparseHandle, cusparseBsrilu02Info
       type(cusparseHandle) :: handle
       type(cusparseBsrilu02Info) :: info
       integer :: enable_boost
       real(8) :: tol
       complex(4) :: boost_val
     end function cusparseCbsrilu02_numericBoost_setHPM

     integer function cusparseCbsrilu02_numericBoost_setDPM(handle, info, enable_boost, tol, boost_val)
       import cusparseHandle, cusparseBsrilu02Info
       type(cusparseHandle) :: handle
       type(cusparseBsrilu02Info) :: info
       integer :: enable_boost
       real(8), device :: tol
       complex(4), device :: boost_val
     end function cusparseCbsrilu02_numericBoost_setDPM
  end interface cusparseCbsrilu02_numericBoost

#ifdef KEEP_V2_INTERFACES
  interface
     integer(c_int) function cusparseCbsrilu02_numericBoost_v2(handle, info, enable_boost, tol, boost_val) &
          bind(C,name='cusparseCbsrilu02_numericBoost')
       use iso_c_binding
       import cusparseHandle, cusparseBsrilu02Info
       type(cusparseHandle), value :: handle
       type(cusparseBsrilu02Info), value :: info
       integer(c_int), value :: enable_boost
       !dir$ ignore_tkr (d) tol, boost_val
       real(c_double) :: tol
       complex(4) :: boost_val
     end function cusparseCbsrilu02_numericBoost_v2
  end interface
#endif

! cusparseZbsrilu02_numericBoost
  interface cusparseZbsrilu02_numericBoost
     integer function cusparseZbsrilu02_numericBoost_setHPM(handle, info, enable_boost, tol, boost_val)
       import cusparseHandle, cusparseBsrilu02Info
       type(cusparseHandle) :: handle
       type(cusparseBsrilu02Info) :: info
       integer :: enable_boost
       real(8) :: tol
       complex(8) :: boost_val
     end function cusparseZbsrilu02_numericBoost_setHPM

     integer function cusparseZbsrilu02_numericBoost_setDPM(handle, info, enable_boost, tol, boost_val)
       import cusparseHandle, cusparseBsrilu02Info
       type(cusparseHandle) :: handle
       type(cusparseBsrilu02Info) :: info
       integer :: enable_boost
       real(8), device :: tol
       complex(8), device :: boost_val
     end function cusparseZbsrilu02_numericBoost_setDPM
  end interface cusparseZbsrilu02_numericBoost

#ifdef KEEP_V2_INTERFACES
  interface
     integer(c_int) function cusparseZbsrilu02_numericBoost_v2(handle, info, enable_boost, tol, boost_val) &
          bind(C,name='cusparseZbsrilu02_numericBoost')
       use iso_c_binding
       import cusparseHandle, cusparseBsrilu02Info
       type(cusparseHandle), value :: handle
       type(cusparseBsrilu02Info), value :: info
       integer(c_int), value :: enable_boost
       !dir$ ignore_tkr (d) tol, boost_val
       real(c_double) :: tol
       complex(8) :: boost_val
     end function cusparseZbsrilu02_numericBoost_v2
  end interface
#endif

! cusparseSbsrilu02_bufferSize
  interface cusparseSbsrilu02_bufferSize
     integer(c_int) function cusparseSbsrilu02_bufferSize(handle, dirA, mb, nnzb, descrA, &
          bsrValA, bsrRowPtrA, bsrColIndA, blockDim, info, pBufferSize) bind(C,name='cusparseSbsrilu02_bufferSize')
       use iso_c_binding
       import cusparseHandle,  cusparseMatDescr, cusparseBsrilu02Info
       type(cusparseHandle), value :: handle
       integer(c_int), value :: dirA
       integer(c_int), value :: mb, nnzb
       type(cusparseMatDescr), value :: descrA
       real(c_float), device :: bsrValA(*)
       integer(c_int), device :: bsrRowPtrA(*), bsrColIndA(*)
       integer(c_int), value :: blockDim
       type(cusparseBsrilu02Info), value :: info
       integer(4) :: pBufferSize
     end function cusparseSbsrilu02_bufferSize
     integer(c_int) function cusparseSbsrilu02_bufferSizeExt(handle, dirA, mb, nnzb, descrA, &
          bsrValA, bsrRowPtrA, bsrColIndA, blockDim, info, pBufferSize) bind(C,name='cusparseSbsrilu02_bufferSizeExt')
       use iso_c_binding
       import cusparseHandle,  cusparseMatDescr, cusparseBsrilu02Info
       type(cusparseHandle), value :: handle
       integer(c_int), value :: dirA
       integer(c_int), value :: mb, nnzb
       type(cusparseMatDescr), value :: descrA
       real(c_float), device :: bsrValA(*)
       integer(c_int), device :: bsrRowPtrA(*), bsrColIndA(*)
       integer(c_int), value :: blockDim
       type(cusparseBsrilu02Info), value :: info
       integer(8) :: pBufferSize
     end function cusparseSbsrilu02_bufferSizeExt
  end interface

! cusparseDbsrilu02_bufferSize
  interface cusparseDbsrilu02_bufferSize
     integer(c_int) function cusparseDbsrilu02_bufferSize(handle, dirA, mb, nnzb, descrA, &
          bsrValA, bsrRowPtrA, bsrColIndA, blockDim, info, pBufferSize) bind(C,name='cusparseDbsrilu02_bufferSize')
       use iso_c_binding
       import cusparseHandle,  cusparseMatDescr, cusparseBsrilu02Info
       type(cusparseHandle), value :: handle
       integer(c_int), value :: dirA
       integer(c_int), value :: mb, nnzb
       type(cusparseMatDescr), value :: descrA
       real(c_double), device :: bsrValA(*)
       integer(c_int), device :: bsrRowPtrA(*), bsrColIndA(*)
       integer(c_int), value :: blockDim
       type(cusparseBsrilu02Info), value :: info
       integer(4) :: pBufferSize
     end function cusparseDbsrilu02_bufferSize
     integer(c_int) function cusparseDbsrilu02_bufferSizeExt(handle, dirA, mb, nnzb, descrA, &
          bsrValA, bsrRowPtrA, bsrColIndA, blockDim, info, pBufferSize) bind(C,name='cusparseDbsrilu02_bufferSizeExt')
       use iso_c_binding
       import cusparseHandle,  cusparseMatDescr, cusparseBsrilu02Info
       type(cusparseHandle), value :: handle
       integer(c_int), value :: dirA
       integer(c_int), value :: mb, nnzb
       type(cusparseMatDescr), value :: descrA
       real(c_double), device :: bsrValA(*)
       integer(c_int), device :: bsrRowPtrA(*), bsrColIndA(*)
       integer(c_int), value :: blockDim
       type(cusparseBsrilu02Info), value :: info
       integer(8) :: pBufferSize
     end function cusparseDbsrilu02_bufferSizeExt
  end interface

! cusparseCbsrilu02_bufferSize
  interface cusparseCbsrilu02_bufferSize
     integer(c_int) function cusparseCbsrilu02_bufferSize(handle, dirA, mb, nnzb, descrA, &
          bsrValA, bsrRowPtrA, bsrColIndA, blockDim, info, pBufferSize) bind(C,name='cusparseCbsrilu02_bufferSize')
       use iso_c_binding
       import cusparseHandle,  cusparseMatDescr, cusparseBsrilu02Info
       type(cusparseHandle), value :: handle
       integer(c_int), value :: dirA
       integer(c_int), value :: mb, nnzb
       type(cusparseMatDescr), value :: descrA
       complex(4), device :: bsrValA(*)
       integer(c_int), device :: bsrRowPtrA(*), bsrColIndA(*)
       integer(c_int), value :: blockDim
       type(cusparseBsrilu02Info), value :: info
       integer(4) :: pBufferSize
     end function cusparseCbsrilu02_bufferSize
     integer(c_int) function cusparseCbsrilu02_bufferSizeExt(handle, dirA, mb, nnzb, descrA, &
          bsrValA, bsrRowPtrA, bsrColIndA, blockDim, info, pBufferSize) bind(C,name='cusparseCbsrilu02_bufferSizeExt')
       use iso_c_binding
       import cusparseHandle,  cusparseMatDescr, cusparseBsrilu02Info
       type(cusparseHandle), value :: handle
       integer(c_int), value :: dirA
       integer(c_int), value :: mb, nnzb
       type(cusparseMatDescr), value :: descrA
       complex(4), device :: bsrValA(*)
       integer(c_int), device :: bsrRowPtrA(*), bsrColIndA(*)
       integer(c_int), value :: blockDim
       type(cusparseBsrilu02Info), value :: info
       integer(8) :: pBufferSize
     end function cusparseCbsrilu02_bufferSizeExt
  end interface

! cusparseZbsrilu02_bufferSize
  interface cusparseZbsrilu02_bufferSize
     integer(c_int) function cusparseZbsrilu02_bufferSize(handle, dirA, mb, nnzb, descrA, &
          bsrValA, bsrRowPtrA, bsrColIndA, blockDim, info, pBufferSize) bind(C,name='cusparseZbsrilu02_bufferSize')
       use iso_c_binding
       import cusparseHandle,  cusparseMatDescr, cusparseBsrilu02Info
       type(cusparseHandle), value :: handle
       integer(c_int), value :: dirA
       integer(c_int), value :: mb, nnzb
       type(cusparseMatDescr), value :: descrA
       complex(8), device :: bsrValA(*)
       integer(c_int), device :: bsrRowPtrA(*), bsrColIndA(*)
       integer(c_int), value :: blockDim
       type(cusparseBsrilu02Info), value :: info
       integer(4) :: pBufferSize
     end function cusparseZbsrilu02_bufferSize
     integer(c_int) function cusparseZbsrilu02_bufferSizeExt(handle, dirA, mb, nnzb, descrA, &
          bsrValA, bsrRowPtrA, bsrColIndA, blockDim, info, pBufferSize) bind(C,name='cusparseZbsrilu02_bufferSizeExt')
       use iso_c_binding
       import cusparseHandle,  cusparseMatDescr, cusparseBsrilu02Info
       type(cusparseHandle), value :: handle
       integer(c_int), value :: dirA
       integer(c_int), value :: mb, nnzb
       type(cusparseMatDescr), value :: descrA
       complex(8), device :: bsrValA(*)
       integer(c_int), device :: bsrRowPtrA(*), bsrColIndA(*)
       integer(c_int), value :: blockDim
       type(cusparseBsrilu02Info), value :: info
       integer(8) :: pBufferSize
     end function cusparseZbsrilu02_bufferSizeExt
  end interface

! cusparseSbsrilu02_analysis
  interface 
     integer(c_int) function cusparseSbsrilu02_analysis(handle, dirA, mb, nnzb, descrA, &
          bsrValA, bsrRowPtrA, bsrColIndA, blockDim, info, policy, pBuffer) bind(C,name='cusparseSbsrilu02_analysis')
       use iso_c_binding
       import cusparseHandle,  cusparseMatDescr, cusparseBsrilu02Info
       type(cusparseHandle), value :: handle
       integer(c_int), value :: dirA
       integer(c_int), value :: mb, nnzb
       type(cusparseMatDescr), value :: descrA
       real(c_float), device :: bsrValA(*)
       integer(c_int), device :: bsrRowPtrA(*), bsrColIndA(*)
       integer(c_int), value :: blockDim
       type(cusparseBsrilu02Info), value :: info
       integer(c_int), value :: policy
       !dir$ ignore_tkr pBuffer
       character(c_char), device :: pBuffer(*)
     end function cusparseSbsrilu02_analysis
  end interface
  
! cusparseDbsrilu02_analysis
  interface 
     integer(c_int) function cusparseDbsrilu02_analysis(handle, dirA, mb, nnzb, descrA, &
          bsrValA, bsrRowPtrA, bsrColIndA, blockDim, info, policy, pBuffer) bind(C,name='cusparseDbsrilu02_analysis')
       use iso_c_binding
       import cusparseHandle,  cusparseMatDescr, cusparseBsrilu02Info
       type(cusparseHandle), value :: handle
       integer(c_int), value :: dirA
       integer(c_int), value :: mb, nnzb
       type(cusparseMatDescr), value :: descrA
       real(c_double), device :: bsrValA(*)
       integer(c_int), device :: bsrRowPtrA(*), bsrColIndA(*)
       integer(c_int), value :: blockDim
       type(cusparseBsrilu02Info), value :: info
       integer(c_int), value :: policy
       !dir$ ignore_tkr pBuffer
       character(c_char), device :: pBuffer(*)
     end function cusparseDbsrilu02_analysis
  end interface

! cusparseCbsrilu02_analysis
  interface 
     integer(c_int) function cusparseCbsrilu02_analysis(handle, dirA, mb, nnzb, descrA, &
          bsrValA, bsrRowPtrA, bsrColIndA, blockDim, info, policy, pBuffer) bind(C,name='cusparseCbsrilu02_analysis')
       use iso_c_binding
       import cusparseHandle,  cusparseMatDescr, cusparseBsrilu02Info
       type(cusparseHandle), value :: handle
       integer(c_int), value :: dirA
       integer(c_int), value :: mb, nnzb
       type(cusparseMatDescr), value :: descrA
       complex(4), device :: bsrValA(*)
       integer(c_int), device :: bsrRowPtrA(*), bsrColIndA(*)
       integer(c_int), value :: blockDim
       type(cusparseBsrilu02Info), value :: info
       integer(c_int), value :: policy
       !dir$ ignore_tkr pBuffer
       character(c_char), device :: pBuffer(*)
     end function cusparseCbsrilu02_analysis
  end interface

! cusparseZbsrilu02_analysis
  interface 
     integer(c_int) function cusparseZbsrilu02_analysis(handle, dirA, mb, nnzb, descrA, &
          bsrValA, bsrRowPtrA, bsrColIndA, blockDim, info, policy, pBuffer) bind(C,name='cusparseZbsrilu02_analysis')
       use iso_c_binding
       import cusparseHandle,  cusparseMatDescr, cusparseBsrilu02Info
       type(cusparseHandle), value :: handle
       integer(c_int), value :: dirA
       integer(c_int), value :: mb, nnzb
       type(cusparseMatDescr), value :: descrA
       complex(8), device :: bsrValA(*)
       integer(c_int), device :: bsrRowPtrA(*), bsrColIndA(*)
       integer(c_int), value :: blockDim
       type(cusparseBsrilu02Info), value :: info
       integer(c_int), value :: policy
       !dir$ ignore_tkr pBuffer
       character(c_char), device :: pBuffer(*)
     end function cusparseZbsrilu02_analysis
  end interface

! cusparseSbsrilu02
  interface 
     integer(c_int) function cusparseSbsrilu02(handle, dirA, mb, nnzb, descrA, &
          bsrValA, bsrRowPtrA, bsrColIndA, blockDim, info, policy, pBuffer) bind(C,name='cusparseSbsrilu02')
       use iso_c_binding
       import cusparseHandle,  cusparseMatDescr, cusparseBsrilu02Info
       type(cusparseHandle), value :: handle
       integer(c_int), value :: dirA
       integer(c_int), value :: mb, nnzb
       type(cusparseMatDescr), value :: descrA
       real(c_float), device :: bsrValA(*)
       integer(c_int), device :: bsrRowPtrA(*), bsrColIndA(*)
       integer(c_int), value :: blockDim
       type(cusparseBsrilu02Info), value :: info
       integer(c_int), value :: policy
       !dir$ ignore_tkr pBuffer
       character(c_char), device :: pBuffer(*)
     end function cusparseSbsrilu02
  end interface
  
! cusparseDbsrilu02
  interface 
     integer(c_int) function cusparseDbsrilu02(handle, dirA, mb, nnzb, descrA, &
          bsrValA, bsrRowPtrA, bsrColIndA, blockDim, info, policy, pBuffer) bind(C,name='cusparseDbsrilu02')
       use iso_c_binding
       import cusparseHandle,  cusparseMatDescr, cusparseBsrilu02Info
       type(cusparseHandle), value :: handle
       integer(c_int), value :: dirA
       integer(c_int), value :: mb, nnzb
       type(cusparseMatDescr), value :: descrA
       real(c_double), device :: bsrValA(*)
       integer(c_int), device :: bsrRowPtrA(*), bsrColIndA(*)
       integer(c_int), value :: blockDim
       type(cusparseBsrilu02Info), value :: info
       integer(c_int), value :: policy
       !dir$ ignore_tkr pBuffer
       character(c_char), device :: pBuffer(*)
     end function cusparseDbsrilu02
  end interface
  
! cusparseCbsrilu02
  interface 
     integer(c_int) function cusparseCbsrilu02(handle, dirA, mb, nnzb, descrA, &
          bsrValA, bsrRowPtrA, bsrColIndA, blockDim, info, policy, pBuffer) bind(C,name='cusparseCbsrilu02')
       use iso_c_binding
       import cusparseHandle,  cusparseMatDescr, cusparseBsrilu02Info
       type(cusparseHandle), value :: handle
       integer(c_int), value :: dirA
       integer(c_int), value :: mb, nnzb
       type(cusparseMatDescr), value :: descrA
       complex(4), device :: bsrValA(*)
       integer(c_int), device :: bsrRowPtrA(*), bsrColIndA(*)
       integer(c_int), value :: blockDim
       type(cusparseBsrilu02Info), value :: info
       integer(c_int), value :: policy
       !dir$ ignore_tkr pBuffer
       character(c_char), device :: pBuffer(*)
     end function cusparseCbsrilu02
  end interface
  
! cusparseZbsrilu02
  interface 
     integer(c_int) function cusparseZbsrilu02(handle, dirA, mb, nnzb, descrA, &
          bsrValA, bsrRowPtrA, bsrColIndA, blockDim, info, policy, pBuffer) bind(C,name='cusparseZbsrilu02')
       use iso_c_binding
       import cusparseHandle,  cusparseMatDescr, cusparseBsrilu02Info
       type(cusparseHandle), value :: handle
       integer(c_int), value :: dirA
       integer(c_int), value :: mb, nnzb
       type(cusparseMatDescr), value :: descrA
       complex(8), device :: bsrValA(*)
       integer(c_int), device :: bsrRowPtrA(*), bsrColIndA(*)
       integer(c_int), value :: blockDim
       type(cusparseBsrilu02Info), value :: info
       integer(c_int), value :: policy
       !dir$ ignore_tkr pBuffer
       character(c_char), device :: pBuffer(*)
     end function cusparseZbsrilu02
  end interface
  
! cusparseXbsrilu02_zeroPivot
  interface cusparseXbsrilu02_zeroPivot
     integer function cusparseXbsrilu02_zeroPivot_setHPM(handle, info, position)
       import cusparseHandle, cusparseBsrilu02Info
       type(cusparseHandle) :: handle
       type(cusparseBsrilu02Info) :: info
       integer :: position
     end function cusparseXbsrilu02_zeroPivot_setHPM

     integer function cusparseXbsrilu02_zeroPivot_setDPM(handle, info, position)
       import cusparseHandle, cusparseBsrilu02Info
       type(cusparseHandle) :: handle
       type(cusparseBsrilu02Info) :: info
       integer, device :: position
     end function cusparseXbsrilu02_zeroPivot_setDPM
  end interface cusparseXbsrilu02_zeroPivot

#ifdef KEEP_V2_INTERFACES
  interface 
     integer(c_int) function cusparseXbsrilu02_zeroPivot_v2(handle, info, position) bind(C,name='cusparseXbsrilu02_zeroPivot')
       use iso_c_binding
       import cusparseHandle, cusparseBsrilu02Info
       type(cusparseHandle), value :: handle
       type(cusparseBsrilu02Info), value :: info
       !dir$ ignore_tkr (d) position
       integer(c_int) :: position
     end function cusparseXbsrilu02_zeroPivot_v2
  end interface
#endif

  ! -------------------
  ! 6.5 Preconditioners
  ! -------------------
  ! none

  ! -------------------
  ! 7.0 Preconditioners
  ! -------------------
  ! none

  ! ---------------
  ! 7.0 Reorderings
  ! ---------------

! cusparseScsrColor
  interface cusparseScsrColor
    integer function cusparseScsrcolor_setHPM(handle, m, nnz, descrA, csrValA, csrRowPtrA, csrColIndA, &
          fractionToColor, ncolors, coloring, reordering, info)
    import cusparseHandle, cusparseMatDescr, cusparseColorInfo
    type(cusparseHandle) :: handle
    type(cusparseMatDescr) :: descrA
    type(cusparseColorInfo) :: info
    integer :: m, nnz
    real(4), device :: csrValA(*)
    integer, device :: csrRowPtrA(*), csrColIndA(*), coloring(*), reordering(*)

    real(4) :: fractionToColor
    integer :: ncolors

    end function cusparseScsrcolor_setHPM
    integer function cusparseScsrcolor_setDPM(handle, m, nnz, descrA, csrValA, csrRowPtrA, csrColIndA, &
          fractionToColor, ncolors, coloring, reordering, info)
    import cusparseHandle, cusparseMatDescr, cusparseColorInfo
    type(cusparseHandle) :: handle
    type(cusparseMatDescr) :: descrA
    type(cusparseColorInfo) :: info
    integer :: m, nnz
    real(4), device :: csrValA(*)
    integer, device :: csrRowPtrA(*), csrColIndA(*), coloring(*), reordering(*)

    real(4), device :: fractionToColor
    integer, device :: ncolors
    end function cusparseScsrcolor_setDPM
  end interface cusparseScsrColor

#ifdef KEEP_V2_INTERFACES
  interface cusparseScsrcolor_v2
     integer(c_int) function cusparseScsrcolor_v2(handle, m, nnz, descrA, csrValA, csrRowPtrA, csrColIndA, &
          fractionToColor, ncolors, coloring, reordering, info) bind(C,name='cusparseScsrcolor')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr, cusparseColorInfo
       type(cusparseHandle), value :: handle
       integer(c_int), value :: m, nnz
       type(cusparseMatDescr), value :: descrA
       real(c_float), device :: csrValA(*)
       integer(c_int), device :: csrRowPtrA(*), csrColIndA(*)       
       !dir$ ignore_tkr (d) fractionToColor
       real(c_float) :: fractionToColor
      !dir$ ignore_tkr (d) ncolors
       integer(c_int) :: ncolors
       integer(c_int), device :: coloring(*), reordering(*)
       type(cusparseColorInfo), value :: info
     end function cusparseScsrcolor_v2
  end interface cusparseScsrcolor_v2
#endif
       
! cusparseDcsrColor
  interface cusparseDcsrColor
    integer function cusparseDcsrcolor_setHPM(handle, m, nnz, descrA, csrValA, csrRowPtrA, csrColIndA, &
          fractionToColor, ncolors, coloring, reordering, info)
    import cusparseHandle, cusparseMatDescr, cusparseColorInfo
    type(cusparseHandle) :: handle
    type(cusparseMatDescr) :: descrA
    type(cusparseColorInfo) :: info
    integer :: m, nnz
    real(8), device :: csrValA(*)
    integer, device :: csrRowPtrA(*), csrColIndA(*), coloring(*), reordering(*)

    real(8) :: fractionToColor
    integer :: ncolors
    end function cusparseDcsrcolor_setHPM
    integer function cusparseDcsrcolor_setDPM(handle, m, nnz, descrA, csrValA, csrRowPtrA, csrColIndA, &
          fractionToColor, ncolors, coloring, reordering, info)
    import cusparseHandle, cusparseMatDescr, cusparseColorInfo
    type(cusparseHandle) :: handle
    type(cusparseMatDescr) :: descrA
    type(cusparseColorInfo) :: info
    integer :: m, nnz
    real(8), device :: csrValA(*)
    integer, device :: csrRowPtrA(*), csrColIndA(*), coloring(*), reordering(*)

    real(8), device :: fractionToColor
    integer, device :: ncolors
    end function cusparseDcsrcolor_setDPM
  end interface cusparseDcsrColor

#ifdef KEEP_V2_INTERFACES
  interface cusparseDcsrcolor_v2
     integer(c_int) function cusparseDcsrcolor_v2(handle, m, nnz, descrA, csrValA, csrRowPtrA, csrColIndA, &
          fractionToColor, ncolors, coloring, reordering, info) bind(C,name='cusparseDcsrcolor')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr, cusparseColorInfo
       type(cusparseHandle), value :: handle
       integer(c_int), value :: m, nnz
       type(cusparseMatDescr), value :: descrA
       real(c_double), device :: csrValA(*)
       integer(c_int), device :: csrRowPtrA(*), csrColIndA(*)       
       !dir$ ignore_tkr (d) fractionToColor
       real(c_double) :: fractionToColor
       !dir$ ignore_tkr (d) ncolors
       integer(c_int) :: ncolors
       integer(c_int), device :: coloring(*), reordering(*)
       type(cusparseColorInfo), value :: info
     end function cusparseDcsrcolor_v2
  end interface cusparseDcsrcolor_v2
#endif
       
! cusparseCcsrColor
  interface cusparseCcsrColor
    integer function cusparseCcsrcolor_setHPM(handle, m, nnz, descrA, csrValA, csrRowPtrA, csrColIndA, &
          fractionToColor, ncolors, coloring, reordering, info)
    import cusparseHandle, cusparseMatDescr, cusparseColorInfo
    type(cusparseHandle) :: handle
    type(cusparseMatDescr) :: descrA
    type(cusparseColorInfo) :: info
    integer :: m, nnz
    complex(4), device :: csrValA(*)
    integer, device :: csrRowPtrA(*), csrColIndA(*), coloring(*), reordering(*)

    real(4) :: fractionToColor
    integer :: ncolors
    end function cusparseCcsrcolor_setHPM
    integer function cusparseCcsrcolor_setDPM(handle, m, nnz, descrA, csrValA, csrRowPtrA, csrColIndA, &
          fractionToColor, ncolors, coloring, reordering, info)
    import cusparseHandle, cusparseMatDescr, cusparseColorInfo
    type(cusparseHandle) :: handle
    type(cusparseMatDescr) :: descrA
    type(cusparseColorInfo) :: info
    integer :: m, nnz
    complex(4), device :: csrValA(*)
    integer, device :: csrRowPtrA(*), csrColIndA(*), coloring(*), reordering(*)

    real(4), device :: fractionToColor
    integer, device :: ncolors
    end function cusparseCcsrcolor_setDPM
  end interface cusparseCcsrColor

#ifdef KEEP_V2_INTERFACES
  interface cusparseCcsrcolor_v2
     integer(c_int) function cusparseCcsrcolor_v2(handle, m, nnz, descrA, csrValA, csrRowPtrA, csrColIndA, &
          fractionToColor, ncolors, coloring, reordering, info) bind(C,name='cusparseCcsrcolor')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr, cusparseColorInfo
       type(cusparseHandle), value :: handle
       integer(c_int), value :: m, nnz
       type(cusparseMatDescr), value :: descrA
       complex(4), device :: csrValA(*)
       integer(c_int), device :: csrRowPtrA(*), csrColIndA(*)       
       !dir$ ignore_tkr (d) fractionToColor
       real(c_float) :: fractionToColor
       !dir$ ignore_tkr (d) ncolors
       integer(c_int) :: ncolors
       integer(c_int), device :: coloring(*), reordering(*)
       type(cusparseColorInfo), value :: info
     end function cusparseCcsrcolor_v2
  end interface cusparseCcsrcolor_v2
#endif
       
! cusparseZcsrColor
  interface cusparseZcsrColor
    integer function cusparseZcsrcolor_setHPM(handle, m, nnz, descrA, csrValA, csrRowPtrA, csrColIndA, &
          fractionToColor, ncolors, coloring, reordering, info)
    import cusparseHandle, cusparseMatDescr, cusparseColorInfo
    type(cusparseHandle) :: handle
    type(cusparseMatDescr) :: descrA
    type(cusparseColorInfo) :: info
    integer :: m, nnz
    complex(8), device :: csrValA(*)
    integer, device :: csrRowPtrA(*), csrColIndA(*), coloring(*), reordering(*)

    real(8) :: fractionToColor
    integer :: ncolors
    end function cusparseZcsrcolor_setHPM
    integer function cusparseZcsrcolor_setDPM(handle, m, nnz, descrA, csrValA, csrRowPtrA, csrColIndA, &
          fractionToColor, ncolors, coloring, reordering, info)
    import cusparseHandle, cusparseMatDescr, cusparseColorInfo
    type(cusparseHandle) :: handle
    type(cusparseMatDescr) :: descrA
    type(cusparseColorInfo) :: info
    integer :: m, nnz
    complex(8), device :: csrValA(*)
    integer, device :: csrRowPtrA(*), csrColIndA(*), coloring(*), reordering(*)

    real(8), device :: fractionToColor
    integer, device :: ncolors
    end function cusparseZcsrcolor_setDPM
  end interface cusparseZcsrColor

#ifdef KEEP_V2_INTERFACES
  interface cusparseZcsrcolor_v2
     integer(c_int) function cusparseZcsrcolor_v2(handle, m, nnz, descrA, csrValA, csrRowPtrA, csrColIndA, &
          fractionToColor, ncolors, coloring, reordering, info) bind(C,name='cusparseZcsrcolor')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr, cusparseColorInfo
       type(cusparseHandle), value :: handle
       integer(c_int), value :: m, nnz
       type(cusparseMatDescr), value :: descrA
       complex(8), device :: csrValA(*)
       integer(c_int), device :: csrRowPtrA(*), csrColIndA(*)       
       !dir$ ignore_tkr (d) fractionToColor
       real(c_double) :: fractionToColor
       !dir$ ignore_tkr (d) ncolors
       integer(c_int) :: ncolors
       integer(c_int), device :: coloring(*), reordering(*)
       type(cusparseColorInfo), value :: info
     end function cusparseZcsrcolor_v2
  end interface cusparseZcsrcolor_v2
#endif
       
  ! ----------------------
  ! 5.5 Format conversions
  ! ----------------------

! cusparseSbsr2csr
  interface
     integer(c_int) function cusparseSbsr2csr(handle, dirA, mb, nb, descrA, bsrValA, bsrRowPtrA, bsrColIndA, &
          blockDim, descrC, csrValC, csrRowPtrC, csrColIndC) bind(C, name='cusparseSbsr2csr')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr
       type(cusparseHandle), value :: handle       
       integer(c_int), value :: dirA, mb, nb, blockDim
       type(cusparseMatDescr), value :: descrA, descrC
       real(c_float), device :: bsrValA(*), csrValC(*)
       integer(c_int), device :: bsrRowPtrA(*), bsrColIndA(*), csrRowPtrC(*), csrColIndC(*)
     end function cusparseSbsr2csr
  end interface

! cusparseDbsr2csr
  interface
     integer(c_int) function cusparseDbsr2csr(handle, dirA, mb, nb, descrA, bsrValA, bsrRowPtrA, bsrColIndA, &
          blockDim, descrC, csrValC, csrRowPtrC, csrColIndC) bind(C, name='cusparseDbsr2csr')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr
       type(cusparseHandle), value :: handle       
       integer(c_int), value :: dirA, mb, nb, blockDim
       type(cusparseMatDescr), value :: descrA, descrC
       real(c_double), device :: bsrValA(*), csrValC(*)
       integer(c_int), device :: bsrRowPtrA(*), bsrColIndA(*), csrRowPtrC(*), csrColIndC(*)
     end function cusparseDbsr2csr
  end interface

! cusparseCbsr2csr
  interface
     integer(c_int) function cusparseCbsr2csr(handle, dirA, mb, nb, descrA, bsrValA, bsrRowPtrA, bsrColIndA, &
          blockDim, descrC, csrValC, csrRowPtrC, csrColIndC) bind(C, name='cusparseCbsr2csr')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr
       type(cusparseHandle), value :: handle       
       integer(c_int), value :: dirA, mb, nb, blockDim
       type(cusparseMatDescr), value :: descrA, descrC
       complex(4), device :: bsrValA(*), csrValC(*)
       integer(c_int), device :: bsrRowPtrA(*), bsrColIndA(*), csrRowPtrC(*), csrColIndC(*)
     end function cusparseCbsr2csr
  end interface

! cusparseZbsr2csr
  interface
     integer(c_int) function cusparseZbsr2csr(handle, dirA, mb, nb, descrA, bsrValA, bsrRowPtrA, bsrColIndA, &
          blockDim, descrC, csrValC, csrRowPtrC, csrColIndC) bind(C, name='cusparseZbsr2csr')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr
       type(cusparseHandle), value :: handle       
       integer(c_int), value :: dirA, mb, nb, blockDim
       type(cusparseMatDescr), value :: descrA, descrC
       complex(8), device :: bsrValA(*), csrValC(*)
       integer(c_int), device :: bsrRowPtrA(*), bsrColIndA(*), csrRowPtrC(*), csrColIndC(*)
     end function cusparseZbsr2csr
  end interface

! cusparseXcoo2csr
  interface
     integer(c_int) function cusparseXcoo2csr(handle, cooRowInd, nnz, m, csrRowPtr, idxBase) &
          bind(C, name='cusparseXcoo2csr')
       use iso_c_binding
       import cusparseHandle       
       type(cusparseHandle), value :: handle       
       integer(c_int), value :: nnz, m, idxBase
       integer(c_int), device :: cooRowInd(*), csrRowPtr(*)
     end function cusparseXcoo2csr
  end interface

! cusparseScsc2dense
  interface
     integer(c_int) function cusparseScsc2dense(handle, m, n, descrA, cscValA, cscRowIndA, cscColPtrA, A, lda) &
          bind(C, name='cusparseScsc2dense')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr       
       type(cusparseHandle), value :: handle       
       integer(c_int), value :: m, n, lda
       type(cusparseMatDescr), value :: descrA
       !dir$ ignore_tkr(r) A
       real(c_float), device :: cscValA(*), A(*)
       integer(c_int), device :: cscRowIndA(*), cscColPtrA(*)
     end function cusparseScsc2dense
  end interface

! cusparseDcsc2dense
  interface
     integer(c_int) function cusparseDcsc2dense(handle, m, n, descrA, cscValA, cscRowIndA, cscColPtrA, A, lda) &
          bind(C, name='cusparseDcsc2dense')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr       
       type(cusparseHandle), value :: handle       
       integer(c_int), value :: m, n, lda
       type(cusparseMatDescr), value :: descrA
       !dir$ ignore_tkr(r) A
       real(c_double), device :: cscValA(*), A(*)
       integer(c_int), device :: cscRowIndA(*), cscColPtrA(*)
     end function cusparseDcsc2dense
  end interface
     
! cusparseCcsc2dense
  interface
     integer(c_int) function cusparseCcsc2dense(handle, m, n, descrA, cscValA, cscRowIndA, cscColPtrA, A, lda) &
          bind(C, name='cusparseCcsc2dense')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr       
       type(cusparseHandle), value :: handle       
       integer(c_int), value :: m, n, lda
       type(cusparseMatDescr), value :: descrA
       !dir$ ignore_tkr(r) A
       complex(4), device :: cscValA(*), A(*)
       integer(c_int), device :: cscRowIndA(*), cscColPtrA(*)
     end function cusparseCcsc2dense
  end interface
     
! cusparseZcsc2dense
  interface
     integer(c_int) function cusparseZcsc2dense(handle, m, n, descrA, cscValA, cscRowIndA, cscColPtrA, A, lda) &
          bind(C, name='cusparseZcsc2dense')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr       
       type(cusparseHandle), value :: handle       
       integer(c_int), value :: m, n, lda
       type(cusparseMatDescr), value :: descrA
       !dir$ ignore_tkr(r) A
       complex(8), device :: cscValA(*), A(*)
       integer(c_int), device :: cscRowIndA(*), cscColPtrA(*)
     end function cusparseZcsc2dense
  end interface

! cusparseScsc2hyb
  interface
     integer(c_int) function cusparseScsc2hyb(handle, m, n, descrA, cscValA, cscRowIndA, cscColPtrA, &
          hybA, userEllWidth, partitionType) bind(C, name='cusparseScsc2hyb')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr, cusparseHybMat       
       type(cusparseHandle), value :: handle       
       integer(c_int), value :: m, n, userEllWidth, partitionType
       type(cusparseMatDescr), value :: descrA
       real(c_float), device :: cscValA(*)
       integer(c_int), device :: cscRowIndA(*), cscColPtrA(*)
       type(cusparseHybMat), value :: hybA
     end function cusparseScsc2hyb
  end interface

! cusparseDcsc2hyb
  interface
     integer(c_int) function cusparseDcsc2hyb(handle, m, n, descrA, cscValA, cscRowIndA, cscColPtrA, &
          hybA, userEllWidth, partitionType) bind(C, name='cusparseDcsc2hyb')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr, cusparseHybMat       
       type(cusparseHandle), value :: handle       
       integer(c_int), value :: m, n, userEllWidth, partitionType
       type(cusparseMatDescr), value :: descrA
       real(c_double), device :: cscValA(*)
       integer(c_int), device :: cscRowIndA(*), cscColPtrA(*)
       type(cusparseHybMat), value :: hybA
     end function cusparseDcsc2hyb
  end interface

! cusparseCcsc2hyb
  interface
     integer(c_int) function cusparseCcsc2hyb(handle, m, n, descrA, cscValA, cscRowIndA, cscColPtrA, &
          hybA, userEllWidth, partitionType) bind(C, name='cusparseCcsc2hyb')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr, cusparseHybMat       
       type(cusparseHandle), value :: handle       
       integer(c_int), value :: m, n, userEllWidth, partitionType
       type(cusparseMatDescr), value :: descrA
       complex(4), device :: cscValA(*)
       integer(c_int), device :: cscRowIndA(*), cscColPtrA(*)
       type(cusparseHybMat), value :: hybA
     end function cusparseCcsc2hyb
  end interface

! cusparseZcsc2hyb
  interface
     integer(c_int) function cusparseZcsc2hyb(handle, m, n, descrA, cscValA, cscRowIndA, cscColPtrA, &
          hybA, userEllWidth, partitionType) bind(C, name='cusparseZcsc2hyb')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr, cusparseHybMat       
       type(cusparseHandle), value :: handle       
       integer(c_int), value :: m, n, userEllWidth, partitionType
       type(cusparseMatDescr), value :: descrA
       complex(8), device :: cscValA(*)
       integer(c_int), device :: cscRowIndA(*), cscColPtrA(*)
       type(cusparseHybMat), value :: hybA
     end function cusparseZcsc2hyb
  end interface

! cusparseXcsr2bsrNnz
  interface cusparseXcsr2bsrNnz
    integer function cusparseXcsr2bsrNnz_setHPM(handle, dirA, m, n, descrA, csrRowPtrA, csrColIndA, &
       blockDim, descrC, bsrRowPtrC, nnzTotalDevHostPtr)
    import cusparseHandle, cusparseMatDescr
    type(cusparseHandle) :: handle       
    integer :: dirA, m, n, blockdim
    type(cusparseMatDescr) :: descrA, descrC
    integer, device :: csrRowPtrA(*), csrColIndA(*), bsrRowPtrC(*)
    integer :: nnzTotalDevHostPtr
    end function cusparseXcsr2bsrNnz_setHPM
    integer function cusparseXcsr2bsrNnz_setDPM(handle, dirA, m, n, descrA, csrRowPtrA, csrColIndA, &
       blockDim, descrC, bsrRowPtrC, nnzTotalDevHostPtr)
    import cusparseHandle, cusparseMatDescr
    type(cusparseHandle) :: handle       
    integer :: dirA, m, n, blockdim
    type(cusparseMatDescr) :: descrA, descrC
    integer, device :: csrRowPtrA(*), csrColIndA(*), bsrRowPtrC(*)
    integer, device :: nnzTotalDevHostPtr
    end function cusparseXcsr2bsrNnz_setDPM
  end interface cusparseXcsr2bsrNnz

#ifdef KEEP_V2_INTERFACES
  interface 
     integer(c_int) function cusparseXcsr2bsrNnz_v2(handle, dirA, m, n, descrA, csrRowPtrA, csrColIndA, &
          blockDim, descrC, bsrRowPtrC, nnzTotalDevHostPtr) bind(C, name='cusparseXcsr2bsrNnz')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr    
       type(cusparseHandle), value :: handle       
       integer(c_int), value :: dirA, m, n, blockdim
       type(cusparseMatDescr), value :: descrA, descrC
       integer(c_int), device :: csrRowPtrA(*), csrColIndA(*), bsrRowPtrC(*)
       !dir$ ignore_tkr (d) nnzTotalDevHostPtr
       integer(c_int) :: nnzTotalDevHostPtr
     end function cusparseXcsr2bsrNnz_v2
  end interface
#endif

! cusparseScsr2bsr
  interface
     integer(c_int) function cusparseScsr2bsr(handle, dirA, m, n, descrA, csrValA, csrRowPtrA, csrColIndA, &
          blockDim, descrC, bsrValC, bsrRowPtrC, bsrColIndC) bind(C, name='cusparseScsr2bsr')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr    
       type(cusparseHandle), value :: handle       
       integer(c_int), value :: dirA, m, n, blockdim
       type(cusparseMatDescr), value :: descrA, descrC
       real(c_float), device :: csrValA(*), bsrValC(*)
       integer(c_int), device :: csrRowPtrA(*), csrColIndA(*), bsrRowPtrC(*), bsrColIndC(*)
     end function cusparseScsr2bsr
  end interface

! cusparseDcsr2bsr
  interface
     integer(c_int) function cusparseDcsr2bsr(handle, dirA, m, n, descrA, csrValA, csrRowPtrA, csrColIndA, &
          blockDim, descrC, bsrValC, bsrRowPtrC, bsrColIndC) bind(C, name='cusparseDcsr2bsr')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr    
       type(cusparseHandle), value :: handle       
       integer(c_int), value :: dirA, m, n, blockdim
       type(cusparseMatDescr), value :: descrA, descrC
       real(c_double), device :: csrValA(*), bsrValC(*)
       integer(c_int), device :: csrRowPtrA(*), csrColIndA(*), bsrRowPtrC(*), bsrColIndC(*)
     end function cusparseDcsr2bsr
  end interface

! cusparseCcsr2bsr
  interface
     integer(c_int) function cusparseCcsr2bsr(handle, dirA, m, n, descrA, csrValA, csrRowPtrA, csrColIndA, &
          blockDim, descrC, bsrValC, bsrRowPtrC, bsrColIndC) bind(C, name='cusparseCcsr2bsr')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr    
       type(cusparseHandle), value :: handle       
       integer(c_int), value :: dirA, m, n, blockdim
       type(cusparseMatDescr), value :: descrA, descrC
       complex(4), device :: csrValA(*), bsrValC(*)
       integer(c_int), device :: csrRowPtrA(*), csrColIndA(*), bsrRowPtrC(*), bsrColIndC(*)
     end function cusparseCcsr2bsr
  end interface

! cusparseZcsr2bsr
  interface
     integer(c_int) function cusparseZcsr2bsr(handle, dirA, m, n, descrA, csrValA, csrRowPtrA, csrColIndA, &
          blockDim, descrC, bsrValC, bsrRowPtrC, bsrColIndC) bind(C, name='cusparseZcsr2bsr')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr    
       type(cusparseHandle), value :: handle       
       integer(c_int), value :: dirA, m, n, blockdim
       type(cusparseMatDescr), value :: descrA, descrC
       complex(8), device :: csrValA(*), bsrValC(*)
       integer(c_int), device :: csrRowPtrA(*), csrColIndA(*), bsrRowPtrC(*), bsrColIndC(*)
     end function cusparseZcsr2bsr
  end interface

! cusparseXcsr2coo
  interface
     integer(c_int) function cusparseXcsr2coo(handle, csrRowPtr, nnz, m, cooRowInd, idxBase) &
          bind(C, name='cusparseXcsr2coo')
       use iso_c_binding
       import cusparseHandle
       type(cusparseHandle), value :: handle       
       integer(c_int), value :: nnz, m, idxBase
       integer(c_int), device :: csrRowPtr(*), cooRowInd(*)
     end function cusparseXcsr2coo
  end interface

! cusparseScsr2csc
  interface
     integer(c_int) function cusparseScsr2csc(handle, m, n, nnz, csrVal, csrRowPtr, csrColInd, &
          cscVal, cscRowInd, cscColPtr, copyValues, idxBase) bind(C,name='cusparseScsr2csc')
       use iso_c_binding
       import cusparseHandle
       type(cusparseHandle), value :: handle       
       integer(c_int), value :: m, n, nnz, copyValues, idxBase
       real(c_float), device :: csrVal(*), cscVal(*)
       integer(c_int), device :: csrRowPtr(*), csrColInd(*), cscRowInd(*), cscColPtr(*)
     end function cusparseScsr2csc
  end interface

! cusparseDcsr2csc
  interface
     integer(c_int) function cusparseDcsr2csc(handle, m, n, nnz, csrVal, csrRowPtr, csrColInd, &
          cscVal, cscRowInd, cscColPtr, copyValues, idxBase) bind(C,name='cusparseDcsr2csc')
       use iso_c_binding
       import cusparseHandle
       type(cusparseHandle), value :: handle       
       integer(c_int), value :: m, n, nnz, copyValues, idxBase
       real(c_double), device :: csrVal(*), cscVal(*)
       integer(c_int), device :: csrRowPtr(*), csrColInd(*), cscRowInd(*), cscColPtr(*)
     end function cusparseDcsr2csc
  end interface

! cusparseCcsr2csc
  interface
     integer(c_int) function cusparseCcsr2csc(handle, m, n, nnz, csrVal, csrRowPtr, csrColInd, &
          cscVal, cscRowInd, cscColPtr, copyValues, idxBase) bind(C,name='cusparseCcsr2csc')
       use iso_c_binding
       import cusparseHandle
       type(cusparseHandle), value :: handle       
       integer(c_int), value :: m, n, nnz, copyValues, idxBase
       complex(4), device :: csrVal(*), cscVal(*)
       integer(c_int), device :: csrRowPtr(*), csrColInd(*), cscRowInd(*), cscColPtr(*)
     end function cusparseCcsr2csc
  end interface

! cusparseZcsr2csc
  interface
     integer(c_int) function cusparseZcsr2csc(handle, m, n, nnz, csrVal, csrRowPtr, csrColInd, &
          cscVal, cscRowInd, cscColPtr, copyValues, idxBase) bind(C,name='cusparseZcsr2csc')
       use iso_c_binding
       import cusparseHandle
       type(cusparseHandle), value :: handle       
       integer(c_int), value :: m, n, nnz, copyValues, idxBase
       complex(8), device :: csrVal(*), cscVal(*)
       integer(c_int), device :: csrRowPtr(*), csrColInd(*), cscRowInd(*), cscColPtr(*)
     end function cusparseZcsr2csc
  end interface

! cusparseCsr2cscEx2
  interface
     integer(c_int) function cusparseCsr2cscEx2_bufferSize(handle, m, n, nnz, csrVal, csrRowPtr, csrColInd, &
          cscVal, cscColPtr, cscRowInd, valType, copyValues, idxBase, alg, bufferSize) &
          bind(C,name='cusparseCsr2cscEx2_bufferSize')
       use iso_c_binding
       import cusparseHandle
       type(cusparseHandle), value :: handle       
       integer(c_int), value :: m, n, nnz, valType, copyValues, idxBase, alg
       !dir$ ignore_tkr csrVal, cscVal
       real(4), device :: csrVal(*), cscVal(*)
       integer(c_int), device :: csrRowPtr(*), csrColInd(*), cscRowInd(*), cscColPtr(*)
       integer(8) :: bufferSize
     end function cusparseCsr2cscEx2_bufferSize
  end interface

  interface
     integer(c_int) function cusparseCsr2cscEx2(handle, m, n, nnz, csrVal, csrRowPtr, csrColInd, &
          cscVal, cscColPtr, cscRowInd, valType, copyValues, idxBase, alg, buffer) &
          bind(C,name='cusparseCsr2cscEx2')
       use iso_c_binding
       import cusparseHandle
       type(cusparseHandle), value :: handle       
       integer(c_int), value :: m, n, nnz, valType, copyValues, idxBase, alg
       !dir$ ignore_tkr csrVal, cscVal
       real(4), device :: csrVal(*), cscVal(*)
       integer(c_int), device :: csrRowPtr(*), csrColInd(*), cscRowInd(*), cscColPtr(*)
       !dir$ ignore_tkr buffer
       character, device :: buffer(*)
     end function cusparseCsr2cscEx2
  end interface

! cusparseScsr2dense
  interface
     integer(c_int) function cusparseScsr2dense(handle, m, n, descrA, csrValA, csrRowPtrA, csrColIndA, A, lda) &
          bind(C,name='cusparseScsr2dense')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr
       type(cusparseHandle), value :: handle       
       integer(c_int), value :: m, n, lda
       type(cusparseMatDescr), value :: descrA
       !dir$ ignore_tkr(r) A
       real(c_float), device :: csrValA(*), A(*)
       integer(c_int), device :: csrRowPtrA(*), csrColIndA(*)
     end function cusparseScsr2dense
  end interface

! cusparseDcsr2dense
  interface
     integer(c_int) function cusparseDcsr2dense(handle, m, n, descrA, csrValA, csrRowPtrA, csrColIndA, A, lda) &
          bind(C,name='cusparseDcsr2dense')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr
       type(cusparseHandle), value :: handle       
       integer(c_int), value :: m, n, lda
       type(cusparseMatDescr), value :: descrA
       !dir$ ignore_tkr(r) A
       real(c_double), device :: csrValA(*), A(*)
       integer(c_int), device :: csrRowPtrA(*), csrColIndA(*)
     end function cusparseDcsr2dense
  end interface

! cusparseCcsr2dense
  interface
     integer(c_int) function cusparseCcsr2dense(handle, m, n, descrA, csrValA, csrRowPtrA, csrColIndA, A, lda) &
          bind(C,name='cusparseCcsr2dense')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr
       type(cusparseHandle), value :: handle       
       integer(c_int), value :: m, n, lda
       type(cusparseMatDescr), value :: descrA
       !dir$ ignore_tkr(r) A
       complex(4), device :: csrValA(*), A(*)
       integer(c_int), device :: csrRowPtrA(*), csrColIndA(*)
     end function cusparseCcsr2dense
  end interface

! cusparseZcsr2dense
  interface
     integer(c_int) function cusparseZcsr2dense(handle, m, n, descrA, csrValA, csrRowPtrA, csrColIndA, A, lda) &
          bind(C,name='cusparseZcsr2dense')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr
       type(cusparseHandle), value :: handle       
       integer(c_int), value :: m, n, lda
       type(cusparseMatDescr), value :: descrA
       !dir$ ignore_tkr(r) A
       complex(8), device :: csrValA(*), A(*)
       integer(c_int), device :: csrRowPtrA(*), csrColIndA(*)
     end function cusparseZcsr2dense
  end interface

! cusparseScsr2hyb
  interface
     integer(c_int) function cusparseScsr2hyb(handle, m, n, descrA, csrValA, csrRowPtrA, csrColIndA, &
          hybA, userEllWidth, partitionType) bind(C,name='cusparseScsr2hyb')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr, cusparseHybMat
       type(cusparseHandle), value :: handle       
       integer(c_int), value :: m, n, userEllWidth, partitionType
       type(cusparseMatDescr), value :: descrA
       type(cusparseHybMat), value :: hybA
       real(c_float), device :: csrValA(*)
       integer(c_int), device :: csrRowPtrA(*), csrColIndA(*)
     end function cusparseScsr2hyb
  end interface

! cusparseDcsr2hyb
  interface
     integer(c_int) function cusparseDcsr2hyb(handle, m, n, descrA, csrValA, csrRowPtrA, csrColIndA, &
          hybA, userEllWidth, partitionType) bind(C,name='cusparseDcsr2hyb')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr, cusparseHybMat
       type(cusparseHandle), value :: handle       
       integer(c_int), value :: m, n, userEllWidth, partitionType
       type(cusparseMatDescr), value :: descrA
       type(cusparseHybMat), value :: hybA
       real(c_double), device :: csrValA(*)
       integer(c_int), device :: csrRowPtrA(*), csrColIndA(*)
     end function cusparseDcsr2hyb
  end interface

! cusparseCcsr2hyb
  interface
     integer(c_int) function cusparseCcsr2hyb(handle, m, n, descrA, csrValA, csrRowPtrA, csrColIndA, &
          hybA, userEllWidth, partitionType) bind(C,name='cusparseCcsr2hyb')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr, cusparseHybMat
       type(cusparseHandle), value :: handle       
       integer(c_int), value :: m, n, userEllWidth, partitionType
       type(cusparseMatDescr), value :: descrA
       type(cusparseHybMat), value :: hybA
       complex(4), device :: csrValA(*)
       integer(c_int), device :: csrRowPtrA(*), csrColIndA(*)
     end function cusparseCcsr2hyb
  end interface

! cusparseZcsr2hyb
  interface
     integer(c_int) function cusparseZcsr2hyb(handle, m, n, descrA, csrValA, csrRowPtrA, csrColIndA, &
          hybA, userEllWidth, partitionType) bind(C,name='cusparseZcsr2hyb')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr, cusparseHybMat
       type(cusparseHandle), value :: handle       
       integer(c_int), value :: m, n, userEllWidth, partitionType
       type(cusparseMatDescr), value :: descrA
       type(cusparseHybMat), value :: hybA
       complex(8), device :: csrValA(*)
       integer(c_int), device :: csrRowPtrA(*), csrColIndA(*)
     end function cusparseZcsr2hyb
  end interface

! cusparseSdense2csc
  interface
     integer(c_int) function cusparseSdense2csc(handle, m, n, descrA, A, lda, nnzPerCol, &
          cscValA, cscRowIndA, cscColPtrA) bind(C, name='cusparseSdense2csc')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr
       type(cusparseHandle), value :: handle       
       integer(c_int), value :: m, n, lda
       type(cusparseMatDescr), value :: descrA
       !dir$ ignore_tkr(r) A
       real(c_float), device :: A(*), cscValA(*)
       integer(c_int), device :: nnzPerCol(*), cscRowIndA(*), cscColPtrA(*)
     end function cusparseSdense2csc
  end interface

! cusparseDdense2csc
  interface
     integer(c_int) function cusparseDdense2csc(handle, m, n, descrA, A, lda, nnzPerCol, &
          cscValA, cscRowIndA, cscColPtrA) bind(C, name='cusparseDdense2csc')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr
       type(cusparseHandle), value :: handle       
       integer(c_int), value :: m, n, lda
       type(cusparseMatDescr), value :: descrA
       !dir$ ignore_tkr(r) A
       real(c_double), device :: A(*), cscValA(*)
       integer(c_int), device :: nnzPerCol(*), cscRowIndA(*), cscColPtrA(*)
     end function cusparseDdense2csc
  end interface

! cusparseCdense2csc
  interface
     integer(c_int) function cusparseCdense2csc(handle, m, n, descrA, A, lda, nnzPerCol, &
          cscValA, cscRowIndA, cscColPtrA) bind(C, name='cusparseCdense2csc')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr
       type(cusparseHandle), value :: handle       
       integer(c_int), value :: m, n, lda
       type(cusparseMatDescr), value :: descrA
       !dir$ ignore_tkr(r) A
       complex(4), device :: A(*), cscValA(*)
       integer(c_int), device :: nnzPerCol(*), cscRowIndA(*), cscColPtrA(*)
     end function cusparseCdense2csc
  end interface

! cusparseZdense2csc
  interface
     integer(c_int) function cusparseZdense2csc(handle, m, n, descrA, A, lda, nnzPerCol, &
          cscValA, cscRowIndA, cscColPtrA) bind(C, name='cusparseZdense2csc')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr
       type(cusparseHandle), value :: handle       
       integer(c_int), value :: m, n, lda
       type(cusparseMatDescr), value :: descrA
       !dir$ ignore_tkr(r) A
       complex(8), device :: A(*), cscValA(*)
       integer(c_int), device :: nnzPerCol(*), cscRowIndA(*), cscColPtrA(*)
     end function cusparseZdense2csc
  end interface

! cusparseSdense2csr
  interface
     integer(c_int) function cusparseSdense2csr(handle, m, n, descrA, A, lda, nnzPerRow, &
          csrValA, csrRowPtrA, csrColIndA) bind(C, name='cusparseSdense2csr')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr
       type(cusparseHandle), value :: handle       
       integer(c_int), value :: m, n, lda
       type(cusparseMatDescr), value :: descrA
       !dir$ ignore_tkr(r) A
       real(c_float), device :: A(*), csrValA(*)
       integer(c_int), device :: nnzPerRow(*), csrRowPtrA(*), csrColIndA(*)
     end function cusparseSdense2csr
  end interface

! cusparseDdense2csr
  interface
     integer(c_int) function cusparseDdense2csr(handle, m, n, descrA, A, lda, nnzPerRow, &
          csrValA, csrRowPtrA, csrColIndA) bind(C, name='cusparseDdense2csr')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr
       type(cusparseHandle), value :: handle       
       integer(c_int), value :: m, n, lda
       type(cusparseMatDescr), value :: descrA
       !dir$ ignore_tkr(r) A
       real(c_double), device :: A(*), csrValA(*)
       integer(c_int), device :: nnzPerRow(*), csrRowPtrA(*), csrColIndA(*)
     end function cusparseDdense2csr
  end interface

! cusparseCdense2csr
  interface
     integer(c_int) function cusparseCdense2csr(handle, m, n, descrA, A, lda, nnzPerRow, &
          csrValA, csrRowPtrA, csrColIndA) bind(C, name='cusparseCdense2csr')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr
       type(cusparseHandle), value :: handle       
       integer(c_int), value :: m, n, lda
       type(cusparseMatDescr), value :: descrA
       !dir$ ignore_tkr(r) A
       complex(4), device :: A(*), csrValA(*)
       integer(c_int), device :: nnzPerRow(*), csrRowPtrA(*), csrColIndA(*)
     end function cusparseCdense2csr
  end interface

! cusparseZdense2csr
  interface
     integer(c_int) function cusparseZdense2csr(handle, m, n, descrA, A, lda, nnzPerRow, &
          csrValA, csrRowPtrA, csrColIndA) bind(C, name='cusparseZdense2csr')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr
       type(cusparseHandle), value :: handle       
       integer(c_int), value :: m, n, lda
       type(cusparseMatDescr), value :: descrA
       !dir$ ignore_tkr(r) A
       complex(8), device :: A(*), csrValA(*)
       integer(c_int), device :: nnzPerRow(*), csrRowPtrA(*), csrColIndA(*)
     end function cusparseZdense2csr
  end interface

! cusparseSdense2hyb
  interface
     integer(c_int) function cusparseSdense2hyb(handle, m, n, descrA, A, lda, &
          nnzPerRow, hybA, userEllWidth, partitionType) bind(C, name='cusparseSdense2hyb')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr, cusparseHybMat
       type(cusparseHandle), value :: handle       
       integer(c_int), value :: m, n, lda, userEllWidth, partitionType
       type(cusparseMatDescr), value :: descrA
       type(cusparseHybMat), value :: hybA
       !dir$ ignore_tkr(r) A
       real(c_float), device :: A(*)
       integer(c_int), device :: nnzPerRow(*)
     end function cusparseSdense2hyb
  end interface

! cusparseDdense2hyb
  interface
     integer(c_int) function cusparseDdense2hyb(handle, m, n, descrA, A, lda, &
          nnzPerRow, hybA, userEllWidth, partitionType) bind(C, name='cusparseDdense2hyb')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr, cusparseHybMat
       type(cusparseHandle), value :: handle       
       integer(c_int), value :: m, n, lda, userEllWidth, partitionType
       type(cusparseMatDescr), value :: descrA
       type(cusparseHybMat), value :: hybA
       !dir$ ignore_tkr(r) A
       real(c_double), device :: A(*)
       integer(c_int), device :: nnzPerRow(*)
     end function cusparseDdense2hyb
  end interface

! cusparseCdense2hyb
  interface
     integer(c_int) function cusparseCdense2hyb(handle, m, n, descrA, A, lda, &
          nnzPerRow, hybA, userEllWidth, partitionType) bind(C, name='cusparseCdense2hyb')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr, cusparseHybMat
       type(cusparseHandle), value :: handle       
       integer(c_int), value :: m, n, lda, userEllWidth, partitionType
       type(cusparseMatDescr), value :: descrA
       type(cusparseHybMat), value :: hybA
       !dir$ ignore_tkr(r) A
       complex(4), device :: A(*)
       integer(c_int), device :: nnzPerRow(*)
     end function cusparseCdense2hyb
  end interface

! cusparseZdense2hyb
  interface
     integer(c_int) function cusparseZdense2hyb(handle, m, n, descrA, A, lda, &
          nnzPerRow, hybA, userEllWidth, partitionType) bind(C, name='cusparseZdense2hyb')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr, cusparseHybMat
       type(cusparseHandle), value :: handle       
       integer(c_int), value :: m, n, lda, userEllWidth, partitionType
       type(cusparseMatDescr), value :: descrA
       type(cusparseHybMat), value :: hybA
       !dir$ ignore_tkr(r) A
       complex(8), device :: A(*)
       integer(c_int), device :: nnzPerRow(*)
     end function cusparseZdense2hyb
  end interface

! cusparseShyb2csc
  interface
     integer(c_int) function cusparseShyb2csc(handle, descrA, hybA, cscValA, cscRowIndA, cscColPtrA) &
          bind(C, name='cusparseShyb2csc')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr, cusparseHybMat
       type(cusparseHandle), value :: handle       
       type(cusparseMatDescr), value :: descrA
       type(cusparseHybMat), value :: hybA
       real(c_float), device :: cscValA(*)
       integer(c_int), device :: cscRowIndA(*), cscColPtrA(*)
     end function cusparseShyb2csc
  end interface

! cusparseDhyb2csc
  interface
     integer(c_int) function cusparseDhyb2csc(handle, descrA, hybA, cscValA, cscRowIndA, cscColPtrA) &
          bind(C, name='cusparseDhyb2csc')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr, cusparseHybMat
       type(cusparseHandle), value :: handle       
       type(cusparseMatDescr), value :: descrA
       type(cusparseHybMat), value :: hybA
       real(c_double), device :: cscValA(*)
       integer(c_int), device :: cscRowIndA(*), cscColPtrA(*)
     end function cusparseDhyb2csc
  end interface

! cusparseChyb2csc
  interface
     integer(c_int) function cusparseChyb2csc(handle, descrA, hybA, cscValA, cscRowIndA, cscColPtrA) &
          bind(C, name='cusparseChyb2csc')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr, cusparseHybMat
       type(cusparseHandle), value :: handle       
       type(cusparseMatDescr), value :: descrA
       type(cusparseHybMat), value :: hybA
       complex(4), device :: cscValA(*)
       integer(c_int), device :: cscRowIndA(*), cscColPtrA(*)
     end function cusparseChyb2csc
  end interface

! cusparseZhyb2csc
  interface
     integer(c_int) function cusparseZhyb2csc(handle, descrA, hybA, cscValA, cscRowIndA, cscColPtrA) &
          bind(C, name='cusparseZhyb2csc')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr, cusparseHybMat
       type(cusparseHandle), value :: handle       
       type(cusparseMatDescr), value :: descrA
       type(cusparseHybMat), value :: hybA
       complex(8), device :: cscValA(*)
       integer(c_int), device :: cscRowIndA(*), cscColPtrA(*)
     end function cusparseZhyb2csc
  end interface

! cusparseShyb2csr
  interface
     integer(c_int) function cusparseShyb2csr(handle, descrA, hybA, csrValA, csrRowPtrA, csrColIndA) &
          bind(C, name='cusparseShyb2csr')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr, cusparseHybMat
       type(cusparseHandle), value :: handle       
       type(cusparseMatDescr), value :: descrA
       type(cusparseHybMat), value :: hybA
       real(c_float), device :: csrValA(*)
       integer(c_int), device :: csrRowPtrA(*), csrColIndA(*)
     end function cusparseShyb2csr
  end interface

! cusparseDhyb2csr
  interface
     integer(c_int) function cusparseDhyb2csr(handle, descrA, hybA, csrValA, csrRowPtrA, csrColIndA) &
          bind(C, name='cusparseDhyb2csr')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr, cusparseHybMat
       type(cusparseHandle), value :: handle       
       type(cusparseMatDescr), value :: descrA
       type(cusparseHybMat), value :: hybA
       real(c_double), device :: csrValA(*)
       integer(c_int), device :: csrRowPtrA(*), csrColIndA(*)
     end function cusparseDhyb2csr
  end interface

! cusparseChyb2csr
  interface
     integer(c_int) function cusparseChyb2csr(handle, descrA, hybA, csrValA, csrRowPtrA, csrColIndA) &
          bind(C, name='cusparseChyb2csr')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr, cusparseHybMat
       type(cusparseHandle), value :: handle       
       type(cusparseMatDescr), value :: descrA
       type(cusparseHybMat), value :: hybA
       complex(4), device :: csrValA(*)
       integer(c_int), device :: csrRowPtrA(*), csrColIndA(*)
     end function cusparseChyb2csr
  end interface

! cusparseZhyb2csr
  interface
     integer(c_int) function cusparseZhyb2csr(handle, descrA, hybA, csrValA, csrRowPtrA, csrColIndA) &
          bind(C, name='cusparseZhyb2csr')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr, cusparseHybMat
       type(cusparseHandle), value :: handle       
       type(cusparseMatDescr), value :: descrA
       type(cusparseHybMat), value :: hybA
       complex(8), device :: csrValA(*)
       integer(c_int), device :: csrRowPtrA(*), csrColIndA(*)
     end function cusparseZhyb2csr
  end interface

! cusparseShyb2dense
  interface
     integer(c_int) function cusparseShyb2dense(handle, descrA, hybA, A, lda) &
          bind(C, name='cusparseShyb2dense')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr, cusparseHybMat
       type(cusparseHandle), value :: handle       
       type(cusparseMatDescr), value :: descrA
       type(cusparseHybMat), value :: hybA
       !dir$ ignore_tkr(r) A
       real(c_float), device :: A(*)
       integer(c_int), value :: lda
     end function cusparseShyb2dense
  end interface

! cusparseDhyb2dense
  interface
     integer(c_int) function cusparseDhyb2dense(handle, descrA, hybA, A, lda) &
          bind(C, name='cusparseDhyb2dense')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr, cusparseHybMat
       type(cusparseHandle), value :: handle       
       type(cusparseMatDescr), value :: descrA
       type(cusparseHybMat), value :: hybA
       !dir$ ignore_tkr(r) A
       real(c_double), device :: A(*)
       integer(c_int), value :: lda
     end function cusparseDhyb2dense
  end interface

! cusparseChyb2dense
  interface
     integer(c_int) function cusparseChyb2dense(handle, descrA, hybA, A, lda) &
          bind(C, name='cusparseChyb2dense')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr, cusparseHybMat
       type(cusparseHandle), value :: handle       
       type(cusparseMatDescr), value :: descrA
       type(cusparseHybMat), value :: hybA
       !dir$ ignore_tkr(r) A
       complex(4), device :: A(*)
       integer(c_int), value :: lda
     end function cusparseChyb2dense
  end interface

! cusparseZhyb2dense
  interface
     integer(c_int) function cusparseZhyb2dense(handle, descrA, hybA, A, lda) &
          bind(C, name='cusparseZhyb2dense')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr, cusparseHybMat
       type(cusparseHandle), value :: handle       
       type(cusparseMatDescr), value :: descrA
       type(cusparseHybMat), value :: hybA
       !dir$ ignore_tkr(r) A
       complex(8), device :: A(*)
       integer(c_int), value :: lda
     end function cusparseZhyb2dense
  end interface

! cusparseSnnz
  interface cusparseSnnz
    integer function cusparseSnnz_setHPM(handle, dirA, m, n, descrA, A, lda, &
       nnzPerRowColumn, nnzTotalDevHostPtr)
    import cusparseHandle, cusparseMatDescr
    type(cusparseHandle) :: handle       
    integer :: dirA, m, n, lda
    type(cusparseMatDescr) :: descrA
    !dir$ ignore_tkr(r) A
    real, device :: A(*)
    integer, device :: nnzPerRowColumn(*)
    integer :: nnzTotalDevHostPtr
    end function cusparseSnnz_setHPM
    integer function cusparseSnnz_setDPM(handle, dirA, m, n, descrA, A, lda, &
       nnzPerRowColumn, nnzTotalDevHostPtr)
    import cusparseHandle, cusparseMatDescr
    type(cusparseHandle) :: handle       
    integer :: dirA, m, n, lda
    type(cusparseMatDescr) :: descrA
    !dir$ ignore_tkr(r) A
    real, device :: A(*)
    integer, device :: nnzPerRowColumn(*)
    integer, device :: nnzTotalDevHostPtr
    end function cusparseSnnz_setDPM
  end interface cusparseSnnz

#ifdef KEEP_V2_INTERFACES
  interface
     integer(c_int) function cusparseSnnz_v2(handle, dirA, m, n, descrA, A, lda, &
          nnzPerRowColumn, nnzTotalDevHostPtr) bind(C, name='cusparseSnnz')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr
       type(cusparseHandle), value :: handle       
       integer(c_int), value :: dirA, m, n, lda
       type(cusparseMatDescr), value :: descrA
       !dir$ ignore_tkr(r) A
       real(c_float), device :: A(*)
       integer(c_int), device :: nnzPerRowColumn(*)
       !dir$ ignore_tkr (d) nnzTotalDevHostPtr
       integer(c_int) :: nnzTotalDevHostPtr
     end function cusparseSnnz_v2
  end interface
#endif

! cusparseDnnz
  interface cusparseDnnz
    integer function cusparseDnnz_setHPM(handle, dirA, m, n, descrA, A, lda, &
       nnzPerRowColumn, nnzTotalDevHostPtr)
    import cusparseHandle, cusparseMatDescr
    type(cusparseHandle) :: handle       
    integer :: dirA, m, n, lda
    type(cusparseMatDescr) :: descrA
    !dir$ ignore_tkr(r) A
    real(8), device :: A(*)
    integer, device :: nnzPerRowColumn(*)
    integer :: nnzTotalDevHostPtr
    end function cusparseDnnz_setHPM
    integer function cusparseDnnz_setDPM(handle, dirA, m, n, descrA, A, lda, &
       nnzPerRowColumn, nnzTotalDevHostPtr)
    import cusparseHandle, cusparseMatDescr
    type(cusparseHandle) :: handle       
    integer :: dirA, m, n, lda
    type(cusparseMatDescr) :: descrA
    !dir$ ignore_tkr(r) A
    real(8), device :: A(*)
    integer, device :: nnzPerRowColumn(*)
    integer, device :: nnzTotalDevHostPtr
    end function cusparseDnnz_setDPM
  end interface cusparseDnnz

#ifdef KEEP_V2_INTERFACES
  interface
     integer(c_int) function cusparseDnnz_v2(handle, dirA, m, n, descrA, A, lda, &
          nnzPerRowColumn, nnzTotalDevHostPtr) bind(C, name='cusparseDnnz')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr
       type(cusparseHandle), value :: handle       
       integer(c_int), value :: dirA, m, n, lda
       type(cusparseMatDescr), value :: descrA
       !dir$ ignore_tkr(r) A
       real(c_double), device :: A(*)
       integer(c_int), device :: nnzPerRowColumn(*)
       !dir$ ignore_tkr (d) nnzTotalDevHostPtr
       integer(c_int) :: nnzTotalDevHostPtr
     end function cusparseDnnz_v2
  end interface
#endif

! cusparseCnnz
  interface cusparseCnnz
    integer function cusparseCnnz_setHPM(handle, dirA, m, n, descrA, A, lda, &
       nnzPerRowColumn, nnzTotalDevHostPtr)
    import cusparseHandle, cusparseMatDescr
    type(cusparseHandle) :: handle       
    integer :: dirA, m, n, lda
    type(cusparseMatDescr) :: descrA
    !dir$ ignore_tkr(r) A
    complex(4), device :: A(*)
    integer, device :: nnzPerRowColumn(*)
    integer :: nnzTotalDevHostPtr
    end function cusparseCnnz_setHPM
    integer function cusparseCnnz_setDPM(handle, dirA, m, n, descrA, A, lda, &
       nnzPerRowColumn, nnzTotalDevHostPtr)
    import cusparseHandle, cusparseMatDescr
    type(cusparseHandle) :: handle       
    integer :: dirA, m, n, lda
    type(cusparseMatDescr) :: descrA
    !dir$ ignore_tkr(r) A
    complex(4), device :: A(*)
    integer, device :: nnzPerRowColumn(*)
    integer, device :: nnzTotalDevHostPtr
    end function cusparseCnnz_setDPM
  end interface cusparseCnnz

#ifdef KEEP_V2_INTERFACES
  interface
     integer(c_int) function cusparseCnnz_v2(handle, dirA, m, n, descrA, A, lda, &
          nnzPerRowColumn, nnzTotalDevHostPtr) bind(C, name='cusparseCnnz')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr
       type(cusparseHandle), value :: handle       
       integer(c_int), value :: dirA, m, n, lda
       type(cusparseMatDescr), value :: descrA
       !dir$ ignore_tkr(r) A
       complex(4), device :: A(*)
       integer(c_int), device :: nnzPerRowColumn(*)
       !dir$ ignore_tkr (d) nnzTotalDevHostPtr
       integer(c_int) :: nnzTotalDevHostPtr
     end function cusparseCnnz_v2
  end interface
#endif

! cusparseZnnz
  interface cusparseZnnz
    integer function cusparseZnnz_setHPM(handle, dirA, m, n, descrA, A, lda, &
       nnzPerRowColumn, nnzTotalDevHostPtr)
    import cusparseHandle, cusparseMatDescr
    type(cusparseHandle) :: handle       
    integer :: dirA, m, n, lda
    type(cusparseMatDescr) :: descrA
    !dir$ ignore_tkr(r) A
    complex(8), device :: A(*)
    integer, device :: nnzPerRowColumn(*)
    integer :: nnzTotalDevHostPtr
    end function cusparseZnnz_setHPM
    integer function cusparseZnnz_setDPM(handle, dirA, m, n, descrA, A, lda, &
       nnzPerRowColumn, nnzTotalDevHostPtr)
    import cusparseHandle, cusparseMatDescr
    type(cusparseHandle) :: handle       
    integer :: dirA, m, n, lda
    type(cusparseMatDescr) :: descrA
    !dir$ ignore_tkr(r) A
    complex(8), device :: A(*)
    integer, device :: nnzPerRowColumn(*)
    integer, device :: nnzTotalDevHostPtr
    end function cusparseZnnz_setDPM
  end interface cusparseZnnz

#ifdef KEEP_V2_INTERFACES
  interface
     integer(c_int) function cusparseZnnz_v2(handle, dirA, m, n, descrA, A, lda, &
          nnzPerRowColumn, nnzTotalDevHostPtr) bind(C, name='cusparseZnnz')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr
       type(cusparseHandle), value :: handle       
       integer(c_int), value :: dirA, m, n, lda
       type(cusparseMatDescr), value :: descrA
       !dir$ ignore_tkr(r) A
       complex(8), device :: A(*)
       integer(c_int), device :: nnzPerRowColumn(*)
       !dir$ ignore_tkr (d) nnzTotalDevHostPtr
       integer(c_int) :: nnzTotalDevHostPtr
     end function cusparseZnnz_v2
  end interface
#endif

  ! ----------------------
  ! 6.0 format conversions
  ! ----------------------

! cusparseSgebsr2gebsc_bufferSize
  interface cusparseSgebsr2gebsc_bufferSize
     integer(c_int) function cusparseSgebsr2gebsc_bufferSize(handle, mb, nb, nnzb, bsrVal, bsrRowPtr, bsrColInd, &
          rowBlockDim, colBlockDim, pBufferSize) bind(C, name='cusparseSgebsr2gebsc_bufferSize')
       use iso_c_binding
       import cusparseHandle
       type(cusparseHandle), value :: handle       
       integer(c_int), value :: mb, nb, nnzb
       real(c_float), device :: bsrVal(*)
       integer(c_int), device :: bsrRowPtr(*), bsrColInd(*)
       integer(c_int), value :: rowBlockDim, colBlockDim
       integer(4) :: pBufferSize
     end function cusparseSgebsr2gebsc_bufferSize
     integer(c_int) function cusparseSgebsr2gebsc_bufferSizeExt(handle, mb, nb, nnzb, bsrVal, bsrRowPtr, bsrColInd, &
          rowBlockDim, colBlockDim, pBufferSize) bind(C, name='cusparseSgebsr2gebsc_bufferSizeExt')
       use iso_c_binding
       import cusparseHandle
       type(cusparseHandle), value :: handle       
       integer(c_int), value :: mb, nb, nnzb
       real(c_float), device :: bsrVal(*)
       integer(c_int), device :: bsrRowPtr(*), bsrColInd(*)
       integer(c_int), value :: rowBlockDim, colBlockDim
       integer(8) :: pBufferSize
     end function cusparseSgebsr2gebsc_bufferSizeExt
  end interface

! cusparseDgebsr2gebsc_bufferSize
  interface cusparseDgebsr2gebsc_bufferSize
     integer(c_int) function cusparseDgebsr2gebsc_bufferSize(handle, mb, nb, nnzb, bsrVal, bsrRowPtr, bsrColInd, &
          rowBlockDim, colBlockDim, pBufferSize) bind(C, name='cusparseDgebsr2gebsc_bufferSize')
       use iso_c_binding
       import cusparseHandle
       type(cusparseHandle), value :: handle       
       integer(c_int), value :: mb, nb, nnzb
       real(c_double), device :: bsrVal(*)
       integer(c_int), device :: bsrRowPtr(*), bsrColInd(*)
       integer(c_int), value :: rowBlockDim, colBlockDim
       integer(4) :: pBufferSize
     end function cusparseDgebsr2gebsc_bufferSize
     integer(c_int) function cusparseDgebsr2gebsc_bufferSizeExt(handle, mb, nb, nnzb, bsrVal, bsrRowPtr, bsrColInd, &
          rowBlockDim, colBlockDim, pBufferSize) bind(C, name='cusparseDgebsr2gebsc_bufferSizeExt')
       use iso_c_binding
       import cusparseHandle
       type(cusparseHandle), value :: handle       
       integer(c_int), value :: mb, nb, nnzb
       real(c_double), device :: bsrVal(*)
       integer(c_int), device :: bsrRowPtr(*), bsrColInd(*)
       integer(c_int), value :: rowBlockDim, colBlockDim
       integer(8) :: pBufferSize
     end function cusparseDgebsr2gebsc_bufferSizeExt
  end interface

! cusparseCgebsr2gebsc_bufferSize
  interface cusparseCgebsr2gebsc_bufferSize
     integer(c_int) function cusparseCgebsr2gebsc_bufferSize(handle, mb, nb, nnzb, bsrVal, bsrRowPtr, bsrColInd, &
          rowBlockDim, colBlockDim, pBufferSize) bind(C, name='cusparseCgebsr2gebsc_bufferSize')
       use iso_c_binding
       import cusparseHandle
       type(cusparseHandle), value :: handle       
       integer(c_int), value :: mb, nb, nnzb
       complex(4), device :: bsrVal(*)
       integer(c_int), device :: bsrRowPtr(*), bsrColInd(*)
       integer(c_int), value :: rowBlockDim, colBlockDim
       integer(4) :: pBufferSize
     end function cusparseCgebsr2gebsc_bufferSize
     integer(c_int) function cusparseCgebsr2gebsc_bufferSizeExt(handle, mb, nb, nnzb, bsrVal, bsrRowPtr, bsrColInd, &
          rowBlockDim, colBlockDim, pBufferSize) bind(C, name='cusparseCgebsr2gebsc_bufferSizeExt')
       use iso_c_binding
       import cusparseHandle
       type(cusparseHandle), value :: handle       
       integer(c_int), value :: mb, nb, nnzb
       complex(4), device :: bsrVal(*)
       integer(c_int), device :: bsrRowPtr(*), bsrColInd(*)
       integer(c_int), value :: rowBlockDim, colBlockDim
       integer(8) :: pBufferSize
     end function cusparseCgebsr2gebsc_bufferSizeExt
  end interface

! cusparseZgebsr2gebsc_bufferSize
  interface cusparseZgebsr2gebsc_bufferSize
     integer(c_int) function cusparseZgebsr2gebsc_bufferSize(handle, mb, nb, nnzb, bsrVal, bsrRowPtr, bsrColInd, &
          rowBlockDim, colBlockDim, pBufferSize) bind(C, name='cusparseZgebsr2gebsc_bufferSize')
       use iso_c_binding
       import cusparseHandle
       type(cusparseHandle), value :: handle       
       integer(c_int), value :: mb, nb, nnzb
       complex(8), device :: bsrVal(*)
       integer(c_int), device :: bsrRowPtr(*), bsrColInd(*)
       integer(c_int), value :: rowBlockDim, colBlockDim
       integer(4) :: pBufferSize
     end function cusparseZgebsr2gebsc_bufferSize
     integer(c_int) function cusparseZgebsr2gebsc_bufferSizeExt(handle, mb, nb, nnzb, bsrVal, bsrRowPtr, bsrColInd, &
          rowBlockDim, colBlockDim, pBufferSize) bind(C, name='cusparseZgebsr2gebsc_bufferSizeExt')
       use iso_c_binding
       import cusparseHandle
       type(cusparseHandle), value :: handle       
       integer(c_int), value :: mb, nb, nnzb
       complex(8), device :: bsrVal(*)
       integer(c_int), device :: bsrRowPtr(*), bsrColInd(*)
       integer(c_int), value :: rowBlockDim, colBlockDim
       integer(8) :: pBufferSize
     end function cusparseZgebsr2gebsc_bufferSizeExt
  end interface

! cusparseSgebsr2gebsc
  interface
     integer(c_int) function cusparseSgebsr2gebsc(handle, mb, nb, nnzb, bsrVal, bsrRowPtr, bsrColInd, rowBlockDim, colBlockDim, &
          bscVal, bscRowInd, bscColPtr, copyValues, baseIdx, pBuffer) bind(C,name='cusparseSgebsr2gebsc')
       use iso_c_binding
       import cusparseHandle
       type(cusparseHandle), value :: handle       
       integer(c_int), value :: mb, nb, nnzb
       real(c_float), device :: bsrVal(*)
       integer(c_int), device :: bsrRowPtr(*), bsrColInd(*)
       integer(c_int), value :: rowBlockDim, colBlockDim
       real(c_float), device :: bscVal(*)
       integer(c_int), device :: bscRowInd(*), bscColPtr(*)
       integer(c_int), value :: copyValues, baseIdx
       !dir$ ignore_tkr pBuffer
       character(c_char), device :: pBuffer(*)
     end function cusparseSgebsr2gebsc
  end interface

! cusparseDgebsr2gebsc
  interface
     integer(c_int) function cusparseDgebsr2gebsc(handle, mb, nb, nnzb, bsrVal, bsrRowPtr, bsrColInd, rowBlockDim, colBlockDim, &
          bscVal, bscRowInd, bscColPtr, copyValues, baseIdx, pBuffer) bind(C,name='cusparseDgebsr2gebsc')
       use iso_c_binding
       import cusparseHandle
       type(cusparseHandle), value :: handle       
       integer(c_int), value :: mb, nb, nnzb
       real(c_double), device :: bsrVal(*)
       integer(c_int), device :: bsrRowPtr(*), bsrColInd(*)
       integer(c_int), value :: rowBlockDim, colBlockDim
       real(c_double), device :: bscVal(*)
       integer(c_int), device :: bscRowInd(*), bscColPtr(*)
       integer(c_int), value :: copyValues, baseIdx
       !dir$ ignore_tkr pBuffer
       character(c_char), device :: pBuffer(*)
     end function cusparseDgebsr2gebsc
  end interface

! cusparseCgebsr2gebsc
  interface
     integer(c_int) function cusparseCgebsr2gebsc(handle, mb, nb, nnzb, bsrVal, bsrRowPtr, bsrColInd, rowBlockDim, colBlockDim, &
          bscVal, bscRowInd, bscColPtr, copyValues, baseIdx, pBuffer) bind(C,name='cusparseCgebsr2gebsc')
       use iso_c_binding
       import cusparseHandle
       type(cusparseHandle), value :: handle       
       integer(c_int), value :: mb, nb, nnzb
       complex(4), device :: bsrVal(*)
       integer(c_int), device :: bsrRowPtr(*), bsrColInd(*)
       integer(c_int), value :: rowBlockDim, colBlockDim
       complex(4), device :: bscVal(*)
       integer(c_int), device :: bscRowInd(*), bscColPtr(*)
       integer(c_int), value :: copyValues, baseIdx
       !dir$ ignore_tkr pBuffer
       character(c_char), device :: pBuffer(*)
     end function cusparseCgebsr2gebsc
  end interface

! cusparseZgebsr2gebsc
  interface
     integer(c_int) function cusparseZgebsr2gebsc(handle, mb, nb, nnzb, bsrVal, bsrRowPtr, bsrColInd, rowBlockDim, colBlockDim, &
          bscVal, bscRowInd, bscColPtr, copyValues, baseIdx, pBuffer) bind(C,name='cusparseZgebsr2gebsc')
       use iso_c_binding
       import cusparseHandle
       type(cusparseHandle), value :: handle       
       integer(c_int), value :: mb, nb, nnzb
       complex(8), device :: bsrVal(*)
       integer(c_int), device :: bsrRowPtr(*), bsrColInd(*)
       integer(c_int), value :: rowBlockDim, colBlockDim
       complex(8), device :: bscVal(*)
       integer(c_int), device :: bscRowInd(*), bscColPtr(*)
       integer(c_int), value :: copyValues, baseIdx
       !dir$ ignore_tkr pBuffer
       character(c_char), device :: pBuffer(*)
     end function cusparseZgebsr2gebsc
  end interface

! cusparseSgebsr2gebsr_bufferSize
  interface cusparseSgebsr2gebsr_bufferSize
     integer(c_int) function cusparseSgebsr2gebsr_bufferSize(handle, mb, nb, nnzb, bsrVal, bsrRowPtr, bsrColInd, &
          rowBlockDimA, colBlockDimA, rowBlockDimC, colBlockDimC, pBufferSize) bind(C, name='cusparseSgebsr2gebsr_bufferSize')
       use iso_c_binding
       import cusparseHandle
       type(cusparseHandle), value :: handle       
       integer(c_int), value :: mb, nb, nnzb
       real(c_float), device :: bsrVal(*)
       integer(c_int), device :: bsrRowPtr(*), bsrColInd(*)
       integer(c_int), value :: rowBlockDimA, colBlockDimA, rowBlockDimC, colBlockDimC
       integer(4) :: pBufferSize
     end function cusparseSgebsr2gebsr_bufferSize
     integer(c_int) function cusparseSgebsr2gebsr_bufferSizeExt(handle, mb, nb, nnzb, bsrVal, bsrRowPtr, bsrColInd, &
          rowBlockDimA, colBlockDimA, rowBlockDimC, colBlockDimC, pBufferSize) bind(C, name='cusparseSgebsr2gebsr_bufferSizeExt')
       use iso_c_binding
       import cusparseHandle
       type(cusparseHandle), value :: handle       
       integer(c_int), value :: mb, nb, nnzb
       real(c_float), device :: bsrVal(*)
       integer(c_int), device :: bsrRowPtr(*), bsrColInd(*)
       integer(c_int), value :: rowBlockDimA, colBlockDimA, rowBlockDimC, colBlockDimC
       integer(8) :: pBufferSize
     end function cusparseSgebsr2gebsr_bufferSizeExt
  end interface

! cusparseDgebsr2gebsr_bufferSize
  interface cusparseDgebsr2gebsr_bufferSize
     integer(c_int) function cusparseDgebsr2gebsr_bufferSize(handle, mb, nb, nnzb, bsrVal, bsrRowPtr, bsrColInd, &
          rowBlockDimA, colBlockDimA, rowBlockDimC, colBlockDimC, pBufferSize) bind(C, name='cusparseDgebsr2gebsr_bufferSize')
       use iso_c_binding
       import cusparseHandle
       type(cusparseHandle), value :: handle       
       integer(c_int), value :: mb, nb, nnzb
       real(c_double), device :: bsrVal(*)
       integer(c_int), device :: bsrRowPtr(*), bsrColInd(*)
       integer(c_int), value :: rowBlockDimA, colBlockDimA, rowBlockDimC, colBlockDimC
       integer(4) :: pBufferSize
     end function cusparseDgebsr2gebsr_bufferSize
     integer(c_int) function cusparseDgebsr2gebsr_bufferSizeExt(handle, mb, nb, nnzb, bsrVal, bsrRowPtr, bsrColInd, &
          rowBlockDimA, colBlockDimA, rowBlockDimC, colBlockDimC, pBufferSize) bind(C, name='cusparseDgebsr2gebsr_bufferSizeExt')
       use iso_c_binding
       import cusparseHandle
       type(cusparseHandle), value :: handle       
       integer(c_int), value :: mb, nb, nnzb
       real(c_double), device :: bsrVal(*)
       integer(c_int), device :: bsrRowPtr(*), bsrColInd(*)
       integer(c_int), value :: rowBlockDimA, colBlockDimA, rowBlockDimC, colBlockDimC
       integer(8) :: pBufferSize
     end function cusparseDgebsr2gebsr_bufferSizeExt
  end interface

! cusparseCgebsr2gebsr_bufferSize
  interface cusparseCgebsr2gebsr_bufferSize
     integer(c_int) function cusparseCgebsr2gebsr_bufferSize(handle, mb, nb, nnzb, bsrVal, bsrRowPtr, bsrColInd, &
          rowBlockDimA, colBlockDimA, rowBlockDimC, colBlockDimC, pBufferSize) bind(C, name='cusparseCgebsr2gebsr_bufferSize')
       use iso_c_binding
       import cusparseHandle
       type(cusparseHandle), value :: handle       
       integer(c_int), value :: mb, nb, nnzb
       complex(4), device :: bsrVal(*)
       integer(c_int), device :: bsrRowPtr(*), bsrColInd(*)
       integer(c_int), value :: rowBlockDimA, colBlockDimA, rowBlockDimC, colBlockDimC
       integer(4) :: pBufferSize
     end function cusparseCgebsr2gebsr_bufferSize
     integer(c_int) function cusparseCgebsr2gebsr_bufferSizeExt(handle, mb, nb, nnzb, bsrVal, bsrRowPtr, bsrColInd, &
          rowBlockDimA, colBlockDimA, rowBlockDimC, colBlockDimC, pBufferSize) bind(C, name='cusparseCgebsr2gebsr_bufferSizeExt')
       use iso_c_binding
       import cusparseHandle
       type(cusparseHandle), value :: handle       
       integer(c_int), value :: mb, nb, nnzb
       complex(4), device :: bsrVal(*)
       integer(c_int), device :: bsrRowPtr(*), bsrColInd(*)
       integer(c_int), value :: rowBlockDimA, colBlockDimA, rowBlockDimC, colBlockDimC
       integer(8) :: pBufferSize
     end function cusparseCgebsr2gebsr_bufferSizeExt
  end interface

! cusparseZgebsr2gebsr_bufferSize
  interface cusparseZgebsr2gebsr_bufferSize
     integer(c_int) function cusparseZgebsr2gebsr_bufferSize(handle, mb, nb, nnzb, bsrVal, bsrRowPtr, bsrColInd, &
          rowBlockDimA, colBlockDimA, rowBlockDimC, colBlockDimC, pBufferSize) bind(C, name='cusparseZgebsr2gebsr_bufferSize')
       use iso_c_binding
       import cusparseHandle
       type(cusparseHandle), value :: handle       
       integer(c_int), value :: mb, nb, nnzb
       complex(8), device :: bsrVal(*)
       integer(c_int), device :: bsrRowPtr(*), bsrColInd(*)
       integer(c_int), value :: rowBlockDimA, colBlockDimA, rowBlockDimC, colBlockDimC
       integer(4) :: pBufferSize
     end function cusparseZgebsr2gebsr_bufferSize
     integer(c_int) function cusparseZgebsr2gebsr_bufferSizeExt(handle, mb, nb, nnzb, bsrVal, bsrRowPtr, bsrColInd, &
          rowBlockDimA, colBlockDimA, rowBlockDimC, colBlockDimC, pBufferSize) bind(C, name='cusparseZgebsr2gebsr_bufferSizeExt')
       use iso_c_binding
       import cusparseHandle
       type(cusparseHandle), value :: handle       
       integer(c_int), value :: mb, nb, nnzb
       complex(8), device :: bsrVal(*)
       integer(c_int), device :: bsrRowPtr(*), bsrColInd(*)
       integer(c_int), value :: rowBlockDimA, colBlockDimA, rowBlockDimC, colBlockDimC
       integer(8) :: pBufferSize
     end function cusparseZgebsr2gebsr_bufferSizeExt
  end interface

! cusparseXgebsr2gebsrNnz
  interface cusparseXgebsr2gebsrNnz
     integer function cusparseXgebsr2gebsrNnz_setHPM(handle, dir, mb, nb, nnzb, descrA, bsrRowPtrA, bsrColIndA, rowBlockDimA, colBlockDimA, &
       descrC, bsrRowPtrC, rowBlockDimC, colBlockDimC, nnzTotalDevHostPtr, pBuffer)
       import cusparseHandle, cusparseMatDescr
       type(cusparseHandle) :: handle
       integer :: dir, mb, nb, nnzb
       type(cusparseMatDescr) :: descrA
       integer, device :: bsrRowPtrA(*), bsrColIndA(*)
       integer :: rowBlockDimA, colBlockDimA
       type(cusparseMatDescr) :: descrC
       integer, device :: bsrRowPtrC(*)
       integer :: rowBlockDimC, colBlockDimC
       integer :: nnzTotalDevHostPtr
       !dir$ ignore_tkr pBuffer
       character, device :: pBuffer(*)
     end function cusparseXgebsr2gebsrNnz_setHPM 

     integer function cusparseXgebsr2gebsrNnz_setDPM(handle, dir, mb, nb, nnzb, descrA, bsrRowPtrA, bsrColIndA, rowBlockDimA, colBlockDimA, &
       descrC, bsrRowPtrC, rowBlockDimC, colBlockDimC, nnzTotalDevHostPtr, pBuffer)
       import cusparseHandle, cusparseMatDescr
       type(cusparseHandle) :: handle
       integer :: dir, mb, nb, nnzb
       type(cusparseMatDescr) :: descrA
       integer, device :: bsrRowPtrA(*), bsrColIndA(*)
       integer :: rowBlockDimA, colBlockDimA
       type(cusparseMatDescr) :: descrC
       integer, device :: bsrRowPtrC(*)
       integer :: rowBlockDimC, colBlockDimC
       integer, device :: nnzTotalDevHostPtr
       !dir$ ignore_tkr pBuffer
       character, device :: pBuffer(*)
     end function cusparseXgebsr2gebsrNnz_setDPM
  end interface cusparseXgebsr2gebsrNnz

#ifdef KEEP_V2_INTERFACES
  interface
     integer(c_int) function cusparseXgebsr2gebsrNnz_v2(handle, dir, mb, nb, nnzb, descrA, bsrRowPtrA, bsrColIndA, rowBlockDimA, colBlockDimA, &
          descrC, bsrRowPtrC, rowBlockDimC, colBlockDimC, nnzTotalDevHostPtr, pBuffer) bind(C,name='cusparseXgebsr2gebsrNnz')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr
       type(cusparseHandle), value :: handle
       integer(c_int), value :: dir, mb, nb, nnzb
       type(cusparseMatDescr), value :: descrA
       integer(c_int), device :: bsrRowPtrA(*), bsrColIndA(*)
       integer(c_int), value :: rowBlockDimA, colBlockDimA
       type(cusparseMatDescr), value :: descrC
       integer(c_int), device :: bsrRowPtrC(*)
       integer(c_int), value :: rowBlockDimC, colBlockDimC
       !dir$ ignore_tkr (d) nnzTotalDevHostPtr
       integer(c_int) :: nnzTotalDevHostPtr
       !dir$ ignore_tkr pBuffer
       character(c_char), device :: pBuffer(*)
     end function cusparseXgebsr2gebsrNnz_v2
  end interface
#endif

! cusparseSgebsr2gebsr
  interface
     integer(c_int) function cusparseSgebsr2gebsr(handle, dir, mb, nb, nnzb, descrA, bsrValA, bsrRowPtrA, bsrColIndA, rowBlockDimA, colBlockDimA, &
          descrC, bsrValC, bsrRowPtrC, bsrColIndC, rowBlockDimC, colBlockDimC, pBuffer) bind(C,name='cusparseSgebsr2gebsr')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr
       type(cusparseHandle), value :: handle
       integer(c_int), value :: dir, mb, nb, nnzb
       type(cusparseMatDescr), value :: descrA
       real(c_float), device :: bsrValA(*), bsrValC(*)
       integer(c_int), device :: bsrRowPtrA(*), bsrColIndA(*)
       integer(c_int), value :: rowBlockDimA, colBlockDimA
       type(cusparseMatDescr), value :: descrC
       integer(c_int), device :: bsrRowPtrC(*), bsrColIndC(*)
       integer(c_int), value :: rowBlockDimC, colBlockDimC
       !dir$ ignore_tkr pBuffer
       character(c_char), device :: pBuffer(*)
     end function cusparseSgebsr2gebsr
  end interface

! cusparseDgebsr2gebsr
  interface
     integer(c_int) function cusparseDgebsr2gebsr(handle, dir, mb, nb, nnzb, descrA, bsrValA, bsrRowPtrA, bsrColIndA, rowBlockDimA, colBlockDimA, &
          descrC, bsrValC, bsrRowPtrC, bsrColIndC, rowBlockDimC, colBlockDimC, pBuffer) bind(C,name='cusparseDgebsr2gebsr')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr
       type(cusparseHandle), value :: handle
       integer(c_int), value :: dir, mb, nb, nnzb
       type(cusparseMatDescr), value :: descrA
       real(c_double), device :: bsrValA(*), bsrValC(*)
       integer(c_int), device :: bsrRowPtrA(*), bsrColIndA(*)
       integer(c_int), value :: rowBlockDimA, colBlockDimA
       type(cusparseMatDescr), value :: descrC
       integer(c_int), device :: bsrRowPtrC(*), bsrColIndC(*)
       integer(c_int), value :: rowBlockDimC, colBlockDimC
       !dir$ ignore_tkr pBuffer
       character(c_char), device :: pBuffer(*)
     end function cusparseDgebsr2gebsr
  end interface

! cusparseCgebsr2gebsr
  interface
     integer(c_int) function cusparseCgebsr2gebsr(handle, dir, mb, nb, nnzb, descrA, bsrValA, bsrRowPtrA, bsrColIndA, rowBlockDimA, colBlockDimA, &
          descrC, bsrValC, bsrRowPtrC, bsrColIndC, rowBlockDimC, colBlockDimC, pBuffer) bind(C,name='cusparseCgebsr2gebsr')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr
       type(cusparseHandle), value :: handle
       integer(c_int), value :: dir, mb, nb, nnzb
       type(cusparseMatDescr), value :: descrA
       complex(4), device :: bsrValA(*), bsrValC(*)
       integer(c_int), device :: bsrRowPtrA(*), bsrColIndA(*)
       integer(c_int), value :: rowBlockDimA, colBlockDimA
       type(cusparseMatDescr), value :: descrC
       integer(c_int), device :: bsrRowPtrC(*), bsrColIndC(*)
       integer(c_int), value :: rowBlockDimC, colBlockDimC
       !dir$ ignore_tkr pBuffer
       character(c_char), device :: pBuffer(*)
     end function cusparseCgebsr2gebsr
  end interface

! cusparseZgebsr2gebsr
  interface
     integer(c_int) function cusparseZgebsr2gebsr(handle, dir, mb, nb, nnzb, descrA, bsrValA, bsrRowPtrA, bsrColIndA, rowBlockDimA, colBlockDimA, &
          descrC, bsrValC, bsrRowPtrC, bsrColIndC, rowBlockDimC, colBlockDimC, pBuffer) bind(C,name='cusparseZgebsr2gebsr')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr
       type(cusparseHandle), value :: handle
       integer(c_int), value :: dir, mb, nb, nnzb
       type(cusparseMatDescr), value :: descrA
       complex(8), device :: bsrValA(*), bsrValC(*)
       integer(c_int), device :: bsrRowPtrA(*), bsrColIndA(*)
       integer(c_int), value :: rowBlockDimA, colBlockDimA
       type(cusparseMatDescr), value :: descrC
       integer(c_int), device :: bsrRowPtrC(*), bsrColIndC(*)
       integer(c_int), value :: rowBlockDimC, colBlockDimC
       !dir$ ignore_tkr pBuffer
       character(c_char), device :: pBuffer(*)
     end function cusparseZgebsr2gebsr
  end interface

! cusparseSgebsr2csr
  interface
     integer(c_int) function cusparseSgebsr2csr(handle, dir, mb, nb, descrA, bsrValA, bsrRowPtrA, bsrColIndA, rowBlockDimA, colBlockDimA, &
          descrC, csrValC, csrRowPtrC, csrColIndC) bind(C,name='cusparseSgebsr2csr')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr
       type(cusparseHandle), value :: handle
       integer(c_int), value :: dir, mb, nb
       type(cusparseMatDescr), value :: descrA
       real(c_float), device :: bsrValA(*), csrValC(*)
       integer(c_int), device :: bsrRowPtrA(*), bsrColIndA(*)
       integer(c_int), value :: rowBlockDimA, colBlockDimA
       type(cusparseMatDescr), value :: descrC
       integer(c_int), device :: csrRowPtrC(*), csrColIndC(*)
     end function cusparseSgebsr2csr
  end interface

! cusparseDgebsr2csr
  interface
     integer(c_int) function cusparseDgebsr2csr(handle, dir, mb, nb, descrA, bsrValA, bsrRowPtrA, bsrColIndA, rowBlockDimA, colBlockDimA, &
          descrC, csrValC, csrRowPtrC, csrColIndC) bind(C,name='cusparseDgebsr2csr')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr
       type(cusparseHandle), value :: handle
       integer(c_int), value :: dir, mb, nb
       type(cusparseMatDescr), value :: descrA
       real(c_double), device :: bsrValA(*), csrValC(*)
       integer(c_int), device :: bsrRowPtrA(*), bsrColIndA(*)
       integer(c_int), value :: rowBlockDimA, colBlockDimA
       type(cusparseMatDescr), value :: descrC
       integer(c_int), device :: csrRowPtrC(*), csrColIndC(*)
     end function cusparseDgebsr2csr
  end interface

! cusparseCgebsr2csr
  interface
     integer(c_int) function cusparseCgebsr2csr(handle, dir, mb, nb, descrA, bsrValA, bsrRowPtrA, bsrColIndA, rowBlockDimA, colBlockDimA, &
          descrC, csrValC, csrRowPtrC, csrColIndC) bind(C,name='cusparseCgebsr2csr')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr
       type(cusparseHandle), value :: handle
       integer(c_int), value :: dir, mb, nb
       type(cusparseMatDescr), value :: descrA
       complex(4), device :: bsrValA(*), csrValC(*)
       integer(c_int), device :: bsrRowPtrA(*), bsrColIndA(*)
       integer(c_int), value :: rowBlockDimA, colBlockDimA
       type(cusparseMatDescr), value :: descrC
       integer(c_int), device :: csrRowPtrC(*), csrColIndC(*)
     end function cusparseCgebsr2csr
  end interface

! cusparseZgebsr2csr
  interface
     integer(c_int) function cusparseZgebsr2csr(handle, dir, mb, nb, descrA, bsrValA, bsrRowPtrA, bsrColIndA, rowBlockDimA, colBlockDimA, &
          descrC, csrValC, csrRowPtrC, csrColIndC) bind(C,name='cusparseZgebsr2csr')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr
       type(cusparseHandle), value :: handle
       integer(c_int), value :: dir, mb, nb
       type(cusparseMatDescr), value :: descrA
       complex(8), device :: bsrValA(*), csrValC(*)
       integer(c_int), device :: bsrRowPtrA(*), bsrColIndA(*)
       integer(c_int), value :: rowBlockDimA, colBlockDimA
       type(cusparseMatDescr), value :: descrC
       integer(c_int), device :: csrRowPtrC(*), csrColIndC(*)
     end function cusparseZgebsr2csr
  end interface

! cusparseScsr2gebsr_bufferSize
  interface cusparseScsr2gebsr_bufferSize
     integer(c_int) function cusparseScsr2gebsr_bufferSize(handle, dir, m, n, descrA, csrVal, csrRowPtr, csrColInd, &
          rowBlockDim, colBlockDim, pBufferSize) bind(C, name='cusparseScsr2gebsr_bufferSize')
       use iso_c_binding
       import cusparseHandle
       type(cusparseHandle), value :: handle  
       integer(c_int), value :: dir, m, n
       real(c_float), device :: csrVal(*)
       integer(c_int), device :: csrRowPtr(*), csrColInd(*)
       integer(c_int), value :: rowBlockDim, colBlockDim
       integer(4) :: pBufferSize
     end function cusparseScsr2gebsr_bufferSize
     integer(c_int) function cusparseScsr2gebsr_bufferSizeExt(handle, dir, m, n, descrA, csrVal, csrRowPtr, csrColInd, &
          rowBlockDim, colBlockDim, pBufferSize) bind(C, name='cusparseScsr2gebsr_bufferSizeExt')
       use iso_c_binding
       import cusparseHandle
       type(cusparseHandle), value :: handle  
       integer(c_int), value :: dir, m, n
       real(c_float), device :: csrVal(*)
       integer(c_int), device :: csrRowPtr(*), csrColInd(*)
       integer(c_int), value :: rowBlockDim, colBlockDim
       integer(8) :: pBufferSize
     end function cusparseScsr2gebsr_bufferSizeExt
  end interface

! cusparseDcsr2gebsr_bufferSize
  interface cusparseDcsr2gebsr_bufferSize
     integer(c_int) function cusparseDcsr2gebsr_bufferSize(handle, dir, m, n, descrA, csrVal, csrRowPtr, csrColInd, &
          rowBlockDim, colBlockDim, pBufferSize) bind(C, name='cusparseDcsr2gebsr_bufferSize')
       use iso_c_binding
       import cusparseHandle
       type(cusparseHandle), value :: handle  
       integer(c_int), value :: dir, m, n
       real(c_double), device :: csrVal(*)
       integer(c_int), device :: csrRowPtr(*), csrColInd(*)
       integer(c_int), value :: rowBlockDim, colBlockDim
       integer(4) :: pBufferSize
     end function cusparseDcsr2gebsr_bufferSize
     integer(c_int) function cusparseDcsr2gebsr_bufferSizeExt(handle, dir, m, n, descrA, csrVal, csrRowPtr, csrColInd, &
          rowBlockDim, colBlockDim, pBufferSize) bind(C, name='cusparseDcsr2gebsr_bufferSizeExt')
       use iso_c_binding
       import cusparseHandle
       type(cusparseHandle), value :: handle  
       integer(c_int), value :: dir, m, n
       real(c_double), device :: csrVal(*)
       integer(c_int), device :: csrRowPtr(*), csrColInd(*)
       integer(c_int), value :: rowBlockDim, colBlockDim
       integer(8) :: pBufferSize
     end function cusparseDcsr2gebsr_bufferSizeExt
  end interface

! cusparseCcsr2gebsr_bufferSize
  interface cusparseCcsr2gebsr_bufferSize
     integer(c_int) function cusparseCcsr2gebsr_bufferSize(handle, dir, m, n, descrA, csrVal, csrRowPtr, csrColInd, &
          rowBlockDim, colBlockDim, pBufferSize) bind(C, name='cusparseCcsr2gebsr_bufferSize')
       use iso_c_binding
       import cusparseHandle
       type(cusparseHandle), value :: handle  
       integer(c_int), value :: dir, m, n
       complex(4), device :: csrVal(*)
       integer(c_int), device :: csrRowPtr(*), csrColInd(*)
       integer(c_int), value :: rowBlockDim, colBlockDim
       integer(4) :: pBufferSize
     end function cusparseCcsr2gebsr_bufferSize
     integer(c_int) function cusparseCcsr2gebsr_bufferSizeExt(handle, dir, m, n, descrA, csrVal, csrRowPtr, csrColInd, &
          rowBlockDim, colBlockDim, pBufferSize) bind(C, name='cusparseCcsr2gebsr_bufferSizeExt')
       use iso_c_binding
       import cusparseHandle
       type(cusparseHandle), value :: handle  
       integer(c_int), value :: dir, m, n
       complex(4), device :: csrVal(*)
       integer(c_int), device :: csrRowPtr(*), csrColInd(*)
       integer(c_int), value :: rowBlockDim, colBlockDim
       integer(8) :: pBufferSize
     end function cusparseCcsr2gebsr_bufferSizeExt
  end interface

! cusparseZcsr2gebsr_bufferSize
  interface cusparseZcsr2gebsr_bufferSize
     integer(c_int) function cusparseZcsr2gebsr_bufferSize(handle, dir, m, n, descrA, csrVal, csrRowPtr, csrColInd, &
          rowBlockDim, colBlockDim, pBufferSize) bind(C, name='cusparseZcsr2gebsr_bufferSize')
       use iso_c_binding
       import cusparseHandle
       type(cusparseHandle), value :: handle  
       integer(c_int), value :: dir, m, n
       complex(8), device :: csrVal(*)
       integer(c_int), device :: csrRowPtr(*), csrColInd(*)
       integer(c_int), value :: rowBlockDim, colBlockDim
       integer(4) :: pBufferSize
     end function cusparseZcsr2gebsr_bufferSize
     integer(c_int) function cusparseZcsr2gebsr_bufferSizeExt(handle, dir, m, n, descrA, csrVal, csrRowPtr, csrColInd, &
          rowBlockDim, colBlockDim, pBufferSize) bind(C, name='cusparseZcsr2gebsr_bufferSizeExt')
       use iso_c_binding
       import cusparseHandle
       type(cusparseHandle), value :: handle  
       integer(c_int), value :: dir, m, n
       complex(8), device :: csrVal(*)
       integer(c_int), device :: csrRowPtr(*), csrColInd(*)
       integer(c_int), value :: rowBlockDim, colBlockDim
       integer(8) :: pBufferSize
     end function cusparseZcsr2gebsr_bufferSizeExt
  end interface

! cusparseXcsr2gebsrNnz
  interface cusparseXcsr2gebsrNnz
     integer function cusparseXcsr2gebsrNnz_setHPM(handle, dir, m, n, descrA, csrRowPtrA, csrColIndA, &
          descrC, bsrRowPtrC, rowBlockDimC, colBlockDimC, nnzTotalDevHostPtr, pBuffer)
       import cusparseHandle, cusparseMatDescr
       type(cusparseHandle) :: handle
       integer :: dir, m, n
       type(cusparseMatDescr) :: descrA
       integer, device :: csrRowPtrA(*), csrColIndA(*)
       type(cusparseMatDescr) :: descrC
       integer, device :: bsrRowPtrC(*)
       integer :: rowBlockDimC, colBlockDimC
       integer :: nnzTotalDevHostPtr
       !dir$ ignore_tkr pBuffer
       character, device :: pBuffer(*)
     end function cusparseXcsr2gebsrNnz_setHPM

     integer function cusparseXcsr2gebsrNnz_setDPM(handle, dir, m, n, descrA, csrRowPtrA, csrColIndA, &
          descrC, bsrRowPtrC, rowBlockDimC, colBlockDimC, nnzTotalDevHostPtr, pBuffer)
       import cusparseHandle, cusparseMatDescr
       type(cusparseHandle) :: handle
       integer :: dir, m, n
       type(cusparseMatDescr) :: descrA
       integer, device :: csrRowPtrA(*), csrColIndA(*)
       type(cusparseMatDescr) :: descrC
       integer, device :: bsrRowPtrC(*)
       integer :: rowBlockDimC, colBlockDimC
       integer, device :: nnzTotalDevHostPtr
       !dir$ ignore_tkr pBuffer
       character, device :: pBuffer(*)
     end function cusparseXcsr2gebsrNnz_setDPM
  end interface cusparseXcsr2gebsrNnz

#ifdef KEEP_V2_INTERFACES
  interface
     integer(c_int) function cusparseXcsr2gebsrNnz_v2(handle, dir, m, n, descrA, csrRowPtrA, csrColIndA, &
          descrC, bsrRowPtrC, rowBlockDimC, colBlockDimC, nnzTotalDevHostPtr, pBuffer) bind(C,name='cusparseXcsr2gebsrNnz')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr
       type(cusparseHandle), value :: handle
       integer(c_int), value :: dir, m, n
       type(cusparseMatDescr), value :: descrA
       integer(c_int), device :: csrRowPtrA(*), csrColIndA(*)
       type(cusparseMatDescr), value :: descrC
       integer(c_int), device :: bsrRowPtrC(*)
       integer(c_int), value :: rowBlockDimC, colBlockDimC
       !dir$ ignore_tkr (d) nnzTotalDevHostPtr
       integer(c_int) :: nnzTotalDevHostPtr
       !dir$ ignore_tkr pBuffer
       character(c_char), device :: pBuffer(*)
     end function cusparseXcsr2gebsrNnz_v2
  end interface
#endif

! cusparseScsr2gebsr
  interface
     integer(c_int) function cusparseScsr2gebsr(handle, dir, m, n, descrA, csrValA, csrRowPtrA, csrColIndA, &
          descrC, bsrValC, bsrRowPtrC, bsrColIndC, rowBlockDimC, colBlockDimC, pBuffer) bind(C,name='cusparseScsr2gebsr')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr
       type(cusparseHandle), value :: handle
       integer(c_int), value :: dir
       type(cusparseMatDescr), value :: descrA
       real(c_float), device :: csrValA(*), bsrValC(*)
       integer(c_int), device :: csrRowPtrA(*), csrColIndA(*)
       type(cusparseMatDescr), value :: descrC
       integer(c_int), device :: bsrRowPtrC(*), bsrColIndC(*)
       integer(c_int), value :: rowBlockDimC, colBlockDimC
       !dir$ ignore_tkr pBuffer
       character(c_char), device :: pBuffer(*)
     end function cusparseScsr2gebsr
  end interface

! cusparseDcsr2gebsr
  interface
     integer(c_int) function cusparseDcsr2gebsr(handle, dir, m, n, descrA, csrValA, csrRowPtrA, csrColIndA, &
          descrC, bsrValC, bsrRowPtrC, bsrColIndC, rowBlockDimC, colBlockDimC, pBuffer) bind(C,name='cusparseDcsr2gebsr')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr
       type(cusparseHandle), value :: handle
       integer(c_int), value :: dir
       type(cusparseMatDescr), value :: descrA
       real(c_double), device :: csrValA(*), bsrValC(*)
       integer(c_int), device :: csrRowPtrA(*), csrColIndA(*)
       type(cusparseMatDescr), value :: descrC
       integer(c_int), device :: bsrRowPtrC(*), bsrColIndC(*)
       integer(c_int), value :: rowBlockDimC, colBlockDimC
       !dir$ ignore_tkr pBuffer
       character(c_char), device :: pBuffer(*)
     end function cusparseDcsr2gebsr
  end interface

! cusparseCcsr2gebsr
  interface
     integer(c_int) function cusparseCcsr2gebsr(handle, dir, m, n, descrA, csrValA, csrRowPtrA, csrColIndA, &
          descrC, bsrValC, bsrRowPtrC, bsrColIndC, rowBlockDimC, colBlockDimC, pBuffer) bind(C,name='cusparseCcsr2gebsr')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr
       type(cusparseHandle), value :: handle
       integer(c_int), value :: dir
       type(cusparseMatDescr), value :: descrA
       complex(4), device :: csrValA(*), bsrValC(*)
       integer(c_int), device :: csrRowPtrA(*), csrColIndA(*)
       type(cusparseMatDescr), value :: descrC
       integer(c_int), device :: bsrRowPtrC(*), bsrColIndC(*)
       integer(c_int), value :: rowBlockDimC, colBlockDimC
       !dir$ ignore_tkr pBuffer
       character(c_char), device :: pBuffer(*)
     end function cusparseCcsr2gebsr
  end interface

! cusparseZcsr2gebsr
  interface
     integer(c_int) function cusparseZcsr2gebsr(handle, dir, m, n, descrA, csrValA, csrRowPtrA, csrColIndA, &
          descrC, bsrValC, bsrRowPtrC, bsrColIndC, rowBlockDimC, colBlockDimC, pBuffer) bind(C,name='cusparseZcsr2gebsr')
       use iso_c_binding
       import cusparseHandle, cusparseMatDescr
       type(cusparseHandle), value :: handle
       integer(c_int), value :: dir
       type(cusparseMatDescr), value :: descrA
       complex(8), device :: csrValA(*), bsrValC(*)
       integer(c_int), device :: csrRowPtrA(*), csrColIndA(*)
       type(cusparseMatDescr), value :: descrC
       integer(c_int), device :: bsrRowPtrC(*), bsrColIndC(*)
       integer(c_int), value :: rowBlockDimC, colBlockDimC
       !dir$ ignore_tkr pBuffer
       character(c_char), device :: pBuffer(*)
     end function cusparseZcsr2gebsr
  end interface

  ! ----------------------
  ! 6.5 format conversions
  ! ----------------------
  ! none

  ! ----------------------
  ! 7.0 format conversions
  ! ----------------------

! cusparseCreateIdentityPermutation
  interface
     integer(c_int) function cusparseCreateIdentityPermutation(handle, n, p) &
          bind(C,name='cusparseCreateIdentityPermutation')
       use iso_c_binding
       import cusparseHandle
       type(cusparseHandle), value :: handle
       integer(c_int), value :: n
       integer(c_int), device :: p(*)
     end function cusparseCreateIdentityPermutation
  end interface

! cusparseXcoosort_bufferSize
  interface cusparseXcoosort_bufferSize
     integer(c_int) function cusparseXcoosort_bufferSizeExt(handle, m, n, nnz, &
          cooRows, cooCols, pBufferSizeInBytes) bind(C, name='cusparseXcoosort_bufferSizeExt')
       use iso_c_binding
       import cusparseHandle
       type(cusparseHandle), value :: handle       
       integer(c_int), value :: m, n, nnz
       integer(c_int), device :: cooRows(*), cooCols(*)
       integer(8) :: pBufferSizeInBytes
     end function cusparseXcoosort_bufferSizeExt
  end interface cusparseXcoosort_bufferSize

! cusparseXcoosortByRow
  interface cusparseXcoosortByRow
     integer(c_int) function cusparseXcoosortByRow(handle, m, n, nnz, &
          cooRows, cooCols, P, pBuffer) bind(C, name='cusparseXcoosortByRow')
       use iso_c_binding
       import cusparseHandle
       type(cusparseHandle), value :: handle       
       integer(c_int), value :: m, n, nnz
       integer(c_int), device :: cooRows(*), cooCols(*), P(*)
       !dir$ ignore_tkr pBuffer
       character(c_char), device :: pBuffer(*)
     end function cusparseXcoosortByRow
  end interface cusparseXcoosortByRow

! cusparseXcoosortByColumn
  interface cusparseXcoosortByColumn
     integer(c_int) function cusparseXcoosortByColumn(handle, m, n, nnz, &
          cooRows, cooCols, P, pBuffer) bind(C, name='cusparseXcoosortByColumn')
       use iso_c_binding
       import cusparseHandle
       type(cusparseHandle), value :: handle       
       integer(c_int), value :: m, n, nnz
       integer(c_int), device :: cooRows(*), cooCols(*), P(*)
       !dir$ ignore_tkr pBuffer
       character(c_char), device :: pBuffer(*)
     end function cusparseXcoosortByColumn
  end interface cusparseXcoosortByColumn
 
! cusparseXcsrsort_bufferSize
  interface cusparseXcsrsort_bufferSize
     integer(c_int) function cusparseXcsrsort_bufferSizeExt(handle, m, n, nnz, &
          csrRowInd, csrColInd, pBufferSizeInBytes) bind(C, name='cusparseXcsrsort_bufferSizeExt')
       use iso_c_binding
       import cusparseHandle
       type(cusparseHandle), value :: handle       
       integer(c_int), value :: m, n, nnz
       integer(c_int), device :: csrRowInd(*), csrColInd(*)
       integer(8) :: pBufferSizeInBytes
     end function cusparseXcsrsort_bufferSizeExt
  end interface cusparseXcsrsort_bufferSize

! cusparseXcsrsort
  interface cusparseXcsrsort
     integer(c_int) function cusparseXcsrsort(handle, m, n, nnz, &
          csrRowInd, csrColInd, P, pBuffer) bind(C, name='cusparseXcsrsort')
       use iso_c_binding
       import cusparseHandle
       type(cusparseHandle), value :: handle       
       integer(c_int), value :: m, n, nnz
       integer(c_int), device :: csrRowInd(*), csrColInd(*), P(*)
       !dir$ ignore_tkr pBuffer
       character(c_char), device :: pBuffer(*)
     end function cusparseXcsrsort
  end interface cusparseXcsrsort

! cusparseXcscsort_bufferSize
  interface cusparseXcscsort_bufferSize
     integer(c_int) function cusparseXcscsort_bufferSizeExt(handle, m, n, nnz, &
          cscColPtr, cscRowInd, pBufferSizeInBytes) bind(C, name='cusparseXcscsort_bufferSizeExt')
       use iso_c_binding
       import cusparseHandle
       type(cusparseHandle), value :: handle       
       integer(c_int), value :: m, n, nnz
       integer(c_int), device :: cscColPtr(*), cscRowInd(*)
       integer(8) :: pBufferSizeInBytes
     end function cusparseXcscsort_bufferSizeExt
  end interface cusparseXcscsort_bufferSize

! cusparseXcscsort
  interface cusparseXcscsort
     integer(c_int) function cusparseXcscsort(handle, m, n, nnz, &
          cscColPtr, cscRowInd, P, pBuffer) bind(C, name='cusparseXcscsort')
       use iso_c_binding
       import cusparseHandle
       type(cusparseHandle), value :: handle       
       integer(c_int), value :: m, n, nnz
       integer(c_int), device :: cscColPtr(*), cscRowInd(*), P(*)
       !dir$ ignore_tkr pBuffer
       character(c_char), device :: pBuffer(*)
     end function cusparseXcscsort
  end interface cusparseXcscsort
 
! cusparseScsru2csr_bufferSize
  interface cusparseScsru2csr_bufferSize
     integer(c_int) function cusparseScsru2csr_bufferSizeExt(handle, m, n, nnz, &
          csrVal, csrRowPtr, csrColInd, info, pBufferSizeInBytes) bind(C, name='cusparseScsru2csr_bufferSizeExt')
       use iso_c_binding
       import cusparseHandle, cusparseCsru2csrInfo
       type(cusparseHandle), value :: handle       
       integer(c_int), value :: m, n, nnz
       real(c_float), device :: csrVal(*)
       integer(c_int), device :: csrRowPtr(*), csrColInd(*)
       type(cusparseCsru2csrInfo), value :: info
       integer(8) :: pBufferSizeInBytes
     end function cusparseScsru2csr_bufferSizeExt
  end interface cusparseScsru2csr_bufferSize

! cusparseDcsru2csr_bufferSize
  interface cusparseDcsru2csr_bufferSize
     integer(c_int) function cusparseDcsru2csr_bufferSizeExt(handle, m, n, nnz, &
          csrVal, csrRowPtr, csrColInd, info, pBufferSizeInBytes) bind(C, name='cusparseDcsru2csr_bufferSizeExt')
       use iso_c_binding
       import cusparseHandle, cusparseCsru2csrInfo
       type(cusparseHandle), value :: handle       
       integer(c_int), value :: m, n, nnz
       real(c_double), device :: csrVal(*)
       integer(c_int), device :: csrRowPtr(*), csrColInd(*)
       type(cusparseCsru2csrInfo), value :: info
       integer(8) :: pBufferSizeInBytes
     end function cusparseDcsru2csr_bufferSizeExt
  end interface cusparseDcsru2csr_bufferSize

! cusparseCcsru2csr_bufferSize
  interface cusparseCcsru2csr_bufferSize
     integer(c_int) function cusparseCcsru2csr_bufferSizeExt(handle, m, n, nnz, &
          csrVal, csrRowPtr, csrColInd, info, pBufferSizeInBytes) bind(C, name='cusparseCcsru2csr_bufferSizeExt')
       use iso_c_binding
       import cusparseHandle, cusparseCsru2csrInfo
       type(cusparseHandle), value :: handle       
       integer(c_int), value :: m, n, nnz
       complex(4), device :: csrVal(*)
       integer(c_int), device :: csrRowPtr(*), csrColInd(*)
       type(cusparseCsru2csrInfo), value :: info
       integer(8) :: pBufferSizeInBytes
     end function cusparseCcsru2csr_bufferSizeExt
  end interface cusparseCcsru2csr_bufferSize

! cusparseZcsru2csr_bufferSize
  interface cusparseZcsru2csr_bufferSize
     integer(c_int) function cusparseZcsru2csr_bufferSizeExt(handle, m, n, nnz, &
          csrVal, csrRowPtr, csrColInd, info, pBufferSizeInBytes) bind(C, name='cusparseZcsru2csr_bufferSizeExt')
       use iso_c_binding
       import cusparseHandle, cusparseCsru2csrInfo
       type(cusparseHandle), value :: handle       
       integer(c_int), value :: m, n, nnz
       complex(8), device :: csrVal(*)
       integer(c_int), device :: csrRowPtr(*), csrColInd(*)
       type(cusparseCsru2csrInfo), value :: info
       integer(8) :: pBufferSizeInBytes
     end function cusparseZcsru2csr_bufferSizeExt
  end interface cusparseZcsru2csr_bufferSize

! cusparseScsru2csr
  interface cusparseScsru2csr
     integer(c_int) function cusparseScsru2csr(handle, m, n, nnz, &
          descrA, csrVal, csrRowPtr, csrColInd, info, pBuffer) bind(C, name='cusparseScsru2csr')
       use iso_c_binding
       import cusparseHandle, cusparseCsru2csrInfo, cusparseMatDescr
       type(cusparseHandle), value :: handle       
       integer(c_int), value :: m, n, nnz
       type(cusparseMatDescr), value :: descrA
       real(c_float), device :: csrVal(*)
       integer(c_int), device :: csrRowPtr(*), csrColInd(*)
       type(cusparseCsru2csrInfo), value :: info
       !dir$ ignore_tkr pBuffer
       character(c_char), device :: pBuffer(*)
     end function cusparseScsru2csr
  end interface cusparseScsru2csr

! cusparseDcsru2csr
  interface cusparseDcsru2csr
     integer(c_int) function cusparseDcsru2csr(handle, m, n, nnz, &
          descrA, csrVal, csrRowPtr, csrColInd, info, pBuffer) bind(C, name='cusparseDcsru2csr')
       use iso_c_binding
       import cusparseHandle, cusparseCsru2csrInfo, cusparseMatDescr
       type(cusparseHandle), value :: handle       
       integer(c_int), value :: m, n, nnz
       type(cusparseMatDescr), value :: descrA
       real(c_double), device :: csrVal(*)
       integer(c_int), device :: csrRowPtr(*), csrColInd(*)
       type(cusparseCsru2csrInfo), value :: info
       !dir$ ignore_tkr pBuffer
       character(c_char), device :: pBuffer(*)
     end function cusparseDcsru2csr
  end interface cusparseDcsru2csr

! cusparseCcsru2csr
  interface cusparseCcsru2csr
     integer(c_int) function cusparseCcsru2csr(handle, m, n, nnz, &
          descrA, csrVal, csrRowPtr, csrColInd, info, pBuffer) bind(C, name='cusparseCcsru2csr')
       use iso_c_binding
       import cusparseHandle, cusparseCsru2csrInfo, cusparseMatDescr
       type(cusparseHandle), value :: handle       
       integer(c_int), value :: m, n, nnz
       type(cusparseMatDescr), value :: descrA
       complex(4), device :: csrVal(*)
       integer(c_int), device :: csrRowPtr(*), csrColInd(*)
       type(cusparseCsru2csrInfo), value :: info
       !dir$ ignore_tkr pBuffer
       character(c_char), device :: pBuffer(*)
     end function cusparseCcsru2csr
  end interface cusparseCcsru2csr

! cusparseZcsru2csr
  interface cusparseZcsru2csr
     integer(c_int) function cusparseZcsru2csr(handle, m, n, nnz, &
          descrA, csrVal, csrRowPtr, csrColInd, info, pBuffer) bind(C, name='cusparseZcsru2csr')
       use iso_c_binding
       import cusparseHandle, cusparseCsru2csrInfo, cusparseMatDescr
       type(cusparseHandle), value :: handle       
       integer(c_int), value :: m, n, nnz
       type(cusparseMatDescr), value :: descrA
       complex(8), device :: csrVal(*)
       integer(c_int), device :: csrRowPtr(*), csrColInd(*)
       type(cusparseCsru2csrInfo), value :: info
       !dir$ ignore_tkr pBuffer
       character(c_char), device :: pBuffer(*)
     end function cusparseZcsru2csr
  end interface cusparseZcsru2csr
  
! cusparseScsr2csru
  interface cusparseScsr2csru
     integer(c_int) function cusparseScsr2csru(handle, m, n, nnz, &
          descrA, csrVal, csrRowPtr, csrColInd, info, pBuffer) bind(C, name='cusparseScsr2csru')
       use iso_c_binding
       import cusparseHandle, cusparseCsru2csrInfo, cusparseMatDescr
       type(cusparseHandle), value :: handle       
       integer(c_int), value :: m, n, nnz
       type(cusparseMatDescr), value :: descrA
       real(c_float), device :: csrVal(*)
       integer(c_int), device :: csrRowPtr(*), csrColInd(*)
       type(cusparseCsru2csrInfo), value :: info
       !dir$ ignore_tkr pBuffer
       character(c_char), device :: pBuffer(*)
     end function cusparseScsr2csru
  end interface cusparseScsr2csru

! cusparseDcsr2csru
  interface cusparseDcsr2csru
     integer(c_int) function cusparseDcsr2csru(handle, m, n, nnz, &
          descrA, csrVal, csrRowPtr, csrColInd, info, pBuffer) bind(C, name='cusparseDcsr2csru')
       use iso_c_binding
       import cusparseHandle, cusparseCsru2csrInfo, cusparseMatDescr
       type(cusparseHandle), value :: handle       
       integer(c_int), value :: m, n, nnz
       type(cusparseMatDescr), value :: descrA
       real(c_double), device :: csrVal(*)
       integer(c_int), device :: csrRowPtr(*), csrColInd(*)
       type(cusparseCsru2csrInfo), value :: info
       !dir$ ignore_tkr pBuffer
       character(c_char), device :: pBuffer(*)
     end function cusparseDcsr2csru
  end interface cusparseDcsr2csru

! cusparseCcsr2csru
  interface cusparseCcsr2csru
     integer(c_int) function cusparseCcsr2csru(handle, m, n, nnz, &
          descrA, csrVal, csrRowPtr, csrColInd, info, pBuffer) bind(C, name='cusparseCcsr2csru')
       use iso_c_binding
       import cusparseHandle, cusparseCsru2csrInfo, cusparseMatDescr
       type(cusparseHandle), value :: handle       
       integer(c_int), value :: m, n, nnz
       type(cusparseMatDescr), value :: descrA
       complex(4), device :: csrVal(*)
       integer(c_int), device :: csrRowPtr(*), csrColInd(*)
       type(cusparseCsru2csrInfo), value :: info
       !dir$ ignore_tkr pBuffer
       character(c_char), device :: pBuffer(*)
     end function cusparseCcsr2csru
  end interface cusparseCcsr2csru

! cusparseZcsr2csru
  interface cusparseZcsr2csru
     integer(c_int) function cusparseZcsr2csru(handle, m, n, nnz, &
          descrA, csrVal, csrRowPtr, csrColInd, info, pBuffer) bind(C, name='cusparseZcsr2csru')
       use iso_c_binding
       import cusparseHandle, cusparseCsru2csrInfo, cusparseMatDescr
       type(cusparseHandle), value :: handle       
       integer(c_int), value :: m, n, nnz
       type(cusparseMatDescr), value :: descrA
       complex(8), device :: csrVal(*)
       integer(c_int), device :: csrRowPtr(*), csrColInd(*)
       type(cusparseCsru2csrInfo), value :: info
       !dir$ ignore_tkr pBuffer
       character(c_char), device :: pBuffer(*)
     end function cusparseZcsr2csru
  end interface cusparseZcsr2csru

  ! ---------------------
  ! Some new functions added in 2021
  ! ---------------------

  interface cusparseCsrmvEx_bufferSize
     integer(c_int) function cusparseCsrmvEx_bufferSize(handle, alg, trans, m, n, nnz, &
          alpha, alphaType, descrA, csrValA, csrValAType, csrRowPtrA, csrColIndA, &
          x, xType, beta, betaType, y, yType, computeType, bufferSizeInBytes) &
          bind(C,name='cusparseCsrmvEx_bufferSize')
       import cusparseHandle, cusparseMatDescr
       type(cusparseHandle), value :: handle
       integer(4), value :: alg, trans
       integer(4), value :: m, n, nnz
       !dir$ ignore_tkr (tkrd) alpha, (tkrd) beta
       real(4) :: alpha, beta
       type(cusparseMatDescr), value :: descrA
       !dir$ ignore_tkr csrValA, x, y
       real(4), device :: csrValA(*)
       integer(4), device :: csrRowPtrA(*)
       integer(4), device :: csrColIndA(*)
       real(4), device :: x(*)
       real(4), device :: y(*) 
       integer(4), value  :: alphaType, csrValAType, xType, betaType, yType, computeType
       integer(8) :: bufferSizeInBytes
     end function cusparseCsrmvEx_bufferSize
  end interface cusparseCsrmvEx_bufferSize

  interface cusparseCsrmvEx
     integer(c_int) function cusparseCsrmvEx(handle, alg, trans, m, n, nnz, &
          alpha, alphaType, descrA, csrValA, csrValAType, csrRowPtrA, csrColIndA, &
          x, xType, beta, betaType, y, yType, computeType, buffer) bind(C,name='cusparseCsrmvEx')
       import cusparseHandle, cusparseMatDescr
       type(cusparseHandle), value :: handle
       integer(4), value :: alg, trans
       integer(4), value :: m, n, nnz
       !dir$ ignore_tkr (tkrd) alpha, (tkrd) beta
       real(4) :: alpha, beta
       type(cusparseMatDescr), value :: descrA
       !dir$ ignore_tkr csrValA, x, y
       real(4), device :: csrValA(*)
       integer(4), device :: csrRowPtrA(*)
       integer(4), device :: csrColIndA(*)
       real(4), device :: x(*)
       real(4), device :: y(*) 
       integer(4), value  :: alphaType, csrValAType, xType, betaType, yType, computeType
       !dir$ ignore_tkr buffer
       character, device :: buffer(*)
     end function cusparseCsrmvEx
  end interface cusparseCsrmvEx

! cusparseSgemvi_bufferSize
  interface
     integer(c_int) function cusparseSgemvi_bufferSize(handle, transA, m, n, nnz, &
          pBufferSize) bind(C,name='cusparseSgemvi_bufferSize')
       import cusparseHandle
       type(cusparseHandle), value :: handle       
       integer(4), value :: transA, m, n, nnz
       integer(4) :: pBufferSize
     end function cusparseSgemvi_bufferSize
  end interface

! cusparseDgemvi_bufferSize
  interface
     integer(c_int) function cusparseDgemvi_bufferSize(handle, transA, m, n, nnz, &
          pBufferSize) bind(C,name='cusparseSgemvi_bufferSize')
       import cusparseHandle
       type(cusparseHandle), value :: handle       
       integer(4), value :: transA, m, n, nnz
       integer(4) :: pBufferSize
     end function cusparseDgemvi_bufferSize
  end interface

! cusparseCgemvi_bufferSize
  interface
     integer(c_int) function cusparseCgemvi_bufferSize(handle, transA, m, n, nnz, &
          pBufferSize) bind(C,name='cusparseSgemvi_bufferSize')
       import cusparseHandle
       type(cusparseHandle), value :: handle       
       integer(4), value :: transA, m, n, nnz
       integer(4) :: pBufferSize
     end function cusparseCgemvi_bufferSize
  end interface

! cusparseZgemvi_bufferSize
  interface
     integer(c_int) function cusparseZgemvi_bufferSize(handle, transA, m, n, nnz, &
          pBufferSize) bind(C,name='cusparseSgemvi_bufferSize')
       import cusparseHandle
       type(cusparseHandle), value :: handle       
       integer(4), value :: transA, m, n, nnz
       integer(4) :: pBufferSize
     end function cusparseZgemvi_bufferSize
  end interface

! cusparseSgemvi
  interface
     integer(c_int) function cusparseSgemvi(handle, transA, m, n, alpha, A, lda, nnz, &
          xVal, xInd, beta, y, idxBase, pBuffer) &
          bind(C,name='cusparseSgemvi')
       import cusparseHandle
       type(cusparseHandle), value :: handle       
       integer(4), value :: transA, m, n, lda, nnz, idxBase
       !dir$ ignore_tkr(r) A
       real(4), device :: A(lda,*), xVal(*), y(*)
       integer(4), device :: xInd(*)
       !dir$ ignore_tkr (d) alpha, (d) beta
       real(4) :: alpha, beta
       !dir$ ignore_tkr pBuffer
       integer(1), device :: pBuffer(*)
     end function cusparseSgemvi
  end interface

! cusparseDgemvi
  interface
     integer(c_int) function cusparseDgemvi(handle, transA, m, n, alpha, A, lda, nnz, &
          xVal, xInd, beta, y, idxBase, pBuffer) &
          bind(C,name='cusparseSgemvi')
       import cusparseHandle
       type(cusparseHandle), value :: handle       
       integer(4), value :: transA, m, n, lda, nnz, idxBase
       !dir$ ignore_tkr(r) A
       real(8), device :: A(lda,*), xVal(*), y(*)
       integer(4), device :: xInd(*)
       !dir$ ignore_tkr (d) alpha, (d) beta
       real(8) :: alpha, beta
       !dir$ ignore_tkr pBuffer
       integer(1), device :: pBuffer(*)
     end function cusparseDgemvi
  end interface

! cusparseCgemvi
  interface
     integer(c_int) function cusparseCgemvi(handle, transA, m, n, alpha, A, lda, nnz, &
          xVal, xInd, beta, y, idxBase, pBuffer) &
          bind(C,name='cusparseSgemvi')
       import cusparseHandle
       type(cusparseHandle), value :: handle       
       integer(4), value :: transA, m, n, lda, nnz, idxBase
       !dir$ ignore_tkr(r) A
       complex(4), device :: A(lda,*), xVal(*), y(*)
       integer(4), device :: xInd(*)
       !dir$ ignore_tkr (d) alpha, (d) beta
       complex(4) :: alpha, beta
       !dir$ ignore_tkr pBuffer
       integer(1), device :: pBuffer(*)
     end function cusparseCgemvi
  end interface

! cusparseZgemvi
  interface
     integer(c_int) function cusparseZgemvi(handle, transA, m, n, alpha, A, lda, nnz, &
          xVal, xInd, beta, y, idxBase, pBuffer) &
          bind(C,name='cusparseSgemvi')
       import cusparseHandle
       type(cusparseHandle), value :: handle       
       integer(4), value :: transA, m, n, lda, nnz, idxBase
       !dir$ ignore_tkr(r) A
       complex(8), device :: A(lda,*), xVal(*), y(*)
       integer(4), device :: xInd(*)
       !dir$ ignore_tkr (d) alpha, (d) beta
       complex(8) :: alpha, beta
       !dir$ ignore_tkr pBuffer
       integer(1), device :: pBuffer(*)
     end function cusparseZgemvi
  end interface

! cusparseSgemmi
  interface
     integer(c_int) function cusparseSgemmi(handle, m, n, k, nnz, alpha, A, lda, &
          cscValB, cscColPtrB, cscRowIndB, beta, C, ldc) &
          bind(C,name='cusparseSgemmi')
       import cusparseHandle
       type(cusparseHandle), value :: handle       
       integer(4), value :: m, n, k, nnz, lda, ldc
       !dir$ ignore_tkr(r) A, (r) C
       real(4), device :: cscValB(*), A(lda,*), C(ldc,*)
       integer(4), device :: cscColPtrB(*), cscRowIndB(*)
       !dir$ ignore_tkr (d) alpha, (d) beta
       real(4) :: alpha, beta
     end function cusparseSgemmi
  end interface

! cusparseDgemmi
  interface
     integer(c_int) function cusparseDgemmi(handle, m, n, k, nnz, alpha, A, lda, &
          cscValB, cscColPtrB, cscRowIndB, beta, C, ldc) &
          bind(C,name='cusparseDgemmi')
       import cusparseHandle
       type(cusparseHandle), value :: handle       
       integer(4), value :: m, n, k, nnz, lda, ldc
       !dir$ ignore_tkr(r) A, (r) C
       real(8), device :: cscValB(*), A(lda,*), C(ldc,*)
       integer(4), device :: cscColPtrB(*), cscRowIndB(*)
       !dir$ ignore_tkr (d) alpha, (d) beta
       real(8) :: alpha, beta
     end function cusparseDgemmi
  end interface

! cusparseCgemmi
  interface
     integer(c_int) function cusparseCgemmi(handle, m, n, k, nnz, alpha, A, lda, &
          cscValB, cscColPtrB, cscRowIndB, beta, C, ldc) &
          bind(C,name='cusparseCgemmi')
       import cusparseHandle
       type(cusparseHandle), value :: handle       
       integer(4), value :: m, n, k, nnz, lda, ldc
       !dir$ ignore_tkr(r) A, (r) C
       complex(4), device :: cscValB(*), A(lda,*), C(ldc,*)
       integer(4), device :: cscColPtrB(*), cscRowIndB(*)
       !dir$ ignore_tkr (d) alpha, (d) beta
       complex(4) :: alpha, beta
     end function cusparseCgemmi
  end interface

! cusparseZgemmi
  interface
     integer(c_int) function cusparseZgemmi(handle, m, n, k, nnz, alpha, A, lda, &
          cscValB, cscColPtrB, cscRowIndB, beta, C, ldc) &
          bind(C,name='cusparseZgemmi')
       import cusparseHandle
       type(cusparseHandle), value :: handle       
       integer(4), value :: m, n, k, nnz, lda, ldc
       !dir$ ignore_tkr(r) A, (r) C
       complex(8), device :: cscValB(*), A(lda,*), C(ldc,*)
       integer(4), device :: cscColPtrB(*), cscRowIndB(*)
       !dir$ ignore_tkr (d) alpha, (d) beta
       complex(8) :: alpha, beta
     end function cusparseZgemmi
  end interface

  ! ---------------------
  ! Generic API Reference
  ! ---------------------

! cusparseCreateSpVec
  interface cusparseCreateSpVec
     integer(c_int) function cusparseCreateSpVec(descr, size, nnz, indices, &
       values, idxType, idxBase, valueType) bind(C, name='cusparseCreateSpVec')
       import cusparseSpVecDescr
       !dir$ ignore_tkr (k) size, (k) nnz
       !dir$ ignore_tkr indices, values 
       type(cusparseSpVecDescr) :: descr
       integer(8), value :: size, nnz
       integer(4), device :: indices(*)
       real(4), device    :: values(*)
       integer(4), value  :: idxType, idxBase, valueType
     end function cusparseCreateSpVec
  end interface cusparseCreateSpVec

! cusparseDestroySpVec
  interface cusparseDestroySpVec
     integer(c_int) function cusparseDestroySpVec(descr) &
       bind(C, name='cusparseDestroySpVec')
       import cusparseSpVecDescr
       type(cusparseSpVecDescr), value :: descr
     end function cusparseDestroySpVec
  end interface cusparseDestroySpVec

! cusparseSpVecGet
  interface cusparseSpVecGet
     integer(c_int) function cusparseSpVecGet(descr, size, nnz, indices, &
       values, idxType, idxBase, valueType) bind(C, name='cusparseSpVecGet')
       import cusparseSpVecDescr, c_devptr
       type(cusparseSpVecDescr), value :: descr
       integer(8)         :: size, nnz
       type(c_devptr)     :: indices
       type(c_devptr)     :: values
       integer(4)         :: idxType, idxBase, valueType
     end function cusparseSpVecGet
  end interface cusparseSpVecGet

! cusparseSpVecGetIndexBase
  interface cusparseSpVecGetIndexBase
     integer(c_int) function cusparseSpVecGetIndexBase(descr, idxBase) &
       bind(C, name='cusparseSpVecGetIndexBase')
       import cusparseSpVecDescr
       type(cusparseSpVecDescr), value :: descr
       integer(4)         :: idxBase
     end function cusparseSpVecGetIndexBase
  end interface cusparseSpVecGetIndexBase

! cusparseSpVecGetValues
  interface cusparseSpVecGetValues
     integer(c_int) function cusparseSpVecGetValues(descr, values) &
       bind(C, name='cusparseSpVecGetValues')
       import cusparseSpVecDescr, c_devptr
       type(cusparseSpVecDescr), value :: descr
       type(c_devptr) :: values
     end function cusparseSpVecGetValues
  end interface cusparseSpVecGetValues

! cusparseSpVecSetValues
  interface cusparseSpVecSetValues
     integer(c_int) function cusparseSpVecSetValues(descr, values) &
       bind(C, name='cusparseSpVecSetValues')
       import cusparseSpVecDescr
       type(cusparseSpVecDescr), value :: descr
       !dir$ ignore_tkr values 
       real(4), device    :: values(*)
     end function cusparseSpVecSetValues
  end interface cusparseSpVecSetValues

! ---------------

! cusparseCreateDnVec
  interface cusparseCreateDnVec
     integer(c_int) function cusparseCreateDnVec(descr, size, &
       values, valueType) bind(C, name='cusparseCreateDnVec')
       import cusparseDnVecDescr
       !dir$ ignore_tkr (k) size
       !dir$ ignore_tkr values 
       type(cusparseDnVecDescr) :: descr
       integer(8), value :: size
       real(4), device    :: values(*)
       integer(4), value  :: valueType
     end function cusparseCreateDnVec
  end interface cusparseCreateDnVec

! cusparseDestroyDnVec
  interface cusparseDestroyDnVec
     integer(c_int) function cusparseDestroyDnVec(descr) &
       bind(C, name='cusparseDestroyDnVec')
       import cusparseDnVecDescr
       type(cusparseDnVecDescr), value :: descr
     end function cusparseDestroyDnVec
  end interface cusparseDestroyDnVec

! cusparseDnVecGet
  interface cusparseDnVecGet
     integer(c_int) function cusparseDnVecGet(descr, size, &
       values, valueType) bind(C, name='cusparseDnVecGet')
       import cusparseDnVecDescr, c_devptr
       type(cusparseDnVecDescr), value :: descr
       integer(8)     :: size
       type(c_devptr) :: values
       integer(4)     :: valueType
     end function cusparseDnVecGet
  end interface cusparseDnVecGet

! cusparseDnVecGetValues
  interface cusparseDnVecGetValues
     integer(c_int) function cusparseDnVecGetValues(descr, values) &
       bind(C, name='cusparseDnVecGetValues')
       import cusparseDnVecDescr, c_devptr
       type(cusparseDnVecDescr), value :: descr
       type(c_devptr) :: values
     end function cusparseDnVecGetValues
  end interface cusparseDnVecGetValues

! cusparseDnVecSetValues
  interface cusparseDnVecSetValues
     integer(c_int) function cusparseDnVecSetValues(descr, values) &
       bind(C, name='cusparseDnVecSetValues')
       import cusparseDnVecDescr
       type(cusparseDnVecDescr), value :: descr
       !dir$ ignore_tkr values 
       real(4), device    :: values(*)
     end function cusparseDnVecSetValues
  end interface cusparseDnVecSetValues

! ---------------

! cusparseSpVV
  interface cusparseSpVV_bufferSize
     integer(c_int) function cusparseSpVV_buffersize(handle, opX, vecX, vecY, &
       result, computeType, bufferSize) bind(C, name='cusparseSpVV_bufferSize')
       import cusparseHandle, cusparseSpVecDescr, cusparseDnVecDescr
       type(cusparseHandle), value :: handle
       type(cusparseSpVecDescr), value :: vecX
       type(cusparseDnVecDescr), value :: vecY
       !dir$ ignore_tkr (tkrd) result
       real(4), device    :: result
       integer(4), value  :: opX, computeType
       integer(8)         :: bufferSize
     end function cusparseSpVV_buffersize
  end interface cusparseSpVV_buffersize

  interface cusparseSpVV
     integer(c_int) function cusparseSpVV(handle, opX, vecX, vecY, &
       result, computeType, buffer) bind(C, name='cusparseSpVV')
       import cusparseHandle, cusparseSpVecDescr, cusparseDnVecDescr
       type(cusparseHandle), value :: handle
       type(cusparseSpVecDescr), value :: vecX
       type(cusparseDnVecDescr), value :: vecY
       !dir$ ignore_tkr (tkrd) result
       !dir$ ignore_tkr buffer
       real(4), device    :: result
       integer(4), value  :: opX, computeType
       integer(4), device :: buffer
     end function cusparseSpVV
  end interface cusparseSpVV

! ---------------

! cusparseCreateCoo
  interface cusparseCreateCoo
     integer(c_int) function cusparseCreateCoo(descr, rows, cols, nnz, &
       cooRowInd, cooColInd, cooValues, idxType, idxBase, &
       valueType) bind(C, name='cusparseCreateCoo')
       import cusparseSpMatDescr
       !dir$ ignore_tkr (k) rows, (k) cols, (k) nnz
       !dir$ ignore_tkr cooRowInd, cooColInd, cooValues
       type(cusparseSpMatDescr) :: descr
       integer(8), value :: rows, cols, nnz
       integer(4), device :: cooRowInd(*), cooColInd(*)
       real(4), device    :: cooValues(*)
       integer(4), value  :: idxType, idxBase, valueType
     end function cusparseCreateCoo
  end interface cusparseCreateCoo

! cusparseCreateCooAoS
  interface cusparseCreateCooAoS
     integer(c_int) function cusparseCreateCooAoS(descr, rows, cols, nnz, &
       cooInd, cooValues, idxType, idxBase, &
       valueType) bind(C, name='cusparseCreateCooAoS')
       import cusparseSpMatDescr
       !dir$ ignore_tkr (k) rows, (k) cols, (k) nnz
       !dir$ ignore_tkr cooInd, cooValues
       type(cusparseSpMatDescr) :: descr
       integer(8), value :: rows, cols, nnz
       integer(4), device :: cooInd(*)
       real(4), device    :: cooValues(*)
       integer(4), value  :: idxType, idxBase, valueType
     end function cusparseCreateCooAoS
  end interface cusparseCreateCooAoS

! cusparseCreateCsr
  interface cusparseCreateCsr
     integer(c_int) function cusparseCreateCsr(descr, rows, cols, nnz, &
       csrRowOffsets, csrColInd, csrValues, csrRowOffsetsType, csrColIndType, &
       idxBase, valueType) bind(C, name='cusparseCreateCsr')
       import cusparseSpMatDescr
       !dir$ ignore_tkr (k) rows, (k) cols, (k) nnz
       !dir$ ignore_tkr csrRowOffsets, csrColInd, csrValues
       type(cusparseSpMatDescr) :: descr
       integer(8), value :: rows, cols, nnz
       integer(4), device :: csrRowOffsets(*), csrColInd(*)
       real(4), device    :: csrValues(*)
       integer(4), value  :: csrRowOffsetsType, csrColIndType, idxBase, valueType
     end function cusparseCreateCsr
  end interface cusparseCreateCsr

! cusparseDestroySpMat
  interface cusparseDestroySpMat
     integer(c_int) function cusparseDestroySpMat(descr) &
       bind(C, name='cusparseDestroySpMat')
       import cusparseSpMatDescr
       type(cusparseSpMatDescr), value :: descr
     end function cusparseDestroySpMat
  end interface cusparseDestroySpMat

! cusparseCooGet
  interface cusparseCooGet
     integer(c_int) function cusparseCooGet(descr, rows, cols, nnz, &
       cooRowInd, cooColInd, cooValues, idxType, idxBase, &
       valueType) bind(C, name='cusparseCooGet')
       import cusparseSpMatDescr, c_devptr
       type(cusparseSpMatDescr), value :: descr
       integer(8) :: rows, cols, nnz
       type(c_devptr) :: cooRowInd, cooColInd, cooValues
       integer(4) :: idxType, idxBase, valueType
     end function cusparseCooGet
  end interface cusparseCooGet

! cusparseCooAoSGet
  interface cusparseCooAoSGet
     integer(c_int) function cusparseCooAoSGet(descr, rows, cols, nnz, &
       cooInd, cooValues, idxType, idxBase, &
       valueType) bind(C, name='cusparseCooAoSGet')
       import cusparseSpMatDescr, c_devptr
       type(cusparseSpMatDescr), value :: descr
       integer(8) :: rows, cols, nnz
       type(c_devptr) :: cooInd, cooValues
       integer(4) :: idxType, idxBase, valueType
     end function cusparseCooAoSGet
  end interface cusparseCooAoSGet

! cusparseCsrGet
  interface cusparseCsrGet
     integer(c_int) function cusparseCsrGet(descr, rows, cols, nnz, &
       csrRowOffsets, csrColInd, csrValues, csrRowOffsetsType, csrColIndType, &
       idxBase, valueType) bind(C, name='cusparseCsrGet')
       import cusparseSpMatDescr, c_devptr
       type(cusparseSpMatDescr), value :: descr
       integer(8) :: rows, cols, nnz
       type(c_devptr) :: csrRowOffsets, csrColInd, csrValues
       integer(4) :: csrRowOffsetsType, csrColIndType, idxBase, valueType
     end function cusparseCsrGet
  end interface cusparseCsrGet

! cusparseSpMatGetFormat
  interface cusparseSpMatGetFormat
     integer(c_int) function cusparseSpMatGetFormat(descr, format) &
       bind(C, name='cusparseSpMatGetFormat')
       import cusparseSpMatDescr
       type(cusparseSpMatDescr), value :: descr
       integer(4) :: format
     end function cusparseSpMatGetFormat
  end interface cusparseSpMatGetFormat

! cusparseSpMatGetIndexBase
  interface cusparseSpMatGetIndexBase
     integer(c_int) function cusparseSpMatGetIndexBase(descr, idxBase) &
       bind(C, name='cusparseSpMatGetIndexBase')
       import cusparseSpMatDescr
       type(cusparseSpMatDescr), value :: descr
       integer(4) :: idxBase
     end function cusparseSpMatGetIndexBase
  end interface cusparseSpMatGetIndexBase

! cusparseSpMatGetValues
  interface cusparseSpMatGetValues
     integer(c_int) function cusparseSpMatGetValues(descr, values) &
       bind(C, name='cusparseSpMatGetValues')
       import cusparseSpMatDescr, c_devptr
       type(cusparseSpMatDescr), value :: descr
       type(c_devptr) :: values
     end function cusparseSpMatGetValues
  end interface cusparseSpMatGetValues

! cusparseSpMatSetValues
  interface cusparseSpMatSetValues
     integer(c_int) function cusparseSpMatSetValues(descr, values) &
       bind(C, name='cusparseSpMatSetValues')
       import cusparseSpMatDescr
       type(cusparseSpMatDescr), value :: descr
       !dir$ ignore_tkr values 
       real(4), device    :: values(*)
     end function cusparseSpMatSetValues
  end interface cusparseSpMatSetValues

! cusparseSpMatGetSize
  interface cusparseSpMatGetSize
     integer(c_int) function cusparseSpMatGetSize(descr, rows, cols, nnz) &
       bind(C, name='cusparseSpMatGetSize')
       import cusparseSpMatDescr, c_devptr
       type(cusparseSpMatDescr), value :: descr
       integer(8) :: rows, cols, nnz
     end function cusparseSpMatGetSize
  end interface cusparseSpMatGetSize

! cusparseSpMatGetStridedBatch
  interface cusparseSpMatGetStridedBatch
     integer(c_int) function cusparseSpMatGetStridedBatch(descr, batchCount) &
       bind(C, name='cusparseSpMatGetStridedBatch')
       import cusparseSpMatDescr
       type(cusparseSpMatDescr), value :: descr
       integer(4) :: batchCount
     end function cusparseSpMatGetStridedBatch
  end interface cusparseSpMatGetStridedBatch

! cusparseSpMatSetStridedBatch
! This is gone?  Leave it for now, but can be removed with 11.x.
  interface cusparseSpMatSetStridedBatch
     integer(c_int) function cusparseSpMatSetStridedBatch(descr, batchCount) &
       bind(C, name='cusparseSpMatSetStridedBatch')
       import cusparseSpMatDescr
       type(cusparseSpMatDescr), value :: descr
       integer(4), value :: batchCount
     end function cusparseSpMatSetStridedBatch
  end interface cusparseSpMatSetStridedBatch

  interface cusparseCooSetStridedBatch
     integer(c_int) function cusparseCooSetStridedBatch(descr, batchCount, &
       batchStride) bind(C, name='cusparseCooSetStridedBatch')
       import cusparseSpMatDescr
       type(cusparseSpMatDescr), value :: descr
       integer(4), value :: batchCount
       !dir$ ignore_tkr (K) batchStride
       integer(8), value :: batchStride
     end function cusparseCooSetStridedBatch
  end interface cusparseCooSetStridedBatch

  interface cusparseCsrSetStridedBatch
     integer(c_int) function cusparseCsrSetStridedBatch(descr, batchCount, &
       offsetsBatchStride, columnsValuesBatchStride) &
       bind(C, name='cusparseCsrSetStridedBatch')
       import cusparseSpMatDescr
       type(cusparseSpMatDescr), value :: descr
       integer(4), value :: batchCount
       !dir$ ignore_tkr (K) offsetsBatchStride, (K) columnsValuesBatchStride
       integer(8), value :: offsetsBatchStride, columnsValuesBatchStride
     end function cusparseCsrSetStridedBatch
  end interface cusparseCsrSetStridedBatch

  interface cusparseBsrSetStridedBatch
     integer(c_int) function cusparseBsrSetStridedBatch(descr, batchCount, &
       offsetsBatchStride, columnsBatchStride, valuesBatchStride) &
       bind(C, name='cusparseBsrSetStridedBatch')
       import cusparseSpMatDescr
       type(cusparseSpMatDescr), value :: descr
       integer(4), value :: batchCount
       !dir$ ignore_tkr (K) offsetsBatchStride, (K) columnsBatchStride
       integer(8), value :: offsetsBatchStride, columnsBatchStride
       !dir$ ignore_tkr (K) valuesBatchStride
       integer(8), value :: valuesBatchStride
     end function cusparseBsrSetStridedBatch
  end interface cusparseBsrSetStridedBatch

! cusparseCsrSetPointers
  interface cusparseCsrSetPointers
     integer(c_int) function cusparseCsrSetPointers(descr, csrRowOffsets, &
                    csrColInd, csrValues) bind(C, name='cusparseCsrSetPointers')
       import cusparseSpMatDescr
       !dir$ ignore_tkr csrRowOffsets, csrColInd, csrValues
       type(cusparseSpMatDescr), value :: descr
       integer(4), device :: csrRowOffsets(*), csrColInd(*)
       real(4), device    :: csrValues(*)
     end function cusparseCsrSetPointers
  end interface cusparseCsrSetPointers

! cusparseCscSetPointers
  interface cusparseCscSetPointers
     integer(c_int) function cusparseCscSetPointers(descr, cscColOffsets, &
                    cscRowInd, cscValues) bind(C, name='cusparseCscSetPointers')
       import cusparseSpMatDescr
       !dir$ ignore_tkr cscColOffsets, cscRowInd, cscValues
       type(cusparseSpMatDescr), value :: descr
       integer(4), device :: cscColOffsets(*), cscRowInd(*)
       real(4), device    :: cscValues(*)
     end function cusparseCscSetPointers
  end interface cusparseCscSetPointers

! cusparseSpMatGetAttribute
  interface cusparseSpMatGetAttribute
     integer(c_int) function cusparseSpMatGetAttribute(descr, attribute, &
       data, dataSize) bind(C, name='cusparseSpMatGetAttribute')
       import cusparseSpMatDescr
       type(cusparseSpMatDescr), value :: descr
       integer(4), value :: attribute
       !dir$ ignore_tkr data, (K) dataSize
       integer(4) :: data
       integer(8), value :: dataSize
     end function cusparseSpMatGetAttribute
  end interface cusparseSpMatGetAttribute

! cusparseSpMatSetAttribute
  interface cusparseSpMatSetAttribute
     integer(c_int) function cusparseSpMatSetAttribute(descr, attribute, &
       data, dataSize) bind(C, name='cusparseSpMatSetAttribute')
       import cusparseSpMatDescr
       type(cusparseSpMatDescr), value :: descr
       integer(4), value :: attribute
       !dir$ ignore_tkr data, (K) dataSize
       integer(4) :: data
       integer(8), value :: dataSize
     end function cusparseSpMatSetAttribute
  end interface cusparseSpMatSetAttribute

! ---------------
! Blocked Ellpack

! cusparseCreateBlockedEll
  interface cusparseCreateBlockedEll
     integer(c_int) function cusparseCreateBlockedEll(descr, rows, cols, &
       ellBlockSize, ellCols, ellColInd, ellValues, ellIdxType, &
       idxBase, valueType) bind(C, name='cusparseCreateBlockedEll')
       import cusparseSpMatDescr
       !dir$ ignore_tkr (k) rows, (k) cols, (k) ellBlockSize, (k) ellCols
       !dir$ ignore_tkr ellColInd, ellValues
       type(cusparseSpMatDescr) :: descr
       integer(8), value :: rows, cols, ellBlockSize, ellCols
       integer(4), device :: ellColInd(*)
       real(4), device    :: ellValues(*)
       integer(4), value  :: ellIdxType, idxBase, valueType
     end function cusparseCreateBlockedEll
  end interface cusparseCreateBlockedEll

! cusparseBlockedEllGet
  interface cusparseBlockedEllGet
     integer(c_int) function cusparseBlockedEllGet(descr, rows, cols, &
       ellBlockSize, ellCols, ellColInd, ellValues, ellIdxType, &
       idxBase, valueType) bind(C, name='cusparseBlockedEllGet')
       import cusparseSpMatDescr, c_devptr
       !dir$ ignore_tkr ellColInd, ellValues
       type(cusparseSpMatDescr), value :: descr
       integer(8) :: rows, cols, ellBlockSize, ellCols
       type(c_devptr) :: ellColInd, ellValues
       integer(4)  :: ellIdxType, idxBase, valueType
     end function cusparseBlockedEllGet
  end interface cusparseBlockedEllGet

! ---------------

! cusparseCreateDnMat
  interface cusparseCreateDnMat
     integer(c_int) function cusparseCreateDnMat(descr, rows, cols, ld, &
       values, valueType, order) bind(C, name='cusparseCreateDnMat')
       import cusparseDnMatDescr
       !dir$ ignore_tkr (k) rows, (k) cols, (k) ld
       !dir$ ignore_tkr values 
       type(cusparseDnMatDescr) :: descr
       integer(8), value :: rows, cols, ld
       real(4), device    :: values(*)
       integer(4), value  :: valueType, order
     end function cusparseCreateDnMat
  end interface cusparseCreateDnMat

! cusparseDestroyDnMat
  interface cusparseDestroyDnMat
     integer(c_int) function cusparseDestroyDnMat(descr) &
       bind(C, name='cusparseDestroyDnMat')
       import cusparseDnMatDescr
       type(cusparseDnMatDescr), value :: descr
     end function cusparseDestroyDnMat
  end interface cusparseDestroyDnMat

! cusparseDnMatGet
  interface cusparseDnMatGet
     integer(c_int) function cusparseDnMatGet(descr, rows, cols, ld, &
       values, valueType, order) bind(C, name='cusparseDnMatGet')
       import cusparseDnMatDescr, c_devptr
       type(cusparseDnMatDescr), value :: descr
       integer(8)     :: rows, cols, ld
       type(c_devptr) :: values
       integer(4)     :: valueType, order
     end function cusparseDnMatGet
  end interface cusparseDnMatGet

! cusparseDnMatGetValues
  interface cusparseDnMatGetValues
     integer(c_int) function cusparseDnMatGetValues(descr, values) &
       bind(C, name='cusparseDnMatGetValues')
       import cusparseDnMatDescr, c_devptr
       type(cusparseDnMatDescr), value :: descr
       type(c_devptr) :: values
     end function cusparseDnMatGetValues
  end interface cusparseDnMatGetValues

! cusparseDnMatSetValues
  interface cusparseDnMatSetValues
     integer(c_int) function cusparseDnMatSetValues(descr, values) &
       bind(C, name='cusparseDnMatSetValues')
       import cusparseDnMatDescr
       type(cusparseDnMatDescr), value :: descr
       !dir$ ignore_tkr values 
       real(4), device    :: values(*)
     end function cusparseDnMatSetValues
  end interface cusparseDnMatSetValues

! cusparseDnMatGetStridedBatch
  interface cusparseDnMatGetStridedBatch
     integer(c_int) function cusparseDnMatGetStridedBatch(descr, batchCount, &
                     batchStride) bind(C, name='cusparseDnMatGetStridedBatch')
       import cusparseDnMatDescr
       type(cusparseDnMatDescr), value :: descr
       integer(4) :: batchCount
       integer(8) :: batchStride
     end function cusparseDnMatGetStridedBatch
  end interface cusparseDnMatGetStridedBatch

! cusparseDnMatSetStridedBatch
  interface cusparseDnMatSetStridedBatch
     integer(c_int) function cusparseDnMatSetStridedBatch(descr, batchCount, &
                     batchStride) bind(C, name='cusparseDnMatSetStridedBatch')
       import cusparseDnMatDescr
       !dir$ ignore_tkr (k) batchStride
       type(cusparseDnMatDescr), value :: descr
       integer(4), value :: batchCount
       integer(8), value :: batchStride
     end function cusparseDnMatSetStridedBatch
  end interface cusparseDnMatSetStridedBatch

! ---------------

! cusparseSpMV
  interface cusparseSpMV_bufferSize
     integer(c_int) function cusparseSpMV_buffersize(handle, opA, alpha, &
                        matA, vecX, beta, vecY, computeType, alg, bufferSize) &
                                        bind(C, name='cusparseSpMV_bufferSize')
       import cusparseHandle, cusparseSpMatDescr, cusparseDnVecDescr
       type(cusparseHandle), value :: handle
       !dir$ ignore_tkr (tkrd) alpha, (tkrd) beta
       real(4) :: alpha, beta
       type(cusparseSpMatDescr), value :: matA
       type(cusparseDnVecDescr), value :: vecX, vecY
       integer(4), value  :: opA, computeType, alg
       integer(8)         :: bufferSize
     end function cusparseSpMV_buffersize
  end interface cusparseSpMV_buffersize

  interface cusparseSpMV
     integer(c_int) function cusparseSpMV(handle, opA, alpha, &
                        matA, vecX, beta, vecY, computeType, alg, buffer) &
                                               bind(C, name='cusparseSpMV')
       import cusparseHandle, cusparseSpMatDescr, cusparseDnVecDescr
       type(cusparseHandle), value :: handle
       !dir$ ignore_tkr (tkrd) alpha, (tkrd) beta
       real(4) :: alpha, beta
       type(cusparseSpMatDescr), value :: matA
       type(cusparseDnVecDescr), value :: vecX, vecY
       integer(4), value  :: opA, computeType, alg
       !dir$ ignore_tkr buffer
       integer(4), device :: buffer
     end function cusparseSpMV
  end interface cusparseSpMV

! ---------------

! cusparseSpMM
  interface cusparseSpMM_bufferSize
     integer(c_int) function cusparseSpMM_buffersize(handle, opA, opB, alpha, &
                        matA, matB, beta, matC, computeType, alg, bufferSize) &
                                        bind(C, name='cusparseSpMM_bufferSize')
       import cusparseHandle, cusparseSpMatDescr, cusparseDnMatDescr
       type(cusparseHandle), value :: handle
       !dir$ ignore_tkr (tkrd) alpha, (tkrd) beta
       real(4) :: alpha, beta
       type(cusparseSpMatDescr), value :: matA
       type(cusparseDnMatDescr), value :: matB, matC
       integer(4), value  :: opA, opB, computeType, alg
       integer(8)         :: bufferSize
     end function cusparseSpMM_buffersize
  end interface cusparseSpMM_buffersize

  interface cusparseSpMM_preprocess
     integer(c_int) function cusparseSpMM_preprocess(handle, opA, opB, alpha, &
                        matA, matB, beta, matC, computeType, alg, buffer) &
                                   bind(C, name='cusparseSpMM_preprocess')
       import cusparseHandle, cusparseSpMatDescr, cusparseDnMatDescr
       type(cusparseHandle), value :: handle
       !dir$ ignore_tkr (tkrd) alpha, (tkrd) beta
       real(4) :: alpha, beta
       type(cusparseSpMatDescr), value :: matA
       type(cusparseDnMatDescr), value :: matB, matC
       integer(4), value  :: opA, opB, computeType, alg
       !dir$ ignore_tkr buffer
       integer(4), device :: buffer
     end function cusparseSpMM_preprocess
  end interface cusparseSpMM_preprocess

  interface cusparseSpMM
     integer(c_int) function cusparseSpMM(handle, opA, opB, alpha, &
                        matA, matB, beta, matC, computeType, alg, buffer) &
                                               bind(C, name='cusparseSpMM')
       import cusparseHandle, cusparseSpMatDescr, cusparseDnMatDescr
       type(cusparseHandle), value :: handle
       !dir$ ignore_tkr (tkrd) alpha, (tkrd) beta
       real(4) :: alpha, beta
       type(cusparseSpMatDescr), value :: matA
       type(cusparseDnMatDescr), value :: matB, matC
       integer(4), value  :: opA, opB, computeType, alg
       !dir$ ignore_tkr buffer
       integer(4), device :: buffer
     end function cusparseSpMM
  end interface cusparseSpMM

! ---------------

! cusparseSDDMM
  interface cusparseSDDMM_bufferSize
     integer(c_int) function cusparseSDDMM_buffersize(handle, opA, opB, alpha, &
                        matA, matB, beta, matC, computeType, alg, bufferSize) &
                                        bind(C, name='cusparseSDDMM_bufferSize')
       import cusparseHandle, cusparseSpMatDescr, cusparseDnMatDescr
       type(cusparseHandle), value :: handle
       !dir$ ignore_tkr (tkrd) alpha, (tkrd) beta
       real(4) :: alpha, beta
       type(cusparseDnMatDescr), value :: matA, matB
       type(cusparseSpMatDescr), value :: matC
       integer(4), value  :: opA, opB, computeType, alg
       integer(8)         :: bufferSize
     end function cusparseSDDMM_buffersize
  end interface cusparseSDDMM_buffersize

  interface cusparseSDDMM_preprocess
     integer(c_int) function cusparseSDDMM_preprocess(handle, opA, opB, alpha, &
                        matA, matB, beta, matC, computeType, alg, buffer) &
                                   bind(C, name='cusparseSDDMM_preprocess')
       import cusparseHandle, cusparseSpMatDescr, cusparseDnMatDescr
       type(cusparseHandle), value :: handle
       !dir$ ignore_tkr (tkrd) alpha, (tkrd) beta
       real(4) :: alpha, beta
       type(cusparseDnMatDescr), value :: matA, matB
       type(cusparseSpMatDescr), value :: matC
       integer(4), value  :: opA, opB, computeType, alg
       !dir$ ignore_tkr buffer
       integer(4), device :: buffer
     end function cusparseSDDMM_preprocess
  end interface cusparseSDDMM_preprocess

  interface cusparseSDDMM
     integer(c_int) function cusparseSDDMM(handle, opA, opB, alpha, &
                        matA, matB, beta, matC, computeType, alg, buffer) &
                                               bind(C, name='cusparseSDDMM')
       import cusparseHandle, cusparseSpMatDescr, cusparseDnMatDescr
       type(cusparseHandle), value :: handle
       !dir$ ignore_tkr (tkrd) alpha, (tkrd) beta
       real(4) :: alpha, beta
       type(cusparseDnMatDescr), value :: matA, matB
       type(cusparseSpMatDescr), value :: matC
       integer(4), value  :: opA, opB, computeType, alg
       !dir$ ignore_tkr buffer
       integer(4), device :: buffer
     end function cusparseSDDMM
  end interface cusparseSDDMM

! ---------------
! cusparseSpSV

! cusparseSpSV_createDescr
  interface
     integer(c_int) function cusparseSpSV_createDescr(descr) bind(C,name='cusparseSpSV_createDescr')
       import cusparseSpSVDescr
       type(cusparseSpSVDescr) :: descr
     end function cusparseSpSV_createDescr
  end interface

! cusparseSpSV_destroyDescr
  interface
     integer(c_int) function cusparseSpSV_destroyDescr(descr) bind(C,name='cusparseSpSV_destroyDescr')
       import cusparseSpSVDescr
       type(cusparseSpSVDescr), value :: descr
     end function cusparseSpSV_destroyDescr
  end interface

! cusparseSpSV_buffersize
  interface cusparseSpSV_bufferSize
     integer(c_int) function cusparseSpSV_buffersize(handle, opA, alpha, &
                   matA, vecX, vecY, computeType, alg, spsvDescr, bufferSize) &
                                        bind(C, name='cusparseSpSV_bufferSize')
       import cusparseHandle, cusparseSpMatDescr, cusparseDnVecDescr
       import cusparseSpSVDescr
       type(cusparseHandle), value :: handle
       !dir$ ignore_tkr (tkrd) alpha
       real(4) :: alpha
       type(cusparseSpMatDescr), value :: matA
       type(cusparseDnVecDescr), value :: vecX, vecY
       integer(4), value  :: opA, computeType, alg
       type(cusparseSpSVDescr), value :: spsvDescr
       integer(8)         :: bufferSize
     end function cusparseSpSV_buffersize
  end interface cusparseSpSV_buffersize

! cusparseSpSV_analysis
  interface cusparseSpSV_analysis
     integer(c_int) function cusparseSpSV_analysis(handle, opA, alpha, &
                   matA, vecX, vecY, computeType, alg, spsvDescr, buffer) &
                                        bind(C, name='cusparseSpSV_analysis')
       import cusparseHandle, cusparseSpMatDescr, cusparseDnVecDescr
       import cusparseSpSVDescr
       type(cusparseHandle), value :: handle
       !dir$ ignore_tkr (tkrd) alpha
       real(4) :: alpha
       type(cusparseSpMatDescr), value :: matA
       type(cusparseDnVecDescr), value :: vecX, vecY
       integer(4), value  :: opA, computeType, alg
       type(cusparseSpSVDescr), value :: spsvDescr
       !dir$ ignore_tkr buffer
       integer(4), device :: buffer
     end function cusparseSpSV_analysis
  end interface cusparseSpSV_analysis

! cusparseSpSV_solve
  interface cusparseSpSV_solve
     integer(c_int) function cusparseSpSV_solve(handle, opA, alpha, &
                   matA, vecX, vecY, computeType, alg, spsvDescr) &
                                        bind(C, name='cusparseSpSV_solve')
       import cusparseHandle, cusparseSpMatDescr, cusparseDnVecDescr
       import cusparseSpSVDescr
       type(cusparseHandle), value :: handle
       !dir$ ignore_tkr (tkrd) alpha
       real(4) :: alpha
       type(cusparseSpMatDescr), value :: matA
       type(cusparseDnVecDescr), value :: vecX, vecY
       integer(4), value  :: opA, computeType, alg
       type(cusparseSpSVDescr), value :: spsvDescr
     end function cusparseSpSV_solve
  end interface cusparseSpSV_solve

! cusparseSpSV_updateMatrix
  interface cusparseSpSV_updateMatrix
     integer(c_int) function cusparseSpSV_updateMatrix(handle, spsvDescr, &
                   newValues, updatePart) &
                              bind(C, name='cusparseSpSV_updateMatrix')
       import cusparseHandle
       import cusparseSpSVDescr
       type(cusparseHandle), value :: handle
       type(cusparseSpSVDescr), value :: spsvDescr
       !dir$ ignore_tkr (tkr) newValues
       real(4), device :: newValues(*)
       integer(4), value  :: updatePart
     end function cusparseSpSV_updateMatrix
  end interface cusparseSpSV_updateMatrix

! ---------------
! cusparseSpSM

! cusparseSpSM_createDescr
  interface
     integer(c_int) function cusparseSpSM_createDescr(descr) bind(C,name='cusparseSpSM_createDescr')
       import cusparseSpSMDescr
       type(cusparseSpSMDescr) :: descr
     end function cusparseSpSM_createDescr
  end interface

! cusparseSpSM_destroyDescr
  interface
     integer(c_int) function cusparseSpSM_destroyDescr(descr) bind(C,name='cusparseSpSM_destroyDescr')
       import cusparseSpSMDescr
       type(cusparseSpSMDescr), value :: descr
     end function cusparseSpSM_destroyDescr
  end interface

! cusparseSpSM_buffersize
  interface cusparseSpSM_bufferSize
     integer(c_int) function cusparseSpSM_buffersize(handle, opA, opB, alpha, &
                   matA, matB, matC, computeType, alg, spsmDescr, bufferSize) &
                                        bind(C, name='cusparseSpSM_bufferSize')
       import cusparseHandle, cusparseSpMatDescr, cusparseDnMatDescr
       import cusparseSpSMDescr
       type(cusparseHandle), value :: handle
       !dir$ ignore_tkr (tkrd) alpha
       real(4) :: alpha
       type(cusparseSpMatDescr), value :: matA
       type(cusparseDnMatDescr), value :: matB, matC
       integer(4), value  :: opA, opB, computeType, alg
       type(cusparseSpSMDescr), value :: spsmDescr
       integer(8)         :: bufferSize
     end function cusparseSpSM_buffersize
  end interface cusparseSpSM_buffersize

! cusparseSpSM_analysis
  interface cusparseSpSM_analysis
     integer(c_int) function cusparseSpSM_analysis(handle, opA, opB, alpha, &
                   matA, matB, matC, computeType, alg, spsmDescr, buffer) &
                                        bind(C, name='cusparseSpSM_analysis')
       import cusparseHandle, cusparseSpMatDescr, cusparseDnMatDescr
       import cusparseSpSMDescr
       type(cusparseHandle), value :: handle
       !dir$ ignore_tkr (tkrd) alpha
       real(4) :: alpha
       type(cusparseSpMatDescr), value :: matA
       type(cusparseDnMatDescr), value :: matB, matC
       integer(4), value  :: opA, opB, computeType, alg
       type(cusparseSpSMDescr), value :: spsmDescr
       !dir$ ignore_tkr buffer
       integer(4), device :: buffer
     end function cusparseSpSM_analysis
  end interface cusparseSpSM_analysis

! cusparseSpSM_solve
  interface cusparseSpSM_solve
     integer(c_int) function cusparseSpSM_solve(handle, opA, opB, alpha, &
                   matA, matB, matC, computeType, alg, spsmDescr) &
                                        bind(C, name='cusparseSpSM_solve')
       import cusparseHandle, cusparseSpMatDescr, cusparseDnMatDescr
       import cusparseSpSMDescr
       type(cusparseHandle), value :: handle
       !dir$ ignore_tkr (tkrd) alpha
       real(4) :: alpha
       type(cusparseSpMatDescr), value :: matA
       type(cusparseDnMatDescr), value :: matB, matC
       integer(4), value  :: opA, opB, computeType, alg
       type(cusparseSpSMDescr), value :: spsmDescr
     end function cusparseSpSM_solve
  end interface cusparseSpSM_solve

! ---------------
! cusparseSparseToDense and cusparseDenseToSparse

! cusparseSparseToDense_buffersize
  interface cusparseSparseToDense_bufferSize
     integer(c_int) function cusparseSparseToDense_buffersize(handle, &
                   matA, matB, alg, bufferSize) &
                   bind(C, name='cusparseSparseToDense_bufferSize')
       import cusparseHandle, cusparseSpMatDescr, cusparseDnMatDescr
       type(cusparseHandle), value :: handle
       type(cusparseSpMatDescr), value :: matA
       type(cusparseDnMatDescr), value :: matB
       integer(4), value  :: alg
       integer(8)         :: bufferSize
     end function cusparseSparseToDense_buffersize
  end interface cusparseSparseToDense_buffersize

! cusparseSparseToDense
  interface cusparseSparseToDense
     integer(c_int) function cusparseSparseToDense(handle, &
                   matA, matB, alg, buffer) &
                   bind(C, name='cusparseSparseToDense')
       import cusparseHandle, cusparseSpMatDescr, cusparseDnMatDescr
       type(cusparseHandle), value :: handle
       type(cusparseSpMatDescr), value :: matA
       type(cusparseDnMatDescr), value :: matB
       integer(4), value  :: alg
       !dir$ ignore_tkr buffer
       integer(4), device :: buffer
     end function cusparseSparseToDense
  end interface cusparseSparseToDense

! cusparseDenseToSparse_buffersize
  interface cusparseDenseToSparse_bufferSize
     integer(c_int) function cusparseDenseToSparse_buffersize(handle, &
                   matA, matB, alg, bufferSize) &
                   bind(C, name='cusparseDenseToSparse_bufferSize')
       import cusparseHandle, cusparseSpMatDescr, cusparseDnMatDescr
       type(cusparseHandle), value :: handle
       type(cusparseDnMatDescr), value :: matA
       type(cusparseSpMatDescr), value :: matB
       integer(4), value  :: alg
       integer(8)         :: bufferSize
     end function cusparseDenseToSparse_buffersize
  end interface cusparseDenseToSparse_buffersize

! cusparseDenseToSparse_analysis
  interface cusparseDenseToSparse_analysis
     integer(c_int) function cusparseDenseToSparse_analysis(handle, &
                   matA, matB, alg, buffer) &
                   bind(C, name='cusparseDenseToSparse_analysis')
       import cusparseHandle, cusparseSpMatDescr, cusparseDnMatDescr
       type(cusparseHandle), value :: handle
       type(cusparseDnMatDescr), value :: matA
       type(cusparseSpMatDescr), value :: matB
       integer(4), value  :: alg
       !dir$ ignore_tkr buffer
       integer(4), device :: buffer
     end function cusparseDenseToSparse_analysis
  end interface cusparseDenseToSparse_analysis

! cusparseDenseToSparse_convert
  interface cusparseDenseToSparse_convert
     integer(c_int) function cusparseDenseToSparse_convert(handle, &
                   matA, matB, alg, buffer) &
                   bind(C, name='cusparseDenseToSparse_convert')
       import cusparseHandle, cusparseSpMatDescr, cusparseDnMatDescr
       type(cusparseHandle), value :: handle
       type(cusparseDnMatDescr), value :: matA
       type(cusparseSpMatDescr), value :: matB
       integer(4), value  :: alg
       !dir$ ignore_tkr buffer
       integer(4), device :: buffer
     end function cusparseDenseToSparse_convert
  end interface cusparseDenseToSparse_convert

! ---------------
! cusparseSpGEMM

! cusparseSpGEMM_createDescr
  interface
     integer(c_int) function cusparseSpGEMM_createDescr(descr) bind(C,name='cusparseSpGEMM_createDescr')
       import cusparseSpGEMMDescr
       type(cusparseSpGEMMDescr) :: descr
     end function cusparseSpGEMM_createDescr
  end interface

! cusparseSpGEMM_destroyDescr
  interface
     integer(c_int) function cusparseSpGEMM_destroyDescr(descr) bind(C,name='cusparseSpGEMM_destroyDescr')
       import cusparseSpGEMMDescr
       type(cusparseSpGEMMDescr), value :: descr
     end function cusparseSpGEMM_destroyDescr
  end interface

! cusparseSpGEMM_workEstimation
  interface cusparseSpGEMM_workEstimation
     integer(c_int) function cusparseSpGEMM_workEstimation(handle, opA, opB, &
             alpha, matA, matB, beta, matC, computeType, alg, spgemmDescr, &
             bufferSize1, buffer1) bind(C, name='cusparseSpGEMM_workEstimation')
       import cusparseHandle, cusparseSpMatDescr
       import cusparseSpGEMMDescr
       type(cusparseHandle), value :: handle
       !dir$ ignore_tkr (tkrd) alpha, (tkrd) beta
       real(4) :: alpha, beta
       type(cusparseSpMatDescr), value :: matA, matB, matC
       integer(4), value  :: opA, opB, computeType, alg
       type(cusparseSpGEMMDescr), value :: spgemmDescr
       integer(8)         :: bufferSize1
       !dir$ ignore_tkr buffer1
       integer(4), device :: buffer1
     end function cusparseSpGEMM_workEstimation
  end interface cusparseSpGEMM_workEstimation

! cusparseSpGEMM_getNumProducts
  interface cusparseSpGEMM_getNumProducts
     integer(c_int) function cusparseSpGEMM_getNumProducts(spgemmDescr, &
             num_prods) bind(C, name='cusparseSpGEMM_getNumProducts')
       import cusparseSpGEMMDescr
       type(cusparseSpGEMMDescr), value :: spgemmDescr
       integer(8)         :: num_prods
     end function cusparseSpGEMM_getNumProducts
  end interface cusparseSpGEMM_getNumProducts

! cusparseSpGEMM_estimateMemory
  interface cusparseSpGEMM_estimateMemory
     integer(c_int) function cusparseSpGEMM_estimateMemory(handle, opA, opB, &
             alpha, matA, matB, beta, matC, computeType, alg, spgemmDescr, &
             chunk_fraction, bufferSize3, buffer3, bufferSize2) &
             bind(C, name='cusparseSpGEMM_estimateMemory')
       import cusparseHandle, cusparseSpMatDescr
       import cusparseSpGEMMDescr
       type(cusparseHandle), value :: handle
       !dir$ ignore_tkr (tkrd) alpha, (tkrd) beta
       real(4) :: alpha, beta
       type(cusparseSpMatDescr), value :: matA, matB, matC
       integer(4), value  :: opA, opB, computeType, alg
       type(cusparseSpGEMMDescr), value :: spgemmDescr
       real(4), value     :: chunk_fraction
       integer(8)         :: bufferSize3
       !dir$ ignore_tkr buffer3
       integer(4), device :: buffer3
       integer(8)         :: bufferSize2
     end function cusparseSpGEMM_estimateMemory
  end interface cusparseSpGEMM_estimateMemory

! cusparseSpGEMM_compute
  interface cusparseSpGEMM_compute
     integer(c_int) function cusparseSpGEMM_compute(handle, opA, opB, &
             alpha, matA, matB, beta, matC, computeType, alg, spgemmDescr, &
             bufferSize2, buffer2) bind(C, name='cusparseSpGEMM_compute')
       import cusparseHandle, cusparseSpMatDescr
       import cusparseSpGEMMDescr
       type(cusparseHandle), value :: handle
       !dir$ ignore_tkr (tkrd) alpha, (tkrd) beta
       real(4) :: alpha, beta
       type(cusparseSpMatDescr), value :: matA, matB, matC
       integer(4), value  :: opA, opB, computeType, alg
       type(cusparseSpGEMMDescr), value :: spgemmDescr
       !dir$ ignore_tkr buffer2
       integer(8)         :: bufferSize2
       integer(4), device :: buffer2
     end function cusparseSpGEMM_compute
  end interface cusparseSpGEMM_compute

! cusparseSpGEMM_copy
  interface cusparseSpGEMM_copy
     integer(c_int) function cusparseSpGEMM_copy(handle, opA, opB, &
             alpha, matA, matB, beta, matC, computeType, alg, spgemmDescr) &
             bind(C, name='cusparseSpGEMM_copy')
       import cusparseHandle, cusparseSpMatDescr
       import cusparseSpGEMMDescr
       type(cusparseHandle), value :: handle
       !dir$ ignore_tkr (tkrd) alpha, (tkrd) beta
       real(4) :: alpha, beta
       type(cusparseSpMatDescr), value :: matA, matB, matC
       integer(4), value  :: opA, opB, computeType, alg
       type(cusparseSpGEMMDescr), value :: spgemmDescr
     end function cusparseSpGEMM_copy
  end interface cusparseSpGEMM_copy

! cusparseSpGEMMreuse_workEstimation
  interface cusparseSpGEMMreuse_workEstimation
     integer(c_int) function cusparseSpGEMMreuse_workEstimation(handle, opA, opB, &
             matA, matB, matC, alg, spgemmDescr, bufferSize1, buffer1) &
             bind(C, name='cusparseSpGEMMreuse_workEstimation')
       import cusparseHandle, cusparseSpMatDescr
       import cusparseSpGEMMDescr
       type(cusparseHandle), value :: handle
       type(cusparseSpMatDescr), value :: matA, matB, matC
       integer(4), value  :: opA, opB, alg
       type(cusparseSpGEMMDescr), value :: spgemmDescr
       integer(8)         :: bufferSize1
       !dir$ ignore_tkr buffer1
       integer(4), device :: buffer1
     end function cusparseSpGEMMreuse_workEstimation
  end interface cusparseSpGEMMreuse_workEstimation

! cusparseSpGEMMreuse_nnz
  interface cusparseSpGEMMreuse_nnz
     integer(c_int) function cusparseSpGEMMreuse_nnz(handle, opA, opB, &
             matA, matB, matC, alg, spgemmDescr, bufferSize2, buffer2, &
             bufferSize3, buffer3, bufferSize4, buffer4) &
             bind(C, name='cusparseSpGEMMreuse_nnz')
       import cusparseHandle, cusparseSpMatDescr
       import cusparseSpGEMMDescr
       type(cusparseHandle), value :: handle
       type(cusparseSpMatDescr), value :: matA, matB, matC
       integer(4), value  :: opA, opB, alg
       type(cusparseSpGEMMDescr), value :: spgemmDescr
       integer(8)         :: bufferSize2, bufferSize3, bufferSize4
       !dir$ ignore_tkr buffer2, buffer3, buffer4
       integer(4), device :: buffer2, buffer3, buffer4
     end function cusparseSpGEMMreuse_nnz
  end interface cusparseSpGEMMreuse_nnz

! cusparseSpGEMMreuse_copy
  interface cusparseSpGEMMreuse_copy
     integer(c_int) function cusparseSpGEMMreuse_copy(handle, opA, opB, &
             matA, matB, matC, alg, spgemmDescr, bufferSize5, buffer5) &
             bind(C, name='cusparseSpGEMMreuse_copy')
       import cusparseHandle, cusparseSpMatDescr
       import cusparseSpGEMMDescr
       type(cusparseHandle), value :: handle
       type(cusparseSpMatDescr), value :: matA, matB, matC
       integer(4), value  :: opA, opB, alg
       type(cusparseSpGEMMDescr), value :: spgemmDescr
       integer(8)         :: bufferSize5
       !dir$ ignore_tkr buffer5
       integer(4), device :: buffer5
     end function cusparseSpGEMMreuse_copy
  end interface cusparseSpGEMMreuse_copy

! cusparseSpGEMMreuse_compute
  interface cusparseSpGEMMreuse_compute
     integer(c_int) function cusparseSpGEMMreuse_compute(handle, opA, opB, &
             alpha, matA, matB, beta, matC, computeType, alg, spgemmDescr) &
             bind(C, name='cusparseSpGEMMreuse_compute')
       import cusparseHandle, cusparseSpMatDescr
       import cusparseSpGEMMDescr
       type(cusparseHandle), value :: handle
       !dir$ ignore_tkr (tkrd) alpha, (tkrd) beta
       real(4) :: alpha, beta
       type(cusparseSpMatDescr), value :: matA, matB, matC
       integer(4), value  :: opA, opB, computeType, alg
       type(cusparseSpGEMMDescr), value :: spgemmDescr
     end function cusparseSpGEMMreuse_compute
  end interface cusparseSpGEMMreuse_compute

end module cusparse
